/* This program reads in an xbase-dbf file and sends 'inserts' to an
   mSQL-server with the records in the xbase-file

   M. Boekhold (boekhold@cindy.et.tudelft.nl)  okt. 1995
*/

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <msql.h>
#include "dbf.h"

#define VERSION	"0.1"

inline void strtoupper(char *string);
char *msqlEscape(char*);
void usage(void);

inline void strtoupper(char *string) {
	while(*string != '\0') {
		*string++ = toupper(*string);
	}
}

char *msqlEscape(char *string) {
    char    *escaped, *foo, *bar;
    u_long  count = 0;

    foo = string;
    while(*foo != '\0') {
        if (*foo++ == '\'') count++;
    }

    if (!(escaped = (char *)malloc(strlen(string) + count + 1))) {
        return (char *)-1;
    }

    foo = escaped;
    bar = string;
    while (*bar != '\0') {
        if (*bar == '\'') {
            *foo++ = '\\';
        }
        *foo++ = *bar++;
    }
    *foo = '\0';

    return escaped;
}


void usage(void){
		printf("dbf2msql %s\n", VERSION);
		printf("\t\tusage: dbf2msql [-u] [-h hostname] ");
		printf("[-p primary key]\n\t\t[-d dbase][-t table] dbf-file\n");
		exit(1);
}

int main(int argc, char **argv)
{
	int 		SQLsock, i, h, upper = 0;
	extern int 	optind;
	extern char	*optarg;
	char 		*host = NULL;
	char 		*dbase = "test";
	char 		*table = "test";
	char		verbose=0;
	char 		primary[11];
	char 		query[2048];
	char 		keys[1024];
	char 		vals[1024];
	char 		t[20];
	char		*foo;
	int 		length=0, result;
	dbhead 		*dbh;
	field 		*fields;

	primary[0] = '\0';

	while ((i = getopt(argc, argv, "uvh:p:d:t:")) != EOF) {
		switch (i) {
			case 'v':
				verbose=1;
				break;
			case 'u':
				upper=1;
				break;
			case 'h':
				host = (char *)strdup(optarg);
				break;
			case 'p':
				strncpy(primary, optarg, 11);
				break;
			case 'd':
				dbase = (char *)strdup(optarg);
				break;
			case 't':
				table = (char *)strdup(optarg);
				break;
			case ':':
				printf("missing argument!\n");
				usage();
			case '?':
				printf("unknown argument: %s\n", argv[0]);
				usage();
			default:
				break;
		}
	}

	argc -= optind;
	argv = &argv[optind];

	if (argc != 1) {
		usage();
	}

	if (verbose) {
		printf("Opening dbf-file\n");
	}

	if ((dbh = dbf_open(argv[0], O_RDONLY)) == (dbhead *)-1) {
		fprintf(stderr, "Couldn't open xbase-file %s\n", argv[0]);
		exit(1);
	}

	if (verbose) {
		printf("Making connection to mSQL-server\n");
	}

	if ((SQLsock = msqlConnect(host)) == -1) {
		fprintf(stderr, "Couldn't get a connection with the ");
		fprintf(stderr, "designated host!\n");
		fprintf(stderr, "Detailed report: %s\n", msqlErrMsg);
		close(dbh->db_fd);
		free(dbh);
		exit(1);
	}

	if (verbose) {
		printf("Selecting database\n");
	}

	if ((msqlSelectDB(SQLsock, dbase)) == -1) {
		fprintf(stderr, "Couldn't select database %s.\n", dbase);
		fprintf(stderr, "Detailed report: %s\n", msqlErrMsg);
		close(dbh->db_fd);
		free(dbh);
		msqlClose(SQLsock);
		exit(1);
	}

	if (verbose) {
		printf("Dropping table\n");
	}

	sprintf(query, "DROP TABLE %s", table);
	msqlQuery(SQLsock, query);

/* Build a CREATE-clause
*/

	if (verbose) {
		printf("Building CREATE-clause\n");
	}

	sprintf(query, "CREATE TABLE %s (", table);
	length = strlen(query);
	for ( i = 0; i < dbh->db_nfields; i++) {
		if ((strlen(query) != length)) {
                        strcat(query, ",");
                }
                strcat(query, dbh->db_fields[i].db_name);
                switch(dbh->db_fields[i].db_type) {
						case 'D':
                        case 'C':
                                strcat(query, " char");
                                sprintf(t," (%d)",dbh->db_fields[i].db_flen);
                                strcat(query, t);
                                strcat(query, "");
                                break;
                        case 'N':
                                if (dbh->db_fields[i].db_dec != 0) {
                                        strcat(query, " real");
                                } else {
                                        strcat(query, " int");
                                }
                                break;
                        case 'L':
                                strcat(query, " char (1)");
                                break;
                }
                if (strcmp(dbh->db_fields[i].db_name, primary) == 0) {
					strcat(query, " primary key");
				}
	}

	strcat(query, ")\n");

	if (verbose) {
		printf("Sending create-clause\n");
		printf("%s\n", query);
	}

	if (msqlQuery(SQLsock, query) == -1) {
		fprintf(stderr, "Error creating table!\n");
		fprintf(stderr, "Detailed report: %s\n", msqlErrMsg);
                close(dbh->db_fd);
                free(dbh);
                msqlClose(SQLsock);
                exit(1);
	}

/* Build an INSERT-clause
*/
	if (verbose) {
		printf("Inserting records\n");
	}

	for ( i = 0; i < dbh->db_records; i++) {
		if ((fields = dbf_build_record(dbh)) != (field *)DBF_ERROR) {
			result = dbf_get_record(dbh, fields,  i);
			if (result == DBF_VALID) {
				keys[0]='\0';
				vals[0]='\0';

				for (h = 0; h < dbh->db_nfields; h++) {
					if (strlen(keys) != 0) {
						strcat(keys, ",");
					}
					if (strlen(vals) != 0) {
						strcat(vals, ",");
					}
					strcat(keys, fields[h].db_name);
					if (fields[h].db_type != 'N') {
						strcat(vals, "'");
					}
					if (upper) {
					   strtoupper(fields[h].db_contents);
					}
					foo=msqlEscape(fields[h].db_contents);
					strcat(vals, foo);
					free(foo);
					if (fields[h].db_type != 'N') {
						strcat(vals, "'");
					}
				}

				sprintf(query,
					"INSERT INTO %s ( %s ) VALUES ( %s )\n",
				    	table, keys, vals);

				if ((verbose) && ((i % 100) == 0)) {
					printf("Inserting record %d\n", i);
				}

				if (msqlQuery(SQLsock, query) == -1) {
					fprintf(stderr,
						"Error sending INSERT in record %04d\n", i);
					fprintf(stderr,
						"Detailed report: %s\n",
						msqlErrMsg);
				}
			}
			dbf_free_record(dbh, fields);
		}
	}

	if (verbose) {
		printf("Closing up....\n");
	}

        close(dbh->db_fd);
        free(dbh);
        msqlClose(SQLsock);
	exit(0);
}
