/* 
 * Nested n-dimensional blocks 
 * David Kotz dfk@cs.duke.edu 
 */

#include <stdio.h>

#define MAXDIMS 10
#define MAXBOXES 30

typedef int boolean;
#define TRUE ((boolean)1)
#define FALSE ((boolean)0)

typedef int BoxType[MAXDIMS];	/* represents one box */
BoxType boxlist[MAXBOXES+1];	/* represents all the boxes 1..numboxes */
						/* note box=0 is special */

struct info {
    int length;			/* length of longest path out of here */
    int next;				/* next box in longest path */
    boolean visited;		/* have we been here yet? */
} boxinfo[MAXBOXES+1];		/* one for each box, plus box 0 */

boolean nests[MAXBOXES+1][MAXBOXES+1];	/* [i][j]=TRUE means i nests in j */

void SortBox();
void PrintBox();
void BuildGraph();
boolean DoesNestIn();
void LongestNesting();
void Visit();

main()
{
    char line[BUFSIZ];		/* input line */
    int numboxes;			/* number of boxes */
    int dims;				/* number of dimensions */
    int b, d;				/* box number, dimension number */

    /* Repeatedly read data sets */
    while (scanf("%d %d\n", &numboxes, &dims) == 2) {
	   for (b = 1; b <= numboxes; b++) {
		  for (d = 0; d < dims; d++)
		    scanf("%d ", &boxlist[b][d]);
		  SortBox(boxlist[b], dims);
/*
		  printf("%d: ", b);
		  PrintBox(boxlist[b], dims);
*/
	   }
	   BuildGraph(numboxes, dims);
	   LongestNesting(numboxes, dims);
    }

}

/* Sort a box's dimensions */
void
SortBox(box, dims)
	BoxType box;			/* a box */
	int dims;				/* number of dimensions */
{
    int i,j;
    int temp;

    for (i = 0; i < dims-1; i++) {
	   for (j = i+1; j < dims; j++) {
		  if (box[i] > box[j]) {
			 temp = box[i];
			 box[i] = box[j];
			 box[j] = temp;
		  }
	   }
    }
}

/* Print a box */
void
PrintBox(box, dims)
	BoxType box;			/* a box */
	int dims;				/* number of dimensions */
{
    int i;

    for (i = 0; i < dims; i++)
	 printf("%d ", box[i]);
    printf("\n");
}

/* Build the nesting graph */
void
BuildGraph(numboxes, dims)
    int numboxes;			/* number of boxes */
    int dims;				/* number of dimensions */
{
    int i,j;				/* box numbers */

    for (i = 1; i <= numboxes; i++) {
	   for (j = 1; j <= numboxes; j++) {
		  if (i == j)
		    nests[i][j] = FALSE;
		  else
		    nests[i][j] = DoesNestIn(boxlist[i],boxlist[j], dims);
	   }
    }

    /* This is a special case: box 0 nests in all boxes */
    for (j = 1; j <= numboxes; j++)
	 nests[0][j] = TRUE;
    
}

boolean					/* does block a nest in block b? */
DoesNestIn(a,b, dims)
	BoxType a,b;			/* the two boxes */
	int dims;				/* number of dimensions */
{
    int i;

    for (i = 0; i < dims; i++) {
	   if (a[i] >= b[i])
		return(FALSE);
    }
    return(TRUE);
}

/* Find the longest nesting sequence */
void
LongestNesting(numboxes, dims)
    int numboxes;			/* number of boxes */
    int dims;				/* number of dimensions */
{
    int b;

    for (b = 0; b <= numboxes; b++)
	   boxinfo[b].visited = FALSE;

    Visit(0, numboxes);		/* 0 means visit them all, top level */

    /* print the longest path */
    printf("%d\n", boxinfo[0].length);
    for (b = boxinfo[0].next; b > 0; b = boxinfo[b].next)
	 printf("%d ", b);
    printf("\n");
}

/* visit one node */
void
Visit(b, numboxes)
	int b;				/* box number */
	int numboxes;			/* number of boxes */
{
    int n;				/* another box */
    int length = 0, which = -1; /* maximum-length path */

    /* first mark that we were here */
    boxinfo[b].visited = TRUE;

    for (n = 1; n <= numboxes; n++) {
	   if (b == n)
		continue;
	   if (nests[b][n]) {	/* remember 0 nests in all */
		  if (!boxinfo[n].visited)
		    Visit(n, numboxes);
		  if (boxinfo[n].length+1 > length) {
			 length = boxinfo[n].length+1;
			 which = n;
		  }		    
	   }
    }
    boxinfo[b].length = length;
    boxinfo[b].next = which;
}
