------------------------------------------------------------------------------
--                                                                          --
--                GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--                    S Y S T E M . R E N D E Z V O U S                     --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                    (corresponds to FSU revision 1.33)                    --
--                                                                          --
--           Copyright (c) 1992,1993,1994 FSU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with System.std; use System.std;

with System.Compiler_Exceptions;
with System.Task_IDs; use System.Task_IDs;
with System.Entries;
with System.Entry_Queue;
with System.Protected_Objects;

package System.Rendezvous is

   Null_Task_Entry : constant := Entries.Null_Entry;

   Max_Task_Entry : constant Integer := Entries.Max_Entry;

   type Task_Entry_Index is new Entries.Entry_Index
     range Entries.Null_Entry .. Entries.Max_Entry;

   procedure Task_Entry_Call
     (Acceptor              : Task_ID;
      E                     : Task_Entry_Index;
      Parameter             : System.Address;
      Mode                  : Protected_Objects.Call_Modes;
      Rendezvous_Successful : out Boolean);
   --  General entry call

   procedure Call_Simple
     (Acceptor  : Task_ID;
      E         : Task_Entry_Index;
      Parameter : System.Address);
   --  Simple entry call

   procedure Cancel_Task_Entry_Call (Cancelled : out Boolean);
   --  Cancel pending task entry call

   type Select_Modes is (Simple_Mode, Else_Mode, Terminate_Mode);

   Max_Select : constant Integer := Integer'Last; -- RTS-defined

   type Select_Index is range 0 .. Max_Select;

   No_Rendezvous : constant Select_Index := 0;

   subtype Positive_Select_Index is
     Select_Index range 1 .. Select_Index'Last;

   type Accept_Alternative is record
      Null_Body : Boolean;
      S         : Task_Entry_Index;
   end record;
   pragma Pack (Accept_Alternative);

   type Accept_List is
     array (Positive_Select_Index range <>) of Accept_Alternative;

   type Accept_List_Access is access constant Accept_List;

   procedure Selective_Wait
     (Open_Accepts : Accept_List_Access;
      Select_Mode  : Select_Modes;
      Parameter    : out System.Address;
      Index        : out Select_Index);
   --  Selective wait

   procedure Accept_Call (E : Task_Entry_Index; Parameter : out Address);
   --  Accept an entry call

   procedure Accept_Trivial (E : Task_Entry_Index);
   --  Accept an entry call that has no parameters and no body

   function Count (E : Task_Entry_Index) return Natural;
   --  Return number of tasks waiting on the entry E (of current task)

   function Callable (T : Task_ID) return Boolean;
   --  Return T'CALLABLE

   procedure Complete_Rendezvous;
   --  Called by acceptor to wake up caller

   procedure Exceptional_Complete_Rendezvous
     (Ex : Compiler_Exceptions.Exception_ID);
   --  Called by acceptor to mark the end of the current rendezvous and
   --  propagate an exception to the caller.

   --------------------------------
   --  RTS Internal Declarations --
   --------------------------------

   procedure Close_Entries (Target : Task_ID);
   --  Not part of the CARTS interface.
   --  close entries, purge entry queues
   --  called by Task_Stages.Complete

   procedure Reset_Priority
     (Acceptor_Prev_Priority : Entry_Queue.Rendezvous_Priority;
      Acceptor               : Task_ID);
   pragma Inline (Reset_Priority);
   --  Not part of the CARTS interface.
   --  reset priority to original value

end System.Rendezvous;
