------------------------------------------------------------------------------
--                                                                          --
--                GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--                      S Y S T E M . P T H R E A D S                       --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                    (corresponds to FSU revision 1.20)                    --
--                                                                          --
--           Copyright (c) 1992,1993,1994 FSU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package interfaces with Pthreads. It is not a complete interface;
--  it only includes what is needed to implement the Ada runtime.

with System.C_Constants;

with System.POSIX_RTE;   use System.POSIX_RTE;
with System.POSIX_ERROR; use System.POSIX_ERROR;

package System.Pthreads is

   Draft5 : constant Boolean := True;

   Pthreads_Error : exception;

   type Interrupt_State is new Integer;

   type Interrupt_Type is new Integer;

   subtype Priority_Type is Integer
     range System.Priority'First ..
           System.Priority'Last + C_Constants.Add_Prio;

   PTHREAD_INTR_ENABLE       : constant Interrupt_State := 2;
   PTHREAD_INTR_DISABLE      : constant Interrupt_State := 1;
   PTHREAD_INTR_CONTROLLED   : constant Interrupt_Type  := 0;
   PTHREAD_INTR_ASYNCHRONOUS : constant Interrupt_Type  := 1;

   subtype size_t is Integer;

   type Byte is range 0 .. 2#1#E8 - 1;
   for Byte'Size use 8;

   type Halfword is range 0 .. 2#1#E16 - 1;
   for Halfword'Size use 16;

   type Word is mod 2 ** 32;
   for Word'SIZE use 32;

   subtype pthread_t is Word;
   --  Really this declaration should be:
   --
   --    type pthread_t is array (1 .. C_Constants.pthread_t_size) of Byte;
   --
   --  This is the only type returned by an Ada function. We declare it as
   --  a Word for efficiency reasons (Verdix in particular manipulates arrays
   --  of bytes a byte at a time, which we want to avoid). Note that we are
   --  counting on pthread_t_size having the value 4.

   type pthread_mutex_t is
     array (1 .. C_Constants.pthread_mutex_t_size)     of Byte;

   type pthread_cond_t is
     array (1 .. C_Constants.pthread_cond_t_size)      of Byte;

   type pthread_attr_t is
     array (1 .. C_Constants.pthread_attr_t_size)      of Byte;

   type pthread_mutexattr_t is
     array (1 .. C_Constants.pthread_mutexattr_t_size) of Byte;

   type pthread_condattr_t is
     array (1 .. C_Constants.pthread_condattr_t_size)  of Byte;

   subtype pthread_key_t is Word;
   --  Logically this should be declared as:
   --
   --   type pthread_key_t is
   --     array (1 .. C_Constants.pthread_key_t_size) of Byte;
   --
   --  This type is passed as a scaler into a C function. It must be declared
   --  as a scaler, not an array, to ensure that it is passed correctly. Note
   --  that we are counting on pthread_key_t_size having the value 4.

   subtype pthread_protocol_t is Integer;

   NO_PRIO_INHERIT : constant pthread_protocol_t := 0;
   PRIO_INHERIT    : constant pthread_protocol_t := 1;
   PRIO_PROTECT    : constant pthread_protocol_t := 2;

   procedure pthread_attr_init
     (attributes    : out pthread_attr_t;
      result        : out Return_Code);
   pragma Inline (pthread_attr_init);

   procedure pthread_attr_setstacksize
     (attr          : in out pthread_attr_t;
      stacksize     : size_t;
      result        : out Return_Code);
   pragma Inline (pthread_attr_setstacksize);

   procedure pthread_attr_setdetachstate
     (attr          : in out pthread_attr_t;
      detachstate   : Integer;
      Result        : out Return_Code);
   pragma Inline (pthread_attr_setdetachstate);

   procedure pthread_create
     (thread        : out pthread_t;
      attributes    : pthread_attr_t;
      start_routine : Address;
      arg           : Address;
      result        : out Return_Code);
   pragma Inline (pthread_create);

   procedure pthread_init;

   function pthread_self return pthread_t;
   pragma Inline (pthread_self);

   procedure pthread_detach
     (thread        : in out pthread_t;
      result        : out Return_Code);
   pragma Inline (pthread_detach);

   procedure pthread_mutexattr_init
     (attributes    : out pthread_mutexattr_t;
      result        : out Return_Code);
   pragma Inline (pthread_mutexattr_init);

   procedure pthread_mutexattr_setprotocol
     (attributes    : in out pthread_mutexattr_t;
      protocol      : pthread_protocol_t;
      result        : out Return_Code);
   pragma Inline (pthread_mutexattr_setprotocol);

   procedure pthread_mutexattr_setprio_ceiling
     (attributes    : in out pthread_mutexattr_t;
      prio_ceiling  : Integer;
      result        : out Return_Code);
   pragma Inline (pthread_mutexattr_setprio_ceiling);

   procedure pthread_mutex_init
     (mutex         : out pthread_mutex_t;
      attributes    : pthread_mutexattr_t;
      result        : out Return_Code);
   pragma Inline (pthread_mutex_init);

   procedure pthread_mutex_destroy
     (mutex         : in out pthread_mutex_t;
      result        : out Return_Code);
   pragma Inline (pthread_mutex_destroy);

   procedure pthread_mutex_trylock
     (mutex         : in out pthread_mutex_t;
      result        : out Return_Code);
   pragma Inline (pthread_mutex_trylock);

   procedure pthread_mutex_lock
     (mutex         : in out pthread_mutex_t;
      result        : out Return_Code);
   pragma Inline (pthread_mutex_lock);

   procedure pthread_mutex_unlock
     (mutex         : in out pthread_mutex_t;
      result        : out Return_Code);
   pragma Inline (pthread_mutex_unlock);

   procedure pthread_cond_init
     (condition     : out pthread_cond_t;
      attributes    : pthread_condattr_t;
      result        : out Return_Code);
   pragma Inline (pthread_cond_init);

   procedure pthread_cond_wait
     (condition     : in out pthread_cond_t;
      mutex         : in out pthread_mutex_t;
      result        : out Return_Code);
   pragma Inline (pthread_cond_wait);

   procedure pthread_cond_timedwait
     (condition     : in out pthread_cond_t;
      mutex         : in out pthread_mutex_t;
      absolute_time : POSIX_RTE.timespec;
      result        : out Return_Code);
   pragma Inline (pthread_cond_timedwait);

   procedure pthread_cond_signal
     (condition     : in out pthread_cond_t;
      result        : out Return_Code);
   pragma Inline (pthread_cond_signal);

   procedure pthread_cond_broadcast
     (condition     : in out pthread_cond_t;
      result        : out Return_Code);
   pragma Inline (pthread_cond_broadcast);

   procedure pthread_cond_destroy
     (condition     : in out pthread_cond_t;
      result        : out Return_Code);
   pragma Inline (pthread_cond_destroy);

   procedure pthread_condattr_init
     (attributes    : out pthread_condattr_t;
      result        : out Return_Code);
   pragma Inline (pthread_condattr_init);

   procedure pthread_condattr_destroy
     (attributes    : in out pthread_condattr_t;
      result        : out Return_Code);
   pragma Inline (pthread_condattr_destroy);

   procedure pthread_setspecific
     (key           : pthread_key_t;
      value         : System.Address;
      result        : out Return_Code);
   pragma Inline (pthread_setspecific);

   procedure pthread_getspecific
     (key           : pthread_key_t;
      value         : out System.Address;
      result        : out Return_Code);
   pragma Inline (pthread_getspecific);

   procedure pthread_key_create
     (key           : out pthread_key_t;
      destructor    : System.Address;
      result        : out Return_Code);
   pragma Inline (pthread_key_create);

   procedure pthread_setprio
     (thread        : pthread_t;
      priority      : Priority_Type;
      result        : out Return_Code);
   pragma Inline (pthread_setprio);

   procedure pthread_attr_setprio
     (attr          : in out pthread_attr_t;
      priority      : Priority_Type;
      result        : out Return_Code);
   pragma Inline (pthread_attr_setprio);

   procedure pthread_attr_getprio
     (attr          : pthread_attr_t;
      priority      : out Priority_Type;
      result        : out Return_Code);
   pragma Inline (pthread_attr_getprio);

   procedure pthread_setschedattr
     (thread        : pthread_t;
      attributes    : pthread_attr_t;
      result        : out Return_Code);
   pragma Inline (pthread_setschedattr);

   procedure pthread_getschedattr
     (thread        : pthread_t;
      attributes    : out pthread_attr_t;
      result        : out Return_Code);
   pragma Inline (pthread_getschedattr);

   procedure pthread_exit
     (status        : System.Address);
   pragma Import (C, pthread_exit, "pthread_exit");

   procedure pthread_kill
     (thread        : pthread_t;
      sig           : Signal_Number;
      result        : out Return_Code);
   pragma Inline (pthread_kill);

   procedure pthread_cleanup_push
     (routine       : System.Address;
      arg           : System.Address);
   pragma Import (C, pthread_cleanup_push, "pthread_cleanup_push");

   procedure pthread_cleanup_pop
     (execute       : Integer);
   pragma Import (C, pthread_cleanup_pop, "pthread_cleanup_pop");

   procedure pthread_yield;
   pragma Import (C, pthread_yield, "pthread_yield");

end System.Pthreads;
