------------------------------------------------------------------------------
--                                                                          --
--                GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--             S Y S T E M . P R O T E C T E D _ O B J E C T S              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                    (corresponds to FSU revision 1.20)                    --
--                                                                          --
--           Copyright (c) 1992,1993,1994 FSU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with System.Compiler_Exceptions;
with System.Primitives;
with System.Entries;
with System.Task_IDs;
with System.Entry_Queue; use System.Entry_Queue;

with Unchecked_Conversion;

package System.Protected_Objects is

   Null_Protected_Entry : constant := Entries.Null_Entry;

   Max_Protected_Entry : constant := Entries.Max_Entry;

   type Protected_Entry_Index is new Entries.Entry_Index
     range Null_Protected_Entry .. Max_Protected_Entry;

   type Call_Modes is  (Simple_Call, Conditional_Call, Asynchronous_Call);

   function C_M_To_EQ_Call_Modes is new
     Unchecked_Conversion (Call_Modes, EQ_Call_Modes);

   function EQ_C_M_To_Call_Modes is new
     Unchecked_Conversion (EQ_Call_Modes, Call_Modes);

   type Protected_Entry_Queue_Array is
     array (Protected_Entry_Index range <>) of
       System.Entry_Queue.Entry_Queue;

   type Protection (Num_Entries : Protected_Entry_Index) is record
      L                : Primitives.Lock;
      Pending_Call     : Entry_Call_Link;
      Call_In_Progress : Entry_Call_Link;
      Entry_Queues     : Protected_Entry_Queue_Array (1 .. Num_Entries);
   end record;

   type Protection_Access is access all Protection;

   function EQ_P_A_To_Protection_Access is new
     Unchecked_Conversion (EQ_Protection_Access, Protection_Access);

   function P_A_To_EQ_Protection_Access is new
     Unchecked_Conversion (Protection_Access, EQ_Protection_Access);

   procedure Initialize_Protection
     (Object           : access Protection;
      Ceiling_Priority : Integer);

   procedure Finalize_Protection (Object : access Protection);

   procedure Lock (Object : access Protection);

   procedure Lock_Read_Only (Object : access Protection);

   procedure Unlock (Object : access Protection);

   procedure Protected_Entry_Call
     (Object    : access Protection;
      E         : Protected_Entry_Index;
      Parameter : System.Address;
      Mode      : Call_Modes);

   procedure Complete_Protected_Entry_Call (Call_Serviced : out Boolean);

   type Barrier_Vector is array (Protected_Entry_Index range <>) of Boolean;

   procedure Next_Entry_Call
     (Object    : access Protection;
      Barriers  : Barrier_Vector;
      Parameter : out System.Address;
      E         : out Protected_Entry_Index);

   procedure Complete_Entry_Body (Object : access Protection);

   procedure Exceptional_Complete_Entry_Body
     (Object : access Protection;
      Ex     : Compiler_Exceptions.Exception_ID);

   procedure Requeue_Protected_Entry
     (Object, New_Object : access Protection;
      E                  : Protected_Entry_Index;
      With_Abort         : Boolean);


   procedure Requeue_Task_Entry
     (New_Object : access Protection;
      T          : in out Task_IDs.Task_ID;
      E          : Protected_Entry_Index;
      With_Abort : Boolean);

   function Count
     (Object : Protection;
      E      : Protected_Entry_Index)
      return Natural;

   procedure Broadcast_Program_Error (Object : access Protection);

   function Get_Call_In_Progress (Object : Protection) return Entry_Call_Link;

   procedure Set_Call_In_Progress
     (Object : access Protection;
      Call   : Entry_Call_Link);

   procedure Check_Exception;

end System.Protected_Objects;
