/*
 *
 *  Function:   dialer program for the different modems
 *
 *  Usage:      gdial -t Type [-d Dialers-file] [-xD] ttyname telnumber: speed  [send expect ...]
 *              gdial -t Type [-d Dialers-file] [-xD] -h ttyname speed
 *              gdial -t Type [-d Dialers-file] [-xD] -z ttyname speed []
 *              
 *              Type .... (    gdial,   
 *                            )
 *
 *  Returns:    0x80    ()   
 *              0x10    ()   ungetty -r
 *               0x8X  X
 *                / 
 *                      1 -  
 *                      2 -   
 *                      3 -   
 *                      4 -   
 *                      5 -    
 *                      6 -    
 *                      7 -  ioctl
 *                      8 - -
 *                      9 - no dial tone
 *                      a -   hangup
 *                      b -     /  ERROR
 *                
 *                      
 *                      d - 
 *                      e -  
 *                      f -  
 *               !0x80,  0x.X X -  
 *                      0 -   
 *                      >0 - boud rate
 *      CLOCAL,     CLOCAL .
 *      
 *          HUPCL
 *  -h    HUPCL  -CLOCAL
 *               -z    ,  
 *                  .
 *
 *                  -   0x80,  
 *
 *
 *  :        HDB UUCP (
 *                    ),  
 *              UUCP     UUCP (  ungetty).
 */
static char rcsid[] = "$Header: gdial.c,v 1.14 92/10/25 23:30:15 alex STAB4-1-1 $";

#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>
# ifdef WITH_UUCP
#include "uucpconf.h"
# else
/*  delay    */
#ifndef M_XENIX
#  define INTERVALTIMER
# else
#   define INTER_FTIME
# endif
/*
 *   SIGTYPE
 */
# ifdef SCO_UNIX
#     define SIGTYPE void
# else
#     define SIGTYPE int
# endif /* SCO_UNIX     */
# ifdef INTERVALTIMER
#    define uucpdelay(num,denom) intervaldelay(num,denom)
# else
#    ifdef INTER_FTIME
#        define uucpdelay(num,denom) ftimedelay(1000*num/denom)
#    else
#        define uucpdelay(num,denom) /* */
#    endif
# endif
# endif

char *DIALERS="/etc/uucp/L.dialers";

#define HDUU

# ifdef ultrix
/* DHU Emulex */
# define B57600 B1800
# endif

#ifndef B19200
#define B19200 EXTA
#endif /*  */

#ifndef B38400
#define B38400 EXTB
#endif /*  */
#define CNULL (char *)0
#define OPTLEN 128


/* return codes: these are set up so that an abort signal at any time can */
/* set the fail bit and return to the caller with the correct status */
#define	RC_BAUD		0x0f	/* CBAUD connected at (0=same as dialed speed)*/
#define	RC_ENABLED	0x10	/* enabled flag: 1 = ungetty -r required to */
				/* restore the line */
#define RC_BADTYPE      0x20    /* Bad modem type */
#define	RC_FAIL		0x80	/* 1 = failed to connect */

/* error return codes */
#define	RCE_NULL	0	/* general purpose or unknown error code */
#define	RCE_INUSE	1	/* line in use */
#define	RCE_SIG		2	/* signal aborted dialer */
#define	RCE_ARGS	3	/* invalid arguments */
#define	RCE_PHNO	4	/* invalid phone number */
#define	RCE_SPEED	5	/* invalid baud rate -or- bad connect baud */
#define	RCE_OPEN	6	/* can't open line */
#define	RCE_IOCTL	7	/* ioctl error */
#define	RCE_TIMOUT	8	/* timeout */
#define	RCE_NOTONE	9	/* no dial tone */
#define	RCE_HANGUP	10	/* hangup failed */
#define RCE_NORESP	11	/* Modem didn't respond. */
#define RCE_BUSY        13      /* phone is busy, no carier, no answer */

#define	SUCCESS	0

/* ungetty return codes */
#define	UG_NOTENAB	0
#define	UG_ENAB		1
#define	UG_RESTART	1
#define	UG_FAIL		2

#define SAME		0
#define MAXLINE		256
#define	UNGETTY		"/usr/lib/uucp/ungetty"

#define DEBUG(l, f, s)	if (Debug >= l) fprintf(stderr, f, s)
#ifndef DBG
#define	DBG	0
#endif /*  */


/*      */
#define	P_ZERO	0
#define	P_ONE	1
#define	P_EVEN	2
#define	P_ODD	3

char	par_tab[128];	/* must be power of two */
char *modem_type;

typedef char **SE; /* Send / expect, \WNNN - Time out */
/*
 *   \xx    L.sys  
 * 
 * \Dn -  DTR   
 * \Wnnn -     nnn
 * \O -   
 * \T -    
 * \H -    Hang Up.
 * \\ - \
 * \C (  ) -     
 * A_Ok -  
 * A_Con -  
 * A_busy - ,  ,  
 * A_Err  -  
 * A_Nod  -   
 *    
 */
/*
 *   -
 */
#define FAIL            -1      /* Read/ write FAIL */
#define OK              0       /* Command succesful */
#define ERROR           1       /* Connect timeout has occurred */
#define NODIALTONE      2       /* Command error encountered */
#define BUSY            3       /* No dial tone was detected */
#define NOANSWER        4       /* Remote site did not answer */
#define RRING           5       /* Remote site is ringing */
#define RING            6       /* Local site is ringing  */
#define CONNECT         7
#define NOCARRIER       8
#define OTHER           9       /* Other answer */


unsigned Bauds[] = {300, 600, 1200, 1800, 2400, 4800, 9600, 19200, 38400, 57600, 0};
/* ---------     ------------- */
/* S_Baud -    */
unsigned *S_Baud = Bauds;
/*
 *  MDSETUP - set up for DIDO - don't assume anything about modem state
 *	or defaults.
 *
 *
 * Note that we have to break the setup string into two command lines.
 */

/* Hang up */
SE S_Hang;

/* Write Profile (Assume comand mode OK */
SE S_Init;

/* Call */
SE S_Call;

/* Host */
SE S_Host;

/* Reset - if can't hang */
SE S_Reset;

/*
 *  
 *
 *
 * S_Valid  -    
 */
char *S_Valid = "0123456789WPTwpt@,/";

/*
 * .  OK   ,    
 * 
 */
# define MAX_ANSWERS 64
struct answers {
char *a_ans;
int   a_type; } A_Codes[MAX_ANSWERS];

/*
 *  RING  ,   ,  
 *   answers
 */
char *S_Ring = "RING";

/*
 *     - ,  
 */
char *P_Chars="-,=W";

/*
 *  -    , , W, ,
 *    RRING
 */
int T_Digit = 5;
int T_Pause = 2;
int T_Wait  = 5;
int T_Conn  = 55;
int T_Rring = 35;
int T_Modem = 10;
int N_Retry = 2;

/*
 *   / 
 *
 * :
 *     ... 0
 *     \O (   )
 */
char **opts1;
/*
 *   
 */
char *Sound1;   /* Sound1 = */
char *Sound2;   /* Sound2 = */

/*    - R-, T-
 */
struct std_opts {
char *std_opt;
int *std_ptr; } std_tab[] =
{
"R",     &N_Retry,
"Retry", &N_Retry,
"T",    &T_Conn,
"Conn", &T_Conn,
"H",    &T_Rring,
"Rring",&T_Rring,
0, 0 };

/* ---------     ------------- */

/*
 *  -     
 */
char *B_Options;
int T_wait;    /*    */
int T_Onnum;   /* .     */

char *strchr(), *strrchr(), *strcpy(), *strdup(), *malloc();
long atol();
SIGTYPE alrmint();
SIGTYPE abort();

struct termios term, terms;              /* for storing tty parameters   */
int Debug = DBG;			/* set when debug flag is given	*/
int dialing;				/* set while modem is dialing	*/
int fd = -1;				/* file descriptor for acu	*/
int errflag = 0;			/* set on errors		*/
int hflag = 0;				/* set to hangup modem		*/
int zflag = 0;				/* set to setup modem		*/
int highbaud, lowbaud;			/* baud rate limits		*/
int retcode = RC_FAIL;			/* return code			*/
int timeout;				/* how long to wait for alarm	*/
int pid;				/* stores child's pid		*/
int c;					/* temporary storage		*/
char command[MAXLINE];			/* modem command buffer		*/
char *p;				/* temporary storage		*/
char *acu;				/* device to dial through	*/
char *phone;				/* phone number to dial		*/
extern int optind;			/* for getopt ()		*/
extern char *optarg;			/* for getopt ()		*/
jmp_buf Sjbuf;
jmp_buf Cjbuf;

#define	toprint(x)	((x)<' '?((x)+'@'):'?')

/*
 * Chat -   
 */
int inchat = 0; /*  -    Chat */
char **chat_script;

/* vgets -  ( )  c   f
 */

char *
vgets(c)
unsigned char c;
{
	static char buffer[10];
	char *pnt;

	pnt = buffer;
	if (iscntrl(c) || !isprint(c)) {
		if (!isascii(c)) {			/* Top bit is set */
			*pnt++ = 'M';
			*pnt++ = '-';
			c = toascii(c);			/* Strip it */
		}
		if (iscntrl(c)) {			/* Not printable */
			*pnt++ = '^';
			c = toprint(c);			/* Make it printable */
		}
	}
	*pnt++ = c;
	*pnt = '\0';
	return(buffer);
}


/*
 *   str     ttab, 
 *     -   -   ,  ..
 * string.
 */
static void
translate(ttab, str)
register char *ttab, *str;
{
	register char *s;

	for(;*ttab && *(ttab+1); ttab += 2)
		for(s=str;*s;s++)
			if(*ttab == *s)
				*s = *(ttab+1);
}


main (argc,argv)
	int argc;
	char *argv[];
{
	register char *p;
	char buf[256];
	int i;
	/*
	 *  Reenable all those signals we want to know about
	 */

	signal(SIGILL, SIG_DFL);
	signal(SIGIOT, SIG_DFL);
	signal(SIGEMT, SIG_DFL);
	signal(SIGFPE, SIG_DFL);
	signal(SIGBUS, SIG_DFL);
	signal(SIGSEGV, SIG_DFL);
	signal(SIGSYS,  SIG_DFL);
	signal(SIGINT,  abort);
	signal(SIGTERM, abort);
	setbuf(stderr, (char *)0);

	modem_type = argv[0];
	if ( (p = strrchr(argv[0], '/')) != 0 )
		argv[0] = modem_type = ++p;
	if ( strcmp(modem_type,"gdial") == 0 )
		modem_type = 0;
	/*
	 *   
	 *      -t type
	 */
	while ((c = getopt(argc, argv, "hzx:t:d:")) != EOF)
	{
		switch(c)
		{
			case 'h':
				hflag++;
				break;
			case 'z':
				zflag++;
				break;
			case 'd':
				DIALERS = strdup( optarg);
				break;
			case 't':
				modem_type = strdup( optarg);
				break;
			case 'x':
				Debug = atoi(optarg);
				break;
			case '?':
				errflag++;
				break;
		}
	}

	if (Debug)
	{
		fprintf (stderr, "dialer for `%s',  args ", modem_type);
		for (c=0; c<argc; c++)
		{
			fprintf (stderr, " %s", argv[c]);
		}
		fprintf(stderr, "\n");
	}
	if ( !modem_type )
	{
		fprintf(stderr, "Modem type must be specified\n");
		errflag++;
	}
	if ( !errflag && do_table(NULL, DIALERS, modem_type) < 0 )
	{
		fprintf(stderr, "Error in %s\n", DIALERS);
		exit(RC_FAIL | RCE_ARGS);
	}

	if (hflag )
	{
		if (argc - optind != 2)  errflag++ ;
	}
	else if ( zflag )
	{
		if ( argc - optind != 2  && argc - optind != 3 )  errflag++ ;
	}
	else
	{
		if (argc - optind < 3)  errflag++ ;
	}

	if (errflag)
	{
		char *typediag = strcmp(argv[0],"gdial") ? "" : " -t modemtype";
			fprintf (stderr,
			"Usage:\t%s%s [-d dialersfile] [-xN] -h devicename speed\n",
			argv[0], typediag);
			fprintf (stderr,
			"or:\t%s%s [-d dialersfile] [-xN] -z devicename\n\
\t\tspeed [:options]\n",
			argv[0], typediag);
			fprintf (stderr,
			"or:\t%s%s [-d dialersfile] [-xN] devicename\n\
\t\tphonenumber[:options] speed\n",
			argv[0], typediag);
		exit(RC_FAIL | RCE_ARGS);
	}

	acu = argv[optind++];

	if (!(hflag|zflag))
	{
		/*
		 *  
		 */
		phone = strdup(argv[optind++]);
		if ( p = strchr(phone,':') )
		{
			errflag = parseopts(p);
			*p = 0;
		} else errflag = parseopts(CNULL);
		/*      */
		T_Onnum = comp_time(phone);
		/*
		 *      
		 */
		translate(P_Chars,phone);
		/*
		 *         
		 */
		strcpy(buf,S_Valid);
		strcat(buf,P_Chars);
		if (strlen(phone) != strspn(phone, buf))
		{
			fprintf(stderr, "dial: Bad phone number %s\n", phone);
			exit(RC_FAIL | RCE_PHNO);
		}
	}

	lowbaud = highbaud = checkbaud ((unsigned)atol(argv[optind++]));
	DEBUG (6, "checkbaud claims low/high baud is 0%o\n", lowbaud);
	if ( zflag && argv[optind] )
	{
		errflag = parseopts(argv[optind]);
	}
	if ( errflag )
	{
		fprintf(stderr, "gdial: bad options\n");
		exit(RC_FAIL|RCE_PHNO);
	}
	/* 1.     */
	if (!hflag)
	{
		ungetty ("dialer");
	}

	/*
	 * Open the modem file.
	 */
	mdopen();

	/*
	 *  Timeout after 10 seconds if no response
	 */
	timeout = 10;
	signal(SIGALRM, alrmint);
	mdflush();

	/*
	 * Hangup and exit if requested
	 */
	if (hflag)
	{
		DEBUG(1, "Hangup sequence begun%c", '\n');
		i = do_sequence(S_Host);
		if ( i != OK && S_Reset != 0 )
			i = do_sequence(S_Reset);
		mdclose(0);
		cleanup (re_getty ("hangup") | i);
	}

	/*
	 *  Initialize the modem
	 */
	if (zflag)
	{
	    DEBUG(6, "Initializing modem at %s\n", acu);
	    i = do_sequence(S_Init);
	    if ( i != OK )
	    {
		fprintf(stderr,"Initialization FAILED\n");
		cleanup(RC_FAIL | i );
	    }
	    else
		fprintf(stderr,"Initialization OK\n");
	    mdclose(1);
	    cleanup (re_getty ("modem init") | (i?RC_FAIL:0) );
	}


	/*
	 *  
	 */
	if ( argv[optind] ) chat_script = &argv[optind];
call:

	i = do_sequence(S_Call);
	switch ( i )
	{
	case CONNECT:
		 /* Wait CD appear */
		 sleep(2);
		 retcode = 0;
		break;
	case BUSY:
	case NOANSWER:
	case NOCARRIER:
		 cleanup(RC_FAIL|RCE_BUSY);
	case ERROR:
	case FAIL:
		if ( N_Retry-- > 0 )
		{
			if ( S_Init && (i = do_sequence(S_Init)) != OK )
			{
				cleanup(RC_FAIL|RCE_NORESP);
			}
			goto call;
		}
	case NODIALTONE:
		cleanup(RC_FAIL|RCE_NOTONE);
	default:
		cleanup(RC_FAIL|RCE_TIMOUT);
	}
	mdclose(0);
	if ( retcode == 0 && chat_script)
	{
		inchat = 1;
		T_Modem = 15;
		retcode = do_sequence(chat_script);
		if ( retcode )
		{
			mdclose(1);
			cleanup(RC_FAIL|RCE_TIMOUT);
		}
	}
	cleanup(retcode);
}

/*
 *   , 
 *   
 */
comp_time(ph)
register char *ph;
{
    register int i = 0;
    register int c;
    int timeo = 2;
    while ( c = *ph++ )
        switch (c)
        {
        case 'T': 
            timeo = 1;  
            break;
        case 'P': 
	    timeo = T_Digit;
            break;
        case '-': 
	    i += T_Pause;
            break;
        case '=': 
	    i += T_Wait;
            break;
        default:  
            i += timeo; 
            break;
        }
    return(i);
}


do_sequence(flds)
register char *flds[];
{
	register char *want, *alt;
	char *send;
	int ko = OK;  static int inhup = 0;
	/*    ,  .   */
	if ( flds == 0 ) return (OK);
	while ( (send = *flds++) && ko == OK )
	{
		/*      */
		want = *flds++;
		alt = 0;
do_send:
		if ( want && (alt = strchr(want,'-')) )
		      *alt++ = 0;
		T_wait = ((want||inchat)? T_Modem: T_Conn);
		mdflush();
		DEBUG(50, "On `%s' Expect ", send);
		DEBUG(50, "`%s'\n", want);
		if ( strncmp(send,"\\H",2) == 0 && inhup == 0 )
		{
			inhup = 1;
			ko = do_sequence(S_Hang);
			inhup = 0;
			if ( ko != OK ) goto alt_ask;
			send += 2;
		}
		if ( mdwrite ( send ) != OK )
		{
			DEBUG(6,"do_sequence return = %d\n", FAIL);
			return(FAIL);
		}
		if ( want == 0 && inchat )
		{
			ko = OK;
			break;
		}
		if ( want == 0 )
			ko = mdread ( CNULL, T_wait);
		else
		if ( want[0] == 0)
			ko == OK;
		else
			ko = mdread ( want, T_wait);
		/*   ,    -  */
alt_ask:
		if ( ko != OK && ko != CONNECT && alt)
		{
			send = alt;
			/*     -  want*/
			if ( alt = strchr(alt,'-') )
			{
				*alt++ = 0;
				want = alt;
			}
			goto do_send;
		}
		if ( want == 0 ) break;
	}
	DEBUG(6,"do_sequence return = %d\n", ko);
	return(ko);
}

/*
 *      Write line to modem
 *
 *      return codes:  OK - , FAIL - 
 */
mdwrite(str)
register char *str;
{
    register char *strptr;
    int i, n, cr = 1;
    register char c;
    static int p_init = 0;
    DEBUG(6, "Sent MODEM %s", "<<");
    /* if ( !inchat) mdflush();         */
    uucpdelay(2,10);
    /*      -   */
    if ( setjmp( Cjbuf ) || setjmp(Sjbuf) )
    {
	DEBUG(6,">> FAIL!\n",0);
	alarm(0);
        return(FAIL);
    }
    if (!p_init) {
        p_init++;
	bld_partab(P_ZERO);
    }
    /* Set parity as needed */
    if (strcmp(str, "P_ZERO") == SAME) {
        bld_partab(P_ZERO);
        str += 6;
    } 
    else
        if (strcmp(str, "P_ONE") == SAME) {
            bld_partab(P_ONE);
            str += 5;
        } 
        else
            if (strcmp(str, "P_EVEN") == SAME) {
                bld_partab(P_EVEN);
                str += 6;
            } 
            else
                if (strcmp(str, "P_ODD") == SAME) {
                    bld_partab(P_ODD);
                    str += 5;
                }
    /* If "", just send '\r' */
    if (strcmp(str, "\"\"") == SAME) {
        p_chwrite('\r');
	DEBUG(6,">>\n", CNULL);
        return (OK);
    }
    strptr = str;
    while ((c = *strptr++) != '\0') {
        if (c == '\\') {
            switch(*strptr++) {
            case '\0':
		DEBUG(12, "TRAILING BACKSLASH IGNORED\n", CNULL);
                --strptr;
                continue;
	    case '\\':
		c = '\\';
		break;
            case 's':
		DEBUG(12,"BLANK\n", CNULL);
                c = ' ';
                break;
            case 'd':
		DEBUG(12, "DELAY\n", CNULL);
		sleep(1);
                continue;
            case 'n':
		DEBUG(12, "NEW LINE\n", CNULL);
                c = '\n';
                break;
            case 'r':
		DEBUG(12, "RETURN\n", CNULL);
                c = '\r';
                break;
            case 'B':
                if (isdigit(*strptr)) {
                    i = (*strptr++ - '0');
                    if (i <= 0 || i > 10)
                        i = 3;
                } 
                else
                    i = 3;
                /* send break */
                genbrk(i);
                if (*strptr == '\0')
                    cr = 0;
                continue;
	    case 'b':
		{
		    unsigned baud;
		    baud = 0;
		    i = 0;
		    while ( isdigit(*strptr) && i++ < 5)
			baud = baud * 10 + (*strptr++ - '0');
		    if ( baud <= 0 || mdsetbaud(baud) != SUCCESS )
			return(FAIL);
		}
                continue;
            case 'c':
                if (*strptr == '\0') {
		    DEBUG(12, "NO CR\n", CNULL);
                    cr = 0;
                } 
                else
                    DEBUG(5, "NO CR - IGNORED NOT EOL\n", CNULL);
                continue;
            case 'D':
                if (isdigit(*strptr)) {
                    i = (*strptr++ - '0');
                    if (i <= 0 || i > 10)
                        i = 2;
                } 
                else
                    i = 2;
                /* Clear DTR */
                clrdtr(i);
                if (*strptr == '\0')
                    cr = 0;
                continue;
            case 'w':
                i = 0;
		while ( isdigit(*strptr) )
                    i = i * 10 + (*strptr++ - '0');
                if ( i > 0 ) {
		    T_wait = i;
		    DEBUG(50,"\nWait time set to %d\n",i);
                }
                continue;
            case 'O':
		p_swrite( B_Options);
                continue;
            case 'T':
		p_swrite( phone);
		T_wait += T_Onnum;
                continue;
            default:
#define isoctal(x)  ((x >= '0') && (x <= '7'))
                if (isoctal(strptr[-1])) {
                    i = 0;
                    n = 0;
                    --strptr;
                    while (isoctal(*strptr) && ++n <= 3)
                        i = i * 8 + (*strptr++ - '0');
                    DEBUG(5, "\\%o\n", i);
                    p_chwrite( (char)i);
                    continue;
                }
		    else
			DEBUG(3,"Unrecognized: %s\n", strptr -1);
            }
        } /*    \ */
        p_chwrite( c);
    }
    if (cr)
        p_chwrite( '\r');
    DEBUG(6,">>\n", CNULL);
    return(OK);
}

/*
 *  mdread (expect, rtime)
 *    expect,   - OK, NOANSWER, FAIL
 *  expect ,    connect,
 *   A_codes
 *    
 *     ,  ,   \r
 *      Chat,     .
 */
mdread(expect,rtime)
char *expect;
int rtime;
{
    register char *bp;
    char buf[MAXLINE];
    register struct answers *pans;
    int i;
    char ch;
    struct answers *tans;
    struct answers tmpans[3];
    int rring;
    /* Expect -  ,    = A_Codes */
    dialing = 1;
    if ( !expect )
        tans = A_Codes;
    else
    {
	dialing = 0;
        tans = tmpans;
        tmpans[0].a_ans  = expect;
        tmpans[0].a_type = OK;
        tmpans[1].a_ans  = S_Ring;
        tmpans[1].a_type = RING;
        tmpans[2].a_ans = 0;
	if ( inchat ) tmpans[1].a_ans = 0;
    }
    if ( Debug >= 50 ) {
	for ( pans = tans; pans->a_ans; pans++ )
	{
	    DEBUG(50, "Wait `%s' ", pans->a_ans);
	    DEBUG(50, " = %d\n",  pans->a_type);
	}
    }
    /*    -  */
    if ( setjmp (Sjbuf ) )
    {
        alarm(0);
        return(NOANSWER);
    }
    /*    */
    bp = buf;
    DEBUG(6,"Time=%d ", rtime);
    alarm(rtime);
    rring = 0;
    DEBUG(6, "MODEM returned %s", "<<");
    /* .  \r -    */
    while (read(fd, &ch, 1) == 1) {
	c = ch&0177;
	if ( c == ' ') continue;
        if ((*bp = c) != '\0')
            *++bp = '\0';
        DEBUG(6, "%s", vgets(c));
        if (bp >= buf + MAXLINE) {
            alarm(0);
            DEBUG(4,">>-%s\n","FAIL");
            return(FAIL);
        }
        for ( pans = tans; pans->a_ans; pans++ )
        {
	    register char *sa;
	    sa = pans->a_ans;
	    if ( sa[0] == ' ' ) sa++;
	    else if ( !inchat && c != '\n' ) continue;
	    if ( substr(sa, buf) == 0 )
            {
                i = pans->a_type;
                if ( i == RRING && T_Rring && rring++ == 0)
                {
                    DEBUG(4,">>-%s\n","RRING");
		    DEBUG(6,"RRING: set timer to %d\n", T_Rring);
                    alarm(T_Rring);
                }
		bp = buf; *bp = 0;
                if ( i == RRING ) continue;
                alarm(0);
                DEBUG(6,">>- got %s\n", pans->a_ans);
                return(pans->a_type);
            }
        }
        /*   LF,   */
	if ( c == '\n' )
            bp = buf;
        continue;
    }
    alarm(0);
    DEBUG(6,">>-%s","FAIL");
    DEBUG(4, " no response\n", 0);

    return(FAIL);
}

/*  mdflush()
 *
 *  Function:	Flushes input clists for modem
 */
mdflush()
{
	DEBUG(12,"MDFLUSH\n",0);
	ioctl(fd, TIOCFLUSH, 0) ;
}

/*
 *  substr(s, l)
 *
 *  Function:	Checks for the presence of the string pointed to by s
 *		somewhere within the string pointed to by l.
 *
 *
 *  Returns:	0 if found.
 *		-1 if not found.
 */
substr(s, l)
	char *s;
	register char *l;
{
	int len;

	len = strlen(s);
	while ((l = strchr(l, *s)) != NULL) {
		if (strncmp(s, l, len) == SAME)
			return(0);
		l++;
	}
	return(-1);
}


/*
 *  alrmint()
 *
 *  Function:	Catches alarm calls (signal 14) and exits.
 *
 *  Returns:
 */

SIGTYPE alrmint()
{
	signal(SIGALRM, alrmint);
	DEBUG(4, "\nTimeout waiting for %s\n", dialing ? "carrier" : "acu");
	longjmp(Sjbuf, 2);
}


/*
 *  cleanup(stat)
 *
 *  Function:	Closes device file and exits.
 *
 *  Returns:	No return.  Exits with status stat.
 */

cleanup (stat)
	int stat;
{
	if (stat & RC_FAIL)	/* if we failed, drop DTR (in abort) */
	{
		retcode = stat;
		abort(0);
	}
	else
	{		/* else, return */
		DEBUG(9,"gdial returns: %o\n", stat);
		exit(stat);
	}
}

/*
 * Exit, making sure the modem hangs up and we
 * don't leave the tty ungetty'ed.
 */
static int abcount;
SIGTYPE
abort (sig)
int sig;
{
	int error = retcode & ~(RC_FAIL | RC_ENABLED);

	signal(SIGINT, SIG_IGN);
	if (sig)
		signal(sig, SIG_IGN);

	if ( error != RCE_HANGUP && error != RCE_IOCTL && !abcount++ )
	{
		if ( fd < 0 || do_sequence (S_Hang) != OK)
		{
			retcode |= (RC_FAIL);
		}
		else
		{
		       /*
		 	* No need to call dial -h again.
		 	*/
			retcode &= ~RC_ENABLED;
		}
	}

	if (re_getty ("abort") == RC_FAIL)
	{
		retcode |= RC_FAIL;
	}

	if ( abcount++ < 2 )
	{
		mdclose (1);
	}

	if (sig)  retcode |= (RC_FAIL | RCE_SIG);
	DEBUG(9,"gdial returns: %o\n", retcode);
	exit (retcode);
}

/*
 *  checkbaud(n)
 *
 *  Function:	Check for valid baud rates
 *
 *  Returns:	The baud rate in struct termio c_cflag fashion
 *
 */

checkbaud(n)
unsigned n;
{
	int baudrate;

	switch(n) {
# ifdef B57600
		case 57600:
			baudrate = B57600;
			break;
# endif
		case 38400:
			baudrate = B38400;
			break;
		case 19200:
			baudrate = B19200;
			break;
		case 9600:
			baudrate = B9600;
			break;
		case 4800:
			baudrate = B4800;
			break;
		case 2400:
			baudrate = B2400;
			break;
		case 1800:
			baudrate = B1800;
			break;
		case 1200:
			baudrate = B1200;
			break;
		case 300:
			baudrate = B300;
			break;

		default:
			fprintf(stderr, "dial: Bad speed: %u\n", n);
			exit(RC_FAIL | RCE_SPEED);
	}		
	return(baudrate);
}

/*
 * ===================== TTY CONTROL =============================
 * Open the modem and set the baud rate.  Sets the global variable
 * "fd".
 */
int
mdopen ()
{
	/*
	 *  Must open with O_NDELAY set or the open may hang.
	 */
	DEBUG (8, "mdopen(): opening %s\n", acu);
	if ((fd = open(acu, O_RDWR | O_NDELAY)) < 0)
	{
		fprintf(stderr, "dial: Can't open device: %s\n", acu);
		exit(RC_FAIL | RCE_OPEN | retcode);
	}
	/*
	 * set line for no echo and correct speed.
	 * If hanging up, issue commands at high baud to enable auto answer
	 * at all speeds.
	 */
	signal(SIGINT, abort);
	errflag = ioctl(fd, TIOCGETA, &term);
	terms = term;
/*        term.c_cflag &= ~(CBAUD | HUPCL);     */
	term.c_cflag = (CLOCAL | CS8 | CREAD /* | CTSFLOW | RTSFLOW */);
	term.c_ispeed = term.c_ospeed = highbaud;
	if (hflag)
	    term.c_cflag |= HUPCL;
	term.c_iflag  = 0;
	term.c_oflag  = 0;
	term.c_lflag &= ~(ECHO|ICANON|ISIG);
	term.c_cc[VMIN] =  1;
	term.c_cc[VTIME] = 1;
	errflag = ioctl(fd, TIOCSETAW, &term);
	if (errflag)
	{
		char buf[16];
		DEBUG(1, "dial: ioctl error on %s", acu);
		DEBUG(1, " errno=%d\n", errno);
		cleanup(RC_FAIL | RCE_IOCTL | retcode);
	}
	DEBUG(6,"set ioctl on %s, ", acu);
	DEBUG(6,"baudrate = 0%o\n", term.c_ospeed);
	/*
	 *  Reopen line with clocal so we can talk without carrier present
	 */
	c = fd;
	if ((fd = open(acu, O_RDWR)) < 0)
	{
		fprintf(stderr, "dial: Can't open device local: %s\n", acu);
		exit(RC_FAIL | RCE_OPEN | retcode);
	}
	close(c);
	return (fd);
}

/*
 *   -
 * mdsetbaud(rate) unsigned rate;
 */
mdsetbaud(baud)
unsigned baud;
{
	highbaud = checkbaud(baud);
	term.c_ospeed = term.c_ispeed = highbaud;
	if (ioctl(fd, TIOCSETAW, &term))
	{
		char buf[16];
		DEBUG(1, "dial: ioctl error on %s", acu);
		DEBUG(1, " errno=%d\n", errno);
		cleanup(RC_FAIL | RCE_IOCTL | retcode);
	}
	DEBUG(6,"set baud rate on on %s, ", acu);
	DEBUG(6,"baudrate = 0%o\n", term.c_ospeed);
	return(SUCCESS);
}

mdclose (old)
{
	if ( fd < 0 ) return;
	if ( old ) term = terms;
/*	else
		term.c_cflag &= ~CLOCAL;
*/	term.c_cflag |= HUPCL;
	DEBUG (8, "mdclose(): closing %s ", acu);
	DEBUG (8, " old=%d", old);
	DEBUG (8, " iflag=%o", term.c_iflag);
	DEBUG (8, " VMIN=%d", term.c_cc[VMIN]);
	DEBUG (8, " VTIME=%d", term.c_cc[VTIME]);
	DEBUG (8, " CLOCAL=%o\n", term.c_cflag&CLOCAL);
	sleep(1); /*       */
# ifdef i386
	sleep(1);
# endif
	errflag = ioctl(fd, TIOCSETAW, &term);
	if (errflag)
	{
		char buf[16];
		DEBUG(1, "dial: ioctl error on %s", acu);
		DEBUG(1, " errno=%d\n", errno);
		cleanup(RC_FAIL | RCE_IOCTL | retcode);
	}
}

/*
 *  DTR  time 
 */
clrdtr(time)
int time;
{
	int t;
	int i;
# ifndef TIOCCDTR
	struct termios nterm;
/*
 *  -   ,    .   (Sjbuf)
 */
	nterm = term;
	nterm.c_cflag &= ~CBAUD;
	DEBUG(5,"DTR CLEAR..", CNULL);
	t = alarm(5);
	i = ioctl(fd, TIOCSETAW, &nterm);
	if ( i < 0 )
	{
		char buf[16];
		DEBUG(1, "dial: ioctl error on %s", acu);
		DEBUG(1, " errno=%d\n", errno);
		ioctl(fd, TIOCSETAW, &term);
		cleanup(RC_FAIL | RCE_IOCTL | retcode);
	}
	alarm(0);
# ifndef ultrix
	sleep(time+1);
# else
	sleep(time);
# endif
	alarm(5);
	i = ioctl(fd, TIOCSETAW, &term);
	if ( i < 0 )
	{
		char buf[16];
		DEBUG(1, "dial: ioctl error on %s", acu);
		DEBUG(1, " errno=%d\n", errno);
		cleanup(RC_FAIL | RCE_IOCTL | retcode);
	}
	sleep(1);
	DEBUG(5,".\n", CNULL);
	alarm(t);
# else /*  */
/*
 *  -   ,    .   (Sjbuf)
 */
	DEBUG(5,"DTR CLEAR..", CNULL);
	t = alarm(5);
	i = ioctl(fd, TIOCCDTR, 0);
	if ( i < 0 )
	{
		char buf[16];
		DEBUG(1, "dial: ioctl error on %s", acu);
		DEBUG(1, " errno=%d\n", errno);
		ioctl(fd, TIOCSDTR, 0);
		cleanup(RC_FAIL | RCE_IOCTL | retcode);
	}
	alarm(0);
	sleep(time);
	alarm(5);
	i = ioctl(fd, TIOCSDTR, 0);
	if ( i < 0 )
	{
		char buf[16];
		DEBUG(1, "dial: ioctl error on %s", acu);
		DEBUG(1, " errno=%d\n", errno);
		cleanup(RC_FAIL | RCE_IOCTL | retcode);
	}
	sleep(1);
	DEBUG(5,".\n", CNULL);
	alarm(t);
# endif /*  */
}

/*
 *  BREAK  time 
 */
genbrk(time)
int time;
{
int t;
t = alarm(2);
if ( ioctl(fd,TIOCSBRK,0) < 0)
	DEBUG(5, "break TCSBRK err=%d\n", errno);
alarm(t);
}

/*
 * Call ungetty to send a signal to the getty on a line.
 *
 * returns:
 * RC_FAIL - failure
 * SUCCESS - success
 */
ungetty (string)
	char *string;
{
#ifndef HDUU
	if ((pid = fork()) == 0)
	{
		DEBUG (7, "%s: ungetty", string);
		DEBUG (7, " %s called\n", acu);
		execl (UNGETTY, "ungetty", acu, NULL);
		exit (-1);
	}
	while (((c = wait(&errflag)) != pid) && c != -1) ;

	switch ((errflag >> 8) & 0xff)
	{
		case UG_NOTENAB:	/* line acquired: not enabled */
			retcode = SUCCESS;
			break;
		case UG_ENAB:	/* line acquired: need ungetty -r when done */
			retcode = RC_ENABLED;
			break;
		case UG_FAIL:		/* could not acquire line */
			DEBUG (1, "%s: ungetty failed:", string);
			DEBUG (1, " %x", UG_FAIL);
			exit (RC_FAIL | RCE_INUSE);
		case 255:
			exit (RC_FAIL);
	}
#else /*  */
	retcode = SUCCESS;	/* uugetty does not require ungetty */
#endif /*  */
}

/*
 * Restore the getty on the acu.
 *
 * returns:
 * RC_FAIL - failure
 * SUCCESS - success
 */

re_getty (string)
	char *string;
{
#ifndef HDUU
	/* call ungetty to see if we need to switch to dialin */
	if ((pid = fork()) == 0)
	{
		DEBUG (7, "%s: ungetty -t", string);
		DEBUG (7, " %s called\n", acu);
		execl (UNGETTY, "ungetty", "-t", acu, NULL);
		exit(-1);
	}
	while (((c = wait(&errflag)) != pid) && c != -1) ;

	if (((errflag>>8) & 0xff) != UG_RESTART)
	{
		DEBUG (7, "%s: no ungetty -r needed\n", string);
		return (SUCCESS);
	}

	DEBUG (7, "%s: ungetty -r needed\n", string);
	if ((pid = fork()) == 0)
	{
		DEBUG (7, "%s: ungetty -r", string);
		DEBUG (7, " %s called\n", acu);
		execl (UNGETTY, "ungetty", "-r", acu, NULL);
		exit (-1);
	}
	while (((c = wait(&errflag)) != pid) && c != -1) ;

	if (((errflag >> 8) & 0xff) == UG_FAIL)
	{
		return (RC_FAIL);
	}
	else
#endif /*  */
	{
		return (SUCCESS);
	}
}

/* #ifndef SCO_UNIX
static char *
strdup(str)
register char *str;
{
	register char *s;

	if ((s = (char *) malloc(strlen(str)+1)))
		strcpy (s, str);
	return(s);
}
# endif /* SCO_UNIX */


p_swrite(s)
register char *s;
{
	if ( s )
		while ( *s)
			p_chwrite(*s++);
}

p_chwrite( c)
char c;
{
	int i;
	i = alarm(4);
	DEBUG(6, "%s", vgets(c));
	c = par_tab[c&0177];
	if (write(fd, &c, 1) != 1) {
	       fprintf (stderr,
		       "Write error! \n");
	       alarm(0);
	       longjmp(Cjbuf, 2);
	}
	alarm(i);
}

/*
 * generate parity table for use by p_chwrite.
 */
bld_partab(type)
int type;
{
	register int i, j, n;

	for (i = 0; i < sizeof(par_tab); i++) {
		n = 0;
		for (j = i&(sizeof(par_tab)-1); j; j = (j-1)&j)
			n++;
		par_tab[i] = i;
		if (type == P_ONE
		 || (type == P_EVEN && (n&01) != 0)
		 || (type == P_ODD && (n&01) == 0))
			par_tab[i] |= sizeof(par_tab);
	}
}


/*
 * parseopts(s) -    s
 * sss:sss:SS=dd:... -  
 *   B_Options   ,   
 * std_tab.
 *  - 0 - OK, -1 - Fail
 */
parseopts(p)
register char *p;
{
    char *pn; /* .  */
    char *pv; /*   = */
    int ko = 0;
    char optline[OPTLEN], *pol = optline, *poe = optline + OPTLEN;
    pn = p;
    if ( Debug > 4  && !zflag )
    {
	if ( Debug > 8 && Sound2 )
        {
            strcpy(pol, Sound2); 
            pol += strlen(Sound2);
        } 
        else if ( Sound1 )
        {
            strcpy(pol, Sound1); 
            pol += strlen(Sound2);
        }
    }
    if ( pn ) DEBUG(9,"Parseopts (%s)\n", pn);
    while ( pn && *pn )
    {
	while ( *pn == ':' )
		pn++;
        p = pn;
        pn = strchr(p, ':');
        if ( pn )
            *pn++ = 0;
        if ( pv = strchr(p, '=') )
        {
            register struct std_opts *ps;
            int i;
            *pv ++ = 0;
            i = 0;
            while ( *pv >= '0' && *pv <= '9' )
            {
                i += (i*10) + (*pv - '0');
                pv++;
            }
            if ( *pv != 0 )
            {
                fprintf(stderr,"Bad value in %s=\n",p);
                ko = -1;
                continue;
            }
            for( ps = std_tab; ps->std_opt;ps++ )
                if ( strcmp(p, ps->std_opt) == 0 ) break;
            if ( ps->std_ptr == 0 )
            {
                fprintf(stderr,"Bad name in %s=\n",p);
                ko = -1;
                continue;
            }
            *(ps->std_ptr) = i;
            DEBUG(5,"Set %s to",p);
            DEBUG(5," %d\n", i);
            continue;
        }
        /* Value - option */
        {
            register char **pl;
            int i;
            for ( pl = opts1; pl[0] && pl[1]; pl += 2)
		if ( strcmp(p, *pl ) == 0 )
                {
                    if ( (pol + (i = strlen(pl[1])) ) < poe )
                    {
                        strcpy(pol, pl[1]);
                        pol += i;
                        goto e_opt;
                    }
                    if ( ko == 0 )
                        fprintf(stderr,"Too many options on %s\n", p);
                    ko = -1;
                    goto e_opt;
                }
            fprintf(stderr,"Not found option %s\n", p);
            ko = -1;
e_opt:
            continue;
        }
    }
    /* Finish - dup line */
    if ( ko != 0 )
        return(ko);
    *pol = 0;
    if ( pol != optline )
    {
        B_Options = strdup(optline);
        DEBUG(5,"Options string set to `%s'\n",B_Options);
    }
    return(ko);
}

/*
 *    
 */
# define TN_ANSWER 0040
# define TN_MASKA  0017
# define TN_STR    0001
# define TN_LIST   0002
# define TN_INT    0003
# define TN_LISTI  0004
# define TN_NAME   0005
# define TN_ALTER  0006
# define TN_FLAG   0007
# define TN_LISTU  0010

struct tn_info {
char *tn_name;
char **tn_ptr;
int  tn_type; } tn_tab[] =
{
"Modem",   0,       TN_NAME,
"Use",     0,       TN_ALTER,
"Hangup"  , (char **)&S_Hang, TN_LIST,
"Save",     (char **)&S_Init, TN_LIST,
"Call",     (char **)&S_Call, TN_LIST,
"Host",     (char **)&S_Host, TN_LIST,
"Reset",    (char **)&S_Reset,TN_LIST,
"Valid",    (char **)&S_Valid,TN_STR,
"Baudrate", (char **)&S_Baud, TN_LISTU,
"Connect",     0,    TN_ANSWER|CONNECT,
"Busy",        0,    TN_ANSWER|BUSY,
"Nocarrier",   0,    TN_ANSWER|NOANSWER,
"Error",       0,    TN_ANSWER|ERROR,
"Nodial",      0,    TN_ANSWER|NODIALTONE,
"Rring",       0,    TN_ANSWER|RRING,
"Ring",        0,    TN_ANSWER|RING,
"Chars",    (char **)&P_Chars,TN_STR,
"Tconnect", (char **)&T_Conn, TN_INT,
"Trring",   (char **)&T_Rring,TN_INT,
"Tmodem",   (char **)&T_Modem,TN_INT,
"Tdigit",   (char **)&T_Digit,TN_INT,
"Tpause",   (char **)&T_Pause,TN_INT,
"Twait",    (char **)&T_Wait, TN_INT,
"X5mode",   (char **)&Sound1, TN_STR,
"X9mode",   (char **)&Sound2, TN_STR,
"Options",  (char **)&opts1,   TN_LIST,
0,          0,       0 };
/*
 *    
 * #   , 
 *      ,      2- 
 * Modem= Use=__
 * = 
 * =___ (  -  )
 *      \,    ,  . 
 *   
 *   -   
 *     ,     
 */
#define MAX_LINE 1024
#define MAX_AV 128
do_table(fd,file, name)
FILE *fd;
char *file;
char *name;
{
    long old_seek;
    long ftell(); int line_break();int ko = 0;
    char line[MAX_LINE];
    char *avline[MAX_AV+1];
    char **pav;
    struct tn_info *line_anl(), *tnptr;
    char **npav;
    char **plist;
    int ival, type;
    register char *pc;
    int *pilist;
    /*
         * 1.  
         */
    if ( fd ) {
        old_seek = ftell(fd);
        fseek(fd,0l, 0);
	DEBUG(99,"Set L.dialers to 0 from %ld\n", old_seek);
    } 
    else
    {
        old_seek = 0;
        fd = fopen(file,"r");
        if ( !fd ) {
            perror(file);
            exit(-1);
        }
    }
    /*
     * 3.   (   , 
     *      \  ,   ..
     */
    while (line_get(fd, line) )
    {
	if ( line[0] == 0 ) continue;
	DEBUG(99,"In: %s\n",line);
	if ( line_break(line, avline) < 0 ) continue;
	pav = avline;
	DEBUG(99,"1-th w=%s \n",*pav);
	if ( pav && (tnptr = line_anl( pav )) && tnptr->tn_type == TN_NAME
	    && strcmp(*pav++,  name) == 0 )
            goto do_read;
        /*
	 * 4.      ,  .   ' \n\0',
	 *     ,     ,   (3)
	 */
	 DEBUG(99,"Skip to next\n",0);
	 while (line_get(fd, line) && line[0] != '.' && line[1] != 0 );
    }
    ko = -1;
    fprintf(stderr,"Modem `%s' not found in `%s'\n", name, file);
    goto retn;
do_read:
    /*
     * 5.   ,  
     */
    DEBUG(99,"Found modem type %s\n",name);
    while ( 1 )
    {
	DEBUG(99,"L in: %s\n",line);
        /*
	 * 1.    .
	 *      ""  ''   
	 */
	if ( !*pav )
	{
		/*   */
		if ( line_get(fd, line) == 0 || ( line[0]=='.' && line[1]==0))
			break; /*   */
		if ( line_break(line, avline) < 0  )
		{
			fprintf(stderr, "Syntax error in line: %s...\n", line);
			ko = -1;
			break;
		}
		pav = avline;
	}
        /*
	 * 2)   
	 */
        while ( *pav )
        {
	    DEBUG(99,"Arg[0] = %s \n", *pav);
            /*
	     * 3)  = ,  ,     =,  
	     * 4)   
	     * 5)   , 
	     *    *pav   
	     *    
	     */
	    {
	    char *key;
		key = *pav;
		tnptr = line_anl( pav );
		/*
		 *    -   ,    
		 */
		if ( !tnptr )
		{
		    DEBUG(1,"Unknown keyword %s\n", key);
		    *pav = 0; /* .  */
		    break;
		}
	    }
            /*
	     *  .  
	     */
	    DEBUG(99,"Found: %s ",tnptr->tn_name);
	    DEBUG(99,"Type = %o\n",tnptr->tn_type);
            type = tnptr->tn_type;
            /*
	     *   ,        
	     */
            if ( type&TN_ANSWER)
                {
                while ( *pav )
		    if ( add_kod(*pav++,type&TN_MASKA) < 0 ) ko = -1;
                continue;
            }
            /*
	     *  
	     */
            switch ( type ) {
            case TN_STR: 
                *(tnptr->tn_ptr) = strdup(*pav++);
		DEBUG(99, "Set *ptr to %s\n", *(tnptr->tn_ptr));
                break;
            case TN_LIST:
                npav = pav;
                while ( *npav ) npav++;
		plist = (char **)malloc((sizeof(char *)) * (npav - pav + 1));
                if ( !plist )   goto no_mem;
		*((char ***)(tnptr->tn_ptr)) = plist;
                while ( *pav )
		{
		    *plist++ = strdup(*pav++);
		    DEBUG(99, "Set to `%s'\n", plist[-1]);
		}
                *plist = 0;
                break;
            case TN_INT:
                if ( a_toi(*pav++, &ival) == -1 ) ko = -1;
                *((int *)(tnptr->tn_ptr)) = ival;
		DEBUG(99, "Val = %d\n", ival);
                break;
            case TN_LISTI:
                npav = pav;
                while ( *npav ) npav++;
		pilist = (int *)malloc((sizeof(int)) * (npav - pav + 1));
                if ( !pilist ) goto no_mem;
                *((int **)(tnptr->tn_ptr)) = pilist;
                while ( *pav )
		{
                    if ( a_toi(*pav++, pilist++) < 0 ) ko = -1;
		    DEBUG(99, "Set to `%d'\n", pilist[-1]);
		}
		*pilist = 0;
                break;
	    case TN_LISTU:
		/* Assume that sizeof(unsigned) == sizeof(int) */
                npav = pav;
                while ( *npav ) npav++;
		pilist = (int *)malloc((sizeof(unsigned)) * (npav - pav + 1));
                if ( !pilist ) goto no_mem;
                *((int **)(tnptr->tn_ptr)) = pilist;
                while ( *pav )
		{
		    if ( a_tou(*pav++, pilist++) < 0 ) ko = -1;
		    DEBUG(99, "Set to `%u'\n", pilist[-1]);
		}
		*pilist = 0;
                break;
            case TN_ALTER:
		ko = do_table(fd,file, *pav++);
                    break;
            case TN_FLAG:
                *((int *)(tnptr->tn_ptr)) = 1;
		DEBUG(99, "Set flag = 1\n", 0);
                pav++;
                break;
            default:
		fprintf(stderr, "Unknown type of field %o\n", type);
                ko = -1;
                break;
            }
        }
    }
retn:
    if ( old_seek ) fseek(fd, old_seek, 0);
    else fclose(fd);
    return(ko);
no_mem:
    fprintf(stderr,"No memory\n");
    ko = -1;
    goto retn;
}

/*
 * add_kod(s,cod) -      
 *  - <0 - 
 */
add_kod(s,cod)
char *s;
int cod;
{
    register struct answers *pCodes;
    for(pCodes = A_Codes;pCodes < A_Codes + MAX_ANSWERS; pCodes++)
    {
        if ( pCodes->a_ans == 0 )
        {
            s = strdup(s);
            pCodes->a_ans = s;
            pCodes->a_type = cod;
	    DEBUG(99, "Add Answer[%d]", pCodes - A_Codes);
	    DEBUG(99, " `%s'", s);
	    DEBUG(99, " kod=%d\n", cod);
            if ( cod == RING )
	    {
                S_Ring = s;
		DEBUG(99, "S_Ring=`%s'\n", s);
	    }
            return(0);
        }
    }
    fprintf(stderr,"Too many Answers, Max=%d\n", MAX_ANSWERS);
    return(-1);
}

/*
 * a_toi(s,pi)      
 *   ,  - -1,  - 0
 */
int a_toi(pc,pi)
register char *pc;
int *pi;
{
	char *pc0 = pc;
	int ival = 0, is = 1;
	if ( *pc == '-' )
	{ 
		is = -1;
		pc++;
	}
	else if ( *pc == '+' ) pc++;
	while ( *pc && *pc >= '0' && *pc <= '9' )
		ival = (ival*10) + (*pc++ -'0');
	*pi = ival * is;
	if ( *pc != 0 )
	{
		fprintf(stderr,"Bad integer %s\n",pc0);
		return(-1);
	}
	return(0);
}

/*
 * a_tou(s,pi)      unsigned
 *   ,  - -1,  - 0
 */
int a_tou(pc,pi)
register char *pc;
unsigned *pi;
{
	char *pc0 = pc;
	unsigned ival = 0;

	if ( *pc == '+' ) pc++;
	while ( *pc && *pc >= '0' && *pc <= '9' )
		ival = (ival*10) + (*pc++ -'0');
	*pi = ival;
	if ( *pc != 0 )
	{
		fprintf(stderr,"Bad unsigned integer %s\n",pc0);
		return(-1);
	}
	return(0);
}

/*
 * line_get(fd, line[MAX_LINE] )
 *  
 *  
 *  - 1 - OK
 *         0 -  
 */
# define L_SKIP while ( (c = getc(fd)) > 0 && c !='\n')
line_get(fd, line)
FILE *fd;
char *line;
{
    register char *s;
    register int c;
    char *se;
    int escape = 0;
    s = line;
    se = s + MAX_LINE-2;
    /* 1.   ,    */
    while ( (c = getc(fd)) >= 0 )
    {
        if ( c == '#' )
        {
	    L_SKIP;
            continue;
        }
        if ( c == '\n' )
        {
            *s = 0;
            return(1);
        }
	break;
    }
    if ( c < 0 ) return(0);
    escape = -1;
    /* 2.  .      */
    while ( escape < 0 || (c = getc(fd)) >= 0)
    {
	/*     -   .   1- */
	if ( escape < 0 ) escape = 0;
        if ( c != '\n' ) {
            *s++ = c;
            if ( s >= se ) {
                *s = 0;
		fprintf(stderr, "Line too long: %.20s...\n", line);
                while ( (c = getc(fd)) >= 0 && c != '\n' );
                return(1);
            }
            escape = (c == '\\');
            continue;
        }
        if ( escape ) {
            escape = 0;
	    s--;
	    while ( (c = getc(fd)) >= 0 && c == '#' )
		L_SKIP;
	    if ( c >= 0 ) escape = -1;
            continue;
        }
        *s = 0;
        return(1);
    }
    if ( s != line )
    {
        *s = 0;
        return(1);
    }
    return(0);
}

/*
 * line_break(line, args)
 *        list
 * args[] = 0
 *  -  , -1 - 
 */
line_break(cp, execargs)
register char *cp;
char *execargs[];
{
    register char **e = execargs;
    register char *wp;
    while (*cp == ' ' ) cp++;
    wp = cp;
    while (*cp != 0) {
	*e++ = wp;              /*   */
	if ((e-execargs) >= MAX_AV) goto noargerr; /*   */
	/*     */
	while ( *cp && *cp != ' ' && *cp != '\t' )
	{
	    if (*cp == '"') {
		cp++;
		while (*cp !=  '"')
		    if (*cp == 0) goto noargerr;
		    else *wp++ = *cp++;
		cp++;
		continue;
	    };
	    if (*cp == '\'') {
		cp++;
		while (*cp !=  '\'')
		    if (*cp == 0) goto noargerr;
		    else *wp++ = *cp++;
		cp++;
		continue;
	    };
	    while (*cp != '"' && *cp != '\'' &&
		   *cp != ' ' && *cp != '\t' && *cp ) *wp++ = *cp++;
	}
	while ( *cp == '\t' || *cp == ' ' ) *cp++ = 0;
	*wp++ = 0;
    }
    *e = 0;
    return(e - execargs);
noargerr:
    return(-1);
}

/*
 * line_anl(av)
 * char **av
 *     
 *  -    
 *   *av     ( 
 *  ,     '='
 */
struct tn_info *line_anl(av)
char **av;
{
    register char *p;
    int lk;
    register struct tn_info *pt;
    /* 1.  p   */
    p = *av;
    /* 2.  '=' */
    *av = strchr(p, '=');
    /* 3.   - ,   *av   . 
     *        -  *av   */
    if ( !*av ) *av = p + strlen(p);
    else
    {
        **av = 0;
        (*av)++;
    }
    /* 4.   ,      
     *   (  1  -  ) */
    lk = strlen(p);
    if ( lk <= 1 ) return(0);
    for ( pt = tn_tab; pt->tn_name; pt++ )
    {
        if ( strncmp(pt->tn_name, p, lk) == 0 )
        {
            /* 5.   */
            return(pt);
        }
    }
    /* 6.    -  0 */
    return(0);
}

# ifndef WITH_UUCP
/*
 * intervaldelay:  delay execution for numerator/denominator seconds.
 */

#ifdef INTERVALTIMER
#include <sys/time.h>
intervaldelay(num,denom)
int num, denom;
{
	struct timeval tv;
	tv.tv_sec = num / denom;
	tv.tv_usec = (num * 1000000L / denom ) % 1000000L;
	(void) select (0, (int *)0, (int *)0, (int *)0, &tv);
}
#endif  /* INTERVALTIMER */


#ifdef FTIME_INTR
# include <sys/timeb.h>
ftimedelay(n)
{
	static struct timeb loctime;
	register i = loctime.millitm;

	ftime(&loctime);
	while (abs((int)(loctime.millitm - i))<n) ftime(&loctime)
		;
}
#endif  /* FTIME_INTR */

# endif
