/*-
 * Copyright (c) 1993, 1994 RELCOM Corp. Igor Chechik.
 */

/*
 * Copyright (c) 1993, 1994 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: if_sl.c,v 1.18 1994/01/29 05:45:44 karels Exp $
 */

/*
 * Copyright (c) 1987, 1989, 1993 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)if_sl.c	7.22 (Berkeley) 4/20/91
 */

/*
 * Serial Line interface
 *
 * Rick Adams
 * Center for Seismic Studies
 * 1300 N 17th Street, Suite 1450
 * Arlington, Virginia 22209
 * (703)276-7900
 * rick@seismo.ARPA
 * seismo!rick
 *
 * Pounded on heavily by Chris Torek (chris@mimsy.umd.edu, umcp-cs!chris).
 * N.B.: this belongs in netinet, not net, the way it stands now.
 * Should have a link-layer type designation, but wouldn't be
 * backwards-compatible.
 *
 * Converted to 4.3BSD Beta by Chris Torek.
 * Other changes made at Berkeley, based in part on code by Kirk Smith.
 * W. Jolitz added slip abort.
 *
 * Hacked almost beyond recognition by Van Jacobson (van@helios.ee.lbl.gov).
 * Added priority queuing for "interactive" traffic; hooks for TCP
 * header compression; ICMP filtering (at 2400 baud, some cretin
 * pinging you can use up all your bandwidth).  Made low clist behavior
 * more robust and slightly less likely to hang serial line.
 * Sped up a bunch of things.
 * 
 * Note that splimp() is used throughout to block both (tty) input
 * interrupts and network activity; thus, splimp must be >= spltty.
 */

/* #define BSD_1_0      /* BSD/386 version */

/* from if_sl.c,v 1.7 89/05/31 02:24:52 van Exp */
/* from if_sl.c,v 1.11 84/10/04 12:54:47 rick Exp */

#ifdef BSD_1_0
# include "sl.h"
#else   /* for BSD_1_1 */
# define NSL 1
#endif

#if NSL > 0

#include "param.h"
#include "proc.h"
#include "mbuf.h"
#include "buf.h"
#include "socket.h"
#include "ioctl.h"
#include "file.h"
#include "tty.h"
#include "kernel.h"
#include "conf.h"

#include "if.h"
#include "if_types.h"
#include "netisr.h"
#include "route.h"
#if INET
#include "netinet/in.h"
#include "netinet/in_systm.h"
#include "netinet/in_var.h"
#include "netinet/ip.h"
#else
Huh? Slip without inet?
#endif

#include "machine/mtpr.h"

#include "slcompress.h"
#include "if_slvar.h"

#include "bpfilter.h"
#if NBPFILTER > 0
#include <sys/time.h>
#include <net/bpf.h>
#endif

#include "syslog.h"

/*
 * SLMAX is a hard limit on input packet size.  To simplify the code
 * and improve performance, we require that packets fit in an mbuf
 * cluster, and if we get a compressed packet, there's enough extra
 * room to expand the header into a max length tcp/ip header (128
 * bytes).  If we configure up BPF, it wants an additional 16 bytes
 * for the largest compressed header plus a direction byte.
 * So, SLMAX can be at most
 *	MCLBYTES - 128 - SLIP_HDRLEN
 *
 * SLMTU is a hard limit on output packet size.  To insure good
 * interactive response, SLMTU wants to be the smallest size that
 * amortizes the header cost.  (Remember that even with
 * type-of-service queuing, we have to wait for any in-progress
 * packet to finish.  I.e., we wait, on the average, 1/2 * mtu /
 * cps, where cps is the line speed in characters per second.
 * E.g., 533ms wait for a 1024 byte MTU on a 9600 baud line.  The
 * average compressed header size is 6-8 bytes so any MTU > 90
 * bytes will give us 90% of the line bandwidth.  A 100ms wait is
 * tolerable (500ms is not), so want an MTU around 296.  (Since TCP
 * will send 256 byte segments (to allow for 40 byte headers), the
 * typical packet size on the wire will be around 260 bytes).  In
 * 4.3tahoe+ systems, we can set an MTU in a route so we do that &
 * leave the interface MTU relatively high (so we don't IP fragment
 * when acting as a gateway to someone using a stupid MTU).
 *
 * Similar considerations apply to SLIP_HIWAT:  It's the amount of
 * data that will be queued 'downstream' of us (i.e., in clists
 * waiting to be picked up by the tty output interrupt).  If we
 * queue a lot of data downstream, it's immune to our t.o.s. queuing.
 * E.g., if SLIP_HIWAT is 1024, the interactive traffic in mixed
 * telnet/ftp will see a 1 sec wait, independent of the mtu (the
 * wait is dependent on the ftp window size but that's typically
 * 1k - 4k).  So, we want SLIP_HIWAT just big enough to amortize
 * the cost (in idle time on the wire) of the tty driver running
 * off the end of its clists & having to call back slstart for a
 * new packet.  For a tty interface with any buffering at all, this
 * cost will be zero.  Even with a totally brain dead interface (like
 * the one on a typical workstation), the cost will be <= 1 character
 * time.  So, setting SLIP_HIWAT to ~100 guarantees that we'll lose
 * at most 1% while maintaining good interactive response.
 */
#if NBPFILTER > 0
#define	BUFOFFSET	(128 + SLIP_HDRLEN)
#else
#define BUFOFFSET	128
#endif
#define SLMAX           (MCLBYTES - BUFOFFSET)/* must be less MCLBYTES XXX */
#define	SLBUFSIZE	(SLMAX + BUFOFFSET)
#define	SLMTU		576	/* Was: 296 CHECH */
#define	SLIP_HIWAT	200	/* Was: 100 CHECH */

/* can't used */
/* #define CLISTRESERVE    1024    /* Can't let buffer space get too low */

/*
 * SLIP_CLSIZE is used as the size of the buffer for the clist
 * routines for a SLIP link.  (wasn't reasonable to try to wad
 * SLIP packets into a 1K buffer)
 */ 
#define SLIP_CLQSIZE    1024*4  /* Was 32768. I suppose ~2 packs is OK */

/*
 * SLIP ABORT ESCAPE MECHANISM:
 *	(inspired by HAYES modem escape arrangement)
 *	1sec escape 1sec escape 1sec escape { 1sec escape 1sec escape }
 *	signals a "soft" exit from slip mode by usermode process
 */

#if 0
#define	ABT_ESC		'\033'	/* can't be t_intr - distant host must know it*/
#define ABT_WAIT	1	/* in seconds - idle before an escape & after */
#define ABT_RECYCLE	(5*2+2)	/* in seconds - time window processing abort */

#define ABT_SOFT	3	/* count of escapes */
#endif

#ifdef GATEWAY
/*
 * The following disgusting hack gets around the problem that IP TOS
 * isn't set yet by some systems.  We want to put "interactive" traffic on
 * a high priority queue.  To decide if traffic is interactive, we check that
 * a) it is TCP and b) one of its ports is telnet, rlogin or ftp control.
 */
static u_short interactive_ports[8] = {
	0,	513,	0,	179,
	0,	21,	0,	23,
};
#define INTERACTIVE(p) (interactive_ports[(p) & 7] == (p))
#endif

struct sl_softc *sl_softc;
struct sl_line  sl_line[SLIPLINES];
#define        SNULL   ((struct sl_softc *) NULL)
#define        LNULL   ((struct sl_line  *) NULL)


#define FRAME_END	 	0xc0		/* Frame End */
#define FRAME_ESCAPE		0xdb		/* Frame Esc */
#define TRANS_FRAME_END	 	0xdc		/* transposed frame end */
#define TRANS_FRAME_ESCAPE 	0xdd		/* transposed frame esc */

#define t_sc T_LINEP

int sloutput(), slioctl(), ttrstrt();
int sl_keepalive(), sl_outfill();
extern struct timeval time;

#ifdef BSD_1_0
extern int tty_net_devices;	/* tty-class devices may be used for net? */
#endif
int	nslip;			/* number of slip interfaces (set at boot) */

/*
 * Called from boot code to establish sl interfaces.
 */
#ifdef BSD_1_0
slattach()
#else  /* BSD_1_1 */
slattach (nsl)
	int nsl;
#endif
{
	register struct sl_softc *sc;
	register int i = 0;

	bzero ((caddr_t) sl_line, sizeof (sl_line));

#ifdef BSD_1_0
	nslip = NSL;
#else /* BSD_1_1 */
	nslip = nsl;
#endif
	sl_softc = malloc(nslip * sizeof *sl_softc, M_DEVBUF, M_WAIT);

	for (sc = sl_softc; i < nslip; sc++) {
		sc->sc_lst = LNULL;
		sc->sc_num = 0;
		sc->sc_if.if_name = "sl";
		sc->sc_if.if_unit = i++;
		sc->sc_if.if_mtu = SLMTU;
#if defined(BSD_1_0) || !defined(MULTICAST)
		sc->sc_if.if_flags = IFF_POINTOPOINT;
#else /* BSD_1_1 && MULTICAST */
		sc->sc_if.if_flags = IFF_POINTOPOINT|IFF_MULTICAST;
#endif
		sc->sc_if.if_type = IFT_SLIP;
		sc->sc_if.if_ioctl = slioctl;
		sc->sc_if.if_output = sloutput;
		sc->sc_if.if_snd.ifq_maxlen = 100;  /* Was: 50 CHECH */
		sc->sc_fastq.ifq_maxlen = 50;       /* Was: 32 CHECH */
		if_attach(&sc->sc_if);
#if NBPFILTER > 0
		bpfattach(&sc->sc_bpf, &sc->sc_if, DLT_SLIP, SLIP_HDRLEN);
#endif
	}
#ifdef BSD_1_0
	tty_net_devices++;
#endif
}

/*
 * Line specific open routine.
 *       su
 */
/* ARGSUSED */
slopen(dev, tp)
	dev_t dev;
	struct tty *tp;
{
	register struct proc *p = curproc;          /* XXX */
	register error;

	if (error = suser(p->p_ucred, &p->p_acflag))
		return (error);
	return (0);
};

/*
 * Line specific close routine.
 * Detach the tty from the sl unit.
 * Mimics part of ttyclose().
 */
slclose(tp)
	register struct tty *tp;
{
	tp->t_line = 0;
	sl_tty_detach (tp);
}

/*
 * Line specific (tty) ioctl routine.
 */
/* ARGSUSED */
sltioctl(tp, cmd, data, flag)
	struct tty *tp;
	caddr_t data;
{
	struct sl_line *l = (struct sl_line *) tp -> t_sc;
	int s, error = 0;
	struct sl_attach sl;

	switch (cmd) {
	case SLIOCATTACH:
	case SLIOCATTW:
		if (l)  return (EBUSY);         /* already attached */
		break;
	case SLIOCDETACH:                       /* all this ioctl's ... */
	case SLIOCGUNIT:                        /* .. must work in ...  */
	case SLIOCGFLAGS:                       /* .. attach mode only  */
	case SLIOCSFLAGS:
	case SLIOCSKEEPAL:
	case SLIOCSOUTFILL:
	case SLIOCGKEEPAL:
	case SLIOCGOUTFILL:
	case SLIOCWAIT:
		if (l == LNULL || l -> l_sc == SNULL)
			return (ENXIO);
	}

	switch (cmd) {
	case SLIOCATTACH:
		error = sl_tty_attach (tp, *(int *)data);
		l = (struct sl_line *) tp -> t_sc;
		l -> l_flags |= SC_KEEPUP;  /*    ? :-( */
		break;

	case SLIOCWAIT:
		/* ignores sl.sl_unit and sl.sl_fill */
		sl = *(struct sl_attach *) data;
		if (sl.sl_cnt)
			error = sl_tty_wait (l, sl.sl_keep, sl.sl_cnt);
		break;

	case SLIOCATTW:
		sl = *(struct sl_attach *) data;
		error = sl_tty_attach (tp, sl.sl_unit);
		if (error)
			break;
		l = (struct sl_line *) tp -> t_sc;
		if (l == LNULL)         /* I'm crazy ? */
			return (ENXIO);
		if (sl.sl_fill) {
			l -> l_outfill   = sl.sl_fill;
			timeout (sl_outfill, (caddr_t) l, sl.sl_fill * hz);
		}
		if (sl.sl_keep) {
			if (sl.sl_cnt) {
				error = sl_tty_wait (l, sl.sl_keep, sl.sl_cnt);
				if (error)  {
					(void) sl_tty_detach (tp);
					break;
				}
			}
			l -> l_keepalive = sl.sl_keep;
			timeout (sl_keepalive, (caddr_t)l, sl.sl_keep * hz);
			l -> l_flags |= SC_KEEPTEST;
		}
		/*   sl.sl_keep == 0 || sl.sl_cnt == 0 */
		l -> l_flags |= SC_KEEPUP;  /*    ? :-( */
		break;

	case SLIOCDETACH:
		/* tty_detach   l  LNULL */
		error = sl_tty_detach (tp);
		if (error == -1)
			error = ENXIO;          /* XXX */
		else {
			*(int *)data = error;
			error = 0;
		}
		break;

	case SLIOCGUNIT:
		*(int *)data = l -> l_sc -> sc_if.if_unit;
		break;

	case SLIOCGFLAGS:   /* same as SIOCGIFFLAGS */
		*(int *)data = l -> l_sc ->sc_if.if_flags & LLC_MASK;
		break;

	case SLIOCSFLAGS:   /* same as SIOCSIFFLAGS */
		s = splimp();
		l -> l_sc -> sc_if.if_flags &= ~LLC_MASK;
		l -> l_sc -> sc_if.if_flags |= (*(int *)data) & LLC_MASK;
		splx(s);
		break;

	case SLIOCSKEEPAL:
		if (l -> l_keepalive)
			untimeout (sl_keepalive, (caddr_t) l);
		if (l -> l_keepalive = *(u_int *) data) {
			timeout (sl_keepalive, (caddr_t) l, l -> l_keepalive*hz);
			l -> l_flags |= SC_KEEPTEST;
		}
		break;

	case SLIOCGKEEPAL:
		*(int *)data = l -> l_keepalive;
		break;

	case SLIOCSOUTFILL:
		if (l -> l_outfill)
			untimeout (sl_outfill, (caddr_t) l);
		if (l -> l_outfill = *(u_int *) data)
			timeout (sl_outfill, (caddr_t) l, l -> l_outfill * hz);
		break;

	case SLIOCGOUTFILL:
		*(int *)data = l -> l_outfill;
		break;

	case SLIOCSETMTU:   /* same as SIOCSIFMTU XXX */
		if (*(short *)data > 128 && *(short *)data < SLMAX) {
			s = splimp();
			l -> l_sc -> sc_if.if_mtu = *(short *)data;
			splx(s);
		}
		else
			error = EINVAL;
		break;

	case SLIOCGETMTU:
		*(short *)data = l -> l_sc -> sc_if.if_mtu;
		break;

	default:
		return (-1);    /* pass thru common ioctl, device ioctl ... */
	}
	return (error);
}

sl_tty_attach (tp, unit)
	register struct tty *tp;
	int unit;
{
	register struct sl_softc *sc;
	register struct sl_line *l, *p;
	int s;
	static slinit ();

	if (unit >= nslip)
		return (ENXIO);
	sc = &sl_softc [unit];
	for (s = 0, l = sl_line; s < SLIPLINES; s++, l++) {
	    if ((l -> l_sc   == SNULL) &&
		(l -> l_nxt  == LNULL) &&
		(l -> l_ttyp == (struct tty *) NULL))
	{
		if (slinit (l, sc) == 0)
			return (ENOBUFS);
		s = splimp();   /* or spltty() ??? XXX */
		tp -> t_sc = (caddr_t) l;
		l -> l_ttyp = tp;
		l -> l_sc = sc;
	    /*         */
		if (sc -> sc_num++) {   /* add line to linked list */
			for (p = sc -> sc_lst;
			    p -> l_nxt != sc -> sc_lst;
			    p = p -> l_nxt) ;
			p -> l_nxt = l;   /* add line to last element of list */
		}
		else
			sc -> sc_lst = l;  /* first entry at linked list */
		l -> l_nxt = sc -> sc_lst;
#ifdef BSD_1_0
		/* 
		 * Free the old clist buffer space and
		 * allocate bigger, better space.
		 */			
		/*** cqttydestroy(tp);  ***/
		if (tp->t_outq.c_cq)
			free((caddr_t)tp->t_outq.c_cq, M_CLIST);
		if (tp->t_outq.c_ct)
			free((caddr_t)tp->t_outq.c_ct, M_CLIST);

		/*** cqttyinit(tp, SLIP_CLQSIZE);   ***/
		tp->t_outq.c_cc = 0;
		tp->t_outq.c_cf = tp->t_outq.c_cl = NULL;
		tp->t_outq.c_cq = (char *)malloc(SLIP_CLQSIZE, M_CLIST, M_WAITOK);
		tp->t_outq.c_ct = (char *)malloc(SLIP_CLQSIZE/NBBY, M_CLIST, M_WAITOK);
		bzero(tp->t_outq.c_ct, SLIP_CLQSIZE/NBBY);
		tp->t_outq.c_ce = (char *)(tp->t_outq.c_cq + SLIP_CLQSIZE - 1);
		tp->t_outq.c_cs = SLIP_CLQSIZE;
#endif /* BSD_1_0 */
		splx(s);
/*??? XXX*/     sc->sc_if.if_baudrate = tp -> t_ospeed;
		ttyflush(tp, FREAD | FWRITE);
#ifndef BSD_1_0 /* BSD_1_1 */
		tty_net_devices(1);     /* add one tty-net device */
#endif
		return (0);
	    }
	}
	return (ENXIO);
}

static int
slinit (l, sc)
	register struct sl_line *l;
	register struct sl_softc *sc;
{
	register caddr_t p;

	if (sc -> sc_num == 0)      /* paranoid */
		sc->sc_if.if_flags &= ~IFF_UP;
	if (l -> l_ep == (u_char *) 0) {
		MCLALLOC(p, M_WAIT);
		if (p)
			l -> l_ep = (u_char *)p + SLBUFSIZE;
		else {
			printf("sl%d: can't allocate buffer\n", sc - sl_softc);
			return (0);
		}
	}
	l -> l_buf = l -> l_ep - SLMAX;
	l -> l_mp  = l -> l_buf;
	if (l -> l_comp)    /* paranoid */
		free (l -> l_comp, M_DEVBUF);
	l -> l_comp = (struct slcompress *)
		malloc (sizeof (struct slcompress), M_DEVBUF, M_WAITOK);
	if (l -> l_comp)
		sl_compress_init (l -> l_comp);
	else {
		printf("sl%d: can't allocate slcompress\n", sc - sl_softc);
		MCLFREE (p);    /* free mbuf */
		return (0);
	}
	l -> l_bytessent    = 0;
	l -> l_bytesrcvd    = 0;
	l -> l_pktin        = 0;
	l -> l_pktout       = 0;
	l -> l_lasttime     = 0;
	l -> l_starttime    = 0;
	l -> l_keepalive    = 0;
	l -> l_outfill      = 0;
	l -> l_abortcount   = 0;
	l -> l_flags        = 0;
	l -> l_keepcnt      = 0;
	/* if POINTOPOINT present don't change interface flags */
	if ( !(sc->sc_if.if_flags & IFF_POINTOPOINT))
#if defined(BSD_1_0) || !defined(MULTICAST)
		sc->sc_if.if_flags = IFF_POINTOPOINT;
#else /* BSD_1_1 && MULTICAST */
		sc->sc_if.if_flags = IFF_POINTOPOINT|IFF_MULTICAST;
#endif
	return (1);
}
/*  -1    ATTACH   
 *    detach-
 */
sl_tty_detach (tp)
	struct tty *tp;
{
	register struct sl_line  *l, *p;
	register struct sl_softc *sc;
	int s, ret;

	ttyflush(tp, FREAD | FWRITE);   /* ??? XXX */
	s = splimp();                   /* actually, max(spltty, splnet) */

	l = (struct sl_line *) tp -> t_sc;
	if (l == LNULL) {
		splx(s);
		return (-1);
	}
	sc = (struct sl_softc *) l -> l_sc;

	/* delete line to linked list */
	for  (p = l; p -> l_nxt != l; p = p-> l_nxt);
	if (p != l)                     /*      */
	    p -> l_nxt = l -> l_nxt;    /*    */

	tp -> t_sc   = (caddr_t) NULL;  /*  "" */
	l -> l_ttyp  = (struct tty *) NULL;     /* -"- */
	l -> l_sc    = SNULL;                   /* -"- */
	l -> l_nxt   = LNULL;                   /* -"- */
	MCLFREE ((caddr_t) (l -> l_ep - SLBUFSIZE));    /* free mbuf */
	l -> l_ep = 0;                          /* clear line struct */
	l -> l_mp = 0;
	l -> l_buf = 0;
	l -> l_flags = 0;
	l -> l_keepalive = 0;
	l -> l_outfill = 0;
	if (l -> l_comp) {                      /*   */
		free (l -> l_comp, M_DEVBUF);   /*   */
		l -> l_comp = NULL;             /*    */
	}
	untimeout (sl_keepalive,(caddr_t) l);   /* clear timeouts */
	untimeout (sl_outfill,  (caddr_t) l);

	/*   */
	if (sc) {                       /* kludge */
	    if (p != l) {               /*     */
		if (sc -> sc_lst == l)  /*      */
		    sc -> sc_lst = p;   /*     */
		sc -> sc_num--;
	    }
	    else {                      /*    */
		sc -> sc_lst = LNULL;   /*      */
		sc -> sc_num = 0;       /*    */
		sc -> sc_if.if_flags &= ~(IFF_UP | IFF_RUNNING);
		if_down (&sc->sc_if);   /*   */
	    }
	    ret = sc -> sc_num;
	}
	else                            /*  :-) */
	    ret = 0;
#ifdef BSD_1_0
	cqttydestroy(tp);               /* paranoid */
	cqttyinit(tp, TTYHOG);          /* -"- -"- */
#endif /* BSD_1_0 */
	splx(s);

#ifndef BSD_1_0 /* BSD_1_1 */
	tty_net_devices (-1);           /* subtract one tty-net device */
#endif
	return (ret);
}

sl_tty_wait (l, time, count)
register struct sl_line  *l;
register time;
{
	register error;

	if (count > 20)                 /* kludge */
		count = 20;
	l -> l_keepcnt = count;
	l -> l_flags |= SC_KEEPWAIT;
	l -> l_flags &= ~SC_KEEPUP;     /*     */
	error = ttysleep (l -> l_ttyp, (caddr_t) l, TTIPRI|PCATCH, "slwait",
			time * hz);
	if (error || (l -> l_keepcnt > 0)) {
		log (LOG_WARNING, "sl%d: no keepalives ? [errno %d cnt %d]",
		    l -> l_sc -> sc_if.if_unit, error, l->l_keepcnt);
		error = ETIMEDOUT;
	}
	else if (l -> l_keepalive == 0)         /* kludge :-( */
		l -> l_flags |= SC_KEEPUP;      /*   */

	return (error);
}

/*
 * Queue a packet.  Start transmission if not active.
 */
sloutput(ifp, m, dst)
	struct ifnet *ifp;
	register struct mbuf *m;
	struct sockaddr *dst;
{
	register struct sl_softc *sc = &sl_softc[ifp->if_unit];
	register struct sl_line  *l;
	register struct ip *ip;
	register struct ifqueue *ifq;
	int s, i;
#if NBPFILTER > 0
	u_char bpfbuf[SLMAX + SLIP_HDRLEN];
	register int len;
#endif

	/*
	 * `Cannot happen' (see slioctl).  Someday we will extend
	 * the line protocol to support other address families.
	 */
	if (dst->sa_family != AF_INET) {
		printf("sl%d: af%d not supported\n", sc->sc_if.if_unit,
			dst->sa_family);
		m_freem(m);
		return (EAFNOSUPPORT);
	}

	if (sc -> sc_lst == LNULL || sc -> sc_num == 0 ||
	    (sc -> sc_if.if_flags & (IFF_UP | IFF_RUNNING)) !=
	    (IFF_UP | IFF_RUNNING)) {
		m_freem(m);
		return (ENETDOWN);	/* sort of */
	}

	l = sc -> sc_lst -> l_nxt;     /* Get next "line" */
	for (i = sc -> sc_num; i--; l = l -> l_nxt) {
		if ((l -> l_flags & SC_KEEPUP) == 0)
		    continue;
		if ((l -> l_ttyp -> t_state & TS_CARR_ON) == 0 &&
		    (l -> l_ttyp -> t_cflag & CLOCAL) == 0)
		    continue;
		if (l -> l_ttyp -> t_ospeed)
		    goto found;
	}
	m_freem(m);                 /* normal "line" not found */
	return (EHOSTUNREACH);
found:                              /* use first normal "line" */
	s = splimp ();              /* ??? XXX */
	sc -> sc_lst = l;           /* make it as "current line" */
	splx (s);

#if NBPFILTER > 0
	if ((sc->sc_if.if_flags & SC_BPFILTER) && sc->sc_bpf) {
/* ??? XXX */
		/*
		 * We need to save the TCP/IP header before it's compressed.
		 * To avoid complicated code, we just copy the entire packet
		 * into a stack buffer (since this is a serial line, packets
		 * should be short and/or the copy should be negligible cost
		 * compared to the packet transmission time).
		 */
		register struct mbuf *m0 = m;
		register u_char *cp = bpfbuf + SLIP_HDRLEN;

		len = 0;
		do {
			register int mlen = m0->m_len;

			bcopy(mtod(m0, caddr_t), cp, (u_int)mlen);
			cp += mlen;
			len += mlen;
		} while (m0 = m0->m_next);
		/*
		 * Put the SLIP pseudo-"link header" in place.  The compressed
		 * header is now at the beginning of the mbuf.
		 */
		bpfbuf[SLX_DIR] = SLIPDIR_OUT;
		bcopy(mtod(m, caddr_t), &bpfbuf[SLX_CHDR], CHDR_LEN);
		bpf_tap(sc->sc_bpf, bpfbuf, (u_int)len + SLIP_HDRLEN);
	}
#endif
/*    .      fastq */
	ifq = &sc->sc_if.if_snd;
	if ((ip = mtod(m, struct ip *))->ip_p == IPPROTO_TCP) {
		if (ip->ip_tos == IPTOS_LOWDELAY)
			ifq = &sc->sc_fastq;
#ifdef GATEWAY
		else if (ip->ip_off == 0) {
			register int p;

		/* Don't need to byte swap ip_off to compare against 0 */
			p = ntohl(((int *)ip)[ip->ip_hl]);
			if (INTERACTIVE(p & 0xffff) || INTERACTIVE(p >> 16))
				ifq = &sc->sc_fastq;
		}
#endif
	}
#ifdef SC_NOICMP /* closed ! */
	else if (sc->sc_if.if_flags & SC_NOICMP && ip->ip_p == IPPROTO_ICMP) {
		m_freem(m);
		return (0);
	}
#endif
	s = splimp();
	if (IF_QFULL(ifq)) {
		IF_DROP(ifq);
		m_freem(m);
		splx(s);
		sc->sc_if.if_oerrors++;
		return (ENOBUFS);
	}
	IF_ENQUEUE(ifq, m);
	sc->sc_if.if_lastchange = time;

/*  .    round robbin */
	for (i = sc -> sc_num; i--; l = l -> l_nxt) {
/*         */
		if ((l -> l_ttyp -> t_state & TS_CARR_ON) == 0 &&
		    (l -> l_ttyp -> t_cflag & CLOCAL) == 0)
		    continue;
		if (l -> l_ttyp -> t_ospeed == 0)
		    continue;
/*    .       
     .  .    
          (. break) */
		if (l -> l_ttyp -> t_outq.c_cc == 0) {
			slstart(l -> l_ttyp);
			break;                  /* ??? XXX */
		}
		else
			(*l -> l_ttyp -> t_oproc)(l -> l_ttyp);
	}

	splx(s);
	return (0);
}

/*
 * Start output on interface.  Get another datagram
 * to send from the interface queue and map it to
 * the interface before starting output.
 */
slstart(tp)
	register struct tty *tp;
{
	register struct sl_line  *l  = (struct sl_line *)  tp -> t_sc;
	register struct sl_softc *sc;
	register struct mbuf *m;
	register u_char *cp;
	register struct ip *ip;
	int s;
	struct mbuf *m2;
	u_int obytessent;
	int colls;

	for (;;) {
		/*
		 * If there is more in the output queue, just send it now.
		 * We are being called in lieu of ttstart and must do what
		 * it would.
		 */
		if (tp -> t_outq.c_cc != 0) {
			if (l)  /*     FRAME_END ??? XXX */
				l -> l_flags &= ~SC_OUTWAIT;
			(*tp->t_oproc)(tp);
			if (tp->t_outq.c_cc > SLIP_HIWAT)
				return;
		}
		/*
		 * This happens briefly when the line shuts down.
		 */
		if (l == LNULL || (sc = l -> l_sc) == SNULL)
			return;

		/*
		 * Get a packet and send it to the interface.
		 */
		s = splimp();
		IF_DEQUEUE(& sc -> sc_fastq, m);
		if (m == NULL)
			IF_DEQUEUE(&sc->sc_if.if_snd, m);
		splx(s);
		if (m == NULL)
			return;
		sc -> sc_if.if_lastchange = time;
		l -> l_flags &= ~SC_OUTWAIT;
		obytessent = l -> l_bytessent;
/********************************************************************
*               
*       SLIP_HIWAT - . . Chech
*
*                * If low on clist buffer space, flush our
*                * output queue (if the stuff was important, it'll
*                * get retransmitted).
*
*               if (SLIP_CLQSIZE - tp->t_outq.c_cc <= (SLMAX*3)/2) {
*                       m_freem(m);
*                       sc->sc_if.if_collisions++;
*                       log(LOG_WARNING, "sl%d: low clist space (%d)\n",
*                           sc->sc_if.if_unit, SLIP_CLQSIZE - tp->t_outq.c_cc);
*                       continue;
*               }
*********************************************************************/
		/*
		 * The extra FRAME_END will start up a new packet, and thus
		 * will flush any accumulated garbage.  We do this whenever
		 * the line may have been idle for some time.
		 */
		if (tp->t_outq.c_cc == 0) {
			++l -> l_bytessent;
			(void) putc(FRAME_END, &tp->t_outq);
		}
		/* Compress TCP header of package */
		if ((ip = mtod(m, struct ip *)) -> ip_p == IPPROTO_TCP) {
			if (sc -> sc_if.if_flags & SC_COMPRESS) {
				u_char  c;
				c = sl_compress_tcp(m, ip, l->l_comp);
				*mtod(m, u_char *) |= c;
			}
		}

		colls = 0;
		while (m) {
			register u_char *ep;

			cp = mtod(m, u_char *); ep = cp + m->m_len;
			while (cp < ep) {
				/*
				 * Find out how many bytes in the string we can
				 * handle without doing something special.
				 */
				register u_char *bp = cp;

				while (cp < ep) {
					switch (*cp++) {
					case FRAME_ESCAPE:
					case FRAME_END:
						--cp;
						goto out;
					}
				}
				out:
				if (cp > bp) {
					/*
					 * Put n characters at once
					 * into the tty output queue.
					 */
					if (b_to_q((char *)bp, cp - bp, &tp->t_outq)) {
						colls++;
						break;
					}
					l->l_bytessent += cp - bp;
				}
				/*
				 * If there are characters left in the mbuf,
				 * the first one must be special..
				 * Put it out in a different form.
				 */
				if (cp < ep) {
					if (putc(FRAME_ESCAPE, &tp->t_outq)) {
						colls++;
						break;
					}
					if (putc(*cp++ == FRAME_ESCAPE ?
					   TRANS_FRAME_ESCAPE : TRANS_FRAME_END,
					   &tp->t_outq)) {
						colls++;
						(void) unputc(&tp->t_outq);
						break;
					}
					l->l_bytessent += 2;
				}
			}
			MFREE(m, m2);
			m = m2;
		}

		if (putc(FRAME_END, &tp->t_outq)) {
			/*
			 * Not enough room.  Remove a char to make room
			 * and end the packet normally.
			 * If you get many collisions (more than one or two
			 * a day) you probably do not have enough clists
			 * and you should increase "nclist" in param.c.
			 */
			(void) unputc(&tp->t_outq);
			(void) putc(FRAME_END, &tp->t_outq);
			colls++;
		} else {
			l->l_bytessent++;
			l->l_pktout++;
			sc->sc_if.if_opackets++;
		}
		if (colls) {
			log(LOG_WARNING, "sl%d: no clist space [%bytes]\n",
			    sc->sc_if.if_unit, colls);
			sc->sc_if.if_collisions++;
		}
		sc->sc_if.if_obytes += l->l_bytessent - obytessent;
	}
}

/*
 * Copy data buffer to mbuf chain; add ifnet pointer.
 */
static struct mbuf *
sl_btom(l, len)
	register struct sl_line *l;
	register int len;
{
	register struct mbuf *m;

	MGETHDR(m, M_DONTWAIT, MT_DATA);
	if (m == NULL)
		return (NULL);

	/*
	 * If we have more than MHLEN bytes, it's cheaper to
	 * queue the cluster we just filled & allocate a new one
	 * for the input buffer.  Otherwise, fill the mbuf we
	 * allocated above.  Note that code in the input routine
	 * guarantees that packet will fit in a cluster.
	 */
	if (len >= MHLEN) {
		MCLGET(m, M_DONTWAIT);
		if ((m->m_flags & M_EXT) == 0) {
			/*
			 * we couldn't get a cluster - if memory's this
			 * low, it's time to start dropping packets.
			 */
			(void) m_free(m);
			return (NULL);
		}
		l->l_ep = mtod(m, u_char *) + SLBUFSIZE;
		m->m_data = (caddr_t)l->l_buf;
		m->m_ext.ext_buf = (caddr_t)((int)l->l_buf &~ MCLOFSET);
	} else
		bcopy((caddr_t)l->l_buf, mtod(m, caddr_t), len);

	m->m_len = len;
	m->m_pkthdr.len = len;
	m->m_pkthdr.rcvif = &(l->l_sc->sc_if);  /* ??? XXX */
	return (m);
}

/*
 * tty interface receiver interrupt.
 */
slinput(c, tp)
	register int c;
	register struct tty *tp;
{
	register struct sl_line *l;
	register struct sl_softc *sc;
	register struct mbuf *m;
	register int len;
	int s;
	extern struct ttytotals ttytotals;
#if NBPFILTER > 0
	u_char chdr[CHDR_LEN];
#endif

	ttytotals.tty_nin++;
	l = (struct sl_line *) tp -> t_sc;
	if (l == LNULL || (sc = l -> l_sc) == SNULL)
		return;

	++l -> l_bytesrcvd;
	++sc -> sc_if.if_ibytes;
	c &= 0xff;			/* XXX */

#ifdef ABT_ESC
	{
		/* if we see an abort after "idle" time, count it */
		if (c == ABT_ESC && time.tv_sec >= l -> l_lasttime + ABT_WAIT) {
			l -> l_abortcount++;
			/* record when the first abort escape arrived */
			if (l -> l_abortcount == 1)
				l -> l_starttime = time.tv_sec;
		}
		/*
		 * if we have an abort, see that we have not run out of time,
		 * or that we have an "idle" time after the complete escape
		 * sequence
		 */
		if (l -> l_abortcount) {
			if (time.tv_sec >= l -> l_starttime + ABT_RECYCLE)
				l -> l_abortcount = 0;
			if (l -> l_abortcount >= ABT_SOFT &&
			    time.tv_sec >= l -> l_lasttime + ABT_WAIT) {
				slclose(tp);
				return;
			}
		}
		l -> l_lasttime = time.tv_sec;
	}
#endif

	switch (c) {

	case TRANS_FRAME_ESCAPE:
		if (l -> l_flags & SC_ESCAPE)
			c = FRAME_ESCAPE;
		break;

	case TRANS_FRAME_END:
		if (l -> l_flags & SC_ESCAPE)
			c = FRAME_END;
		break;

	case FRAME_ESCAPE:
		l -> l_flags |= SC_ESCAPE;
		return;

	case FRAME_END:
	    /*    keepalives */
		if ((l -> l_flags & SC_KEEPWAIT) && (l -> l_keepcnt-- <= 0)) {
			wakeup ((caddr_t) l);
			l -> l_flags &= ~SC_KEEPWAIT;
		}
		if (l -> l_keepalive) {
		/*      */
			l -> l_flags &= ~SC_KEEPTEST;/*   */
			l -> l_flags |= SC_KEEPUP;  /*  UP  */
			if (sc -> sc_if.if_flags & IFF_RUNNING)
				sc -> sc_if.if_flags |= IFF_UP;
		}

		len = l -> l_mp - l -> l_buf;
		if ((len < 3) ||                /*     */
		    ((sc -> sc_if.if_flags & (IFF_UP | IFF_RUNNING)) !=
		    (IFF_UP | IFF_RUNNING)))    /*     */
			goto newpack;           /*    */

#if NBPFILTER > 0
		if ((sc->sc_if.if_flags & SC_BPFILTER) && sc->sc_bpf)
			/*
			 * Save the compressed header, so we can
			 * tack it on later.  Note that we just
			 * copy the maximum number of bytes (16) --
			 * we will end up copying garbage in some
			 * cases but this is okay.  We remember
			 * where the buffer started so we can
			 * compute the new header length.
			 */
			bcopy((caddr_t)l -> l_buf, (caddr_t)chdr, CHDR_LEN);
#endif
		if ((c = (*l -> l_buf & 0xf0)) != (IPVERSION << 4)) {
			if (c & 0x80)
				c = TYPE_COMPRESSED_TCP;
			else if (c == TYPE_UNCOMPRESSED_TCP)
				*l -> l_buf &= 0x4f; /* XXX */
			/*
			 * We've got something that's not an IP packet.
			 * If compression is enabled, try to decompress it.
			 * Otherwise, if `auto-enable' compression is on and
			 * it's a reasonable packet, decompress it and then
			 * enable compression.  Otherwise, drop it.
			 */
			if (sc->sc_if.if_flags & SC_COMPRESS) {
				len = sl_uncompress_tcp(&l -> l_buf, len,
							(u_int)c, l -> l_comp);
				if (len <= 0)
					goto error;
			} else if ((sc->sc_if.if_flags & SC_AUTOCOMP) &&
			    c == TYPE_UNCOMPRESSED_TCP && len >= 40) {
				len = sl_uncompress_tcp(&l -> l_buf, len,
							(u_int)c, l -> l_comp);
				if (len <= 0)
					goto error;
				sc->sc_if.if_flags |= SC_COMPRESS;
			} else
				goto error;
		}
#if NBPFILTER > 0
		if ((sc->sc_if.if_flags & SC_BPFILTER) && sc->sc_bpf) {
			/*
			 * Put the SLIP pseudo-"link header" in place.
			 * We couldn't do this any earlier since
			 * decompression probably moved the buffer
			 * pointer.  Then, invoke BPF.
			 */
			register u_char *hp = l -> l_buf - SLIP_HDRLEN;
			hp[SLX_DIR] = SLIPDIR_IN;
			bcopy((caddr_t)chdr, (caddr_t)&hp[SLX_CHDR], CHDR_LEN);
			bpf_tap(sc->sc_bpf, hp, (u_int)len + SLIP_HDRLEN);
		}
#endif
		m = sl_btom(l, len);
		if (m == NULL)
			goto error;

		l -> l_pktin++;
		sc->sc_if.if_ipackets++;
		sc->sc_if.if_lastchange = time;
		s = splimp();
		if (IF_QFULL(&ipintrq)) {
			IF_DROP(&ipintrq);
			sc->sc_if.if_iqdrops++;
			m_freem(m);
		} else {
			IF_ENQUEUE(&ipintrq, m);
			schednetisr(NETISR_IP);
		}
		splx(s);
		goto newpack;
	}
	if (l -> l_mp < l -> l_ep) {
		*l -> l_mp++ = c;
		l -> l_flags &= ~SC_ESCAPE;     /* ??? XXX */
		return;
	}
error:
	sc->sc_if.if_ierrors++;
newpack:
	l -> l_mp = l -> l_buf = l -> l_ep - SLMAX;
	l -> l_flags &= ~SC_ESCAPE;
}

/*
 * Process an ioctl request.
 */
slioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	register struct ifaddr *ifa = (struct ifaddr *)data;
	register struct sl_softc *sc;
#ifndef BSD_1_0
	register struct ifreq *ifr;
#endif
	int s = splimp(), error = 0;

	if (data == NULL)
		return (EINVAL);

	switch (cmd) {

	case SIOCSIFADDR:
		if (ifa->ifa_addr->sa_family == AF_INET)
			ifp->if_flags |= IFF_UP | IFF_RUNNING;
		else
			error = EAFNOSUPPORT;
		break;

	case SIOCSIFDSTADDR:
		if (ifa->ifa_addr->sa_family != AF_INET)
			error = EAFNOSUPPORT;
		break;

#if !defined(BSD_1_0) && defined(MULTICAST) /* BSD_1_1 && MUTLICAST */
	case SIOCADDMULTI:
	case SIOCDELMULTI:
		ifr = (struct ifreq *)data;
		if (ifr == 0) {
			error = EAFNOSUPPORT;		/* XXX */
			break;
		}
		switch (ifr->ifr_addr.sa_family) {
#ifdef INET
		case AF_INET:
			break;
#endif
		default:
			error = EAFNOSUPPORT;
			break;
		}
		break;
#endif /* BSD_1_1 && MULTICAST */

	case SIOCSIFFLAGS:
		if (ifp -> if_flags & IFF_UP)
		    ifp -> if_flags |= IFF_RUNNING;
		else
		    ifp -> if_flags &= ~IFF_RUNNING;
		break;

	default:
		error = EINVAL;
	}
	splx(s);
	return (error);
}

sl_keepalive (l)
    register struct sl_line *l;
{ 
    register struct sl_softc *sc;
    register struct sl_line *n;

    sc = l -> l_sc;
    if (l && l -> l_keepalive) {
	if (l -> l_flags & SC_KEEPTEST) {       /* was not keepalive */
	    l -> l_flags &= ~SC_KEEPUP;         /*   DOWN */
	/*       
	 *        
	 */ if (sc -> sc_if.if_flags & IFF_RUNNING) {
/* XXX ???*/    for (n = l -> l_nxt; n != l; n = n -> l_nxt)/*  ""  */
		    if (l -> l_flags & SC_KEEPUP)
			break;
		if (n == l) {
		    int s = splimp ();
		    sc -> sc_if.if_flags &= ~(IFF_UP | IFF_RUNNING);
		    if_down (&sc -> sc_if);
		    splx (s);
		}
	    }
	    sl_signal (l);
	}
	else {
	    l -> l_flags |= SC_KEEPTEST;
	}
	timeout (sl_keepalive, (caddr_t) l, l -> l_keepalive * hz);
    } else
	untimeout (sl_keepalive, (caddr_t) l);
}
	
sl_outfill (l)
	register struct sl_line *l;
{
	register struct tty *tp = l -> l_ttyp;
	int s;

	if (l && tp && l -> l_outfill) {
		if (l -> l_flags & SC_OUTWAIT) {
			s = splimp ();
			putc(FRAME_END, &tp -> t_outq);
			(*tp->t_oproc)(tp);
			splx (s);
			l->l_bytessent++;
			if (l -> l_sc)
			    l -> l_sc -> sc_if.if_obytes++;
		} else
			l -> l_flags |= SC_OUTWAIT;
		timeout (sl_outfill, (caddr_t) l, l -> l_outfill * hz);
	} else
		untimeout (sl_keepalive, (caddr_t) l);
}

sl_signal (l)
	register struct sl_line *l;
{
	pgsignal (l -> l_ttyp -> t_pgrp, SIGUSR1, 1);
}
#endif
