/*
 * tickadj - read, and possibly modify, the kernel `tick' and
 *	     `tickadj' variables, as well as `dosynctodr'.  Note that
 *	     this operates on the running kernel only.  I'd like to be
 *	     able to read and write the binary as well, but haven't
 *	     mastered this yet.
 */
#include <stdio.h>
#include <nlist.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>

#ifdef RS6000
#undef hz
#endif RS6000

#define	KMEM	"/dev/kmem"
#define	STREQ(a, b)	(*(a) == *(b) && strcmp((a), (b)) == 0)

char *progname;
int debug;

int dokmem = 1;
int writetickadj = 0;
int writeopttickadj = 0;
int unsetdosync = 0;
int writetick = 0;
int quiet = 0;

char *kmem = KMEM;
char *kernel = 0;


/*
 * main - parse arguments and handle options
 */
main(argc, argv)
int argc;
char *argv[];
{
	int c;
	int errflg = 0;
	extern int optind;
	extern char *optarg;
	unsigned long tickadj_offset;
	unsigned long tick_offset;
	unsigned long dosync_offset;
	int tickadj;
	int tick;
	int dosynctodr;
	int hz, hz_hundredths;
	int recommend_tickadj;
	long tmp;
	char *file;
	int fd;
	int openfile();
	char *getoffsets();
	void readvar();
	void writevar();

	progname = argv[0];
	while ((c = getopt(argc, argv, "a:Adkqst:")) != EOF)
		switch (c) {
		case 'd':
			++debug;
			break;
		case 'k':
			dokmem = 1;
			break;
		case 'q':
			quiet = 1;
			break;
		case 'a':
			writetickadj = atoi(optarg);
			if (writetickadj <= 0) {
				(void) fprintf(stderr,
				    "%s: unlikely value for tickadj: %s\n",
				    progname, optarg);
				errflg++;
			}
			break;
		case 'A':
			writeopttickadj = 1;
			break;
		case 's':
			unsetdosync = 1;
			break;
		case 't':
			writetick = atoi(optarg);
			if (writetick <= 0) {
				(void) fprintf(stderr,
				    "%s: unlikely value for tick: %s\n",
				    progname, optarg);
				errflg++;
			}
			break;
		default:
			errflg++;
			break;
		}
	if (errflg || optind != argc) {
		(void) fprintf(stderr,
		    "usage: %s [-Aqs] [-a newadj] [-t newtick]\n", progname);
		exit(2);
	}
	kernel = getoffsets(kernel, &tick_offset,
	    &tickadj_offset, &dosync_offset);
	if (debug) {
		(void) printf("tick offset = %lu\n", tick_offset);
		(void) printf("tickadj offset = %lu\n", tickadj_offset);
		(void) printf("dosynctodr offset = %lu\n", dosync_offset);
	}

	if (dokmem)
		file = kmem;
	else
		file = kernel;

	fd = openfile(file, O_RDONLY);
	readvar(fd, tickadj_offset, &tickadj);
	readvar(fd, tick_offset, &tick);
	if (dosync_offset != 0)
		readvar(fd, dosync_offset, &dosynctodr);
	(void) close(fd);

	if (unsetdosync && dosync_offset == 0) {
		(void) fprintf(stderr,
		    "%s: can't find _dosynctodr in namelist\n", progname);
		exit(1);
	}

	if (!quiet) {
		if (dosync_offset == 0)
			(void) printf("tick = %d us, tickadj = %d us\n",
			     tick, tickadj);
		else
			(void) printf(
			    "tick = %d us, tickadj = %d us, dosynctodr is %s\n",
			     tick, tickadj, dosynctodr ? "on" : "off");
	}

	if (tick <= 0) {
		(void) fprintf(stderr, "%s: the value of tick is silly!\n");
		exit(1);
	}

	hz = (int)(1000000L / (long)tick);
	hz_hundredths = (int)((100000000L / (long)tick) - ((long)hz * 100L));
	if (!quiet)
		(void) printf("calculated hz = %d.%02d Hz\n", hz,
		    hz_hundredths);
	tmp = (long) tick * 500L;
	recommend_tickadj = (int)(tmp / 1000000L);
	if (tmp % 1000000L > 0)
		recommend_tickadj++;
	if (!quiet)
		(void) printf("recommended value of tickadj = %d us\n",
		    recommend_tickadj);
	
	if (writetickadj == 0 && !writeopttickadj &&
	    !unsetdosync && writetick == 0)
		exit(0);

	if (writetickadj == 0 && writeopttickadj)
		writetickadj = recommend_tickadj;

	fd = openfile(file, O_WRONLY);
	if (writetick > 0) {
		if (!quiet) {
			(void) fprintf(stderr, "writing tick, value %d: ",
			    writetick);
			(void) fflush(stderr);
		}
		writevar(fd, tick_offset, writetick);
		if (!quiet)
			(void) fprintf(stderr, "done!\n");
	}
	if (writetickadj > 0) {
		if (!quiet) {
			(void) fprintf(stderr, "writing tickadj, value %d: ",
			    writetickadj);
			(void) fflush(stderr);
		}
		writevar(fd, tickadj_offset, writetickadj);
		if (!quiet)
			(void) fprintf(stderr, "done!\n");
	}
	if (unsetdosync) {
		if (!quiet) {
			(void) fprintf(stderr, "zeroing dosynctodr: ");
			(void) fflush(stderr);
		}
		writevar(fd, dosync_offset, 0);
		if (!quiet)
			(void) fprintf(stderr, "done!\n");
	}
	(void) close(fd);
	exit(0);
}

/*
 * getoffsets - read the magic offsets from the specified file
 */
char *
getoffsets(file, tick_off, tickadj_off, dosync_off)
	char *file;
	unsigned long *tick_off;
	unsigned long *tickadj_off;
	unsigned long *dosync_off;
{
	char **kname;
	int err = 0;
	static struct nlist nl[] =
	{	{"_tickadj"},
		{"_tick"},
		{"_dosynctodr"},
		{""},
	};
	static char *kernels[] = {
		"/vmunix",
		"/unix",
		NULL
	};
	struct stat stbuf;

#define	K_TICKADJ	0
#define	K_TICK		1
#define	K_DOSYNC	2
	for (kname = kernels; *kname != NULL; kname++) {
		if (stat(*kname, &stbuf) == -1)
			continue;
		if (nlist(*kname, nl) >= 0)
			break;
	}
	if (*kname == NULL) {
		(void) fprintf(stderr,
		    "%s: nlist fails: can't find/read /vmunix or /unix\n",
		    progname);
		exit(1);
	}

	if (nl[K_TICKADJ].n_value == 0) {
		(void) fprintf(stderr, "%s: namelist can't find `_tickadj'\n",
		    progname);
		err++;
	}
	if (nl[K_TICK].n_value == 0) {
		(void) fprintf(stderr, "%s: namelist can't find `_tick'\n",
		    progname);
		err++;
	}
	if (err)
		exit(1);

	*tickadj_off = nl[K_TICKADJ].n_value;
	*tick_off = nl[K_TICK].n_value;
	*dosync_off = nl[K_DOSYNC].n_value;
	return *kname;
#undef K_TICKADJ
#undef K_TICK
#undef K_DOSYNC
}


/*
 * openfile - open the file, check for errors
 */
int
openfile(name, mode)
	char *name;
	int mode;
{
	int fd;

	fd = open(name, mode);
	if (fd < 0) {
		(void) fprintf(stderr, "%s: open %s: ", progname, name);
		perror("");
		exit(1);
	}
	return fd;
}


/*
 * writevar - write a variable into the file
 */
void
writevar(fd, off, var)
	int fd;
	unsigned long off;
	int var;
{
	
	if (lseek(fd, off, L_SET) == -1) {
		(void) fprintf(stderr, "%s: lseek fails: ", progname);
		perror("");
		exit(1);
	}
	if (write(fd, (char *)&var, sizeof(int)) != sizeof(int)) {
		(void) fprintf(stderr, "%s: write fails: ", progname);
		perror("");
		exit(1);
	}
}


/*
 * readvar - read a variable from the file
 */
void
readvar(fd, off, var)
	int fd;
	unsigned long off;
	int *var;
{
	int i;
	
	if (lseek(fd, off, L_SET) == -1) {
		(void) fprintf(stderr, "%s: lseek fails: ", progname);
		perror("");
		exit(1);
	}
	i = read(fd, (char *)var, sizeof(int));
	if (i < 0) {
		(void) fprintf(stderr, "%s: read fails: ", progname);
		perror("");
		exit(1);
	}
	if (i != sizeof(int)) {
		(void) fprintf(stderr, "%s: read expected %d, got %d\n",
		    progname, sizeof(int), i);
		exit(1);
	}
}
