/*======================================================================

   Header file for Sound Blaster user programs.
   [ This file is a part of SBlast-BSD-1.5 ]

   Steve Haehnichen <shaehnic@ucsd.edu>

   sblast.h,v 2.2 1992/09/14 03:09:25 steve Exp

   Copyright (C) 1992 Steve Haehnichen.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 * sblast.h,v
 * Revision 2.2  1992/09/14  03:09:25  steve
 * Released as distribution v1.5.
 * Mostly stable, still lacking stereo recording.
 *
 * Revision 2.1  1992/08/02  21:55:03  steve
 * Added select handler and streamlined a few things.
 * Now allows read/write on a single open.
 *
 * Revision 2.0  1992/07/13  01:25:23  steve
 * Promoting to v2.0 with start of CVS management.
 *
 * Revision 1.5.1.1  1992/07/13  00:39:32  steve
 * Stable double-buffering Sound Blaster driver.
 *
 * Revision 1.5  1992/06/13  01:48:15  steve
 * Released in SBlast-BSD-1.5

======================================================================*/

#ifndef SBLAST_H
#define SBLAST_H
#include <sys/ioctl.h>

/*
 * Here is where you select which card you have.
 * Define only one of these.
 *
 * NOTE: Version 1.5 supports ONLY the SBPRO selection.
 *       Feel free to add support for the rest. :-) 
 */

/* Define this if you have a Sound Blaster Pro  (DSP v3.x) */
#define SBPRO
/* Define this if you have a Sound Blaster with a DSP version 2.x */
/* #define SB20 */
/* Define this if you have an older Sound Blaster (DSP v1.x) */
/* #define SB10 */

typedef unsigned char BYTE;
typedef unsigned char FLAG;

/*
 * Available compression modes:
 * (These are actually DSP DMA-start commands, but you don't care)
 */
enum
{
  PCM_8 	= 0x14,		/* Straight 8-bit PCM */
  ADPCM_4 	= 0x74,		/* 4-bit ADPCM compression */
  ADPCM_3	= 0x76,		/* 2.5-bit ADPCM compression */
  ADPCM_2 	= 0x16,		/* 2-bit ADPCM compression */
};

enum { FM_LEFT, FM_RIGHT, FM_BOTH };	/* Stereo channel choices */



/*   IOCTLs for FM, Mixer, and DSP control.   */

/*
 * FM: Reset chips to silence
 *     Play a note as defined by struct sb_fm_note
 *     Set global FM parameters as defined by struct sb_fm_params
 */
#define FM_IOCTL_RESET      	_IO('s', 10)
#define FM_IOCTL_PLAY_NOTE  	_IOW('s', 11, struct sb_fm_note)
#define FM_IOCTL_SET_PARAMS 	_IOW('s', 13, struct sb_fm_params)

/* Mixer: Set mixer volume levels
 *        Set mixer control parameters (non-volume stuff)
 *	  Read current mixer volume levels
 *	  Read current mixer parameters
 *	  Reset the mixer to a default state
 */	  
#define MIXER_IOCTL_SET_LEVELS 	_IOW('s', 20, struct sb_mixer_levels)
#define MIXER_IOCTL_SET_PARAMS 	_IOW('s', 21, struct sb_mixer_params)
#define MIXER_IOCTL_READ_LEVELS	_IOR('s', 22, struct sb_mixer_levels)
#define MIXER_IOCTL_READ_PARAMS _IOR('s', 23, struct sb_mixer_params)
#define MIXER_IOCTL_RESET 	_IO('s', 24)

/* DSP: Reset the DSP and terminate any transfers.
 *      Set the speed (in Hz) of DSP playback/record.
 *      (Note that the speed parameter is modified to be the actual speed.)
 *      Turn the DSP voice output on (non-zero) or off (0)
 *      Flush any pending written data.
 *      Set the DSP decompression mode to one of the above modes.
 *      Set Stereo playback/record on (non-zero) or off (0)
 *	Set the DMA transfer buffer size.
 *	Errno generated when there is recording overrun.  (ESUCCESS for ignore)
 */
#define DSP_IOCTL_RESET 	_IO('s', 0)
#define DSP_IOCTL_SPEED 	_IOWR('s', 1, int)
#define DSP_IOCTL_VOICE 	_IOW('s', 2, FLAG)
#define DSP_IOCTL_FLUSH 	_IO('s', 3)
#define DSP_IOCTL_COMPRESS	_IOW('s', 4, BYTE)
#define DSP_IOCTL_STEREO	_IOW('s', 5, FLAG) /* Can go to mixer too */
#define DSP_IOCTL_BUFSIZE	_IOWR('s', 6, int)
#define DSP_IOCTL_OVERRUN_ERRNO	_IOW('s', 7, int)

/*
 * DSP legal speed range:
 * For cards other than the SBPRO, there are separate limits
 * on recording and playback speed.  Future driver versions will
 * address this.  Also, compression effects the valid range.  Whew..
 */
# define DSP_MIN_SPEED	3906
#ifdef SBPRO
#  define DSP_MAX_SPEED	47619
#endif
#ifdef SB20
#  define DSP_MAX_SPEED 43478
#endif
#ifdef SB10
#  define DSP_MAX_SPEED 22222
#endif

/* The smallest and largest allowable DSP transfer buffer size.
   Note that the size must also be divisible by two */
#define DSP_MIN_BUFSIZE 256
#define DSP_MAX_BUFSIZE (64 * 1024)

struct stereo_vol
{
  BYTE l;			/* Left volume */
  BYTE r;			/* Right volume */
};


/*
 * Mixer volume levels for MIXER_IOCTL_SET_VOL & MIXER_IOCTL_READ_VOL
 */
struct sb_mixer_levels
{
  struct stereo_vol master;	/* Master volume */
  struct stereo_vol voc;	/* DSP Voice volume */
  struct stereo_vol fm;		/* FM volume */
  struct stereo_vol line;	/* Line-in volume */
  struct stereo_vol cd;		/* CD audio */
  BYTE mic;			/* Microphone level */
};

/*
 * Mixer parameters for MIXER_IOCTL_SET_PARAMS & MIXER_IOCTL_READ_PARAMS
 */
struct sb_mixer_params
{
  BYTE record_source;		/* Recording source (See SRC_xxx below) */
  FLAG hifreq_filter;		/* Filter frequency (hi/low) */
  FLAG filter_input;		/* ANFI input filter */
  FLAG filter_output;		/* DNFI output filter */
  FLAG dsp_stereo;		/* 1 if DSP is in Stereo mode */
};
#define SRC_MIC         1	/* Select Microphone recording source */
#define SRC_CD          3	/* Select CD recording source */
#define SRC_LINE        7	/* Use Line-in for recording source */


/*
 * Data structure composing an FM "note" or sound event.
 */
struct sb_fm_note
{
  BYTE channel;			/* LEFT, RIGHT, or BOTH */
  BYTE operator;		/* Operator cell (0 or 1) */
  BYTE voice;			/* FM voice (0 to 8) */

  /* Per operator: */
  BYTE waveform;		/* 2 bits: Select wave shape (see WAVEFORM) */
  FLAG am;			/* Amplitude modulation */
  FLAG vibrato;			/* Vibrato effect */
  FLAG do_sustain;		/* Do sustain phase, or skip it */
  FLAG kbd_scale;		/* Keyboard scaling? */

  BYTE harmonic;		/* 4 bits: Which harmonic to generate */
  BYTE scale_level;		/* 2 bits: Decrease output as freq rises*/
  BYTE volume;			/* 6 bits: Intuitive volume */

  BYTE attack;			/* 4 bits: Attack rate */
  BYTE decay;			/* 4 bits: Decay rate */
  BYTE sustain;			/* 4 bits: Sustain level */
  BYTE release;			/* 4 bits: Release rate */

  /* Apply to both operators of one voice: */
  BYTE feedback;		/* 3 bits: How much feedback for op0 */
  FLAG key_on;			/* Set for active, Clear for silent */
  FLAG indep_op;		/* Clear for op0 to modulate op1,
				   Set for parallel tones. */

  /* Freq = 50000 * Fnumber * 2^(octave - 20) */
  BYTE octave;			/* 3 bits: What octave to play (0 = low) */
  unsigned int fnum;		/* 10 bits: Frequency "number" */
};

/*
 * FM parameters that apply globally to all voices, and thus are not "notes"
 */
struct sb_fm_params
{
  BYTE channel;			/* LEFT, RIGHT, or BOTH, as defined above */

  FLAG am_depth;		/* Amplitude Modulation depth (1=hi) */
  FLAG vib_depth;		/* Vibrato depth (1=hi) */
  FLAG wave_ctl;		/* Let voices select their waveform */
  FLAG speech;			/* Composite "Speech" mode (?) */
  FLAG kbd_split;		/* Keyboard split */
  FLAG rhythm;			/* Percussion mode select */

  /* Percussion instruments */
  FLAG bass;
  FLAG snare;
  FLAG tomtom;
  FLAG cymbal;
  FLAG hihat;
};

#endif				/* !def SBLAST_H */
