#pragma C+


/*
 *  TADSMap
 *
 *  An HTML-based mapping libary 
 *  Copyright (c) 2001-2002 Andrew Pontious
 *  All rights reserved. 
 *  Anyone may use and modify this code for their own TADS games.
 *
 *  http://www.umbar.com/TADS/tadsmap/
 *
 *  Version 1.0
 *
 *  July 2002
 *
 */
 
 
 /*
  *  tadsmap.t
  *
  *  Main file for TADSMap support.
  *  Include this in your main game file after adv.t (and std.t, if used)
  *  but before any of your game's code.
  */




/*
 *  Override points so we can take special steps when a game is restored.
 *  Note this wouldn't be necessary if there were a single function that was 
 *  always called after you restore a game.
 *  Note that function is *not* initRestore, which is only called if you're
 *  restoring a game when you first start up your TADS interpreter.
 */
modify restoreVerb

    restoreGame(actor) =
    {
        local result = inherited.restoreGame(actor);
        // Only proceed if restore was successful.
        if(result)
            tadsmap.map_restore;
    }
;
modify strObj

    restoreGame(actor) =
    {
        local result = inherited.restoreGame(actor);
        // Only proceed if restore was successful.
        if(result)
            tadsmap.map_restore;
    }
;




// These are labels for constant values that you should never override.
class tadsmap_defines: object

	// Mapping room algorithm settings defines.
	NORTH = 1
	NORTHEAST = 2
	EAST = 3
	SOUTHEAST = 4
	SOUTH = 5
	SOUTHWEST = 6
	WEST = 7
	NORTHWEST = 8
	
	OPPOSING_DIRECTIONS =
	[
		5,
		6,
		7,
		8,
		1,
		2,
		3,
		4
	]

	ROOM_EXITPROPERTY_STRINGS =
	[
		'north',
		'ne',
		'east',
		'se',
		'south',
		'sw',
		'west',
		'nw'
	]

	ROOM_EXITPROPERTIES =
		[
		    &north,
		    &ne,
		    &east,
		    &se,
		    &south,
		    &sw,
		    &west,
		    &nw
		]

	ROOM_EXITPROPERTY_DISTANCES =
		[
		    &north_distance,
		    &ne_distance,
		    &east_distance,
		    &se_distance,
		    &south_distance,
		    &sw_distance,
		    &west_distance,
		    &nw_distance
		]

	ROOM_EXITPROPERTY_TAKENS =
		[
		    &north_taken,
		    &ne_taken,
		    &east_taken,
		    &se_taken,
		    &south_taken,
		    &sw_taken,
		    &west_taken,
		    &nw_taken
		]

	CLOCKWISE = 1
	COUNTERCLOCKWISE = -1

	ENTRY_HAS_EXIT_FLAGS =
	[
		&has_north_exit,
		&has_ne_exit,
		&has_east_exit,
		&has_se_exit,
		&has_south_exit,
		&has_sw_exit,
		&has_west_exit,
		&has_nw_exit,
		&has_up_exit
	]

	ROOM_EXITPROPERTY_X_DELTA =
	[
		0,
		1,
		1,
		1,
		0,
		-1,
		-1,
		-1
	]
	ROOM_EXITPROPERTY_Y_DELTA =
	[
		-1,
		-1,
		0,
		1,
		1,
		1,
		0,
		-1
	]

	HALLWAY_TYPE_NONE = 0
	HALLWAY_TYPE_N_TO_S = 1
	HALLWAY_TYPE_NE_TO_SW = 2
	HALLWAY_TYPE_E_TO_W = 3
	HALLWAY_TYPE_NW_TO_SE = 4
	
	HALLWAY_TYPE_STRINGS =
	[
		'north to south',
		'northeast to southwest',
		'east to west',
		'northwest to southeast'
	]
	
	EXITPROPERTY_TO_HALLWAY_TYPE =
	[
		self.HALLWAY_TYPE_N_TO_S, 
		self.HALLWAY_TYPE_NE_TO_SW, 
		self.HALLWAY_TYPE_E_TO_W, 
		self.HALLWAY_TYPE_NW_TO_SE,
		self.HALLWAY_TYPE_N_TO_S, 
		self.HALLWAY_TYPE_NE_TO_SW, 
		self.HALLWAY_TYPE_E_TO_W, 
		self.HALLWAY_TYPE_NW_TO_SE	
	]
;

// Change values here to change initial behavior.
class tadsmap_initial: object
	
	init_entries_across = 5
	init_entries_down = 5
	
	map_location = 'LEFT'

	map_percentage_span = 15

	map_use_pixel_span = true
	
	/*
	 *  All subentries must be the same width and the same height
	 *  (though height and width can be different)
	 *  in order for each entry of the map to be the same size and
	 *  therefore for all of them to fit together.
	 *  Change these if you come up with a completely different 
	 *  set of subentries for your map that have a different width/height.
	 */
	map_subentry_pixel_width = 15
	map_subentry_pixel_height = self.map_subentry_pixel_width

	// Pretty much all the logic of TADSMap depends on these proportions.
	map_subentries_across_per_entry = 3
	map_subentries_down_per_entry = self.map_subentries_across_per_entry
	
	default_dir_distance = 1
	
	map_right_left_pixel_buffer = 10
	map_top_bottom_pixel_buffer = 10
	
	should_map = true
	
	image_file_directory = 'tadsmap'

	map_show_controls = true
;

class tadsmap_working: object

	// Working variables needed during mapping process

	// True/nil, whether we're currently mapping.
	// Used by tadsmap_room and subclasses.
	ismapping_flag = nil
	
	/*
	 *  Two-dimensional grid of tadsmap_entry's. 
	 *  Across (horizontal) is represented by elements in map_entries list
	 *  Down (vertical) is represented by the the lists that are *in* the elements
	 *  of the map_entries list.
	 *  Indexes are 1-based.
	 *  So to reach 3 across, 4 down, you would access workentries[3][4].
	 */
	map_entries = []
	
	// Counter for current turn: if tadsmap_room's map_counter is equal to this,
	// it's been checked already this turn.
	map_counter = 0
	
	// Flag that an error has occurred.
	map_error_flag = nil
	map_error_string = ''
	
	map_inited = nil
	
	map_draw_this_turn = nil
	old_location = nil
	
	// Whether GOTO functions are occurring.
	going = nil
;


tadsmap: tadsmap_defines, tadsmap_initial, tadsmap_working

	// Attributes

	map_pixelorpercentage =
	{
		local 
			output_string, 
			temp_number;
	
		// If we're counting pixels
		if(self.map_use_pixel_span)
		{
			// If we're horizontal, we want how far *down* map goes.
			if(self.banner_is_horizontal)
				temp_number =
					self.map_subentry_pixel_height * self.map_subentries_down_per_entry * self.entries_down +
						self.map_top_bottom_pixel_buffer * 2;
			// If we're vertical, we want how far *across* map goes.
			else
				temp_number =
					self.map_subentry_pixel_width * self.map_subentries_across_per_entry * self.entries_across +
						self.map_right_left_pixel_buffer * 2;
			
			output_string = cvtstr(temp_number);
		}
		else
		{
			// TODO
		}
		
		return output_string;
	}

	map_bannerid = 'TADSMap'	

	// Default room values.
	default_room_maptype = tadsmap_room_basic.maptype
	
	default_room_mapcolor = [0, 0, 0]




	// Accessor Functions
	
	
	
	
	
	can_map =
	{
		// Interpreter is new enough to support systemInfo() itself
		// and interpreter supports HTML mode.
		if(systemInfo(__SYSINFO_SYSINFO) == true &&
		   systemInfo(__SYSINFO_HTML) == 1)
		{
			return true;
		}
		else
			return nil;
	}
	
	banner_is_horizontal =
	{
		local alignment = upper(self.map_location);
		if(alignment == 'TOP' || alignment == 'BOTTOM')
			return true;
		else
			return nil;
	}
	banner_is_vertical = 
	{
		return !self.banner_is_horizontal;
	}
	entries_across =
	{
		return length(self.map_entries);
	}
	entries_down =
	{
		return (length(self.map_entries) == 0 ? 0 : length(self.map_entries[1]));
	}
	current_entry_x =
	{
		return self.entries_across/2+1;
	}
	current_entry_y =
	{
		return self.entries_down/2+1;
	}
	center_room =
	{
		return parserGetMe().location;
	}
	center_entry =
	{
		return self.map_entries[self.current_entry_x][self.current_entry_y];
	}
	entry_for_xandy(x, y) =
	{
		if(x >= 1 && x <= self.entries_across && y >= 1 && y <= self.entries_down)
			return self.map_entries[x][y];
		else
			return nil;
	}


    // Combine TADSMap activity and GOTO activity in one accessor.
    ismapping =
    {
        return (global.justTesting && tadsmap.ismapping_flag);
    }




	// Functions





	new_entry_setxandy(x, y) =
	{
		local entry;
		
		entry = self.new_entry;
		entry.setxandy(x, y);
		
		return entry;
	}

	/*
	 *  Method called by other tadsmap methods to dynamically-create a new
	 *  tadsmap_entry instance.
	 *  Override if you want to use a subclass of tadsmap_entry in your own
	 *  code.
	 */
	new_entry =
	{
		return new tadsmap_entry;
	}




	draw_image(source) =
	{
		"<IMG SRC=\"";
		if(self.image_file_directory != nil)
		{
			say(self.image_file_directory); "/";
		}
		say(source);
		"\" BORDER=0 CELLSPACING=0 CELLPADDING=0>";
		
		return true;
	}

	
	
	
	// Sets error message to be shown
	// Always returns nil, signifying error.
	map_error(functionname, errorstring) =
	{
		// If error has already occurred, ignore new one: we'll print first error instead.
		if(self.map_error_flag == nil)
		{
			self.map_error_string = 
				'\b' + functionname + '(): ' +
				errorstring;
			
			self.map_error_flag = true;
		}
	
		// Caller depend on this returning nil so *they* can return nil, signifying error.
		return nil;
	}



	// Called within tadsmap.map() when first run.
	// should not be called from any other location.
	map_init =
	{
		if(self.map_resizeentries(self.init_entries_across, self.init_entries_down) == nil)
			return nil;

		self.map_inited = true;
	
		return true;
	}
	
	
	
    /*
     *  Restoring game logic:
     *  Display doesn't automatically revert to saved game's state; they're independently
     *  owned and operated.
     *  So we have to take steps to make sure TADSMap does the Right Thing in all cases.
     *  This must be called from *all* possible places where a game is restored.
     */
	map_restore =
	{
	    // If we can map at all (if we can't, don't worry about any HTML commands)
	    if(self.can_map == true)
	    {
	        // If we should be mapping per restored game's settings
    	    if(self.should_map == true)
    	    {
    	        // Be sure we draw right now to refresh banner.
    	        self.draw_this_turn = true;
                self.map;
    	    }
            // Otherwise
    	    else
    	    {
        	    // Be sure to turn off banner.
        	    "\H+ <BANNER ID="; say(self.map_bannerid); " REMOVE> \H-";
        	}
        }
	}



	map_error_resizeentries(errorstring) =
	{
		return self.map_error('map_resizeentries', errorstring);
	}

	// Return true on success, nil on error.
	map_resizeentries(new_entries_across, new_entries_down) =
	{
		local 
			i, j, 
			loopmax_outer, loopmax_inner,
			initial_entries_across = self.entries_across,
			initial_entries_down = self.entries_down;
	
		// Parameter checks.
		if(datatype(new_entries_across) != 1)
			return self.map_error_resizeentries('entries-across param is not a number');
		if(datatype(new_entries_down) != 1)
			return self.map_error_resizeentries('entries-down param is not a number');
		if(new_entries_across <= 0)
			return self.map_error_resizeentries('entries-across param is ' + cvtstr(new_entries_across));
		if(new_entries_down <= 0)
			return self.map_error_resizeentries('entries-down param is ' + cvtstr(new_entries_down));


		// Prep outer loop maximum, which is used for case #1 or case #2.
		// Is entries-across total, unless we're going to shrink that in case #3,
		// in which case we only want as many as will be left remaining.
		loopmax_outer = (initial_entries_across > new_entries_across ? new_entries_across : initial_entries_across);

	
		// Case #1:
		// If we want to shrink entries-down (grid height)
		if(new_entries_down < initial_entries_down)
		{
			// Loop through all (or remaining) row elements.
			for(i = 1; i <= loopmax_outer; i++)
			{
				// Loop through first x column elements, where x is
				// the difference between what we have and what we want.
				loopmax_inner = initial_entries_down - new_entries_down;
				
				for(j = 1; j <= loopmax_inner; j++)
				{
					// Delete dynamically-allocated tadsmap_entry referenced by element.
					delete car(self.map_entries[i]);
					// Lop off element from column list 
					// (This, unfortunately, makes a whole new list and recopies it into 
					// row list element).
					self.map_entries[i] = cdr(self.map_entries[i]);
				}

				// If there's anything left in this column.
				loopmax_inner = length(self.map_entries[i]);
				if(loopmax_inner > 0)
				{
					// Loop remaining column elements.
					// Reset x and y.
					for(j = 1; j <= loopmax_inner; j++)
						self.map_entries[i][j].setxandy(i, j);
				}
			}
		}
		// Case #2:
		// If we want to grow entries-down (grid height)
		else if(new_entries_down > initial_entries_down)
		{
			// Loop through all (or remaining) row elements.
			for(i = 1; i <= loopmax_outer; i++)
			{
				// Loop through as many new column list elements as we want to add.
				for(j = initial_entries_down+1; j <= new_entries_down; j++)
					self.map_entries[i] += tadsmap.new_entry_setxandy(i, j);
			}
		}

	
		// Case #3:
		// If we want to shrink entries-across (grid width)
		if(new_entries_across < initial_entries_across)
		{
			// Loop through first x row elements, where x is
			// the difference between what we have and what we want.
			loopmax_outer = initial_entries_across - new_entries_across;
		
			for(i = 1; i <= loopmax_outer; i++)
			{
				// Loop through all column elements for this row element.
				loopmax_inner = length(self.map_entries[i]);
				
				for(j = 1; j <= loopmax_inner; j++)
					// Delete dynamically-allocated tadsmap_entry referenced by element.
					delete self.map_entries[1][j];
				
				// Now delete entire row element.
				self.map_entries = cdr(self.map_entries);
			}

			// Reset coordinates.
			if(length(self.map_entries) > 0 && new_entries_across > 0)
			{
				loopmax_inner = length(self.map_entries[1]);
				for(i = 1; i <= new_entries_across; i++)
				{
					for(j = 1; j <= loopmax_inner; j++)
					{
						self.map_entries[i][j].setxandy(i, j);
					}
				}
			}
		}
		// Case #4:
		// If we want to grow entries-across (grid width)
		else if(new_entries_across > initial_entries_across)
		{
			// Loop through as many new column list elements as we want to add.
			for(i = initial_entries_across+1; i <= new_entries_across; i++)
			{
				// Add new row element.
				self.map_entries += nil;
				self.map_entries[length(self.map_entries)] = [];
			
				// Loop through all column elements.
				for(j = 1; j <= new_entries_down; j++)
					self.map_entries[i] += tadsmap.new_entry_setxandy(i, j);
			}

			// Reset coordinates.
			if(length(self.map_entries) > 0)
			{
				loopmax_inner = length(self.map_entries[1]);
				for(i = 1; i <= new_entries_across; i++)
				{
					for(j = 1; j <= loopmax_inner; j++)
					{
						self.map_entries[i][j].setxandy(i, j);
					}
				}
			}
		}
		
		// Everything went well. Return true.
		return true;
	}




	// Return true on success, nil on error.
	map_resetentries =
	{
		local 
			i, j, 
			width = self.entries_across,
			height = self.entries_down;
		
		for(i = 1; i <= width; i++)
		{
			for(j = 1; j <= height; j++)
			{
				if(self.map_entries[i][j].reset == nil)
					return nil;
			}
		}
		
		return true;
	}



	
	// Called by TMShowVerb
	show =
	{
		if(!self.should_map)
		{
			self.should_map = true;
			self.draw_this_turn = true;
			self.map;
		}
		else
		{
			"[The map is already being shown.]\n";
		}
	}
	// Called by TMHideVerb
	hide =
	{
		if(self.should_map)
		{
			self.should_map = nil;
			"\H+ <BANNER ID="; say(self.map_bannerid); " REMOVE> \H-";
			"[To show the map again, type MAPSHOW]";
		}
		else
		{
			"[The map is already hidden.]\n";
		}
	}	




	// Called by TMAboveVerb
	move_to_above =
	{
		if(self.map_location == 'TOP')
			"[The map is already at the top.]\n";
		else
		{
			"\H+ <BANNER ID="; say(self.map_bannerid); " REMOVE> \H-";
			self.map_location = 'TOP';
			self.draw_this_turn = true;
			self.map;
		}
	}
	// Called by TMBelowVerb
	move_to_below =
	{
		if(self.map_location == 'BOTTOM')
			"[The map is already at the bottom.]\n";
		else
		{
			"\H+ <BANNER ID="; say(self.map_bannerid); " REMOVE> \H-";
			self.map_location = 'BOTTOM';
			self.draw_this_turn = true;
			self.map;
		}
	}
	// Called by TMRightVerb
	move_to_right =
	{
		if(self.map_location == 'RIGHT')
			"[The map is already to the right.]\n";
		else
		{
			"\H+ <BANNER ID="; say(self.map_bannerid); " REMOVE> \H-";
			self.map_location = 'RIGHT';
			self.draw_this_turn = true;
			self.map;
		}
	}
	// Called by TMLeftVerb
	move_to_left =
	{
		if(self.map_location == 'LEFT')
			"[The map is already to the left.]\n";
		else
		{
			"\H+ <BANNER ID="; say(self.map_bannerid); " REMOVE> \H-";
			self.map_location = 'LEFT';
			self.draw_this_turn = true;
			self.map;
		}
	}




	// Called by TMWiderVerb
	make_wider =
	{
		if(self.map_resizeentries(self.entries_across+2, self.entries_down))
		{
			self.draw_this_turn = true;
			self.map;
		}
	}
	// Called by TMNarrowerVerb
	make_narrower =
	{
		if(self.entries_across < 3)
			"[The map is already as narrow as possible.]";
		else
		{
			if(self.map_resizeentries(self.entries_across-2, self.entries_down))
			{
				self.draw_this_turn = true;
				self.map;
			}
		}
	}
	// Called by TMTallerVerb
	make_taller =
	{
		if(self.map_resizeentries(self.entries_across, self.entries_down+2))
		{
			self.draw_this_turn = true;
			self.map;
		}
	}
	// Called by TMShorterVerb
	make_shorter =
	{
		if(self.entries_down < 3)
			"[The map is already as short as possible.]";
		else
		{
			if(self.map_resizeentries(self.entries_across, self.entries_down-2))
			{
				self.draw_this_turn = true;
				self.map;
			}
		}
	}




	// Called by TMControlsVerb
	show_controls =
	{
		if(!self.map_show_controls)
		{
			self.map_show_controls = true;
			self.draw_this_turn = true;
			self.map;
		}
		else
			"[The map's controls are already showing.]\n";
	}
	// Called by TMNoControlsVerb
	hide_controls =
	{
		if(self.map_show_controls)
		{
			self.map_show_controls = nil;
			"[To show controls again, type MAPCONTROLS]";
			self.draw_this_turn = true;
			self.map;
		}
		else
			"[The map's controls are already hidden.]\n";
	}




	// Should be called at ___.
	// (Beginning of turn? After player has moved?)
	map =
	{
		local i, j;

		/*
		 *  Only draw if we *can* draw (HTML is available) 
		 *  and we *should* draw (user has turned on mapping)
		 *  and either we've moved or we've been told to draw explicitly.
		 */
		if(self.can_map == true && 
		   self.should_map == true &&
		   (self.draw_this_turn || old_location != self.center_room))
		{
			old_location = self.center_room;
			self.draw_this_turn = nil;
		
			// Working variables
			self.ismapping_flag = true;
			self.map_error_flag = nil;
			self.map_error_string = '';
			
			self.map_counter++;

			if(self.map_inited == nil)
				self.map_init;

			// Reset all map grid entries.
			map_resetentries;
			
			self.center_entry.is_center = true;
			
			/*
			 *  Fill in all map grid entries by mapping the center room:
			 *  it will search every exit and branch out until all rooms to
			 *  border of grid have been mapped.
			 */
			self.map_maproom(self.center_room, 
							 self.center_entry.map_x,
							 self.center_entry.map_y);


			// Draw map.
			self.map_draw;

			self.ismapping_flag = nil;
		}
	}
	
	
	
	map_draw =
	{
		local 
			i = 1, j = 1,
			across = self.entries_across, down = self.entries_down,
			turn_on_and_off_html = (systemInfo(__SYSINFO_HTML_MODE) == nil);

		if(turn_on_and_off_html)
			"\H+";
		
		"<BANNER ID="; say(self.map_bannerid);
		" ALIGN="; say(self.map_location);
		" ";
		if(self.banner_is_horizontal)
			"HEIGHT";
		else
			"WIDTH";
		
		"="; say(self.map_pixelorpercentage); ">";

		// Outer outer table, containing both outer table and controls
		
		if(self.map_show_controls)
		{
			"<TABLE";
			" BORDER=0 CELLSPACING=0 CELLPADDING=0>";
			"<TR>";
			"<TD>";
		}


		// Outer table, containing both coordinates and map.
		"<TABLE";
		" BORDER=0 CELLSPACING=0 CELLPADDING=0>";

		"<TR>";

		// Empty upper left element.
		"<TD HEIGHT=12 WIDTH=12></TD>";
		
		
		// Horizontal coordinates
		"<TD>";
		"<TABLE";
		" BORDER=0 CELLSPACING=0 CELLPADDING=0>";
		"<TR>";
		for(i = 0; i < self.entries_across; i++)
		{
			"<TD";
			" ALIGN=CENTER WIDTH=";
      		say(self.map_subentry_pixel_width * self.map_subentries_across_per_entry);
			">"; say(i); "</TD>";
		}
		"</TR>";
		"</TABLE>";
		"</TD>";
		
		"</TR>";
		
		"<TR>";
		
		// Vertical coordinates
		"<TD>";
		"<TABLE";
		" BORDER=0 CELLSPACING=0 CELLPADDING=0>";
		for(i = 0; i < self.entries_down; i++)
		{
			"<TR>";
			"<TD";
			" VALIGN=BOTTOM HEIGHT=";
 	      	say(self.map_subentry_pixel_height * self.map_subentries_down_per_entry);
			">"; say(i); "</TD>";
			"</TR>";
		}
		"</TABLE>";
		"</TD>";
		
		
		
		// Lower left-hand element of outer table: Map itself
		"<TD>";
		"<TABLE BORDER=0 CELLSPACING=0 CELLPADDING=0 HEIGHT=";
  		say(self.map_subentry_pixel_height * self.map_subentries_down_per_entry * self.entries_down);
      	" WIDTH=";
  		say(self.map_subentry_pixel_width * self.map_subentries_across_per_entry * self.entries_across);
      	">";
		for(i=1; i <= down; i++)
		{
	      	tadsmap.open_row;
	      	
	      	for(j = 1; j <= across; j++)
	      	{
 	      		"<TD VALIGN=TOP ";
 	      		
 	      		"HEIGHT=";
 	      		say(self.map_subentry_pixel_height * self.map_subentries_down_per_entry);
 	      		" WIDTH=";
 	      		say(self.map_subentry_pixel_width * self.map_subentries_across_per_entry);
 	      		
 	      		">";
	      		
	      		self.map_entries[j][i].draw;
	      		
	      		tadsmap.close_element;
	      	}
	      	tadsmap.close_row;
      	}
      	"</TABLE>";
      	"</TD>";

      	"</TR>";
      	"</TABLE>";


		if(self.map_show_controls)
		{
			"</TD>";
			
			if(self.banner_is_horizontal)
			{
				"<TD>";
				self.draw_controls;
				"</TD>";
			}
			else
			{
				"</TR>";
				"<TR>";
				"<TD>";
				self.draw_controls;
				"</TD>";
			}

			"</TR>";
			"</TABLE>";
		}
		
		"</BANNER>";
		

		if(turn_on_and_off_html)
			"\H-";


      	if(self.map_error_flag == true)
      	{
      		"\n\("; say(self.map_error_string); "\)\n";
      	}
	}
	



	// Gets output of a room's exit property without displaying 
	// any of its text.
	get_exitproperty_output(maproom, exitproperty) =
	{
		if(datatype(maproom) != 2)
			return nil;
		else
			return maproom.exitproperty_output(exitproperty);
	}
	
	
	
	map_error_maproom(errorstring) =
	{
		return self.map_error('map_maproom', errorstring);
	}

	// Returns nil on any problems, true on success.
	map_maproom(maproom, x, y) =
	{
		local 
			i = 1, j,
			mapentry = nil, 
			exitproperty_index, exitproperty_output,
			distance_to, distance_from,
			distance_x, distance_y, distance_entry;
	
		// Parameter checks.
		// - Is maproom param an object?
		if(datatype(maproom) != 2)
			return self.map_error_maproom('room param is datatype ' + 
										  cvtstr(datatype(maproom)) + 
										  ', not an object');
		// - Is maproom param object a tadsmap_room?
		if(isclass(maproom, tadsmap_room) == nil)
			return self.map_error_maproom('room param is not an of class tadsmap_room');


		// Find corresponding tadsmap_entry, if there is one.
		mapentry = self.entry_for_xandy(x, y);
		// If there isn't one, return true: no work to do here.
		if(mapentry == nil)
			return true;
		else mapentry.map_room = maproom;


		// Update counter to show that we've touched this room already
		maproom.map_counter = self.map_counter;
		
		
		//
		// Loop through cardinal directions.
		//

		// Start loop where room indicates: default is north.
		exitproperty_index = maproom.map_initialexitproperty;
		for(i = 1; i <= 8; i++)
		{
			// Get output of this cardinal direction's exit property (north, ne, etc.)
			// without showing any of its text.
			exitproperty_output = 
				tadsmap.get_exitproperty_output(maproom, self.ROOM_EXITPROPERTIES[exitproperty_index]);
			
			// If an object, should be a room.
			if(datatype(exitproperty_output) == 2)
			{
				// Check that it inherits from tadsmap_room.
				if(isclass(exitproperty_output, tadsmap_room) == nil)
				{
					return self.map_error_maproom('The reference in exit property \'' +
												  tadsmap.ROOM_EXITPROPERTY_STRINGS[exitproperty_index] +
												  '\' of room "' + maproom.sdesc_as_string +
												  '" at grid coordinates (' +
												  cvtstr(mapentry.map_x) + ', ' +
												  cvtstr(mapentry.map_y) +
												  '), is of an object, but that object doesn\'t derive from tadsmap_room.');
				}

				// Mark an exit in this direction.
				mapentry.(self.ENTRY_HAS_EXIT_FLAGS[exitproperty_index]) = true;

				// Figure out offset from map that this room is at.
				
				// Distance from this room to that room.
				distance_to = maproom.(self.ROOM_EXITPROPERTY_DISTANCES[exitproperty_index]);
				// To doublecheck, get distance that other room thinks it is to *this* room.
				// if they're different, flag an error.
				distance_from = 
					exitproperty_output.(self.ROOM_EXITPROPERTY_DISTANCES[self.OPPOSING_DIRECTIONS[exitproperty_index]]);
				if(distance_to != distance_from)
				{
					return self.map_error_maproom('Distance mismatch: room "' +
												  maproom.sdesc_as_string + 
												  '" thinks that the room to the ' +
												  tadsmap.ROOM_EXITPROPERTY_STRINGS[exitproperty_index] +
												  ' of it,  room "' + exitproperty_output.sdesc_as_string +
												  '", is ' + cvtstr(distance_to) +
												  ' away. But room "' + 
												  exitproperty_output.sdesc_as_string +
												  '" thinks that the room to its ' +
												  tadsmap.ROOM_EXITPROPERTY_STRINGS[self.OPPOSING_DIRECTIONS[exitproperty_index]] +
												  ' ("' + maproom.sdesc_as_string +
												  '") is ' + cvtstr(distance_from) +
												  ' away.');
				}
				
				// If room has already been seen by user *and* the hallway has been taken from either end *or* it's only 1 long, 
				// then attempt to (a) trace path to it and (b) map the room itself.
				if( exitproperty_output.isseen == true &&
				    (distance_to == 1 ||
				     maproom.(self.ROOM_EXITPROPERTY_TAKENS[exitproperty_index]) == true || 
				     exitproperty_output.(self.ROOM_EXITPROPERTY_TAKENS[self.OPPOSING_DIRECTIONS[exitproperty_index]]) == true) )
				{
					// Loop through each step to that other room.
					for(j = 1; j <= distance_from; j++)
					{
						// Figure out new coordinate.
						distance_x = x + j * self.ROOM_EXITPROPERTY_X_DELTA[exitproperty_index];
						distance_y = y + j * self.ROOM_EXITPROPERTY_Y_DELTA[exitproperty_index];
						
						// Figure out if coordinate is within entries grid.
						distance_entry = self.entry_for_xandy(distance_x, distance_y);

						// If not, stop loop
						if(distance_entry == nil)
						{
							break;
						}
						// If so...
						else
						{
							// If we're not yet at the other room.
							if(j < distance_from)
							{
								// If hallway has been set, assume it's set all the way to the room and the
								// room is set, so stop.
								if(distance_entry.has_been_set)
								{
									break;
								}
								/*
								 *  If hallway hasn't already been set... (it may have been set from mapping
								 *  logic going in other direction).
								 *  NOTE: The *whole point* of the tracing hallway-route to a room that may
								 *  already have been mapped is to *also* be sure the hallway itself is mapped
								 *  from either direction.
								 */
								else
								{
									// Set entry as "non-room hallway" of given type:
									// n-s, e-w, nw-se, ne-sw.
									distance_entry.hallway_type = self.EXITPROPERTY_TO_HALLWAY_TYPE[exitproperty_index];
								}
							}
							// If we are at other room.
							else
							{
								// If room in this direction has already been mapped...
								if(exitproperty_output.map_counter == self.map_counter)
								{
									// ...do nothing.
								}
								// If room has not yet been mapped, recurse to map the other room.
								else if(self.map_maproom(exitproperty_output, distance_x, distance_y) == nil)
								{
									return nil;
								}
							}
						}
					}
				}
			}
			/*
			 *  If returned true, there's an exit in this direction, but
			 *  map shouldn't show the actual destination, just the exit itself.
			 *  (Good for exits that would end game if taken, etc.)
			 */
			else if(datatype(exitproperty_output) == 8)
			{
				// Mark an exit in this direction.
				mapentry.(self.ENTRY_HAS_EXIT_FLAGS[exitproperty_index]) = true;
			
				// Don't proceed further with this "room".
			}
			// If null or anything else, no exit in that direction.
			else //if(datatype(exitproperty_output) == 1)
			{
				// Don't mark an exit in this direction.
			
				// Don't proceed further with this "room".
			}


			// Update values needed for next iteration of loop.
			
			// Which direction to travel in to get to next iteration's exit property.
			// (This is a room setting, normally clockwise.)
			exitproperty_index += maproom.map_searchdirection;
			// Bounds checking for above.
			if(exitproperty_index == 0)
				exitproperty_index = length(self.ROOM_EXITPROPERTIES);
			else if(exitproperty_index > length(self.ROOM_EXITPROPERTIES))
				exitproperty_index = 1;
		}
		
		
		//
		// Handle other directions.
		//

		if(tadsmap.get_exitproperty_output(maproom, &up) != nil)
			mapentry.has_up_exit = true;
		if(tadsmap.get_exitproperty_output(maproom, &down) != nil)
			mapentry.has_down_exit = true;
		if(maproom.map_has_special_symbol != nil)
			mapentry.has_special = true;
			
		
		return true;
	}


	draw_controls =
	{
		// Hide
		"<A HREF='mhide'>hide</A>";
		"\n";

		"\b";

		// Dimensions
		"<A HREF='mwide'>make wider</A>";
		"\n";
		if(self.entries_across >= 3)
		{
			"<A HREF='mnarrow'>make narrower</A>";
			"\n";
		}
		"<A HREF='mtall'>make taller</A>";
		"\n";
		if(self.entries_down >= 3)
		{
			"<A HREF='mshort'>make shorter</A>";
			"\n";
		}

		"\b";

		// Orientation
		if(self.map_location != 'TOP')
		{
			"<A HREF='mabove'>move to top</A>";
			"\n";
		}
		if(self.map_location != 'BOTTOM')
		{
			"<A HREF='mbelow'>move to bottom</A>";
			"\n";
		}
		if(self.map_location != 'LEFT')
		{
			"<A HREF='mleft'>move to left</A>";
			"\n";
		}
		if(self.map_location != 'RIGHT')
		{
			"<A HREF='mright'>move to right</A>";
			"\n";
		}
		
		// Directions
		
		"<TABLE";
		" BORDER=0 CELLSPACING=0 CELLPADDING=0>";

		"<TR>";
		"<TD>";
			if(self.center_room.has_exit(&nw))
				"<A HREF='nw'>nw</A> ";
		"</TD>";
		"<TD>";
			if(self.center_room.has_exit(&north))
				"<A HREF='n'>n</A> ";
		"</TD>";
		"<TD>";
			if(self.center_room.has_exit(&ne))
				"<A HREF='ne'>ne</A> ";
		"</TD>";
		"</TR>";

		"<TR>";
		"<TD>";
			if(self.center_room.has_exit(&west))
				"<A HREF='w'>w</A> ";
		"</TD>";
		"<TD>";
			if(self.center_room.has_exit(&up))
				"<A HREF='up'>up</A> ";
			if(self.center_room.has_exit(&down))
				"<A HREF='down'>down</A> ";
		"</TD>";
		"<TD>";
			if(self.center_room.has_exit(&east))
				"<A HREF='e'>e</A> ";
		"</TD>";
		"</TR>";

		"<TR>";
		"<TD>";
			if(self.center_room.has_exit(&sw))
				"<A HREF='sw'>sw</A> ";
		"</TD>";
		"<TD>";
			if(self.center_room.has_exit(&south))
				"<A HREF='s'>s</A> ";
		"</TD>";
		"<TD>";
			if(self.center_room.has_exit(&se))
				"<A HREF='se'>se</A> ";
		"</TD>";
		"</TR>";
		"</TABLE>";
	}



	open_row =
	{
		"<TR>";
	}
	close_row =
	{
		"</TR>";
	}
	open_element =
	{
		"<TD>";
	}
	close_element =
	{
		"</TD>";
	}
;


#pragma C-


#include "tadsmap_entry.t"
#include "tadsmap_room.t"
#include "tadsmap_verbs.t"
#include "tadsmap_preparse.t"
#include "GOTO.T"