;;;; Pseudo-Random inexact real numbers for scheme.
;;; Copyright (C) 1991 Aubrey Jaffer.

;This file is loaded by random.scm if inexact numbers are supported by
;the implementation.

;  (random:uniform state)				procedure

;Returns an uniformly distributed inexact real random number in the
;range between 0 and 1.

;  (random:solid-sphere! <vect>)			procedure
;  (random:solid-sphere! <vect> state)			procedure

;Fills <vect> with inexact real random numbers the sum of whose
;squares is less than 1.0.  Thinking of <vect> as coordinates in space
;of dimension n = (vector-length <vect>), the coordinates are
;uniformly distributed within the unit n-shere.  The sum of the
;squares of the numbers is returned.

;Random:solid-sphere! is inefficient for large <vect>.

;  (random:hollow-sphere! <vect>)			procedure
;  (random:hollow-sphere! <vect> state)			procedure

;Fills <vect> with inexact real random numbers the sum of whose
;squares is equal to 1.0.  Thinking of <vect> as coordinates in space
;of dimension n = (vector-length <vect>), the coordinates are
;uniformly distributed over the surface of the unit n-shere.

;Random:hollow-sphere! is inefficient for large <vect>.

;  (random:normal)					procedure
;  (random:normal state)				procedure

;Returns an inexact real in a normal distribution with mean 0 and
;standard deviation 1.  For a normal distribution with mean M and
;standard deviation D use (+ M (* D (random:normal))).

;  (random:exp)						procedure
;  (random:exp state)					procedure

;Returns an inexact real in an exponential distribution with mean 1.
;For an exponential distribution with mean U use (* U (random:exp)).
;;;;-----------------------------------------------------------------

(define random:float-radix
  (+ 1 (exact->inexact random:MASK)))

;;; This determines how many chunks will be neccessary to completely
;;; fill up an inexact real.
(define (random:size-float l x)
  (if (= 1.0 (+ 1 x))
      l
      (random:size-float (+ l 1) (/ x random:float-radix))))
(define random:chunks/float (random:size-float 1 1.0))

(define (random:uniform-chunk n state)
  (if (= 1 n)
      (/ (exact->inexact (random:chunk state))
	 random:float-radix)
      (/ (+ (random:uniform-chunk (- n 1) state)
	    (exact->inexact (random:chunk state)))
	 random:float-radix)))

;;; Generate an inexact real between 0 and 1.
(define (random:uniform state)
  (random:uniform-chunk random:chunks/float state))

(define (random:solid-sphere! vect . args)
  (let ((state (if (null? args) *random-state* (car args)))
	(x 0)
	(x2 0)
	(ms 0))
    (do ((n (vector-length vect) (- n 1)))
	((zero? n))
      (set! x (- (* 2 (random:uniform state)) 1))
      (vector-set! vect (- n 1) x)
      (set! x2 (* x x))
      (set! ms (+ ms x2)))
    (do ((i 0 (modulo (- i 1) (vector-length vect))))
	((<= ms 1.0) ms)
      (set! ms (- ms x2))
      (set! x (random:uniform state))
      (vector-set! vect i x)
      (set! x2 (* x x))
      (set! ms (+ ms x2)))))

(define (random:hollow-sphere! vect . args)
  (let ((ms (sqrt (apply random:solid-sphere! vect args))))
    (do ((n (vector-length vect) (- n 1)))
	((zero? n))
      (vector-set! vect (- n 1) (/ (vector-ref vect (- n 1)) ms)))))

;;; These two distributions from algorithms in Knuth Vol. II.

(define (random:normal . args)
  (let* ((v (make-vector 2 0.0))
	 (s (apply random:solid-sphere! v args)))
    (* (vector-ref v 0)
       (sqrt (/ (* -2 (log s)) s)))))

(define (random:exp . args)
  (let ((state (if (null? args) *random-state* (car args))))
    (- (log (random:uniform state)))))

(require 'random)
