package CryptoTron::ParseAccount;

# Load the Perl pragmas.
use 5.010000;
use strict;
use warnings;

# Load the Perl pragma Exporter.
use vars qw(@ISA @EXPORT @EXPORT_OK);
use Exporter 'import';

# Exporting the implemented subroutine.
our @EXPORT = qw(
    TotalBalance
    FreeBalance
    FrozenBalance
);

# Base class of this (tron_addr) module.
our @ISA = qw(Exporter);

# Set the package version. 
our $VERSION = '0.01';

# Load the required Perl module.
use JSON::PP;

# Set the variable $SUN.
our $SUN = 1000000;

# ---------------------------------------------------------------------------- #
# Subroutine TotalBalance()                                                    #
#                                                                              #
# Description:                                                                 #
# Parse the JSON account data and retrieve the total balance.                  #
#                                                                              #
# @argument $json_data      Raw JSON data  (scalar)                            #
# @return   $total_balance  Total balance  (scalar)                            #
# ---------------------------------------------------------------------------- #
sub TotalBalance {
    # Assign the argument to the local variable.
    my $json_data = $_[0];
    # Set the variable $json.
    my $json = 'JSON::PP'->new->pretty;
    # Decode the JSON data to get a valid hash.
    my $decoded = $json->decode($json_data);
    # Set the array with the keys.
    my @keys = ('balance', 'frozen', 'frozen_balance',
                'account_resource', 'frozen_balance_for_energy');
    # Get the frozen account data.
    my $free = $decoded->{$keys[0]};
    my $frozen = $decoded->{$keys[1]}[0]{$keys[2]};
    my $energy = $decoded->{$keys[3]}{$keys[4]}{$keys[2]};
    # Calculate the total balance.
    my $total_balance = ($free + $frozen + $energy) / $SUN;
    # Return the total balance.
    return $total_balance;
};

# ---------------------------------------------------------------------------- #
# Subroutine FreeBalance()                                                     #
#                                                                              #
# Description:                                                                 #
# Parse the JSON account data and retrieve the free balance.                   #
#                                                                              #
# @argument $content       Raw json content  (scalar)                          #
# @return   $free_balance  Free balance      (scalar)                          #
# ---------------------------------------------------------------------------- #
sub FreeBalance {
    # Assign the argument to the local variable.
    my $json_data = $_[0];
    # Set the variable $json.
    my $json = 'JSON::PP'->new->pretty;
    # Decode the JSON data to get a valid hash.
    my $decoded = $json->decode($json_data);
    # Set the key.
    my $key = ('balance');
    # Get the free balance.
    my $free = $decoded->{$key};
    # Calculate the free balance.
    my $free_balance = $free / $SUN;
    # Return the free balance.
    return $free_balance;
};

# ---------------------------------------------------------------------------- #
# Subroutine FrozenBalance()                                                   #
#                                                                              #
# Description:                                                                 #
# Parse the JSON account data and retrieve the frozen balance.                 #
#                                                                              #
# @argument $content         Raw json content  (scalar)                        #
# @return   $frozen_balance  Frozen balance    (scalar)                        #
# ---------------------------------------------------------------------------- #
sub FrozenBalance {
    # Assign the argument to the local variable.
    my $json_data = $_[0];
    # Set the variable $json.
    my $json = 'JSON::PP'->new->pretty;
    # Decode the JSON data to get a valid hash.
    my $decoded = $json->decode($json_data);
    # Set the $keys.
    my @keys = ('balance', 'frozen', 'frozen_balance',
                'account_resource', 'frozen_balance_for_energy');
    my $frozen = ($decoded->{$keys[1]}[0]{$keys[2]}) / $SUN;
    my $energy = ($decoded->{$keys[3]}{$keys[4]}{$keys[2]}) / $SUN;
    # Calculate the frozen balance.
    my $total_frozen = $frozen + $energy;
    # Return the frozen balance.
    return $total_frozen;
};

1;

__END__

=head1 NAME

CryptoTron::GetAccount - Perl extension for use with the blockchain of the crypto coin Tron.

=head1 SYNOPSIS

  use CryptoTron::ParseAccount;

  # Declare the variable $balance.
  my $balance;

  # Set the JSON data.
  my $json_data = '{"balance": 1000000000, "frozen": [{"frozen_balance": 2000000000}],
                    "account_resource": {"frozen_balance_for_energy": {"frozen_balance": 300000000}}}';

  # Get the total balance..
  $balance = TotalBalance($json_data);
  print $balance;

  # Get the free balance..
  $balance = FreeBalance($json_data);
  print $balance;

  # Get the free balance..
  $balance = FrozenBalance($json_data);
  print $balance;

=head1 DESCRIPTION

The module consists of methods for parsing raw JSON data. The raw JSON can 
come from module CryptoTron::GetAccount or other sources.

=head1 SEE ALSO

JSON::PP;

=head1 AUTHOR

Dr. Peter Netz, E<lt>ztenretep@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2022 by Dr. Peter Netz

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.30.0 or,
at your option, any later version of Perl 5 you may have available.

=cut
