#-----------------------------------------------------------------
# SADI::Data::Def::OWLClass
# Author: Edward Kawas <edward.kawas@gmail.com>,
# For copyright and disclaimer see below.
#
# $Id: OWLClass.pm,v 1.21 2009-11-12 21:17:02 ubuntu Exp $
#-----------------------------------------------------------------
package SADI::Data::Def::OWLClass;
use base qw( SADI::Base );
use strict;

# add versioning to this module
use vars qw /$VERSION/;
$VERSION = sprintf "%d.%02d", q$Revision: 1.21 $ =~ /: (\d+)\.(\d+)/;

=head1 NAME

SADI::Data::Def::OWLClass - definition of an owl class

=head1 SYNOPSIS

 use SADI::Data::Def::OWLClass;

 # create a new data type
 my $class = new SADI::Data::Def::OWLClass
    ( name        => 'MySequenceClass',
      type        => 'http://some.domain.com/classes#MySequenceClass',
      parent      => 'http://some.domain.com/classes#MySequenceClassParent',
    );

 # get the name of this owl class
 print $class->name;


=cut

=head1 DESCRIPTION

A container representing an OWL class definition

=head1 AUTHORS

 Edward Kawas (edward.kawas [at] gmail [dot] com)

=cut

#-----------------------------------------------------------------
# A list of allowed attribute names. See SADI::Base for details.
#-----------------------------------------------------------------

=head1 ACCESSIBLE ATTRIBUTES

Details are in L<SADI::Base>. Here just a list of them:

=over

=item B<name>

A name of this owl class

=item B<parent>

A parent for this owl class ... defaults to SADI::Data::OWL::Class

=item B<type>

The type of this owl class

=back

=cut
{
	my %_allowed = (
		name => {
			# name set when you set the type!
			type => SADI::Base->STRING,
		},
		parent => {
			type => SADI::Base->STRING,
			post => sub {
				my ($self) = shift;
				my $name = @{$self->parent}[-1];
				return unless defined $name;
#				$name = 'SADI::Data::OWL::Class' unless defined $name;
				if ($name eq 'SADI::Data::OWL::Class') {
					# add to parents array
                    $self->add_module_parent($name);
				} else {
				    # add to parents array
				    $self->add_module_parent($self->owlClass2module( $self->uri2package($name) ));
				}
			},
			is_array => 1,
		},
		type => {
			type => SADI::Base->STRING,
			post => sub {
				my ($self) = shift;
				my $package = $self->uri2package($self->type);
				$self->{module_name} = $self->owlClass2module($package);
				# extract our name and set it
				my $name = $1 if $package =~ m|\:\:(\w+)$|gi;
				$name = $package unless $name; 
				$self->name($name);
				# bnodes are typed differently than URI nodes
				$self->{type} = "" if $package =~ m/^genid/ig;
			  }
		},
		object_properties =>
		  { type => 'SADI::Data::Def::ObjectProperty', is_array => 1 },
		datatype_properties =>
		  { type => 'SADI::Data::Def::DatatypeProperty', is_array => 1 },

# used internally  (but cannot start with underscore - Template would ignore them)
        # the full package name for this class
		module_name   => undef,
		# the full package names for parents
		module_parent => { type => SADI::Base->STRING, is_array => 1 },
	);

	sub _accessible {
		my ( $self, $attr ) = @_;
		exists $_allowed{$attr} or $self->SUPER::_accessible($attr);
	}

	sub _attr_prop {
		my ( $self, $attr_name, $prop_name ) = @_;
		my $attr = $_allowed{$attr_name};
		return ref($attr) ? $attr->{$prop_name} : $attr if $attr;
		return $self->SUPER::_attr_prop( $attr_name, $prop_name );
	}
}

#-----------------------------------------------------------------

=head1 SUBROUTINES

=cut
#-----------------------------------------------------------------
# init
#-----------------------------------------------------------------
sub init {
	my ($self) = shift;
	$self->SUPER::init();
	$self->add_parent('SADI::Data::OWL::Class');
}
1;
__END__
