/*  tzinit.c -- FreeMiNT.
    Copyright (C) 1998 Guido Flohr <gufl0000@stud.uni-sb.de>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*  $Header$  */

/* Define to 1 if your struct tm has a member tm_gmtoff.  */
/* #undef HAVE_TM_GMTOFF */

/*  Set the system timezone according to the TZ environment variable.
 *  This program should be run from mint.cnf if and only if your
 *  system clock is ticking in local time (which is deprecated).
 */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <mintbind.h>
#include <unistd.h>

char* myname = "tzinit";

/* Kludge for broken time.h in MiNTLib.  */
extern char* tzname[2];

extern char* getenv ();

struct kernel_timezone {
  long tz_minuteswest;
  long tz_dstflag;
};

int main (int argc, char* argv[])
{
  struct kernel_timezone tz;
  time_t t = time (&t);
  struct tm* tm = localtime (&t);
  char* mint_clockmode = getenv ("MINT_CLOCKMODE");
  long clock_mode = 0;  /* Let clock tick in UTC.  */
  
  if (argc != 0 && argv[0][0] != '\0')
    myname = argv[0];
  
#ifndef __USE_BSD
# define tm_gmtoff __tm_gmtoff
#endif

  if (tm == NULL) {
    fflush (stdout);
    fprintf (stderr, "%s: localtime failed:\n", myname);
    return 1;
  }
  
  if (mint_clockmode)
    if (strcmp ("GMT", mint_clockmode) && strcmp ("UTC", mint_clockmode))
      clock_mode = 1;

#ifdef HAVE_TM_GMTOFF
  tz.tz_minuteswest = -(tm->tm_gmtoff) / 60;
#else
  /* This may seem awkward but I think it is the only possible way
   * with the MiNTLib.  As far as I can see it the (undocumented)
   * global variable _timezone has not the same meaning as 
   * timezone in a POSIX libc because it doesn't reflect if DST
   * is in use.  Because the whole time stuff in the MiNTLib
   * screams for a re-write from the scratch we don't want to
   * depend on a particular patchlevel.  Better go the thorny way.
   */
  {
    struct tm localtm = *tm;
    struct tm gmtm;
    time_t loc, gm;
    
    tm = gmtime (&t);
    if (tm == NULL) {
      fflush (stdout);
      fprintf (stderr, "%s: gmtime failed\n", myname);
      return 1;
    }
    gmtm = *tm;
    loc = mktime (&localtm);
    gm = mktime (&gmtm);
    if (loc == (time_t) -1 || gm == (time_t) -1) {
      fflush (stdout);
      fprintf (stderr, "%s: mktime failed\n", myname);
      return 1;
    }
    tz.tz_minuteswest = (gm - loc) / 60;
    tm->tm_isdst = localtm.tm_isdst;
  }
#endif

  tz.tz_dsttime = 0;

  printf ("%s: setting system timezone information.\n", myname);
  printf ("Timezone in use: %s.\n", tm->tm_isdst > 0 ? tzname[1] : tzname[0]);
  printf ("Offset to Greenwich Mean Time: %d minutes.\n", tz.tz_minuteswest);
  printf ("Daylight savings time in use: %s.\n", 
      tm->tm_isdst > 0 ? "yes" :
      tm->tm_isdst == 0 ? "no" : "not determinable");
  printf ("Kernel clock ticks in %s.\n", clock_mode ? "local time." : "UTC");

  if ((errno = -(Ssystem (100, clock_mode, 0))) != 0) {
    fflush (stdout);
    fprintf (stderr, "%s: warning: Ssystem failed: %s\n", myname, strerror (errno));
  }
  if ((errno = -(Tsettimeofday (NULL, &tz))) != 0) {
    fflush (stdout);
    fprintf (stderr, "%s: Tsettimeofday failed: %s\n", myname, strerror (errno));
    return 1;
  }
  return 0;
}
