/*
 * IBM PC VGA/SVGA driver for starchart 3.0 by Doug MCDonald
 * 
 * Uses my own public domain graphics package, included. This package works on
 * EGA VGA and super VGA cards. Its upper-level routines draw points, lines,
 * ellipses, filled ellipses, filled rectangles and text. The "symbol"
 * routine draws "text" that is defined by a user-supplied string of bytes,
 * which are painted onto the screen from top to bottom. The "zsetup" and
 * "curmod" routines must bracket any drawing done with this package. If you
 * exit the program or try to use BIOS text between them you get garbage.
 * 
 * 
 * Development was done with a Dell 310 20MHz 386, Microsoft C 5.1, a Video 7
 * VGA16 display card, and (thank goodness) an 80387 numeric co-processor, at
 * with a 18 msec disk.
 * 
 * 
 * The Microsoft library is pretty compatible with U**x System V, so compile
 * everything with the SYSV flag set and you will be OK.
 * 
 * The main glitch is the file "con.locs". You have absolutely GOT to rename
 * this file under DOS, otherwise it tries to talk to the CONsole. I use
 * "cons.loc", which is fairly close. Renaming the built in file names is
 * done by modifying pcstar.h.
 * 
 * 
 * Copyright (c) 1990 by Doug McDonald, Tom Horsley, and Craig Counterman.
 * All rights reserved.
 * StarChart Version 3.2 copyright (c) March 1990 by Craig Counterman 
 * original StarChart Software Suite copyright (c) 1987 by Alan Paeth
 *
 * All rights reserved. Redistribution granted for non-commercial
 * non-profit use only. Disclaimer: users of this work understand that
 * (a) the authors' cannot undertake to support this software (b) users
 * agree to acknowledge the use of the software in any published work
 * arising from its application and (c) any subsequent redistribution of
 * this work retains this warranty placard. 
 *
 * The graphics library is public domain, and may be used any way you wish.
 * 
 * No representation is made about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty, to
 * the extent permitted by applicable law.
 * 
 *  The followinf file will compile and link it:

masm svgagraf
cl -c -DSYSV -Dindex=strchr -AL -Gt32 starcust.c interact.c parse_in.c
cl -c -DSYSV -Dindex=strchr -AL -Gt32 staribmv.c starsupp.c readfile.c
cl -c -DSYSV -Dindex=strchr -AL -Gt32 -DINTERACTIVE_CONTROL starm2.c starmain.c 
link staribmv+readfile+starcust+interact+parse_in+starsupp+starm2+starmain+svgagraf;
exemod staribmv.exe /stack 4200

 *  for plain VGA  add -DPLAIN_VGA in the line that compiles staribmv
 * 
 */

/* for plain VGA  #define PLAIN_VGA */ 


#include <stdio.h>
#include <math.h>
#ifndef SYSV
#include <strings.h>
#else
#include <string.h>
#endif
#include <ctype.h>

#include <dos.h>

#include "star3.h"


int             yscale1;
long            yscale2;
long            xscale1;
int             control_mode = -1;
int             yrad[7] = {7, 6, 5, 4, 3, 2, 1};
int             xrad[7] = {7, 6, 5, 4, 3, 2, 1};


static int      xlast, ylast;
#define scx(x) ((int)((((long)(x))*xscale1)/1000L))
#define scy(y) (yscale1-((int)((((long)(y))*yscale2)/10000L)))


extern char    *title;	/* Title of page */

extern mapwindow *mapwin[MAXWINDOWS];
extern int      numwins;

extern int      cur_function;
extern int      cur_map_type;
extern int      cur_map_tag;
extern char    *cur_tag_field;

/*
 * Set by initxform One could use elements of the mapwindow structure, but
 * these should be faster for the current window
 */
extern int      xproj_mode;
extern double   xwest, xeast, xnorth, xsouth, xbot;
extern int      cenx, ceny, strty;
extern double   xracen, sindlcen, cosdlcen, chart_scale;
extern double   yscale;
extern double   xc_scale;
extern double   inv_;


/*
 * Scale multiplier, minimum, mangitude change, maximum, for thumbnail,
 */
#define THSMUL 1.2
#define THSMIN 10.0
#define THMADJ 2.5
#define THMMAX 5.0

#define MIN_MAG		(-1.5)	/* all magnitude below become this */
#define MAX_MAG		(11.4)	/* all magnitude above become this */


/* Exports */

/* The variables in the first few lines MUST be set by driver */
mapwindow       fullpage = {
    880, 700, 20, 65,	/* width, height, x and y offsets */
    5.9, 2.0, 2.05,	/* default limiting mags for glyph, name, label */

    /*
     * The next several variables SHOULD be set by the driver, but are only
     * used by the driver
     */
    FULLPAGEMAP,	/* Type of map: THUMBNAIL may have some restrictions */
    0,	   /* May be used by driver for whatever */
    "String",	/* May be used by driver for whatever */

    /*
     * The next several variables may be set by the driver, but the main
     * routines may reset them (and the driver routines may then override
     * that)
     */
    SANSONS,	/* Projection mode */
    FALSE, FALSE,	/* Draw grids */
    0.5, 5.0,	/* grid step size */
    0.0, 0.0,	/* grid origin */

    FALSE, /* Invert (flip north south) */
};

/* The variables in the first few lines MUST be set by driver */
mapwindow       mainmap = {
    880, 500, 20, 265,	/* width, height, x and y offsets */
    5.9, 2.0, 2.05,	/* default limiting mags for glyph, name, label */

    /*
     * The next several variables SHOULD be set by the driver, but are only
     * used by the driver
     */
    MAINMAP,	/* Type of map: THUMBNAIL may have some restrictions */
    0,	   /* May be used by driver for whatever */
    "String",	/* May be used by driver for whatever */

    /*
     * The next several variables may be set by the driver, but the main
     * routines may reset them (and the driver routines may then override
     * that)
     */
    SANSONS,	/* Projection mode */
    FALSE, FALSE,	/* Draw grids */
    0.5, 5.0,	/* grid step size */
    0.0, 0.0,	/* grid origin */

    FALSE, /* Invert (flip north south) */
};


/* The variables in the first few lines MUST be set by driver */
mapwindow       thumbmap = {
    480, 195, 420, 35,	/* width, height, x and y offsets */
    3.0 + THMADJ, 1.0 + THMADJ, 2.05 + THMADJ,
    /* default limiting mags for glyph, name, label */

    /*
     * The next several variables SHOULD be set by the driver, but are only
     * used by the driver
     */
    THUMBNAIL,	/* Type of map: THUMBNAIL may have some restrictions */
    0,	   /* May be used by driver for whatever */
    "String",	/* May be used by driver for whatever */

    /*
     * The next several variables may be set by the driver, but the main
     * routines may reset them (and the driver routines may then override
     * that)
     */
    SANSONS,	/* Projection mode */
    FALSE, FALSE,	/* Draw grids */
    0.5, 5.0,	/* grid step size */
    0.0, 0.0,	/* grid origin */

    FALSE, /* Invert (flip north south) */
};

/* h & v tick text controls */
int             htick_lim = 2;
int             htext_lim = 80;
int             htext_xoff = -29;	/* was +2 */
int             htext_yoff = 27;
int             vtick_lim = 2;
int             vtext_lim = 20;
int             vtext_xoff = 24;
int             vtext_yoff = 0;

/* externs for labels */
int             x_nameoffset = 10, y_nameoffset = 0;
int             x_lbloffset = 0, y_lbloffset = 10;
int             x_magoffset = 7, y_magoffset = -15;

/* externs for legend: variables of positioning are here */
int             l_til = 220;
int             l_stil = 185;

int             l_lmar1 = 40;
int             l_lmar2 = 65;
int             l_ltext = 95;
int             l_rmar1 = 205;
int             l_rmar2 = 230;
int             l_rtext = 260;

int             l_line1 = 150;
int             l_line2 = 125;
int             l_line3 = 100;
int             l_line4 = 75;
int             l_line5 = 50;
int             l_line6 = 25;

/* Point sizes for font calls */
int             titlesize = 16;
int             subtlsize = 12;
int             namesize = 10;
int             lblsize = 8;
int             magsize = 8;

/* Fonts for font calls */
int             namefnt = TIMESROMAN;
int             lblfnt = HELV;
int             magfnt = COURIER;
int             titlefnt = TIMESBOLD;
int             subtlfnt = TIMESROMAN;

/*
 * Scale multiplier, minimum, mangitude change, maximum, for thumbnail,
 */
double          th_smul = THSMUL;
double          th_smin = THSMIN;
double          th_madj = THMADJ;
double          th_mmax = THMMAX;

#define MAX(a,b) ((a)>(b)?(a):(b))
#define MIN(a,b) ((a)<(b)?(a):(b))

int             star_size_adj = 0;
int             alwayslabel = 0;

/*
 * Generic Star Drawing Stuff
 */


int             n, oldmode, currentcolor;
union REGS      rg;


/* Override graphics mode */
D_control_arg(s)
    char           *s;
{
int             c, i, j;
    i = 0;
    j = 1;
    while (c = s[i++]) {
	if (c == 'D') {
	    if (s[i] == '-') {
		j = -1;
		i++;
	    } else if (s[i] == '+')
		i++;
	    star_size_adj = s[i] - '0';
	    if (star_size_adj < 0 || star_size_adj > 9)
		star_size_adj = 0;
	    star_size_adj *= j;
	}
	if (c == 'X')
	    alwayslabel = 1;
    }
}

extern int      readstar();
extern int      (*readfile) ();

struct IRGB {
    int             i;
    int             r;
    int             g;
    int             b;
};

struct IRGB     irgb[16] = {
    {0, 0, 0, 0},
    {1, 22, 22, 63},
    {2, 33, 33, 56},
    {3, 44, 44, 40},
    {4, 49, 49, 33},
    {5, 47, 47, 20},
    {20, 48, 29, 13},
    {7, 45, 45, 45},
    {56, 48, 15, 5},
    {57, 40, 0, 0},
    {58, 18, 18, 18},
    {59, 27, 27, 27},
    {60, 50, 0, 63},
    {61, 0, 60, 25},
    {62, 60, 29, 29},
    {63, 60, 60, 60}
};

#define COLOR_BLACK 0
#define COLOR_O_STAR 1
#define COLOR_B_STAR 2
#define COLOR_A_STAR 3
#define COLOR_F_STAR 4
#define COLOR_G_STAR 5
#define COLOR_K_STAR 6
#define COLOR_DEFTEXT 7
#define COLOR_M_STAR 8
#define COLOR_RED 9
#define COLOR_DARKGRAY 10
#define COLOR_LIGHTGRAY 11
#define COLOR_VIOLET 12
#define COLOR_GREEN 13
#define COLOR_PINK 14
#define COLOR_WHITE 15

#ifndef PLAIN_VGA
#define NUM_HORPIXELS 800
#define NUM_VERTPIXELS 600
#define MODE_SET 98
#else
#define NUM_HORPIXELS 640
#define NUM_VERTPIXELS 480
#define MODE_SET 18
#endif




/* Planetary images */

char           *Sun[] = {
    "        ",
    "   *    ",
    " *****  ",
    "**   ** ",
    "** * ** ",
    "**   ** ",
    " *****  ",
    "   *    ",
    "        ",
    NULL
};

unsigned char   xSun[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Luna[] = {
    " **     ",
    "  ***   ",
    "   ***  ",
    "   ** **",
    "   ** **",
    "   ** **",
    "   ***  ",
    "  ***   ",
    " **     ",
    NULL
};
unsigned char   xLuna[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};


char           *Mercury[] = {

    "**    **",
    "  ****  ",
    "**    **",
    "**    **",
    "**    **",
    "  ****  ",
    "   **   ",
    "  ****  ",
    "   **   ",
    NULL
};

unsigned char   xMercury[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Venus[] = {
    "   **   ",
    "  ****  ",
    "**    **",
    "**    **",
    "**    **",
    "  ****  ",
    "   **   ",
    "  ****  ",
    "   **   ",
    NULL
};
unsigned char   xVenus[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Mars[] = {
    "   **** ",
    "    *** ",
    "    *** ",
    "   ** * ",
    "   **   ",
    " *****  ",
    "**   ** ",
    "**   ** ",
    " *****  ",
    NULL
};
unsigned char   xMars[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Jupiter[] = {
    " **     ",
    "****  **",
    "  ** ** ",
    "  ** ** ",
    " **  ** ",
    " **  ** ",
    " *****  ",
    "    **  ",
    "    **  ",
    NULL
};
unsigned char   xJupiter[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Saturn[] = {
    " ****   ",
    "  **    ",
    "  **    ",
    "  ***** ",
    "  **  **",
    "  ** ** ",
    "  ** ** ",
    "  **  **",
    "       *",
    NULL
};
unsigned char   xSaturn[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Uranus[] = {
    "   **   ",
    " ****** ",
    "** ** **",
    "   **   ",
    " ****** ",
    "**    **",
    "** ** **",
    "**    **",
    " ****** ",
    NULL
};
unsigned char   xUranus[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Neptune[] = {
    "        ",
    "** ** **",
    "** ** **",
    "** ** **",
    "** ** **",
    " ****** ",
    "   **   ",
    "  ****  ",
    "   **   ",
    NULL
};
unsigned char   xNeptune[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Pluto[] = {
    "******  ",
    "**   ** ",
    "**    **",
    "**   ** ",
    "******  ",
    "**      ",
    "**      ",
    "**      ",
    "********",
    NULL
};
unsigned char   xPluto[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Comet[] = {
    "        ",
    "     ***",
    "    *** ",
    " ***    ",
    "********",
    "********",
    " ***    ",
    "    *** ",
    "     ***",
    NULL
};
unsigned char   xComet[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Aster[] = {
    "        ",
    "*     * ",
    "*     * ",
    " * * *  ",
    "  ***   ",
    "  ***   ",
    " * * *  ",
    "*     * ",
    "*     * ",
    NULL
};
unsigned char   xAster[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

/* Galaxy images */

char           *Gal_E[] = {
    "        ",
    "**      ",
    "****    ",
    " ****   ",
    " *****  ",
    "  ***** ",
    "   **** ",
    "    ****",
    "      **",
    NULL
};
unsigned char   xGal_E[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Gal_S[] = {
    "        ",
    "  ***   ",
    " *   *  ",
    "   **   ",
    "  ****  ",
    "   **   ",
    "  *   * ",
    "   ***  ",
    "        ",
    NULL
};
unsigned char   xGal_S[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

/* Cluster images */

char           *Clu_G[] = {
    "        ",
    "   *    ",
    " *   *  ",
    "   *    ",
    "* *** * ",
    "   *    ",
    " *   *  ",
    "   *    ",
    "        ",
    NULL
};
unsigned char   xClu_G[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Clu_O[] = {
    "        ",
    "   *    ",
    " *   *  ",
    "        ",
    "*     * ",
    "        ",
    " *   *  ",
    "   *    ",
    "        ",
    NULL
};
unsigned char   xClu_O[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

/* Nebula images */

char           *Neb_D[] = {
    "        ",
    "******* ",
    "*     * ",
    "*     * ",
    "*     * ",
    "*     * ",
    "*     * ",
    "******* ",
    "        ",
    NULL
};
unsigned char   xNeb_D[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Neb_P[] = {
    "        ",
    "   *    ",
    " * * *  ",
    "   *    ",
    "******* ",
    "   *    ",
    " * * *  ",
    "   *    ",
    "        ",
    NULL
};
unsigned char   xNeb_P[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

/* Other images */

char           *Unk_U[] = {
    "        ",
    "   ***  ",
    "  *   * ",
    "  *   * ",
    "     *  ",
    "    *   ",
    "    *   ",
    "        ",
    "    *   ",
    NULL
};
unsigned char   xUnk_U[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *Other_O[] = {
    "        ",
    "        ",
    "* * *   ",
    " * * *  ",
    "* * *   ",
    " * * *  ",
    "* * *   ",
    " * * *  ",
    "        ",
    NULL
};
unsigned char   xOther_O[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *C300[] = {
    "*    ** ",
    " *****  ",
    "  **    ",
    " **     ",
    " **     ",
    " **     ",
    "  ****  ",
    "     ** ",
    "    **  ",
    NULL
};
unsigned char   xC300[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *C305[] = {
    "* ****  ",
    " *      ",
    "  ***   ",
    "**      ",
    "**      ",
    " **     ",
    "  ***   ",
    "     ** ",
    "    **  ",
    NULL
};
unsigned char   xC305[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *C301[] = {
    "        ",
    "**      ",
    "******  ",
    "**   ** ",
    "**   ** ",
    "**   ** ",
    "     ** ",
    "     ** ",
    "     ** ",
    NULL
};
unsigned char   xC301[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *C302[] = {
    "        ",
    "        ",
    "        ",
    " ***    ",
    "  **    ",
    "  **    ",
    "  **    ",
    "  **    ",
    "  ***   ",
    NULL
};
unsigned char   xC302[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *C303[] = {
    "        ",
    "        ",
    "**   ** ",
    "**  **  ",
    "** **   ",
    "****    ",
    "** **   ",
    "**  **  ",
    "**   ** ",
    NULL
};
unsigned char   xC303[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *C304[] = {
    "**      ",
    " **     ",
    "  **    ",
    "   **   ",
    "   **   ",
    "   **   ",
    "  ****  ",
    " **  ** ",
    "**    **",
    NULL
};
unsigned char   xC304[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *C306[] = {
    "        ",
    "  ****  ",
    " **  ** ",
    "**   ** ",
    "**   ** ",
    "**  **  ",
    "*****   ",
    "**      ",
    "**      ",
    NULL
};
unsigned char   xC306[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *C307[] = {
    "*  *  * ",
    "*  *  * ",
    "*  *  * ",
    "*  *  * ",
    " * * *  ",
    "  ***   ",
    "   *    ",
    "   *    ",
    "   *    ",
    NULL
};
unsigned char   xC307[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

char           *C308[] = {
    "        ",
    "        ",
    "        ",
    " *   *  ",
    "**   ** ",
    "** * ** ",
    "** * ** ",
    "** * ** ",
    " *****  ",
    NULL
};
unsigned char   xC308[9] = {0, 0, 0, 0, 0, 0, 0, 0, 0};



void            cimage(char **, unsigned char *);

void 
cimage(char **in, unsigned char *out)
{
int             i, j, bits;

    for (i = 0; i < 9; i++) {
	bits = 0;
	if (in[i][0] == '*')
	    bits |= 128;
	if (in[i][1] == '*')
	    bits |= 64;
	if (in[i][2] == '*')
	    bits |= 32;
	if (in[i][3] == '*')
	    bits |= 16;
	if (in[i][4] == '*')
	    bits |= 8;
	if (in[i][5] == '*')
	    bits |= 4;
	if (in[i][6] == '*')
	    bits |= 2;
	if (in[i][7] == '*')
	    bits |= 1;
	out[i] = bits;
    }
}
void            dimage(int, int, unsigned char *);



int             firsttimethrough = 1;




/* Open the device */
int
D_open()
{
int             i;
float           aspect;
int             limit;

    readfile = readstar;	/* No memory to spare on PC */

    rg.h.ah = 0x0f;
    int86(16, &rg, &rg);
    oldmode = rg.h.al;
    if (MODE_SET == 18) {
	rg.x.ax = 18;
    } else if (MODE_SET == 98) {	/* This will need to be changed for
					 * other video cards               */
	rg.x.ax = 0x6f05;
	rg.h.bl = 98;
    } else
	exit(1);
    int86(16, &rg, &rg);


    setmod(MODE_SET);


    /*
     * This next bit resets the VGA pallette registers to nice values. Since
     * the EGA doesn't have these registers, this program won't work there.
     * You could get it to work by dropping this "for" loop and changing the
     * #defines for the colors, and using graphics mode 16 (decimal). There
     * are 64 colors in an EGA, which probably will be enough. Use the BIOS
     * to set them.
     */

    for (i = 0; i < 16; i++) {
	outp(0x3c8, irgb[i].i);
	outp(0x3c9, irgb[i].r);
	outp(0x3c9, irgb[i].g);
	outp(0x3c9, irgb[i].b);
    }

    setcolor(COLOR_DEFTEXT);

    yscale1 = NUM_VERTPIXELS - 1;
    yscale2 = (long) (13 * NUM_VERTPIXELS);
    xscale1 = (long) (0.5 + 0.9765625 * NUM_HORPIXELS);
    xlast = 0;
    ylast = yscale1;
    aspect = 1.;
    for (i = 0; i < 7; ++i) {
	if (yrad[i] < 1) {
	    yrad[i] = 1;
	}
    }
    return TRUE;
}

int 
setcolor(c)
    int             c;
{
    currentcolor = c;
}

/* Close the device */
D_close()
{
    /* wait for keystroke */
    getch();
    /* reset screen */
    rg.x.ax = oldmode;
    int86(16, &rg, &rg);
}

int             cur__x, cur__y;

/* Move to (x, y) */
D_move(x, y)
    int             x, y;
{
    cur__x = scx(x);
    cur__y = scy(y);
}


/* Draw a line of style line_style from the current point to (x, y) */
/* Note, this replaces vecdraw vecdrawdot and vecdrawhyph */
D_draw(x, y, line_style)
    int             x, y;
    int             line_style;	/* SOLID, DOTTED, DASHED, etc. */
{
unsigned short  ls;
int             tx, ty;

    switch (line_style) {
    case SOLID:
    case VECSOLID:
	ls = COLOR_DEFTEXT;
	break;
    case DOTTED:
    case VECDOT:
	ls = COLOR_LIGHTGRAY;
	break;
    case DASHED:
    case VECDASH:
	ls = COLOR_DARKGRAY;
    }
    zsetup();
    zline(cur__x, cur__y, (tx = scx(x)), (ty = scy(y)), ls);
    curmod();
    cur__x = tx;
    cur__y = ty;
}
/*
 * This routine is encouraged to look at the extern cur_funtion and change
 * the line style drawn as desired
 */


/* Move to (x1, y1) then draw a line of style line_style to (x2, y2) */
D_movedraw(x1, y1, x2, y2, line_style)
    int             x1, y1, x2, y2;
    int             line_style;	/* SOLID, DOTTED, DASHED, etc. */
{
    D_move(x1, y1);
    D_draw(x2, y2, line_style);
}


/* Set the color to be used for lines and text */
/*
 * color_str is a 2 char (+ '\0') string containing a specification for a
 * color, e.g. "G2" for the color of a star of spectral class G2, or "r7" for
 * red, level seven.  The interpretation of the color string is left to the
 * device driver
 */
D_color(color_str)
    char           *color_str;
{
    switch (color_str[0]) {
    case 'O':
    case 'b':
	setcolor(COLOR_O_STAR);
	break;
    case 'B':
	setcolor(COLOR_B_STAR);
	break;
    case 'A':
	setcolor(COLOR_A_STAR);
	break;
    case 'F':
	setcolor(COLOR_F_STAR);
	break;
    case 'y':
    case 'G':
	setcolor(COLOR_G_STAR);
	break;
    case 'K':
    case 'o':
	setcolor(COLOR_K_STAR);
	break;
    case 'M':
	setcolor(COLOR_M_STAR);
	break;
    case 'r':
    case 'C':
    case 'R':
    case 'N':
    case 'S':
	setcolor(COLOR_RED);
	break;
    case 'g':
    case 'c':
	setcolor(COLOR_GREEN);
	break;
    case 'p':
	setcolor(COLOR_VIOLET);
	break;
    case 16:
	setcolor(COLOR_PINK);
	break;
    default:
	setcolor(COLOR_DEFTEXT);
	break;
    }
}


/* Set the font and font size to be used for text. */
/* Note order of args */
D_fontsize(fsize, font)
    int             fsize;	/* Size of font */
    int             font;	/* e.g. TIMES, HELV, TIMES+ITALIC */
{
}
/*
 * This routine is encouraged to look at the extern cur_funtion and change
 * the font used as desired
 */


/*
 * Fill in one character using the 8x8 character table that BIOS uses to
 * write chars in graphics mode. Advance the xlast value by 8 bits.
 */
one_char(c)
    int             c;
{
char far       *xtab;
int             i, j;
int             x, y;


    if ((c < 0)) {
	xlast += 8;
	return;
    }
    y = ylast - 4;
    x = xlast + 7;
    if (c < 300) {
	slettr(x, y, c, 7);
    } else {
	switch (c) {
	case 300:
	    symbol(x, y, 9, 7, xC300);
	    break;
	case 301:
	    symbol(x, y, 9, 7, xC301);
	    break;
	case 302:
	    symbol(x, y - 2, 9, 7, xC302);
	    break;
	case 303:
	    symbol(x, y - 2, 9, 7, xC303);
	    break;
	case 304:
	    symbol(x, y - 2, 9, 7, xC304);
	    xlast += 1;
	    break;
	case 305:
	    symbol(x, y, 9, 7, xC305);
	    break;
	case 306:
	    symbol(x, y - 2, 9, 7, xC306);
	    break;
	case 307:
	    symbol(x, y - 1, 9, 7, xC307);
	    break;
	case 308:
	    symbol(x, y - 2, 9, 7, xC308);
	    xlast += 1;
	    break;
	default:
	    break;
	}
    }
    xlast += 8;
}

static int      intable[25] = {'a', 'b', 'g', 'd', 'e', 'z', 'h', 'q',
                 'i', 'k', 'l', 'm', 'n', 'x', 'o', 'p', 'r', 's', 't',
                 'u', 'j', 'c', 'y', 'w', 'f'};
static int      outtable[25] = {224, 225, 226, 235, 238, 300, 301, 233,
                 302, 303, 304, 230, 'v', 305, 'o', 227, 306, 229, 231,
                 'u', 237, 'x', 307, 308, 237};

greekfirsttime = 1;

/*
 * Display text string str at x,y, in current font and font size, using greek
 * characters (if possible) if gk_flag is TRUE
 */
D_text(x, y, str, gk_flag)
    int             x, y;
    char           *str;
    int             gk_flag;
{
int             c, i;
    zsetup();

    if (greekfirsttime) {
	cimage(C300, xC300);
	cimage(C301, xC301);
	cimage(C302, xC302);
	cimage(C303, xC303);
	cimage(C304, xC304);
	cimage(C305, xC305);
	cimage(C306, xC306);
	cimage(C307, xC307);
	cimage(C308, xC308);
	greekfirsttime = 0;
    }
    if (gk_flag == 2)
	gk_flag = 0;
    xlast = scx(x);
    ylast = scy(y);
    if (!gk_flag) {
	while ((c = *str++) != '\0') {
	    one_char(c);
	}
    } else {
	if (isgreek(str[0]) && (isdigit(str[1]) || (str[1] == ' '))) {
	    /*
	     * Greek if first character is greek encoded, and the second is
	     * space or a digit
	     */
	    c = 0;
	    for (i = 0; i < 25; i++) {
		if (intable[i] == str[0]) {
		    c = outtable[i];
		    break;
		}
	    }
	    if (c) {
		one_char(c);
		one_char(str[1]);
	    }
	} else {   /* Star label, but not greek */
	    /* remove leading spaces */
	    while (*str == ' ')
		str++;
	    while ((c = *str++) != '\0')
		one_char(c);
	}




    }

    curmod();

}

isgreek(c)
    char            c;
{
char           *cp;

    cp = "abgdezhqiklmnxoprstujcywf";	/* f and j are both phi */
    while (*cp && (*cp != c))
	cp++;
    return (*cp != '\0');	/* True if letter was in greek string */
}


/* Return input coordinate in device coords where there are pointing devices */
D_inxy(x, y)
    int            *x, *y;
{
}


/*
 * Put non-displayed comment in output.  Allowed in postscript, but few other
 * drivers will be able to support this.
 */
D_comment(str)
    char           *str;
{
    /*
     * fprintf(stderr, "%s\n", str);
     */
}

drawlen(x, y, dx, dy, len)
{
int             x1, x2, y1;

    x1 = x + dx * 2;
    y1 = y + dy * 2;
    x2 = x1 + len * 2 - 1;
    x1 = scx(x1);
    y1 = scy(y1);
    x2 = scx(x2);
    zsetup();
    zline(x2, y1, x1, y1, currentcolor);
    curmod();
    cur__x = x2;
    cur__y = y1;
}



/* Interface Function */
/* Draw object at x, y.  properties set by other parameters */
drawobj(x, y, mag, type, color_str, label_field, con_str, obj_name,
	comment_str, file_line,
	draw_glyph, draw_text, use_lbl, use_name, use_mag)
    int             x, y;
    double          mag;	/* Magnitude of object */
    char           *type;	/* 2 chars, object code and subcode */
    char           *color_str;	/* 2 chars, spectral type for stars, color
				 * code otherwise */
    char           *label_field;	/* 2 chars, Bayer or flamsteed for
					 * stars, size in seconds for nebulae
					 * and planets */
    char           *con_str;	/* 3 chars, the constellation the object is
				 * in */
    char           *obj_name;	/* Name of object */
    char           *comment_str;	/* Comment field */
    char           *file_line;	/* The full line from the file, containing
				 * the above if it is in standard format */
    int             draw_glyph;	/* Draw object symbol */
    int             draw_text;	/* Draw text */
    int             use_lbl;	/* Label object with the label_field string */
    int             use_name;	/* Label object with the obj_name string */
    int             use_mag;	/* Label object with a 2 or 3 character
				 * string containing the magnitude * 10
				 * without decimal point */
{
char            magstr[10];


    if (firsttimethrough) {
	cimage(Sun, xSun);
	cimage(Luna, xLuna);
	cimage(Mercury, xMercury);
	cimage(Venus, xVenus);
	cimage(Mars, xMars);
	cimage(Jupiter, xJupiter);
	cimage(Saturn, xSaturn);
	cimage(Uranus, xUranus);
	cimage(Neptune, xNeptune);
	cimage(Pluto, xPluto);
	cimage(Comet, xComet);
	cimage(Aster, xAster);
	cimage(Gal_E, xGal_E);
	cimage(Gal_S, xGal_S);
	cimage(Clu_G, xClu_G);
	cimage(Clu_O, xClu_O);
	cimage(Neb_D, xNeb_D);
	cimage(Neb_P, xNeb_P);
	cimage(Unk_U, xUnk_U);
	cimage(Other_O, xOther_O);

	firsttimethrough = 0;
    }
    /*
     * fprintf(stderr, "%d %d %f <%s> <%s> <%s> <%s> <%s> <%s> <%s>\n", x, y,
     * mag, type, color_str, label_field, con_str, obj_name, comment_str,
     * file_line);
     */

    if (draw_glyph)
	switch (type[0]) {
	case 'S':
	    drawStar(x, y, mag, type[1], color_str);
	    break;
	case 'P':
	    drawPlan(x, y, mag, type[1], color_str, size_obj(label_field),
		     comment_str);
	    goto extraname;
	    /*
	     * I, J. D. McDonald, have sworn to include a "goto" in every
	     * program I write. This is a perfect use of one.
	     */
	case 'N':
	    drawNebu(x, y, mag, type[1], color_str, size_obj(label_field));
	    goto extraname;
	case 'G':
	    drawGalx(x, y, mag, type[1], color_str, size_obj(label_field));
	    goto extraname;
	case 'C':
	    drawClus(x, y, mag, type[1], color_str, size_obj(label_field));
    extraname:
	    if (alwayslabel) {
		draw_text = 1;
		use_name = 1;
	    }
	    break;
	case 'U':
	    drawUnknown(x, y, mag, type[1], color_str, size_obj(label_field));
	    break;
	case 'O':
	    drawOther(x, y, mag, type[1], color_str, size_obj(label_field));
	    break;
	case 'V':
	case 'A':
	case 'I':
	    break;
	case '#':
	default:
	    break;
	};


    /*
     * use name or label
     */
    if (draw_text) {
	if (type[0] == 'I')
	    D_color(color_str);
	else
	    D_color("  ");

	if (use_name && obj_name[0]) {
	    D_fontsize(namesize, namefnt);
	    D_text(x + x_nameoffset, y + y_nameoffset, obj_name, FALSE);
	} else if (use_lbl &&
		   ((label_field[0] != ' ') || (label_field[1] != ' '))) {
	    D_fontsize(lblsize, lblfnt);
	    D_text(x + x_lbloffset, y + y_lbloffset, label_field, TRUE);
	}
	/* If you want to mag label other objects, change this */
	if (use_mag && (type[0] == 'S')) {
	    sprintf(magstr, "%02d", (int) (mag * 10.0 + 0.5));
	    D_fontsize(magsize, magfnt);
	    D_text(x + x_magoffset, y + y_magoffset, magstr, FALSE);
	}
    }
}

drawStar(x, y, mag, type, color)
    int             x, y;
    double          mag;
    char            type, *color;
{
int            *coord;
int             i;
int             i_mag;
int             maxlen;

    mag -=  star_size_adj;

    if (mag < MIN_MAG) {
	mag = MIN_MAG;
    } else if (mag > MAX_MAG) {
	mag = MAX_MAG;
    }

    i_mag =  9 - (int)(mag + 0.5);

    D_color(color);

    x = scx(x);
    y = scy(y);
    zsetup();
    if (i_mag > 1)
	fillelip(x, y, i_mag, i_mag, currentcolor);
    else if (i_mag == -2)
	zpoint(x, y, currentcolor);
    else if (i_mag == -1)
	zline(x, y, x+1, y, currentcolor);
    else if (i_mag == 0)
        fillelip(x,y,1,1,currentcolor);
    else 
        rectfill(x-1,y-1,x+1,y+1,currentcolor);

    if (type == 'D' && i_mag > 2) {
	zline(x - i_mag - 2, y, x + i_mag + 2, y, currentcolor);
    }
    if (type == 'V' && i_mag > 3) {
	ellipse(x, y, i_mag - 2, i_mag - 2, 0);
    }
    curmod();
    D_color(" ");
}


drawPlan(x, y, mag, pcode, color, plansize, comment_str)
    int             x, y;
    double          mag;
    char            pcode, *color;
    long            plansize;
    char           *comment_str;
{

    D_color(color);
    zsetup();
    switch (pcode) {
    case 'S':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xSun);
	break;
    case 'L':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xLuna);
	break;
    case 'M':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xMercury);
	break;
    case 'V':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xVenus);
	break;
    case 'm':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xMars);
	break;
    case 'J':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xJupiter);
	break;
    case 's':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xSaturn);
	break;
    case 'U':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xUranus);
	break;
    case 'N':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xNeptune);
	break;
    case 'P':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xPluto);
	break;
    case 'C':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xComet);
	break;
    case 'A':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xAster);
	break;
    default:
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xUnk_U);
	break;
    }
    curmod();
}


drawGalx(x, y, mag, type, color, nebsize)
    int             x, y;
    double          mag;
    char            type, *color;
    long            nebsize;	/* -1 should give default size */
{
    D_color(color);

    zsetup();
    switch (type) {
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'B':
    case 'S':
    case 'O':
    case 'Q':
    case 'I':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xGal_S);
	break;
    default:
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xGal_E);

    }
    curmod();

}

drawClus(x, y, mag, type, color, nebsize)
    int             x, y;
    double          mag;
    char            type, *color;
    long            nebsize;	/* -1 should give default size */
{
    D_color(color);
    zsetup();
    switch (type) {
    case 'G':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xClu_G);
	break;
    default:
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xClu_O);

    }
    curmod();
}

drawNebu(x, y, mag, type, color, nebsize)
    int             x, y;
    double          mag;
    char            type, *color;
    long            nebsize;	/* -1 should give default size */
{

    if (color[0] == ' ') {
	if (type == 'P')
	    color[0] = 'g';
	else
	    color[0] = 16;
    }
    D_color(color);

    zsetup();
    switch (type) {
    case 'P':
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xNeb_P);
	break;
    default:
	symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xNeb_D);

    }
    curmod();
}

drawUnknown(x, y, mag, type, color, nebsize)
    int             x, y;
    double          mag;
    char            type, *color;
    long            nebsize;	/* -1 should give default size */
{
    D_color(color);

    zsetup();
    symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xUnk_U);
    curmod();
}

drawOther(x, y, mag, type, color, nebsize)
    int             x, y;
    double          mag;
    char            type, *color;
    long            nebsize;	/* -1 should give default size */
{
    D_color(color);

    zsetup();
    symbol((int) scx(x) - 3, (int) scy(y) - 4, 9, currentcolor, xOther_O);
    curmod();
}




/* Note externs which are used */

chartlegend(win)
    mapwindow      *win;
{
char            ras[20], dls[20], outstr[40];
    if (!title[0])
	title = "LEGEND";
    rastr(ras, win->racen);
    declstr(dls, win->dlcen);

    if (win->map_type != FULLPAGEMAP) {
	sprintf(outstr, "(%s,%s lim: %2.1f)", ras, dls, win->maglim);
	D_fontsize(titlesize, titlefnt);
	D_text(l_lmar1, l_til, title, FALSE);
	D_fontsize(subtlsize, subtlfnt);
	D_text(l_lmar1, l_stil, outstr, FALSE);

	drawStar(l_lmar2, l_line1, 1.0, 'S', "  ");
	D_fontsize(namesize, namefnt);
	D_text(l_ltext, l_line1, "<1.5", FALSE);
	if (win->maglim >= 2.5) {
	    drawStar(l_rmar2, l_line1, 3.0, 'S', "  ");
	    D_fontsize(namesize, namefnt);
	    D_text(l_rtext, l_line1, "<3.5", FALSE);
	}
	if (win->maglim >= 4.5) {
	    drawStar(l_lmar2, l_line2, 5.0, 'S', "  ");
	    D_fontsize(namesize, namefnt);
	    D_text(l_ltext, l_line2, "<5.5", FALSE);
	}
	if (win->maglim >= 6.5) {
	    drawStar(l_rmar2, l_line2, 7.0, 'S', "  ");
	    D_fontsize(namesize, namefnt);
	    D_text(l_rtext, l_line2, "<7.5", FALSE);
	}
	if (win->maglim >= 8.5) {
	    drawStar(l_lmar2, l_line3, 9.0, 'S', "  ");
	    D_fontsize(namesize, namefnt);
	    D_text(l_ltext, l_line3, "<9.5", FALSE);
	}
	if (win->maglim > 9.5) {
	    drawStar(l_rmar2, l_line3, 10.0, 'S', "  ");
	    D_fontsize(namesize, namefnt);
	    D_text(l_rtext, l_line3, ">9.5", FALSE);
	}
	D_fontsize(namesize, namefnt);
	D_text(l_ltext, l_line4, "double", FALSE);
	drawStar(l_lmar2, l_line4, 4.0, 'D', "  ");
	D_fontsize(namesize, namefnt);
	D_text(l_rtext, l_line4, "variable", FALSE);
	drawStar(l_rmar2, l_line4, 2.0, 'V', "  ");

	D_fontsize(namesize, namefnt);
	D_text(l_ltext, l_line5, "planet", FALSE);
	drawPlan(l_lmar2, l_line5, 1.0, 'J', "  ", (long) -1, "");

	D_fontsize(namesize, namefnt);
	D_text(l_rtext, l_line5, "galaxy", FALSE);
	drawGalx(l_rmar2, l_line5, 1.0, 'E', "  ", (long) -1);
	drawGalx(l_rmar1, l_line5, 1.0, 'S', "  ", (long) -1);

	D_fontsize(namesize, namefnt);
	D_text(l_ltext, l_line6, "nebula", FALSE);
	drawNebu(l_lmar2, l_line6, 1.0, 'D', "  ", (long) -1);
	drawNebu(l_lmar1, l_line6, 1.0, 'P', "  ", (long) -1);

	D_fontsize(namesize, namefnt);
	D_text(l_rtext, l_line6, "cluster", FALSE);
	drawClus(l_rmar2, l_line6, 1.0, 'O', "  ", (long) -1);
	drawClus(l_rmar1, l_line6, 1.0, 'G', "  ", (long) -1);
    } else {
	D_fontsize(namesize, namefnt);

	sprintf(outstr, "%s: %s,%s lim: %2.1f", title, ras, dls, win->maglim);
	D_text(15, 15, outstr, FALSE);
    }
}

/* Functions for areas, drawn as lines for now */
static struct {
    int             x, y;
}               areapts[1000];
static int      nareapts;
/* Move to (x, y) to begin an area */
D_areamove(x, y)
    int             x, y;
{
    nareapts = 0;
    areapts[nareapts].x = x;
    areapts[nareapts].y = y;
    nareapts++;
}

/* Add a segment to the area border */
D_areaadd(x, y)
    int             x, y;
{
    areapts[nareapts].x = x;
    areapts[nareapts].y = y;
    nareapts++;
}

/* Fill the area, after adding the last segment */
D_areafill(x, y)
    int             x, y;
{
int             i;

    areapts[nareapts].x = x;
    areapts[nareapts].y = y;
    nareapts++;


    D_move(areapts[0].x, areapts[0].y);

    for (i = 1; i < nareapts; i++)
	D_draw(areapts[i].x, areapts[i].y);
}



/* Draw an ellipse with width irx and height iry                         */
/* from a routine by Tim Hogan in Dr. Dobb's Journal May '85 p.40        */
/* Improved by calculating increments incrementally, thus removing all   */
/* multiplies from the loops. These multiplies were very bad since they  */
/* were (long)*(long). This code, when compiled by Microsoft C Version 5,*/
/* can't be significantly improved by hand optimization.                 */
/* Written Sept. 7, 1987 by J.D. McDonald (public domain)                */

static long     alpha, beta, alpha2, alpha4, beta2, beta4, d;
static long     ddx, ddy, alphadx, betady;
static int      dy, dx;

extern void     e_start();	/* Starts off by writing right and left
				 * points                                */
extern void     e_xd();	/* Moves one step to lower x (in all 4 quadrants) */
extern void     e_xdyu();	/* Moves to lower x, higher y            */
extern void     e_yu();	/* Moves to higher y                     */

ellipse(x, y, irx, iry, c)
    int             x, y, irx, iry;
    unsigned        c;
{

beta = (long) irx *(long) irx;
alpha = (long) iry *(long) iry;

    if (alpha == 0L)
	alpha = 1L;
    if (beta == 0L)
	beta = 1L;

    dy = 0;
    dx = irx;
    alpha2 = alpha << 1;
    alpha4 = alpha2 << 1;
    beta2 = beta << 1;
    beta4 = beta2 << 1;
    alphadx = alpha * dx;
    betady = 0;
    ddx = alpha4 * (1 - dx);
    ddy = beta2 * 3;

    d = alpha2 * ((long) (dx - 1) * dx) + alpha + beta2 * (1 - alpha);
    e_start(x - dx, x + dx, y, c);

    do {
	if (d >= 0) {
	    d += ddx;
	    dx--;
	    alphadx -= alpha;
	    ddx += alpha4;
	    e_xdyu();
	} else
	    e_yu();
	d += ddy;
	dy++;
	betady += beta;
	ddy += beta4;
    } while (alphadx > betady);

    d = beta2 * ((long) dy * (dy + 1)) + alpha2 * ((long) dx * (dx - 2) + 1)
	+ beta * (1 - alpha2);
    ddx = alpha2 * (3 - (dx << 1));
    ddy = beta4 * (1 + dy);

    do {
	if (d <= 0) {
	    d += ddy;
	    ddy += beta4;
	    dy++;
	    e_xdyu();
	} else
	    e_xd();
	d += ddx;
	ddx += alpha4;
	dx--;
    } while (dx > 0);
}

fillelip(x, y, irx, iry, c)
    int             x, y, irx, iry;
    unsigned        c;
{

beta = (long) irx *(long) irx;
alpha = (long) iry *(long) iry;

    if (alpha == 0L)
	alpha = 1L;
    if (beta == 0L)
	beta = 1L;

    dy = 0;
    dx = irx;
    alpha2 = alpha << 1;
    alpha4 = alpha2 << 1;
    beta2 = beta << 1;
    beta4 = beta2 << 1;
    alphadx = alpha * dx;
    betady = 0;
    ddx = alpha4 * (1 - dx);
    ddy = beta2 * 3;

    d = alpha2 * ((long) (dx - 1) * dx) + alpha + beta2 * (1 - alpha);
    rectfill(x - dx, y, x + dx, y, c);

    do {
	if (d >= 0) {
	    d += ddx;
	    dx--;
	    alphadx -= alpha;
	    ddx += alpha4;
	}
	d += ddy;
	dy++;
	betady += beta;
	ddy += beta4;
	rectfill(x - dx, y + dy, x + dx, y + dy, c);
	rectfill(x - dx, y - dy, x + dx, y - dy, c);
    } while (alphadx > betady);

    d = beta2 * ((long) dy * (dy + 1)) + alpha2 * ((long) dx * (dx - 2) + 1)
	+ beta * (1 - alpha2);
    ddx = alpha2 * (3 - (dx << 1));
    ddy = beta4 * (1 + dy);

    do {
	dx--;
	if (d <= 0) {
	    d += ddy;
	    ddy += beta4;
	    dy++;
	    rectfill(x - dx, y + dy, x + dx, y + dy, c);
	    rectfill(x - dx, y - dy, x + dx, y - dy, c);
	}
	d += ddx;
	ddx += alpha4;
    } while (dx > 0);
}
