/* Functions for manipulating a PR.
   Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.
   Contributed by Tim Wicinski (wicinski@barn.com)
   and Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"

PR_entry pr[NUM_PR_ITEMS];

static PR_Name field_name		PARAMS((char *));

/* Read the file pointed to by FP, and look for matching field names, trying
   to attach values to those names.  */

void 
read_pr (fp, prune)
     FILE *fp;
     int prune;
{
  char token[STR_MAX];
  char line[STR_MAX];
  int i, oldi = -1;
  char *l;
  bool multi_text = FALSE;
  char *buffer, *unformatted;
  char *b, *u, *p;
  int buffer_size = BUFSIZ, unformatted_size = BUFSIZ;
  int unformatted_len = 0, buffer_len = 0;
  int len;

  if (fp == (FILE *) NULL)
    return;

  memset (token, 0, STR_MAX);
  memset (line, 0, STR_MAX);
  clean_pr ();

  buffer = (char*) xmalloc (BUFSIZ);
  unformatted = (char*) xmalloc (BUFSIZ);

  b = buffer; u = unformatted;
  while ((len = read_string (line, fp)) != -1)
    {
      i = -1;

      if (len > 1)
	{
	  *token = '\0';

	  l = get_token (line, token);
	  if (token[0] == PR_START_FIELD && token[1])
	    {
	      i = (int) field_name (token);
	      multi_text = FALSE;
	    }
	}

      /* Heuristic for finding included PR fields */
      if (i >= 0 && multi_text && (i == oldi || pr[i].value != NULL))
	i = -1;

      /* If we can't find the name and we are not in multi_text line mode, 
         then it goes into UNFORMATTED */
      if (i != -1)
	{
	  /* prune should be used when only reading w/no hint 
	     of writing PR's back out (ie, query-pr).  */
	  if (prune && pr[i].datatype == MultiText && i != ORGANIZATION)
	    break;

	  if (pr[i].datatype == MultiText)
	    {
	      multi_text = TRUE;
	      len = strlen (l);
	      if (oldi != -1)
		{
		  buffer[buffer_len++] = '\0';
		  if (pr[oldi].value)
		    xfree (pr[oldi].value);
		  pr[oldi].value = (char *) xmalloc (buffer_len);
		  memcpy (pr[oldi].value, buffer, buffer_len);
		}
	      b = buffer;
	      memcpy (b, l, len);
	      buffer_len = len;
	      b += len;
	      oldi = i;
	    }
          else 
            {
	      SKIP_WHITE_SPACE (l);

	      /* Fields that aren't MultiText shouldn't have a newline
		 at the end, since the field's value is important.  */
	      if (line[len - 1] == '\n')
		line[--len] = '\0';

	      if (is_space[ (unsigned char) line[len - 1] ])
		for (p = &line[0] + len - 1; p != &line[0]; p--)
		  {
		    if (is_space[(unsigned char) *p])
		      *p = '\0';
		    else
		      break;
		  }

	      if (pr[i].datatype == Enum)
		{
		  p = (char *) strchr (l, ' ');
		  if (p)
		    *p = '\0';
		}
	      if (pr[i].value)
		xfree (pr[i].value);
	      pr[i].value = (char *) strdup (l);
	      multi_text = FALSE;
            }
	}
      else if (multi_text) 
        {
	  if ((len + buffer_len) >= buffer_size)
	    {
	      buffer_size += BUFSIZ;
	      buffer = (char *) xrealloc (buffer, buffer_size);
	      b = buffer + buffer_len;
	    }
	  memcpy (b, line, len);
	  buffer_len += len;
	  b += len;
        }
      else
	{
	  /* It must be unformatted.  This is done separately since an
	     unformatted field can show up anywhere.  */
	  multi_text = FALSE;

	  if ((len + unformatted_len) >= unformatted_size)
	    {
	      unformatted_size += BUFSIZ;
	      unformatted = (char *) xrealloc (unformatted,
					       unformatted_size);
	      u = unformatted + unformatted_len;
	    }

	  memcpy (u, line, len);
	  unformatted_len += len;
	  u += len;
	}
    }

  /* we don't really care about these buffers with prune set.  */
  /* OLDI could've never changed if we didn't read anything from the file.  */
  /*   BUT - it could be an entirely hosed PR too... */
  if (prune || (unformatted_len == 0 && oldi == -1))
    {
      xfree (buffer);
      xfree (unformatted);
      return;
    }

  /* Quick check to see if the last case was dealt with.  It usually isn't,
     especially since the final cases are normally MultiText.  */
  if (oldi != -1 && pr[oldi].value == NULL) 
    {
      buffer[buffer_len] = '\0';
      pr[oldi].value = buffer;      
    }
  else
    xfree (buffer);

  /* Check to see if the unformatted field was used.  */
  if (unformatted_len != 0)
    {
      unformatted[unformatted_len] = '\0';
      if (pr[UNFORMATTED].value)
	{
	  len = strlen (pr[UNFORMATTED].value);
	  if ((len + unformatted_len + 1) > unformatted_size)
	    {
	      unformatted_size = len + unformatted_len + 1;
	      unformatted = (char *) xrealloc (unformatted,
					       unformatted_size);
	      u = unformatted + unformatted_len;
	    }

	  strcat (u, pr[UNFORMATTED].value);
	  xfree (pr[UNFORMATTED].value);
	}

      pr[UNFORMATTED].value = unformatted;
    }
  else
    xfree (unformatted);
}

/* If TRUE, output both a linefeed and a newline, rather than just a
   newline.  Needed for the network portion.  */
int doret = 0;

/* Write either a single PR field, or the entire PR into the file passed
   in via FP.  */
void 
write_pr (fp, string)
     FILE *fp;
     PR_Name string;
{
  register int i;
  extern char *ret;
  int len;

#define MAYBE_NL(str) { \
    len = strlen(str); \
    if (len && (str[len-1] != '\n')) \
      fputc('\n', fp); \
  }

  if (string != NUM_PR_ITEMS)
    {
      char fmt[10];
      if (pr[string].datatype == MultiText)
	sprintf (fmt, "%%s%s%%s%s", ret, ret);
      else
	sprintf (fmt, "%%-16s %%s%s", ret);

      if (pr[string].value == NULL)
	fprintf (fp, "%s%s", pr[string].name, ret);
      else
	{
	  fprintf (fp, fmt, pr[string].name, pr[string].value);
	  if (pr[string].datatype == MultiText)
	    {
	      MAYBE_NL(pr[string].value);
	    }
	}
    }
  else
    {
      for (i = 0; i < NUM_PR_ITEMS; i++)
	if (pr[i].value != NULL)
	  {
	    /* For multi-text lines, always send a newline after the
		field name, and don't emit one after the value, since
		it will have the newline we need.  */
	    if (pr[i].datatype == MultiText)
	      {
		fprintf (fp, "%s%s%s", pr[i].name, ret, pr[i].value);
		MAYBE_NL(pr[i].value);
	      }
	    else
	      fprintf (fp, "%-16s %s%s", pr[i].name, pr[i].value, ret);
	  }
        else
  	  fprintf (fp, "%s%s", pr[i].name, ret);
    }
}

/* Init all the fields in the PR.  */
void 
init_pr ()
{
  memset (&pr[0], 0, sizeof (PR_entry) * NUM_PR_ITEMS);

  pr[ORIGINATOR].name = ORIGINATOR_STRING;
  pr[ORIGINATOR].datatype = Text;

  pr[ORGANIZATION].name = ORGANIZATION_STRING;
  pr[ORGANIZATION].datatype = MultiText;

  pr[SUBMITTER].name = SUBMITTER_STRING;
  pr[SUBMITTER].datatype = Text;
  pr[SUBMITTER].default_value = def_subm;

  pr[START_DATE].name = START_DATE_STRING;
  pr[START_DATE].datatype = StartDate;
 
  pr[END_DATE].name = END_DATE_STRING;
  pr[END_DATE].datatype = EndDate;

  pr[ARRIVAL_DATE].name = ARRIVAL_DATE_STRING;
  pr[ARRIVAL_DATE].default_value = '\0';
  pr[ARRIVAL_DATE].datatype = Date;

  pr[SYNOPSIS].name = SYNOPSIS_STRING;
  pr[SYNOPSIS].datatype = Text;

  pr[CONFIDENTIAL].name = CONFIDENTIAL_STRING;
  pr[CONFIDENTIAL].enum_values = "no | yes";
  pr[CONFIDENTIAL].default_value = "yes";
  pr[CONFIDENTIAL].datatype = Enum;

  pr[SEVERITY].name = SEVERITY_STRING;
  pr[SEVERITY].enum_values = "non-critical | serious | critical";
  pr[SEVERITY].default_value = "serious";
  pr[SEVERITY].datatype = Enum;

  pr[PRIORITY].name = PRIORITY_STRING;
  pr[PRIORITY].enum_values = "low | medium | high";
  pr[PRIORITY].default_value = "medium";
  pr[PRIORITY].datatype = Enum;

  pr[CATEGORY].name = CATEGORY_STRING;
  pr[CATEGORY].default_value = PENDING;
  pr[CATEGORY].datatype = Text;	/* really enum, matching category file */

#ifdef GNATS_RELEASE_BASED
  pr[QUARTER].name = QUARTER_STRING;
  pr[QUARTER].datatype = Text;

  pr[KEYWORDS].name = KEYWORDS_STRING;
  pr[KEYWORDS].datatype = Text;
#endif

  pr[RELEASE].name = RELEASE_STRING;
  pr[RELEASE].datatype = Text;

  pr[LAST_MODIFIED].name = LAST_MODIFIED_STRING;
  pr[LAST_MODIFIED].datatype = Date;

#ifdef GNATS_RELEASE_BASED
  pr[DATE_REQUIRED].name = DATE_REQUIRED_STRING;
  pr[DATE_REQUIRED].datatype = Date;
#endif

  pr[ENVIRONMENT].name = ENVIRONMENT_STRING;
  pr[ENVIRONMENT].datatype = MultiText;

  pr[DESCRIPTION].name = DESCRIPTION_STRING;
  pr[DESCRIPTION].datatype = MultiText;

  pr[HOW_TO_REPEAT].name = HOW_TO_REPEAT_STRING;
  pr[HOW_TO_REPEAT].datatype = MultiText;

  pr[NUMBER].name = NUMBER_STRING;
  pr[NUMBER].datatype = Integral;
  pr[NUMBER].default_value = "-1";

  pr[RESPONSIBLE].name = RESPONSIBLE_STRING;
  pr[RESPONSIBLE].datatype = Text;

  pr[AUDIT_TRAIL].name = AUDIT_TRAIL_STRING;
  pr[AUDIT_TRAIL].datatype = MultiText;
  pr[AUDIT_TRAIL].default_value = "\nState-Changed-From-To:\n\
State-Changed-By:\n\
State-Changed-When:\n\
State-Changed-Why:";

  pr[STATE].name = STATE_STRING;
  pr[STATE].enum_values = "open | assigned | feedback | suspended | closed";
  pr[STATE].default_value = "open";
  pr[STATE].datatype = Enum;

  pr[CLASS].name = CLASS_STRING;
  pr[CLASS].enum_values
    = "unscheduled | scheduled | support | sw-bug | doc-bug | change-request | mistaken | duplicate ";
  pr[CLASS].default_value = "sw-bug";
  pr[CLASS].datatype = Enum;

  pr[FIX].name = FIX_STRING;
  pr[FIX].datatype = MultiText;
  pr[FIX].default_value = "\nUnknown";

  pr[UNFORMATTED].name = UNFORMATTED_STRING;
  pr[UNFORMATTED].datatype = MultiText;
}

void
clean_pr ()
{
  PR_Name i;

  for (i = NUMBER; i < NUM_PR_ITEMS; i++)
    {
      xfree (pr[i].value);
      pr[i].value = (char *) NULL;
    }
}

/* We deliberately don't alloca the token here, to avoid a bottleneck.  */
int
verify_enum (i, v, token)
     PR_Name i;
     char *v, *token;
{
  char *p, *p2;

  if (pr[i].datatype != Enum)
    return -1;

  for (p2 = pr[i].enum_values, p = get_next_field (p2, token, '|');
       p || p2;
       (p2 = p) && (p = get_next_field (p, token, '|')))
    {
      char *t;
      int j;

      t = strchr (v, ' ');

      if (t)
	j = strncasecmp (v, token, t - v);
      else
	j = strcasecmp (v, token);

      if (j == 0)
	return 1;
    }

  return 0;
}

/* Check all the enumerated typed fields that are declared, and check
   to make sure they're all good values.  Reset any null or invalid
   entries.  */
struct bad_enum *
check_enum_types (check)
     int check;
{
  int i = -1;
  bool found = FALSE;
  struct bad_enum *bad_enums = NULL, *bad_enums_end = NULL;
  char *token = (char *) alloca (1024); /* FIXME */

  while (++i < (int) NUM_PR_ITEMS)
    {
      found = FALSE;
      if (pr[i].datatype != (PR_Type) 0 && pr[i].datatype == Enum)
	{
	  if (pr[i].value == NULL || *pr[i].value == '\0')
	    {
	      if (pr[i].default_value != NULL)
		pr[i].value = pr[i].default_value;
	      else
		break;
	    }

	  if (verify_enum (i, pr[i].value, token))
	    found = TRUE;

	  if (found == FALSE)
	    {
	      if (check)
		{
		  char *msg = (char *) xmalloc (100 + strlen (pr[i].value)
						+ strlen (pr[i].name)
						+ strlen (pr[i].default_value));
		  if (bad_enums == NULL)
		    bad_enums = bad_enums_end
		    = (struct bad_enum *) xmalloc (sizeof (struct bad_enum));
		  else
		    {
		      bad_enums_end->next
			= (struct bad_enum *) xmalloc (sizeof (struct bad_enum));
		      bad_enums_end = bad_enums_end->next;
		    }

		  if (check == 1)
		    sprintf (msg,
			     "\tNote: There was a bad value `%s' for the field `%s'.\n\tIt was set to the default value of `%s'.\n",
			     pr[i].value, pr[i].name, pr[i].default_value);
		  else if (check == 2)
		    sprintf (msg, "%s %s\n", pr[i].name, pr[i].value);

		  bad_enums_end->msg = msg;
		  bad_enums_end->next = (struct bad_enum *) NULL;
		}
	      pr[i].value = pr[i].default_value;
	    }
	}
    }

  return bad_enums;
}

/* Look to see if STRING is a known GNATS string.  */
static PR_Name 
field_name (string)
     char *string;
{
  PR_Name i;
  int len;

  if (*string == '\0')
    return (PR_Name) -1;

  len = strlen(string);

  for (i = (PR_Name) 0; i < NUM_PR_ITEMS; i++)
    if ((pr[i].name != NULL && string != NULL) &&
	(strncmp (pr[i].name, string, len) == 0))
      return i;

  /* We check this, for backwards compatability with the old
     way of doing things (referring to them as customers, instead
     of submitters).  */
  if (strncmp (ALTERNATE_SUBMITTER, string, len) == 0)
    return SUBMITTER;

  return (PR_Name) -1;
}

/* These two routines are used for external procedures. This way, other files
   don't need to see the pr structure.  */
char *
field_value (name)
     PR_Name name;
{
  if (name >= NUM_PR_ITEMS)
    return NULL;

  if (pr[name].value == NULL)
    {
      if (pr[name].default_value != NULL)
	pr[name].value = (char *) strdup (pr[name].default_value);
      else if (pr[name].datatype == Date)
	return NULL;
      else
	pr[name].value = (char *) strdup ("");
    }

  return pr[name].value;
}

/* Set the value of the PR entry NAME to STRING.  Returns TRUE if the
   operation was successful, FALSE if not (e.g., if STRING was a bad
   value for an enumerated field).  */
bool 
set_field (name, string)
     PR_Name name;
     char *string;
{
  char *p, *p2;
  char *token;
  bool found = FALSE;

  if (name >= NUM_PR_ITEMS || string == NULL)
    return found;

  if (pr[name].datatype != Enum)
    {
      xfree (pr[name].value);
      pr[name].value = (char *) strdup (string);
      found = TRUE;
    }
  else
    {
      token = (char *) xmalloc(STR_MAX);
      p = pr[name].enum_values;

      for (p2 = pr[name].enum_values, p = get_next_field (p2, token, '|');
	   p || p2;
	   (p2 = p) && (p = get_next_field(p, token, '|')))
	{
	  if (strcmp (string, token) == 0)
	    {
	      found = TRUE;
	      break;
	    }
	}

      if (found)
	{
	  xfree (pr[name].value);
	  pr[name].value = (char *) strdup (string);
	}
     }

  return found;
}
