;;; hyperspec.el --- Browse documentation from the Common Lisp HyperSpec

;; Copyright 1997 Naggum Software

;; Author: Erik Naggum <erik@naggum.no>
;; Keywords: lisp

;; This file is not part of GNU Emacs, but distributed under the same
;; conditions as GNU Emacs, and is useless without GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Kent Pitman and the Harlequin Group have made the text of American
;; National Standard for Information Technology -- Programming Language --
;; Common Lisp, ANSI X3.226-1994 available on the WWW, in the form of the
;; Common Lisp HyperSpec.  This package makes it convenient to peruse this
;; documentation from within Emacs.

;;; Code:

(require 'cl)
(require 'browse-url)			;you need the Emacs 20 version
(require 'thingatpt)

(defvar common-lisp-hyperspec-root
  "http://www.xanalys.com/software_tools/reference/HyperSpec/"
  "The root of the Common Lisp HyperSpec URL.
If you copy the HyperSpec to your local system, set this variable to
something like \"file:/usr/local/doc/HyperSpec/\".")

(defvar common-lisp-hyperspec-history nil
  "History of symbols looked up in the Common Lisp HyperSpec.")

;;if only we had had packages or hash tables..., but let's fake it.

(defvar common-lisp-hyperspec-symbols (make-vector 67 0))

(defun common-lisp-hyperspec (symbol-name)
  "View the documentation on SYMBOL-NAME from the Common Lisp HyperSpec.
If SYMBOL-NAME has more than one definition, all of them are displayed with
your favorite browser in sequence.  The browser should have a \"back\"
function to view the separate definitions.

The Common Lisp HyperSpec is the full ANSI Standard Common Lisp, provided
by Kent Pitman and the Harlequin Group.  By default, the Harlequin WWW site
is visited to retrieve the information.  The Harlequin Group allows you to
transfer the entire Common Lisp HyperSpec to your own site under certain
conditions.  Visit http://www.harlequin.com/books/HyperSpec/ for more
information.  If you copy the HyperSpec to another location, customize the
variable `common-lisp-hyperspec-root' to point to that location."
  (interactive (list (let ((symbol-at-point (thing-at-point 'symbol)))
		       (if (and symbol-at-point
				(intern-soft (downcase symbol-at-point)
					     common-lisp-hyperspec-symbols))
			 symbol-at-point
			 (completing-read
			  "Look up symbol in Common Lisp HyperSpec: "
			  common-lisp-hyperspec-symbols #'boundp
			  t symbol-at-point
			  'common-lisp-hyperspec-history)))))
  (maplist (lambda (entry)
	    (browse-url (concat common-lisp-hyperspec-root "Body/" (car entry)))
	    (if (cdr entry)
	      (sleep-for 1.5)))
    (let ((symbol (intern-soft (downcase symbol-name)
			       common-lisp-hyperspec-symbols)))
      (if (and symbol (boundp symbol))
	(symbol-value symbol)
	(error "The symbol `%s' is not defined in Common Lisp"
	       symbol-name)))))

;;; Added the following just to provide a common entry point according
;;; to the various 'hyperspec' implementations.
;;;
;;; 19990820 Marco Antoniotti

(eval-when (load eval)
  (setf (symbol-function 'hyperspec-lookup)
	(symbol-function 'common-lisp-hyperspec)))


(mapcar (lambda (entry)
	  (let ((symbol (intern (car entry) common-lisp-hyperspec-symbols)))
	    (if (boundp symbol)
	      (push (cadr entry) (symbol-value symbol))
	      (set symbol (cdr entry)))))
	'(("&allow-other-keys" "sec_3-4-1.html")
	  ("&aux" "sec_3-4-1.html")
	  ("&body" "sec_3-4-4.html")
	  ("&environment" "sec_3-4-4.html")
	  ("&key" "sec_3-4-1.html")
	  ("&optional" "sec_3-4-1.html")
	  ("&rest" "sec_3-4-1.html")
	  ("&whole" "sec_3-4-4.html")
	  ("*" "fun_st.html")
	  ("*" "var_stcm_ststcm_ststst.html")
	  ("**" "var_stcm_ststcm_ststst.html")
	  ("***" "var_stcm_ststcm_ststst.html")
	  ("*break-on-signals*" "var_stbreak-on-signalsst.html")
	  ("*compile-file-pathname*" "var_stcompile_e-truenamest.html")
	  ("*compile-file-truename*" "var_stcompile_e-truenamest.html")
	  ("*compile-print*" "var_stcompile_le-verbosest.html")
	  ("*compile-verbose*" "var_stcompile_le-verbosest.html")
	  ("*debug-io*" "var_stdebug-i_ace-outputst.html")
	  ("*debugger-hook*" "var_stdebugger-hookst.html")
	  ("*default-pathname-defaults*" "var_stdefault_e-defaultsst.html")
	  ("*error-output*" "var_stdebug-i_ace-outputst.html")
	  ("*features*" "var_stfeaturesst.html")
	  ("*gensym-counter*" "var_stgensym-counterst.html")
	  ("*load-pathname*" "var_stload-pa_d-truenamest.html")
	  ("*load-print*" "var_stload-pr_ad-verbosest.html")
	  ("*load-truename*" "var_stload-pa_d-truenamest.html")
	  ("*load-verbose*" "var_stload-pr_ad-verbosest.html")
	  ("*macroexpand-hook*" "var_stmacroexpand-hookst.html")
	  ("*modules*" "var_stmodulesst.html")
	  ("*package*" "var_stpackagest.html")
	  ("*print-array*" "var_stprint-arrayst.html")
	  ("*print-base*" "var_stprint-b_rint-radixst.html")
	  ("*print-case*" "var_stprint-casest.html")
	  ("*print-circle*" "var_stprint-circlest.html")
	  ("*print-escape*" "var_stprint-escapest.html")
	  ("*print-gensym*" "var_stprint-gensymst.html")
	  ("*print-length*" "var_stprint-l_int-lengthst.html")
	  ("*print-level*" "var_stprint-l_int-lengthst.html")
	  ("*print-lines*" "var_stprint-linesst.html")
	  ("*print-miser-width*" "var_stprint-miser-widthst.html")
	  ("*print-pprint-dispatch*" "var_stprint-p_t-dispatchst.html")
	  ("*print-pretty*" "var_stprint-prettyst.html")
	  ("*print-radix*" "var_stprint-b_rint-radixst.html")
	  ("*print-readably*" "var_stprint-readablyst.html")
	  ("*print-right-margin*" "var_stprint-right-marginst.html")
	  ("*query-io*" "var_stdebug-i_ace-outputst.html")
	  ("*random-state*" "var_strandom-statest.html")
	  ("*read-base*" "var_stread-basest.html")
	  ("*read-default-float-format*" "var_stread-de_oat-formatst.html")
	  ("*read-eval*" "var_stread-evalst.html")
	  ("*read-suppress*" "var_stread-suppressst.html")
	  ("*readtable*" "var_streadtablest.html")
	  ("*standard-input*" "var_stdebug-i_ace-outputst.html")
	  ("*standard-output*" "var_stdebug-i_ace-outputst.html")
	  ("*terminal-io*" "var_stterminal-iost.html")
	  ("*trace-output*" "var_stdebug-i_ace-outputst.html")
	  ("+" "fun_pl.html")
	  ("+" "var_plcm_plplcm_plplpl.html")
	  ("++" "var_plcm_plplcm_plplpl.html")
	  ("+++" "var_plcm_plplcm_plplpl.html")
	  ("-" "fun_-.html")
	  ("-" "var_-.html")
	  ("/" "fun_sl.html")
	  ("/" "var_slcm_slslcm_slslsl.html")
	  ("//" "var_slcm_slslcm_slslsl.html")
	  ("///" "var_slcm_slslcm_slslsl.html")
	  ("/=" "fun_eqcm_sleq__lteqcm_gteq.html")
	  ("1+" "fun_1plcm_1-.html")
	  ("1-" "fun_1plcm_1-.html")
	  ("<" "fun_eqcm_sleq__lteqcm_gteq.html")
	  ("<=" "fun_eqcm_sleq__lteqcm_gteq.html")
	  ("=" "fun_eqcm_sleq__lteqcm_gteq.html")
	  (">" "fun_eqcm_sleq__lteqcm_gteq.html")
	  (">=" "fun_eqcm_sleq__lteqcm_gteq.html")
	  ("abort" "fun_abortcm_c_cm_use-value.html")
	  ("abort" "res_abort.html")
	  ("abs" "fun_abs.html")
	  ("acons" "fun_acons.html")
	  ("acos" "fun_asincm_acoscm_atan.html")
	  ("acosh" "fun_sinhcm_co_coshcm_atanh.html")
	  ("add-method" "stagenfun_add-method.html")
	  ("adjoin" "fun_adjoin.html")
	  ("adjust-array" "fun_adjust-array.html")
	  ("adjustable-array-p" "fun_adjustable-array-p.html")
	  ("allocate-instance" "stagenfun_all_ate-instance.html")
	  ("alpha-char-p" "fun_alpha-char-p.html")
	  ("alphanumericp" "fun_alphanumericp.html")
	  ("and" "mac_and.html")
	  ("and" "typspe_and.html")
	  ("append" "fun_append.html")
	  ("apply" "fun_apply.html")
	  ("apropos" "fun_aproposcm_apropos-list.html")
	  ("apropos-list" "fun_aproposcm_apropos-list.html")
	  ("aref" "acc_aref.html")
	  ("arithmetic-error" "contyp_arithmetic-error.html")
	  ("arithmetic-error-operands" "fun_arithmeti_or-operation.html")
	  ("arithmetic-error-operation" "fun_arithmeti_or-operation.html")
	  ("array" "syscla_array.html")
	  ("array-dimension" "fun_array-dimension.html")
	  ("array-dimension-limit" "convar_array-_ension-limit.html")
	  ("array-dimensions" "fun_array-dimensions.html")
	  ("array-displacement" "fun_array-displacement.html")
	  ("array-element-type" "fun_array-element-type.html")
	  ("array-has-fill-pointer-p" "fun_array-has_ll-pointer-p.html")
	  ("array-in-bounds-p" "fun_array-in-bounds-p.html")
	  ("array-rank" "fun_array-rank.html")
	  ("array-rank-limit" "convar_array-rank-limit.html")
	  ("array-row-major-index" "fun_array-row-major-index.html")
	  ("array-total-size" "fun_array-total-size.html")
	  ("array-total-size-limit" "convar_array-_l-size-limit.html")
	  ("arrayp" "fun_arrayp.html")
	  ("ash" "fun_ash.html")
	  ("asin" "fun_asincm_acoscm_atan.html")
	  ("asinh" "fun_sinhcm_co_coshcm_atanh.html")
	  ("assert" "mac_assert.html")
	  ("assoc" "fun_assoccm_a_assoc-if-not.html")
	  ("assoc-if" "fun_assoccm_a_assoc-if-not.html")
	  ("assoc-if-not" "fun_assoccm_a_assoc-if-not.html")
	  ("atan" "fun_asincm_acoscm_atan.html")
	  ("atanh" "fun_sinhcm_co_coshcm_atanh.html")
	  ("atom" "fun_atom.html")
	  ("atom" "typ_atom.html")
	  ("base-char" "typ_base-char.html")
	  ("base-string" "typ_base-string.html")
	  ("bignum" "typ_bignum.html")
	  ("bit" "acc_bitcm_sbit.html")
	  ("bit" "typ_bit.html")
	  ("bit-and" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-andc1" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-andc2" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-eqv" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-ior" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-nand" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-nor" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-not" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-orc1" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-orc2" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("bit-vector" "syscla_bit-vector.html")
	  ("bit-vector-p" "fun_bit-vector-p.html")
	  ("bit-xor" "fun_bit-andcm_c2cm_bit-xor.html")
	  ("block" "speope_block.html")
	  ("boole" "fun_boole.html")
	  ("boole-1" "convar_boole-_cm_boole-xor.html")
	  ("boole-2" "convar_boole-_cm_boole-xor.html")
	  ("boole-and" "convar_boole-_cm_boole-xor.html")
	  ("boole-andc1" "convar_boole-_cm_boole-xor.html")
	  ("boole-andc2" "convar_boole-_cm_boole-xor.html")
	  ("boole-c1" "convar_boole-_cm_boole-xor.html")
	  ("boole-c2" "convar_boole-_cm_boole-xor.html")
	  ("boole-clr" "convar_boole-_cm_boole-xor.html")
	  ("boole-eqv" "convar_boole-_cm_boole-xor.html")
	  ("boole-ior" "convar_boole-_cm_boole-xor.html")
	  ("boole-nand" "convar_boole-_cm_boole-xor.html")
	  ("boole-nor" "convar_boole-_cm_boole-xor.html")
	  ("boole-orc1" "convar_boole-_cm_boole-xor.html")
	  ("boole-orc2" "convar_boole-_cm_boole-xor.html")
	  ("boole-set" "convar_boole-_cm_boole-xor.html")
	  ("boole-xor" "convar_boole-_cm_boole-xor.html")
	  ("boolean" "typ_boolean.html")
	  ("both-case-p" "fun_upper-cas__both-case-p.html")
	  ("boundp" "fun_boundp.html")
	  ("break" "fun_break.html")
	  ("broadcast-stream" "syscla_broadcast-stream.html")
	  ("broadcast-stream-streams" "fun_broadcast_ream-streams.html")
	  ("built-in-class" "syscla_built-in-class.html")
	  ("butlast" "fun_butlastcm_nbutlast.html")
	  ("byte" "fun_bytecm_by_yte-position.html")
	  ("byte-position" "fun_bytecm_by_yte-position.html")
	  ("byte-size" "fun_bytecm_by_yte-position.html")
	  ("caaaar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("caaadr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("caaar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("caadar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("caaddr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("caadr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("caar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cadaar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cadadr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cadar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("caddar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cadddr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("caddr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cadr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("call-arguments-limit" "convar_call-a_uments-limit.html")
	  ("call-method" "locmac_call-m__make-method.html")
	  ("call-next-method" "locfun_call-next-method.html")
	  ("car" "acc_carcm_cdr_darcm_cddddr.html")
	  ("case" "mac_casecm_ccasecm_ecase.html")
	  ("catch" "speope_catch.html")
	  ("ccase" "mac_casecm_ccasecm_ecase.html")
	  ("cdaaar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cdaadr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cdaar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cdadar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cdaddr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cdadr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cdar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cddaar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cddadr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cddar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cdddar" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cddddr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cdddr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cddr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("cdr" "acc_carcm_cdr_darcm_cddddr.html")
	  ("ceiling" "fun_floorcm_f_undcm_fround.html")
	  ("cell-error" "contyp_cell-error.html")
	  ("cell-error-name" "fun_cell-error-name.html")
	  ("cerror" "fun_cerror.html")
	  ("change-class" "stagenfun_change-class.html")
	  ("char" "acc_charcm_schar.html")
	  ("char-code" "fun_char-code.html")
	  ("char-code-limit" "convar_char-code-limit.html")
	  ("char-downcase" "fun_char-upca_har-downcase.html")
	  ("char-equal" "fun_chareqcm__ar-not-lessp.html")
	  ("char-greaterp" "fun_chareqcm__ar-not-lessp.html")
	  ("char-int" "fun_char-int.html")
	  ("char-lessp" "fun_chareqcm__ar-not-lessp.html")
	  ("char-name" "fun_char-name.html")
	  ("char-not-equal" "fun_chareqcm__ar-not-lessp.html")
	  ("char-not-greaterp" "fun_chareqcm__ar-not-lessp.html")
	  ("char-not-lessp" "fun_chareqcm__ar-not-lessp.html")
	  ("char-upcase" "fun_char-upca_har-downcase.html")
	  ("char/=" "fun_chareqcm__ar-not-lessp.html")
	  ("char<" "fun_chareqcm__ar-not-lessp.html")
	  ("char<=" "fun_chareqcm__ar-not-lessp.html")
	  ("char=" "fun_chareqcm__ar-not-lessp.html")
	  ("char>" "fun_chareqcm__ar-not-lessp.html")
	  ("char>=" "fun_chareqcm__ar-not-lessp.html")
	  ("character" "fun_character.html")
	  ("character" "syscla_character.html")
	  ("characterp" "fun_characterp.html")
	  ("check-type" "mac_check-type.html")
	  ("cis" "fun_cis.html")
	  ("class" "syscla_class.html")
	  ("class-name" "stagenfun_class-name.html")
	  ("class-of" "fun_class-of.html")
	  ("clear-input" "fun_clear-input.html")
	  ("clear-output" "fun_finish-ou_clear-output.html")
	  ("close" "fun_close.html")
	  ("clrhash" "fun_clrhash.html")
	  ("code-char" "fun_code-char.html")
	  ("coerce" "fun_coerce.html")
	  ("compilation-speed" "dec_optimize.html")
	  ("compile" "fun_compile.html")
	  ("compile-file" "fun_compile-file.html")
	  ("compile-file-pathname" "fun_compile-file-pathname.html")
	  ("compiled-function" "typ_compiled-function.html")
	  ("compiled-function-p" "fun_compiled-function-p.html")
	  ("compiler-macro" "stagenfun_doc_umentationcp.html")
	  ("compiler-macro-function" "acc_compiler-_cro-function.html")
	  ("complement" "fun_complement.html")
	  ("complex" "fun_complex.html")
	  ("complex" "syscla_complex.html")
	  ("complexp" "fun_complexp.html")
	  ("compute-applicable-methods" "stagenfun_com_able-methods.html")
	  ("compute-restarts" "fun_compute-restarts.html")
	  ("concatenate" "fun_concatenate.html")
	  ("concatenated-stream" "syscla_concatenated-stream.html")
	  ("concatenated-stream-streams" "fun_concatena_ream-streams.html")
	  ("cond" "mac_cond.html")
	  ("condition" "contyp_condition.html")
	  ("conjugate" "fun_conjugate.html")
	  ("cons" "fun_cons.html")
	  ("cons" "syscla_cons.html")
	  ("consp" "fun_consp.html")
	  ("constantly" "fun_constantly.html")
	  ("constantp" "fun_constantp.html")
	  ("continue" "fun_abortcm_c_cm_use-value.html")
	  ("continue" "res_continue.html")
	  ("control-error" "contyp_control-error.html")
	  ("copy-alist" "fun_copy-alist.html")
	  ("copy-list" "fun_copy-list.html")
	  ("copy-pprint-dispatch" "fun_copy-pprint-dispatch.html")
	  ("copy-readtable" "fun_copy-readtable.html")
	  ("copy-seq" "fun_copy-seq.html")
	  ("copy-structure" "fun_copy-structure.html")
	  ("copy-symbol" "fun_copy-symbol.html")
	  ("copy-tree" "fun_copy-tree.html")
	  ("cos" "fun_sincm_coscm_tan.html")
	  ("cosh" "fun_sinhcm_co_coshcm_atanh.html")
	  ("count" "fun_countcm_c_count-if-not.html")
	  ("count-if" "fun_countcm_c_count-if-not.html")
	  ("count-if-not" "fun_countcm_c_count-if-not.html")
	  ("ctypecase" "mac_typecasec_cm_etypecase.html")
	  ("debug" "dec_optimize.html")
	  ("decf" "mac_incfcm_decf.html")
	  ("declaim" "mac_declaim.html")
	  ("declaration" "dec_declaration.html")
	  ("declare" "sym_declare.html")
	  ("decode-float" "fun_decode-fl_decode-float.html")
	  ("decode-universal-time" "fun_decode-universal-time.html")
	  ("defclass" "mac_defclass.html")
	  ("defconstant" "mac_defconstant.html")
	  ("defgeneric" "mac_defgeneric.html")
	  ("define-compiler-macro" "mac_define-compiler-macro.html")
	  ("define-condition" "mac_define-condition.html")
	  ("define-method-combination" "mac_define-me_-combination.html")
	  ("define-modify-macro" "mac_define-modify-macro.html")
	  ("define-setf-expander" "mac_define-setf-expander.html")
	  ("define-symbol-macro" "mac_define-symbol-macro.html")
	  ("defmacro" "mac_defmacro.html")
	  ("defmethod" "mac_defmethod.html")
	  ("defpackage" "mac_defpackage.html")
	  ("defparameter" "mac_defparametercm_defvar.html")
	  ("defsetf" "mac_defsetf.html")
	  ("defstruct" "mac_defstruct.html")
	  ("deftype" "mac_deftype.html")
	  ("defun" "mac_defun.html")
	  ("defvar" "mac_defparametercm_defvar.html")
	  ("delete" "fun_removecm__elete-if-not.html")
	  ("delete-duplicates" "fun_remove-du_e-duplicates.html")
	  ("delete-file" "fun_delete-file.html")
	  ("delete-if" "fun_removecm__elete-if-not.html")
	  ("delete-if-not" "fun_removecm__elete-if-not.html")
	  ("delete-package" "fun_delete-package.html")
	  ("denominator" "fun_numerator__denominator.html")
	  ("deposit-field" "fun_deposit-field.html")
	  ("describe" "fun_describe.html")
	  ("describe-object" "stagenfun_describe-object.html")
	  ("destructuring-bind" "mac_destructuring-bind.html")
	  ("digit-char" "fun_digit-char.html")
	  ("digit-char-p" "fun_digit-char-p.html")
	  ("directory" "fun_directory.html")
	  ("directory-namestring" "fun_namestrin_h-namestring.html")
	  ("disassemble" "fun_disassemble.html")
	  ("division-by-zero" "contyp_division-by-zero.html")
	  ("do" "mac_docm_dost.html")
	  ("do*" "mac_docm_dost.html")
	  ("do-all-symbols" "mac_do-symbol_-all-symbols.html")
	  ("do-external-symbols" "mac_do-symbol_-all-symbols.html")
	  ("do-symbols" "mac_do-symbol_-all-symbols.html")
	  ("documentation" "stagenfun_doc_umentationcp.html")
	  ("dolist" "mac_dolist.html")
	  ("dotimes" "mac_dotimes.html")
	  ("double-float" "typ_short-flo_m_long-float.html")
	  ("double-float-epsilon" "convar_short-_tive-epsilon.html")
	  ("double-float-negative-epsilon" "convar_short-_tive-epsilon.html")
	  ("dpb" "fun_dpb.html")
	  ("dribble" "fun_dribble.html")
	  ("dynamic-extent" "dec_dynamic-extent.html")
	  ("ecase" "mac_casecm_ccasecm_ecase.html")
	  ("echo-stream" "syscla_echo-stream.html")
	  ("echo-stream-input-stream" "fun_echo-stre_utput-stream.html")
	  ("echo-stream-output-stream" "fun_echo-stre_utput-stream.html")
	  ("ed" "fun_ed.html")
	  ("eighth" "acc_firstcm_s_inthcm_tenth.html")
	  ("elt" "acc_elt.html")
	  ("encode-universal-time" "fun_encode-universal-time.html")
	  ("end-of-file" "contyp_end-of-file.html")
	  ("endp" "fun_endp.html")
	  ("enough-namestring" "fun_namestrin_h-namestring.html")
	  ("ensure-directories-exist" "fun_ensure-di_tories-exist.html")
	  ("ensure-generic-function" "fun_ensure-ge_ric-function.html")
	  ("eq" "fun_eq.html")
	  ("eql" "fun_eql.html")
	  ("eql" "typspe_eql.html")
	  ("equal" "fun_equal.html")
	  ("equalp" "fun_equalp.html")
	  ("error" "fun_error.html")
	  ("error" "contyp_error.html")
	  ("etypecase" "mac_typecasec_cm_etypecase.html")
	  ("eval" "fun_eval.html")
	  ("eval-when" "speope_eval-when.html")
	  ("evenp" "fun_evenpcm_oddp.html")
	  ("every" "fun_everycm_s_erycm_notany.html")
	  ("exp" "fun_expcm_expt.html")
	  ("export" "fun_export.html")
	  ("expt" "fun_expcm_expt.html")
	  ("extended-char" "typ_extended-char.html")
	  ("fboundp" "fun_fboundp.html")
	  ("fceiling" "fun_floorcm_f_undcm_fround.html")
	  ("fdefinition" "acc_fdefinition.html")
	  ("ffloor" "fun_floorcm_f_undcm_fround.html")
	  ("fifth" "acc_firstcm_s_inthcm_tenth.html")
	  ("file-author" "fun_file-author.html")
	  ("file-error" "contyp_file-error.html")
	  ("file-error-pathname" "fun_file-error-pathname.html")
	  ("file-length" "fun_file-length.html")
	  ("file-namestring" "fun_namestrin_h-namestring.html")
	  ("file-position" "fun_file-position.html")
	  ("file-stream" "syscla_file-stream.html")
	  ("file-string-length" "fun_file-string-length.html")
	  ("file-write-date" "fun_file-write-date.html")
	  ("fill" "fun_fill.html")
	  ("fill-pointer" "acc_fill-pointer.html")
	  ("find" "fun_findcm_fi__find-if-not.html")
	  ("find-all-symbols" "fun_find-all-symbols.html")
	  ("find-class" "acc_find-class.html")
	  ("find-if" "fun_findcm_fi__find-if-not.html")
	  ("find-if-not" "fun_findcm_fi__find-if-not.html")
	  ("find-method" "stagenfun_find-method.html")
	  ("find-package" "fun_find-package.html")
	  ("find-restart" "fun_find-restart.html")
	  ("find-symbol" "fun_find-symbol.html")
	  ("finish-output" "fun_finish-ou_clear-output.html")
	  ("first" "acc_firstcm_s_inthcm_tenth.html")
	  ("fixnum" "typ_fixnum.html")
	  ("flet" "speope_fletcm_scm_macrolet.html")
	  ("float" "fun_float.html")
	  ("float" "syscla_float.html")
	  ("float-digits" "fun_decode-fl_decode-float.html")
	  ("float-precision" "fun_decode-fl_decode-float.html")
	  ("float-radix" "fun_decode-fl_decode-float.html")
	  ("float-sign" "fun_decode-fl_decode-float.html")
	  ("floating-point-inexact" "contyp_floati_oint-inexact.html")
	  ("floating-point-invalid-operation" "contyp_floati_id-operation.html")
	  ("floating-point-overflow" "contyp_floati_int-overflow.html")
	  ("floating-point-underflow" "contyp_floati_nt-underflow.html")
	  ("floatp" "fun_floatp.html")
	  ("floor" "fun_floorcm_f_undcm_fround.html")
	  ("fmakunbound" "fun_fmakunbound.html")
	  ("force-output" "fun_finish-ou_clear-output.html")
	  ("format" "fun_format.html")
	  ("formatter" "mac_formatter.html")
	  ("fourth" "acc_firstcm_s_inthcm_tenth.html")
	  ("fresh-line" "fun_terpricm_fresh-line.html")
	  ("fround" "fun_floorcm_f_undcm_fround.html")
	  ("ftruncate" "fun_floorcm_f_undcm_fround.html")
	  ("ftype" "dec_ftype.html")
	  ("funcall" "fun_funcall.html")
	  ("function" "speope_function.html")
	  ("function" "stagenfun_doc_umentationcp.html")
	  ("function" "syscla_function.html")
	  ("function-keywords" "stagenfun_fun_ion-keywords.html")
	  ("function-lambda-expression" "fun_function-_a-expression.html")
	  ("functionp" "fun_functionp.html")
	  ("gcd" "fun_gcd.html")
	  ("generic-function" "syscla_generic-function.html")
	  ("gensym" "fun_gensym.html")
	  ("gentemp" "fun_gentemp.html")
	  ("get" "acc_get.html")
	  ("get-decoded-time" "fun_get-unive_decoded-time.html")
	  ("get-dispatch-macro-character" "fun_set-dispa_ro-character.html")
	  ("get-internal-real-time" "fun_get-internal-real-time.html")
	  ("get-internal-run-time" "fun_get-internal-run-time.html")
	  ("get-macro-character" "fun_set-macro_ro-character.html")
	  ("get-output-stream-string" "fun_get-outpu_tream-string.html")
	  ("get-properties" "fun_get-properties.html")
	  ("get-setf-expansion" "fun_get-setf-expansion.html")
	  ("get-universal-time" "fun_get-unive_decoded-time.html")
	  ("getf" "acc_getf.html")
	  ("gethash" "acc_gethash.html")
	  ("go" "speope_go.html")
	  ("graphic-char-p" "fun_graphic-char-p.html")
	  ("handler-bind" "mac_handler-bind.html")
	  ("handler-case" "mac_handler-case.html")
	  ("hash-table" "syscla_hash-table.html")
	  ("hash-table-count" "fun_hash-table-count.html")
	  ("hash-table-p" "fun_hash-table-p.html")
	  ("hash-table-rehash-size" "fun_hash-table-rehash-size.html")
	  ("hash-table-rehash-threshold" "fun_hash-tabl_sh-threshold.html")
	  ("hash-table-size" "fun_hash-table-size.html")
	  ("hash-table-test" "fun_hash-table-test.html")
	  ("host-namestring" "fun_namestrin_h-namestring.html")
	  ("identity" "fun_identity.html")
	  ("if" "speope_if.html")
	  ("ignorable" "dec_ignorecm_ignorable.html")
	  ("ignore" "dec_ignorecm_ignorable.html")
	  ("ignore-errors" "mac_ignore-errors.html")
	  ("imagpart" "fun_realpartcm_imagpart.html")
	  ("import" "fun_import.html")
	  ("in-package" "mac_in-package.html")
	  ("incf" "mac_incfcm_decf.html")
	  ("initialize-instance" "stagenfun_ini_ize-instance.html")
	  ("inline" "dec_inlinecm_notinline.html")
	  ("input-stream-p" "fun_input-str_put-stream-p.html")
	  ("inspect" "fun_inspect.html")
	  ("integer" "syscla_integer.html")
	  ("integer-decode-float" "fun_decode-fl_decode-float.html")
	  ("integer-length" "fun_integer-length.html")
	  ("integerp" "fun_integerp.html")
	  ("interactive-stream-p" "fun_interactive-stream-p.html")
	  ("intern" "fun_intern.html")
	  ("internal-time-units-per-second" "convar_intern_s-per-second.html")
	  ("intersection" "fun_intersect_intersection.html")
	  ("invalid-method-error" "fun_invalid-method-error.html")
	  ("invoke-debugger" "fun_invoke-debugger.html")
	  ("invoke-restart" "fun_invoke-restart.html")
	  ("invoke-restart-interactively" "fun_invoke-re_nteractively.html")
	  ("isqrt" "fun_sqrtcm_isqrt.html")
	  ("keyword" "typ_keyword.html")
	  ("keywordp" "fun_keywordp.html")
	  ("labels" "speope_fletcm_scm_macrolet.html")
	  ("lambda" "mac_lambda.html")
	  ("lambda" "sym_lambda.html")
	  ("lambda-list-keywords" "convar_lambda_ist-keywords.html")
	  ("lambda-parameters-limit" "convar_lambda_meters-limit.html")
	  ("last" "fun_last.html")
	  ("lcm" "fun_lcm.html")
	  ("ldb" "acc_ldb.html")
	  ("ldb-test" "fun_ldb-test.html")
	  ("ldiff" "fun_ldiffcm_tailp.html")
	  ("least-negative-double-float" "convar_most-p_d-long-float.html")
	  ("least-negative-long-float" "convar_most-p_d-long-float.html")
	  ("least-negative-normalized-double-float" "convar_most-p_d-long-float.html")
	  ("least-negative-normalized-long-float" "convar_most-p_d-long-float.html")
	  ("least-negative-normalized-short-float" "convar_most-p_d-long-float.html")
	  ("least-negative-normalized-single-float" "convar_most-p_d-long-float.html")
	  ("least-negative-short-float" "convar_most-p_d-long-float.html")
	  ("least-negative-single-float" "convar_most-p_d-long-float.html")
	  ("least-positive-double-float" "convar_most-p_d-long-float.html")
	  ("least-positive-long-float" "convar_most-p_d-long-float.html")
	  ("least-positive-normalized-double-float" "convar_most-p_d-long-float.html")
	  ("least-positive-normalized-long-float" "convar_most-p_d-long-float.html")
	  ("least-positive-normalized-short-float" "convar_most-p_d-long-float.html")
	  ("least-positive-normalized-single-float" "convar_most-p_d-long-float.html")
	  ("least-positive-short-float" "convar_most-p_d-long-float.html")
	  ("least-positive-single-float" "convar_most-p_d-long-float.html")
	  ("length" "fun_length.html")
	  ("let" "speope_letcm_letst.html")
	  ("let*" "speope_letcm_letst.html")
	  ("lisp-implementation-type" "fun_lisp-impl_tion-version.html")
	  ("lisp-implementation-version" "fun_lisp-impl_tion-version.html")
	  ("list" "fun_listcm_listst_list.html")
	  ("list" "syscla_list.html")
	  ("list*" "fun_listcm_listst.html")
	  ("list-all-packages" "fun_list-all-packages.html")
	  ("list-length" "fun_list-length.html")
	  ("listen" "fun_listen.html")
	  ("listp" "fun_listp.html")
	  ("load" "fun_load.html")
	  ("load-logical-pathname-translations" "fun_load-logi_translations.html")
	  ("load-time-value" "speope_load-time-value.html")
	  ("locally" "speope_locally.html")
	  ("log" "fun_log.html")
	  ("logand" "fun_logandcm__rc2cm_logxor.html")
	  ("logandc1" "fun_logandcm__rc2cm_logxor.html")
	  ("logandc2" "fun_logandcm__rc2cm_logxor.html")
	  ("logbitp" "fun_logbitp.html")
	  ("logcount" "fun_logcount.html")
	  ("logeqv" "fun_logandcm__rc2cm_logxor.html")
	  ("logical-pathname" "fun_logical-pathname.html")
	  ("logical-pathname" "syscla_logical-pathname.html")
	  ("logical-pathname-translations" "acc_logical-p_translations.html")
	  ("logior" "fun_logandcm__rc2cm_logxor.html")
	  ("lognand" "fun_logandcm__rc2cm_logxor.html")
	  ("lognor" "fun_logandcm__rc2cm_logxor.html")
	  ("lognot" "fun_logandcm__rc2cm_logxor.html")
	  ("logorc1" "fun_logandcm__rc2cm_logxor.html")
	  ("logorc2" "fun_logandcm__rc2cm_logxor.html")
	  ("logtest" "fun_logtest.html")
	  ("logxor" "fun_logandcm__rc2cm_logxor.html")
	  ("long-float" "typ_short-flo_m_long-float.html")
	  ("long-float-epsilon" "convar_short-_tive-epsilon.html")
	  ("long-float-negative-epsilon" "convar_short-_tive-epsilon.html")
	  ("long-site-name" "fun_short-sit_ng-site-name.html")
	  ("loop" "mac_loop.html")
	  ("loop-finish" "locmac_loop-finish.html")
	  ("lower-case-p" "fun_upper-cas__both-case-p.html")
	  ("machine-instance" "fun_machine-instance.html")
	  ("machine-type" "fun_machine-type.html")
	  ("machine-version" "fun_machine-version.html")
	  ("macro-function" "acc_macro-function.html")
	  ("macroexpand" "fun_macroexpa_acroexpand-1.html")
	  ("macroexpand-1" "fun_macroexpa_acroexpand-1.html")
	  ("macrolet" "speope_fletcm_scm_macrolet.html")
	  ("make-array" "fun_make-array.html")
	  ("make-broadcast-stream" "fun_make-broadcast-stream.html")
	  ("make-concatenated-stream" "fun_make-conc_nated-stream.html")
	  ("make-condition" "fun_make-condition.html")
	  ("make-dispatch-macro-character" "fun_make-disp_ro-character.html")
	  ("make-echo-stream" "fun_make-echo-stream.html")
	  ("make-hash-table" "fun_make-hash-table.html")
	  ("make-instance" "stagenfun_make-instance.html")
	  ("make-instances-obsolete" "stagenfun_mak_ces-obsolete.html")
	  ("make-list" "fun_make-list.html")
	  ("make-load-form" "stagenfun_make-load-form.html")
	  ("make-load-form-saving-slots" "fun_make-load_saving-slots.html")
	  ("make-method" "locmac_call-m__make-method.html")
	  ("make-package" "fun_make-package.html")
	  ("make-pathname" "fun_make-pathname.html")
	  ("make-random-state" "fun_make-random-state.html")
	  ("make-sequence" "fun_make-sequence.html")
	  ("make-string" "fun_make-string.html")
	  ("make-string-input-stream" "fun_make-stri_input-stream.html")
	  ("make-string-output-stream" "fun_make-stri_utput-stream.html")
	  ("make-symbol" "fun_make-symbol.html")
	  ("make-synonym-stream" "fun_make-synonym-stream.html")
	  ("make-two-way-stream" "fun_make-two-way-stream.html")
	  ("makunbound" "fun_makunbound.html")
	  ("map" "fun_map.html")
	  ("map-into" "fun_map-into.html")
	  ("mapc" "fun_mapccm_ma_istcm_mapcon.html")
	  ("mapcan" "fun_mapccm_ma_istcm_mapcon.html")
	  ("mapcar" "fun_mapccm_ma_istcm_mapcon.html")
	  ("mapcon" "fun_mapccm_ma_istcm_mapcon.html")
	  ("maphash" "fun_maphash.html")
	  ("mapl" "fun_mapccm_ma_istcm_mapcon.html")
	  ("maplist" "fun_mapccm_ma_istcm_mapcon.html")
	  ("mask-field" "acc_mask-field.html")
	  ("max" "fun_maxcm_min.html")
	  ("member" "fun_membercm__ember-if-not.html")
	  ("member" "typspe_member.html")
	  ("member-if" "fun_membercm__ember-if-not.html")
	  ("member-if-not" "fun_membercm__ember-if-not.html")
	  ("merge" "fun_merge.html")
	  ("merge-pathnames" "fun_merge-pathnames.html")
	  ("method" "syscla_method.html")
	  ("method-combination" "stagenfun_doc_umentationcp.html")
	  ("method-combination" "syscla_method-combination.html")
	  ("method-combination-error" "fun_method-co_nation-error.html")
	  ("method-qualifiers" "stagenfun_met_d-qualifiers.html")
	  ("min" "fun_maxcm_min.html")
	  ("minusp" "fun_minuspcm_plusp.html")
	  ("mismatch" "fun_mismatch.html")
	  ("mod" "fun_modcm_rem.html")
	  ("mod" "typspe_mod.html")
	  ("most-negative-double-float" "convar_most-p_d-long-float.html")
	  ("most-negative-fixnum" "convar_most-p_ative-fixnum.html")
	  ("most-negative-long-float" "convar_most-p_d-long-float.html")
	  ("most-negative-short-float" "convar_most-p_d-long-float.html")
	  ("most-negative-single-float" "convar_most-p_d-long-float.html")
	  ("most-positive-double-float" "convar_most-p_d-long-float.html")
	  ("most-positive-fixnum" "convar_most-p_ative-fixnum.html")
	  ("most-positive-long-float" "convar_most-p_d-long-float.html")
	  ("most-positive-short-float" "convar_most-p_d-long-float.html")
	  ("most-positive-single-float" "convar_most-p_d-long-float.html")
	  ("muffle-warning" "fun_abortcm_c_cm_use-value.html")
	  ("muffle-warning" "res_muffle-warning.html")
	  ("multiple-value-bind" "mac_multiple-value-bind.html")
	  ("multiple-value-call" "speope_multiple-value-call.html")
	  ("multiple-value-list" "mac_multiple-value-list.html")
	  ("multiple-value-prog1" "speope_multip_-value-prog1.html")
	  ("multiple-value-setq" "mac_multiple-value-setq.html")
	  ("multiple-values-limit" "convar_multip_values-limit.html")
	  ("name-char" "fun_name-char.html")
	  ("namestring" "fun_namestrin_h-namestring.html")
	  ("nbutlast" "fun_butlastcm_nbutlast.html")
	  ("nconc" "fun_nconc.html")
	  ("next-method-p" "locfun_next-method-p.html")
	  ("nil" "convar_nil.html")
	  ("nil" "typ_nil.html")
	  ("nintersection" "fun_intersect_intersection.html")
	  ("ninth" "acc_firstcm_s_inthcm_tenth.html")
	  ("no-applicable-method" "stagenfun_no-_cable-method.html")
	  ("no-next-method" "stagenfun_no-next-method.html")
	  ("not" "fun_not.html")
	  ("not" "typspe_not.html")
	  ("notany" "fun_everycm_s_erycm_notany.html")
	  ("notevery" "fun_everycm_s_erycm_notany.html")
	  ("notinline" "dec_inlinecm_notinline.html")
	  ("nreconc" "fun_revappendcm_nreconc.html")
	  ("nreverse" "fun_reversecm_nreverse.html")
	  ("nset-difference" "fun_set-diffe_t-difference.html")
	  ("nset-exclusive-or" "fun_set-exclu_exclusive-or.html")
	  ("nstring-capitalize" "fun_string-up_g-capitalize.html")
	  ("nstring-downcase" "fun_string-up_g-capitalize.html")
	  ("nstring-upcase" "fun_string-up_g-capitalize.html")
	  ("nsublis" "fun_subliscm_nsublis.html")
	  ("nsubst" "fun_substcm_s_subst-if-not.html")
	  ("nsubst-if" "fun_substcm_s_subst-if-not.html")
	  ("nsubst-if-not" "fun_substcm_s_subst-if-not.html")
	  ("nsubstitute" "fun_substitut_itute-if-not.html")
	  ("nsubstitute-if" "fun_substitut_itute-if-not.html")
	  ("nsubstitute-if-not" "fun_substitut_itute-if-not.html")
	  ("nth" "acc_nth.html")
	  ("nth-value" "mac_nth-value.html")
	  ("nthcdr" "fun_nthcdr.html")
	  ("null" "fun_null.html")
	  ("null" "syscla_null.html")
	  ("number" "syscla_number.html")
	  ("numberp" "fun_numberp.html")
	  ("numerator" "fun_numerator__denominator.html")
	  ("nunion" "fun_unioncm_nunion.html")
	  ("oddp" "fun_evenpcm_oddp.html")
	  ("open" "fun_open.html")
	  ("open-stream-p" "fun_open-stream-p.html")
	  ("optimize" "dec_optimize.html")
	  ("or" "mac_or.html")
	  ("or" "typspe_or.html")
	  ("otherwise" "mac_casecm_ccasecm_ecase.html")
	  ("output-stream-p" "fun_input-str_put-stream-p.html")
	  ("package" "syscla_package.html")
	  ("package-error" "contyp_package-error.html")
	  ("package-error-package" "fun_package-error-package.html")
	  ("package-name" "fun_package-name.html")
	  ("package-nicknames" "fun_package-nicknames.html")
	  ("package-shadowing-symbols" "fun_package-s_wing-symbols.html")
	  ("package-use-list" "fun_package-use-list.html")
	  ("package-used-by-list" "fun_package-used-by-list.html")
	  ("packagep" "fun_packagep.html")
	  ("pairlis" "fun_pairlis.html")
	  ("parse-error" "contyp_parse-error.html")
	  ("parse-integer" "fun_parse-integer.html")
	  ("parse-namestring" "fun_parse-namestring.html")
	  ("pathname" "fun_pathname.html")
	  ("pathname" "syscla_pathname.html")
	  ("pathname-device" "fun_pathname-_name-version.html")
	  ("pathname-directory" "fun_pathname-_name-version.html")
	  ("pathname-host" "fun_pathname-_name-version.html")
	  ("pathname-match-p" "fun_pathname-match-p.html")
	  ("pathname-name" "fun_pathname-_name-version.html")
	  ("pathname-type" "fun_pathname-_name-version.html")
	  ("pathname-version" "fun_pathname-_name-version.html")
	  ("pathnamep" "fun_pathnamep.html")
	  ("peek-char" "fun_peek-char.html")
	  ("phase" "fun_phase.html")
	  ("pi" "convar_pi.html")
	  ("plusp" "fun_minuspcm_plusp.html")
	  ("pop" "mac_pop.html")
	  ("position" "fun_positionc_ition-if-not.html")
	  ("position-if" "fun_positionc_ition-if-not.html")
	  ("position-if-not" "fun_positionc_ition-if-not.html")
	  ("pprint" "fun_writecm_p_rintcm_princ.html")
	  ("pprint-dispatch" "fun_pprint-dispatch.html")
	  ("pprint-exit-if-list-exhausted" "locmac_pprint_st-exhausted.html")
	  ("pprint-fill" "fun_pprint-fi_rint-tabular.html")
	  ("pprint-indent" "fun_pprint-indent.html")
	  ("pprint-linear" "fun_pprint-fi_rint-tabular.html")
	  ("pprint-logical-block" "mac_pprint-logical-block.html")
	  ("pprint-newline" "fun_pprint-newline.html")
	  ("pprint-pop" "locmac_pprint-pop.html")
	  ("pprint-tab" "fun_pprint-tab.html")
	  ("pprint-tabular" "fun_pprint-fi_rint-tabular.html")
	  ("prin1" "fun_writecm_p_rintcm_princ.html")
	  ("prin1-to-string" "fun_write-to-_nc-to-string.html")
	  ("princ" "fun_writecm_p_rintcm_princ.html")
	  ("princ-to-string" "fun_write-to-_nc-to-string.html")
	  ("print" "fun_writecm_p_rintcm_princ.html")
	  ("print-not-readable" "contyp_print-not-readable.html")
	  ("print-not-readable-object" "fun_print-not_dable-object.html")
	  ("print-object" "stagenfun_print-object.html")
	  ("print-unreadable-object" "mac_print-unr_dable-object.html")
	  ("probe-file" "fun_probe-file.html")
	  ("proclaim" "fun_proclaim.html")
	  ("prog" "mac_progcm_progst.html")
	  ("prog*" "mac_progcm_progst.html")
	  ("prog1" "mac_prog1cm_prog2.html")
	  ("prog2" "mac_prog1cm_prog2.html")
	  ("progn" "speope_progn.html")
	  ("program-error" "contyp_program-error.html")
	  ("progv" "speope_progv.html")
	  ("provide" "fun_providecm_require.html")
	  ("psetf" "mac_setfcm_psetf.html")
	  ("psetq" "mac_psetq.html")
	  ("push" "mac_push.html")
	  ("pushnew" "mac_pushnew.html")
	  ("quote" "speope_quote.html")
	  ("random" "fun_random.html")
	  ("random-state" "syscla_random-state.html")
	  ("random-state-p" "fun_random-state-p.html")
	  ("rassoc" "fun_rassoccm__assoc-if-not.html")
	  ("rassoc-if" "fun_rassoccm__assoc-if-not.html")
	  ("rassoc-if-not" "fun_rassoccm__assoc-if-not.html")
	  ("ratio" "syscla_ratio.html")
	  ("rational" "fun_rationalcm_rationalize.html")
	  ("rational" "syscla_rational.html")
	  ("rationalize" "fun_rationalcm_rationalize.html")
	  ("rationalp" "fun_rationalp.html")
	  ("read" "fun_readcm_re_g-whitespace.html")
	  ("read-byte" "fun_read-byte.html")
	  ("read-char" "fun_read-char.html")
	  ("read-char-no-hang" "fun_read-char-no-hang.html")
	  ("read-delimited-list" "fun_read-delimited-list.html")
	  ("read-from-string" "fun_read-from-string.html")
	  ("read-line" "fun_read-line.html")
	  ("read-preserving-whitespace" "fun_readcm_re_g-whitespace.html")
	  ("read-sequence" "fun_read-sequence.html")
	  ("reader-error" "contyp_reader-error.html")
	  ("readtable" "syscla_readtable.html")
	  ("readtable-case" "acc_readtable-case.html")
	  ("readtablep" "fun_readtablep.html")
	  ("real" "syscla_real.html")
	  ("realp" "fun_realp.html")
	  ("realpart" "fun_realpartcm_imagpart.html")
	  ("reduce" "fun_reduce.html")
	  ("reinitialize-instance" "stagenfun_rei_ize-instance.html")
	  ("rem" "fun_modcm_rem.html")
	  ("remf" "mac_remf.html")
	  ("remhash" "fun_remhash.html")
	  ("remove" "fun_removecm__elete-if-not.html")
	  ("remove-duplicates" "fun_remove-du_e-duplicates.html")
	  ("remove-if" "fun_removecm__elete-if-not.html")
	  ("remove-if-not" "fun_removecm__elete-if-not.html")
	  ("remove-method" "stagenfun_remove-method.html")
	  ("remprop" "fun_remprop.html")
	  ("rename-file" "fun_rename-file.html")
	  ("rename-package" "fun_rename-package.html")
	  ("replace" "fun_replace.html")
	  ("require" "fun_providecm_require.html")
	  ("rest" "acc_rest.html")
	  ("restart" "syscla_restart.html")
	  ("restart-bind" "mac_restart-bind.html")
	  ("restart-case" "mac_restart-case.html")
	  ("restart-name" "fun_restart-name.html")
	  ("return" "mac_return.html")
	  ("return-from" "speope_return-from.html")
	  ("revappend" "fun_revappendcm_nreconc.html")
	  ("reverse" "fun_reversecm_nreverse.html")
	  ("room" "fun_room.html")
	  ("rotatef" "mac_rotatef.html")
	  ("round" "fun_floorcm_f_undcm_fround.html")
	  ("row-major-aref" "acc_row-major-aref.html")
	  ("rplaca" "fun_rplacacm_rplacd.html")
	  ("rplacd" "fun_rplacacm_rplacd.html")
	  ("safety" "dec_optimize.html")
	  ("satisfies" "typspe_satisfies.html")
	  ("sbit" "acc_bitcm_sbit.html")
	  ("scale-float" "fun_decode-fl_decode-float.html")
	  ("schar" "acc_charcm_schar.html")
	  ("search" "fun_search.html")
	  ("second" "acc_firstcm_s_inthcm_tenth.html")
	  ("sequence" "syscla_sequence.html")
	  ("serious-condition" "contyp_serious-condition.html")
	  ("set" "fun_set.html")
	  ("set-difference" "fun_set-diffe_t-difference.html")
	  ("set-dispatch-macro-character" "fun_set-dispa_ro-character.html")
	  ("set-exclusive-or" "fun_set-exclu_exclusive-or.html")
	  ("set-macro-character" "fun_set-macro_ro-character.html")
	  ("set-pprint-dispatch" "fun_set-pprint-dispatch.html")
	  ("set-syntax-from-char" "fun_set-syntax-from-char.html")
	  ("setf" "mac_setfcm_psetf.html")
	  ("setf" "stagenfun_doc_umentationcp.html")
	  ("setq" "spefor_setq.html")
	  ("seventh" "acc_firstcm_s_inthcm_tenth.html")
	  ("shadow" "fun_shadow.html")
	  ("shadowing-import" "fun_shadowing-import.html")
	  ("shared-initialize" "stagenfun_sha_d-initialize.html")
	  ("shiftf" "mac_shiftf.html")
	  ("short-float" "typ_short-flo_m_long-float.html")
	  ("short-float-epsilon" "convar_short-_tive-epsilon.html")
	  ("short-float-negative-epsilon" "convar_short-_tive-epsilon.html")
	  ("short-site-name" "fun_short-sit_ng-site-name.html")
	  ("signal" "fun_signal.html")
	  ("signed-byte" "typ_signed-byte.html")
	  ("signum" "fun_signum.html")
	  ("simple-array" "typ_simple-array.html")
	  ("simple-base-string" "typ_simple-base-string.html")
	  ("simple-bit-vector" "typ_simple-bit-vector.html")
	  ("simple-bit-vector-p" "fun_simple-bit-vector-p.html")
	  ("simple-condition" "contyp_simple-condition.html")
	  ("simple-condition-format-arguments" "fun_simple-co_at-arguments.html")
	  ("simple-condition-format-control" "fun_simple-co_at-arguments.html")
	  ("simple-error" "contyp_simple-error.html")
	  ("simple-string" "typ_simple-string.html")
	  ("simple-string-p" "fun_simple-string-p.html")
	  ("simple-type-error" "contyp_simple-type-error.html")
	  ("simple-vector" "typ_simple-vector.html")
	  ("simple-vector-p" "fun_simple-vector-p.html")
	  ("simple-warning" "contyp_simple-warning.html")
	  ("sin" "fun_sincm_coscm_tan.html")
	  ("single-float" "typ_short-flo_m_long-float.html")
	  ("single-float-epsilon" "convar_short-_tive-epsilon.html")
	  ("single-float-negative-epsilon" "convar_short-_tive-epsilon.html")
	  ("sinh" "fun_sinhcm_co_coshcm_atanh.html")
	  ("sixth" "acc_firstcm_s_inthcm_tenth.html")
	  ("sleep" "fun_sleep.html")
	  ("slot-boundp" "fun_slot-boundp.html")
	  ("slot-exists-p" "fun_slot-exists-p.html")
	  ("slot-makunbound" "fun_slot-makunbound.html")
	  ("slot-missing" "stagenfun_slot-missing.html")
	  ("slot-unbound" "stagenfun_slot-unbound.html")
	  ("slot-value" "fun_slot-value.html")
	  ("software-type" "fun_software-_ware-version.html")
	  ("software-version" "fun_software-_ware-version.html")
	  ("some" "fun_everycm_s_erycm_notany.html")
	  ("sort" "fun_sortcm_stable-sort.html")
	  ("space" "dec_optimize.html")
	  ("special" "dec_special.html")
	  ("special-operator-p" "fun_special-operator-p.html")
	  ("speed" "dec_optimize.html")
	  ("sqrt" "fun_sqrtcm_isqrt.html")
	  ("stable-sort" "fun_sortcm_stable-sort.html")
	  ("standard" "sec_7-6-6-2.html")
	  ("standard-char" "typ_standard-char.html")
	  ("standard-char-p" "fun_standard-char-p.html")
	  ("standard-class" "syscla_standard-class.html")
	  ("standard-generic-function" "syscla_standa_ric-function.html")
	  ("standard-method" "syscla_standard-method.html")
	  ("standard-object" "cla_standard-object.html")
	  ("step" "mac_step.html")
	  ("storage-condition" "contyp_storage-condition.html")
	  ("store-value" "fun_abortcm_c_cm_use-value.html")
	  ("store-value" "res_store-value.html")
	  ("stream" "syscla_stream.html")
	  ("stream-element-type" "fun_stream-element-type.html")
	  ("stream-error" "contyp_stream-error.html")
	  ("stream-error-stream" "fun_stream-error-stream.html")
	  ("stream-external-format" "fun_stream-external-format.html")
	  ("streamp" "fun_streamp.html")
	  ("string" "fun_string.html")
	  ("string" "syscla_string.html")
	  ("string-capitalize" "fun_string-up_g-capitalize.html")
	  ("string-downcase" "fun_string-up_g-capitalize.html")
	  ("string-equal" "fun_stringeqc_ng-not-lessp.html")
	  ("string-greaterp" "fun_stringeqc_ng-not-lessp.html")
	  ("string-left-trim" "fun_string-tr_g-right-trim.html")
	  ("string-lessp" "fun_stringeqc_ng-not-lessp.html")
	  ("string-not-equal" "fun_stringeqc_ng-not-lessp.html")
	  ("string-not-greaterp" "fun_stringeqc_ng-not-lessp.html")
	  ("string-not-lessp" "fun_stringeqc_ng-not-lessp.html")
	  ("string-right-trim" "fun_string-tr_g-right-trim.html")
	  ("string-stream" "syscla_string-stream.html")
	  ("string-trim" "fun_string-tr_g-right-trim.html")
	  ("string-upcase" "fun_string-up_g-capitalize.html")
	  ("string/=" "fun_stringeqc_ng-not-lessp.html")
	  ("string<" "fun_stringeqc_ng-not-lessp.html")
	  ("string<=" "fun_stringeqc_ng-not-lessp.html")
	  ("string=" "fun_stringeqc_ng-not-lessp.html")
	  ("string>" "fun_stringeqc_ng-not-lessp.html")
	  ("string>=" "fun_stringeqc_ng-not-lessp.html")
	  ("stringp" "fun_stringp.html")
	  ("structure" "stagenfun_doc_umentationcp.html")
	  ("structure-class" "syscla_structure-class.html")
	  ("structure-object" "cla_structure-object.html")
	  ("style-warning" "contyp_style-warning.html")
	  ("sublis" "fun_subliscm_nsublis.html")
	  ("subseq" "acc_subseq.html")
	  ("subsetp" "fun_subsetp.html")
	  ("subst" "fun_substcm_s_subst-if-not.html")
	  ("subst-if" "fun_substcm_s_subst-if-not.html")
	  ("subst-if-not" "fun_substcm_s_subst-if-not.html")
	  ("substitute" "fun_substitut_itute-if-not.html")
	  ("substitute-if" "fun_substitut_itute-if-not.html")
	  ("substitute-if-not" "fun_substitut_itute-if-not.html")
	  ("subtypep" "fun_subtypep.html")
	  ("svref" "acc_svref.html")
	  ("sxhash" "fun_sxhash.html")
	  ("symbol" "syscla_symbol.html")
	  ("symbol-function" "acc_symbol-function.html")
	  ("symbol-macrolet" "speope_symbol-macrolet.html")
	  ("symbol-name" "fun_symbol-name.html")
	  ("symbol-package" "fun_symbol-package.html")
	  ("symbol-plist" "acc_symbol-plist.html")
	  ("symbol-value" "acc_symbol-value.html")
	  ("symbolp" "fun_symbolp.html")
	  ("synonym-stream" "syscla_synonym-stream.html")
	  ("synonym-stream-symbol" "fun_synonym-stream-symbol.html")
	  ("t" "convar_t.html")
	  ("t" "syscla_t.html")
	  ("t" "stagenfun_doc_umentationcp.html")
	  ("tagbody" "speope_tagbody.html")
	  ("tailp" "fun_ldiffcm_tailp.html")
	  ("tan" "fun_sincm_coscm_tan.html")
	  ("tanh" "fun_sinhcm_co_coshcm_atanh.html")
	  ("tenth" "acc_firstcm_s_inthcm_tenth.html")
	  ("terpri" "fun_terpricm_fresh-line.html")
	  ("the" "speope_the.html")
	  ("third" "acc_firstcm_s_inthcm_tenth.html")
	  ("throw" "speope_throw.html")
	  ("time" "mac_time.html")
	  ("trace" "mac_tracecm_untrace.html")
	  ("translate-logical-pathname" "fun_translate_cal-pathname.html")
	  ("translate-pathname" "fun_translate-pathname.html")
	  ("tree-equal" "fun_tree-equal.html")
	  ("truename" "fun_truename.html")
	  ("truncate" "fun_floorcm_f_undcm_fround.html")
	  ("two-way-stream" "syscla_two-way-stream.html")
	  ("two-way-stream-input-stream" "fun_two-way-s_utput-stream.html")
	  ("two-way-stream-output-stream" "fun_two-way-s_utput-stream.html")
	  ("type" "dec_type.html")
	  ("type" "stagenfun_doc_umentationcp.html")
	  ("type-error" "contyp_type-error.html")
	  ("type-error-datum" "fun_type-erro_xpected-type.html")
	  ("type-error-expected-type" "fun_type-erro_xpected-type.html")
	  ("type-of" "fun_type-of.html")
	  ("typecase" "mac_typecasec_cm_etypecase.html")
	  ("typep" "fun_typep.html")
	  ("unbound-slot" "contyp_unbound-slot.html")
	  ("unbound-slot-instance" "fun_unbound-slot-instance.html")
	  ("unbound-variable" "contyp_unbound-variable.html")
	  ("undefined-function" "contyp_undefined-function.html")
	  ("unexport" "fun_unexport.html")
	  ("unintern" "fun_unintern.html")
	  ("union" "fun_unioncm_nunion.html")
	  ("unless" "mac_whencm_unless.html")
	  ("unread-char" "fun_unread-char.html")
	  ("unsigned-byte" "typ_unsigned-byte.html")
	  ("untrace" "mac_tracecm_untrace.html")
	  ("unuse-package" "fun_unuse-package.html")
	  ("unwind-protect" "speope_unwind-protect.html")
	  ("update-instance-for-different-class" "stagenfun_upd_ferent-class.html")
	  ("update-instance-for-redefined-class" "stagenfun_upd_efined-class.html")
	  ("upgraded-array-element-type" "fun_upgraded-_element-type.html")
	  ("upgraded-complex-part-type" "fun_upgraded-_ex-part-type.html")
	  ("upper-case-p" "fun_upper-cas__both-case-p.html")
	  ("use-package" "fun_use-package.html")
	  ("use-value" "fun_abortcm_c_cm_use-value.html")
	  ("use-value" "res_use-value.html")
	  ("user-homedir-pathname" "fun_user-homedir-pathname.html")
	  ("values" "acc_values.html")
	  ("values" "typspe_values.html")
	  ("values-list" "fun_values-list.html")
	  ("variable" "stagenfun_doc_umentationcp.html")
	  ("vector" "fun_vector.html")
	  ("vector" "syscla_vector.html")
	  ("vector-pop" "fun_vector-pop.html")
	  ("vector-push" "fun_vector-pu_-push-extend.html")
	  ("vector-push-extend" "fun_vector-pu_-push-extend.html")
	  ("vectorp" "fun_vectorp.html")
	  ("warn" "fun_warn.html")
	  ("warning" "contyp_warning.html")
	  ("when" "mac_whencm_unless.html")
	  ("wild-pathname-p" "fun_wild-pathname-p.html")
	  ("with-accessors" "mac_with-accessors.html")
	  ("with-compilation-unit" "mac_with-compilation-unit.html")
	  ("with-condition-restarts" "mac_with-cond_ion-restarts.html")
	  ("with-hash-table-iterator" "mac_with-hash_ble-iterator.html")
	  ("with-input-from-string" "mac_with-input-from-string.html")
	  ("with-open-file" "mac_with-open-file.html")
	  ("with-open-stream" "mac_with-open-stream.html")
	  ("with-output-to-string" "mac_with-output-to-string.html")
	  ("with-package-iterator" "mac_with-package-iterator.html")
	  ("with-simple-restart" "mac_with-simple-restart.html")
	  ("with-slots" "mac_with-slots.html")
	  ("with-standard-io-syntax" "mac_with-stan_rd-io-syntax.html")
	  ("write" "fun_writecm_p_rintcm_princ.html")
	  ("write-byte" "fun_write-byte.html")
	  ("write-char" "fun_write-char.html")
	  ("write-line" "fun_write-str_m_write-line.html")
	  ("write-sequence" "fun_write-sequence.html")
	  ("write-string" "fun_write-str_m_write-line.html")
	  ("write-to-string" "fun_write-to-_nc-to-string.html")
	  ("y-or-n-p" "fun_y-or-n-pcm_yes-or-no-p.html")
	  ("yes-or-no-p" "fun_y-or-n-pcm_yes-or-no-p.html")
	  ("zerop" "fun_zerop.html")))

(provide 'hyperspec)

;;; hyperspec.el ends here
