/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: usredtecc.c
 * User interface technology editor: interactive technology library editing
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "egraphics.h"
#include "efunction.h"
#include "usr.h"
#include "drc.h"
#include "tecgen.h"
#include "tecart.h"
#include "usredtec.h"
#include "usrdiacom.h"

#define MAXNAMELEN 25		/* max chars in a new name */

INTBIG us_teceddrclayers = 0;
char **us_teceddrclayernames = 0;

/* the known technology variables */
TECHVAR us_knownvars[] =
{
	{"DRC_ecad_deck",               NOTECHVAR, 0, 0, 0.0, "", VSTRING|VISARRAY,
		"Dracula design-rule deck"},
	{"IO_cif_polypoints",           NOTECHVAR, 0, 0, 0.0, "", VINTEGER,
		"Maximum points in a CIF polygon"},
	{"IO_cif_resolution",           NOTECHVAR, 0, 0, 0.0, "", VINTEGER,
		"Minimum resolution of CIF coordinates"},
	{"IO_gds_polypoints",           NOTECHVAR, 0, 0, 0.0, "", VINTEGER,
		"Maximum points in a GDS-II polygon"},
	{"SIM_spice_min_resistance",    NOTECHVAR, 0, 0, 0.0, "", VFLOAT,
		"Minimum resistance of SPICE elements"},
	{"SIM_spice_min_capacitance",   NOTECHVAR, 0, 0, 0.0, "", VFLOAT,
		"Minimum capacitance of SPICE elements"},
	{"SIM_spice_mask_scale",        NOTECHVAR, 0, 0, 0.0, "", VFLOAT,
		"Scaling factor for SPICE decks"},
	{"SIM_spice_header_level1",     NOTECHVAR, 0, 0, 0.0, "", VSTRING|VISARRAY,
		"Level 1 header for SPICE decks"},
	{"SIM_spice_header_level2",     NOTECHVAR, 0, 0, 0.0, "", VSTRING|VISARRAY,
		"Level 2 header for SPICE decks"},
	{"SIM_spice_header_level3",     NOTECHVAR, 0, 0, 0.0, "", VSTRING|VISARRAY,
		"Level 3 header for SPICE decks"},
	{"SIM_spice_model_file",        NOTECHVAR, 0, 0, 0.0, "", VSTRING,
		"Disk file with SPICE header cards"},
	{"SIM_spice_trailer_file",      NOTECHVAR, 0, 0, 0.0, "", VSTRING,
		"Disk file with SPICE trailer cards"},
	{NULL, NULL,0, 0, 0.0, NULL, 0, NULL}  /* 0 */
};

/* these must correspond to the layer functions in "efunction.h" */
LIST us_teclayer_functions[] =
{
	{"unknown",           "LFUNKNOWN",     LFUNKNOWN},
	{"metal-1",           "LFMETAL1",      LFMETAL1},
	{"metal-2",           "LFMETAL2",      LFMETAL2},
	{"metal-3",           "LFMETAL3",      LFMETAL3},
	{"metal-4",           "LFMETAL4",      LFMETAL4},
	{"metal-5",           "LFMETAL5",      LFMETAL5},
	{"metal-6",           "LFMETAL6",      LFMETAL6},
	{"metal-7",           "LFMETAL7",      LFMETAL7},
	{"metal-8",           "LFMETAL8",      LFMETAL8},
	{"metal-9",           "LFMETAL9",      LFMETAL9},
	{"metal-10",          "LFMETAL10",     LFMETAL10},
	{"metal-11",          "LFMETAL11",     LFMETAL11},
	{"metal-12",          "LFMETAL12",     LFMETAL12},
	{"poly-1",            "LFPOLY1",       LFPOLY1},
	{"poly-2",            "LFPOLY2",       LFPOLY2},
	{"poly-3",            "LFPOLY3",       LFPOLY3},
	{"gate",              "LFGATE",        LFGATE},
	{"diffusion",         "LFDIFF",        LFDIFF},
	{"implant",           "LFIMPLANT",     LFIMPLANT},
	{"contact-1",         "LFCONTACT1",    LFCONTACT1},
	{"contact-2",         "LFCONTACT2",    LFCONTACT2},
	{"contact-3",         "LFCONTACT3",    LFCONTACT3},
	{"contact-4",         "LFCONTACT4",    LFCONTACT4},
	{"contact-5",         "LFCONTACT5",    LFCONTACT5},
	{"contact-6",         "LFCONTACT6",    LFCONTACT6},
	{"plug",              "LFPLUG",        LFPLUG},
	{"overglass",         "LFOVERGLASS",   LFOVERGLASS},
	{"resistor",          "LFRESISTOR",    LFRESISTOR},
	{"capacitor",         "LFCAP",         LFCAP},
	{"transistor",        "LFTRANSISTOR",  LFTRANSISTOR},
	{"emitter",           "LFEMITTER",     LFEMITTER},
	{"base",              "LFBASE",        LFBASE},
	{"collector",         "LFCOLLECTOR",   LFCOLLECTOR},
	{"substrate",         "LFSUBSTRATE",   LFSUBSTRATE},
	{"well",              "LFWELL",        LFWELL},
	{"guard",             "LFGUARD",       LFGUARD},
	{"isolation",         "LFISOLATION",   LFISOLATION},
	{"bus",               "LFBUS",         LFBUS},
	{"art",               "LFART",         LFART},
	{"control",           "LFCONTROL",     LFCONTROL},

	{"p-type",            "LFPTYPE",       LFPTYPE},
	{"n-type",            "LFNTYPE",       LFNTYPE},
	{"depletion",         "LFDEPLETION",   LFDEPLETION},
	{"enhancement",       "LFENHANCEMENT", LFENHANCEMENT},
	{"light",             "LFLIGHT",       LFLIGHT},
	{"heavy",             "LFHEAVY",       LFHEAVY},
	{"pseudo",            "LFPSEUDO",      LFPSEUDO},
	{"nonelectrical",     "LFNONELEC",     LFNONELEC},
	{"connects-metal",    "LFCONMETAL",    LFCONMETAL},
	{"connects-poly",     "LFCONPOLY",     LFCONPOLY},
	{"connects-diff",     "LFCONDIFF",     LFCONDIFF},
	{"inside-transistor", "LFINTRANS",     LFINTRANS},
	{NULL, NULL, 0}
};

/* these must correspond to the layer functions in "efunction.h" */
LIST us_tecarc_functions[] =
{
	{"unknown",             "APUNKNOWN",  APUNKNOWN},
	{"metal-1",             "APMETAL1",   APMETAL1},
	{"metal-2",             "APMETAL2",   APMETAL2},
	{"metal-3",             "APMETAL3",   APMETAL3},
	{"metal-4",             "APMETAL4",   APMETAL4},
	{"metal-5",             "APMETAL5",   APMETAL5},
	{"metal-6",             "APMETAL6",   APMETAL6},
	{"metal-7",             "APMETAL7",   APMETAL7},
	{"metal-8",             "APMETAL8",   APMETAL8},
	{"metal-9",             "APMETAL9",   APMETAL9},
	{"metal-10",            "APMETAL10",  APMETAL10},
	{"metal-11",            "APMETAL11",  APMETAL11},
	{"metal-12",            "APMETAL12",  APMETAL12},
	{"polysilicon-1",       "APPOLY1",    APPOLY1},
	{"polysilicon-2",       "APPOLY2",    APPOLY2},
	{"polysilicon-3",       "APPOLY3",    APPOLY3},
	{"diffusion",           "APDIFF",     APDIFF},
	{"p-Diffusion",         "APDIFFP",    APDIFFP},
	{"n-Diffusion",         "APDIFFN",    APDIFFN},
	{"substrate-Diffusion", "APDIFFS",    APDIFFS},
	{"well-Diffusion",      "APDIFFW",    APDIFFW},
	{"bus",                 "APBUS",      APBUS},
	{"unrouted",            "APUNROUTED", APUNROUTED},
	{"nonelectrical",       "APNONELEC",  APNONELEC},
	{NULL, NULL, 0}
};

static GRAPHICS us_edtechigh = {LAYERH, HIGHLIT, SOLIDC, SOLIDC,
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

/* prototypes for local routines */
static void us_teceditdrc(void);
static void us_teceditcolormap(void);
static void us_teceditcreat(INTBIG, char*[]);
static void us_teceditidentify(BOOLEAN);
static void us_teceditinquire(void);
static void us_teceditmodobject(INTBIG, char*[]);
static void us_tecedlayer3dheight(NODEINST*, INTBIG, char*[]);
static void us_tecedlayer3dthick(NODEINST*, INTBIG, char*[]);
static void us_tecedlayercolor(NODEINST*, INTBIG, char*[]);
static void us_tecedlayerdrcminwid(NODEINST*, INTBIG, char*[]);
static void us_tecedlayerstyle(NODEINST*, INTBIG, char*[]);
static void us_tecedlayercif(NODEINST*, INTBIG, char*[]);
static void us_tecedlayerdxf(NODEINST*, INTBIG, char*[]);
static void us_tecedlayergds(NODEINST*, INTBIG, char*[]);
static void us_tecedlayerspires(NODEINST*, INTBIG, char*[]);
static void us_tecedlayerspicap(NODEINST*, INTBIG, char*[]);
static void us_tecedlayerspiecap(NODEINST*, INTBIG, char*[]);
static void us_tecedlayerfunction(NODEINST*, INTBIG, char*[]);
static void us_tecedlayerletters(NODEINST*, INTBIG, char*[]);
static void us_tecedlayerpattern(NODEINST*);
static void us_tecedlayertype(NODEINST*, INTBIG, char*[]);
static void us_tecedmodport(NODEINST*, INTBIG, char*[]);
static void us_tecedarcfunction(NODEINST*, INTBIG, char*[]);
static void us_tecedarcfixang(NODEINST*, INTBIG, char*[]);
static void us_tecedarcwipes(NODEINST*, INTBIG, char*[]);
static void us_tecedarcnoextend(NODEINST*, INTBIG, char*[]);
static void us_tecedarcinc(NODEINST*, INTBIG, char*[]);
static void us_tecednodefunction(NODEINST*, INTBIG, char*[]);
static void us_tecednodeserpentine(NODEINST*, INTBIG, char*[]);
static void us_tecednodesquare(NODEINST*, INTBIG, char*[]);
static void us_tecednodewipes(NODEINST*, INTBIG, char*[]);
static void us_tecednodelockable(NODEINST*, INTBIG, char*[]);
static void us_tecedinfolambda(NODEINST*, INTBIG, char*[]);
static void us_tecedinfodescript(NODEINST*, INTBIG, char*[]);
static void us_tecedsetnode(NODEINST*, char*);
static NODEPROTO *us_tecedenterfacet(char*);
static void us_tecedredolayergraphics(NODEPROTO*);
static void us_tecedloadlibmap(LIBRARY*);
static INTBIG us_teceditparsefun(char*);
static BOOLEAN us_tecedgetdrc(char *str, BOOLEAN *connected, BOOLEAN *wide, BOOLEAN *multi,
			INTBIG *widrule, BOOLEAN *edge, INTBIG *amt, INTBIG *layer1, INTBIG *layer2,
			char **rule, INTBIG maxlayers, char **layernames);
static void us_tecedrenamesequence(char *varname, char *oldname, char *newname);

/*
 * the entry routine for all technology editing
 */
void us_tecedentry(INTBIG count, char *par[])
{
	REGISTER TECHNOLOGY *tech;
	REGISTER LIBRARY *lib;
	LIBRARY **dependentlibs;
	NODEPROTO **sequence;
	REGISTER char *pp, **dependentlist;
	char *facetname, *newpar[2];
	UINTSML stip[8];
	REGISTER INTBIG i, l;
	REGISTER INTBIG dependentlibcount;
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	REGISTER void *infstr;

	if (count == 0)
	{
		ttyputusage("technology edit OPTION");
		return;
	}

	l = strlen(pp = par[0]);
	if (namesamen(pp, "library-to-tech-and-c", l) == 0 && l >= 17)
	{
		if (count <= 1) pp = 0; else
			pp = par[1];
		us_tecfromlibinit(el_curlib, pp, TRUE);
		return;
	}
	if (namesamen(pp, "library-to-tech", l) == 0)
	{
		if (count <= 1) pp = 0; else
			pp = par[1];
		us_tecfromlibinit(el_curlib, pp, FALSE);
		return;
	}
	if (namesamen(pp, "tech-to-library", l) == 0)
	{
		if (count == 1) tech = el_curtech; else
		{
			tech = gettechnology(par[1]);
			if (tech == NOTECHNOLOGY)
			{
				us_abortcommand(_("Technology '%s' unknown"), par[1]);
				return;
			}
		}
		if ((tech->userbits&NONSTANDARD) != 0)
		{
			us_abortcommand(_("Cannot convert technology '%s', it is nonstandard"), tech->techname);
			return;
		}

		/* see if there is already such a library */
		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			if (namesame(lib->libname, tech->techname) == 0) break;
		if (lib != NOLIBRARY)
			ttyputmsg(_("Already a library called %s, using that"), lib->libname); else
				lib = us_tecedmakelibfromtech(tech);
		if (lib != NOLIBRARY)
		{
			newpar[0] = "use";
			newpar[1] = lib->libname;
			us_library(2, newpar);
			us_tecedloadlibmap(lib);
		}
		return;
	}
	if (namesamen(pp, "inquire-layer", l) == 0 && l >= 2)
	{
		us_teceditinquire();
		return;
	}
	if (namesamen(pp, "place-layer", l) == 0)
	{
		if (count < 2)
		{
			ttyputusage("technology edit place-layer SHAPE");
			return;
		}

		us_teceditcreat(count-1, &par[1]);
		return;
	}
	if (namesamen(pp, "change", l) == 0)
	{
		us_teceditmodobject(count-1, &par[1]);
		return;
	}
	if (namesamen(pp, "edit-node", l) == 0 && l >= 6)
	{
		if (count < 2)
		{
			ttyputusage("technology edit edit-node NODENAME");
			return;
		}
		infstr = initinfstr();
		addstringtoinfstr(infstr, "node-");
		addstringtoinfstr(infstr, par[1]);
		(void)allocstring(&facetname, returninfstr(infstr), el_tempcluster);

		/* first make sure all fields exist */
		np = getnodeproto(facetname);
		if (np != NONODEPROTO)
		{
			us_tecedmakenode(np, NPUNKNOWN, FALSE, FALSE, FALSE, FALSE);
			(*el_curconstraint->solve)(np);
		}

		np = us_tecedenterfacet(facetname);
		efree(facetname);
		if (np == NONODEPROTO) return;
		us_tecedmakenode(np, NPUNKNOWN, FALSE, FALSE, FALSE, FALSE);
		(*el_curconstraint->solve)(np);
		(void)us_tecedenterfacet(describenodeproto(np));
		return;
	}
	if (namesamen(pp, "edit-arc", l) == 0 && l >= 6)
	{
		if (count < 2)
		{
			ttyputusage("technology edit edit-arc ARCNAME");
			return;
		}
		infstr = initinfstr();
		addstringtoinfstr(infstr, "arc-");
		addstringtoinfstr(infstr, par[1]);
		(void)allocstring(&facetname, returninfstr(infstr), el_tempcluster);
		np = us_tecedenterfacet(facetname);
		efree(facetname);
		if (np == NONODEPROTO) return;
		us_tecedmakearc(np, APUNKNOWN, 1, 1, 0, 90);
		(*el_curconstraint->solve)(np);
		(void)us_tecedenterfacet(describenodeproto(np));
		return;
	}
	if (namesamen(pp, "edit-layer", l) == 0 && l >= 6)
	{
		if (count < 2)
		{
			ttyputusage("technology edit edit-layer LAYERNAME");
			return;
		}
		infstr = initinfstr();
		addstringtoinfstr(infstr, "layer-");
		addstringtoinfstr(infstr, par[1]);
		(void)allocstring(&facetname, returninfstr(infstr), el_tempcluster);

		/* first make sure all fields exist */
		for(i=0; i<8; i++) stip[i] = 0;
		np = getnodeproto(facetname);
		if (np != NONODEPROTO)
		{
			us_tecedmakelayer(np, COLORT1, stip, SOLIDC, "XX", LFUNKNOWN, "x", "",
				-1, 0.0, 0.0, 0.0, 0, 0);
			(*el_curconstraint->solve)(np);
		}

		np = us_tecedenterfacet(facetname);
		efree(facetname);
		if (np == NONODEPROTO) return;
		us_tecedmakelayer(np, COLORT1, stip, SOLIDC, "XX", LFUNKNOWN, "x", "",
			-1, 0.0, 0.0, 0.0, 0, 0);
		(*el_curconstraint->solve)(np);
		(void)us_tecedenterfacet(describenodeproto(np));
		return;
	}
	if (namesamen(pp, "edit-subsequent", l) == 0 && l >= 6)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		if (namesamen(np->cell->cellname, "node-", 5) == 0)
		{
			dependentlibcount = us_teceditgetdependents(el_curlib, &dependentlibs);
			i = us_teceditfindsequence(dependentlibs, dependentlibcount, "node-",
				"EDTEC_nodesequence", &sequence);
			if (i == 0) return;
			for(l=0; l<i; l++) if (sequence[l] == np)
			{
				if (l == i-1) np = sequence[0]; else np = sequence[l+1];
				(void)us_tecedenterfacet(describenodeproto(np));
				break;
			}
			efree((char *)sequence);
			return;
		}
		if (namesamen(np->cell->cellname, "arc-", 4) == 0)
		{
			dependentlibcount = us_teceditgetdependents(el_curlib, &dependentlibs);
			i = us_teceditfindsequence(dependentlibs, dependentlibcount, "arc-",
				"EDTEC_arcsequence", &sequence);
			if (i == 0) return;
			for(l=0; l<i; l++) if (sequence[l] == np)
			{
				if (l == i-1) np = sequence[0]; else np = sequence[l+1];
				(void)us_tecedenterfacet(describenodeproto(np));
				break;
			}
			efree((char *)sequence);
			return;
		}
		if (namesamen(np->cell->cellname, "layer-", 6) == 0)
		{
			dependentlibcount = us_teceditgetdependents(el_curlib, &dependentlibs);
			i = us_teceditfindsequence(dependentlibs, dependentlibcount, "layer-",
				"EDTEC_layersequence", &sequence);
			if (i == 0) return;
			for(l=0; l<i; l++) if (sequence[l] == np)
			{
				if (l == i-1) np = sequence[0]; else np = sequence[l+1];
				(void)us_tecedenterfacet(describenodeproto(np));
				break;
			}
			efree((char *)sequence);
			return;
		}
		ttyputerr(_("Must be editing a layer, node, or arc to advance to the next"));
		return;
	}
	if (namesamen(pp, "edit-colors", l) == 0 && l >= 6)
	{
		us_teceditcolormap();
		return;
	}
	if (namesamen(pp, "edit-design-rules", l) == 0 && l >= 6)
	{
		us_teceditdrc();
		return;
	}
	if (namesamen(pp, "edit-misc-information", l) == 0 && l >= 6)
	{
		/* first make sure all fields exist */
		np = getnodeproto("factors");
		if (np != NONODEPROTO)
		{
			us_tecedmakeinfo(np, 2000, el_curlib->libname);
			(*el_curconstraint->solve)(np);
		}

		/* now edit the facet */
		np = us_tecedenterfacet("factors");
		if (np == NONODEPROTO) return;
		us_tecedmakeinfo(np, 2000, el_curlib->libname);
		(*el_curconstraint->solve)(np);
		(void)us_tecedenterfacet(describenodeproto(np));
		return;
	}
	if (namesamen(pp, "identify-layers", l) == 0 && l >= 10)
	{
		us_teceditidentify(FALSE);
		return;
	}
	if (namesamen(pp, "identify-ports", l) == 0 && l >= 10)
	{
		us_teceditidentify(TRUE);
		return;
	}
	if (namesamen(pp, "dependent-libraries", l) == 0 && l >= 2)
	{
		if (count < 2)
		{
			/* display dependent library names */
			var = getval((INTBIG)el_curlib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_dependent_libraries");
			if (var == NOVARIABLE) ttyputmsg(_("There are no dependent libraries")); else
			{
				i = getlength(var);
				ttyputmsg(_("%ld dependent %s:"), i, makeplural("library", i));
				for(l=0; l<i; l++)
				{
					pp = ((char **)var->addr)[l];
					lib = getlibrary(pp);
					ttyputmsg("    %s%s", pp, (lib == NOLIBRARY ? _(" (not read in)") : ""));
				}
			}
			return;
		}

		/* clear list if just "-" is given */
		if (count == 2 && strcmp(par[1], "-") == 0)
		{
			var = getval((INTBIG)el_curlib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_dependent_libraries");
			if (var != NOVARIABLE)
				(void)delval((INTBIG)el_curlib, VLIBRARY, "EDTEC_dependent_libraries");
			return;
		}

		/* create a list */
		dependentlist = (char **)emalloc((count-1) * (sizeof (char *)), el_tempcluster);
		if (dependentlist == 0) return;
		for(i=1; i<count; i++) dependentlist[i-1] = par[i];
		(void)setval((INTBIG)el_curlib, VLIBRARY, "EDTEC_dependent_libraries", (INTBIG)dependentlist,
			VSTRING|VISARRAY|((count-1)<<VLENGTHSH));
		efree((char *)dependentlist);
		return;
	}
	ttyputbadusage("technology edit");
}

/*
 * Routine for editing the DRC tables.
 */
void us_teceditdrc(void)
{
	REGISTER INTBIG i, changed, nodecount;
	NODEPROTO **nodesequence;
	LIBRARY *liblist[1];
	REGISTER VARIABLE *var;
	REGISTER DRCRULES *rules;

	/* get the current list of layer and node names */
	us_tecedgetlayernamelist();
	liblist[0] = el_curlib;
	nodecount = us_teceditfindsequence(liblist, 1, "node-", "EDTEC_nodesequence", &nodesequence);

	/* create a RULES structure */
	rules = dr_allocaterules(us_teceddrclayers, nodecount, "EDITED TECHNOLOGY");
	if (rules == NODRCRULES) return;
	for(i=0; i<us_teceddrclayers; i++)
		(void)allocstring(&rules->layernames[i], us_teceddrclayernames[i], el_tempcluster);
	for(i=0; i<nodecount; i++)
		(void)allocstring(&rules->nodenames[i], &nodesequence[i]->cell->cellname[5], el_tempcluster);
	if (nodecount > 0) efree((char *)nodesequence);

	/* get the text-list of design rules and convert them into arrays */
	var = getval((INTBIG)el_curlib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_DRC");
	us_teceditgetdrcarrays(var, rules);

	/* edit the design-rule arrays */
	changed = dr_rulesdlog(NOTECHNOLOGY, rules);

	/* if changes were made, convert the arrays back into a text-list */
	if (changed != 0)
	{
		us_tecedloaddrcmessage(rules, el_curlib);
	}

	/* free the arrays */
	dr_freerules(rules);
}

/*
 * Routine to update tables to reflect that cell "oldname" is now called "newname".
 * If "newname" is not valid, any rule that refers to "oldname" is removed.
 * 
 */
void us_tecedrenamecell(char *oldname, char *newname)
{
	REGISTER VARIABLE *var;
	REGISTER INTBIG i, len;
	REGISTER BOOLEAN valid;
	INTBIG count;
	REGISTER char *origstr, *firstkeyword, *keyword;
	char *str, **strings;
	REGISTER void *infstr, *sa;

	/* if this is a layer, rename the layer sequence array */
	if (namesamen(oldname, "layer-", 6) == 0 && namesamen(newname, "layer-", 6) == 0)
	{
		us_tecedrenamesequence("EDTEC_layersequence", &oldname[6], &newname[6]);
	}

	/* if this is an arc, rename the arc sequence array */
	if (namesamen(oldname, "arc-", 4) == 0 && namesamen(newname, "arc-", 4) == 0)
	{
		us_tecedrenamesequence("EDTEC_arcsequence", &oldname[4], &newname[4]);
	}

	/* if this is a node, rename the node sequence array */
	if (namesamen(oldname, "node-", 5) == 0 && namesamen(newname, "node-", 5) == 0)
	{
		us_tecedrenamesequence("EDTEC_nodesequence", &oldname[5], &newname[5]);
	}

	/* see if there are design rules in the current library */
	var = getval((INTBIG)el_curlib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_DRC");
	if (var == NOVARIABLE) return;

	/* examine the rules and convert the name */
	len = getlength(var);
	sa = newstringarray(us_tool->cluster);
	for(i=0; i<len; i++)
	{
		/* parse the DRC rule */
		str = ((char **)var->addr)[i];
		origstr = str;
		firstkeyword = getkeyword(&str, " ");
		if (firstkeyword == NOSTRING) return;

		/* pass wide wire limitation through */
		if (*firstkeyword == 'l')
		{
			addtostringarray(sa, origstr);
			continue;
		}

		/* rename nodes in the minimum node size rule */
		if (*firstkeyword == 'n')
		{
			if (namesamen(oldname, "node-", 5) == 0 &&
				namesame(&oldname[5], &firstkeyword[1]) == 0)
			{
				/* substitute the new name */
				if (namesamen(newname, "node-", 5) == 0)
				{
					infstr = initinfstr();
					addstringtoinfstr(infstr, "n");
					addstringtoinfstr(infstr, &newname[5]);
					addstringtoinfstr(infstr, str);
					addtostringarray(sa, returninfstr(infstr));
				}
				continue;
			}
			addtostringarray(sa, origstr);
			continue;
		}

		/* rename layers in the minimum layer size rule */
		if (*firstkeyword == 's')
		{
			valid = TRUE;
			infstr = initinfstr();
			formatinfstr(infstr, "%s ", firstkeyword);
			keyword = getkeyword(&str, " ");
			if (keyword == NOSTRING) return;
			if (namesamen(oldname, "layer-", 6) == 0 &&
				namesame(&oldname[6], keyword) == 0)
			{
				if (namesamen(newname, "layer-", 6) != 0) valid = FALSE; else
					addstringtoinfstr(infstr, &newname[6]);
			} else
				addstringtoinfstr(infstr, keyword);
			addstringtoinfstr(infstr, str);
			str = returninfstr(infstr);
			if (valid) addtostringarray(sa, str);
			continue;
		}

		/* layer width rule: substitute layer names */
		infstr = initinfstr();
		formatinfstr(infstr, "%s ", firstkeyword);
		valid = TRUE;

		/* get the first layer name and convert it */
		keyword = getkeyword(&str, " ");
		if (keyword == NOSTRING) return;
		if (namesamen(oldname, "layer-", 6) == 0 &&
			namesame(&oldname[6], keyword) == 0)
		{
			/* substitute the new name */
			if (namesamen(newname, "layer-", 6) != 0) valid = FALSE; else
				addstringtoinfstr(infstr, &newname[6]);
		} else
			addstringtoinfstr(infstr, keyword);
		addtoinfstr(infstr, ' ');

		/* get the second layer name and convert it */
		keyword = getkeyword(&str, " ");
		if (keyword == NOSTRING) return;
		if (namesamen(oldname, "layer-", 6) == 0 &&
			namesame(&oldname[6], keyword) == 0)
		{
			/* substitute the new name */
			if (namesamen(newname, "layer-", 6) != 0) valid = FALSE; else
				addstringtoinfstr(infstr, &newname[6]);
		} else
			addstringtoinfstr(infstr, keyword);

		addstringtoinfstr(infstr, str);
		str = returninfstr(infstr);
		if (valid) addtostringarray(sa, str);
	}
	strings = getstringarray(sa, &count);
	setval((INTBIG)el_curlib, VLIBRARY, "EDTEC_DRC", (INTBIG)strings,
		VSTRING|VISARRAY|(count<<VLENGTHSH));
	killstringarray(sa);
}

/*
 * Routine to rename the layer/arc/node sequence arrays to account for a name change.
 * The sequence array is in variable "varname", and the item has changed from "oldname" to
 * "newname".
 */
void us_tecedrenamesequence(char *varname, char *oldname, char *newname)
{
	REGISTER VARIABLE *var;
	char **strings;
	REGISTER INTBIG i, len;
	INTBIG count;
	void *sa;

	var = getval((INTBIG)el_curlib, VLIBRARY, VSTRING|VISARRAY, varname);
	if (var == NOVARIABLE) return;

	strings = (char **)var->addr;
	len = getlength(var);
	sa = newstringarray(us_tool->cluster);
	for(i=0; i<len; i++)
	{
		if (namesame(strings[i], oldname) == 0)
			addtostringarray(sa, newname); else
				addtostringarray(sa, strings[i]);
	}
	strings = getstringarray(sa, &count);
	setval((INTBIG)el_curlib, VLIBRARY, varname, (INTBIG)strings,
		VSTRING|VISARRAY|(count<<VLENGTHSH));
	killstringarray(sa);
}

void us_tecedgetlayernamelist(void)
{
	REGISTER INTBIG i;
	REGISTER NODEPROTO *np;

	/* free any former layer name information */
	if (us_teceddrclayernames != 0)
	{
		for(i=0; i<us_teceddrclayers; i++) efree(us_teceddrclayernames[i]);
		efree((char *)us_teceddrclayernames);
		us_teceddrclayernames = 0;
	}

	/* count the number of layers */
	us_teceddrclayers = 0;
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (namesamen(np->cell->cellname, "layer-", 6) == 0) us_teceddrclayers++;
	if (us_teceddrclayers <= 0)
	{
		ttyputerr(_("Define layers before creating design rules"));
		return;
	}

	/* build and fill array of layers for DRC parsing */
	us_teceddrclayernames = (char **)emalloc(us_teceddrclayers * (sizeof (char *)), us_tool->cluster);
	if (us_teceddrclayernames == 0) return;
	for(i = 0, np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (namesamen(np->cell->cellname, "layer-", 6) == 0)
			(void)allocstring(&us_teceddrclayernames[i++], &np->cell->cellname[6], us_tool->cluster);
}

/*
 * Routine to create arrays describing the design rules in the variable "var" (which is
 * from "EDTEC_DRC" on a library).  The arrays are stored in "rules".
 */
void us_teceditgetdrcarrays(VARIABLE *var, DRCRULES *rules)
{
	REGISTER INTBIG i, l;
	INTBIG amt;
	BOOLEAN connected, wide, multi, edge;
	INTBIG widrule, layer1, layer2, j;
	REGISTER char *str, *pt;
	char *rule;

	/* get the design rules */
	if (var == NOVARIABLE) return;

	l = getlength(var);
	for(i=0; i<l; i++)
	{
		/* parse the DRC rule */
		str = ((char **)var->addr)[i];
		while (*str == ' ') str++;
		if (*str == 0) continue;

		/* special case for node minimum size rule */
		if (*str == 'n')
		{
			str++;
			for(pt = str; *pt != 0; pt++) if (*pt == ' ') break;
			if (*pt == 0)
			{
				ttyputmsg(_("Bad node size rule (line %ld): %s"), i+1, str);
				continue;
			}
			*pt = 0;
			for(j=0; j<rules->numnodes; j++)
				if (namesame(str, rules->nodenames[j]) == 0) break;
			*pt = ' ';
			if (j >= rules->numnodes)
			{
				ttyputmsg(_("Unknown node (line %ld): %s"), i+1, str);
				continue;
			}
			while (*pt == ' ') pt++;
			rules->minnodesize[j*2] = atofr(pt);
			while (*pt != 0 && *pt != ' ') pt++;
			while (*pt == ' ') pt++;
			rules->minnodesize[j*2+1] = atofr(pt);
			while (*pt != 0 && *pt != ' ') pt++;
			while (*pt == ' ') pt++;
			if (*pt != 0) reallocstring(&rules->minnodesizeR[j], pt, el_tempcluster);
			continue;
		}

		/* parse the layer rule */
		if (us_tecedgetdrc(str, &connected, &wide, &multi, &widrule, &edge,
			&amt, &layer1, &layer2, &rule, rules->numlayers, rules->layernames))
		{
			ttyputmsg(_("DRC line %ld is: %s"), i+1, str);
			continue;
		}

		/* set the layer spacing */
		if (widrule == 1)
		{
			rules->minwidth[layer1] = amt;
			if (*rule != 0)
				(void)reallocstring(&rules->minwidthR[layer1], rule, el_tempcluster);
		} else if (widrule == 2)
		{
			rules->widelimit = amt;
		} else
		{
			if (layer1 > layer2) { j = layer1;  layer1 = layer2;  layer2 = j; }
			j = (layer1+1) * (layer1/2) + (layer1&1) * ((layer1+1)/2);
			j = layer2 + rules->numlayers * layer1 - j;
			if (edge)
			{
				rules->edgelist[j] = amt;
				if (*rule != 0)
					(void)reallocstring(&rules->edgelistR[j], rule, el_tempcluster);
			} else if (wide)
			{
				if (connected)
				{
					rules->conlistW[j] = amt;
					if (*rule != 0)
						(void)reallocstring(&rules->conlistWR[j], rule, el_tempcluster);
				} else
				{
					rules->unconlistW[j] = amt;
					if (*rule != 0)
						(void)reallocstring(&rules->unconlistWR[j], rule, el_tempcluster);
				}
			} else if (multi)
			{
				if (connected)
				{
					rules->conlistM[j] = amt;
					if (*rule != 0)
						(void)reallocstring(&rules->conlistMR[j], rule, el_tempcluster);
				} else
				{
					rules->unconlistM[j] = amt;
					if (*rule != 0)
						(void)reallocstring(&rules->unconlistMR[j], rule, el_tempcluster);
				}
			} else
			{
				if (connected)
				{
					rules->conlist[j] = amt;
					if (*rule != 0)
						(void)reallocstring(&rules->conlistR[j], rule, el_tempcluster);
				} else
				{
					rules->unconlist[j] = amt;
					if (*rule != 0)
						(void)reallocstring(&rules->unconlistR[j], rule, el_tempcluster);
				}
			}
		}
	}
}

/*
 * routine to parse DRC line "str" and fill the factors "connected" (set nonzero
 * if rule is for connected layers), "amt" (rule distance), "layer1" and "layer2"
 * (the layers).  Presumes that there are "maxlayers" layer names in the
 * array "layernames".  Returns true on error.
 */
BOOLEAN us_tecedgetdrc(char *str, BOOLEAN *connected, BOOLEAN *wide, BOOLEAN *multi, INTBIG *widrule,
	BOOLEAN *edge, INTBIG *amt, INTBIG *layer1, INTBIG *layer2, char **rule, INTBIG maxlayers,
	char **layernames)
{
	REGISTER char *pt;
	REGISTER INTBIG save;

	*connected = *wide = *multi = *edge = FALSE;
	for( ; *str != 0; str++)
	{
		if (tolower(*str) == 'c')
		{
			*connected = TRUE;
			continue;
		}
		if (tolower(*str) == 'w')
		{
			*wide = TRUE;
			continue;
		}
		if (tolower(*str) == 'm')
		{
			*multi = TRUE;
			continue;
		}
		if (tolower(*str) == 'e')
		{
			*edge = TRUE;
			continue;
		}
		break;
	}
	*widrule = 0;
	if (tolower(*str) == 's')
	{
		*widrule = 1;
		str++;
	} else if (tolower(*str) == 'l')
	{
		*widrule = 2;
		str++;
	}

	/* get the distance */
	pt = str;
	while (*pt != 0 && *pt != ' ' && *pt != '\t') pt++;
	while (*pt == ' ' || *pt == '\t') pt++;
	*amt = atofr(str);

	/* get the first layer */
	if (*widrule != 2)
	{
		str = pt;
		if (*str == 0)
		{
			ttyputerr(_("Cannot find layer names on DRC line"));
			return(TRUE);
		}
		while (*pt != 0 && *pt != ' ' && *pt != '\t') pt++;
		if (*pt == 0)
		{
			ttyputerr(_("Cannot find layer name on DRC line"));
			return(TRUE);
		}
		save = *pt;
		*pt = 0;
		for(*layer1 = 0; *layer1 < maxlayers; (*layer1)++)
			if (namesame(str, layernames[*layer1]) == 0) break;
		*pt++ = (char)save;
		if (*layer1 >= maxlayers)
		{
			ttyputerr(_("First DRC layer name unknown"));
			return(TRUE);
		}
		while (*pt == ' ' || *pt == '\t') pt++;
	}

	/* get the second layer */
	if (*widrule == 0)
	{
		str = pt;
		while (*pt != 0 && *pt != ' ' && *pt != '\t') pt++;
		save = *pt;
		*pt = 0;
		for(*layer2 = 0; *layer2 < maxlayers; (*layer2)++)
			if (namesame(str, layernames[*layer2]) == 0) break;
		*pt = (char)save;
		if (*layer2 >= maxlayers)
		{
			ttyputerr(_("Second DRC layer name unknown"));
			return(TRUE);
		}
	}

	while (*pt == ' ' || *pt == '\t') pt++;
	*rule = pt;
	return(FALSE);
}

/*
 * Helper routine to examine the arrays describing the design rules and create
 * the variable "EDTEC_DRC" on library "lib".
 */
void us_tecedloaddrcmessage(DRCRULES *rules, LIBRARY *lib)
{
	REGISTER INTBIG drccount, drcindex, i, k, j;
	REGISTER char **drclist;
	REGISTER void *infstr;

	/* determine the number of lines in the text-version of the design rules */
	drccount = 0;
	for(i=0; i<rules->utsize; i++)
	{
		if (rules->conlist[i] >= 0) drccount++;
		if (rules->unconlist[i] >= 0) drccount++;
		if (rules->conlistW[i] >= 0) drccount++;
		if (rules->unconlistW[i] >= 0) drccount++;
		if (rules->conlistM[i] >= 0) drccount++;
		if (rules->unconlistM[i] >= 0) drccount++;
		if (rules->edgelist[i] >= 0) drccount++;
	}
	for(i=0; i<rules->numlayers; i++)
	{
		if (rules->minwidth[i] >= 0) drccount++;
	}
	for(i=0; i<rules->numnodes; i++)
	{
		if (rules->minnodesize[i*2] > 0 || rules->minnodesize[i*2+1] > 0) drccount++;
	}

	/* load the arrays */
	if (drccount != 0)
	{
		drccount++;
		drclist = (char **)emalloc((drccount * (sizeof (char *))), el_tempcluster);
		if (drclist == 0) return;
		drcindex = 0;

		/* write the width limit */
		infstr = initinfstr();
		formatinfstr(infstr, "l%s", frtoa(rules->widelimit));
		(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);

		/* write the minimum width for each layer */
		for(i=0; i<rules->numlayers; i++)
		{
			if (rules->minwidth[i] >= 0)
			{
				infstr = initinfstr();
				formatinfstr(infstr, "s%s %s %s", frtoa(rules->minwidth[i]),
					rules->layernames[i], rules->minwidthR[i]);
				(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);
			}
		}

		/* write the minimum size for each node */
		for(i=0; i<rules->numnodes; i++)
		{
			if (rules->minnodesize[i*2] <= 0 && rules->minnodesize[i*2+1] <= 0) continue;
			{
				infstr = initinfstr();
				formatinfstr(infstr, "n%s %s %s %s", rules->nodenames[i],
					frtoa(rules->minnodesize[i*2]), frtoa(rules->minnodesize[i*2+1]),
					rules->minnodesizeR[i]);
				(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);
			}
		}

		/* now do the distance rules */
		k = 0;
		for(i=0; i<rules->numlayers; i++) for(j=i; j<rules->numlayers; j++)
		{
			if (rules->conlist[k] >= 0)
			{
				infstr = initinfstr();
				formatinfstr(infstr, "c%s %s %s %s", frtoa(rules->conlist[k]),
					rules->layernames[i], rules->layernames[j],
						rules->conlistR[k]);
				(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);
			}
			if (rules->unconlist[k] >= 0)
			{
				infstr = initinfstr();
				formatinfstr(infstr, "%s %s %s %s", frtoa(rules->unconlist[k]),
					rules->layernames[i], rules->layernames[j],
						rules->unconlistR[k]);
				(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);
			}
			if (rules->conlistW[k] >= 0)
			{
				infstr = initinfstr();
				formatinfstr(infstr, "cw%s %s %s %s", frtoa(rules->conlistW[k]),
					rules->layernames[i], rules->layernames[j],
						rules->conlistWR[k]);
				(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);
			}
			if (rules->unconlistW[k] >= 0)
			{
				infstr = initinfstr();
				formatinfstr(infstr, "w%s %s %s %s", frtoa(rules->unconlistW[k]),
					rules->layernames[i], rules->layernames[j],
						rules->unconlistWR[k]);
				(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);
			}
			if (rules->conlistM[k] >= 0)
			{
				infstr = initinfstr();
				formatinfstr(infstr, "cm%s %s %s %s", frtoa(rules->conlistM[k]),
					rules->layernames[i], rules->layernames[j],
						rules->conlistMR[k]);
				(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);
			}
			if (rules->unconlistM[k] >= 0)
			{
				infstr = initinfstr();
				formatinfstr(infstr, "m%s %s %s %s", frtoa(rules->unconlistM[k]),
					rules->layernames[i], rules->layernames[j],
						rules->unconlistMR[k]);
				(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);
			}
			if (rules->edgelist[k] >= 0)
			{
				infstr = initinfstr();
				formatinfstr(infstr, "e%s %s %s %s", frtoa(rules->edgelist[k]),
					rules->layernames[i], rules->layernames[j],
						rules->edgelistR[k]);
				(void)allocstring(&drclist[drcindex++], returninfstr(infstr), el_tempcluster);
			}
			k++;
		}

		(void)setval((INTBIG)lib, VLIBRARY, "EDTEC_DRC", (INTBIG)drclist,
			VSTRING|VISARRAY|(drccount<<VLENGTHSH));
		for(i=0; i<drccount; i++) efree(drclist[i]);
		efree((char *)drclist);
	} else
	{
		/* no rules: remove the variable */
		if (getval((INTBIG)lib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_DRC") != NOVARIABLE)
			(void)delval((INTBIG)lib, VLIBRARY, "EDTEC_DRC");
	}
}

/*
 * routine for manipulating color maps
 */
void us_teceditcolormap(void)
{
	REGISTER INTBIG i, k;
	INTBIG func, gds, drcminwid, height3d, thick3d;
	char *layerlabel[5], *layerabbrev[5], *cif, *layerletters, *dxf;
	GRAPHICS desc;
	REGISTER NODEPROTO *np;
	float spires, spicap, spiecap;
	REGISTER INTBIG *mapptr, *newmap;
	REGISTER VARIABLE *varred, *vargreen, *varblue;

	if (us_needwindow()) return;

	/* load the color map of the technology */
	us_tecedloadlibmap(el_curlib);

	/* now fill in real layer names if known */
	for(i=0; i<5; i++) layerlabel[i] = 0;
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (namesamen(np->cell->cellname, "layer-", 6) != 0) continue;
		cif = layerletters = 0;
		if (us_teceditgetlayerinfo(np, &desc, &cif, &func, &layerletters,
			&dxf, &gds, &spires, &spicap, &spiecap, &drcminwid, &height3d,
				&thick3d)) return;
		switch (desc.bits)
		{
			case LAYERT1: k = 0;   break;
			case LAYERT2: k = 1;   break;
			case LAYERT3: k = 2;   break;
			case LAYERT4: k = 3;   break;
			case LAYERT5: k = 4;   break;
			default:      k = -1;  break;
		}
		if (k >= 0)
		{
			if (layerlabel[k] == 0)
			{
				layerlabel[k] = &np->cell->cellname[6];
				layerabbrev[k] = (char *)emalloc(2, el_tempcluster);
				layerabbrev[k][0] = *layerletters;
				layerabbrev[k][1] = 0;
			}
		}
		if (cif != 0) efree(cif);
		if (layerletters != 0) efree(layerletters);
	}

	/* set defaults */
	if (layerlabel[0] == 0)
	{
		layerlabel[0] = _("Ovrlap 1");
		(void)allocstring(&layerabbrev[0], "1", el_tempcluster);
	}
	if (layerlabel[1] == 0)
	{
		layerlabel[1] = _("Ovrlap 2");
		(void)allocstring(&layerabbrev[1], "2", el_tempcluster);
	}
	if (layerlabel[2] == 0)
	{
		layerlabel[2] = _("Ovrlap 3");
		(void)allocstring(&layerabbrev[2], "3", el_tempcluster);
	}
	if (layerlabel[3] == 0)
	{
		layerlabel[3] = _("Ovrlap 4");
		(void)allocstring(&layerabbrev[3], "4", el_tempcluster);
	}
	if (layerlabel[4] == 0)
	{
		layerlabel[4] = _("Ovrlap 5");
		(void)allocstring(&layerabbrev[4], "5", el_tempcluster);
	}

	/* run the color mixing palette */
	us_colormixdlog(layerlabel);
	for(i=0; i<5; i++) efree(layerabbrev[i]);

	/* save the map on the library */
	varred = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_red_key);
	vargreen = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_green_key);
	varblue = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_blue_key);
	if (varred != NOVARIABLE && vargreen != NOVARIABLE && varblue != NOVARIABLE)
	{
		newmap = emalloc(256*3*SIZEOFINTBIG, el_tempcluster);
		mapptr = newmap;
		for(i=0; i<256; i++)
		{
			*mapptr++ = ((INTBIG *)varred->addr)[i];
			*mapptr++ = ((INTBIG *)vargreen->addr)[i];
			*mapptr++ = ((INTBIG *)varblue->addr)[i];
		}
		(void)setval((INTBIG)el_curlib, VLIBRARY, "EDTEC_colormap", (INTBIG)newmap,
			VINTEGER|VISARRAY|((256*3)<<VLENGTHSH));
		efree((char *)newmap);
	}
}

/*
 * routine for creating a new layer with shape "pp"
 */
void us_teceditcreat(INTBIG count, char *par[])
{
	REGISTER INTBIG l;
	REGISTER char *name, *pp;
	char *dummy[1];
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np, *savenp, *facet;
	HIGHLIGHT high;
	REGISTER VARIABLE *var;

	l = strlen(pp = par[0]);
	np = NONODEPROTO;
	if (namesamen(pp, "port", l) == 0 && l >= 1) np = gen_portprim;
	if (namesamen(pp, "highlight", l) == 0 && l >= 1) np = art_boxprim;
	if (namesamen(pp, "rectangle-filled", l) == 0 && l >= 11) np = art_filledboxprim;
	if (namesamen(pp, "rectangle-outline", l) == 0 && l >= 11) np = art_boxprim;
	if (namesamen(pp, "rectangle-crossed", l) == 0 && l >= 11) np = art_crossedboxprim;
	if (namesamen(pp, "polygon-filled", l) == 0 && l >= 9) np = art_filledpolygonprim;
	if (namesamen(pp, "polygon-outline", l) == 0 && l >= 9) np = art_closedpolygonprim;
	if (namesamen(pp, "lines-solid", l) == 0 && l >= 7) np = art_openedpolygonprim;
	if (namesamen(pp, "lines-dotted", l) == 0 && l >= 8) np = art_openeddottedpolygonprim;
	if (namesamen(pp, "lines-dashed", l) == 0 && l >= 8) np = art_openeddashedpolygonprim;
	if (namesamen(pp, "lines-thicker", l) == 0 && l >= 7) np = art_openedthickerpolygonprim;
	if (namesamen(pp, "circle-outline", l) == 0 && l >= 8) np = art_circleprim;
	if (namesamen(pp, "circle-filled", l) == 0 && l >= 8) np = art_filledcircleprim;
	if (namesamen(pp, "circle-half", l) == 0 && l >= 8) np = art_circleprim;
	if (namesamen(pp, "circle-arc", l) == 0 && l >= 8) np = art_circleprim;
	if (namesamen(pp, "text", l) == 0 && l >= 1) np = gen_invispinprim;

	if (np == NONODEPROTO)
	{
		ttyputerr(_("Unrecoginzed shape: '%s'"), pp);
		return;
	}

	/* make sure the facet is right */
	facet = us_needfacet();
	if (facet == NONODEPROTO) return;
	if (namesamen(facet->cell->cellname, "node-", 5) != 0 &&
		namesamen(facet->cell->cellname, "arc-", 4) != 0)
	{
		us_abortcommand(_("Must be editing a node or arc to place geometry"));
		if ((us_tool->toolstate&NODETAILS) == 0)
			ttyputmsg(_("Use 'edit-node' or 'edit-arc' options"));
		return;
	}
	if (np == gen_portprim &&
		namesamen(facet->cell->cellname, "node-", 5) != 0)
	{
		us_abortcommand(_("Can only place ports in node descriptions"));
		if ((us_tool->toolstate&NODETAILS) == 0)
			ttyputmsg(_("Use the 'edit-node' options"));
		return;
	}

	/* create the node */
	us_clearhighlightcount();
	savenp = us_curnodeproto;
	us_curnodeproto = np;
	dummy[0] = "wait-for-down";
	us_create(1, dummy);
	us_curnodeproto = savenp;

	var = getvalkey((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, us_highlightedkey);
	if (var == NOVARIABLE) return;
	(void)us_makehighlight(((char **)var->addr)[0], &high);
	if (high.fromgeom == NOGEOM) return;
	if (!high.fromgeom->entryisnode) return;
	ni = high.fromgeom->entryaddr.ni;
	(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERPATCH, VINTEGER);

	/* postprocessing on the nodes */
	if (namesamen(pp, "port", l) == 0 && l >= 1)
	{
		/* a port layer */
		if (count == 1)
		{
			name = ttygetline(M_("Port name: "));
			if (name == 0 || name[0] == 0)
			{
				us_abortedmsg();
				return;
			}
		} else name = par[1];
		var = setval((INTBIG)ni, VNODEINST, "EDTEC_portname", (INTBIG)name, VSTRING|VDISPLAY);
		if (var != NOVARIABLE) defaulttextdescript(var->textdescript, ni->geom);
		if ((us_tool->toolstate&NODETAILS) == 0)
			ttyputmsg(_("Use 'change' option to set arc connectivity and port angle"));
	}
	if (namesamen(pp, "highlight", l) == 0 && l >= 1)
	{
		/* a highlight layer */
		us_teceditsetpatch(ni, &us_edtechigh);
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_layer", (INTBIG)NONODEPROTO, VNODEPROTO);
		ttyputmsg(_("Keep highlight a constant distance from the example edge"));
	}
	if (namesamen(pp, "circle-half", l) == 0 && l >= 8)
		setarcdegrees(ni, 0.0, 180.0*EPI/180.0);
	if ((us_tool->toolstate&NODETAILS) == 0)
	{
		if (namesamen(pp, "rectangle-", 10) == 0)
			ttyputmsg(_("Use 'change' option to set a layer for this shape"));
		if (namesamen(pp, "polygon-", 8) == 0)
		{
			ttyputmsg(_("Use 'change' option to set a layer for this shape"));
			ttyputmsg(_("Use 'outline edit' to describe polygonal shape"));
		}
		if (namesamen(pp, "lines-", 6) == 0)
			ttyputmsg(_("Use 'change' option to set a layer for this line"));
		if (namesamen(pp, "circle-", 7) == 0)
			ttyputmsg(_("Use 'change' option to set a layer for this circle"));
		if (namesamen(pp, "text", l) == 0 && l >= 1)
		{
			ttyputmsg(_("Use 'change' option to set a layer for this text"));
			ttyputmsg(_("Use 'var textedit ~.ART_message' command to set text"));
			ttyputmsg(_("Then use 'var change ~.ART_message display' command"));
		}
	}
	if (namesamen(pp, "circle-arc", l) == 0 && l >= 8)
	{
		setarcdegrees(ni, 0.0, 45.0*EPI/180.0);
		if ((us_tool->toolstate&NODETAILS) == 0)
			ttyputmsg(_("Use 'setarc' command to set portion of circle"));
	}
}

/*
 * routine to highlight information about all layers (or ports if "doports" is true)
 */
void us_teceditidentify(BOOLEAN doports)
{
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	REGISTER INTBIG total, qtotal, i, j, bestrot, indent;
	REGISTER INTBIG xsep, ysep, *xpos, *ypos, dist, bestdist, *style;
	INTBIG lx, hx, ly, hy;
	REGISTER EXAMPLE *nelist;
	REGISTER SAMPLE *ns, **whichsam;
	static POLYGON *poly = NOPOLYGON;
	extern GRAPHICS us_hbox;

	np = us_needfacet();
	if (np == NONODEPROTO) return;

	if (doports)
	{
		if (namesamen(np->cell->cellname, "node-", 5) != 0)
		{
			us_abortcommand(_("Must be editing a node to identify ports"));
			if ((us_tool->toolstate&NODETAILS) == 0)
				ttyputmsg(M_("Use the 'edit-node' option"));
			return;
		}
	} else
	{
		if (namesamen(np->cell->cellname, "node-", 5) != 0 &&
			namesamen(np->cell->cellname, "arc-", 4) != 0)
		{
			us_abortcommand(_("Must be editing a node or arc to identify layers"));
			if ((us_tool->toolstate&NODETAILS) == 0)
				ttyputmsg(M_("Use 'edit-node' or 'edit-arc' options"));
			return;
		}
	}

	/* get examples */
	if (namesamen(np->cell->cellname, "node-", 5) == 0)
		nelist = us_tecedgetexamples(np, TRUE); else
			nelist = us_tecedgetexamples(np, FALSE);
	if (nelist == NOEXAMPLE) return;

	/* count the number of appropriate samples in the main example */
	total = 0;
	for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
	{
		if (!doports)
		{
			if (ns->layer != gen_portprim) total++;
		} else
		{
			if (ns->layer == gen_portprim) total++;
		}
	}
	if (total == 0)
	{
		us_tecedfreeexamples(nelist);
		us_abortcommand(_("There are no %s to identify"), (!doports ? _("layers") : _("ports")));
		return;
	}

	/* make arrays for position and association */
	xpos = (INTBIG *)emalloc(total * SIZEOFINTBIG, el_tempcluster);
	if (xpos == 0) return;
	ypos = (INTBIG *)emalloc(total * SIZEOFINTBIG, el_tempcluster);
	if (ypos == 0) return;
	style = (INTBIG *)emalloc(total * SIZEOFINTBIG, el_tempcluster);
	if (style == 0) return;
	whichsam = (SAMPLE **)emalloc(total * (sizeof (SAMPLE *)), el_tempcluster);
	if (whichsam == 0) return;

	/* fill in label positions */
	qtotal = (total+3) / 4;
	ysep = (el_curwindowpart->screenhy-el_curwindowpart->screenly) / qtotal;
	xsep = (el_curwindowpart->screenhx-el_curwindowpart->screenlx) / qtotal;
	indent = (el_curwindowpart->screenhy-el_curwindowpart->screenly) / 15;
	for(i=0; i<qtotal; i++)
	{
		/* label on the left side */
		xpos[i] = el_curwindowpart->screenlx + indent;
		ypos[i] = el_curwindowpart->screenly + ysep * i + ysep/2;
		style[i] = TEXTLEFT;
		if (i+qtotal < total)
		{
			/* label on the top side */
			xpos[i+qtotal] = el_curwindowpart->screenlx + xsep * i + xsep/2;
			ypos[i+qtotal] = el_curwindowpart->screenhy - indent;
			style[i+qtotal] = TEXTTOP;
		}
		if (i+qtotal*2 < total)
		{
			/* label on the right side */
			xpos[i+qtotal*2] = el_curwindowpart->screenhx - indent;
			ypos[i+qtotal*2] = el_curwindowpart->screenly + ysep * i + ysep/2;
			style[i+qtotal*2] = TEXTRIGHT;
		}
		if (i+qtotal*3 < total)
		{
			/* label on the bottom side */
			xpos[i+qtotal*3] = el_curwindowpart->screenlx + xsep * i + xsep/2;
			ypos[i+qtotal*3] = el_curwindowpart->screenly + indent;
			style[i+qtotal*3] = TEXTBOT;
		}
	}

	/* fill in sample associations */
	i = 0;
	for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
	{
		if (!doports)
		{
			if (ns->layer != gen_portprim) whichsam[i++] = ns;
		} else
		{
			if (ns->layer == gen_portprim) whichsam[i++] = ns;
		}
	}

	/* rotate through all configurations, finding least distance */
	bestdist = MAXINTBIG;
	for(i=0; i<total; i++)
	{
		/* find distance from each label to its sample center */
		dist = 0;
		for(j=0; j<total; j++)
			dist += computedistance(xpos[j], ypos[j], whichsam[j]->xpos, whichsam[j]->ypos);
		if (dist < bestdist)
		{
			bestdist = dist;
			bestrot = i;
		}

		/* rotate the samples */
		ns = whichsam[0];
		for(j=1; j<total; j++) whichsam[j-1] = whichsam[j];
		whichsam[total-1] = ns;
	}

	/* rotate back to the best orientation */
	for(i=0; i<bestrot; i++)
	{
		ns = whichsam[0];
		for(j=1; j<total; j++) whichsam[j-1] = whichsam[j];
		whichsam[total-1] = ns;
	}

	/* get polygon */
	(void)needstaticpolygon(&poly, 2, us_tool->cluster);

	/* draw the highlighting */
	us_clearhighlightcount();
	for(i=0; i<total; i++)
	{
		ns = whichsam[i];
		poly->xv[0] = xpos[i];
		poly->yv[0] = ypos[i];
		poly->count = 1;
		if (ns->layer == NONODEPROTO)
		{
			poly->string = "HIGHLIGHT";
		} else if (ns->layer == gen_facetcenterprim)
		{
			poly->string = "GRAB";
		} else if (ns->layer == gen_portprim)
		{
			var = getval((INTBIG)ns->node, VNODEINST, VSTRING, "EDTEC_portname");
			if (var == NOVARIABLE) poly->string = "?"; else
				poly->string = (char *)var->addr;
		} else poly->string = &ns->layer->cell->cellname[6];
		poly->desc = &us_hbox;
		poly->style = style[i];
		TDCLEAR(poly->textdescript);
		TDSETSIZE(poly->textdescript, TXTSETQLAMBDA(4));
		poly->tech = el_curtech;
		us_hbox.col = HIGHLIT;

		nodesizeoffset(ns->node, &lx, &ly, &hx, &hy);
		switch (poly->style)
		{
			case TEXTLEFT:
				poly->xv[1] = ns->node->lowx+lx;
				poly->yv[1] = (ns->node->lowy + ns->node->highy) / 2;
				poly->style = TEXTBOTLEFT;
				break;
			case TEXTRIGHT:
				poly->xv[1] = ns->node->highx-hx;
				poly->yv[1] = (ns->node->lowy + ns->node->highy) / 2;
				poly->style = TEXTBOTRIGHT;
				break;
			case TEXTTOP:
				poly->xv[1] = (ns->node->lowx + ns->node->highx) / 2;
				poly->yv[1] = ns->node->highy-hy;
				poly->style = TEXTTOPLEFT;
				break;
			case TEXTBOT:
				poly->xv[1] = (ns->node->lowx + ns->node->highx) / 2;
				poly->yv[1] = ns->node->lowy+ly;
				poly->style = TEXTBOTLEFT;
				break;
		}
		us_showpoly(poly, el_curwindowpart);

		/* now draw the vector polygon */
		poly->count = 2;
		poly->style = VECTORS;
		us_showpoly(poly, el_curwindowpart);
	}

	/* free rotation arrays */
	efree((char *)xpos);
	efree((char *)ypos);
	efree((char *)style);
	efree((char *)whichsam);

	/* free all examples */
	us_tecedfreeexamples(nelist);
}

/*
 * routine to print information about selected object
 */
void us_teceditinquire(void)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var, *var2;
	REGISTER INTBIG opt;
	REGISTER HIGHLIGHT *high;

	high = us_getonehighlight();
	if ((high->status&HIGHTYPE) != HIGHTEXT && (high->status&HIGHTYPE) != HIGHFROM)
	{
		us_abortcommand(_("Must select a single object for inquiry"));
		return;
	}
	if ((high->status&HIGHTYPE) == HIGHFROM && !high->fromgeom->entryisnode)
	{
		/* describe currently highlighted arc */
		ai = high->fromgeom->entryaddr.ai;
		if (ai->proto != gen_universalarc)
		{
			ttyputmsg(_("This is an unimportant %s arc"), describearcproto(ai->proto));
			return;
		}
		if (ai->end[0].nodeinst->proto != gen_portprim ||
			ai->end[1].nodeinst->proto != gen_portprim)
		{
			ttyputmsg(_("This arc makes an unimportant connection"));
			return;
		}
		var = getval((INTBIG)ai->end[0].nodeinst, VNODEINST, VSTRING, "EDTEC_portname");
		var2 = getval((INTBIG)ai->end[1].nodeinst, VNODEINST, VSTRING, "EDTEC_portname");
		if (var == NOVARIABLE || var2 == NOVARIABLE)
			ttyputmsg(_("This arc connects two port objects")); else
				ttyputmsg(_("This arc connects ports '%s' and '%s'"),
					(char *)var->addr, (char *)var2->addr);
		return;
	}
	ni = high->fromgeom->entryaddr.ni;
	np = ni->parent;
	opt = us_tecedgetoption(ni);
	if (opt < 0)
	{
		ttyputmsg(_("This object has no relevance to technology editing"));
		return;
	}

	switch (opt)
	{
		case ARCFIXANG:
			ttyputmsg(_("This object defines the fixed-angle factor of %s"), describenodeproto(np));
			break;
		case ARCFUNCTION:
			ttyputmsg(_("This object defines the function of %s"), describenodeproto(np));
			break;
		case ARCINC:
			ttyputmsg(_("This object defines the prefered angle increment of %s"), describenodeproto(np));
			break;
		case ARCNOEXTEND:
			ttyputmsg(_("This object defines the arc extension of %s"), describenodeproto(np));
			break;
		case ARCWIPESPINS:
			ttyputmsg(_("This object defines the arc coverage of %s"), describenodeproto(np));
			break;
		case CENTEROBJ:
			ttyputmsg(_("This object identifies the grab point of %s"), describenodeproto(np));
			break;
		case LAYER3DHEIGHT:
			ttyputmsg(_("This object defines the 3D height of %s"), describenodeproto(np));
			break;
		case LAYER3DTHICK:
			ttyputmsg(_("This object defines the 3D thickness of %s"), describenodeproto(np));
			break;
		case LAYERCIF:
			ttyputmsg(_("This object defines the CIF name of %s"), describenodeproto(np));
			break;
		case LAYERCOLOR:
			ttyputmsg(_("This object defines the color of %s"), describenodeproto(np));
			break;
		case LAYERDXF:
			ttyputmsg(_("This object defines the DXF name(s) of %s"), describenodeproto(np));
			break;
		case LAYERDRCMINWID:
			ttyputmsg(_("This object defines the minimum DRC width of %s (OBSOLETE)"), describenodeproto(np));
			break;
		case LAYERFUNCTION:
			ttyputmsg(_("This object defines the function of %s"), describenodeproto(np));
			break;
		case LAYERGDS:
			ttyputmsg(_("This object defines the Calma GDS-II number of %s"), describenodeproto(np));
			break;
		case LAYERLETTERS:
			ttyputmsg(_("This object defines the letters to use for %s"), describenodeproto(np));
			break;
		case LAYERPATTERN:
			ttyputmsg(_("This is one of the bitmap squares in %s"), describenodeproto(np));
			break;
		case LAYERSPICAP:
			ttyputmsg(_("This object defines the SPICE capacitance of %s"), describenodeproto(np));
			break;
		case LAYERSPIECAP:
			ttyputmsg(_("This object defines the SPICE edge capacitance of %s"), describenodeproto(np));
			break;
		case LAYERSPIRES:
			ttyputmsg(_("This object defines the SPICE resistance of %s"), describenodeproto(np));
			break;
		case LAYERSTYLE:
			ttyputmsg(_("This object defines the style of %s"), describenodeproto(np));
			break;
		case LAYERPATCH:
		case HIGHLIGHTOBJ:
			np = us_tecedgetlayer(ni);
			if (np == 0)
				ttyputerr(_("This is an object with no valid layer!")); else
			{
				if (np == NONODEPROTO) ttyputmsg(_("This is a highlight box")); else
					ttyputmsg(_("This is a '%s' layer"), &np->cell->cellname[6]);
				var = getval((INTBIG)ni, VNODEINST, VSTRING, "EDTEC_minbox");
				if (var != NOVARIABLE)
					ttyputmsg(_("   It is at minimum size"));
			}
			break;
		case NODEFUNCTION:
			ttyputmsg(_("This object defines the function of %s"), describenodeproto(np));
			break;
		case NODELOCKABLE:
			ttyputmsg(_("This object tells if %s can be locked (used in array technologies)"),
				describenodeproto(np));
			break;
		case NODESERPENTINE:
			ttyputmsg(_("This object tells if %s is a serpentine transistor"), describenodeproto(np));
			break;
		case NODESQUARE:
			ttyputmsg(_("This object tells if %s is square"), describenodeproto(np));
			break;
		case NODEWIPES:
			ttyputmsg(_("This object tells if %s disappears when conencted to one or two arcs"),
				describenodeproto(np));
			break;
		case PORTOBJ:
			var = getval((INTBIG)ni, VNODEINST, VSTRING, "EDTEC_portname");
			if (var == NOVARIABLE) ttyputmsg(_("This is a port object")); else
				ttyputmsg(_("This is port '%s'"), (char *)var->addr);
			break;
		case TECHDESCRIPT:
			ttyputmsg(_("This object contains the technology description"));
			break;
		case TECHLAMBDA:
			ttyputmsg(_("This object defines the value of lambda"));
			break;
		default:
			ttyputerr(_("This object has unknown information"));
			break;
	}
}

/*
 * routine for modifying the selected object.  If two are selected, connect them.
 */
void us_teceditmodobject(INTBIG count, char *par[])
{
	REGISTER NODEINST *ni;
	GEOM *fromgeom, *togeom;
	PORTPROTO *fromport, *toport;
	INTBIG opt;
	REGISTER HIGHLIGHT *high;
	HIGHLIGHT newhigh;
	char *newpar[2];

	/* make sure something is highlighted */
	high = us_getonehighlight();
	if (high == NOHIGHLIGHT) return;

	/* if two are highlighted, connect them */
	if (!us_gettwoobjects(&fromgeom, &fromport, &togeom, &toport))
	{
		newpar[0] = "angle";   newpar[1] = "0";
		us_create(2, newpar);
		return;
	}

	/* must be one node highlighted */
	ni = (NODEINST *)us_getobject(VNODEINST, TRUE);
	if (ni == NONODEINST) return;

	/* determine technology editor relevance */
	opt = us_tecedgetoption(ni);

	/* special case for pattern blocks: node is replaced */
	if (opt == LAYERPATTERN)
	{
		us_clearhighlightcount();
		us_tecedlayerpattern(ni);
		return;
	}

	/* special case for port modification: reset highlighting by hand */
	if (opt == PORTOBJ)
	{
		/* pick up old highlight values and then remove highlighting */
		newhigh = *high;
		us_clearhighlightcount();

		/* modify the port */
		us_tecedmodport(ni, count, par);

		/* set new highlighting variable */
		newhigh.fromvar = getval((INTBIG)ni, VNODEINST, VSTRING, "EDTEC_portname");
		us_addhighlight(&newhigh);
		return;
	}

	/* ignore if no parameter given */
	if (count <= 0) return;

	/* handle other cases */
	us_pushhighlight();
	us_clearhighlightcount();
	switch (opt)
	{
		case ARCFIXANG:      us_tecedarcfixang(ni, count, par);      break;
		case ARCFUNCTION:    us_tecedarcfunction(ni, count, par);    break;
		case ARCINC:         us_tecedarcinc(ni, count, par);         break;
		case ARCNOEXTEND:    us_tecedarcnoextend(ni, count, par);    break;
		case ARCWIPESPINS:   us_tecedarcwipes(ni, count, par);       break;
		case LAYER3DHEIGHT:  us_tecedlayer3dheight(ni, count, par);  break;
		case LAYER3DTHICK:   us_tecedlayer3dthick(ni, count, par);   break;
		case LAYERCIF:       us_tecedlayercif(ni, count, par);       break;
		case LAYERCOLOR:     us_tecedlayercolor(ni, count, par);     break;
		case LAYERDXF:       us_tecedlayerdxf(ni, count, par);       break;
		case LAYERDRCMINWID: us_tecedlayerdrcminwid(ni, count, par); break;
		case LAYERFUNCTION:  us_tecedlayerfunction(ni, count, par);  break;
		case LAYERGDS:       us_tecedlayergds(ni, count, par);       break;
		case LAYERLETTERS:   us_tecedlayerletters(ni, count, par);   break;
		case LAYERPATCH:     us_tecedlayertype(ni, count, par);      break;
		case LAYERSPICAP:    us_tecedlayerspicap(ni, count, par);    break;
		case LAYERSPIECAP:   us_tecedlayerspiecap(ni, count, par);   break;
		case LAYERSPIRES:    us_tecedlayerspires(ni, count, par);    break;
		case LAYERSTYLE:     us_tecedlayerstyle(ni, count, par);     break;
		case NODEFUNCTION:   us_tecednodefunction(ni, count, par);   break;
		case NODELOCKABLE:   us_tecednodelockable(ni, count, par);   break;
		case NODESERPENTINE: us_tecednodeserpentine(ni, count, par); break;
		case NODESQUARE:     us_tecednodesquare(ni, count, par);     break;
		case NODEWIPES:      us_tecednodewipes(ni, count, par);      break;
		case TECHDESCRIPT:   us_tecedinfodescript(ni, count, par);   break;
		case TECHLAMBDA:     us_tecedinfolambda(ni, count, par);     break;
		default:             us_abortcommand(_("Cannot modify this object"));   break;
	}
	us_pophighlight(FALSE);
}

/***************************** OBJECT MODIFICATION *****************************/

void us_tecedlayer3dheight(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Enter the height of the layer when viewed in 3D"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXT3DHEIGHT);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayer3dthick(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Enter the thickness of the layer when viewed in 3D"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXT3DTHICK);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayerdrcminwid(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Enter the minimum DRC width (negative for none)"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTDRCMINWID);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayercolor(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("New color required"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTCOLOR);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));

	/* redraw the demo layer in this facet */
	us_tecedredolayergraphics(ni->parent);
}

void us_tecedlayerstyle(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Layer style required"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTSTYLE);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));

	/* redraw the demo layer in this facet */
	us_tecedredolayergraphics(ni->parent);
}

void us_tecedlayercif(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTCIF);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayerdxf(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTDXF);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayergds(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Enter the GDS layer number (-1 for none)"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTGDS);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayerspires(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Enter the SPICE layer resistance"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTSPICERES);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayerspicap(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Enter the SPICE layer capacitance"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTSPICECAP);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayerspiecap(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Enter the SPICE layer edge capacitance"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTSPICEECAP);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayerfunction(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER INTBIG func, newfunc;
	REGISTER char *str;
	REGISTER VARIABLE *var;
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Layer function required"));
		return;
	}

	/* get the existing function */
	var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
	func = 0;
	if (var != NOVARIABLE)
	{
		str = (char *)var->addr;
		if (namesamen(str, TECEDNODETEXTFUNCTION, strlen(TECEDNODETEXTFUNCTION)) == 0)
			func = us_teceditparsefun(&str[10]);
	}

	/* add in the new function */
	newfunc = us_teceditparsefun(par[0]);
	if (newfunc <= LFTYPE) func = newfunc; else func |= newfunc;

	/* build the string corresponding to this function */
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTFUNCTION);
	us_tecedaddfunstring(infstr, func);

	/* rewrite the node with the new function */
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayerletters(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER NODEPROTO *np;
	REGISTER char *pt;
	REGISTER INTBIG i;
	char *cif, *layerletters, *dxf;
	GRAPHICS desc;
	float spires, spicap, spiecap;
	INTBIG func, gds, drcminwid, height3d, thick3d;
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Layer letter(s) required"));
		return;
	}

	/* check layer letters for uniqueness */
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (namesamen(np->cell->cellname, "layer-", 6) != 0) continue;
		cif = layerletters = 0;
		if (us_teceditgetlayerinfo(np, &desc, &cif, &func, &layerletters,
			&dxf, &gds, &spires, &spicap, &spiecap, &drcminwid, &height3d,
				&thick3d)) return;
		if (cif != 0) efree(cif);
		if (layerletters == 0) continue;

		/* check these layer letters for uniqueness */
		for(pt = layerletters; *pt != 0; pt++)
		{
			for(i=0; par[0][i] != 0; i++) if (par[0][i] == *pt)
			{
				us_abortcommand(_("Cannot use letter '%c', it is in %s"), *pt, describenodeproto(np));
				efree(layerletters);
				return;
			}
		}
		efree(layerletters);
	}

	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTLETTERS);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedlayerpattern(NODEINST *ni)
{
	REGISTER NODEINST *newni;
	HIGHLIGHT high;
	REGISTER VARIABLE *var;
	INTBIG i;
	UINTSML col[8], color;

	if (ni->proto == art_boxprim)
	{
		startobjectchange((INTBIG)ni, VNODEINST);
		newni = replacenodeinst(ni, art_filledboxprim, FALSE, FALSE);
		if (newni == NONODEINST) return;
		endobjectchange((INTBIG)newni, VNODEINST);
		ni = newni;
	} else
	{
		if (ni->proto == art_filledboxprim)
		{
			startobjectchange((INTBIG)ni, VNODEINST);
			var = getvalkey((INTBIG)ni, VNODEINST, VSHORT|VISARRAY, art_patternkey);
			if (var == NOVARIABLE) color = 0xFFFF; else color = ((INTSML *)var->addr)[0];
			for(i=0; i<8; i++) col[i] = ~color;
			(void)setvalkey((INTBIG)ni, VNODEINST, art_patternkey, (INTBIG)col,
				VSHORT|VISARRAY|(8<<VLENGTHSH));
			endobjectchange((INTBIG)ni, VNODEINST);
		} else return;
	}

	high.status = HIGHFROM;
	high.facet = ni->parent;
	high.fromgeom = ni->geom;
	high.fromport = NOPORTPROTO;
	high.frompoint = 0;
	high.fromvar = NOVARIABLE;
	high.fromvarnoeval = NOVARIABLE;
	us_addhighlight(&high);

	/* redraw the demo layer in this facet */
	us_tecedredolayergraphics(ni->parent);
}

/*
 * routine to modify the layer information in node "ni".
 */
void us_tecedlayertype(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER NODEPROTO *np;
	char *cif, *layerletters, *dxf;
	REGISTER char *name;
	GRAPHICS desc;
	float spires, spicap, spiecap;
	INTBIG func, gds, drcminwid, height3d, thick3d;
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a layer name"));
		return;
	}

	np = us_needfacet();
	if (np == NONODEPROTO) return;
	if (namesame(par[0], "SET-MINIMUM-SIZE") == 0)
	{
		if (namesamen(np->cell->cellname, "node-", 5) != 0)
		{
			us_abortcommand(_("Can only set minimum size in node descriptions"));
			if ((us_tool->toolstate&NODETAILS) == 0) ttyputmsg(_("Use 'edit-node' option"));
			return;
		}
		startobjectchange((INTBIG)ni, VNODEINST);
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_minbox", (INTBIG)"MIN", VSTRING|VDISPLAY);
		endobjectchange((INTBIG)ni, VNODEINST);
		return;
	}

	if (namesame(par[0], "CLEAR-MINIMUM-SIZE") == 0)
	{
		if (getval((INTBIG)ni, VNODEINST, VSTRING, "EDTEC_minbox") == NOVARIABLE)
		{
			ttyputmsg(_("Minimum size is not set on this layer"));
			return;
		}
		startobjectchange((INTBIG)ni, VNODEINST);
		(void)delval((INTBIG)ni, VNODEINST, "EDTEC_minbox");
		endobjectchange((INTBIG)ni, VNODEINST);
		return;
	}

	/* find the actual facet with that layer specification */
	infstr = initinfstr();
	addstringtoinfstr(infstr, "layer-");
	addstringtoinfstr(infstr, par[0]);
	name = returninfstr(infstr);
	np = getnodeproto(name);
	if (np == NONODEPROTO)
	{
		ttyputerr(_("Cannot find layer primitive %s"), name);
		return;
	}

	/* get the characteristics of that layer */
	cif = layerletters = 0;
	if (us_teceditgetlayerinfo(np, &desc, &cif, &func, &layerletters,
		&dxf, &gds, &spires, &spicap, &spiecap, &drcminwid, &height3d,
			&thick3d)) return;
	if (cif != 0) efree(cif);
	if (layerletters != 0) efree(layerletters);

	startobjectchange((INTBIG)ni, VNODEINST);
	us_teceditsetpatch(ni, &desc);
	(void)setval((INTBIG)ni, VNODEINST, "EDTEC_layer", (INTBIG)np, VNODEPROTO);
	endobjectchange((INTBIG)ni, VNODEINST);
}

/*
 * routine to modify port characteristics
 */
void us_tecedmodport(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER INTBIG total, i, len, j, yes;
	BOOLEAN changed;
	REGISTER BOOLEAN *yesno;
	REGISTER NODEPROTO *np, **conlist;
	REGISTER VARIABLE *var;

	/* build an array of arc connections */
	for(total = 0, np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (namesamen(np->cell->cellname, "arc-", 4) == 0) total++;
	conlist = (NODEPROTO **)emalloc(total * (sizeof (NODEPROTO *)), el_tempcluster);
	if (conlist == 0) return;
	for(total = 0, np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (namesamen(np->cell->cellname, "arc-", 4) == 0) conlist[total++] = np;
	yesno = (BOOLEAN *)emalloc(total * (sizeof (BOOLEAN)), el_tempcluster);
	if (yesno == 0) return;
	for(i=0; i<total; i++) yesno[i] = FALSE;

	/* put current list into the array */
	var = getval((INTBIG)ni, VNODEINST, VNODEPROTO|VISARRAY, "EDTEC_connects");
	if (var != NOVARIABLE)
	{
		len = getlength(var);
		for(j=0; j<len; j++)
		{
			for(i=0; i<total; i++)
				if (conlist[i] == ((NODEPROTO **)var->addr)[j]) break;
			if (i < total) yesno[i] = TRUE;
		}
	}

	/* parse the command parameters */
	changed = FALSE;
	for(i=0; i<count-1; i += 2)
	{
		/* search for an arc name */
		for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			if (namesamen(np->cell->cellname, "arc-", 4) == 0 &&
				namesame(&np->cell->cellname[4], par[i]) == 0) break;
		if (np != NONODEPROTO)
		{
			for(j=0; j<total; j++) if (conlist[j] == np)
			{
				if (*par[i+1] == 'y' || *par[i+1] == 'Y') yesno[j] = TRUE; else
					yesno[j] = FALSE;
				changed = TRUE;
				break;
			}
			continue;
		}

		if (namesame(par[i], "PORT-ANGLE") == 0)
		{
			(void)setval((INTBIG)ni, VNODEINST, "EDTEC_portangle", myatoi(par[i+1]), VINTEGER);
			continue;
		}
		if (namesame(par[i], "PORT-ANGLE-RANGE") == 0)
		{
			(void)setval((INTBIG)ni, VNODEINST, "EDTEC_portrange", myatoi(par[i+1]), VINTEGER);
			continue;
		}
	}

	/* store list back if it was changed */
	if (changed)
	{
		yes = 0;
		for(i=0; i<total; i++)
		{
			if (!yesno[i]) continue;
			conlist[yes++] = conlist[i];
		}
		if (yes == 0 && var != NOVARIABLE)
			(void)delval((INTBIG)ni, VNODEINST, "EDTEC_connects"); else
		{
			(void)setval((INTBIG)ni, VNODEINST, "EDTEC_connects", (INTBIG)conlist,
				VNODEPROTO|VISARRAY|(yes<<VLENGTHSH));
		}
	}
	efree((char *)conlist);
	efree((char *)yesno);
}

void us_tecedarcfunction(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a layer function"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTFUNCTION);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedarcfixang(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a yes or no"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Fixed-angle: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedarcwipes(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a yes or no"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Wipes pins: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedarcnoextend(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a yes or no"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Extend arcs: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedarcinc(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires an angle increment in degrees"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Angle increment: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecednodefunction(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a node function"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, TECEDNODETEXTFUNCTION);
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecednodeserpentine(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a yes or no"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Serpentine transistor: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecednodesquare(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a yes or no"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Square node: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecednodewipes(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a yes or no"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Invisible with 1 or 2 arcs: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecednodelockable(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a yes or no"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Lockable: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedinfolambda(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a value of lambda"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Lambda: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

void us_tecedinfodescript(NODEINST *ni, INTBIG count, char *par[])
{
	REGISTER void *infstr;

	if (par[0][0] == 0)
	{
		us_abortcommand(_("Requires a technology description"));
		return;
	}
	infstr = initinfstr();
	addstringtoinfstr(infstr, "Description: ");
	addstringtoinfstr(infstr, par[0]);
	us_tecedsetnode(ni, returninfstr(infstr));
}

/****************************** UTILITIES ******************************/

void us_tecedsetnode(NODEINST *ni, char *chr)
{
	UINTBIG descript[TEXTDESCRIPTSIZE];
	REGISTER VARIABLE *var;

	startobjectchange((INTBIG)ni, VNODEINST);
	var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
	if (var == NOVARIABLE) TDCLEAR(descript); else TDCOPY(descript, var->textdescript);
	var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)chr, VSTRING|VDISPLAY);
	if (var != NOVARIABLE) modifydescript((INTBIG)ni, VNODEINST, var, descript);
	endobjectchange((INTBIG)ni, VNODEINST);
}

/*
 * routine to call up the facet "facetname" (either create it or reedit it)
 * returns NONODEPROTO if there is an error or the facet exists
 */
NODEPROTO *us_tecedenterfacet(char *facetname)
{
	REGISTER NODEPROTO *np;
	char *newpar[2];

	np = getnodeproto(facetname);
	if (np != NONODEPROTO && np->primindex == 0)
	{
		newpar[0] = "editfacet";
		newpar[1] = facetname;
		telltool(us_tool, 2, newpar);
		return(NONODEPROTO);
	}

	/* create the facet */
	np = newnodeproto(facetname, el_curlib);
	if (np == NONODEPROTO) return(NONODEPROTO);

	/* now edit the facet */
	newpar[0] = "editfacet";
	newpar[1] = facetname;
	telltool(us_tool, 2, newpar);
	return(np);
}

/*
 * routine to redraw the demo layer in "layer" facet "np"
 */
void us_tecedredolayergraphics(NODEPROTO *np)
{
	REGISTER VARIABLE *var;
	REGISTER NODEINST *ni;
	GRAPHICS desc;
	REGISTER NODEPROTO *onp;
	char *cif, *layerletters, *dxf;
	INTBIG func, gds, drcminwid, height3d, thick3d;
	float spires, spicap, spiecap;

	/* find the demo patch in this facet */
	var = getval((INTBIG)np, VNODEPROTO, VNODEINST, "EDTEC_colornode");
	if (var == NOVARIABLE) return;
	ni = (NODEINST *)var->addr;

	/* get the current description of this layer */
	cif = layerletters = 0;
	if (us_teceditgetlayerinfo(np, &desc, &cif, &func, &layerletters,
		&dxf, &gds, &spires, &spicap, &spiecap, &drcminwid, &height3d,
			&thick3d)) return;
	if (cif != 0) efree(cif);
	if (layerletters != 0) efree(layerletters);

	/* modify the demo patch to reflect the color and pattern */
	startobjectchange((INTBIG)ni, VNODEINST);
	us_teceditsetpatch(ni, &desc);
	endobjectchange((INTBIG)ni, VNODEINST);

	/* now do this to all layers in all facets! */
	for(onp = el_curlib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
	{
		if (namesamen(onp->cell->cellname, "arc-", 4) != 0 &&
			namesamen(onp->cell->cellname, "node-", 5) != 0) continue;
		for(ni = onp->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			if (us_tecedgetoption(ni) != LAYERPATCH) continue;
			var = getval((INTBIG)ni, VNODEINST, VNODEPROTO, "EDTEC_layer");
			if (var == NOVARIABLE) continue;
			if ((NODEPROTO *)var->addr != np) continue;
			us_teceditsetpatch(ni, &desc);
		}
	}
}

void us_teceditsetpatch(NODEINST *ni, GRAPHICS *desc)
{
	REGISTER INTBIG i;
	UINTBIG pattern[8];
	UINTSML spattern[8];

	(void)setvalkey((INTBIG)ni, VNODEINST, art_colorkey, desc->col, VINTEGER);
	if ((desc->colstyle&NATURE) == PATTERNED)
	{
		if ((desc->colstyle&OUTLINEPAT) == 0)
		{
			for(i=0; i<8; i++) pattern[i] = desc->raster[i];
			(void)setvalkey((INTBIG)ni, VNODEINST, art_patternkey, (INTBIG)pattern,
				VINTEGER|VISARRAY|(8<<VLENGTHSH));
		} else
		{
			for(i=0; i<8; i++) spattern[i] = desc->raster[i];
			(void)setvalkey((INTBIG)ni, VNODEINST, art_patternkey, (INTBIG)spattern,
				VSHORT|VISARRAY|(8<<VLENGTHSH));
		}
	} else
	{
		if (getvalkey((INTBIG)ni, VNODEINST, -1, art_patternkey) != NOVARIABLE)
			(void)delvalkey((INTBIG)ni, VNODEINST, art_patternkey);
	}
}

/*
 * routine to load the color map associated with library "lib"
 */
void us_tecedloadlibmap(LIBRARY *lib)
{
	REGISTER VARIABLE *var;
	REGISTER INTBIG i;
	REGISTER INTBIG *mapptr;
	INTBIG redmap[256], greenmap[256], bluemap[256];

	var = getval((INTBIG)lib, VLIBRARY, VINTEGER|VISARRAY, "EDTEC_colormap");
	if (var != NOVARIABLE)
	{
		mapptr = (INTBIG *)var->addr;
		for(i=0; i<256; i++)
		{
			redmap[i] = *mapptr++;
			greenmap[i] = *mapptr++;
			bluemap[i] = *mapptr++;
		}

		/* disable option tracking */
		(void)setvalkey((INTBIG)us_tool, VTOOL, us_ignoreoptionchangeskey, 1,
			VINTEGER|VDONTSAVE);

		startobjectchange((INTBIG)us_tool, VTOOL);
		(void)setvalkey((INTBIG)us_tool, VTOOL, us_colormap_red_key, (INTBIG)redmap,
			VINTEGER|VISARRAY|(256<<VLENGTHSH));
		(void)setvalkey((INTBIG)us_tool, VTOOL, us_colormap_green_key, (INTBIG)greenmap,
			VINTEGER|VISARRAY|(256<<VLENGTHSH));
		(void)setvalkey((INTBIG)us_tool, VTOOL, us_colormap_blue_key, (INTBIG)bluemap,
			VINTEGER|VISARRAY|(256<<VLENGTHSH));
		endobjectchange((INTBIG)us_tool, VTOOL);

		/* re-enable option tracking */
		var = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER, us_ignoreoptionchangeskey);
		if (var != NOVARIABLE)
			(void)delvalkey((INTBIG)us_tool, VTOOL, us_ignoreoptionchangeskey);
	}
}

/*
 * routine to parse the layer facet in "np" and fill these reference descriptors:
 *   "desc" (a GRAPHICS structure)
 *   "cif" (the name of the CIF layer)
 *   "dxf" (the name of the DXF layer)
 *   "func" (the integer function number)
 *   "layerletters" (the letters associated with this layer),
 *   "gds" (the Calma GDS-II layer number)
 *   "spires" (the SPICE resistance)
 *   "spicap" (the SPICE capacitance)
 *   "spiecap" (the SPICE edge capacitance)
 *   "drcminwid" (the DRC minimum width)
 *   "height3d" (the 3D height)
 *   "thick3d" (the 3D thickness)
 * All of the reference parameters except "func", "gds", "spires", "spicap", and "spiecap"
 * get allocated.  Returns true on error.
 */
BOOLEAN us_teceditgetlayerinfo(NODEPROTO *np, GRAPHICS *desc, char **cif, INTBIG *func,
	char **layerletters, char **dxf, INTBIG *gds, float *spires, float *spicap,
	float *spiecap, INTBIG *drcminwid, INTBIG *height3d, INTBIG *thick3d)
{
	REGISTER NODEINST *ni;
	REGISTER INTBIG patterncount, i, color, l;
	REGISTER INTBIG lowx, highx, lowy, highy, x, y;
	REGISTER char *str;
	REGISTER VARIABLE *var;

	/* create and initialize the GRAPHICS structure */
	desc->colstyle = SOLIDC;
	desc->bwstyle = PATTERNED;
	for(i=0; i<8; i++) desc->raster[i] = 0;

	/* look at all nodes in the layer description facet */
	patterncount = 0;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto == gen_invispinprim)
		{
			var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
			if (var == NOVARIABLE) continue;
			str = (char *)var->addr;
			l = strlen(TECEDNODETEXTCOLOR);
			if (namesamen(str, TECEDNODETEXTCOLOR, l) == 0)
			{
				color = getecolor(&str[l]);
				if (color < 0)
				{
					ttyputerr(_("Unknown color '%s' in %s"), &str[l], describenodeproto(np));
					return(TRUE);
				}
				desc->col = color;
				switch (color)
				{
					case COLORT1: desc->bits = LAYERT1; break;
					case COLORT2: desc->bits = LAYERT2; break;
					case COLORT3: desc->bits = LAYERT3; break;
					case COLORT4: desc->bits = LAYERT4; break;
					case COLORT5: desc->bits = LAYERT5; break;
					default:      desc->bits = LAYERO;  break;
				}
				continue;
			}
			l = strlen(TECEDNODETEXTSTYLE);
			if (namesamen(str, TECEDNODETEXTSTYLE, l) == 0)
			{
				if (namesame(&str[l], "solid") == 0)
					desc->colstyle = desc->bwstyle = SOLIDC;
				if (namesame(&str[l], "patterned") == 0)
					desc->colstyle = desc->bwstyle = PATTERNED;
				if (namesame(&str[l], "patterned/outlined") == 0)
					desc->colstyle = desc->bwstyle = PATTERNED | OUTLINEPAT;
				continue;
			}
			l = strlen(TECEDNODETEXTCIF);
			if (namesamen(str, TECEDNODETEXTCIF, l) == 0)
			{
				if (allocstring(cif, &str[l], us_tool->cluster)) return(TRUE);
				continue;
			}
			l = strlen(TECEDNODETEXTDXF);
			if (namesamen(str, TECEDNODETEXTDXF, l) == 0)
			{
				if (allocstring(dxf, &str[l], us_tool->cluster)) return(TRUE);
				continue;
			}
			l = strlen(TECEDNODETEXTGDS);
			if (namesamen(str, TECEDNODETEXTGDS, l) == 0)
			{
				*gds = myatoi(&str[l]);
				continue;
			}
			l = strlen(TECEDNODETEXTGDSOLD);
			if (namesamen(str, TECEDNODETEXTGDSOLD, l) == 0)
			{
				*gds = myatoi(&str[l]);
				continue;
			}
			l = strlen(TECEDNODETEXTFUNCTION);
			if (namesamen(str, TECEDNODETEXTFUNCTION, l) == 0)
			{
				*func = us_teceditparsefun(&str[l]);
				continue;
			}
			l = strlen(TECEDNODETEXTLETTERS);
			if (namesamen(str, TECEDNODETEXTLETTERS, l) == 0)
			{
				if (allocstring(layerletters, &str[l], us_tool->cluster)) return(TRUE);
				continue;
			}
			l = strlen(TECEDNODETEXTSPICERES);
			if (namesamen(str, TECEDNODETEXTSPICERES, l) == 0)
			{
				*spires = (float)atof(&str[l]);
				continue;
			}
			l = strlen(TECEDNODETEXTSPICECAP);
			if (namesamen(str, TECEDNODETEXTSPICECAP, l) == 0)
			{
				*spicap = (float)atof(&str[l]);
				continue;
			}
			l = strlen(TECEDNODETEXTSPICEECAP);
			if (namesamen(str, TECEDNODETEXTSPICEECAP, l) == 0)
			{
				*spiecap = (float)atof(&str[l]);
				continue;
			}
			l = strlen(TECEDNODETEXTDRCMINWID);
			if (namesamen(str, TECEDNODETEXTDRCMINWID, l) == 0)
			{
				*drcminwid = atofr(&str[l]);
				continue;
			}
			l = strlen(TECEDNODETEXT3DHEIGHT);
			if (namesamen(str, TECEDNODETEXT3DHEIGHT, l) == 0)
			{
				*height3d = myatoi(&str[l]);
				continue;
			}
			l = strlen(TECEDNODETEXT3DTHICK);
			if (namesamen(str, TECEDNODETEXT3DTHICK, l) == 0)
			{
				*thick3d = myatoi(&str[l]);
				continue;
			}
			continue;
		}

		if (ni->proto == art_boxprim || ni->proto == art_filledboxprim)
		{
			var = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_option");
			if (var == NOVARIABLE) continue;
			if (var->addr != LAYERPATTERN) continue;
			if (patterncount == 0)
			{
				lowx = ni->lowx;   highx = ni->highx;
				lowy = ni->lowy;   highy = ni->highy;
			} else
			{
				if (ni->lowx < lowx) lowx = ni->lowx;
				if (ni->highx > highx) highx = ni->highx;
				if (ni->lowy < lowy) lowy = ni->lowy;
				if (ni->highy > highy) highy = ni->highy;
			}
			patterncount++;
		}
	}

	if (patterncount != 16*8)
	{
		ttyputerr(_("Incorrect number of pattern boxes in %s (has %ld, not %d)"),
			describenodeproto(np), patterncount, 16*8);
		return(TRUE);
	}

	/* construct the pattern */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto != art_filledboxprim) continue;
		var = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_option");
		if (var == NOVARIABLE) continue;
		if (var->addr != LAYERPATTERN) continue;
		var = getvalkey((INTBIG)ni, VNODEINST, VSHORT|VISARRAY, art_patternkey);
		if (var != NOVARIABLE)
		{
			for(i=0; i<8; i++) if (((INTSML *)var->addr)[i] != 0) break;
			if (i >= 8) continue;
		}
		x = (ni->lowx - lowx) / ((highx-lowx) / 16);
		y = (highy - ni->highy) / ((highy-lowy) / 8);
		desc->raster[y] |= (1 << (15-x));
	}
	return(FALSE);
}

/*
 * routine to parse the layer function string "str" and return the
 * actual function codes
 */
INTBIG us_teceditparsefun(char *str)
{
	REGISTER INTBIG func, save, i;
	REGISTER char *pt;

	func = 0;
	for(;;)
	{
		/* find the next layer function name */
		pt = str;
		while (*pt != 0 && *pt != ',') pt++;

		/* parse the name */
		save = *pt;
		*pt = 0;
		for(i=0; us_teclayer_functions[i].name != 0; i++)
			if (namesame(str, us_teclayer_functions[i].name) == 0) break;
		*pt = (char)save;
		if (us_teclayer_functions[i].name == 0)
		{
			ttyputerr(_("Unknown layer function: %s"), str);
			return(0);
		}

		/* mix in the layer function */
		if (us_teclayer_functions[i].value <= LFTYPE)
		{
			if (func != 0)
			{
				ttyputerr(_("Cannot be both function %s and %s"),
					us_teclayer_functions[func&LFTYPE].name, us_teclayer_functions[i].name);
				func = 0;
			}
			func = us_teclayer_functions[i].value;
		} else func |= us_teclayer_functions[i].value;

		/* advance to the next layer function name */
		if (*pt == 0) break;
		str = pt + 1;
	}
	return(func);
}

/*
 * routine to return the option index of node "ni"
 */
INTBIG us_tecedgetoption(NODEINST *ni)
{
	REGISTER VARIABLE *var, *var2;
	REGISTER NODEPROTO *np;

	/* port objects are readily identifiable */
	if (ni->proto == gen_portprim) return(PORTOBJ);

	/* center objects are also readily identifiable */
	if (ni->proto == gen_facetcenterprim) return(CENTEROBJ);

	var = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_option");
	if (var == NOVARIABLE) return(-1);
	if (var->addr == LAYERPATCH)
	{
		/* may be a highlight object */
		var2 = getval((INTBIG)ni, VNODEINST, VNODEPROTO, "EDTEC_layer");
		if (var2 != NOVARIABLE)
		{
			np = (NODEPROTO *)var2->addr;
			if (np == NONODEPROTO) return(HIGHLIGHTOBJ);
		}
	}
	return(var->addr);
}

/*
 * Routine called when facet "np" has been deleted (and it may be a layer facet because its name
 * started with "layer-").
 */
void us_teceddeletelayerfacet(NODEPROTO *np)
{
	REGISTER VARIABLE *var;
	REGISTER NODEPROTO *onp;
	REGISTER NODEINST *ni;
	REGISTER BOOLEAN warned, isnode;
	REGISTER char *layername;
	static INTBIG edtec_layer_key = 0;
	REGISTER void *infstr;

	/* may have deleted layer facet in technology library */
	if (edtec_layer_key == 0) edtec_layer_key = makekey("EDTEC_layer");
	layername = &np->cell->cellname[6];
	warned = FALSE;
	for(onp = np->cell->lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
	{
		if (namesamen(onp->cell->cellname, "node-", 5) == 0) isnode = TRUE; else
			if (namesamen(onp->cell->cellname, "arc-", 4) == 0) isnode = FALSE; else
				continue;
		for(ni = onp->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			var = getvalkey((INTBIG)ni, VNODEINST, VNODEPROTO, edtec_layer_key);
			if (var == NOVARIABLE) continue;
			if ((NODEPROTO *)var->addr == np) break;
		}
		if (ni != NONODEINST)
		{
			if (warned) addtoinfstr(infstr, ','); else
			{
				infstr = initinfstr();
				formatinfstr(infstr, _("Warning: layer %s is used in"), layername);
				warned = TRUE;
			}
			if (isnode) formatinfstr(infstr, _(" node %s"), &onp->cell->cellname[5]); else
				formatinfstr(infstr, _(" arc %s"), &onp->cell->cellname[4]);
		}
	}
	if (warned)
		ttyputmsg("%s", returninfstr(infstr));

	/* see if this layer is mentioned in the design rules */
	us_tecedrenamecell(np->cell->cellname, "");
}

/*
 * Routine called when facet "np" has been deleted (and it may be a node facet because its name
 * started with "node-").
 */
void us_teceddeletenodefacet(NODEPROTO *np)
{
	/* see if this node is mentioned in the design rules */
	us_tecedrenamecell(np->cell->cellname, "");
}

/******************** SUPPORT FOR "usredtecp.c" ROUTINES ********************/

/*
 * routine to return the actual bounding box of layer node "ni" in the
 * reference variables "lx", "hx", "ly", and "hy"
 */
void us_tecedgetbbox(NODEINST *ni, INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy)
{
	*lx = ni->geom->lowx;
	*hx = ni->geom->highx;
	*ly = ni->geom->lowy;
	*hy = ni->geom->highy;
	if (ni->proto != gen_portprim) return;
	*lx += 4000;   *hx -= 4000;
	*ly += 4000;   *hy -= 4000;
}

void us_tecedpointout(NODEINST *ni, NODEPROTO *np)
{
	REGISTER WINDOWPART *w;
	char *newpar[2];

	for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		if (w->curnodeproto == np) break;
	if (w == NOWINDOWPART)
	{
		newpar[0] = describenodeproto(np);
		us_editfacet(1, newpar);
	}
	if (ni != NONODEINST)
	{
		us_clearhighlightcount();
		(void)asktool(us_tool, "show-object", (INTBIG)ni->geom);
	}
}

/*
 * routine to swap entries "p1" and "p2" of the port list in "tlist"
 */
void us_tecedswapports(INTBIG *p1, INTBIG *p2, TECH_NODES *tlist)
{
	REGISTER INTBIG temp, *templ;
	REGISTER char *tempc;

	templ = tlist->portlist[*p1].portarcs;
	tlist->portlist[*p1].portarcs = tlist->portlist[*p2].portarcs;
	tlist->portlist[*p2].portarcs = templ;

	tempc = tlist->portlist[*p1].protoname;
	tlist->portlist[*p1].protoname = tlist->portlist[*p2].protoname;
	tlist->portlist[*p2].protoname = tempc;

	temp = tlist->portlist[*p1].initialbits;
	tlist->portlist[*p1].initialbits = tlist->portlist[*p2].initialbits;
	tlist->portlist[*p2].initialbits = temp;

	temp = tlist->portlist[*p1].lowxmul;
	tlist->portlist[*p1].lowxmul = tlist->portlist[*p2].lowxmul;
	tlist->portlist[*p2].lowxmul = (INTSML)temp;
	temp = tlist->portlist[*p1].lowxsum;
	tlist->portlist[*p1].lowxsum = tlist->portlist[*p2].lowxsum;
	tlist->portlist[*p2].lowxsum = (INTSML)temp;

	temp = tlist->portlist[*p1].lowymul;
	tlist->portlist[*p1].lowymul = tlist->portlist[*p2].lowymul;
	tlist->portlist[*p2].lowymul = (INTSML)temp;
	temp = tlist->portlist[*p1].lowysum;
	tlist->portlist[*p1].lowysum = tlist->portlist[*p2].lowysum;
	tlist->portlist[*p2].lowysum = (INTSML)temp;

	temp = tlist->portlist[*p1].highxmul;
	tlist->portlist[*p1].highxmul = tlist->portlist[*p2].highxmul;
	tlist->portlist[*p2].highxmul = (INTSML)temp;
	temp = tlist->portlist[*p1].highxsum;
	tlist->portlist[*p1].highxsum = tlist->portlist[*p2].highxsum;
	tlist->portlist[*p2].highxsum = (INTSML)temp;

	temp = tlist->portlist[*p1].highymul;
	tlist->portlist[*p1].highymul = tlist->portlist[*p2].highymul;
	tlist->portlist[*p2].highymul = (INTSML)temp;
	temp = tlist->portlist[*p1].highysum;
	tlist->portlist[*p1].highysum = tlist->portlist[*p2].highysum;
	tlist->portlist[*p2].highysum = (INTSML)temp;

	/* finally, swap the actual identifiers */
	temp = *p1;   *p1 = *p2;   *p2 = temp;
}

char *us_tecedsamplename(NODEPROTO *layernp)
{
	if (layernp == gen_portprim) return("PORT");
	if (layernp == gen_facetcenterprim) return("GRAB");
	if (layernp == NONODEPROTO) return("HIGHLIGHT");
	return(&layernp->cell->cellname[6]);
}
