# --------------------------------------------------------------------------
# Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
#
# You can use and distribute this software under the terms of the license
# version 1 you should have received along with this software.
# If not or if you want additional information, write to
# Forschungszentrum Informatik, "OBST", Haid-und-Neu-Strasse 10-14,
# D-76131 Karlsruhe, Germany.
# --------------------------------------------------------------------------
##############################################################################
# Module:	ObjectKernel (Version 2.2-1)
# Author:	Rainer Neumann
# Date  :	June 17, 1994
# State :	finished
##############################################################################

if { [info command ok_version] == "" } {

  if { [catch {set env(OK_KNL_PATH)} _tmp_knl_path] } {
    set _tmp_knl_path [pwd]
  }

  proc ok_version {} { return 2.2-1 }
  set  ok_version [ok_version]

  set  ok_release_notes(version)       $ok_version
  set  ok_release_notes(date)          "June 17, 1994"
  set  ok_release_notes(author)        "Rainer Neumann"
  set  ok_release_notes(email)         "rneumann@fzi.de"
  set  ok_release_notes(email-subject) "Using OK"

  if { [catch "exec cat $_tmp_knl_path/ObjectKernel.changes" ok_release_notes(changes)] } {
    set ok_release_notes(changes) "No registered changes to former version"
  }

##############################################################################
# Set ok-paths

if { [catch "set env(OK_KNL_PATH)" OK_KNL_PATH] } {
  set OK_KNL_PATH [pwd]
}
if { [catch "set env(OK_LIB_PATH)" _tmp_ok_lib_path] } {
  set OK_LIB_PATH $_tmp_ok_knl_path
} else {
  set OK_LIB_PATH {}
  while { $_tmp_ok_lib_path != "" } {
    set path [ctoken _tmp_ok_lib_path :]
    if { $path != "" } {
      lappend OK_LIB_PATH $path
    }
  }
}
unset _tmp_ok_lib_path
if { [catch "set env(OK_ROOT_PATH)" OK_ROOT_PATH] } {
  set OK_ROOT_PATH [pwd]
}
if { [regexp {.*:.*} $OK_ROOT_PATH] } {
  ok_error Kernel "Environment variable OK_ROOT_PATH must be single path"
}

##############################################################################
# Module- and Load-management functions

set ok_display_load_msg 0
set _ok_loaded_modules {}
set _ok_load_message ""

proc ok_which { filename } {
  # function returns the complete filename if such the given file
  # exists in the set path or an empty string if not
  global OK_LIB_PATH OK_ROOT_PATH
  set found ""
  if { [lsearch {/ .} [cindex $filename 0]] != -1 } {
    if { [file exists $filename] } {
      set found $filename
    }
  } else {
    foreach path $OK_LIB_PATH {
      if { [file exists $path/$filename] } {
	set found $path/$filename
	break
      }
    }
    if { $found == "" } {
      if { [file exists $OK_ROOT_PATH/$filename] } {
        set found $OK_ROOT_PATH/$filename
      } elseif { [file exists $OK_ROOT_PATH/bin/$filename] } {
        set found $OK_ROOT_PATH/bin/$filename
      } elseif { [file exists $OK_ROOT_PATH/lib/$filename] } {
        set found $OK_ROOT_PATH/lib/$filename
      }
    }
  }
  return $found
}

proc ok_include { filelist } {
  foreach file $filelist {
    set found [ok_which $file]
    if { $found != "" } {
      source $found
    } else {
      ok_error ok_include "No file \"$file\" found in path."
    }
  }
}

proc ok_module { name module } {
  global _ok_loaded_modules _ok_load_message ok_display_load_msg ok_syntax
  if { [lsearch $_ok_loaded_modules $name] == -1 } {
    lappend _ok_loaded_modules $name
    set _ok_load_message "loading module $name"
    if { $ok_display_load_msg } {
      puts stdout $_ok_load_message
    }
    set ok_syntax [ok_version]
    uplevel #0 $module
    set _ok_load_message "module $name loaded"
    if { $ok_display_load_msg } {
      puts stdout $_ok_load_message
    }
  }
}

##############################################################################
# expand the info-command

rename info tcl_info
proc info { args } {
  global _ok_ClassTable
  if { [llength $args] == 0 } {
    error "wrong # args: should be \"info option ?arg arg ...?\""
  } else {
    set item [lindex $args 0]
    if { [regexp -nocase "class.*" $item] } {
      return [array name _ok_ClassTable]
    } else {
      return [uplevel 1 tcl_info $args]
    }
  }
}

##############################################################################
# define procedure intersect3 if it does not exist

if { [info command intersect3] == "" } {
  proc intersect3 { list1 list2 } {
    foreach e $list1 {
      set is3a($e) 0
    }
    foreach e $list2 {
      if { [catch "set is3a($e)"] } {
	set is3a($e) 2
      } else {
	set is3a($e) 1
      }
    }
    set r0 {}
    set r1 {}
    set r2 {}
    if { ![catch {array names is3a} elements] } {
      foreach e [lsort [array names is3a]] {
	lappend r$is3a($e) $e
      }
    }
    return [list $r0 $r1 $r2]
  }
}

##############################################################################

set ok_syntax $ok_version

if { [info global ok_objectcheck] == "" } {
  set ok_objectcheck 1
}

##############################################################################
# Error, Warning, and Message-functions

proc ok_error { sender { error "" } } {
  if { $error == "" } { set error "Unspecified error" }
  error "Errormessage from $sender : $error"
}
proc ok_warning { sender { wrn "" } } {
  puts stdout "Warning from $sender : $wrn"
}
proc ok_message { sender { msg "" } } {
  puts stdout "Message from $sender : $msg"
}

##############################################################################
# Some useful constants

set NULL  NULL
set Null  NULL
set null  NULL
set TRUE  1
set True  1
set true  1
set FALSE 0
set False 0
set false 0

##############################################################################
# Kernel variables

global _ok_ObjectCount _ok_InterpreterOffset
global _ok_ClassTable _ok_AttributeTable _ok_MethodTable

# Description of the important kernel variables
#   _ok_ObjectCount:
#       no comment
#   _ok_InterpreterOffset:
#       String added to OIDs
#   _ok_ClassTable: 
#       _ok_ClassTable[<class>] = { <superclass> <direct subclasses> }
#   _ok_AttributeTable:
#       _ok_AttributeTable[<class>] = { <attributes> <defaults>
#                                       <static attrib.> <their defaults> }
#       _ok_AttributeTable[<class>,<attr>] = { <default> <type> [<prop>] }
#           where <default> is the default value of the attribute,
#           type is its specified type and prop is either static or empty
#   _ok_MethodTable:
#       _ok_MethodTable[<class>] = <methodlist>
#       _ok_MethodTable[<class>,<method>] = <implementation class>

##############################################################################
# Preparing OIDs

set _ok_ObjectCount 0

proc ok_short_ids {} {
  global _ok_InterpreterOffset
  set _ok_InterpreterOffset ""
}

proc ok_long_ids {} {
  global _ok_InterpreterOffset
  set __d [exec date]
  regsub -all ":" $__d "" __d
  set _ok_InterpreterOffset \
         "[exec hostid][lindex $__d 1][lindex $__d 2][lindex $__d 3][pid]"
}

ok_short_ids

##############################################################################
# Registration functions

global _ok_registered_procedures
set _ok_registered_procedures {}

proc ok_proc { procname arguments body } {
  global _ok_registered_procedures
  uplevel 1 [list proc $procname $arguments $body]
  if { [lsearch $_ok_registered_procedures $procname] == -1 } {
    lappend _ok_registered_procedures $procname
  }
  return {}
}

global _ok_registered_includes
set _ok_registered_includes {}

proc ok_source { filename } {
  global _ok_registered_includes
  set file [ok_which $filename]
  if { $file != "" } {
    uplevel 1 [list source $file]
    if { [lsearch $_ok_registered_includes $file] == -1 } {
      lappend _ok_registered_includes $file
    }
  } else {
    ok_error ok_source "$filename not found"
  }
  return {}
}

global _ok_registered_variables
set _ok_registered_variables {}

proc ok_variable { args } {
  global _ok_registered_variables
  foreach arg $args {
    if { [lsearch $_ok_registered_variables $arg] == -1 } {
      lappend _ok_registered_variables $arg
    }
  }
  return {}
}

##############################################################################
# Define enumeration type

global _ok_EnumerationTable

proc ok_enumeration { name values } {
  global _ok_EnumerationTable
  set _ok_EnumerationTable($name) $values
  eval "ok_proc is_$name { val } {
  global _ok_EnumerationTable
  if { \[lsearch \$_ok_EnumerationTable($name) \$val\] > -1 } {
    return 1
  } else {
    return 0
  }\n}"
  eval "ok_proc ${name}_values {} {
  global _ok_EnumerationTable
  return \$_ok_EnumerationTable($name)\n}"
  return {}
}

ok_enumeration BOOL { 0 1 }

##############################################################################
# Basic kernel functions

proc ok_newObject { } {
  global _ok_ObjectCount _ok_InterpreterOffset
  set objID "OKID_${_ok_InterpreterOffset}@${_ok_ObjectCount}"
  incr _ok_ObjectCount
  return $objID
}

proc ok_getSuperclasses { class } {
  global _ok_ClassTable
  if { [catch "set _ok_ClassTable($class)" ci] } {
    ok_error ok_getSuperclasses "No such class $class defined!"
  } else {
    return [lindex $ci 0]
  }
}

proc ok_getAllSuperclasses { class } {
  global _ok_ClassTable
  if { [catch "set _ok_ClassTable($class)" ci] } {
    ok_error ok_getAllSuperclasses "No such class $class defined!"
  } else {
    set ci [lindex $ci 0]
    set res {}
    set i [llength $ci]
    incr i -1
    while { 0 <= $i } {
      set c [lindex $ci $i]
      if { [lsearch $res $c] == -1 } {
	set res [linsert $res 0 $c]
	set ci [concat [lindex $_ok_ClassTable($c) 0] $ci]
	set i [llength $ci]
      }
      incr i -1
    }
  }
  return $res
}

proc ok_getDirectSubclasses { class } {
  global _ok_ClassTable
  if { [catch "set _ok_ClassTable($class)" ci] } {
    ok_error ok_getDirectSubclasses "No such class $class defined!"
  } else {
    return [lindex $ci 1]
  }
}

proc ok_getInstanceAttributes { class } {
  global _ok_AttributeTable
  if { [catch "set _ok_AttributeTable($class)" ai] } {
    ok_error ok_getInstanceAttributes "No such class $class defined!"
  } else {
    return [lreplace $ai 2 3]
  }
}

proc ok_getAllInstanceAttributes { class } {
  set sc [ok_getAllSuperclasses $class]
  set sc [concat $class $sc]
  set al {}
  set dl {}
  foreach cl $sc {
    if { $cl != {} } { 
      set l [ok_getInstanceAttributes $cl]
      set al [concat [lindex $l 0] $al]
      set dl [concat [lindex $l 1] $dl]
    }
  }
  return [list $al $dl]
}

proc ok_getClassAttributes { class } {
  global _ok_AttributeTable
  if { [catch "set _ok_AttributeTable($class)" ai] } {
    ok_error ok_getClassAttributes "No such class $class defined!"
  } else {
    return [lreplace $ai 0 1]
  }
}

proc ok_getAllClassAttributes { class } {
  set sc [ok_getAllSuperclasses $class]
  set sc [concat $class $sc]
  set al {}
  set dl {}
  foreach cl $sc {
    if { $cl != {} } { 
      set l [ok_getClassAttributes $cl]
      set al [concat [lindex $l 0] $al]
      set dl [concat [lindex $l 1] $dl]
    }
  }
  return [list $al $dl]
}

##############################################################################
# Function to check weather an expression describes a valid object
# or not - this function may be redefined in another context.

proc ok_valid_object { obj } {
  upvar #0 $obj inst
  if { $obj == "NULL" } { ok_error Kernel "Trying to dereference NULL Object" }
  if { [catch "set inst(Class)" res] } { return 0 } else { return 1 }
}

##############################################################################
# Following function will be called, when a direct method call to a non-object
# has taken place - this function must be overwritten if needed

proc ok_invalid_methodcall { method instance args } {
  ok_error Kernel "Invalid method call $method $instance ..."
}

##############################################################################
# Functions to create a new class

proc ok_class { classname superclasses args } {
  global _ok_ClassTable _ok_AttributeTable _ok_MethodTable
  global ok_syntax
  if { ![catch "set _ok_ClassTable($classname)" res] } {
    ok_error ok_class "Class $classname is already defined"
  } else {
    if { $classname != "Object" } {
      set ok 1
      foreach sc $superclasses {
	if { [catch "set _ok_ClassTable($sc)" res] } {
	  ok_error ok_class "Superclass $sc of class $classname is undefined"
	  set ok 0
	}
      }
      if { $ok } {
	# insert the current class into the subclass list of the superclass
	foreach sc $superclasses {
          set ci $_ok_ClassTable($sc)
	  set ssc [lindex $ci 0]
	  set sdc [lindex $ci 1]
	  lappend sdc $classname
	  set _ok_ClassTable($sc) [list $ssc $sdc]
	}
	set _ok_ClassTable($classname) [list $superclasses {}]
	# "copy" the methods of the superclasses into the current class
	set _ok_MethodTable($classname) {}
	foreach sc $superclasses {
	  foreach cm $_ok_MethodTable($sc) {
	    set ic $_ok_MethodTable($sc,$cm)
	    lappend _ok_MethodTable($classname) $cm
	    set _ok_MethodTable($classname,$cm) $ic
	    eval "proc ${classname}::$cm { args } { uplevel 1 ${ic}::$cm \$args }"
	  }
        }
      }
    } else {
      # class is "Object"
      set _ok_ClassTable(Object) {{} {}}
      set _ok_MethodTable(Object) {}
    }
    # parsing attributes
    if { [regexp "^1\[\.\].*" $ok_syntax] } {
      set att [lindex $args 0]
      set def [lindex $args 1]
      set _ok_AttributeTable($classname) [list $att $def {} {}]
      set i 0
      foreach a $att {
        set _ok_AttributeTable($classname,$a) \
	  [list [lindex $def $i] UNTYPED]
        incr i
      }
    } else {
      set att {} ; set def {} ; set satt {} ; set sdef {}
      foreach l $args {
        if { [llength $l] < 3 } {
          ok_error ok_class \
	    "Describing an attribute needs: \[static\] <name> <default> <type>"
	} else {
	  if { [lindex $l 0] == "static" } {
	    lappend satt [lindex $l 1]
            lappend sdef [lindex $l 2]
            set _ok_AttributeTable($classname,[lindex $l 1]) \
	      [list [lindex $l 2] [lindex $l 3] static]
	  } else {
	    lappend att [lindex $l 0]
            lappend def [lindex $l 1]
            set _ok_AttributeTable($classname,[lindex $l 0]) \
	      [list [lindex $l 1] [lindex $l 2]]
	  }
	}
      }
      set _ok_AttributeTable($classname) [list $att $def $satt $sdef]
    }
    # creating attribute access methods
    set att  [lindex $_ok_AttributeTable($classname) 0]
    set satt [lindex $_ok_AttributeTable($classname) 2]
    foreach a $att {
      _ok_create_method $classname get_$a 0 {} " return \$$a " $a
      _ok_create_method $classname set_$a 0 { value } " set $a \$value " $a
    }
    upvar #0 $classname statics
    set i 0
    foreach a $satt {
      set statics($a) [lindex $sdef $i]
      eval "ok_staticmethod $classname get_$a { {self NULL} } { return \$$a }"
      eval "ok_staticmethod $classname set_$a { args } {
  set $a \[lindex \$args \[expr {\[llength \$args\] - 1}\]\] }"
      incr i
    }
    # create standard destructor
    eval "ok_destructor $classname {} { [lindex $superclasses 0]::destroy \$self }"
  }
}

##############################################################################
# Functions to create methods of a class
# this function can expand the argument list an the body of a method

proc _ok_create_method { classname methodname isStatic 
                         arglist body { attr "" }} {
  global _ok_ClassTable _ok_MethodTable _ok_AttributeTable
  global ok_syntax ok_objectcheck
  if { !$isStatic } {
    set arglist [concat { self } $arglist]
    if { $ok_objectcheck } {
      set newbody "\n  if { !\[ok_valid_object \$self\] } \{
    ok_invalid_methodcall ${classname}::${methodname} \$self"
      foreach a $arglist {
	set newbody "$newbody \$$a"
      }
      set newbody "$newbody\n  \} else \{\n"
    } else {
      set newbody ""
    }
    # providing attribute access to the method
    if { $attr != "" } {
      set attribs "  upvar #0 \${self}($attr) $attr"
    } else {
      if { [regexp "^1\[\.\].*" $ok_syntax] } {
        set attribs "  upvar #0 \$self attr"
      } else {
	set attribs "  upvar #0"
	set attributes [lindex [ok_getAllInstanceAttributes $classname] 0]
	foreach a $attributes {
	  set attribs "$attribs \${self}($a) $a"
	}
      }
    }
    set newbody "$attribs\n$newbody\n$body"
    if { $ok_objectcheck } {
      set newbody "$newbody\n  \}"
    }
  } else {
    set newbody $body
  }
  # provide direct access on static attributes
  set attribs ""
  set sc [concat [ok_getAllSuperclasses $classname] $classname]
  foreach c $sc {
    set att [lindex [ok_getClassAttributes $c] 0]
    foreach a $att {
      if { $attribs == "" } {
        set attribs "  upvar #0 ${c}($a) $a"
      } else {
	set attribs "$attribs ${c}($a) $a"
      }
    }
  }
  set newbody "$attribs\n$newbody"
  set newbody "  global TRUE True true FALSE False false NULL Null null\n$newbody"
  if { $classname != "Object" } {
    set newbody "  set super {[lindex $_ok_ClassTable($classname) 0]}\n$newbody"
  }
  eval "proc $classname::$methodname { $arglist } {\n$newbody }"
  # Now write the method information into method table
  if { [lsearch $_ok_MethodTable($classname) $methodname] == -1 } {
    lappend _ok_MethodTable($classname) $methodname
  }
  set _ok_MethodTable($classname,$methodname) $classname
  set scs [ok_getAllSuperclasses $classname]
  set visited {}
  set dc [ok_getDirectSubclasses $classname]
  while { $dc != {} } {
    set newdc {}
    foreach cl $dc {
      if { [catch "set _ok_MethodTable($cl,$methodname)" implClass] } {
	# Method was not known to that class
	lappend _ok_MethodTable($cl) $methodname
        set _ok_MethodTable($cl,$methodname) $classname
	eval "proc ${cl}::$methodname { args } \
              { uplevel 1 \"${classname}::$methodname\" \$args }"
      } else {
	if { [lsearch $scs $implClass] != -1 } {
	  set _ok_MethodTable($cl,$methodname) $classname
	  eval "proc ${cl}::$methodname { args } \
                { uplevel 1 \"${classname}::$methodname\" \$args }"
        }
      }
      set newdc [concat $newdc [ok_getDirectSubclasses $cl]]
    }
    set dc $newdc
  }
}

proc ok_method { classname methodname arglist body args } {
  if { $args != {} } {
    ok_error ok_method \
	"Too many parameters: $classname $methodname $arglist { ... } $args"
  }
  _ok_create_method $classname $methodname 0 $arglist $body
}

proc ok_staticmethod { classname methodname arglist body args } {
  if { $args != {} } {
    ok_error ok_staticmethod \
	"Too many parameters: $classname $methodname $arglist { ... } $args"
  }
  _ok_create_method $classname $methodname 1 $arglist $body
}

proc ok_abstractmethod { classname methodname arglist args } {
  if { $args != {} } {
    ok_error ok_abstractmethod \
	"Too many parameters: $classname $methodname $arglist $args"
  }
  ok_method $classname $methodname $arglist \
      "\n  ok_error $classname::$methodname {Calling abstract method}\n"
}

proc ok_constructor { classname arglist body args } {
  if { $args != {} } {
    ok_error ok_constructor \
         "Too many parameters: $classname $arglist { ... } $args"
  }
  ok_staticmethod $classname create $arglist $body
}

proc ok_destructor { classname arglist body args } {
  if { $args != {} } {
    ok_error ok_destructor \
	"Too many parameters: $classname $arglist { ... } $args"
  }
  ok_method $classname destroy $arglist $body
}

##############################################################################
# Function to signal an error, when a NULL-value is dereferenced

proc NULL { args } {
  global _ok_ClassTable
  set res {}
  set method [lindex $args 0]
  switch $method {
    isA     { if { [lsearch [array names _ok_ClassTable] \
		      [lindex $args 1]] > -1 } {
                set res 1
              } else {
                set res 0
              }
            }
    type    { set res NULL }
    destroy {}
    default { ok_error NULL {Trying to access NULL-Object} }
  }
  return $res
}

##############################################################################
# Function to copy several instances into a single one and destroy the others

proc ok_single_instance { args } {
  set res [lindex $args 0]
  set args [lreplace $args 0 0]
  foreach inst $args {
    Object::copy $inst $res 0
    Object::destroy $inst
  }
  return $res
}

##############################################################################

ok_class Object {} { Class "Object" STR }

ok_constructor Object { classname } {
  if { $classname == "" } { set classname Object }
  set objID [ok_newObject]
  set newObj [list $classname {}]
  upvar #0 $objID attr
  set at [ok_getAllInstanceAttributes $classname]
  set al [lindex $at 0]
  set dl [lindex $at 1]
  set i 0
  foreach a $al {
    set attr($a) [lindex $dl $i]
    incr i
  }
  set attr(Class) $classname
  eval "proc $objID { method args } { 
  global _ok_MethodTable
  if { \[catch \"set _ok_MethodTable($classname,\$method)\" res\] } {
    ok_error ${classname}::\$method \"Method is undefined\"
  } else {
    uplevel 1 \"\${res}::\$method $objID\" \$args
  }
}"
  return $objID
}

ok_destructor Object {} {
  upvar #0 $self attributes
  unset attributes
  rename $self ""
}

ok_staticmethod Object type { self } {
  if { $self == $NULL } {
    return [NULL type]
  } else {
    upvar #0 ${self}(Class) Class
    return $Class
  }
}

ok_staticmethod Object isA { self classname } {
  if { $self == $NULL } {
    set result [NULL isA $classname]
  } else {
    if { [ok_valid_object $self] } {
      upvar #0 ${self}(Class) Class
      if { $Class == $classname } {
        set result $True
      } else {
        if { [lsearch [ok_getAllSuperclasses $Class] $classname] > -1 } {
          set result $True
        } else {
          set result $False
        }
      }
    } else {
      set result $False
    }
  }
  return $result
}

ok_method Object copy { { to NULL } { reset 1 } } {
  if { $to == $NULL } {
    set newObj [Object::create $Class]
  } else {
    set newObj $to
  }
  upvar #0 $self   attr
  upvar #0 $newObj newAttr
  if { $reset } {
    unset newAttr
  }
  foreach a [array names attr] {
    set newAttr($a) $attr($a)
  }
  return $newObj
}

ok_method Object convert { newClass } {
  global _ok_ClassList
  if { [lsearch $_ok_ClassList $newClass] > -1 } {
    set at1 [ok_getAllInstanceAttributes $Class]
    set al1 [lindex $at1 0]
    set at2 [ok_getAllInstanceAttributes $newClass]
    set al2 [lindex $at2 0]
    set dl2 [lindex $at2 1]
    set il [intersect3 $al1 $al2]
    foreach a [lindex $il 0] {
      unset $a
    }
    foreach a [lindex $il 2] {
      set attr($a) [lindex $dl2 [lsearch $al2 $a]]
    }
    set attr(Class) $newClass
  } else {
    ok_error Object::convert \
      "Cannot convert to non-class $newClass!"
  }
}

ok_method Object handleMessage { msg sender info } {}

ok_method Object toStr {} {
  upvar #0 $self attr
  set res {}
  foreach a [lindex [ok_getAllInstanceAttributes $Class] 0] {
    lappend res [list $a $attr($a)]
  }
  return $res
}

ok_method Object fromStr { str } {
  upvar #0 $self attr
  foreach pair $str {
    set attr([lindex $pair 0]) [lindex $pair 1]
  }
}

ok_method Object get__referred_Objects {} {
  set result {}
  foreach a [lindex [ok_getAllAttributes $attr(Class)] 0] {
    if { [regexp "^OKID_.+" $attr($a)] } {
      lappend result $attr($a)
    }
  }
  return $result
}

  unset _tmp_knl_path
}
