#line 1 "./src/mta/mta_class.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST Projekt", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
*/
/* OBST LIBRARY MODULE */

#include "mta.h"

LOCAL sos_Bool mta_valid_type_name (const sos_Object& o,
				    const sos_String& name)
{  return INVALID(sos_Schema_module::make(o).lookup_type(name)); }

LOCAL inline smg_String mta_generate_type_name
			  (const sos_Schema_module& sm,
			   const smg_String&        basic_name)
{  return mta_generate_valid_name (basic_name,sos_Object(sm),
				   mta_valid_type_name);
}

EXPORT void mta_set_offsets_and_size (sos_Class_type ct)
// Set ct.local_size, ct.object_size,sc.super_class...local_size
// and offsets of the components.
// precondition: local_sizes of superclasses have to be set
{  T_PROC("mta_set_offsets_and_size");
   TT(mta_H, T_ENTER);

   sos_Super_class_List scl = ct.get_super_closure();
   sos_Int offset = 0;
   mta_open_for_writing (ct.container());
   agg_iterate (scl,sos_Super_class sc)
   {  sc.set_offset (offset);
      if (sc.get_super_class().operator==(ct))
      {  sos_Int	     local_size = 0;
         sos_Comp_descr_List components = ct.get_components();
         agg_iterate (components, sos_Comp_descr cd)
         {  sos_Method get_cm = cd.get_get_method();
            if (get_cm.get_defined_in().operator==(ct))
            {  cd.set_offset (local_size);
               local_size += mta_comp_size(cd);
            }
         }
         agg_iterate_end (components, cd)
         ct.set_local_size(local_size);
      }
      offset += sos_Class_type::make (sc.get_super_class()).get_local_size();
   }
   agg_iterate_end (scl, sc)
   ct.set_object_size(offset);

   TT(mta_H, T_LEAVE);
}


EXPORT sos_Class_type_List mta_using_classes (const sos_Type_descr& td)
{
   // Determines all sos_Class_types using td.
   // Only look for schemas importing the schema, in which td is declared!

   T_PROC ("mta_using_classes");
   TT (mta_H, T_ENTER);

   sos_Class_type_List result = sos_Class_type_List::create(TEMP_CONTAINER);

   sos_Schema_module td_decl = sos_Schema_module::retrieve (td.container());
   sos_Schema_module_Directory sd = sos_Schema_module::schema_dir();
   agg_iterate_association (sd, sos_String name, sos_Schema_module sm)
   {
      // First check, if sm imports td_decl. If not, no further check
      // for that schema is necessary.
      if (sm == td_decl OR VALID(mta_get_import (sm, td_decl)))
      {
	 sos_Type_descr_List types = sm.get_types();
	 agg_iterate (types, sos_Type_descr using_td)
	    if (using_td.has_type(sos_Class_type_type))
	    {  sos_Class_type ct = sos_Class_type::make(using_td);
	       if (mta_uses (ct, td) != sos_IMP_NONE)
		  result.append(ct);
	    }
	 agg_iterate_end (types, td)
      }
   }
   agg_iterate_association_end (sd, name, sm)

   TT (mta_H, T_LEAVE);
   return result;
}

EXPORT sos_Import_mode mta_uses (const sos_Class_type& ct,
				 const sos_Type_descr& td)
// sos_IMP_TYPE if td used in a Method or Component Declaration and 
//              ct is no generic instantiation
// sos_GENERIC_TYPE if td is a generic instantiation and td is the most
//              general instantiation
// sos_SUPER_CLASS if a superclass == td, or a superclass is a generic
// 	         instantiation of td 
{  
   T_PROC("mta_uses");
   TT(mta_H, T_ENTER);
   
   sos_Import_mode result = sos_IMP_NONE;
   if (ct == td)
      result = sos_IMP_TYPE;
   else
   {
      // is ct a generic instanitation of td ?
      // actual_gen_params is valid for an instantiation, 
      // but not for the generic template.
      if VALID(ct.get_actual_gen_params()) 
      {  if (ct.get_generic_class() == td)
	    result = sos_IMP_GENERIC;
	 else
	    result = sos_IMP_NONE;
      }
      sos_Type type = td.isa (sos_Type_type) ? sos_Type::make (td)
					     : sos_Type::make (NO_OBJECT);
      if (result == sos_IMP_NONE
	  AND NOT (   td.has_type (sos_Enum_type_type)
		   OR td.has_type (sos_Extern_type_type)))
      {   
	    // super classes:
	    // is a superclass == td, or a superclass is a generic
	    // instantiation of td ? In this case, td has to be the most
	    // general instantiation
	 sos_Super_class_List scl = ct.get_super_classes();
	 agg_iterate (scl, sos_Super_class sc)
	 {  sos_Type_descr sctd = sc.get_super_class();
	    if (sctd == td  OR  sctd.make_type().root() == type)
	    {   result = sos_IMP_SUPERCLASS;
		break;
	    }
	 }
	 agg_iterate_end (scl, sc)
      }

      if (result == sos_IMP_NONE)
      {  sos_Param_List crpar = ct.get_create_params();
	 if (VALID(crpar))
	    agg_iterate (crpar, sos_Param p)
	       sos_Type partype = p.get_type().make_type().root();
	       if (partype == type OR p.get_type() == td)
	       {  if (int(result) < int (sos_IMP_TYPE))
		  {  result = sos_IMP_TYPE;
		     break;
		  }
	       }
	    agg_iterate_end (crpar, p)
      }

      if (result == sos_IMP_NONE)
      {  sos_Comp_descr_List comps = ct.get_components();
	 agg_iterate (comps, sos_Comp_descr cd)
	    sos_Type_descr rtd = cd.get_get_method().get_result_type();

	    if (    (rtd == td  OR  rtd.make_type().root() == type)
		AND (int (result) < int (sos_IMP_TYPE)))
	    {  result = sos_IMP_TYPE;
	       break;
	    }
	 agg_iterate_end (comps,cd)
      }

      if (result == sos_IMP_NONE)
      {  sos_Method_List ml = ct.get_local_methods();
	 agg_iterate (ml, sos_Method m)
	    if (INVALID (m.get_comp_descr()))
	    {  sos_Type_descr rtd = m.get_result_type();

	       if (    (rtd == td  OR  rtd.make_type().root() == type)
		   AND (int (result) < int (sos_IMP_TYPE)))
	       {  result = sos_IMP_TYPE;
		  break;
	       }
     
	       sos_Param_List params = m.get_params();
	       if (VALID (params))
		  agg_iterate (params, sos_Param p)
		    sos_Type_descr ptd = p.get_type();

		    if (    (ptd == td  OR  ptd.make_type().root() == type)
			AND (int (result) < int (sos_IMP_TYPE)))
		    {  result = sos_IMP_TYPE;
		       break;
		    }
		  agg_iterate_end (params, p)
	    }
	 agg_iterate_end (ml, m)
      }
   }

   TT(mta_H, T_LEAVE);
   return result;
}

EXPORT sos_Class_type mta_create_class (const sos_Schema_module& sm)
{  
   T_PROC("mta_create_class");
   TT (mta_H, T_ENTER);

   sos_Class_type new_ct;
   err_block
      sos_Container cnt = sm.container();
      mta_open_for_writing (cnt);
      sos_String name = 
	 mta_generate_type_name (sm, "new_class").make_String (cnt);

      new_ct = sos_Class_type::create (cnt);
      new_ct.set_name (name);
      new_ct.set_is_abstract (FALSE);
      new_ct.set_root_class (new_ct);
      new_ct.set_methods (sos_Method_table::make(NO_OBJECT));

      sm.get_type_table().insert (name, new_ct);
      sm.get_types().append (new_ct);
    
      sos_Super_class_List scl = sos_Super_class_List::create (cnt);
      sos_Super_class      sc  = sos_Super_class::create (cnt);
      sc.set_create_params (sos_Expr_List::make (NO_OBJECT)) ;
      sc.set_super_class (sos_Type_descr::make (sos_Object_type));
      scl.append (sc);
      new_ct.set_super_classes (scl);
    
      mta_set_super_closure (new_ct);
      new_ct.set_friends (sos_Type_List::make (NO_OBJECT));
      new_ct.set_components (sos_Comp_descr_List::create (cnt));
      mta_init_methods (new_ct);
      mta_set_offsets_and_size (new_ct);
      mta_build_methodtable  (new_ct);
    
      mta_complete_local_methods (new_ct);
      mta_complete_components (new_ct);
   err_exception
      mta_internal_error ("mta_create_class");
      new_ct = sos_Class_type::make (NO_OBJECT);
   err_block_end 

   TT (mta_H, T_LEAVE);
   return new_ct;
}

LOCAL sos_Bool mta_derived_from (const sos_Type_descr& sub,
				 const sos_Type_descr& super)
// returns true, if ct is a superclass of td
// Special case: A generic class/a most general instantiation is a superclass
//		 of all instantiations.
{  
   T_PROC("mta_derived_from");
   TT (mta_H, T_ENTER);

   sos_Bool result     = FALSE;
   sos_Type super_type = super.make_type().root();
   sos_Type sub_type   = sub.make_type().root();
   if (sub_type == super_type  OR  sub_type.is_derived_from_some (super_type))
   {  
      if (super_type.operator==(sub_type))
      {
	 if (super.has_type (sos_Class_type_type))
         {
	    sos_Class_type super_ct = sos_Class_type::make (super);
            sos_Class_type sub_ct   = sos_Class_type::make (sub);
            result = sos_Bool(sub_ct.is_instantiation()
			      AND (   super_ct.is_generic_class()
				   OR super_ct == super_ct.get_root_class()));
         }
	 else
	    result = TRUE;
      }
      else
	 result = TRUE;
   }
   
   TT (mta_H, T_LEAVE);
   return result;
}

LOCAL sos_Class_type_List mta_compute_subclasses (const sos_Schema_module& sm, 
						  const sos_Class_type&    ct)
// computes all sos_Type_descr who derive from sto. This lists contains
// sos_Generic_instantiations and sos_Class_types.
// NOTE: generic instantiations are considered to inherit from their
// root class for this operation. Thus if ct is a generic class or a
// most general instantiation, then all instantiations of this generic
// class are contained in the resulting list.
// If the class has no subclasses, a list containing only the given class
// is returned.
// Bug: If a most general instantiation is passed, the generic class is
// contained in the list
{  T_PROC ("mta_compute_subclasses");
   TT (mta_H, T_ENTER);
   
   sos_Schema_module_Directory schema_dir = sos_Schema_module::schema_dir();
   sos_Class_type_List	result = sos_Class_type_List::create(TEMP_CONTAINER);
   agg_iterate_association(schema_dir,sos_String n,sos_Schema_module tmp_sm)
   {     
      if (tmp_sm.operator==(sm)  OR  VALID(mta_get_import (tmp_sm, sm)))
      {  sos_Type_descr_List types = tmp_sm.get_types();
         agg_iterate(types,sos_Type_descr tmp_st)
               // Don't collect the typedefs, because the
               // base class is in the list, too.
            if (tmp_st.container() == tmp_sm.container())
	    {  
	       if (tmp_st.has_type (sos_Class_type_type))
	       { 
		  if (tmp_st == ct  OR  MTA_DERIVED_FROM (tmp_st,ct))
                     result.append(sos_Class_type::make(tmp_st));
	       }
            }
         agg_iterate_end(types,tmp_st);
      }
   }
   agg_iterate_association_end(schema_dir,n,tmp_sm)

   TT (mta_H, T_LEAVE);
   return result;
}
 
LOCAL void mta_sort_subclasses(sos_Class_type_List &inhcl)
// sorts the hierachy tree in horizontal order. That means that every
// superclass of the list appears before its subclasses.
{  T_PROC ("mta_sort_subclasses");
   TT (mta_H, T_ENTER);
   
   sos_Class_type_List new_list = sos_Class_type_List::create(TEMP_CONTAINER);

      // work as long as the passed list is empty
   while (inhcl.card() != 0)
   {  for (sos_Cursor c = inhcl.open_cursor(); inhcl.is_valid(c);)
      {  sos_Bool       found         = FALSE;
         sos_Class_type desired_super = inhcl.get(c);
	 agg_iterate (inhcl, sos_Class_type desired_sub)
	    if (desired_super != desired_sub)
	    {  
	       if (MTA_DERIVED_FROM (desired_super, desired_sub))
	       {  found = TRUE;
		  inhcl.move_cursor (c, inhcl.find (desired_sub));
		  break;
	       }
            }
	 agg_iterate_end (inhcl, desired_sub)
         if (NOT found)
         {  new_list.append(desired_super);
            inhcl.remove_at(c);
         }
      }
      inhcl.close_cursor(c);
   }
   inhcl.destroy();
   inhcl = new_list;

   TT (mta_H, T_LEAVE);
} 

LOCAL sos_Object_sos_Object_Mapping get_subclass_cache()
{
   T_PROC ("get_subclass_cache");
   TT (mta_M, T_ENTER);

   static sos_Object_sos_Object_Mapping
      		cache = sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);

   TT (mta_M, T_LEAVE);
   return cache;
} 

// **************************************************************************
void _sos_Class_type::local_initialize (OBST_PARDECL(sos_Class_type))
// **************************************************************************
{
   T_PROC ("sos_Class_type::local_initialize");
   TT (mta_H, T_ENTER);
 
 // Not yet implemented (bs)
 
   TT (mta_H, T_LEAVE);
} // ** local_initialize **


// ************************************************************************
sos_Class_type_List _sos_Class_type::get_subclasses(const sos_Typed_id&_OBSThis)
// ************************************************************************
// computes all classes that inherit from ct. To save time, the result
// is cached, so repeated calls don't force another computation.
// Do NOT destroy the resulting List ! It won't be copied , but directly
// taken from the cache !!
// yields a sos_Type_descr_List because sos_Generic_instantriations are 
// returned too.
const{
   T_PROC ("sos_Class_type::get_subclasses");
   TT (mta_H, T_ENTER);

   sos_Class_type_List result;

   sos_Object_sos_Object_Mapping cache = get_subclass_cache();
   sos_Class_type ct = sos_Class_type::make(_OBSThis,this);
   if (cache.is_key (ct))
      result = sos_Class_type_List::make(cache[ct]);
   else
   {  sos_Schema_module   sm  = _sos_Schema_module::retrieve (ct.container());
      sos_Class_type_List stl = mta_compute_subclasses(sm, ct);
      mta_sort_subclasses (stl);
      cache.insert (ct, stl);
      result = stl;
   }

   TT (mta_H, T_LEAVE);
   return result;
} // *** sos_Class_type::get_subclasses ***


EXPORT void mta_reset_subclass_cache (sos_Class_type ct)
{     
   T_PROC ("mta_reset_subclass_cache");
   TT (mta_M, T_ENTER);

   sos_Object_sos_Object_Mapping cache = get_subclass_cache();
   if (cache.is_key (ct))
   {  sos_Type_descr_List tdl = sos_Type_descr_List::make(cache[ct]);
      agg_iterate (tdl, sos_Type_descr td)
      {  sos_Class_type ct_td = sos_Class_type::make (td);
	 if (cache.is_key (ct_td))
	 {  sos_Type_descr_List ct_tdl = 
	       sos_Type_descr_List::make(cache[ct_td]);
	    cache.remove (ct_td);
	    if (VALID(ct_tdl))
	       ct_tdl.destroy();
	 }
      }
      agg_iterate_end (tdl, td)

      if (VALID(tdl))
	 tdl.destroy(); // null_list ?
      cache.remove (ct);
   }

   TT (mta_M, T_LEAVE);
}

// **************************************************************************
void _sos_Class_type::insert_friend (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Class_type) new_friend)
// **************************************************************************
const{  
   T_PROC ("sos_Class_type::insert_friend");
   TT (mta_H, T_ENTER);

   sos_Class_type ct  = sos_Class_type::make(_OBSThis,this);
   sos_Container  cnt = ct.container();
#ifdef OBST_PROT_ALL
   sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD(cnt, ct));

   if (VALID(cmod)) // genereate protocol information only if needed
      cmod.add_friend (new_friend);
#endif
   sos_Type_List friends = ct.get_friends();
   if (INVALID(friends))
      ct.set_friends (friends = sos_Type_List::create (cnt));
   if (NOT friends.find (new_friend.make_type()))
   {  mta_open_for_writing (cnt);
      friends.append (new_friend.make_type());
   }

   TT (mta_H, T_LEAVE);
} // *** sos_Class_type:insert_friend ***
 
 
// **************************************************************************
void _sos_Class_type::remove_friend (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Class_type) friend_to_remove)
// **************************************************************************
const{  T_PROC ("sos_Class_type;:remove_friend");
   TT (mta_H, T_ENTER);
   
   sos_Class_type ct     = sos_Class_type::make(_OBSThis,this);
   sos_Container  cnt    = ct.container();
   sos_Type_List friends = ct.get_friends();
#ifdef OBST_PROT_ALL
   sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD(cnt, ct));

   if (VALID(cmod)) // generate protocol information only if needed
      cmod.del_friend(friend_to_remove);
#endif
   if (VALID (friends))
      if (friends.find (friend_to_remove.make_type()))
      {  mta_open_for_writing (cnt);
         mta_remove_from_list (friends, friend_to_remove.make_type());
         if (friends.card() == 0)
         {  ct.set_friends (sos_Type_List::make (NO_OBJECT));
#ifdef OBST_PROT_ALL
	    sos_ModifHistory::trash(friends);
#else
	    friends.destroy();
#endif
         }
      }
   
   TT (mta_H, T_LEAVE);
} // *** sos_Class_type:remove_friend ***


LOCAL sos_Bool mta_create_param_used (sos_String name, sos_Class_type ct)
{
   // Determines, if the create-parameter name is used in class ct.
   // It may be used as init expression for components, as default parameter
   // for methods and for initialization of superclasses.
   T_PROC ("mta_create_param_used")
   TT (mta_M, T_ENTER);

   sos_Bool	       used = FALSE;
   sos_Comp_descr_List cl   = ct.get_components();

   agg_iterate (cl, sos_Comp_descr cd)
      sos_Expr ie = cd.get_init_expr();
      if (VALID(ie) AND ie.isa (sos_Identifier_type))
      {
	 sos_Identifier id = sos_Identifier::make (ie);
	 if (id.get_id().equal (name))
	 {
	    used = TRUE;
	    break;
         }
      }
   agg_iterate_end (cl, cd) 
   // Now do the check for initialization of superclasses, if not used yet:
   if (NOT used)
   {
      sos_Super_class_List scl = ct.get_super_classes();
      agg_iterate (scl, sos_Super_class sc)
	 sos_Expr_List acpl = sc.get_create_params();
	 if (VALID(acpl))
	 {
	    agg_iterate (acpl, sos_Expr cp)
	       if (cp.isa (sos_Identifier_type))
	       {
		  sos_Identifier id = sos_Identifier::make (cp);
		  if (id.get_id().equal (name))
		  {
		     used = TRUE;
		     break;
		  }
	       }
	    agg_iterate_end (acpl, cp)
	    if (used)
	       break;
	 }
      agg_iterate_end (scl, sc)
   }

   TT (mta_M, T_LEAVE);
   return used;
}

EXPORT sos_Expr mta_get_default_from_param (const sos_Param& p,
					    const sos_Container& cnt)
{
   // Constructs a default for a parameter, even if this doesn't have
   // a default expression. See below, how it is obtained.
   // The constructed expression is placed into the container cnt.
   // If the type of the parameter is NO_OBJECT, then this is an error,
   // which is announced by returning NO_OBJECT.

   T_PROC ("mta_get_default_from_param");
   TT (mta_M, T_ENTER);

   sos_Expr actual_value;

   if (VALID(p) AND VALID (p.get_type()))
   {
      if (VALID(p.get_default_expr()))
	 actual_value = mta_copy (p.get_default_expr(), cnt);
      else // construct a default expression for p
      {
	 /* Default values are obtained in the following way:
	    For types being not scalar			NO_OBJECT
	    For types being an enum (NOTE: enums are scalar)
							   1st literal
	    For types being no enum, but scalar		conversion of 0
	 */
	 sos_Type_descr td = p.get_type();
	 if (NOT td.make_type().is_scalar())
	 {
	    sos_Identifier actual_id = sos_Identifier::create (cnt);
	    actual_id.set_id (sos_String::create (cnt, "NO_OBJECT"));
	    actual_value = actual_id;
	 }
	 else  // type was not scalar
	 {
	    if (td.isa (sos_Enum_type_type))
	    {
	       sos_Enum_type  et	= sos_Enum_type::make (td);
	       sos_Identifier actual_id = sos_Identifier::create (cnt);
	       actual_id.set_id (
		      sos_String::clone (et.get_literals().get_nth(1), cnt));
	       actual_value = actual_id;
	    }
	    else
	       actual_value = sos_Int_expr::create(cnt); // default is 0
	 }
      }
   }
   else // never return NO_OBJECT, hence create 0 expression:
      actual_value = sos_Int_expr::create(cnt);

   TT (mta_M, T_LEAVE);
   return actual_value;
}

LOCAL void mta_insert_as_actual_create_param (const sos_Expr_List&  acpl,
					      const sos_Param_List& fcpl,
					      const sos_Param&	    p,
					      sos_Int		    pos)
{
   // The formal create parameter p is converted into its default expression.
   // This expression is inserted into the actual create parameter list
   // at the specified position. If the list was shorter by two than the
   // insert position, then this came due to default parameters. The
   // gap has to be filled with the default expressions obtained from
   // the formal create parameter list of the associated class.

   T_PROC ("mta_insert_as_actual_create_param");
   TT (mta_L, T_ENTER);
   
   sos_Container cnt = acpl.container();
   mta_open_for_writing (cnt);

   if (acpl.card() < pos-1) // GAP, fill with default expressions!
      for (sos_Int i = acpl.card()+1;  i < pos;  i++)
	 acpl.append (fcpl.get_nth(i).get_default_expr());

   acpl.insert (pos, mta_get_default_from_param (p, cnt));

   TT (mta_L, T_LEAVE);
}

// **************************************************************************
void _sos_Class_type::insert_create_param (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_String)     name,OBST_PARDECL(
					  sos_Type_descr) t,OBST_PARDECL(
					  sos_Expr)       def, sos_Int no)
// *************************************************************************
const{  // NOTE: is_ref is set to FALSE, since it is not specified by parameter.
   // If no is greater than the length of the current create parameter list
   // incremented by one, then the parameter is appended at the end.

   T_PROC ("sos_Class_type::insert_create_param");
   TT (mta_H, T_ENTER);

   sos_Param_List fcpl = get_create_params(_OBSThis);
   sos_Container  cnt  = _OBSThis.container();
   
   mta_open_for_writing (_OBSThis.container());
   if (INVALID(fcpl)) // class had no create parameters before
   {
      fcpl = sos_Param_List::create (_OBSThis.container());
      set_create_params(_OBSThis,fcpl);
   }
   if (no > fcpl.card()+1)
      no = fcpl.card()+1;

   sos_Param p = sos_Param::create (_OBSThis.container());
   p.set_name (sos_String::clone (name, _OBSThis.container()));
   p.set_type(t);
   p.set_is_ref(FALSE);
   p.set_default_expr(VALID(def) ? mta_copy (def, _OBSThis.container())
		      		 : def);
   fcpl.insert(no,p);

#ifdef OBST_PROT_ALL
   sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD(cnt, sos_Class_type::make(_OBSThis,this)));
   if (VALID(cmod)) // genereate protocol information only if needed
      cmod.add_create_par (p, no);
#endif

   // Now insert the parameter in the superclass init lists of all
   // subclasses. Use the conversion of NO_OBJECT as default value.
   // NOTE: self occurs in the super_closure of every subclass, but
   //       in the super_classes only of the direct subclasses and the
   //       classes providing an explicit initialization for self.
   sos_Class_type_List subcl = get_subclasses(_OBSThis);
   agg_iterate (subcl, sos_Class_type ct)
      mta_open_for_writing (ct.container());
      sos_Super_class_List ct_scl = ct.get_super_closure();
      sos_Super_class      sc     = MTA_SUPER_CLASS_IN_LIST (ct_scl, sos_Class_type::make(_OBSThis,this));
      if (INVALID(sc))
	 mta_error (err_SYS, err_MTA_INTERNAL, ct.get_name());

      sos_Expr_List cpl = sc.get_create_params();
      if INVALID(cpl) // class had no create parameters up to now
      {
	 cpl = sos_Expr_List::create (ct.container());
	 sc.set_create_params(cpl);
      }
      mta_insert_as_actual_create_param (cpl, fcpl, p, no);

      ct_scl = ct.get_super_classes();
      sc     = MTA_SUPER_CLASS_IN_LIST (ct_scl, sos_Class_type::make(_OBSThis,this));
      if (VALID(sc))
      {
	 sos_Expr_List cpl = sc.get_create_params();
	 if (INVALID(cpl)) // class had no create parameters up to now
	 {
	    cpl = sos_Expr_List::create (ct.container());
	    sc.set_create_params(cpl);
	 }
	 mta_insert_as_actual_create_param (cpl, fcpl, p, no);
      }
   agg_iterate_end (subcl, ct)

   TT (mta_H, T_LEAVE);
} // *** sos_Class_type::insert_create_param ***

// *************************************************************************
void _sos_Class_type::remove_create_param (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_String) name)
// *************************************************************************
const{  
   T_PROC ("sos_Class_type::remove_create_param");
   TT (mta_H, T_ENTER);
   
   sos_Param_List cpl	      = get_create_params(_OBSThis);
   sos_Bool       found	      = FALSE;
   sos_Bool	  was_only_cp = FALSE;
   sos_Param	  p;
   Index	  pos;
   sos_Container  cnt	      = _OBSThis.container();

   sos_Cursor c = cpl.open_cursor();
   if (cpl.is_valid(c))
   do
   {
      p = cpl.get (c);
      if (p.get_name().equal (name))
      {  found=TRUE;
	 break;
      }
   } while (cpl.to_succ (c));
   if (found)
   {  // Remove only, if the param is not used in this class.
      if (mta_create_param_used (p.get_name(), sos_Class_type::make(_OBSThis,this)))
	 mta_error (err_SYS, err_MTA_CREATE_PARAM_USED, p.get_name());
      mta_open_for_writing (_OBSThis.container());
#ifdef OBST_PROT_ALL
      sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD (cnt, sos_Class_type::make(_OBSThis,this)));
      if (VALID (cmod))
	 cmod.del_create_par (p);
#endif
      pos = cpl.current_pos (c);
      cpl.remove_at (c);
      if (cpl.card() == 0)
      {  // Was only cp, destroy the whole list and replace by NO_OBJECT
	 was_only_cp = TRUE;
#ifdef OBST_PROT_ALL
	 sos_ModifHistory::trash(cpl);
#else
	 cpl.destroy();
#endif
	 set_create_params(_OBSThis,sos_Param_List::make(NO_OBJECT));
      }

      // Now iterate over all subclasses and remove the parameter from
      // superclass init lists.
      sos_Class_type_List sub_cl = get_subclasses(_OBSThis);
      agg_iterate (sub_cl, sos_Class_type ct)
	 mta_open_for_writing(ct.container());

	 sos_Super_class_List scl = ct.get_super_closure();
	 sos_Super_class      sc  = MTA_SUPER_CLASS_IN_LIST (scl, sos_Class_type::make(_OBSThis,this));
	 if (INVALID(sc))
	    mta_error (err_SYS, err_MTA_INTERNAL, ct.get_name());
	 // NOTE: default values are inserted in the super_closure
	 //	  during its construction,
	 //       if no other value was provided. This is not done for
	 //	  the super_classes. Hence the create_params in super_classes
	 //	  may be NO_OBJECT, while that in super_closure is always valid
	 //	  for a class with at least one create parameter.
	 sos_Expr_List sub_cpl = sc.get_create_params();
	 if (pos <= sc.get_create_params().card())
	 {
	    sub_cpl.remove (pos);
	    if (was_only_cp)
	    {  // empty cp-lists must be destroyed!
#ifdef OBST_PROT_ALL
	       sos_ModifHistory::trash(sub_cpl);
#else
	       sub_cpl.destroy();
#endif
	       sc.set_create_params(sos_Expr_List::make(NO_OBJECT));
	    }
	 }
	 scl = ct.get_super_classes();
	 sc  = MTA_SUPER_CLASS_IN_LIST (scl, sos_Class_type::make(_OBSThis,this));
	 if (VALID(sc))
	 {
	    sub_cpl = sc.get_create_params();
	    if (VALID(sub_cpl) AND pos <= sub_cpl.card())
	    {
	       sub_cpl.remove (pos);
	       if (was_only_cp)
	       {  // empty cp-lists must be destroyed!
#ifdef OBST_PROT_ALL
		  sos_ModifHistory::trash(sub_cpl);
#else
		  sub_cpl.destroy();
#endif
		  sc.set_create_params(sos_Expr_List::make(NO_OBJECT));
	       }
	    }
	 }
      agg_iterate_end (sub_cl, ct)
   }
   else
      mta_error (err_WNG, err_MTA_NOT_FOUND, name);

   cpl.close_cursor (c);

   TT (mta_H, T_LEAVE);
} // *** sos_Class_type::remove_create_param ***


LOCAL void mta_replace_cp_in_class (const sos_Class_type& ct,
				    const sos_String&	  old_name,
				    const sos_String&	  new_name,
				    const sos_Type_descr& old_td,
				    const sos_Type_descr& new_td)
{
   /* This function replaces all occurrences of the formal create parameter
      with name old_name in the class ct by new_name if the name changed.
      Possible occurrences are as init-expression for components
      and as actual create parameters for superclasses in super_classes
      and super_closure. Note, that sos_Identifier's used in super_classes
      and super_closure may be identical.
      For all occurrences the types are checked: the new create parameter type
      must be derived from the type of the component or the type of the
      superclass' formal create parameter respectively. */

   T_PROC ("mta_replace_cp_in_class");
   TT (mta_L, T_ENTER);

   sos_Type	       old_type     = old_td.make_type();
   sos_Type	       new_type     = new_td.make_type();
   sos_Bool	       name_changed = (sos_Bool) NOT new_name.equal (old_name);
   sos_Comp_descr_List cl	    = ct.get_components();

   agg_iterate (cl, sos_Comp_descr cd)   // look at component list
      sos_Expr ie = cd.get_init_expr();
      if (VALID(ie) AND ie.isa (sos_Identifier_type))
      {
	 sos_String name = (sos_Identifier::make(ie)).get_id();
	 if (name.equal (old_name))
	 {
	    if (name_changed)
	       name.assign (new_name);
	    if (new_type != old_type)   // type changed, issue warning?
	       if (NOT new_type.is_derived_from_some (
			cd.get_get_method().get_result_type().make_type()))
		  mta_error (err_WNG, err_MTA_CREATE_PARAM_TYPE_MISMATCH,
			     cd.get_name());
	 }
      }
   agg_iterate_end (cl, cd)

   sos_Super_class_List scl = ct.get_super_classes();
   agg_iterate (scl, sos_Super_class sc) // look at super_classes list
      sos_Expr_List acpl = sc.get_create_params();
      if (VALID(acpl))
      {
	 sos_Class_type sct  =
		sos_Class_type::make(sc.get_super_class().make_type());
	 sos_Param_List fcpl = sct.get_create_params();
	 sos_Int	comp;

	 agg_iterate_double (acpl, sos_Expr e, fcpl, sos_Param p, comp)
	    if (e.isa (sos_Identifier_type))
	    {
	       sos_Identifier i = sos_Identifier::make (e);
	       if (i.get_id().equal (old_name))
	       {
		  if (name_changed)
		     i.get_id().assign (new_name);
		  if (new_type != old_type)
		     if (NOT new_type.is_derived_from_some (
			   p.get_type().make_type()))
			mta_error (err_WNG, err_MTA_CREATE_PARAM_TYPE_MISMATCH,
				   sct.get_name());
	       }
	    }
	 agg_iterate_double_end (acpl, e, fcpl, p, comp)
      }
   agg_iterate_end (scl, sc)
   // Now investigate super_closure. Note, that some identifiers
   // might already have been changed while iterating over the super_classes.
   // Here, no type check is necessary, since type correctness for
   // super_classes implies correctness for super_closure.
   // Hence, this loop must only be entered, if the parameter'a name changed.
   if (name_changed)
   {
      scl = ct.get_super_closure();
      agg_iterate (scl, sos_Super_class sc)
	 sos_Expr_List cpl = sc.get_create_params();
	 if (VALID (cpl))
	 {
	    agg_iterate (cpl, sos_Expr e)
	       if (e.isa (sos_Identifier_type))
	       {
		  sos_String id = sos_Identifier::make (e).get_id();
		  if (id.equal (old_name))
		     id.assign (new_name);
	       }
	    agg_iterate_end (cpl, e)
	 }
      agg_iterate_end (scl, sc)
   }

   TT (mta_L, T_LEAVE);
}


LOCAL sos_Type_descr mta_is_formal_create_param (sos_Expr e, sos_Class_type ct)
{
   // Checks, if e denotes a formal create parameter of class ct
   // If so, then the type of the parameter is the result, otherwise
   // NO_OBJECT.

   T_PROC ("mta_is_formal_create_param")
   TT (mta_L, T_ENTER);

   sos_Type_descr result = sos_Type_descr::make (NO_OBJECT);
   if (e.isa (sos_Identifier_type))
   {
      sos_String     name = sos_Identifier::make (e).get_id();
      sos_Param_List fcpl = ct.get_create_params();
      if (VALID (fcpl))
      {
	 agg_iterate (fcpl, sos_Param cp)
	    if (cp.get_name().equal (name))
	    {
	       result = cp.get_type();
	       break;
	    }
	 agg_iterate_end (fcpl, cp)
      }
   }
   TT (mta_L, T_LEAVE);
   return result;
}


LOCAL void mta_check_cptype_in_subclasses (const sos_Class_type& ct,
					   Index		 pos,
					   const sos_Type_descr& new_type)
{
   /* This function checks for all subclasses of ct, if the <pos>th actual
      create parameter for ct is a valid expression for the type new_type,
      if this is possible (for scalar types, expression types are difficult
      to determine).
      The validity can be checked in two cases:
      - The actual create parameter is a formal create parameter of the
	subclass. Then the type of the formal create parameter can be determined
	and checked to be derived from new_type.
      - new_type is an enum type. Then either the first case occurs, or
	the expression must be an sos_Identifier, of which the id must
	be a valid literal of the enumeration new_type.
      In case of a mismatch, an err_WNG is issued.
   */
   T_PROC ("mta_check_cptype_in_subclasses");
   TT (mta_L, T_ENTER);
   
   sos_Type	       new_t  = new_type.make_type();
   sos_Class_type_List sub_cl = ct.get_subclasses();
   agg_iterate (sub_cl, sos_Class_type sub_ct)
      sos_Super_class_List scl = sub_ct.get_super_classes();
      sos_Super_class	   sc  = MTA_SUPER_CLASS_IN_LIST (scl, ct);
      if (VALID(sc))  // sub_ct is a subclass of ct, which provides an
      {		      // initialization for ct.
	 sos_Expr e = sc.get_create_params().get_nth(pos);
	 sos_Type_descr formal_type = mta_is_formal_create_param (e, sub_ct);
	 if (VALID (formal_type))// e was identifier denoting a formal create
	 {			 // parameter of sub_ct
	    if (NOT formal_type.make_type().is_derived_from_some (new_t))
	       mta_error (err_WNG, err_MTA_CREATE_PARAM_TYPE_MISMATCH,
			  sub_ct.get_name());
	 }
	 else if (new_t.is_derived_from (sos_Enum_type_type))
	 {
	    sos_Enum_type ent = sos_Enum_type::make (new_t);
	    if (e.isa (sos_Identifier_type))
	    {
	       sos_String lit = sos_Identifier::make(e).get_id();
	       // NOTE: The following find needs literals to be
	       // based_on_equal=TRUE, which is to be supported by CFE
	       // and corresponding evolution methods in MTA.
	       if (NOT ent.get_literals().find (lit))
		  mta_error (err_WNG, err_MTA_CREATE_PARAM_TYPE_MISMATCH,
			     sub_ct.get_name());
	    }
	    else  // sos_Int expression is invalid for enum types.
	       mta_error (err_WNG, err_MTA_CREATE_PARAM_TYPE_MISMATCH,
			  sub_ct.get_name());
	 }
      }
   agg_iterate_end (sub_cl, sub_ct)

   TT (mta_L, T_LEAVE);
}

// *************************************************************************
void _sos_Class_type::modify_create_param (const sos_Typed_id&_OBSThis,sos_Int    	 pos,OBST_PARDECL(
					  sos_String)	 new_name_tmp,OBST_PARDECL(
					  sos_Type_descr) new_t,OBST_PARDECL(
					  sos_Expr)	 new_def_tmp)
// *************************************************************************
const{  
   T_PROC ("sos_Class_type::modify_create_param");
   TT (mta_H, T_ENTER);

   sos_Param_List cpl	   = get_create_params(_OBSThis);
   sos_Bool       found    = FALSE;
   sos_Type_descr old_t;
   sos_String	  new_name;
   sos_Param	  p	   = cpl.get_nth(pos);
   sos_String	  old_name = p.get_name();
   sos_Container  cnt	   = _OBSThis.container();

   // First check, if the name is not already used for a create
   // parameter of this class:
   if (INVALID(new_name_tmp) OR INVALID(old_name))
      mta_error (err_SYS, err_MTA_INVALID_NAME);

   mta_open_for_writing (_OBSThis.container());
   if (NOT new_name_tmp.equal (old_name))
   {
      agg_iterate (cpl, sos_Param p)
	 if (p.get_name().equal(new_name_tmp))
	    mta_error (err_SYS, err_MTA_NAME_IN_USE, new_name_tmp);
      agg_iterate_end (cpl, p)
      new_name = sos_String::clone(new_name_tmp, _OBSThis.container());
   }
   else
      new_name = old_name;

#ifdef OBST_PROT_ALL
   sos_ClassModif cmod = sos_ClassModif::make (MTA_GET_TYPE_MOD (cnt, sos_Class_type::make(_OBSThis,this)));
   if (VALID (cmod))
      cmod.mod_create_par (p);
#endif
   if (NOT new_name.equal (old_name)) // change name only if needed
      p.set_name (new_name);
   old_t = p.get_type();
   p.set_type (new_t);
   // change default expression only, if it changed
   if (   VALID(new_def_tmp) != VALID(p.get_default_expr())
       OR (VALID(new_def_tmp) AND NOT new_def_tmp.equal(p.get_default_expr())))
      p.set_default_expr (VALID(new_def_tmp) ?
			  mta_copy (new_def_tmp, _OBSThis.container()) :
			  new_def_tmp);

   // Now rename occurrences within this class, if the name changed.
   // Further check the type against the type of components, where
   // the modified create parameter was used as init expression,
   // and against the type of the formal create parameters of superclasses,
   // for which the changed parameter was used as actual param.
   // A further imporovement would be to check initializations done
   // by subclasses for correct type for this create parameter.

   if (NOT old_name.equal (new_name) OR old_t != new_t)
      mta_replace_cp_in_class (sos_Class_type::make(_OBSThis,this), old_name, new_name, old_t, new_t);
   if (old_t != new_t)
      mta_check_cptype_in_subclasses (sos_Class_type::make(_OBSThis,this), pos, new_t);

   TT (mta_H, T_LEAVE);

} // *** sos_Class_type::modify_create_param ***

// *************************************************************************
void _sos_Class_type::modify_superclass_create_param (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Type_descr) td,
						     sos_Int	    pos,OBST_PARDECL(
						     sos_Expr)	    e)
// *************************************************************************
const{
   T_PROC ("sos_Class_type::modify_superclass_create_param");
   TT (mta_H, T_ENTER);

   /* This method modifies the <pos>th actual create parameter of
      direct superclass td. It replaces the current expression with the
      expression e */

   sos_Expr	   exp;
   sos_Container   cnt = _OBSThis.container();
   sos_Super_class sc  = mta_super_class_in_list (get_super_classes(_OBSThis), td);

   if (INVALID (sc))
      mta_error(err_SYS, err_MTA_INVALID_SUPERCLASS, td.make_type().get_name());

   sos_Expr_List acpl = sc.get_create_params();
   if (INVALID (acpl) OR pos > acpl.card())
      mta_error (err_SYS, err_MTA_FEW_CREATE_PARAMS, td.make_type().get_name());

   mta_open_for_writing (cnt);

   if (INVALID(e)) // NO_OBJECT, then fetch a default expression
      exp = mta_get_default_from_param (sos_Class_type::make(
					   sc.get_super_class().make_type())
					     .get_create_params().get_nth(pos),
					cnt);
   else
      exp = sos_Expr::clone (e, cnt);

   acpl.set_nth (pos, exp);
   mta_build_class (sos_Class_type::make(_OBSThis,this), 
		    TRUE,  // derived
		    FALSE, // methodtable
		    TRUE); // super_closure

   TT (mta_H, T_LEAVE);
} // *** sos_Class_type::modify_superclass_create_param ***


// --------------------------------------------------------------------------
// Methods to modify generic classes
// --------------------------------------------------------------------------

LOCAL sos_Gen_param mta_copy (sos_Gen_param gp, sos_Container cnt)
{  // Copies the generic parameter gp into the container cnt.
   // The sos_String representing its name is also copied into cnt.

   T_PROC ("mta_copy");
   TT (mta_H, T_ENTER);

   sos_Gen_param new_gp = sos_Gen_param::copy (gp, cnt);
   sos_String    s      = sos_String::create (cnt);
   s.assign (gp.get_name());
   new_gp.set_name(s);

   TT (mta_H, T_LEAVE);
   return new_gp;
} 

LOCAL sos_Type_descr_List mta_copy_gen_params (sos_Type_descr_List agpl,
					       sos_Container	   cnt)
{  T_PROC ("mta_copy_gen_params");
   TT (mta_H, T_ENTER);
   
   sos_Type_descr_List new_agpl = sos_Type_descr_List::create (cnt);
   agg_iterate (agpl, sos_Type_descr td)
      if (td.has_type (sos_Class_type_type))
	 new_agpl.append (td);
      else
      if (td.has_type (sos_Gen_param_type))
         new_agpl.append(mta_copy (sos_Gen_param::make (td), cnt));
      else
	 mta_error (err_SYS, err_NOT_IMPLEMENTED,mta_get_name_declaration(td));
   agg_iterate_end (agpl, td)

   TT (mta_H, T_LEAVE);
   return new_agpl;
}

EXPORT void mta_destroy (sos_Generic_instantiation gi)
{  T_PROC ("mta_destroy (sos_Generic_instantation)");
   TT (mta_H, T_ENTER);
   
   if (VALID (gi))
   {  sos_Type_descr_List agpl = gi.get_act_gen_params();
      for (sos_Cursor c = agpl.open_cursor();  agpl.is_valid(c); )
      {  sos_Type_descr agp = agpl.get(c);
         if (agpl.has_type (sos_Gen_param_type))
         {  sos_Gen_param gp = sos_Gen_param::make (agp);
            agpl.to_succ(c);
            mta_destroy (gp);
         }
         else
         if (agp.has_type (sos_Generic_instantiation_type))
         {  sos_Generic_instantiation agi = sos_Generic_instantiation::make(agp);
            agpl.to_succ(c);
            mta_destroy (agi);
         }
         else
            agpl.to_succ(c);
      }
      agpl.close_cursor(c);
#ifdef OBST_HAVE_PROT
      sos_ModifHistory::trash(gi);
#else
      gi.destroy();
#endif
   }

   TT (mta_H, T_LEAVE);
}


// **********************************************************************
EXPORT sos_Type_descr _sos_Class_type::lookup_or_create_instantiation
		 (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Schema_module),OBST_PARDECL( sos_Type_descr_List), sos_Bool)
// **********************************************************************
const{  mta_error (err_SYS, err_NOT_IMPLEMENTED);
  sos_Class_type::make(_OBSThis,this); // to supress 'not used' warnings
  return sos_Type_descr::make (NO_OBJECT);
} // sos_Class_type_lookup_generic_instantiation
 

// *************************************************************************
EXPORT void _sos_Class_type::insert_generic_param (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_String), sos_Int)
// *************************************************************************
const{  sos_Class_type::make(_OBSThis,this); // to supress 'not used' warning
   mta_error (err_SYS, err_NOT_IMPLEMENTED);
} // *** sos_Class_type::insert_generic_param ***
 

// *************************************************************************
EXPORT void _sos_Class_type::remove_generic_param (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_String))
// *************************************************************************
const{  sos_Class_type::make(_OBSThis,this); // to supress 'not used' warnings
   mta_error (err_SYS, err_NOT_IMPLEMENTED);
} // *** sos_Class_type::remove_generic_param ***
 
// *************************************************************************
EXPORT void _sos_Class_type::modify_generic_param (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_String),OBST_PARDECL(
						  sos_String),OBST_PARDECL( sos_Type_descr))
// *************************************************************************
const{  sos_Class_type::make(_OBSThis,this); // to supress 'not used' warnings
   mta_error (err_SYS, err_NOT_IMPLEMENTED);
} // *** sos_Class_type::modify_generic_param ***
