/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module gen_front                                          Juergen Uhl (ju)
//                                                        Michael Gravenhorst
// **************************************************************************
// OBST interface generator / front end
// **************************************************************************
//
// tracing conventions: see trc_genCC.h


#include "obst_progstd.h"
#include "smg.h"
#include "genCC_err.h"
#include "trc_genCC.h"
#include "cci_use.h"
#include "mta_use.h"

#include "genCC.h"


//--------- local definitions ----------------------------------------------

LOCAL void front_class (sos_Class_type, sos_Bool);
LOCAL void front_methods (sos_Class_type, sos_Bool);


//-------- module and type handling ---------------------------------------- 

void front_module (sos_Schema_module mdl)
{
   T_PROC ("front_module");
   TT (genCC_H, T_ENTER);

   gen_inline = FALSE; // Generates buggy code for generic classes (mg) 

   sos_Super_class_List scalar_closure
      = sos_Class_type::make (_obst_knl_type(_THE_SCALAR_TYPE))
	   .get_super_closure();

   sos_Container ct = mdl.container();
   ct.open (WRITING, WAITING);

   cci_Schema_impl::make_impl (mdl);

   code_initialize();

   back_schema_start(mdl.get_name(),
                     (sos_Bool)(mdl.get_has_external_types()
                     		OR mdl.get_has_external_import()));
   
   sos_Import_List imports = mdl.get_imports();
   agg_iterate (imports, sos_Import imp)
      back_import (imp.get_module().get_name(), imp.get_mode());
   agg_iterate_end (imports, imp);

   while (back_types_start())
   {  sos_Type_descr_List type_decls = mdl.get_types();

      agg_iterate (type_decls, sos_Type_descr td)

         sos_Type     tp      = td.make_type();
         smg_String   tp_name = tp.get_name();
         sos_Typed_id tpid    = tp._typed_id();
 
	 if (td.has_type(sos_Class_type_type))
	 {  sos_Class_type ct = sos_Class_type::make(td);

            if (NOT ct.is_generic_class())
	    {  sos_Type rtp     = tp.root();
               sos_Bool is_root = (tp == rtp);
	       if (back_class_start(tp_name, tpid,
                                    rtp.get_name(), is_root,
				    ct.get_is_abstract(),
                                    ct.get_object_size()))
	          front_class(ct, is_root);
               back_class_end();
	    }
	 }
	 else if (td.has_type(sos_Enum_type_type))
	 {  sos_Enum_type et = sos_Enum_type::make(td);

	    if (back_enum_start(tp_name, tpid, et.get_object_size()))
	    {  sos_String_List sl = et.get_literals();
	       agg_iterate (sl, sos_String l)
		  back_enum_literal(l);
	       agg_iterate_end (sl, l);
               agg_iterate (scalar_closure, sos_Super_class sc);
                  back_scalar_superclass(sc.make_type().get_name());
               agg_iterate_end (scalar_closure, sc);
	       back_enum_end();
	    }
	 }
	 else if (td.has_type(sos_Extern_type_type))
         {  if (back_extern_start(tp_name, tpid))
	    {  
               agg_iterate (scalar_closure, sos_Super_class sc);
                  back_scalar_superclass(sc.make_type().get_name());
               agg_iterate_end (scalar_closure, sc);
               back_extern_end();
   	    }
	 }
	 else if (td.has_type(sos_Typedef_type_type))
	 {  sos_Typedef_type tdt = sos_Typedef_type::make(td);
	  
	    if (back_typedef_start (tdt.get_name(), tpid))
	    {  sos_Type nt = tdt.get_named_type();
	       back_typedef_end (nt, nt.get_name(), nt.is_scalar());
	    }
	 }
	 else if (td.has_type(sos_Union_type_type))
            if (back_union_start(tp_name, tpid))
	    {
               sos_Type_List uniteds = sos_Union_type::make(td).get_united();
               agg_iterate (uniteds, sos_Type tp)
                  back_union_type(tp.get_name(), tp.is_scalar());
               agg_iterate_end (uniteds, tp);
               back_union_end();	
	    }
      agg_iterate_end (type_decls, td);
   }
   back_schema_end();

   ct.close ();

   TT (genCC_H, T_LEAVE);
}


//-------- class handling ---------------------------------------------------

LOCAL void front_class (sos_Class_type ct, sos_Bool is_root)
{
   T_PROC ("front_class")
   TT (genCC_H, T_ENTER);

   sos_Super_class_List super_closure = ct.get_super_closure();
   agg_iterate (super_closure, sos_Super_class sc)
      sos_Class_type sctp = sos_Class_type::make (
			       sc.get_super_class().make_type());
      back_super_class(sctp.get_name(),
		       sctp.get_root_class().get_name(), 
		       sc.get_offset(),
		       (sos_Bool)(sctp.get_local_size() > 0),
		       sc.get_is_direct());
   agg_iterate_end (super_closure, sc);

   back_super_class_end();
  
   sos_Type_List friend_list = ct.get_friends();
   if VALID(friend_list)
      agg_iterate (friend_list, sos_Type tp);
         back_friend (tp.root().get_name());
      agg_iterate_end (friend_list, tp);

   code_class_start (ct, ct.get_create_params());

   front_methods (ct, is_root);

   back_methods_end();

   TT (genCC_H, T_LEAVE);
}


//-------- method handling --------------------------------------------------

LOCAL void front_methods (sos_Class_type ct, sos_Bool is_root)
{
   T_PROC ("front_methods");
   TT (genCC_H, T_ENTER);

   back_methods_start();
  
   code_additional_methods ();

   sos_Super_class_List super_classes = ct.get_super_classes();
   sos_Bool generic_patch_necessary = FALSE;

   if (is_root)
   {
      agg_iterate (super_classes, sos_Super_class scl)
      {  sos_Class_type sct = sos_Class_type::make(scl.get_super_class()
                                                               .make_type());
         if (sct != sct.get_root_class())
	 {  generic_patch_necessary = TRUE;
	    break;
	 }
      }
      agg_iterate_end (super_classes, scl);
   }

   sos_Method_table mt = ct.get_methods();

   agg_iterate_association (mt, sos_String mname, sos_Method_List ml)

      int	mlcard		     = (INVALID(ml) ? 0 : ml.card());
      sos_Bool	overloaded_inherited = (sos_Bool)((mlcard > 1) AND is_root);

      if (overloaded_inherited)
      {  sos_Bool local_methods          = FALSE;
	 sos_Bool inherited_methods      = FALSE;
         sos_Bool inherited_from_several = FALSE;
         sos_Bool first_inherited        = TRUE;
         sos_Type first_defined_in;

	 agg_iterate (ml, sos_Method m)
	    sos_Type defined_in = m.get_defined_in();
          
	    if (defined_in == ct  OR  defined_in.root() == ct)
	       local_methods     = TRUE;
	    else
            {
	       inherited_methods = TRUE;

               if (first_inherited)
               {
                  first_defined_in = defined_in;
                  first_inherited  = FALSE;
               }
               else
               {
                  if (defined_in == first_defined_in)
                     inherited_from_several = TRUE;
               }
            }  

	 agg_iterate_end (ml, m);

	 if (NOT ((local_methods AND inherited_methods) 
                  OR inherited_from_several))
	    overloaded_inherited = FALSE;
      }

      agg_iterate (ml, sos_Method m)
      {
	 sos_Type        result_type  = m.get_result_type().make_type();
	 sos_Type	 defined_in   = m.get_defined_in();
	 sos_Param_List  params       = m.get_params();
	 smg_String      methname     = mname;
         sos_Bool        is_local     = (sos_Bool)(defined_in == ct OR 
                                                   defined_in.root() == ct);
         sos_Bool        is_comp      = (sos_Bool) VALID(m.get_comp_descr()); 
         sos_Bool        is_abstract  = m.get_is_abstract();
         sos_Bool        is_generated = m.get_is_generated();
         sos_Bool        is_operator  = m.get_is_operator();
         sos_Bool        is_static    = m.get_is_static();
         sos_Bool        is_definite  = m.get_is_definite();
	 int	         pcard        = INVALID(params) ? 0 : params.card();
         Predef_Method   actmeth      = code_whatis(methname,
						    pcard, is_generated);
         sos_Method_kind mkind        = m.get_kind();

	 if (back_method_start (m,
				methname,
			    	result_type.get_name(),
                                is_local,
			    	is_comp,
			    	is_abstract,
			    	is_static,
			    	is_operator,
                            	is_generated,
				is_definite,
			    	mkind,
			    	actmeth,
                            	pcard))
	 {  code_method_start (m, result_type, params);
	    code_param_list();
	    back_param_end();
	    back_method_end();

            if (overloaded_inherited AND NOT is_local)
            {
               sos_Class_type sc     = m.get_defined_in().get_root_class();
               sos_Method     scm    = sc.get_methods().lookup(m);
               sos_Type       scrt   = scm.get_result_type().make_type();

               if (back_method_start(scm, scm.get_name(), scrt.get_name(),
                                     FALSE /*is_local*/,
                                     FALSE /*is_comp*/,
                                     FALSE /*is_abstract*/,
                                     FALSE /*is_static*/,
                                     is_operator,
                                     FALSE /*is_generated*/,
                                     is_definite,
                                     mkind,
                                     OVERLOADED_INHERITED,
                                     pcard))
               {
                  code_method_start(scm, scrt, scm.get_params());
                  code_init_patch(m);
                  code_param_list();
                  back_param_end();
                  back_method_end();
               }
            }                    

            if (generic_patch_necessary AND is_local AND mkind == sos_PUBLIC
                AND NOT (is_generated OR is_static))
            {  
               sos_Class_type non_root_superclass;

               agg_iterate(super_classes, sos_Super_class scl)
               {  sos_Class_type sct = sos_Class_type::make(
                                          scl.get_super_class().make_type());
                  if (sct != sct.get_root_class())
	          {  non_root_superclass = sct;
                     break;
                  }  
               }
               agg_iterate_end(super_classes, scl); 

               sos_Method_table sc_mt = non_root_superclass.get_methods();
               sos_Method       sc_m  = sc_mt.lookup(m);
               sos_Bool         sc_m_found = FALSE; 

               if VALID (sc_m)
               {  while (! sc_m_found)
                  {  sos_Method sc_m_help = sc_m.get_generated_from();
                     if VALID (sc_m_help)
                     {  sc_m_found = (sos_Bool)INVALID(sc_m_help
                                                       .get_generated_from()); 
                        if (! sc_m_found)
                           sc_m = sc_m_help;
                     }
                     else
                        sc_m_found = TRUE;
                  }

                  sos_Param_List sc_m_params = sc_m.get_params();
                  sos_Type       sc_m_result = sc_m.get_result_type()
                                                   .make_type();

                  sos_Bool params_differ = FALSE;
                  int      comp;

                  if (INVALID (params) OR INVALID (sc_m_params))
                     params_differ = params != sc_m_params; 
                  else
                  {    
                     agg_iterate_double (params,      sos_Param ps, 
                                         sc_m_params, sos_Param pm, comp)
                        if (ps.get_type() != pm.get_type())
                        {  params_differ = TRUE;
                           break;
                        }
                     agg_iterate_double_end (params,      ps, 
                                             sc_m_params, pm, comp);
                  }
 
                  if (params_differ)
                  {
                     if (back_method_start(sc_m,
                                           methname,
                                           sc_m_result.get_name(),
                                           TRUE  /* is_local */,
                                           FALSE /* is_comp */,
		 	     	           FALSE /* is_abstract */,
			    	           FALSE /* is_static */,
					   is_operator,
					   TRUE  /* is_generated */,
					   sc_m.get_is_definite(),
					   sos_PUBLIC,
					   GENERIC_PATCH))
                     {  code_method_start(sc_m, sc_m_result, sc_m_params);
                        code_param_list();
                        code_init_patch(m);
                        back_param_end();
                        back_method_end();
                     }
		  }
                  else if (sc_m_result != result_type)
                  {  
                     smg_String where = smg_String(m.get_defined_in()
                                                                .get_name())
                                      + "::"
                                      + methname;  

                     err_raise(err_USE, err_GEN_ILLEGAL_REDEFINITION, 
                               where.make_Cstring());
                  }
               }
	    }                                           
	 }
      }
      agg_iterate_end (ml, m);
   agg_iterate_association_end(mt, mname, ml);

   TT (genCC_H, T_LEAVE);
}
