/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */

// tracing conventions: see trc_dir.h

#define  OBST_IMP_STDCONST
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "_obst_config.h"
#include "obst.h"
#include "smg.h"
#include "trc_dir.h"
#include "dir_err.h"
#include "dir_obst.h"

// *************************************************************************
void sos_Object_Directory::local_initialize (sos_Object_Directory dir)
// *************************************************************************
{  T_PROC ("sos_Object_Directory::local_initialize")
   TT (dir_M, T_ENTER);

   if (VALID (dir.get_name()))
      dir.set_name (sos_String::copy (dir.get_name(), dir.container()));

   TT (dir_M, T_LEAVE);
}

// *************************************************************************
void sos_Object_Directory::local_finalize (sos_Object_Directory dir)
// *************************************************************************
{  T_PROC ("sos_Object_Directory::local_finalize")
   TT (dir_M, T_ENTER);

   if (VALID (dir.get_name()))
      dir.get_name().destroy();
   agg_iterate_association (dir, sos_String name, sos_Object elem)
      name.destroy();
   agg_iterate_association_end (dir, name, elem);

   TT (dir_M, T_LEAVE);
}


// *************************************************************************
sos_Object_Directory sos_Object_Directory::root()
// *************************************************************************
{  T_PROC ("sos_Object_Directory::root")
   TT (dir_H, T_ENTER);

   sos_Bool open = (sos_Bool) ((ROOT_CONTAINER.status() == READABLE) OR
		               (ROOT_CONTAINER.status() == WRITEABLE));

   if (NOT open) ROOT_CONTAINER.open (READING, WAITING);

   sos_Object_Directory d =
      sos_Object_Directory::make (ROOT_CONTAINER.root_object());

   if (NOT open) ROOT_CONTAINER.close ();

   TT (dir_H, T_LEAVE);

   return d;
}

// *************************************************************************
sos_Object sos_Object_Directory::lookup (sos_String path)
// *************************************************************************
{  T_PROC ("sos_Object_Directory::lookup")
   TT (dir_H, T_ENTER);

   sos_Object  result = sos_Object_Directory::root();
   smg_String  s0     = path;
   char        *s     = s0.make_Cstring (SMG_BORROW);
   char        *buf   = new char[s0.length()];
   sos_String  key    = sos_String::create (TEMP_CONTAINER);

   sos_Object_Directory dir;

   do
   {  if (*s == '/'  AND  result.is_some (sos_Object_Directory_type))
	 dir = sos_Object_Directory::make (result);
      else
      {  result = NO_OBJECT;
	 break;
      }
      if (! *(++s))
	 break;
      
      char *nptr = buf;
      while ((*nptr = *s) AND *nptr != '/')
      {  nptr ++;   s ++;
      }
      *nptr = '\0';

      if (*buf)
      {  key.assign_Cstring (buf);
	 result = dir [key];
      }
   }
   while (VALID (result) AND *s);

   delete buf;
   key.destroy();

   TT (dir_H, T_LEAVE);
   return result;
}

// *************************************************************************
void sos_Object_Directory::insert (sos_String so, sos_Object o)
// *************************************************************************
{  T_PROC ("sos_Object_Directory::insert")
   TT (dir_H, T_ENTER);

   sos_String name;
   if (self.is_key (so))
   {  sos_Cursor c = self.open_cursor (TEMP_CONTAINER);
      self.move_cursor (c, so);
      name = sos_String::make (self.get_key (c));
      self.close_cursor (c);
   }
   else
      name = sos_String::copy (so, self.container());
   self.sos_Object_sos_Object_Mapping::insert (name, o);
   TT (dir_H, T_LEAVE);
}


// *************************************************************************
void sos_Object_Directory::remove (sos_String so)
// *************************************************************************
{  T_PROC ("sos_Object_Directory::remove")
   TT (dir_H, T_ENTER);

   sos_Cursor c = self.open_cursor ();
   sos_Bool found = self.move_cursor (c, so);
   err_assert (found,"Directory::remove:name not found");

   sos_Object name = self.get_key (c);
   self.sos_Object_sos_Object_Mapping::remove (so);
   name.destroy();
   self.close_cursor (c);

   TT (dir_H, T_LEAVE);
}

static sos_Object_Directory *wd;

// *************************************************************************
void sos_Object_Directory::set_wd (sos_String path)
// *************************************************************************
{  T_PROC ("sos_Object_Directory::set_wd")
   TT (dir_H, T_ENTER);

   sos_Object d = sos_Object_Directory::lookup (path);
   
   if (INVALID (d)  OR  NOT d.isa (sos_Object_Directory_type))
   {  sos_Cstring pname = path.make_Cstring ();
      err_raise (err_SYS, err_DIR_NO_DIR, pname);
      delete pname;
   }
   else
   {  wd = new sos_Object_Directory;
     *wd = sos_Object_Directory::make (d);
   }

   TT (dir_H, T_LEAVE);
}

// *************************************************************************
void sos_Object_Directory::set_wd_from_env ()
// *************************************************************************
{  T_PROC ("sos_Object_Directory::set_wd_from_env")
   TT (dir_H, T_ENTER);

   char* e;
   if (e = obst_getenv (ENVVAR_DIR))
   {  sos_String s = sos_String::create (TEMP_CONTAINER, e);
      sos_Object_Directory::set_wd (s);
      s.destroy();
   }
   else
      err_raise (err_SYS, err_DIR_NO_DIR, NULL, FALSE);

   TT (dir_H, T_LEAVE);
}

// *************************************************************************
sos_Object_Directory sos_Object_Directory::get_wd ()
// *************************************************************************
{  T_PROC ("sos_Object_Directory::get_wd")
   TT (dir_H, T_ENTER);

   sos_Object_Directory result;

   if (wd)
      result = *wd;
   else
   {  err_raise (err_SYS, err_DIR_NO_WD, NULL, FALSE);;
      result = sos_Object_Directory::make (NO_OBJECT);
   }

   TT (dir_H, T_LEAVE);
   return result;
}
