/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */

#include "agg.h"

// **************************************************************************
// Module List              20/10/92                  Jochen Alt (ja)
//                                   modified 19.2.94 Bernhard Schiefer (bs)
//				     read_node replaced by read_prev_next
//				     and read_node_entry
// **************************************************************************
// implements methods of classes: List
// **************************************************************************

// Die Knoten werden direkt ueber psm abgespeichert, um bei einheitlicher
// Typinformation bzw. einheitlichem Container viiiiel Platz zu sparen. 
// Da eventuell der Container und der Typ nicht abgespeichert werden muss,
// kann dadurch pro Knoten 12 Byte gespart werden. 
// 
// Es wird fuer den Eintrag ein default Typ und ein default
// Container abgespeichert, beide werden beim Eintragen des ersten Elementes 
// gesetzt. Entspricht ein folgender Eintrag einem solchen Defaultwert,
// wird nur das unbedingt notwendige gespeichert. In der Standardanwendung,
// in der der Eintrag immer denselben Typ hat und im selben Container liegt,
// braucht ein Knoten dann 16 Bytes 
// 
// 
// Ein Knoten wird in folgendem Format abgespeichert:
//   sos_Offset : Verweis auf Vorgaengerknoten
//   sos_Offset : Verweis auf Nachfolgerknoten
//   sos_Int    : Typinformation des Knotens und Groesse des Knotens
//      evt. sos_Typed_id Typinformation des Eintrages
//      evt. sos_Container Container des Eintrages
//   sos_Offset : Offset des Eintrages
// 
//
// Der Typ eines Knotens wird durch die folgenden Konstanten identifiziert,
// die die Wertigkeit des Bits angeben, welches bestimmt, ob ein jeweiliges
// Datum abgespeichert wurde. Beispiel: Ist in einem Knoten
// der Typ abgespeichert, weil er dem Standard-typ nicht entspricht, 
// so ist das ENTRY_TYPE_DIFF.te Bit gesetzt.
// Die Groesse des Knotens ist im 2.Byte abgespeichert (Koennte man sich
// aus dem Type bequem berechnen, den Platz im sos_Int kriegt man jedoch 
// geschenkt).

const sos_Offset NO_OFFSET=0;
#define ENTRY_CNT_DIFF 1
#define ENTRY_TYPE_DIFF 2
// Die Groesse befindet sich im 2.ten Byte der Typinformation eines Knotens
#define SIZE_POS 256  

// Erzeuge den Typ Node_type mit NODE_TYPE_SIZE und den entsprechenden bcopies
#define Node_type sos_Int
#define NODE_TYPE_SIZE SOS_INT_SIZE
#define bcopy_from_Node_type bcopy_from_sos_Int
#define bcopy_to_Node_type bcopy_to_sos_Int


LOCAL const int PREV_OFFSET		= 0;
LOCAL const int NEXT_OFFSET		= SOS_OFFSET_SIZE;
LOCAL const int NODE_TYPE_OFFSET	= 2*SOS_OFFSET_SIZE;
LOCAL const int ENTRY_OFFSET		= 2*SOS_OFFSET_SIZE + NODE_TYPE_SIZE;
LOCAL const int MAX_NODE_SIZE		= ENTRY_OFFSET + 
			  			(SOS_CONTAINER_SIZE + 
			   			 SOS_OFFSET_SIZE +
			   			 SOS_ID_SIZE);

// **************************************************************************
void sos_Object_List::change_list_cursor (sos_Bool new_list_cursor)
// **************************************************************************
{  self.set_list_cursor (new_list_cursor);
}

inline LOCAL void write_offset (sos_Offset offset, sos_Container cnt,
                                sos_Offset contents)
{
   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;
 
   bcopy_from_sos_Offset (&contents, &u.mem);
 
   cnt.write (offset, SOS_OFFSET_SIZE, &u.mem);
}


// **************************************************************************
void sos_Object_List::set_next (sos_Offset offset, sos_Offset next)
// **************************************************************************
// Set the field with the successor to 'next'
{
   if (offset == NO_OFFSET)
      self.set_first (next);
   else
      write_offset (offset+NEXT_OFFSET, self.container(), next);
}

// **************************************************************************
void sos_Object_List::set_prev (sos_Offset offset, sos_Offset prev)
// **************************************************************************
{
   if (offset == NO_OFFSET)
      self.set_last (prev);
   else 
      write_offset (offset+PREV_OFFSET, self.container(), prev);
}
 
 

// ************************************************************************** 
sos_Offset sos_Object_List::create_node (sos_Object entry, 
					 sos_Offset prev, sos_Offset next)
// ************************************************************************** 
// create a node with contents (entry, prev, next) and return the offset
{  
   T_PROC("List::create_node")
   TT(agg_L, T_ENTER; TOBJ(entry); TI(prev); TI(next))

   sos_Container cnt        = self.container();
   sos_Container entry_cnt  = entry.container();
   sos_Id        entry_type = entry._type_id();
   sos_Id        def_type;
   sos_Container def_cnt;

   if (self.get_cardinality() == 0)
   {  self.set_default_type (def_type = entry_type);
      self.set_default_container (def_cnt = entry_cnt);
   }
   else
   {
      def_type = self.get_default_type();
      def_cnt  = self.get_default_container();
   }
   
   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;

   char* ptr = u.mem + ENTRY_OFFSET;
   Node_type node_type = 0;

   if (entry_type != def_type)
   {  bcopy_from_sos_Id (&entry_type, ptr);
      ptr += SOS_ID_SIZE;
      node_type += ENTRY_TYPE_DIFF;
   }
   if (entry_cnt != def_cnt)
   {  bcopy_from_sos_Container (&entry_cnt, ptr);
      ptr += SOS_CONTAINER_SIZE;
      node_type += ENTRY_CNT_DIFF;
   }
   sos_Offset entry_offset = entry.offset();
   bcopy_from_sos_Offset (&entry_offset, ptr);
   ptr += SOS_OFFSET_SIZE;

   sos_Int size = ptr - u.mem;
   node_type += size*SIZE_POS;

   ptr = u.mem + PREV_OFFSET;
   bcopy_from_sos_Offset (&prev, ptr);
   ptr = u.mem + NEXT_OFFSET;
   bcopy_from_sos_Offset (&next, ptr);
   ptr = u.mem + NODE_TYPE_OFFSET;
   bcopy_from_Node_type (&node_type, ptr);

   sos_Offset offset = cnt.allocate (size);
   cnt.write (offset, size, &u);

   self.set_size (self.get_size() + size);
   self.set_cardinality (self.get_cardinality() + 1);
   
   self.set_prev (next, offset);
   self.set_next (prev, offset);

   TT(agg_L, T_LEAVE; TI(offset); TI(size))
   return offset;
}


inline LOCAL void get_size_and_node_type (
		sos_Offset    offset,
		sos_Container cnt,
		Node_type     &node_type,
		sos_Int       &size)
// 19.2.94 (bs)
{
   TT(agg_L, TI(offset); TI(int(cnt)))

   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;

   cnt.read (offset + NODE_TYPE_OFFSET, NODE_TYPE_SIZE, &u);

   bcopy_to_Node_type (&node_type, u.mem);
   size = node_type/SIZE_POS;

   TT(agg_L, TI(size))
}

// ************************************************************************** 
void sos_Object_List::remove_node(sos_Offset offset,
				  sos_Offset pred,
                                  sos_Offset next)
// ************************************************************************** 
// remove a node at position - the size is determined inside !
{
   T_PROC("List:sos_Object_List::remove_node")
   TT(agg_L, T_ENTER; TI(offset); TI(pred); TI(next))

   sos_Container cnt = self.container();
   Node_type node_type;
   sos_Int size;
   get_size_and_node_type (offset, cnt, node_type, size);

   cnt.deallocate (offset, size);
   self.set_size (self.get_size() - size);
   self.set_cardinality  (self.get_cardinality() - 1);

   self.set_next (pred, next);
   self.set_prev (next, pred);

   TT (agg_L, T_LEAVE)
}

LOCAL sos_Object read_node_entry (
		     sos_Offset    offset,
                     sos_Container cnt,
                     const sos_Id& def_type,
                     sos_Container def_cnt)
// read a node entry at position offset
// 19.2.94 (bs)
{ 
   T_PROC("List:read_node_entry")
   TT(agg_L, T_ENTER; TI(offset); TI((int)cnt); TI((int)def_cnt))

   Node_type node_type;
   int size;
   get_size_and_node_type (offset, cnt, node_type, size);

   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;
   char* ptr = u.mem;

   cnt.read (offset+ENTRY_OFFSET,
	     size-ENTRY_OFFSET, &u);

   sos_Id entry_type = def_type;
   if (node_type BITAND ENTRY_TYPE_DIFF)
   {  bcopy_to_sos_Id (&entry_type, ptr);  
      ptr += SOS_ID_SIZE;
   }
   sos_Container entry_cnt = def_cnt;
   if (node_type BITAND ENTRY_CNT_DIFF)
   {  bcopy_to_sos_Container (&entry_cnt, ptr);  
      ptr += SOS_CONTAINER_SIZE;
   }
   sos_Offset entry_offset;
   bcopy_to_sos_Offset (&entry_offset, ptr);
 
   sos_Typed_id entry_tpid = 
      sos_Typed_id::make(sos_Id::make(entry_cnt,entry_offset),entry_type);
   sos_Object entry = sos_Object::make (entry_tpid);

   TT (agg_L, T_LEAVE; TOBJ(entry))
   return entry;
}

LOCAL void read_prev_next (
		     sos_Offset    offset,
                     sos_Container cnt,
                     sos_Offset    &prev,
                     sos_Offset    &next)
// read prev and next of a node at position offset
// 19.2.94 (bs)
{
   T_PROC("List:read_prev_next")
   TT(agg_L, T_ENTER; TI(offset); TI((int)cnt))

   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;
   char* ptr = u.mem;

   cnt.read (offset, 2*SOS_OFFSET_SIZE, &u);

   bcopy_to_sos_Offset (&prev, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_to_sos_Offset (&next, ptr);

   TT (agg_L, T_LEAVE; TI(prev); TI(next))
}

// ************************************************************************** 
void sos_Object_List::local_initialize (sos_Object_List l)
// ************************************************************************** 
{ 
   T_PROC("List::local_initialize");
   TT(agg_H, T_ENTER; TOBJ(l));

   l.set_cardinality (0);
   l.set_size (0);
   l.set_first (NO_OFFSET);
   l.set_last (NO_OFFSET);

   TT(agg_H,T_LEAVE)
}


// ************************************************************************** 
void sos_Object_List::local_finalize (sos_Object_List l)
// ************************************************************************** 
{  T_PROC("List::local_finalize");
   TT (agg_H, T_ENTER; TOBJ(l));

   l.clear();

   TT (agg_H, T_LEAVE)
}


// ************************************************************************** 
void sos_Object_List::append (sos_Object o)
// ************************************************************************** 
{  T_PROC("List::append");
   TT(agg_H, T_ENTER; TOBJ(o));

   self.create_node (o, self.get_last(), NO_OFFSET);

   TT(agg_H,T_LEAVE)   
}

// ************************************************************************** 
void sos_Object_List::insert (sos_Int pos, sos_Object o)
// ************************************************************************** 
{  T_PROC("List::insert");
   TT(agg_H, T_ENTER; TI(pos); TOBJ(o))

   sos_Int crd = self.get_cardinality();

   err_assert ((0 <= pos) AND (pos <= crd+1),
               "List::insert position invalid");

      // pos == 0 forces the list to append the entity
   if (pos == 0)
      pos = crd + 1;

   sos_Container cnt = self.container();
   sos_Offset offset = NO_OFFSET;

   if (pos <=crd/2)
   {
      sos_Offset prev = NO_OFFSET;
      sos_Offset next = self.get_first();
      for (sos_Int i = 1;(i<pos) AND (next != NO_OFFSET); i++)
      {  
         offset = next;
         read_prev_next (offset, cnt, prev, next);
      }
      self.create_node (o, offset, next);
   }
   else
   {
      sos_Offset next   = NO_OFFSET;
      sos_Offset prev   = self.get_last();
      for (sos_Int i = crd;(i>=pos) AND (prev != NO_OFFSET); i--)
      {
         offset = prev;
         read_prev_next (offset, cnt, prev, next);
      }
      self.create_node (o, prev, offset);
   }

   TT(agg_H,T_LEAVE)   
}

// ************************************************************************** 
void sos_Object_List::remove (sos_Int pos)
// ************************************************************************** 
{  T_PROC("List::remove");
   TT(agg_H, T_ENTER; TI(pos))

   sos_Int crd = self.get_cardinality();

   err_assert ((0  <  pos) AND (pos  <=  crd),
               "List::remove pos invalid");

   sos_Container cnt = self.container();
   sos_Offset offset = NO_OFFSET;

   if (pos <=crd/2)
   {
      sos_Offset prev = NO_OFFSET;
      sos_Offset next = self.get_first();
 
      for (sos_Int i = 0;(i<pos); i++)
      {
         offset = next;
         read_prev_next (offset, cnt, prev, next);
      }
      self.remove_node (offset, prev, next);
   }
   else
   {  sos_Offset prev = self.get_last();
      sos_Offset next = NO_OFFSET;
  
      for (sos_Int i = crd;(i>=pos); i--)
      {  
         offset = prev;
         read_prev_next (offset, cnt, prev, next);
      }
      self.remove_node (offset, prev, next);
   }
 
   TT(agg_H,T_LEAVE)
}

// ************************************************************************** 
sos_Object sos_Object_List::get_nth(sos_Int pos)
// ************************************************************************** 
{  T_PROC("List::get_nth")
   TT(agg_H, T_ENTER; TI(pos))

   err_assert ((0 < pos) AND (pos <= self.get_cardinality()),
	       "List::get_nth position invalid");

   sos_Offset prev   = NO_OFFSET;
   sos_Offset next   = self.get_first();;
   sos_Offset offset = NO_OFFSET;
   sos_Object entry  = NO_OBJECT;

   sos_Container cnt = self.container();

   for (sos_Int i = 0;(i<pos) AND (next != NO_OFFSET); i++)
   {  
      offset = next;
      read_prev_next (offset, cnt, prev, next);
   }
   entry = read_node_entry (offset, cnt,
                            self.get_default_type(), self.get_default_container());
   TT(agg_H,T_LEAVE)
   return entry;
}

// ************************************************************************** 
void sos_Object_List::set_nth (Index pos, sos_Object o)
// ************************************************************************** 
{  T_PROC("List::set_nth")
   TT(agg_H, T_ENTER; TI(pos))

   err_assert ((0 < pos) AND (pos <= self.get_cardinality()),
	       "List::set_nth position invalid");

   sos_Cursor c = self.open_cursor();
   self.move_cursor (c, pos);
   self.set (c,o);
   self.close_cursor(c);

   TT(agg_H,T_LEAVE)
}

// ************************************************************************** 
Index sos_Object_List::find(sos_Object o)
// ************************************************************************** 
{  T_PROC("List::find(sos_Object)");
   TT(agg_H, T_ENTER; TOBJ(o));

   Index result = 0; // assume not found
   sos_Bool boe = self.get_based_on_equal();
 
   sos_Cursor c = self.open_cursor();
   for (sos_Bool valid = self.is_valid(c); valid; valid = self.to_succ(c))
   {
      sos_Object x = self.get(c);
      if (agg_same_entity (o, x, boe, EQ_WEAK))
      {  result = self.current_pos(c);
         break;
      }
   }
   self.close_cursor(c);

   TT(agg_H,T_LEAVE)

   return result;
}


// ************************************************************************** 
sos_Object sos_Object_List::find_or_append (sos_Object o)
// ************************************************************************** 
{  T_PROC ("List::find_or_append");
   TT( agg_H, T_ENTER; TOBJ(o));
   
   Index      i = self.find(o);
   sos_Object result;
   if (i == 0) // not found, then append
   {  self.append(o);
      result = o;
   }
   else
      result = self.get_nth(i);
  

   TT(agg_H,T_LEAVE)
   return result;
}

// **************************************************************************
void sos_Object_List::operator+= (sos_Object_List alist)
// **************************************************************************
{
   T_PROC ("List::operator+=");
   TT (agg_H, T_ENTER; TOBJ(alist))
 
   agg_iterate (alist, sos_Object o)
      self.append (o);
   agg_iterate_end(alist,o)
 
   TT (agg_H, T_LEAVE)
} // ** operator+= **
 

#define valid_list_cursor(node) (sos_Bool)(node.get_offset()!=NO_OFFSET)

// LOCAL inline sos_Bool valid_list_cursor (sos_List_node node)
// {  return (sos_Bool)(node.get_offset() != NO_OFFSET); }

// **************************************************************************
sos_Bool sos_Object_List::is_valid (sos_Cursor c)
// **************************************************************************
{
   sos_List_node node = sos_List_node::make (c.get_current());
   return valid_list_cursor (node);
}

// **************************************************************************
Index sos_Object_List::current_pos (sos_Cursor c)
// **************************************************************************
{
   T_PROC ("List::current_pos");
   TT (agg_H, T_ENTER)
 
   Index result = 0;

   sos_List_node node = sos_List_node::make(c.get_current());
   sos_Offset offset  = node.get_offset();
 
   err_assert (valid_list_cursor(node) AND c.defined_for(self),
               "List::current_pos cursor invalid");

   sos_Container cnt = self.container();

   for (;offset != NO_OFFSET;result++)
   {  
      sos_Offset prev, next;
      read_prev_next (offset, cnt, prev, next);
      offset = prev;
   }

   TT (agg_H, T_LEAVE; TI (result))
   return result;
} 

// **************************************************************************
sos_Bool sos_Object_List::move_cursor (sos_Cursor c, Index pos)
// **************************************************************************
{
   T_PROC ("List::move_cursor")
   TT (agg_H, T_ENTER; TI (pos))

   sos_Int  crd = self.get_cardinality();
   sos_Bool result = FALSE;

   err_assert ((0 < pos) AND (pos <= crd),
	       "List::move_cursor position invalid");

   if (pos <= crd/2)
   {  result = self.to_first (c);
      if (pos > 1)
         result = self.to_succ (c,pos-1);
   }
   else
   {  result = self.to_last(c);
      if (pos < crd)
	 result = self.to_pred (c, crd-pos);
   }

   TT (agg_H, T_LEAVE; TB(result))
   return result;
}
 
 
// ************************************************************************** 
void sos_Object_List::insert_before (sos_Cursor c, sos_Object o)
// ************************************************************************** 
{  T_PROC("List::insert_before");
   TT(agg_H, T_ENTER; TOBJ(c))

   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(self),
               "List::insert_before cursor invalid");

   sos_Offset    offset = node.get_offset();
   sos_Offset prev, next;
   read_prev_next (offset, self.container(), prev, next);
   node.set_prev (self.create_node (o, prev, offset));
   node.set_entry (o);

   TT(agg_H, T_LEAVE)
}

// ************************************************************************** 
void sos_Object_List::insert_after (sos_Cursor c, sos_Object o)
// ************************************************************************** 
{  T_PROC("List::insert_after")
   TT(agg_H, T_ENTER; TOBJ(c))
 
   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(self),
               "List::insert_after cursor invalid");

   sos_Offset offset = node.get_offset();
   sos_Offset prev, next;
   read_prev_next (offset, self.container(), prev, next);
   node.set_next  (self.create_node (o, offset, next));
   node.set_entry (o);

   TT(agg_H, T_LEAVE)
}

// ************************************************************************** 
void sos_Object_List::local_assign (sos_Object_List x,
                                    sos_Object      o)
// ************************************************************************** 
{  T_PROC("List::local_assign");

   sos_Object_List y = sos_Object_List::make (o);
   x.clear();
   x.operator+= (y);

   TT(agg_H,T_LEAVE) 
}

// ************************************************************************** 
sos_Bool sos_Object_List::local_equal (sos_Object_List x,
                                       sos_Object      o,
                                       sos_Eq_kind     eq_kind) 
// ************************************************************************** 
{  sos_Bool result;
 
   sos_Object_List y = sos_Object_List::make (o);
   if (x.get_cardinality() != y.get_cardinality())
      result = FALSE;
   else
   {  result = TRUE;
      sos_Bool based_on_equal = x.get_based_on_equal();
      int  comp = 0;
      agg_iterate_double (x, sos_Object x_e, y, sos_Object y_e, comp)
      {  result = agg_same_entity (x_e, y_e, based_on_equal, eq_kind);
	 if (NOT result) break;
      }
      agg_iterate_double_end (x, x_e, y, y_e, comp);
      result = (sos_Bool)(result AND comp == 0);
   }
   
   return result;
} // ** List::local_equal **

// ************************************************************************** 
sos_Int sos_Object_List::local_hash_value (sos_Object_List x)
// ************************************************************************** 
{
   T_PROC ("List::local_hash_value");
   TT (agg_H, T_ENTER );
 
   sos_Int result;
 
   if (x.is_empty())
      result = 0;
   else
      result = x.get_nth (1).hash_value();
 
   TT (agg_H, T_LEAVE; TB(result));
 
   return result;
}

// **************************************************************************
sos_Object sos_Object_List::get (sos_Cursor c)
// **************************************************************************
{
   T_PROC ("List::get (sos_Cursor)");
   TT (agg_H, T_ENTER)
 
   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(self),
               "List::get(sos_Cursor) cursor invalid");

   sos_Object entry = NO_OBJECT;

   if (NOT node.get_list_cursor())
      entry = node.get_entry();
   else
   {  sos_Offset offset = node.get_offset();
      sos_Offset prev, next;
      entry = read_node_entry (offset, self.container(),
			       self.get_default_type(), self.get_default_container());
   }

   TT (agg_H, T_LEAVE; TOBJ(entry))
   return entry;
}

// **************************************************************************
void sos_Object_List::set (sos_Cursor c, sos_Object o)
// **************************************************************************
{
   T_PROC ("List::set")
   TT (agg_H, T_ENTER; TOBJ(o))

   sos_List_node node = sos_List_node::make (c.get_current());
 
   err_assert (valid_list_cursor(node) AND c.defined_for(self),
	       "List::set cursor invalid");

   sos_Offset offset = node.get_offset();
   sos_Offset prev, next;

   read_prev_next (offset, self.container(), prev, next);
   self.remove_node (offset, prev, next);

   offset = self.create_node (o, prev, next);

   node.set_offset (offset);
   node.set_prev   (prev);
   node.set_next   (next);
   node.set_entry  (o);
 
   TT (agg_H, T_LEAVE)
}

// ************************************************************************** 
void sos_Object_List::remove_at(sos_Cursor c)
// ************************************************************************** 
{  T_PROC("List::remove_at")
   TT(agg_H, T_ENTER)

   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(self),
               "List::remove_at cursor invalid");

   sos_Container cnt  = self.container();
   sos_Offset  offset = node.get_offset();
   sos_Offset  next,prev;

   read_prev_next (offset, cnt, prev, next);
   self.remove_node (offset,prev, next);

   if (next == NO_OFFSET)
      node.set_offset (NO_OFFSET);
   else
   {  offset = next;
      read_prev_next (offset, cnt, prev, next);
      sos_Object entry = read_node_entry (offset, cnt, self.get_default_type(),
                                          self.get_default_container());
      node.set_offset (offset);
      node.set_prev (prev);
      node.set_next (next);
      node.set_entry (entry);
   }

   TT(agg_H, T_LEAVE)
} 
 

// ************************************************************************** 
sos_Int sos_Object_List::size()
// ************************************************************************** 
{  return self.get_size() +self.sos_Object::size();
}

// **************************************************************************
void sos_Object_List::clear()
// **************************************************************************
{  T_PROC("List::clear")
   TT(agg_H, T_ENTER)
 
   for (;self.get_cardinality() > 0;)
      self.remove (1);
 
   TT(agg_H, T_LEAVE)
}

// **************************************************************************
sos_Int sos_Object_List::card ()
// **************************************************************************
{  T_PROC ("List::card")
   TT (agg_H, T_ENTER)

   sos_Int crd = self.get_cardinality();

   TT (agg_H, T_LEAVE; TI(crd));
   return crd;
}

// ************************************************************************** 
sos_Cursor sos_Object_List::open_cursor(sos_Container cnt)
// ************************************************************************** 
{  T_PROC ("List::open_cursor")
   TT( agg_H, T_ENTER)

   sos_Cursor c = sos_Cursor::create (cnt, self);
   sos_List_node node = sos_List_node::create (cnt, self.get_list_cursor());
   c.set_current (node);
   self.to_first (c);

   TT(agg_H, T_LEAVE)
   return c;
}

// ************************************************************************** 
void sos_Object_List::close_cursor(sos_Cursor c)
// ************************************************************************** 
{  T_PROC ("List::close_cursor")
   TT( agg_H, T_ENTER; TOBJ(c))

   err_assert (c.defined_for(self),
               "List::close_cursor cursor not legal for this List");

   c.get_current().destroy();
   c.destroy();

   TT(agg_H, T_LEAVE)
}

// ************************************************************************** 
sos_Cursor sos_Object_List::duplicate(sos_Cursor c, 
				      sos_Container cnt /*= TEMP_CONTAINER */)
// ************************************************************************** 
{  T_PROC ("List::duplicate")
   TT( agg_H, T_ENTER; TOBJ(c))

   err_assert (c.defined_for(self),
               "List::duplicate cursor not legal for this List");

   sos_Cursor new_c = self.open_cursor (cnt);
   new_c.get_current().assign (c.get_current());

   TT(agg_H, T_LEAVE)
   return new_c;
}


// ************************************************************************** 
sos_Bool sos_Object_List::to_first(sos_Cursor c)
// ************************************************************************** 
{
   err_assert (c.defined_for(self),
               "List::to_first cursor not legal for this List");

   sos_Object entry;
   sos_Offset offset = self.get_first();
   sos_Offset prev, next;
   sos_Container cnt = self.container();

   if (offset != NO_OFFSET)
   {
      read_prev_next (offset, cnt, prev, next);
      entry = read_node_entry (offset, cnt,
                               self.get_default_type(),
                               self.get_default_container());
   }
   sos_List_node node = sos_List_node::make (c.get_current());

   node.set_offset (offset);
   node.set_prev (prev);
   node.set_next (next);
   node.set_entry (entry);

   return valid_list_cursor (node);
}

// ************************************************************************** 
sos_Bool sos_Object_List::to_last(sos_Cursor c)
// ************************************************************************** 
{
   err_assert (c.defined_for(self),
               "List::to_last cursor not legal for this List");

   sos_Object entry;
   sos_Offset offset = self.get_last();
   sos_Offset prev, next;
   sos_Container cnt = self.container();

   if (offset != NO_OFFSET)
   {
      read_prev_next (offset, cnt, prev, next);
      entry = read_node_entry (offset, cnt,
			       self.get_default_type(),
                               self.get_default_container());
   }

   sos_List_node node = sos_List_node::make (c.get_current());

   node.set_offset (offset);
   node.set_prev (prev);
   node.set_next (next);
   node.set_entry (entry);

   return valid_list_cursor (node);
}

// ************************************************************************** 
sos_Bool sos_Object_List::to_succ (sos_Cursor c, Index steps) 
// ************************************************************************** 
{  T_PROC ("List::to_succ")
   TT( agg_H, T_ENTER; TI(steps))

   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(self),
               "List::to_succ cursor invalid");

   if (steps>0)
   {
      sos_Container cnt	= self.container();
      sos_Object entry  = NO_OBJECT;
      sos_Offset offset = NO_OFFSET;
      sos_Offset prev, next;

      if (node.get_list_cursor())
      {  next = node.get_offset();
	 steps++;
      }
      else
         next = node.get_next();
   
      for (; (steps > 0) AND (next != NO_OFFSET); steps--)
      {
         offset = next;
         read_prev_next (offset, cnt, prev, next);
      }

      if (offset != NO_OFFSET)
      {
         entry = read_node_entry (offset, cnt,
				  self.get_default_type(),
				  self.get_default_container());
         if (steps != 0)
	    offset = NO_OFFSET;
      }
   
      node.set_offset (offset);
      node.set_next   (next);
      node.set_prev   (prev);
      node.set_entry  (entry);
   }
   else
      if (steps < 0)
	 self.to_pred (c, -steps);

   TT(agg_H, T_LEAVE; TB(valid_list_cursor (node)))
   return valid_list_cursor (node);
}

// ************************************************************************** 
sos_Bool sos_Object_List::to_pred (sos_Cursor c, Index steps)
// ************************************************************************** 
{  T_PROC ("List::to_pred")
   TT( agg_H, T_ENTER; TI(steps))

   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(self),
               "List::to_pred cursor invalid");

   if (steps > 0)
   { 
      sos_Container cnt	= self.container();
      sos_Object entry  = NO_OBJECT;
      sos_Offset offset = NO_OFFSET;
      sos_Offset prev, next;

      if (node.get_list_cursor())
      {  prev = node.get_offset();
	 steps++;
      }
      else
         prev = node.get_prev();
   
      for (; (steps > 0) AND (prev != NO_OFFSET); steps--)
      {  
         offset = prev;
         read_prev_next (offset, cnt, prev, next);
      }

      if (offset != NO_OFFSET)
      {
         entry = read_node_entry (offset, cnt,
				  self.get_default_type(),
				  self.get_default_container());
         if (steps != 0)
	    offset = NO_OFFSET;
      }

      node.set_offset (offset);
      node.set_next   (next);
      node.set_prev   (prev);
      node.set_entry  (entry);
  }
  else 
     if (steps<0)
        self.to_succ (c, -steps);

   TT(agg_H, T_LEAVE; TB(valid_list_cursor (node)))
   return valid_list_cursor (node);
}
