;;; hanja-util.el --- Korean Hanja util module  -*- lexical-binding: t; -*-

;; Copyright (C) 2008-2022 Free Software Foundation, Inc.

;; Author: Jihyun Cho <jihyun.jo@gmail.com>
;; Keywords: multilingual, input method, Korean, Hanja

;; This file is part of GNU Emacs.

;; GNU Emacs is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; This file defines the Korean Hanja table and symbol table.

;;; Code:

;; How many characters to show at once.
(defconst hanja-list-width 10)

(defvar hanja-table nil
  "A char table for Hanja characters.
It maps a Hangul character to a list of the corresponding Hanja characters.
Each element of the list has the form CHAR or (CHAR . STRING)
where CHAR is a Hanja character and STRING is the meaning of that
character.  This variable is initialized by `hanja-init-load'.")

(defun hanja-init-load ()
  "Initialize `hanja-table' (which see)."
  (when (not hanja-table)
    (message "Hanja table loading...")
    (setq hanja-table (make-char-table nil))
    (mapc
     (lambda (x) (aset hanja-table (car x) (cdr x)))
     '((12593 . (12288 65281 65287 65292 65294 65295 65306 65307
		       65311 65342 65343 65344 65372 65507 12289 12290
		       183 8229 8230 168 12291 173 8213 8741 65340
		       8764 180 65374 711 728 733 730 729 184 731
		       161 191 720))
       (12594 . (198 208 294 306 319 321 216 338 222
		     358 330 230 273 240 295 305 307 312 320 322
		     248 339 223 254 359 331 329))
       (12596 . (65282 65288 65289 65339 65341 65371 65373
		       8216 8217 8220 8221 12308 12309 12296 12297 12298
		       12299 12300 12301 12302 12303 12304 12305))
       (12599 . (65291 65293 65308 65309 65310 177 215
		       247 8800 8804 8805 8734 8756 9794 9792 8736
		       8869 8978 8706 8711 8801 8786 8810 8811 8730
		       8765 8733 8757 8747 8748 8712 8715 8838 8839
		       8834 8835 8746 8745 8743 8744 65506 8658 8660
		       8704 8707 8750 8721 8719))
       (12600 . (12353 12354 12355 12356 12357 12358 12359
		       12360 12361 12362 12363 12364 12365 12366 12367
		       12368 12369 12370 12371 12372 12373 12374 12375
		       12376 12377 12378 12379 12380 12381 12382 12383
		       12384 12385 12386 12387 12388 12389 12390 12391
		       12392 12393 12394 12395 12396 12397 12398 12399
		       12400 12401 12402 12403 12404 12405 12406 12407
		       12408 12409 12410 12411 12412 12413 12414 12415
		       12416 12417 12418 12419 12420 12421 12422 12423
		       12424 12425 12426 12427 12428 12429 12430 12431
		       12432 12433 12434 12435))
       (12601 . (65284 65285 65510 65318 8242 8243 8451
		       8491 65504 65505 65509 164 8457 8240 63 13205
		       13206 13207 13208 13252 13219 13220 13221 13222
		       13209 13210 13211 13212 13213 13214 13215 13216
		       13217 13218 13258 13197 13198 13199 13263 13192
		       13193 13256 13223 13224 13232 13233 13234 13235
		       13236 13237 13238 13239 13240 13241 13184 13185
		       13186 13187 13188 13242 13243 13244 13245 13246
		       13247 13200 13201 13202 13203 13204 8486 13248
		       13249 13194 13195 13196 13270 13253 13229 13230
		       13231 13275 13225 13226 13227 13228 13277 13264
		       13267 13251 13257 13276 13254))
       (12609 . (65283 65286 65290 65312 167 8251 9734
		       9733 9675 9679 9678 9671 9670 9633 9632 9651
		       9650 9661 9660 8594 8592 8593 8595 8596 12307
		       9655 9664 9655 9654 9828 9824 9825 9829 9831
		       8857 9672 9635 9680 9681 9618 9636 9637 9640
		       9639 9638 9641 9832 9743 9742 9756 9758 182
		       8224 8225 8597 8599 8601 8598 8600 9837 9833
		       9834 9836 12927 12828 8470 13255 8482 13250 13272
		       8481 63 170 186))
       (12610 . (9472 9474 9484 9488 9496 9492 9500 9516
		      9508 9524 9532 9473 9475 9487 9491 9499 9495
		      9507 9523 9515 9531 9547 9504 9519 9512 9527
		      9535 9501 9520 9509 9528 9538 9490 9489 9498
		      9497 9494 9493 9486 9485 9502 9503 9505 9506
		      9510 9511 9513 9514 9517 9518 9522 9525 9526
		      9529 9530 9533 9534 9536 9537 9539 9540 9541
		      9542 9543 9544 9545 9546))
       (12611 . (12449 12450 12451 12452 12453 12454 12455
		       12456 12457 12458 12459 12460 12461 12462 12463
		       12464 12465 12466 12467 12468 12469 12470 12471
		       12472 12473 12474 12475 12476 12477 12478 12479
		       12480 12481 12482 12483 12484 12485 12486 12487
		       12488 12489 12490 12491 12492 12493 12494 12495
		       12496 12497 12498 12499 12500 12501 12502 12503
		       12504 12505 12506 12507 12508 12509 12510 12511
		       12512 12513 12514 12515 12516 12517 12518 12519
		       12520 12521 12522 12523 12524 12525 12526 12527
		       12528 12529 12530 12531 12532 12533 12534))
       (12613 . (12896 12897 12898 12899 12900 12901 12902
		       12903 12904 12905 12906 12907 12908 12909 12910
		       12911 12912 12913 12914 12915 12916 12917 12918
		       12919 12920 12921 12922 12923 12800 12801 12802
		       12803 12804 12805 12806 12807 12808 12809 12810
		       12811 12812 12813 12814 12815 12816 12817 12818
		       12819 12820 12821 12822 12823 12824 12825 12826
		       12827))
       (12614 . (1040 1041 1042 1043 1044 1045 1025 1046
		      1047 1048 1049 1050 1051 1052 1053 1054 1055
		      1056 1057 1058 1059 1060 1061 1062 1063 1064
		      1065 1066 1067 1068 1069 1070 1071 1072 1073
		      1074 1075 1076 1077 1105 1078 1079 1080 1081
		      1082 1083 1084 1085 1086 1087 1088 1089 1090
		      1091 1092 1093 1094 1095 1096 1097 1098 1099
		      1100 1101 1102 1103))
       (12615 . (9424 9425 9426 9427 9428 9429 9430 9431
		      9432 9433 9434 9435 9436 9437 9438 9439 9440
		      9441 9442 9443 9444 9445 9446 9447 9448 9449
		      9312 9313 9314 9315 9316 9317 9318 9319 9320
		      9321 9322 9323 9324 9325 9326 9372 9373 9374
		      9375 9376 9377 9378 9379 9380 9381 9382 9383
		      9384 9385 9386 9387 9388 9389 9390 9391 9392
		      9393 9394 9395 9396 9397 9332 9333 9334 9335
		      9336 9337 9338 9339 9340 9341 9342 9343 9344
		      9345 9346))
       (12616 . (65296 65297 65298 65299 65300 65301 65302
		       65303 65304 65305 8560 8561 8562 8563 8564 8565
		       8566 8567 8568 8569 8544 8545 8546 8547 8548
		       8549 8550 8551 8552 8553))
       (12618 . (189 8531 8532 188 190 8539 8540 8541
		     8542 185 178 179 8308 8319 8321 8322 8323 8324))
       (12619 . (12593 12594 12595 12596 12597 12598 12599
		       12600 12601 12602 12603 12604 12605 12606 12607
		       12608 12609 12610 12611 12612 12613 12614 12615
		       12616 12617 12618 12619 12620 12621 12622 12623
		       12624 12625 12626 12627 12628 12629 12630 12631
		       12632 12633 12634 12635 12636 12637 12638 12639
		       12640 12641 12642 12643))
       (12620 . (12645 12646 12647 12648 12649 12650 12651
		       12652 12653 12654 12655 12656 12657 12658 12659
		       12660 12661 12662 12663 12664 12665 12666 12676
		       12668 12669 12670 12671 12672 12673 12674 12675
		       12676 12677 12678 12679 12680 12681 12682 12683
		       12684 12685 12686))
       (12621 . (65313 65314 65315 65316 65317 65318 65319
		       65320 65321 65322 65323 65324 65325 65326 65327
		       65328 65329 65330 65331 65332 65333 65334 65335
		       65336 65337 65338 65345 65346 65347 65348 65349
		       65350 65351 65352 65353 65354 65355 65356 65357
		       65358 65359 65360 65361 65362 65363 65364 65365
		       65366 65367 65368 65369 65370))
       (12622 . (913 914 915 916 917 918 919 920 921
		     922 923 924 925 926 927 928 929 931 932 933
		     934 935 936 937 945 946 947 948 949 950 951
		     952 953 954 955 956 957 958 959 960 961 963
		     964 965 966 967 968 969))
       (44032 . ((21487 . "옳을 가") (23478 . "집 가") (21152 . "더할 가")
                 (27468 . "노래 가") (20729 . "값 가") (34903 . "거리 가")
                 (20551 . "거짓 가") (20339 . "아름다울 가") (26247 . "겨를 가")
                 (26550 . "시렁 가") 20215 (36040 . "값 가, 값/성 가")
                 (20285 . "절 가") (26607 . "가지 가") (36838 . "부처이름 가")
                 (36603 . "수레 가기 힘들 가, 바퀴굴대 가") (22025 . "아름다울 가")
                 (39381 . "멍에 가") (23233 . "시집갈 가") (31292 . "심을 가, 곡식심을 가")
                 (21733 . "노래 가, 형언니 가") (21621 . "꾸짖을 가")
                 (33499 . "매울 가, 까다로울 가") (34952 . "가사 가")
                 (35382 . "꾸짖을 가") (26551 . "목칼 가, 도리깨채 가, 형구 가")
                 (29634 . "옥 이름 가") (33540 . "가지 가") (30146 . "헌데 딱지 가")
                 (36303 . "책상다리할 가, 도사리어 앉을 가") 26933
                 21466 20385 (26013 . "옥잔 가") (33901 . "갈대 가")
                 (33336 . "큰 배 가") (31539 . "갈잎피리 가") (21759 . "좋을 가")
                 29491 39618 (22391 . "험할 가") (27295 . "오동나무 가")
                 (35596 . "歌와 同字") (29640 . "머리꾸미개 가") 30264
                 30754 40208 19749 25281 14606 20297 21631 (27022 . "檟와 同字")
                 27042 27616 28298 29249 35949 (20206 . "假의 略字")
                 27098 35779 36158 36722 38227 39550 40068 (20642 . "세간 가")
                 (21654 . "커피차 가, 커피의 음차로 쓰는자 가") 30229
                 (32798 . "도리깨 가") 23434 23165 23580 23714 24143
                 24385 24486 24856 25107 25128 26010 27158 27847
                 28206 28370 28835 29291 29297 29324 29581 30335
                 31540 31960 32994 33141 34485 35985 36306 37216
                 37483 37490 37813 37814 39802 40218 40602 13774
                 13977 14265 14502 14601 14813 14979 15036 15442
                 15970 16543 16916 17501 17669 17746 19114 19402
                 19863))
       (44033 . ((21508 . "각각 각") (35282 . "뿔 각") (33050 . "다리 각, 종아리 각")
                 (35258 . "깨달을 각") (38307 . "문설주 각, 다락집 각")
                 (21051 . "새길 각") (21364 . "물리칠 각") (29647 . "쌍옥 각")
                 (24682 . "삼갈 각, 공경할 각") (27580 . "껍질 각") (24932 . "삼갈 각, 정성 각")
                 38401 24872 27063 35226 39612 (26743 . "서까래 각")
                 31021 (25609 . "칠 각") 30992 (21371 . "却의 本字")
                 27579 33011 36521 20629 23813 33558 (21679 . "토할 각")
                 (22470 . "메마를 각") (25841 . "놓을 각") 20331 22695
                 25997 33139 38586 14913 15372 15670 22771 24747
                 25601 35273 22617 24857 25428 26016 27338 27587
                 29608 35004 35216 36363 40261 40292 13410 13590
                 13633 13950 14633 15435 15487 15497 16289 16503
                 16886 17432 17852 18409 18884 19845))
       (44036 . ((38291 . "틈 간, 사이 간") (24178 . "방패 간")
                 (30475 . "볼 간") (20094 . "마를 간") (24185 . "줄기 간, 우물난간 간")
                 (23014 . "간사할 간") (21002 . "책 펴낼 간") (32925 . "간 간")
                 (31777 . "대쪽 간") (25031 . "정성 간, 간절할 간") (22904 . "범할 간, 간악할 간, 간사할 간")
                 (33390 . "괘이름 간, 간방 간") (26438 . "지레 간, 나무이름 간, 산뽕나무 간")
                 (35563 . "간할 간") (33393 . "어려울 간, 고생살이 간")
                 (25536 . "가릴 간, 분별할 간") (28567 . "산골물 간")
                 (31487 . "장대 간, 화살대 간, 횃대 간, 대줄기 간")
                 (22718 . "따비질할 간, 개간할 간") (30286 . "경풍 간, 간질 간")
                 (20355 . "강직할 간") (26604 . "가릴 간") (30965 . "산골 시내 간, 석간수 간, 산골물 간")
                 (26751 . "줄기") (31240 . "짚 간, 볏짚 간") 38388 24691
                 26096 (36214 . "달릴 간") (34894 . "즐길 간, 바를 간")
                 31616 (24947 . "아낄 간") (27046 . "幹의 本字") (29589 . "옥돌 간")
                 15892 31795 30639 15694 36245 26261 (36800 . "구할 간, 권할 간")
                 30448 35256 38981 33417 (20568 . "侃과 同字") 22223
                 28038 35975 39709 16379 (31174 . "稈과 同字") 24749
                 25315 25792 28071 30187 32671 33392 35045 35855
                 36396 38159 40840 (26654 . "도표 간") (33563 . "독초이름 간, 미나리아재비 간")
                 (40806 . "물을 간") 20509 22713 23015 23132 23407
                 24531 25916 27274 30279 30327 30383 30797 30976
                 31608 31919 32349 32677 33868 34129 34163 34423
                 34918 35017 35143 35145 35940 37911 37927 38764
                 39343 39708 40177 40363 15060 15765 15781 15911
                 16973 17656 17717 18930 19136 19406 19674 19807))
       (44040 . ((28212 . "목마를 갈, 급할 갈") (33883 . "칡 갈")
                 (38824 . "말갈나라 갈, 말갈보석 갈") (31469 . "다할 갈, 마를 갈")
                 (21917 . "꾸짖을 갈, 성낸 소리 갈") (35088 . "털베 갈, 굵은 베 갈, 천한사람 갈, 갈옷 갈")
                 (26359 . "어찌 갈, 그칠 갈, 벌레이름 갈") (30883 . "비 갈, 우뚝선 돌 갈, 동해의 산이름 갈")
                 (34638 . "전갈 갈, 빈대 갈") (20075 . "땅이름 갈") 26253
                 (32687 . "불친 양 갈, 땅이름 갈") 28167 (31224 . "볏짚 갈, 짚 고갱이 갈")
                 36661 36693 40353 (26988 . "패를 써서 표시할 갈")
                 37755 40534 (22134 . "벼슬이름 갈, 땅이름 갈") (34829 . "전갈 갈")
                 22335 23889 23985 25204 25814 27644 29542 30989
                 38786 39444 39601 13331 13534 13540 14384 15134
                 15254 15715 15719 16530 16741 16915))
       (44048 . ((24863 . "감동할 감, 찌를 감, 깨달을 감, 느낄 감")
                 (28187 . "덜 감, 무지러질 감, 가벼울 감, 물이름 감, 감할 감")
                 (25954 . "감히 감, 구태여 감, 날랠 감, 용맹할 감")
                 (29976 . "달 감, 달게 여길 감") (30435 . "거느릴 감, 살필 감, 볼 감, 벼슬 감, 감옥 감")
                 (37969 . "거울 감, 밝을 감, 비칠 감, 경계할 감")
                 (25022 . "섭섭할 감, 한할 감, 한혐할 감") (37039 . "조나라 서울")
                 (22570 . "견딜 감, 이길 감, 맡을 감, 하늘 감, 산 모양이 기괴할 감")
                 (21208 . "마감할 감, 정할 감") (30640 . "굽어볼 감, 내려다 볼 감")
                 (32058 . "아청 감, 보랏빛 감, 연보라 감") (26577 . "감귤 감, 감자 감")
                 (30131 . "감질병 감, 궤양 감, 창병 감, 종기 아플 감")
                 (22350 . "구덩이 감, 험할 감, 작은 잔 감, 괘 이름 감, 불우하여 뜻을 이루지 못할 감")
                 (25121 . "찌를 감, 이길 감, 죽일 감") (37970 . "鑑과 同字, 거울 감")
                 (23884 . "깊은 골 감, 곁굴 감, 깊은 산 감, 산골짜기 감")
                 (27204 . "감람나무 감") (40853 . "취할 감, 이길 감, 감실 감")
                 20943 30417 27483 37219 (25788 . "흔들 감, 움직일 감, 깨뜨릴 감")
                 29770 (27455 . "서운할 감, 찌푸릴 감, 걱정할 감")
                 (24327 . "뚜껑 감, 사람이름 감") 37747 (23873 . "산 험할 감")
                 38366 (25000 . "어리석을 감, 미련할 감") 30733 26935
                 27457 28553 40572 36695 22591 22634 24770 25750
                 27765 34486 35404 30897 32448 37492 40161 40528
                 40859 (22377 . "도가니 감") (22515 . "구덩이 감") 27860
                 28134 30681 40571 20292 20890 20978 (20981 . "입벌리다")
                 21425 21633 22664 23602 23606 23809 23877 29610
                 29621 29981 30945 31003 31479 31695 31891 33527
                 33712 35708 36113 36641 36673 37996 39760 40036
                 40302 40568 13448 13579 13622 13851 13939 14111
                 14604 14737 15008 15232 15416 16002 16205 16397
                 16621 16862 17083 17184 17891 17939 18131 18711
                 18921 19642))
       (44049 . ((30002 . "갑옷 갑, 첫째 천간 갑, 비롯할 갑, 떡잎 날 갑, 법령 갑, 대궐 갑")
                 (38296 . "문 빗장 갑, 물 문 갑") (23724 . "산 허구리 갑, 줄이 느런할 갑, 산허리 갑")
                 (37440 . "갑옷 갑") (21283 . "궤 갑, 상자 갑, 궤짝 갑")
                 (32987 . "어깻죽지 갑") 30604 24098 14391 38078 38392
                 (38864 . "가죽바지 갑") 21394 25621 25926 29630 31514
                 31594 33338 37272 13539 13637 13641 13776 14785
                 14985 15217 17446 17836 17880 17957 19130 19174))
       (44051 . (13721))
       (44053 . ((27743 . "가람 강, 물 강") (24378 . "강할 강, 나머지 강, 바구미 강")
                 (35611 . "강론할 강, 강구할 강, 익힐 강, 강화할 강")
                 (38477 . "내릴 강, 떨어질 강, 돌아갈 강") (24247 . "편안할 강, 즐거울 강, 다섯거리 강")
                 (32177 . "벼리 강, 근본 강, 법 강, 대강 강") (37628 . "쇠 강, 강철 강")
                 (23004 . "성씨 강, 강할 강") (21083 . "굳을 강, 굳셀 강, 꼬장할 강, 바야흐로 강")
                 (30086 . "땅 경계 강, 갈피 강, 굳셀 강, 변방 강, 지경 강")
                 (23713 . "멧동 강, 산등성이 강") (24394 . "굳셀 강, 센 활 강, 사나울 강, 송장 뻣뻣할 강, 꿋꿋할 강")
                 (23831 . "岡의 俗字, 산등성이 강") (24951 . "강개할 강")
                 (33108 . "창자 강, 속 빌 강, 뼈대 강, 말 허구리 강")
                 (34193 . "생강 강, 새앙 강") (31968 . "겨 강, 번쇄할 강")
                 (33313 . "오나라 배 강") (32115 . "깊게 붉을 강, 강초 강, 깊게 붉은 강")
                 (35137 . "포대기 강") (32652 . "오랑캐 강, 말 끝낼 강")
                 (22536 . "언덕 강, 밭 두둑 강, 독 강") (30074 . "疆과 同字, 지경 강")
                 (40007 . "천징어 강") (24375 . "강할") (26464 . "외나무 다리 강, 깃대 강, 상 앞 가로막대 강, 올릴 강")
                 (20725 . "엎드러질 강, 쓰러질 강") 20872 22859 (24572 . "강개할 강, 뜻이 크고 기개있을 강, 탄식할 강")
                 36331 (27263 . "참죽나무 강, 호미자루 강, 사닥다리 강, 억셀 강")
                 (38849 . "고삐 강") (33587 . "강리풀 강") (22723 . "疆과 同字")
                 26849 (25179 . "마주 들 강") 37357 25478 (27565 . "죽어 썩지 않을 강, 마른 누에 강")
                 (30716 . "돌다리 강, 징검다리 강, 성실할 강") (32328 . "繦의 本字")
                 29370 29586 31976 32366 (31301 . "겨 강, 악기이름 강")
                 (32609 . "강별 강") (32663 . "羌과 同字") (32683 . "밭 갈 강")
                 23765 23931 37881 16216 32560 21018 23703 32475
                 35762 38050 38250 (20619 . "아첨 아니할 강") (35911 . "광저기 강")
                 20910 21221 21279 21812 21836 22004 22245 22589
                 22789 23261 23929 24348 24374 25667 25726 26892
                 27130 28332 28434 28462 28981 29288 29317 29343
                 29456 29511 29596 29751 29928 30037 30085 30730
                 30873 30995 31611 31879 31969 32358 32734 32809
                 33177 33857 34051 34595 34998 35645 36543 37792
                 37870 39004 13563 13588 14056 14185 14304 14325
                 14492 14799 14934 15171 15182 15426 15614 15956
                 16154 16185 16292 16579 16616 16818 17161 17772
                 17895 17909 18071 18505 18770 19221 19738))
       (44060 . ((25913 . "고칠 개, 거듭할 개, 바꿀 개")
                 (38283 . "열 개, 통할 개, 발할 개, 베풀 개, 풀 개")
                 (20491 . "낱 개, 치우칠 개") (30342 . "모두 개, 한 가지 개, 같을 개, 다 개")
                 (35912 . "승전악 개, 일찍 개") (27114 . "평두목 개, 절개 개, 대강 개, 거리낄 개, 칠한 술통 개, 대개 개")
                 (20171 . "낄 개, 도울 개, 맬 개, 클 개, 중매할 개")
                 (33995 . "뚜껑 개, 덮개 개, 이엉 개, 우산 개, 가리울 개, 덮을 개")
                 (24936 . "분할 개, 강개할 개, 슬플 개, 분개할 개")
                 (20215 . "착할 개, 클 개, 사신 개") (22607 . "시원한 땅 개")
                 (31623 . "낱 개, 개수 개") (28433 . "물 댈 개, 씻을 개, 물 천천히 흐를 개")
                 (33445 . "겨자 개, 갓 개, 지푸라기 개, 티끌 개")
                 (20977 . "싸움 이긴 풍류 개, 착할 개, 마파람 개, 좋은 사람 개, 승전악 개")
                 (24894 . "한숨 쉴 개, 성낼 개, 노할 개") (30422 . "蓋의 俗字, 뚜껑 개")
                 (37799 . "투구 개, 갑옷 개") (24887 . "탐할 개, 급할 개, 편안할 개")
                 (24669 . "걱정 없을 개") (30117 . "옴 개") (27113 . "槪와 同字")
                 20010 19984 33890 (21108 . "낫 개, 가까울 개, 간절할 개")
                 (29600 . "큰 서옥 개") (25577 . "문지를 개, 문질러 훔칠 개")
                 15267 (21251 . "청구할 개, 줄 개, 빌 개") 22021 (38355 . "열 개, 풀 개, 활깍지 개")
                 21896 28253 31092 31002 26271 34471 31277 (30933 . "돌 부딪는 소리 개")
                 27010 28887 16471 18551 20975 21056 22450 24320
                 24698 38041 38112 38158 38196 38399 20015 20066
                 21203 21252 21540 22000 22866 23192 23596 23701
                 24134 24206 24523 24550 25124 25697 26042 28297
                 28812 28815 29039 29335 29758 29890 30734 31044
                 32096 33097 33928 34644 34936 37058 37411 37703
                 37774 37926 38435 39101 39744 39786 40347 13400
                 13666 14159 14393 14879 15009 15356 15486 16333
                 16584 16850 17367 17449 17460 17555 18724 19121
                 19440 19514 19524 19640))
       (44061 . ((23458 . "손님 객, 부칠 객, 나그네 객, 지날 객")
                 (21888 . "기침할 객, 토할 객") 22629 25570 30886
                 34889 13766 17940))
       (44079 . (13707 13710))
       (44081 . ((26356 . "다시 갱") (22353 . "빠질 갱, 묻을 갱, 구덩이 갱")
                 (32697 . "국 갱") (31923 . "메벼 갱") (36065 . "이을 갱, 일 갱")
                 (37847 . "금석 소리 갱, 사람이름 갱") (30812 . "아릿아릿할 갱, 알랑알랑할 갱, 돌소리 갱")
                 31188 37726 38444 30843 17332 30785 36179 38143
                 21157 22344 22932 31241 32686 37557 14838 15394
                 15988 16600 18518 18528 18544 18586))
       (44089 . ((37301 . "술추렴") 22129 36475 35899 33220
                 21226 13752 15870 18672))
       (44144 . ((36554 . "수레 거, 그물 거, 잇몸 거") (21435 . "갈 거, 버릴 거, 오래될 거, 떨어질 거, 쫓을 거, 감출 거")
                 (23621 . "살 거, 곳 거, 앉을 거, 놓을 거, 쌓을 거")
                 (24040 . "클 거, 많을 거, 억 거") (25831 . "들 거, 받들 거, 움직일 거, 일걸을 거")
                 (25298 . "막을 거, 맞설 거, 다닥칠 거, 다다를 거")
                 (36317 . "며느리 발톱 거, 지낼 거, 뛸 거, 겨룰 거, 미눌 거, 떨어질 거")
                 (25818 . "의지할 거, 기댈 거, 웅거할 거, 짚을 거")
                 (28192 . "개천 거, 도랑 거, 휑덩그렁할 거, 클 거, 껄껄 웃을 거, 무엇 거")
                 (20520 . "거만할 거, 걸터 앉을 거, 굽을 거") (37301 . "술추렴 거, 추렴 걷을 거")
                 (36989 . "급할 거, 두려울 거, 역말 수레 거") (25454 . "가질 거, 길거할 거, 의지할 거, 손병 거")
                 (31067 . "물리칠 거, 빌 거, 굳셀 거") (37445 . "갈고리 거, 클 거, 강한 쇠 거, 갈구리 거")
                 (28844 . "횃불 거, 불 땔 거, 햇불 거") (36382 . "걸터앉을 거")
                 (37624 . "톱 거, 다리 꺾는 형벌 거") 36850 33289
                 35406 (35070 . "옷 뒷자락 거") (34986 . "소매 거, 소매 아귀 거")
                 (34343 . "패랭이꽃 거, 석죽화 거, 마음에 든든할 거")
                 29722 26588 (33682 . "모시풀 거, 나라이름 거, 고을이름 거")
                 14911 (34134 . "연꽃 거") (31212 . "검은 기장 거") 14912
                 32799 31796 14224 37947 (31847 . "대자리 거, 천상바라기 거, 양잠기 거")
                 29846 (39375 . "튀기 거, 거공벌레 거") 34401 (33106 . "건치 거, 꿩 포 거")
                 (33507 . "상추 거, 검은 깨 거") 24326 (32992 . "열 거, 갈비 거")
                 30962 38457 (31589 . "쌀 담는 광주리 거, 볏단 거")
                 26896 29865 (20297 . "나라이름 거") (21631 . "입 딱 벌일 거")
                 20322 20453 22729 23712 24583 25029 28090 40331
                 20030 25312 25369 27017 35765 38044 38190 38191
                 (26139 . "밝을 거") 20257 20965 21230 21303 21434
                 21881 22998 23150 23820 25873 27384 27397 27499
                 27952 28141 28605 31892 32054 32727 33357 33731
                 34487 34587 34742 34781 34896 35191 35411 35942
                 36486 39828 40011 40622 40705 64020 14162 14378
                 14532 15023 15074 15189 15339 15398 16725 16829
                 17575 18159 18275 18299 18378 18617 19551 19797
                 19865))
       (44145 . (24042))
       (44148 . ((24314 . "세울 건, 둘 건, 심을 건, 칼집 건, 별이름 건")
                 (20094 . "하늘 건, 괘 이름 건, 서북쪽 건, 굳셀 건, 임금 건")
                 (20214 . "조건 건, 가지 건, 구별할 건, 나눌 건")
                 (20581 . "굳셀 건, 병 없을 건, 탐할 건") (24062 . "수건 건, 덮을 건")
                 (37749 . "열쇠 건, 수레 굴대 건, 열쇄 건") (34388 . "정성 건, 빼앗을 건, 죽일 건, 공경할 건")
                 (33137 . "힘줄 건") (24838 . "죄 건, 허물 건, 어기어질 건")
                 (36423 . "절름거릴 건, 험할 건, 교만할 건, 절뚝발이 건")
                 (39467 . "말 배 앓을 건, 이지러질 건, 휙 뛰어 나아갈 건, 이지러지는 건")
                 (26967 . "문 빗장 건, 문지방 건") (35120 . "바지 건, 걷을 건, 발 걷을 건")
                 (35591 . "말 더듬거릴 건, 바른말할 건") (38828 . "칼집 건, 동개 건")
                 25652 40369 (25589 . "멜 건, 막을 건, 들 건") 17805
                 35536 28245 (29325 . "불친 소 건, 짐승이름 건, 촉나라 고을이름 건")
                 24484 (30583 . "눈으로 세어볼 건") 20089 23499 14876
                 39582 20097 22237 22680 23496 24383 25872 25875
                 26068 27049 27645 28455 28733 34246 36410 38003
                 39453 39950 39980 13562 13587 13780 14900 14915
                 14958 14967 15915 16834 16863 17045 17948 18029
                 18367 19272 19391))
       (44152 . ((20625 . "호걸 걸, 준걸 걸, 뛰어날 걸, 거만할 걸, 떡잎 걸")
                 (20062 . "구걸할 걸, 요구할 걸, 원할 걸, 거지 걸")
                 (26480 . "傑의 俗字, 준걸 걸") (26688 . "찢을 걸, 홰 걸, 빼어날 걸, 멜 걸, 사나을 걸, 왕이름 걸")
                 (20552 . "힘 빼두를 걸, 빠를 걸, 헌걸찰 걸") (26373 . "갈 걸")
                 (20076 . "갈 걸") 23909 13350 28368 (27044 . "닭의 홰 걸")
                 25641 27373 33438 34258 13878 15943 17314 17559))
       (44160 . ((27298 . "교정할 검, 금제할 검, 법 검, 봉할 검, 책메뚜기 검, 수결둘 검, 검사할 검")
                 (20745 . "검소할 검, 흉년들 검, 적을 검, 가난할 검")
                 (21133 . "칼 검, 칼로 찔러 죽일 검, 칼 쓰는법 검")
                 (21138 . "劎의 譌字, 칼 검") (40660 . "검을 검") (37392 . "굴대 비녀장 검, 자물쇠 검, 큰 보습 검")
                 (30652 . "눈시울 검") (25791 . "잡을 검, 구속할 검, 관찰할 검, 비교할 검")
                 (33441 . "마름 검, 가시연밥 검") 33225 21134 20461
                 20537 21073 21091 25441 26816 26908 30545 33080
                 38052 23698 20209 21105 21140 24564 14788 16860
                 16937 19022))
       (44161 . ((24623 . "무서워할 겁, 겁낼 겁, 으를 겁")
                 (21163 . "위협할 겁, 겁탈할 겁, 부지런할 겁, 대궐 층계 겁, 겁 겁, 화재 겁, 빼앗을 겁")
                 (36850 . "자내") (21030 . "겁탈할 겁, 구속할 겁") 21052
                 14652 14680 34999 (21031 . "刦의 俗字") 36338 26117
                 25278 26497 30749 34576 34929 37475 39781 40278
                 15218 15972 16256 16439 16562 16566))
       (44163 . (21791))
       (44172 . ((25581 . "높이 들 게, 옷 걷고 물 건널 게, 등에 질 게")
                 (25001 . "쉴 게") (20552 . "쉴 게, 게타 게, 글귀 게")
                 24850 22413 25522 24967))
       (44201 . ((26684 . "이를 격, 올 격, 바를 격, 대적할 격, 감동할 격, 격식 격")
                 (28608 . "물결 부딪혀 흐를 격, 찌를 격, 맑은 소리 격, 말 과격히 할 격, 급할 격")
                 (25802 . "撃과 同字, 칠 격") (38548 . "막을 격, 멀 격")
                 (27268 . "격서 격, 격문 격, 과격할 격") (33160 . "명치 격")
                 (35233 . "박수 격, 남자무당 격") 25731 32750 32371
                 (39612 . "마른 뼈 격, 드러난 뼈 격, 짐승의 뼈 격")
                 (38339 . "고요할 격") (39730 . "오지병 격, 땅이름 격, 다리굽은솥 격")
                 38324 32140 40281 (25356 . "칠 격, 들 격") (40195 . "왜가리 격, 때까치 격")
                 24837 28358 34178 35581 20987 32484 32514 32564
                 35275 38418 19735 (27588 . "맞부딛는 소리 격, 충돌할 격, 창 격")
                 21572 21981 22414 22716 25657 25663 25931 26945
                 27435 28264 29161 29329 29386 29541 30986 32217
                 33269 34514 35297 35684 36698 37113 37139 38839
                 40362 40452 40755 13375 14744 14772 15038 16502
                 16518 16938 17433 18000 18175 18342 18448))
       (44204 . ((35211 . "볼 견, 만날 견, 당할 견") (29356 . "큰 개 견")
                 (22533 . "굳을 견, 굳셀 견, 반드시 견") (36963 . "보낼 견, 쫓을 견, 견전제 견")
                 (29309 . "당길 견, 이끌 견, 빠를 견, 거리낄 견, 볏줄 견")
                 (32937 . "어깨 견, 이길 견, 맡길 견, 멜 견, 세 살 된 짐승 견")
                 (32121 . "비단 견, 깁 견") (29956 . "질그릇 견, 살필 견, 표시할 견, 진 이름 견, 면할 견, 밝을 견")
                 (35700 . "꾸짖을 견, 성낼 견, 귀양갈 견") (40273 . "접동새 견, 두견새 견, 뻐꾹새 견")
                 (32365 . "고치 견, 발 부르틀 견, 목 쉴 견, 비단 견")
                 20436 (34866 . "밝을 견, 조촐할 견, 반딧불 견") 29301
                 (30030 . "밭 도랑 견, 산골 도랑 견") 32307 (29431 . "고집스러울 견, 조수할 견, 편협할 견")
                 (32382 . "곡진할 견, 마지 않을 견") 25492 27751 23693
                 29086 32308 26776 32613 (31591 . "대 홈통 견") (32642 . "걸 견, 걸릴 견")
                 27761 30013 35395 36284 37627 22362 26535 31509
                 32482 32561 35265 35892 38040 40099 40515 17208
                 (27179 . "떡갈나무 견") (39993 . "큰 가물치 견, 물치 견, 강고도리 견")
                 20761 24181 24723 25400 25518 25729 26532 28032
                 (29357 . "개사슴변") 29455 29543 29945 30538 32120
                 33598 33786 34440 34616 34834 35024 35194 35932
                 35939 37124 37379 37522 37874 37971 38595 40189
                 40307 40585 13408 14145 14526 15220 15326 15328
                 15938 16687 16716 16720 17100 17111 17169 17527
                 17879 18056 18058 18515 18682 19080 19300 19660
                 19798 19803 19809 19812 19884))
       (44208 . ((32080 . "맺을 결, 마칠 결, 나중 결, 몫 결")
                 (27770 . "결단할 결, 물이름 결, 물골 터 놓을 결, 끊을 결, 이별할 결")
                 (28500 . "맑을 결, 정결할 결, 조촐할 결") (32570 . "이지러질 결, 번개 번쩍거릴 결, 깨어질 결, 이 빠질 결")
                 (35363 . "이별할 결, 열결할 결, 사례할 결, 비결 결")
                 (25326 . "길거할 결, 열심히 일할 결") (25225 . "긁을 결, 당길 결")
                 20915 (38347 . "쉴 결, 다할 결, 풍류 마칠 결, 문 닫을 결")
                 (35286 . "서운할 결, 들쳐낼 결, 바랄 결") 32092 (29606 . "옥패 결, 활 깍지 결")
                 40194 37388 39363 40195 35002 21012 27951 36907
                 13527 32467 35776 38421 20714 22940 25367 28820
                 28934 30724 32572 33461 33947 34455 34538 35324
                 36221 36281 36860 37352 37905 39773 13558 14019
                 15176 15278 15967 16164 16407 17085 17267 17360
                 18252 18751 18779 18860 18876))
       (44216 . ((20860 . "겸할 겸, 아우를 겸, 둘 얻을 겸, 모을 겸")
                 (35609 . "사양할 겸, 겸손할 겸, 괘 이름 겸") (24906 . "앙심먹을 겸, 불만족하게 생각할 겸, 싫을 겸")
                 (31645 . "재갈 겸, 족집개 겸, 목 잠글 겸") (37463 . "목사슬 겸, 세력으로써 남을 누를 겸, 입 다물 겸")
                 (37772 . "낫 겸") (27465 . "흉년들 겸, 적을 겸, 나쁠 겸")
                 (33977 . "갈대 겸") (21979 . "부족히 여길 겸, 겸손 겸, 입에 밥 물고 있을 겸")
                 (32273 . "합사비단 겸") (20628 . "겸종 겸") 16105 (25297 . "자갈 먹일 겸")
                 40380 32547 35878 38067 40146 40547 27318 40571
                 (40666 . "얕은 금향빛 겸") (40760 . "두더쥐 겸") 23698
                 23920 27087 32684 33153 36253 39964 16932 16948
                 17076 19296 19425))
       (44217 . (35020 21770 18143 37087 37071 22326 25488
		       27534 40266 16427 16461 19041))
       (44221 . ((20140 . "서울 경") (25964 . "공경할 경") (36629 . "가벼울 경")
                 (32147 . "날 경, 경서 경") (24950 . "경사 경") (26356 . "고칠 경, 다시 경")
                 (26223 . "별 경, 볕 경") (24218 . "일곱째 천간 경")
                 (39514 . "놀랄 경") (32789 . "밭갈 경") (31478 . "겨룰 경, 다툴 경")
                 (22659 . "지경 경") (31455 . "다할 경, 마칠 경") (35686 . "경계할 경")
                 (37857 . "거울 경") (20670 . "기울 경, 기울어질 경")
                 (21375 . "벼슬 경") (38915 . "밭 넓이 단위 경, 백이랑 경")
                 (24465 . "지름길 경") (30828 . "굳을 경, 단단할 경")
                 (26775 . "대개 경, 곧을 경") (29898 . "옥 경, 붉은옥 경")
                 (20742 . "경계할 경") (29855 . "옥 빛 경, 옥광채날 경")
                 (28805 . "빛날 경, 빗날 경") (21185 . "굳셀 경") (39912 . "고래 경")
                 (30956 . "경쇠 경") (38968 . "목 경") (33686 . "줄기 경")
                 (33051 . "정강이 경") (25004 . "깨달을 경, 깨우칠 경")
                 (30169 . "심줄 땅길 경, 목뻣뻣할 경") (32069 . "끌어 쥘 경, 낚을 경")
                 (32831 . "빛날 경, 반짝거릴 경") (36885 . "소로 경, 가까울 경")
                 (22384 . "들 경") (29861 . "경옥 경") (25806 . "들 경")
                 (21197 . "셀 경, 굳셀 경") (28039 . "통할 경") (28913 . "빛날 경, 불김 경")
                 (26299 . "밝을 경") (20435 . "徑과 同字, 곧을 경") (20510 . "굳셀 경")
                 32076 36605 (32580 . "빌 경") 20144 (25155 . "빗장 경")
                 (31480 . "競과 同字") (20879 . "빛날 경") (29026 . "외로울 경")
                 (24824 . "근심할 경") (30026 . "耕의 古字") (39873 . "생선뼈 경")
                 29517 (21757 . "목멜 경") (27296 . "도지개 경") (32134 . "두레박줄 경")
                 (40677 . "묵형할 경") 29850 (35111 . "홑옷 경") 26324
                 15347 (35622 . "기침 경") (39369 . "목장 경") 28547
                 25389 (21060 . "목 벨 경") (29106 . "빛날 경") 22491
                 40598 29149 29308 38805 22247 34257 15808 20665
                 21088 26737 28013 32151 33668 35481 40330 19115
                 24452 33550 20542 21037 21170 24198 24362 27682
                 27902 28867 29756 30153 31454 32463 32480 33003
                 33557 36731 36851 38236 39031 39048 39054 40096
                 40120 40530 17209 (25020 . "공경할 경") (28973 . "근싱할 경")
                 (38984 . "홑옷 경") 20765 (20866 . "멀다") 20875 22357
                 22361 22466 23131 23802 24032 24156 24252 24270
                 24371 25395 25526 25724 25807 26878 26921 27144
                 27241 27326 27366 27532 27691 28400 28416 28916
                 28991 29018 29019 29147 29381 29444 30552 30607
                 33496 33774 33885 34319 34812 35497 36217 36353
                 37088 37700 37963 38938 39403 39614 40283 40321
                 40608 13452 13519 14100 14499 14639 14953 14981
                 15116 15307 15841 15851 15977 15989 15993 16135
                 16280 16580 17089 17156 17684 17691 18200 19344
                 19431 19604 19806))
       (44228 . ((35336 . "꾀 계, 셀 계") (30028 . "지경 계")
                 (30328 . "열째 천간 계") (28330 . "시내 계") (23395 . "끝 계")
                 (40388 . "닭 계") (21843 . "열 계") (32380 . "이을 계")
                 (25106 . "경계할 계") (20418 . "걸릴 계, 이을 계") (31995 . "이을 계, 실마리 계")
                 (38542 . "섬돌 계") (22865 . "맺을 계, 계약할 계") (26690 . "계수나무 계")
                 (26800 . "형틀 계, 기계 계") (32363 . "맬 계, 맺을 계")
                 (24760 . "두근거를 계, 두근거릴 계") (35903 . "시내 계")
                 (31293 . "머무를 계") (26856 . "창 계, 창틀 계") (35489 . "경계할 계")
                 36426 (23622 . "이를 계, 다다를 계") (30926 . "谿와 同字, 시냇물 계")
                 (22586 . "界와 同字, 지경 계") 23626 (38622 . "닭 계")
                 32153 (28883 . "화덕 계") (32637 . "물고기 그물 계")
                 (34186 . "삽주 계") 27918 (22566 . "섬돌 계") (39675 . "상투 계")
                 32331 40335 (31114 . "계제 계") (31571 . "비녀 계")
                 31492 (32174 . "발 고운 비단 계") 31271 24712 32018
                 16081 26501 17990 40450 21855 (30216 . "미칠 계")
                 21343 21575 26229 26281 27285 29960 30235 37733
                 21551 40481 32487 34015 35745 35819 38194 38454
                 15753 (32280 . "繫와 同字") (33898 . "베일 계, 삽주 계")
                 20073 21784 21844 21946 21976 23772 23904 (24400 . "고슴도치머리")
                 (24401 . "고슴도치머리") 26123 26644 26960 27317
                 28721 30029 30197 30902 31552 32586 33884 34005
                 34350 34363 34815 35564 37512 37765 37977 38361
                 39898 40174 40792 13415 13445 13635 13713 14071
                 14445 14984 15126 15287 15493 15590 15605 16309
                 16408 16456 16457 17116 17135 17407 18057 18170
                 18447 18516 18823 19156 19307 19308 19702))
       (44256 . ((21476 . "옛 고") (39640 . "높을 고") (21578 . "알릴 고, 고할 고")
                 (33510 . "쓸 고, 괴로울 고") (22266 . "굳을 고") (25925 . "옛 고, 연고 고")
                 (32771 . "상고할 고") (22993 . "시어미 고, 시어머니 고")
                 (23396 . "외로울 고") (24235 . "곳집 고") (40723 . "북 고")
                 (31295 . "볏집 고") (26543 . "마를 고") (39015 . "돌아볼 고")
                 (38599 . "새이름 고, 머슴 고") (36040 . "장사 고, 값/성 고")
                 (30352 . "부르는 소리 고, 언덕 고") (37678 . "땜질할 고")
                 (25335 . "가질 고, 때릴 고") (30204 . "고질 고, 고질병 고")
                 (36764 . "허물 고") (33167 . "살찔 고, 기름 고") (21481 . "두드릴 고")
                 (32929 . "넓적다리 고") (34996 . "바지 고") (25970 . "두드릴 고")
                 (21617 . "울 고, 아이우는 소리 고") (35493 . "고할 고, 가르칠 고")
                 (27073 . "槀와 同字, 마른나무 고") (34241 . "마른 나무 고, 짚 고")
                 (25911 . "考의 古字, 생각할 고") (27837 . "팔 고") (34865 . "독 고, 회충 고")
                 (32660 . "새끼양 고") (33533 . "줄 고") (30590 . "못 고, 고환 고")
                 (33776 . "향초 고") (26272 . "흴 고, 밝고흰 고") (23611 . "꽁무니 고")
                 29330 (30653 . "소경 고") 33263 39641 30391 (21043 . "가를 고")
                 (38752 . "기댈 고") (32762 . "날 고") 27858 31294 (32607 . "그물 고")
                 (35290 . "술잔 고") 40724 (26482 . "밝을 고") 21567
                 19451 (26674 . "북나무 고") 26971 (27072 . "마를 고")
                 (20272 . "값 고") (35393 . "주석 고") (30444 . "염지 고")
                 (31705 . "상앗대 고") (40227 . "자고 고") 32758 (30347 . "못 고")
                 (37220 . "계명주 고") (27356 . "활집 고") (31957 . "떡 고")
                 (33007 . "사타구니 고") 39227 (37084 . "나라이름 고")
                 26599 (32662 . "검은 암양 고") 20388 34500 (27129 . "두레박 고")
                 23914 31519 32665 32973 25630 14328 (37431 . "다리미 고")
                 26701 26845 27092 27248 29404 32753 34215 34291
                 40731 14024 24211 32468 34506 35044 35778 35824
                 36721 38068 38096 38150 38178 39038 40083 40116
                 40490 (20933 . "얼어붙을 고") (29295 . "암소 고") (31233 . "稿와 同字")
                 (31629 . "테 고") (40433 . "새이름 고") 22617 20721
                 21186 21469 21653 21708 22288 22540 22787 23135
                 23284 23827 23854 24600 25358 27245 27322 27928
                 28380 28900 28984 29040 29178 30228 31086 31088
                 31099 31134 31193 31250 31606 31643 31696 32093
                 32603 33228 33735 33746 34211 34508 35122 36301
                 36593 36594 37056 37426 37548 37615 38879 39607
                 39642 39643 39829 39859 39884 39901 40398 13420
                 13548 13661 13789 13884 13961 13974 14568 14610
                 14845 15011 15375 15456 15628 15878 16139 16165
                 16253 16312 16321 16391 16422 16769 16866 16969
                 16976 17243 17291 17467 17513 17624 17634 17676
                 17711 18060 18178 18647 18936 19175 19269 19404
                 19432 19434))
       (44257 . ((26354 . "굽을 곡") (35895 . "골짜기 곡") (31296 . "곡식 곡")
                 (21741 . "울 곡") (40288 . "고니 곡, 과녁 곡") (26767 . "쇠고랑 곡, 수갑 곡")
                 (26011 . "휘 곡, 헤아릴 곡") 31955 (36674 . "바퀴 곡")
                 22195 27030 27586 (32288 . "주름 비단 곡") (35315 . "뿔잔 곡")
                 29311 29812 21950 40516 (27122 . "떡갈나무 곡") 20424
                 21284 21762 21763 22045 27968 28658 28692 28715
                 28928 28933 30814 30834 31553 31916 34075 34512
                 34716 36903 39671 64015 13702 15030 15234 16067
                 16432 16952 17317 17448 17596 18075 18583 18890
                 18940 19668 19784 19868))
       (44260 . ((22256 . "괴로울 곤, 곤할 곤") (22372 . "땅 곤")
                 (26118 . "형 곤, 맏.형 곤") (26829 . "몽둥이 곤, 곤장 곤")
                 (34974 . "곤룡포 곤, 천자의 옷 곤") (23825 . "산이름 곤")
                 (26801 . "문지방 곤") (39908 . "준마 이름 곤, 물고기 알 곤")
                 (28414 . "흐를 곤, 끓는 물 곤") (29736 . "옥돌 곤, 아름다운옥 곤")
                 (34926 . "袞과 同字") (38315 . "문지방 곤") (24707 . "정성 곤")
                 39872 22780 (39649 . "머리 깍을 곤") 28378 39648
                 (40318 . "댓닭 곤") (23824 . "곤륜산 곤") 29869 32145
                 (35084 . "잠방이 곤") 31239 26204 34057 (25414 . "두드릴 곤")
                 34008 34603 39820 16578 22776 32498 35016 36746
                 38175 38403 40103 40114 40525 (22531 . "따 곤") (32196 . "띠 곤")
                 (35021 . "걷어 올릴 곤") (37653 . "붉은 쇠 곤") (40356 . "댓닭 곤")
                 (40811 . "이 빠질 곤") (20008 . "뚫다") 22546 23585
                 24771 25409 26846 28035 28489 28945 29108 29457
                 29819 30543 30548 30580 30833 30937 31093 33239
                 33742 35049 35636 36645 37260 38328 39433 39656
                 13485 13925 14419 14896 15099 15159 15438 17342
                 17845 18011 18183 18437 19818))
       (44264 . ((39592 . "뼈 골") (28369 . "어지러울 골") (27752 . "골몰할")
                 27753 (40379 . "송골매 골") 30715 (27070 . "등걸 골")
                 (25648 . "팔 골") 26458 22307 39305 40536 20646 23603
                 24882 25223 32270 33991 39005 39222 13332 13347
                 13492 16310 19369 19399))
       (44273 . (19433))
       (44275 . (24292 33930 13723 13724))
       (44277 . ((24037 . "장인 공") (20844 . "공변될 공") (21151 . "공 공")
                 (31354 . "빌 공") (20849 . "함께 공, 한가지 공") (24656 . "두려울 공")
                 (24685 . "공손할 공, 공순할 공") (20379 . "이바지할 공, 베풀 공")
                 (36002 . "바칠 공") (23380 . "구멍 공") (25915 . "칠 공")
                 (25511 . "당길 공") (25329 . "두 손 맞잡을 공, 팔짱길 공")
                 (38799 . "묶을 공, 가죽테 공") (29657 . "큰 옥 공, 옥이름 공")
                 (34467 . "지네 공") 31559 13531 15583 (36331 . "발자국 소리 공")
                 (20517 . "어리석을 공") (23814 . "산이름 공") 32702
                 (34537 . "메뚜기 공") (37357 . "굴대 공") 38810 (24318 . "들다")
                 (31644 . "공후 공") 37019 (24766 . "정성 공") (26673 . "두공 공")
                 31547 13515 (27091 . "지렛대 공") 28083 (36123 . "줄 공")
                 21357 30104 31999 15199 16556 24041 36129 36195
                 40858 (34540 . "귀뚜라미 공") (40852 . "공손할 공")
                 22506 22632 24138 24873 24912 25330 26703 28776
                 29077 30847 30909 31714 31996 33340 33926 36539
                 36609 37518 39672 40316 40847 13494 13530 13676
                 14537 14623 14632 14828 14829 15058 15274 15343
                 15489 15900 16044 16166 16565 16707 16733 16872
                 17028 17672 18212 18519 18620 18639 19512 19531
                 19634 19661))
       (44278 . ((20018 . "땅이름 곶")))
       (44284 . ((26524 . "실과 과, 과실 과") (36942 . "지날 과")
                 (31185 . "과정 과, 과거 과") (35506 . "매길 과, 시험할 과")
                 (23521 . "적을 과") (35463 . "자랑할 과") (29916 . "오이 과, 외 과")
                 (25096 . "창 과") (33747 . "과일 과, 과자 과") (38982 . "낟알 과, 알갱이 과")
                 34996 34680 (36328 . "타넘을 과") (37707 . "노구솥 과")
                 (35065 . "쌀 과") (22840 . "자랑할 과") (31392 . "보금자리 과")
                 (22821 . "많을 과") 23025 (23207 . "사람 이름 과") 34198
                 (33007 . "사타구니 과") (25790 . "칠 과") 34622 37403
                 (20361 . "자랑할 과") (36381 . "복사뼈 과") (37529 . "대구 과")
                 (34636 . "올챙이 과") 27291 (39437 . "암말 과") 21102
                 22529 28105 33602 36640 21072 22490 25373 35838
                 36807 38149 38174 39063 39299 39570 15805 19396
                 20526 (22557 . "도가니 과") (29459 . "긴 꼬리 원숭이 과")
                 (31262 . "보리 과") 20878 21685 22380 22446 23897
                 24663 24776 25956 26183 26869 27460 28612 29041
                 29328 30225 31803 31935 32182 33090 33212 33343
                 33834 35555 36411 37633 37945 39196 39611 39617
                 13316 13819 13964 14075 14401 14803 15001 15552
                 15585 15919 16171 16255 16295 16326 16828 16913
                 16918 17152 18024 18488 18668 19162 19422 19769))
       (44285 . ((37101 . "성곽 곽") (24275 . "둘레 곽, 클 곽")
                 (27112 . "덧널 곽, 궤 곽") (34303 . "콩잎 곽") (38669 . "빠를 곽")
                 (30312 . "곽란 곽") 28471 36521 38017 (38841 . "무두질한 가죽 곽")
                 23838 28694 (26881 . "관 곽") 30689 22670 27412 31831
                 38815 13797 15884 18660 18822 18896))
       (44288 . ((23448 . "벼슬 관") (35264 . "볼 관") (38364 . "빗장 관")
                 (39208 . "객사 관") (31649 . "피리 관, 대통 관") (20896 . "갓 관")
                 (23532 . "너그러울 관") (36011 . "꿸 관") (24931 . "버릇 관, 익숙할 관")
                 (27454 . "정성 관") (28748 . "물 댈 관") (20018 . "곶")
                 (33694 . "골풀 관") (29743 . "옥피리 관, 옥저 관") (26874 . "널 관")
                 (39028 . "광대뼈 관") (29912 . "옥 이름 관, 구슬 관")
                 (32592 . "두레박 관, 물동이 관") (33733 . "골풀 관, 왕골 관")
                 26785 33304 38306 (30437 . "대야 관") 35251 18298
                 27445 (31100 . "강신제 관") (37671 . "비녀장 관") 30237
                 23515 (32176 . "얽을 관") (20017 . "쌍상투 관") (40475 . "황새 관")
                 27596 (31590 . "피리 관") (31422 . "빌 관") (29215 . "봉화 관")
                 (38618 . "황새 관") (39638 . "허리뼈 관") 20492 31421
                 38359 (38005 . "두레박 관") 31030 36480 18830 23485
                 24815 25532 32510 35266 36143 39079 39302 39627
                 40563 (28075 . "끓을 관") (36648 . "줏대 관") 21341
                 24761 24762 25692 27148 27423 27456 27892 28485
                 28906 30191 30676 31396 32582 33924 35212 36055
                 36966 37830 37991 40057 40164 15265 15320 17363
                 17728 17862 17904 17982 18030 18749 18775 19050
                 19072 19596 19608))
       (44292 . ((25324 . "묶을 괄, 헤아릴 괄") (27292 . "전나무 괄, 나라이름 괄")
                 (21038 . "깍을 괄, 긁는 괄") (36866 . "빠를 괄") (24669 . "걱정 없을 괄, 깜짝 괄")
                 (32850 . "떠들석할 괄") (26653 . "노송나무 괄") (31560 . "하눌타리 괄")
                 (39674 . "머리 묶을 괄") 40505 (20344 . "힘 쓸 괄")
                 (40240 . "재두루미 괄") 21120 25046 26736 33856 37565
                 38946 39091 39712 17591 18415 18678 18842 19407
                 19450))
       (44305 . ((20809 . "빛 광") (24291 . "넓을 광") (29378 . "미질 광, 미칠 광")
                 (37979 . "쇳돌 광, 쇳덩이 광") (21281 . "바룰 광, 바를 광")
                 (26336 . "밝을 광, 빌 광") (22745 . "광 광, 텅빈 광")
                 (33009 . "오줌통 광, 방광 광") (31568 . "광주리 광")
                 (27960 . "물 용솟음할 광, 물솟을 광") (20362 . "성한 모양 광, 클 광")
                 (29654 . "옥피리 광") (28826 . "햇빚 뜨거울 광, 밝을 광")
                 (32086 . "솜 광") 31014 (35473 . "속일 광") (32394 . "솜 광")
                 (24647 . "겁낼 광") (29559 . "사나울 광") 25751 21179
                 30518 36811 30668 (30970 . "쇳돌 광") (26692 . "광랑나무 광")
                 28782 22425 (33578 . "초결명 광") 20697 25218 37543
                 16133 22329 24195 26103 29367 30719 32425 35795
                 35827 37021 37489 (26694 . "문테 광") (29196 . "불 빛 환할 광")
                 (35462 . "속일 광") 20423 20771 21289 21712 23023
                 25068 25069 26175 27949 28823 28827 28897 30783
                 30788 31340 31610 32075 33254 33257 36566 36589
                 36612 36891 37052 37178 37595 40287 40646 40651
                 14638 15061 15067 15984 16288 16352 16842 17071
                 17584 17841 19409))
       (44312 . ((25499 . "걸 괘") (21350 . "걸 괘, 점괘 괘")
                 (32619 . "줄 괘, 줄칠 괘") (25346 . "그림족자 괘") (35455 . "그르칠 괘")
                 (32611 . "걸 괘") 21902 32083 21593 35798 13726 (21692 . "입 비뚤어질 괘")
                 21479 21865 26548 26618 35074))
       (44340 . ((24618 . "기이할 괴") (24871 . "부끄러워할 괴, 부끄러울 괴")
                 (22750 . "무너질 괴") (22602 . "흙덩이 괴, 흙덩어리 괴")
                 (27088 . "홰나무 괴, 느티나무 괴") (20608 . "클 괴, 괴이할 괴")
                 (20054 . "어그러질 괴, 어긋날 괴") (39745 . "으뜸 괴, 괴수 괴")
                 (25296 . "속일 괴") (23231 . "창피줄 괴") 22730 22351
                 (24672 . "怪의 俗字") (29808 . "슬이름 괴") (33967 . "황모 괴")
                 34146 29900 38368 28896 (24293 . "여물광 괴") (35160 . "띠매듭 괴")
                 20983 27443 33197 22359 33929 (29853 . "옥돌 괴")
                 22692 24028 26077 26182 27334 30288 31625 31977
                 34366 35593 13598 13663 13864 13909 13965 14804
                 16065 16297 16559 16567 16630 16912 16923 17618
                 17625 17996 19427 19428))
       (44341 . ((39320 . "벨 괵") 34402 15745 34760 24151
                 24124 25524 34632 21096 21871 22035 24918 25681
                 28429 32861 33112 33173 16568 19214))
       (44361 . ((23439 . "클 굉, 넓을 굉") (32945 . "팔뚝 굉")
                 (36703 . "울릴 굉, 수레몰아가는 소리 굉") (32024 . "갓끈 굉")
                 (38286 . "마을문 굉") (35301 . "뿔잔 굉") 30817 40652
                 37728 37735 (35335 . "큰 소리 굉") (28004 . "용솟음할 굉")
                 27759 32707 32733 32430 36720 38205 38387 21431
                 21667 28217 31460 32139 32830 35317 36579 37651
                 38787 13978 16790 18508 18511 18880 19025 19779))
       (44368 . ((26657 . "학교 교") (20132 . "사귈 교") (25934 . "가르침 교, 가르칠 교")
                 (27211 . "다리 교") (24039 . "공교할 교, 교묘할 교")
                 (36611 . "견줄 교, 비교할 교") (37066 . "성 밖 교, 들 교")
                 (30703 . "바로잡을 교") (33184 . "아교 교") (32094 . "목맬 교")
                 (20689 . "높을 교, 붙어살 교") (23308 . "아리따울 교")
                 (25898 . "어지러울 교") 37237 (39509 . "교만할 교")
                 (21932 . "높을 교") (36686 . "가마 교") (29409 . "교활할 교, 간교할 교")
                 (30350 . "달빛 교, 달밝을 교") (34527 . "교룡 교, 도룡뇽 교")
                 (21676 . "새소리 교, 새지저귀는 소리 교") (23968 . "뾰족하게 높을 교, 산길험할 교")
                 (32761 . "꼬리 긴 깃털 교, 우뚝우뚝솟은 모양 교")
                 (39851 . "상어 교") (34126 . "메밀 교") (39171 . "경단 교")
                 25945 24508 (27063 . "외나무 다리 교") 30374 (22032 . "닭 울 교")
                 25357 (22169 . "깨물 교") 36679 (30973 . "메마른 땅 교")
                 23657 37496 (23011 . "예쁠 교") 31382 36475 (20348 . "예쁠 교")
                 36267 23930 37157 (25759 . "들 교") 24660 25963 29651
                 31322 (22125 . "부르짖을 교") (24973 . "교만할 교")
                 33581 34780 35665 (26184 . "달 밝을 교") 20748 22717
                 28438 29535 30985 31372 31781 36324 36474 38845
                 38941 40430 14191 15744 19366 21870 20052 20392
                 23047 23780 25378 25605 26725 30699 30807 32478
                 32728 33630 36343 36735 36739 37036 38128 38802
                 39290 39556 40091 19732 (22020 . "부르짖을 교") (26270 . "밝을 교")
                 27207 39609 40257 40809 19970 21835 22018 22054
                 22074 22107 23315 23362 23659 23783 25785 25789
                 25981 25983 26322 27095 28376 28762 28868 29126
                 29342 29868 30810 30980 32337 32400 33259 33613
                 34400 34754 35685 36043 36268 36365 36488 37115
                 37153 37896 39644 40014 40245 40397 13953 13987
                 14013 14352 14487 14605 14637 15157 15422 15586
                 16018 16241 16242 16393 16394 16554 16557 16605
                 17226 17303 17960 18089 18504 18578 18602 18782
                 19521 19646))
       (44396 . ((20061 . "아홉 구") (21475 . "입 구") (27714 . "찾을 구, 구할 구")
                 (21477 . "글귀절 구") (25937 . "구원할 구") (31350 . "궁리할 구")
                 (20037 . "오랠 구") (33290 . "옛 구") (20855 . "갖출 구")
                 (33503 . "진실로 구, 풀 구") (20465 . "함께 구") (21312 . "감출 구, 지경 구")
                 (19992 . "언덕 구") (25304 . "잡을 구") (29699 . "옥경쇠 구, 구슬 구")
                 (27083 . "집세울 구") (25084 . "두려워할 구") (39493 . "몰 구")
                 (29399 . "개 구") (40860 . "나라 이름 구, 거북 구")
                 (37041 . "언덕 구") (27472 . "토할 구") (40169 . "비둘기 구")
                 (36092 . "살(구입할) 구") (40407 . "갈매기 구") (29590 . "옥 다음가는 돌 구, 옥돌 구")
                 (23495 . "도둑 구, 불한당 구") (20167 . "짝 구, 원수 구")
                 (30697 . "곡척 구") (28317 . "개천 구") (33285 . "외삼촌 구")
                 (34914 . "네거리 구, 갈림길 구") (36544 . "몸 구") (37476 . "갈고랑쇠 구, 갈구리 구")
                 (24272 . "마구간 구") (26601 . "관 구") (28792 . "지질 구, 구울 구")
                 (35635 . "읊조릴 구, 노래 구") (22434 . "때 구, 더러울 구, 먼지 구")
                 (39378 . "망아지 구") (22036 . "토할 구, 게워내는 구")
                 (33276 . "절구 구") (23943 . "산험준할 구") (27590 . "때릴 구, 구박할 구")
                 (26552 . "구기자 구, 굽을 구") (21646 . "허물 구") (32777 . "늙을 구, 늙은이 구")
                 (21246 . "글귀절 구") (27628 . "제기 구") (22389 . "丘의 俗字, 언덕 구")
                 26231 (30655 . "놀라서 볼 구, 놀라서 보는 구") (36881 . "짝 구")
                 (32127 . "급박할 구, 급할 구") (26622 . "관 구, 널 구")
                 (37558 . "끌 구") 24807 (25606 . "얽어맬 구") (25187 . "두드릴 구")
                 26087 (35032 . "갓옷 구") (35436 . "꾸짖을 구") (30106 . "오랜 병 구")
                 (36952 . "만날 구") (23230 . "겹사돈 구") (23656 . "삼으로 짠 신 구")
                 21417 20038 25978 (23255 . "할미 구") (21164 . "수고로울 구")
                 (25412 . "흙파올릴 구") (39368 . "驅와 同字") (29964 . "사발 구")
                 (25715 . "치마걷어들 구") (24271 . "마구간") (31414 . "가난하여 예절없을 구")
                 32775 (24384 . "활 잔뜩 당길 구") (39094 . "사방풍 구")
                 27807 33246 32241 29629 38602 (28442 . "물거품 구")
                 (38893 . "부추 구") (40476 . "구관조 구") 32776 (20660 . "구부릴 구")
                 21620 (27032 . "곡척 구") (31959 . "마른밥가루 구")
                 (34479 . "지렁이 구") 35247 19127 (30319 . "여윌 구")
                 20534 36053 20096 (38877 . "깍지 구") 20902 14449
                 (23012 . "만날 구") (21302 . "柩의 古字") (29830 . "옥경쇠 구")
                 (37350 . "그릇에 금테 두를 구") 25221 27661 35283
                 35389 (40391 . "새 새끼 구") 36515 38834 (20421 . "공손할 구")
                 23520 26887 27432 31468 32071 37390 15034 15492
                 16786 (20317 . "꼽추 구") (23715 . "산이름 구") (26155 . "해돋아 따뜻할 구")
                 (31709 . "불덮개 구") (32970 . "포(얇게 저미어서 말린 고기) 구")
                 31537 (22392 . "垢와 同字") 27991 34682 35104 14552
                 16219 (20883 . "재목 어긋매겨 쌓을 구") 20297 26425
                 26754 27843 29323 33405 33462 34098 34107 15600
                 16053 18004 21306 26500 27431 27572 38057 20251
                 21589 22815 22954 23654 23702 24047 25248 27812
                 29935 30477 31405 32529 35279 35764 35807 36141
                 36167 36527 38022 38404 39123 39366 39537 39545
                 40480 40485 40498 40863 20619 (21433 . "세모창 구")
                 (21492 . "소리 높일 구") (25141 . "창 구") (30144 . "곱사등이 구")
                 (31387 . "노략질할 구") (33951 . "구장 구") (38894 . "韭와 同字")
                 (39726 . "제비뽑을 구") 20000 20009 20483 21118 21174
                 21244 21267 21275 22503 22816 22906 23484 23732
                 24048 24260 24388 24514 24592 24883 24934 25167
                 25629 25728 25922 26026 26384 26470 26709 27403
                 27405 27535 27747 28401 28744 29049 29104 29216
                 29278 29360 30018 30387 30426 30487 30601 30616
                 31448 31576 31750 31804 31874 32012 32721 32869
                 32909 33049 33170 33295 33577 33819 33867 34034
                 34322 34492 34551 34871 34876 34983 35305 35332
                 35333 35967 36308 36581 37037 37331 37338 38010
                 38265 38895 39816 39876 39910 39992 40206 40221
                 40596 40745 40765 40808 13312 13340 13348 13506
                 13594 13668 13749 13781 13847 13875 13888 14001
                 14031 14137 14156 14180 14231 14280 14426 14649
                 14824 14860 15148 15197 15414 15491 15512 15563
                 15778 15912 16003 16043 16100 16416 16514 16595
                 16660 16723 16942 17001 17030 17077 17127 17439
                 17510 17659 18158 18218 18285 18340 18412 18421
                 18498 18631 18715 18802 18864 18914 18929 18934
                 19240 19250 19331 19479 19621 19627 19662 19705
                 19708 19830 19866))
       (44397 . ((22283 . "나라 국") (23616 . "부분 국, 판 국")
                 (33738 . "국화 국") (38816 . "기를 국") (38827 . "문초받을")
                 (40628 . "누룩 국") 22269 (21258 . "움켜 뜰 국") 22255
                 (36348 . "구부릴 국") (25516 . "두 손으로 움킬 국")
                 (40623 . "麴과 同字") 36376 38320 (22231 . "둘러쌀")
                 26888 40308 38164 20407 22262 22267 22272 23109
                 24008 25398 26798 27275 27625 27633 27878 28119
                 28951 31927 34332 35530 36252 36537 36610 37606
                 38513 39414 39527 13874 14668 14964 15996 16228
                 16784 17136 17316 17393 17774 17863 18223 18526
                 19093 19553))
       (44400 . ((36557 . "군사 군") (21531 . "임금 군") (37089 . "고을 군")
                 (32676 . "무리 군") (31384 . "군색할 군, 급할 군") (35033 . "치마 군")
                 32675 (25411 . "주울 군") 38965 28050 (30392 . "얼어터질 군")
                 (26750 . "고욤나무 군") 20891 30386 40106 20400 20690
                 23469 24108 25864 25887 29690 30393 33689 34592
                 35040 36625 37717 39862 40280 13733 14986 15659
                 16335 18253 19325))
       (44404 . ((23624 . "굽을 굴") (31391 . "구멍 굴") (25496 . "팔 굴")
                 (22528 . "굴뚝 굴") (23835 . "산 높을 굴") (35416 . "굽을 굴")
                 30715 (20500 . "딱딱할 굴") 26383 (28104 . "흐릴 굴")
                 32976 35790 23851 24353 28992 34419 34982 35507
                 36233 38268 40332 13319 13341 14975 14978 15230
                 15712 16085 16535 17627 17830 17983 18359 18439))
       (44415 . (13708))
       (44417 . ((24339 . "활 궁") (23470 . "굴궐 궁, 궁궐 궁")
                 (31406 . "다할 궁") (36524 . "몸 궁") (31353 . "높을 궁")
                 (33422 . "궁궁이 궁") 23467 31351 34285 21268 (36531 . "躬의 本字")
                 26459 31430 13387 21265 23430 28970 29069 15654
                 17622 18154 18491 18717))
       (44425 . (40204))
       (44428 . ((27402 . "권세 권") (21367 . "굽을 권, 책 권")
                 (21240 . "권할 권") (21048 . "엄쪽(어음을 쪼갠 한 쪽) 권, 문서 권")
                 (25331 . "주먹 권") (22280 . "둥글 권") (30519 . "돌아볼 권")
                 (20518 . "게으를 권") (25458 . "거둘 권, 주먹쥘 권")
                 (28099 . "물돌아 흐를 권, 물도는 모양 권") (24787 . "삼갈 권")
                 21223 (32163 . "정다울 권") (30560 . "眷과 同字") (34615 . "움추러질 권")
                 21173 24059 (26860 . "휘어만든 나무그릇 권") 36385
                 39688 (21196 . "倦과 同字") 24099 16645 21149 26435
                 27177 32507 38185 21228 21875 22287 22498 22854
                 23128 23369 24015 24955 25660 26698 26918 29302
                 29320 31646 32109 33764 34088 34375 34872 35063
                 35296 37640 38863 39148 39179 39731 40804 13501
                 14305 14503 14962 15035 15162 16656 16730 17038
                 17199 17487 17837 18386 18480))
       (44432 . ((21413 . "그 궐") (38357 . "대궐 궐") (36470 . "우물벌레 궐, 쓰러질 궐")
                 (34152 . "고사리 궐") (29527 . "도둑 일어날 궐, 뛰놀 궐")
                 21122 27227 36471 27228 40022 27438 28749 34792
                 40418 38242 38425 40156 25733 (20101 . "갈고리")
                 22104 23378 23379 23969 23973 24992 27666 30234
                 34793 37917 13933 15696 18016))
       (44452 . ((36556 . "수레바퀴 사이 궤, 바퀴굴데 궤")
                 (36470 . "급히걸을 궤") (28528 . "흩어질 궤, 무너뜨릴 궤")
                 (26426 . "느티나무 궤, 책상 궤") (35437 . "속일 궤, 간사할 궤")
                 (27331 . "상자 궤, 궤짝 궤") (20960 . "안석 궤") (39243 . "먹일 궤")
                 (36330 . "끓어앉을 궤") (21297 . "갑 궤") (27195 . "가마테나무 궤")
                 (31755 . "대제기 궤") (39229 . "饋와 同字") (24978 . "심란할 궤")
                 31779 21122 (32354 . "톱끝 궤") 18500 21293 (38368 . "저자문 궤")
                 (20345 . "포갤 궤") 14826 22429 35300 37888 38482
                 39304 21286 21294 24870 28291 31697 32523 35809
                 36712 38419 15470 (27711 . "물가 궤") (40578 . "큰고라니 궤")
                 25733 20967 21420 22298 23037 23303 24657 25905
                 27560 31082 31746 31812 34094 34539 37966 38844
                 40037 13618 13619 13946 14246 16637 16746 17263
                 17754 18473 18624 18642 18725))
       (44480 . ((36020 . "높을 귀, 귀할 귀") (21477 . "글귀 귀")
                 (27512 . "돌아올 귀") (39740 . "귀신 귀") (40860 . "거북 귀")
                 (26231 . "시각 귀, 해그림자 귀") 24112 18588 23428
                 24011 30646 21132 25740 15445 22067 23599 33091
                 34348 39465 18303 21055 24402 30598 36149 26914
                 27126 27383 30344 32877 15925 16475 17158 18017
                 18312 19470))
       (44508 . ((35215 . "그림쇠 규, 법 규") (22862 . "별 규")
                 (22317 . "서옥 규") (31998 . "삼합노 규, 살필 규") (21483 . "부를 규, 부르짖을 규")
                 (25542 . "헤아릴 규") (29674 . "圭와 同字, 서옥 규")
                 (38312 . "색시 규, 안방 규") (31418 . "엿볼 규") (33909 . "해바라기 규, 아욱 규")
                 (36917 . "구거리 규, 큰길 규") (30789 . "유리만드는 흙 규, 규소 규")
                 32326 30054 (31429 . "구멍 규, 엿볼 규") (36211 . "헌걸찰 규, 위엄스러울 규")
                 (27131 . "물푸레나무 규") (34412 . "虯와 同字") 21580
                 (36332 . "한발자국 규") (30589 . "어그러질 규") (34415 . "뿔없는 용 규")
                 (38362 . "갸웃이 볼 규") (27163 . "나무가지 휘어질 규")
                 16531 24011 (31994 . "거둘 규") (26959 . "호미자루 규")
                 (21042 . "찌를 규") (23296 . "성 규") 28505 (26252 . "해질 규")
                 (38925 . "고깔비녀 규") (39319 . "아홉거리 규") 25123
                 28995 (37053 . "고을이름 규") 25678 27132 39853 24664
                 33444 38959 22955 27817 23743 31373 31397 32416
                 35268 38394 (23266 . "가는 허리 규") 20067 21945
                 23215 24695 25707 26427 26909 26961 28224 28296
                 29354 29580 30625 31376 32036 32871 33023 33409
                 33573 34248 34672 35007 35334 36446 37068 37128
                 37736 37751 38916 39734 39737 40650 40764 13907
                 14787 14866 16709 16827 17839 18037 18344 18424
                 18449 18694 19411 19589 19663 19691 19887))
       (44512 . ((22343 . "고를 균") (33740 . "버섯 균") (40860 . "(피부가)틀 균, 거북 균")
                 (37406 . "서른 근 균") (31584 . "대껍질 균, 속빈 대 균")
                 (21243 . "고를 균") (30023 . "밭개간할 균") 21248 (22263 . "둥근곳집 균")
                 31259 (40591 . "고라니 균") 40597 34944 38055 21569
                 23790 25251 27758 30853 31640 31647 34448 35232
                 37505 37534 40583 13996 14290 16889 17411))
       (44516 . ((27224 . "귤나무 귤") 36459 18702 19500))
       (44537 . ((26997 . "가운데 극, 지극할 극") (20811 . "이길 극")
                 (21127 . "심할 극") (38553 . "틈 극") (26840 . "가시나무 극")
                 (25119 . "갈래진 창 극") (21067 . "깍일 극, 이길 극")
                 (20127 . "빠를 극") 27547 (23632 . "나막신 극") 37092
                 (37060 . "隙과 同字") 14776 33549 37620 38559 14211
                 34112 14683 21095 (23557 . "剋과 同字") 20014 21184
                 21194 23060 25760 27254 27690 35027 35147 13836
                 15603 16119 17639 18893 19055 19074))
       (44540 . ((36817 . "가까울 근") (26681 . "뿌리 근") (21220 . "부지런할 근")
                 (35641 . "삼갈 근") (26020 . "근 근") (20677 . "겨우 근")
                 (31563 . "힘줄 근") (39241 . "주릴 근, 푸성귀흉년드는 근")
                 (29822 . "붉은옥 근") (27135 . "무궁화 근") (35250 . "뵈울")
                 (25027 . "은근할 근") (33465 . "미나리 근") (33771 . "오랑캐 꽃 근, 쓴나물 근")
                 (21156 . "힘많을 근, 힘셀 근") (36319 . "발두꿈치 근")
                 (38771 . "아낄 근") 22535 (21370 . "합환주잔 근") 34308
                 13862 (21418 . "적을 근") 27555 (35284 . "筋과 同字")
                 24940 26035 (22672 . "진흙 근") (24057 . "받들 근")
                 (37375 . "대패 근") (24273 . "작은집 근") (28428 . "맑을 근")
                 16957 35280 35880 39313 20242 21711 23268 25028
                 27471 30269 33621 33766 34035 34748 36222 39352
                 14203 14252 14489 14790 14903 15255 15951 16933
                 17165 17594 18746))
       (44544 . ((22865 . "부족 이름 글, 계약할 글") 35350
                 13326 13589 18328 19508))
       (44552 . ((37329 . "성 금, 쇠 금") (20170 . "이제 금")
                 (31105 . "금할 금") (37670 . "비단 금") (31165 . "날짐승 금, 새 금")
                 (29748 . "거문고 금") (35167 . "옷깃 금, 옷섶 금") (34942 . "이불 금")
                 (25810 . "사로잡을 금") (34943 . "옷깃 금, 옷고름 금")
                 37392 (33449 . "풀 이름 금, 금풀 금") (27278 . "능금나무 금")
                 (26129 . "밝을 금") (22935 . "외숙모 금, 싱글벙글 금")
                 (22116 . "입 다물 금") (23956 . "높고 험할 금") (31506 . "첨대 금")
                 (21803 . "입 다물 금") 32031 29753 38985 33779 36218
                 15346 23898 38021 38182 (40645 . "누른 빛 금") 20728
                 20942 20954 22105 22341 24200 24781 25202 25446
                 25607 28607 29665 29755 30747 32825 32931 34457
                 34820 36577 37330 37401 37698 37922 38594 38770
                 40185 40301 40829 13643 14743 14930 14977 15432
                 15485 15504 16291 16610 17134 17181 17508 17719
                 17971 18757 18758 18854 19188 19862))
       (44553 . ((32102 . "넉넉할 급, 줄 급") (24613 . "급할 급")
                 (21450 . "미칠 급") (32026 . "등급 급") (27762 . "길을 급, 물길을 급")
                 (25201 . "미칠 급, 취급할 급") (20235 . "속일 급, 생각할 급")
                 (23692 . "높을 급") (30336 . "고소할 급") (31496 . "책상자 급")
                 (30991 . "산 높은 모양 급") (22334 . "위태할 급") (33448 . "말오줌나무 급")
                 24438 32423 32473 21948 23302 24547 31000 35289
                 40401 64036 14594 14784 15140 16250 17372 17376
                 19631))
       (44557 . ((32943 . "옳이 여길 긍, 즐기어할 긍") (20834 . "삼갈 긍, 조심할 긍")
                 (30684 . "불쌍히 여길 긍, 자랑할 긍") (20120 . "뻗칠")
                 30685 (20121 . "뻗칠 긍") 22569 32090 32234 32910
                 (27537 . "까무라칠 긍") 21039 22879 25519 25583 25604
                 32262 32955 35075 37689 40566 14692 15251 19533
                 19534 19565 19572))
       (44592 . ((24049 . "자기 기, 몸 기") (35352 . "기록할 기")
                 (27683 . "기운 기") (25216 . "재주 기") (36215 . "일어날 기")
                 (26399 . "기약할 기") (22522 . "터 기") (20854 . "그 기")
                 (26083 . "이미 기") (24190 . "기미 기, 몇 기") (35912 . "어찌 기, 일찍 기")
                 (22120 . "그릇 기") (22855 . "기이할 기") (27231 . "틀 기, 베틀 기")
                 (26071 . "기 기") (32000 . "벼리 기") (26820 . "버릴 기")
                 (24524 . "꺼릴 기") (20225 . "꾀할 기, 계획할 기") (27450 . "속일 기")
                 (39138 . "주릴 기") (31048 . "빌 기, 기도할 기") (20062 . "줄 기")
                 (39438 . "말탈 기") (23492 . "부칠 기, 붙여살 기") (30079 . "경기 기")
                 (20864 . "바랄 기") (31637 . "키 기") (32774 . "늙은이 기")
                 (23696 . "갈림길 기, 높을 기") (27778 . "물 이름 기")
                 (39525 . "천리마 기") (29734 . "옥 이름 기") (40594 . "기린 기")
                 (29859 . "구슬 기, 작은구슬 기") (29738 . "옥 기, 옥이름 기")
                 (39439 . "털총이 기, 천리마 기") (28103 . "강 이름 기, 물이름 기")
                 (27773 . "김 기, 물끓는김 기") (35663 . "나무랄 기")
                 (22931 . "기생 기") (32648 . "굴레 기, 말 굴레 기")
                 26426 (21980 . "즐길 기") (32186 . "비단 기, 무늬놓은 비단 기")
                 (26462 . "나무 이름 기, 구기자 기") (30072 . "뙈기 밭 기, 기이할 기")
                 (26398 . "돌 기") (32908 . "살 기") (26827 . "바둑 기")
                 (20238 . "재주 기") (23822 . "험할 기, 산길험할 기")
                 (26547 . "해할 기, 사타구니 기") 20960 (35244 . "바랄 기")
                 (22331 . "경기 기, 지경 기") (39249 . "주릴 기, 흉년들 기")
                 (31047 . "토지의 신 기, 편안할 기") (22804 . "조심할 기, 외발짐승 기")
                 (31041 . "성할 기, 성하게 기") (30959 . "물가 기, 물속자갈 기")
                 (31098 . "복 기, 길할 기") (30849 . "棋와 同字, 바둑 기")
                 (37665 . "솥 기, 세발가마 기") (29592 . "패옥 기") (29826 . "피변 꾸미개 기, 고깔꾸미개 기")
                 (22524 . "崎와 同字, 언덕머리 기") (32813 . "갈 기, 밭갈 기")
                 (37668 . "호미 기") (27495 . "岐와 同字") (24323 . "棄의 古字")
                 (26305 . "및 기") 33425 31442 (26050 . "기 기") 35210
                 27449 14444 (38926 . "헌걸찰 기") (22669 . "맥질할 기")
                 (26826 . "棋와 同字") 26081 (39696 . "갈기 기") (34787 . "서캐 기")
                 (36290 . "육발 기") (21086 . "새김칼 기") (25486 . "끌 기")
                 (34308 . "풀 이름 기") (25906 . "기울기 기") (33456 . "세발 마름 기")
                 29449 (32166 . "연두빛 비단 기") 24782 (24558 . "해칠 기")
                 36349 (23674 . "민둥산 기") 38847 (24877 . "공손할 기")
                 24638 (33450 . "단너삼 기") 30869 36390 (28801 . "기운 기")
                 19980 24475 26082 31141 34359 34937 (32647 . "나그네 기")
                 (35209 . "구속할 기, 말 굴레 기") 20507 24234 25083
                 26280 27071 13928 14291 17771 (20699 . "취하여 춤추는 모양 기")
                 (24203 . "시렁 기") (31144 . "조짐 기") (32949 . "적대 기")
                 (34369 . "夔와 同字") 20862 21001 22064 22096 22600
                 25959 31306 31318 33262 34449 34452 34590 35309
                 36536 39743 39756 13985 14669 17595 18027 18574
                 19463 39269 21501 23682 26724 (27668 . "기운") 27671
                 29585 30710 32426 32494 32641 34162 34414 35274
                 35749 35760 38172 39040 39568 39569 39589 40062
                 40111 40141 (26275 . "볕기운 기") (32165 . "비단 쑥색 기")
                 (34589 . "방계 기") (39981 . "지느러미 기") 38545 40662
                 20115 20703 21007 21080 21594 21609 21614 21805
                 22358 22817 22912 23096 23117 23699 23900 24122
                 25489 25904 26782 27107 27181 27313 27328 28175
                 28336 28637 28943 29570 30400 30453 31256 31273
                 31754 31791 31793 31823 31928 32168 33793 34055
                 34244 34318 34458 34727 35071 35072 35467 35525
                 35526 36369 36520 36573 37012 37183 37910 38509
                 39742 39765 39778 39893 40000 40267 40312 40320
                 40814 64017 13483 13705 13899 14011 14214 14227
                 14235 14255 14271 14298 14306 14348 14385 14446
                 14675 14899 14971 14973 15045 15059 15095 15334
                 15391 15462 15545 16132 16246 16392 16582 16654
                 16683 16754 16875 17019 17119 17307 17408 17524
                 17605 17643 17661 17671 17764 17857 17947 18135
                 18164 18296 18330 18375 18394 18507 18571 18611
                 18706 18771 19027 19053 19151 19318 19324 19682
                 19870))
       (44596 . ((32202 . "굳게 얽힐 긴, 요긴할 긴") 32039
                 33763))
       (44600 . ((21513 . "길할 길") (25326 . "일할 길, 열심히일할 길")
                 (20342 . "건장할 길, 바를 길") (26708 . "도라지 길")
                 27905 36236 (23006 . "성 길") 40246 40082 (34531 . "장구벌레 길")
                 21677 29412 37062 37537 38794 39834 13342 13343
                 14559 17600 18380))
       (44608 . ((37329 . "사람의 성 김, 쇠 김")))
       (45143 . (13725))
       (45149 . (21788))
       (45181 . ((21931 . "마실 끽") 22084))
       (45208 . ((32645 . "새 그물 나") (37027 . "어찌 나")
                 (22856 . "어찌 나") (25343 . "붙잡을 나") (35064 . "벌거벗을 라, 털 없는 벌레 라, 벌거숭이 나")
                 (25078 . "게으를 나") (20794 . "역귀 쫓을 나, 법도있는 나")
                 (25062 . "나약할 나, 부드러울 나") (34746 . "소라 나")
                 (25295 . "붙잡을 나") (37007 . "돌 나") (30313 . "약물 중독 나")
                 (34367 . "소나무겨우살이 나") (21895 . "나팔 라, 라마교 라")
                 (23068 . "아리따울 나, 휘청거릴 나") (25360 . "붙잡을 나")
                 (25386 . "옮길 나") (31983 . "찰벼 나") 37069 30728
                 31332 20649 38222 (26779 . "나무 이름 나") (31973 . "糯와 同字")
                 20088 21738 22001 25063 27232 31276 33944 35517
                 37823 13402 13614 13728 14405 14833 15385 18132))
       (45209 . ((33853 . "떨어질 낙") (27138 . "즐길 낙") (35582 . "대답할 낙, 허락할 낙")
                 (27931 . "강 이름 낙") (39409 . "낙타 낙") (28889 . "지질 낙")
                 (37226 . "타락 낙") (29662 . "구슬 목걸이 낙") 35834
                 25535))
       (45212 . ((38627 . "어려울 난") (26262 . "따뜻할 난")
                 (21365 . "알 난") (20098 . "어지러울 난") (34349 . "난초 난")
                 (27396 . "난간 난") (29211 . "문드러질 난") (40478 . "난새 난")
                 (29014 . "따뜻할 난, 더울 난") (36199 . "얼굴 붉힐 난")
                 15113 (29015 . "터울 난") 38590 (20548 . "언약할 난")
                 (39210 . "풀보기 잔치 난") 22907 25089 28188 15089
                 15150 17313))
       (45216 . ((25466 . "누를 날, 도장찍을 날") (25423 . "이길 날, 찍을 날")
                 (28037 . "앙금 흙 날") 38503 33526 17811 17831))
       (45224 . ((21335 . "남녘 남") (30007 . "사내 남") (28651 . "퍼질 남")
                 (34253 . "쪽 남") (23888 . "남기 남") (26976 . "녹나무 남, 들메나무 남")
                 (26511 . "녹나무 남, 매화나무 남") (28275 . "강 이름 남, 물이름 남")
                 (35172 . "누더기 남") 23066 (21891 . "재잘거릴 남")
                 (26591 . "枏의 俗字") 25543 13426 20413 23163 26260
                 29045 30040 32625 33129 33710 33843 34683 35573
                 36950 13523 14574 15223 15708 16214 16914 17046))
       (45225 . ((32013 . "바칠 납, 들일 납") (25289 . "꺽을 납, 꺾을 납")
                 (33240 . "납향 납") (34847 . "밀 납") (34930 . "기울 납, 장삼 납")
                 33971 37385 32435 38048 22944 31517 36572 38777
                 39798 14885 16939 19087))
       (45229 . ((28010 . "물결 낭") (23064 . "아가씨 낭, 젊은계집 낭")
                 (24266 . "복도 낭") (29436 . "이리 낭") (22218 . "주머니 낭")
                 (26391 . "밝을 낭") (37070 . "사나이 낭") (26345 . "접때 낭")
                 22178 20074 20798 22228 25795 25902 27420 28770
                 39266 40777 13954 15774 16504))
       (45236 . ((20839 . "안 내") (20358 . "올 내") (20035 . "이에 내")
                 (32784 . "견딜 내") (22856 . "어찌 내") (26608 . "능금나무 내, 어찌 내")
                 20869 (36858 . "이에 내") (40720 . "가마솥 내") (21250 . "향내 내")
                 (23341 . "젖 내") 24316 15247 (22902 . "젖 내") 35110
                 15240 25734 32783 37346 23419 25704 27670 27677
                 28223 29067 30099 33816 34714 37692 13371 14007
                 14120 15174 15521 15630 16269 19438 19550 19598))
       (45257 . ((20919 . "차가울 냉, 찰 냉")))
       (45265 . (36419 36925))
       (45285 . (23064 23363))
       (45376 . ((22899 . "계집 녀") 17096 35005 38037 24100
                 37369 15702 17963))
       (45377 . ((24772 . "마음 졸일 녁") (30098 . "병질") 19270))
       (45380 . ((24180 . "해 년") (25754 . "비틀 년, 잡을 년")
                 (31178 . "年의 本字, 해 년") (30910 . "맷돌 년") 28042
                 16693 21718 23017 31760 33036 36494 40263 40752
                 15127 15263 16685))
       (45384 . ((28037 . "죽을 녈") 22332 31710 22824 13711
                 13887 16572))
       (45392 . ((24565 . "생각 념") (24684 . "편안할 념") (25288 . "집을")
                 (25467 . "비틀 념, 찍을 념") 39822 22493 28990 33356
                 36375 13419 18900 19247))
       (45393 . (38007 21926 24791 19670 24071 24137 25948
		       37410 37481 37660 13972 14966 15690 16867 17200
		       17294))
       (45397 . ((23527 . "편안할 녕") 29552 (23511 . "편안할 녕, 차라리 녕")
                 (20318 . "재주 녕") 20395 23509 (28632 . "진흙 녕")
                 23516 29999 (22144 . "정녕할 녕") 25840 32889 21659
                 25319 32845 (20764 . "고달플 녕") 23331 23501 27235
                 27320 28573 30659 34228 37967 39713 40459 14165
                 14583 15500 15536 16358 16808 17709 17919 19298))
       (45400 . (27026))
       (45404 . (31152 31074 16690))
       (45432 . ((32769 . "늙을 노, 늙은이 노") (36335 . "길 노")
                 (21214 . "수고할 노, 일할 노") (24594 . "성낼 노") (38706 . "이슬 노")
                 (22900 . "종 노") (21162 . "힘쓸 노") (30439 . "밥그릇 노")
                 (39791 . "미련할 노, 노둔할 노") (29200 . "화로 노, 화초 노")
                 (39377 . "둔할 노, 노둔한 말 노") (34310 . "갈대 노")
                 (40442 . "해오라기 노") (34396 . "포로 노") (25796 . "사로잡을 노")
                 (24361 . "쇠뇌(여러 개의 화살이나 돌을 잇따라 쏘게 된 큰 활) 노")
                 (24081 . "처자식 노") (27347 . "큰 방패 노") (29785 . "마노 노, 옥돌 노")
                 (23397 . "자식 노") (21622 . "들랠 노") 30766 24025
                 (29489 . "못할 노") (31535 . "새장 노") 24595 (33233 . "팔마디 노")
                 37989 38245 39549 18215 (23793 . "산이름 노") 20246
                 20270 22802 24014 29567 31913 33004 34543 35401
                 40209 13986 14250 16001 16028 18141 18176))
       (45433 . ((32160 . "초록빛 녹") (37636 . "기록할 녹")
                 (31103 . "행복 녹") (40575 . "사슴 녹") (30860 . "푸른 빛 녹, 돌모양 녹")
                 (33737 . "조개풀 녹") 20617 25625))
       (45436 . ((35542 . "말할 논") 23273 15310))
       (45440 . (13328 13335))
       (45448 . (13704))
       (45453 . ((36786 . "농사 농") (24324 . "희롱할 농") (31840 . "대그릇 농")
                 (28611 . "짙을 농") (33215 . "고름 농") (32894 . "귀머거리 농")
                 (22751 . "밭두둑 농, 언덕 농") (31328 . "후한 제사 농")
                 (20738 . "나 농") (37298 . "텁텁한 술 농") 31151 30289
                 20396 20892 21725 27987 31230 33043 15798 (22117 . "달게 먹을 농")
                 23977 27266 27393 29174 32375 34173 35163 36787
                 39710 40776 18038 18569 19804))
       (45460 . (38635))
       (45468 . (35965 35968))
       (45516 . ((38647 . "우뢰 뇌, 우리 뇌") (33126 . "머리 뇌, 머릿골 뇌")
                 (24817 . "괴로워할 뇌, 번뇌할 뇌") (36034 . "뇌물 뇌, 뇌물줄 뇌")
                 (29282 . "가축우리 뇌") (30922 . "돌이 많이 쌓인 모양 뇌, 돌무더기 뇌")
                 (39186 . "굶길 뇌") 33075 39207 25404 25017 22452
                 24700 24745 33041 39297 21272 22550 23070 25468
                 30895 33095 39870 39896 13390 14068 16143))
       (45544 . ((28346 . "우줌눌 뇨") (23615 . "오줌 뇨") 25747
                 (39719 . "시끄러울 뇨") 27208 38297 (23243 . "날릴 뇨")
                 (35018 . "끄을 뇨") (28118 . "진흙 뇨") (37891 . "작은 징 뇨")
                 35117 35658 (23346 . "샘낼 뇨") 18010 34949 38105
                 38393 23304 30944 23248 23325 30791 30837 33074
                 13471 14133 14233 14369 15598 16617 16629 18826
                 19341 19715))
       (45572 . ((27155 . "다락 누") (28431 . "샐 누") (28122 . "눈물 누")
                 (32047 . "포갤 누, 묶을 누") (23650 . "(빛이 들어오는)창 누")
                 (38475 . "줍을 누, 좁을 누") (22744 . "진 누") (32311 . "자세할 누, 실 누")
                 (32808 . "김맬 누") 14149 37778 27080 27325 21826
                 29555 32698 35688 35699 37918 14092 14201 16758
                 18994))
       (45576 . ((23273 . "언약할 눈, 연약할 눈") 23174 23280
                 14107 14442 15783))
       (45580 . ((35365 . "말 더듬거릴 눌, 말을 더듬는 눌")
                 21584 (21558 . "말 더듬을 눌") (32941 . "물개 눌") 35767
                 25232 13679 18396))
       (45684 . ((32016 . "잡색비단 뉴, 단추 뉴") (26491 . "옷 부드러울 뉴, 싸리 뉴")
                 29379 (24568 . "익을 뉴") 37397 25197 17140 32445
                 38062 (38773 . "단추 뉴") 22942 27793 28804 31880
                 33701 13755 16050 16519 17364))
       (45685 . (24679 26386 24568 (34884 . "코피 뉵") 34882
		       17825 17936 18108 19850))
       (45713 . ((21202 . "굴레 늑") (32907 . "갈비대 늑") 16734
                 17322))
       (45728 . ((20956 . "찰 늠")))
       (45733 . ((33021 . "능할 능") (38517 . "능 능, 큰언덕 능")
                 (20940 . "얼음 능, 능가할 능") (32190 . "무늬 놓은 비단 능")
                 (33777 . "마름 능") (31260 . "모질 능") 17403 15664
                 17925))
       (45734 . (33723))
       (45768 . ((27877 . "막힐 니, 진흙 니") (23612 . "여승 니, 여자중 니")
                 20320 31152 (24617 . "부끄러워할 니") (33193 . "살찔 니")
                 26062 (21602 . "소근거릴 니") 22510 (26565 . "얼레자루 니")
                 28628 33249 22963 27319 34477 36833 31074 33147
                 38092 20273 20535 22381 22958 23636 24744 25267
                 29396 31196 32978 33512 36316 37422 37480 37960
                 38572 39324 14270 14736 17765 17958 18127 18266))
       (45769 . ((28346 . "빠질 닉") (21311 . "숨쉴 닉, 숨을 닉")
                 25638 23263 31953 23354 24885 25659 27708 14718
                 15547))
       (45776 . ((26165 . "가깝게 할 닐") (26289 . "칠근할 닐")
                 16485 30150 17932 19793 19794))
       (45784 . (25328 17147 17214))
       (45785 . (13853 22241))
       (45796 . ((22810 . "많을 다") (33590 . "차풀 다") (29241 . "아비 다")
                 22002 22811 33572 35312 36322 39930 19138 19436))
       (45800 . ((21934 . "홀로 단") (30701 . "짧을 단") (31471 . "끝 단")
                 (20294 . "다만 단") (20025 . "붉을 단") (26039 . "끊을 단")
                 (27573 . "층계 단") (22296 . "둥글 단") (22727 . "제터 단")
                 (27264 . "향나무 단, 박달나무 단") (26086 . "밝을 단, 아침 단")
                 (28237 . "여울 단") (37723 . "쇠불릴 단") (32222 . "비단 단")
                 (31774 . "소쿠리 단") (34507 . "오랑캐 이름 단, 새알 단")
                 (20150 . "믿을 단") 34036 (34962 . "웃옷 벗을 단, 옷벗어멜 단")
                 (24406 . "결단할 단, 결단내릴 단") (37170 . "조나라 서울 단, 땅이름 단")
                 (22349 . "언덕") 26029 21333 (25285 . "원숭이 같은 짐승 단")
                 (32966 . "침비깨 단") 25718 27115 (26932 . "자작나무 단")
                 (24945 . "근심할 단") (28441 . "이슬방울 맺을 단")
                 27897 38780 35148 (30281 . "황달 단") 15041 (34577 . "오랑캐 이름 단")
                 (32785 . "시초 단") 29782 28598 29487 35094 (33142 . "약포 단")
                 37828 20595 28524 29011 27592 40224 15068 22242
                 22363 38077 21336 22243 25247 30213 31658 31766
                 32526 37112 38203 21008 21100 21231 21296 21772
                 22597 23183 26540 26598 27314 28266 28997 29612
                 29870 30093 30723 30891 31850 31984 32373 33902
                 34951 35085 35101 35170 35291 35986 35994 36409
                 36502 37485 37748 39411 39513 40360 40402 40420
                 40443 14312 14342 14458 14539 14582 14939 15469
                 16618 16993 17052 17463 17567 17738 18205 18213
                 18254 18337 18474 18615 18836 19045 19790))
       (45804 . ((36948 . "통달할 달") (25787 . "매질할 달, 빠를 달")
                 (30136 . "황달 달") (29562 . "수달 달") (28606 . "미끄러울")
                 16026 (24603 . "깜짝 놀랄 달") (38373 . "대문 달") 36921
                 (38851 . "칠 달") (38780 . "다룬가죽 달") (22962 . "계집의 이름 달")
                 27261 37949 21630 29173 16128 28831 29402 21714
                 25374 33625 36342 38396 38801 22112 22447 22710
                 31530 32360 32653 34200 34813 35418 36482 13459
                 14581 15584 16377 16622 19811))
       (45812 . ((35527 . "말씀 담") (25812 . "질 담") (28129 . "물모양 담, 묽을 담")
                 (28525 . "깊을 담, 못 담") (33213 . "쓸개 담") (35674 . "클 담, 말씀할 담")
                 (30192 . "가래 담") (28601 . "맑을 담, 싱거울 담") (26311 . "날 흐릴 담, 구름낀 담")
                 (25018 . "고요할 담") (28251 . "즐거울 담") (32835 . "노자이름 담, 귀바퀴없는 담")
                 (35203 . "미칠 담, 고요할 담") (21846 . "씹을 담") (34113 . "말 담, 지모 담")
                 (22349 . "물이 언덕 찰 담") (37663 . "긴 창 담") 25285
                 32966 (22704 . "목긴 술병 담") (31147 . "담제 담") (21847 . "씹을 담")
                 24788 32828 32888 32030 (27631 . "담자리 담") (37103 . "나라이름 담")
                 20499 29972 (22089 . "씹을 담") (20747 . "기운차고 날쌜해집 담")
                 (34205 . "치자나무 담") 33807 (40686 . "검을 담") 16619
                 27229 31390 39204 13782 22527 28600 32584 33093
                 33788 34795 38702 38979 39508 39655 18997 33640
                 35848 35885 38188 (22748 . "술병 담") 24528 (32590 . "壜과 同字")
                 (40693 . "문신할 담") 20577 21887 22078 22186 22709
                 22921 22991 23122 23173 24078 24987 26315 27694
                 29933 30298 32194 33297 33301 34283 34932 36105
                 37256 37296 39247 39358 40661 13674 13873 13998
                 14116 15516 15533 15543 15593 15618 16124 16206
                 16209 16806 16817 16990 17060 17466 17497 17692
                 17866 18198 18978 19291))
       (45813 . ((31572 . "젖을 답, 대답 답") (36367 . "떨어지는 소리 답, 밟을 답")
                 (30035 . "논 답") (36957 . "가다가설 답, 뒤섞일 답")
                 (27795 . "거듭 답, 물넘칠 답") 33605 21107 36427
                 28094 28314 35105 35702 21266 23817 27033 28620
                 30051 30484 35103 35515 36473 36514 37769 40854
                 40856 13934 14099 14327 14842 15228 15322 15595
                 16273 16367 16372 16575 16907 17163 17223 17245
                 17299 17435 17632 18202 18444 19098 19700 19820
                 19840 19841))
       (45817 . ((22530 . "집 당") (30070 . "마땅할 당") (21776 . "당나라 당")
                 (40680 . "무리 당") (31958 . "엿 당") (22616 . "연못 당")
                 (25758 . "두드릴 당, 칠 당") (26848 . "아가위 당, 사당나무 당")
                 (34739 . "버마 재비 당") (24162 . "기 당") (25095 . "어리석을 당, 놀랄 당")
                 (20504 . "혹시 당") (20795 . "빼어날 당") (20826 . "黨의 略字")
                 (35740 . "곧은 말 당") (25642 . "당동할 당") (29867 . "귀고리 당")
                 (29805 . "옥이름 당") (30624 . "바로볼 당") (37946 . "죄인 자물쇠 당")
                 34711 31801 25093 (35168 . "잠방이 당") (37852 . "북방축소 당")
                 23907 (28303 . "진수렁 당") (39219 . "엿 당") 25091
                 16751 20741 22049 22087 25690 26349 28578 29051
                 33179 38018 38371 38842 24403 20645 22449 25094
                 25377 26723 29680 31580 35014 35872 38107 38219
                 38231 39271 13541 (27284 . "책상 당") (30993 . "밑바닥 당")
                 (34807 . "사마귀 당") (39225 . "엿 당") 20623 20988
                 21135 21882 22137 22371 22731 24966 25803 25897
                 27062 27411 27705 28447 28761 29219 29949 30602
                 30680 30916 31135 31702 31939 31963 32805 33157
                 33377 33998 35681 36207 36442 36538 37132 37283
                 37781 37810 38363 38554 39236 40374 40734 13437
                 13942 14093 14125 15227 15525 16149 16186 16198
                 16361 16613 16974 17157 17739 18066 18634 18648
                 18659 18908 19290))
       (45824 . ((22823 . "클 대, 큰 대") (20195 . "대신할 대")
                 (24453 . "기다릴 대") (23565 . "마주볼 대, 대답할 대")
                 (24118 . "띠 대") (38538 . "군대 대, 떼 대") (36024 . "빌릴 대")
                 (33274 . "능 대") 21488 (25140 . "받들 대, 머리에일 대")
                 (22408 . "터 대, 집터 대") (34955 . "자루 대, 주머니 대")
                 (25825 . "들 대") (23729 . "태산 대, 산이름 대") (40667 . "눈썹 그릴 대, 새파랄 대")
                 (29619 . "대모(바다거북) 대") (37662 . "창고달 대")
                 (22382 . "들 대, 집대 대") 13362 17491 24111 (25055 . "원망할 대")
                 24082 24989 35176 28647 (30867 . "방아 대") (20755 . "하인 대")
                 34225 35190 13916 (27727 . "씻을 대") 27311 31051
                 34233 36443 13830 15190 17579 18996 38582 23545
                 23550 24102 24636 36151 36714 38246 38431 (26098 . "햇빛 대")
                 (37907 . "창고달 대") 20131 23343 24279 24990 25051
                 26571 28713 28731 29767 29977 31780 31817 33372
                 35656 35701 36561 36570 36699 37916 39832 40689
                 13463 13547 13941 13966 14079 14195 14253 14362
                 14549 14752 15107 15139 15298 15572 15832 15872
                 16086 16589 16902 17994 18078 18079 19423 19622))
       (45825 . ((23429 . "집 댁")))
       (45909 . ((24503 . "큰 덕") (24755 . "德의 古字, 덕 덕")
                 24499 24692 24810 14635 14657))
       (46020 . ((20992 . "칼 도") (36947 . "길 도") (24230 . "법도 도")
                 (22294 . "그림 도") (23798 . "섬 도") (37117 . "도읍 도")
                 (21040 . "이를 도") (24466 . "무리 도") (36867 . "달아날 도")
                 (23566 . "이끌 도, 인도할 도") (28193 . "건널 도") (30428 . "도둑 도")
                 (36884 . "길 도") (20498 . "넘어질 도, 거꾸러질 도")
                 (26691 . "복숭아 나무 도") (36339 . "뛸 도, 미끄러질 도")
                 (38518 . "질 그도 도, 질그릇 도") (22615 . "바를 도, 진흙 도")
                 (31291 . "벼 도") (25361 . "돋을 도, 돋울 도") (31153 . "빌 도")
                 (24764 . "슬퍼할 도, 애도할 도") (29182 . "비칠 도, 덮일 도")
                 (28644 . "물결 도, 큰물결 도") (36424 . "밟을 도") (23648 . "잡을 도, 죽일 도")
                 (28372 . "물 넘칠 도, 도도히 흐르는 도") (30585 . "볼 도")
                 (37709 . "도금할 도") (25481 . "흔들 도, 흔들릴 도")
                 (22581 . "담 도") 20828 (25623 . "찧을 도") (33796 . "들머루")
                 (36077 . "노름 도, 내기 도") (28120 . "일 도, 물흐를 도")
                 (35241 . "睹와 同字, 볼 도") (26873 . "노 저을 도")
                 (38876 . "활집 도, 감출 도") (23947 . "섬 도") (27330 . "상앗대 도, 노 도")
                 (21480 . "탐할 도") 30423 33759 (40727 . "땡땡이 도")
                 (25827 . "찧을 도") (39253 . "탐할 도") 35615 36847
                 (28034 . "길 도") 31106 32091 33660 26482 (38349 . "성문 층대 도")
                 32767 16771 22270 (24521 . "근심할 도") 29078 33312
                 (24354 . "활집 도") (31244 . "찰벼 도") 22834 (38833 . "너그러울 도")
                 24141 21687 37334 32175 35050 (25647 . "두드릴 도")
                 (27310 . "그루터기 도") 14902 26423 32282 (24902 . "기뻐할 도")
                 22133 38793 19802 (22740 . "작은 성 도") 27950 21771
                 25416 26255 27101 30223 34908 36351 40173 16573
                 16699 25443 22259 23548 23707 26812 28059 28952
                 31095 31282 32505 36172 38208 38407 38892 40061
                 (25487 . "가릴(선택)할 도") (37236 . "두번 빚은 술 도")
                 (38784 . "소고 도") 20931 (20994 . "선칼도방") 21259
                 21845 21925 22047 22295 23245 23670 23801 23902
                 23936 23948 23993 24126 24233 24284 25455 26091
                 26764 26913 27076 27077 27300 27672 28531 29757
                 29803 29913 31097 31585 33364 33791 33956 34014
                 34413 34602 34911 35420 36551 37252 37263 37630
                 38502 38557 38575 38767 39160 39232 39327 39420
                 39434 39771 40268 40395 40437 64041 14261 14336
                 14509 14789 15070 15160 15224 15646 15865 15891
                 15959 16076 16108 16111 16414 16446 16759 16945
                 17069 17190 17516 17522 17816 18156 18350 18427
                 18628 18653 18750 19043 19058 19230 19234 19245
                 19387 19676))
       (46021 . ((35712 . "읽을 독") (29544 . "홀로 독") (30563 . "감독할 독")
                 (27602 . "독 독, 독할 독") (31716 . "도타울 독, 두터울 독")
                 (28678 . "도랑 독, 개천 독") (31167 . "대머리 독, 벗어진 독")
                 (29272 . "편지 독") (29346 . "송아지 독") (32411 . "깃일산")
                 29420 (40695 . "더러울 독") 35743 (27357 . "함 독")
                 31171 39633 20959 21301 27568 34675 37654 38883
                 26911 28174 29261 29322 31491 35501 35835 40681
                 15465 23355 28043 28060 29892 30172 30398 30881
                 35067 35908 36117 37621 37983 38855 38885 39475
                 40282 13460 14232 15935 16260 16675 16926 17409
                 17647 18146 18609 19187 19383))
       (46024 . ((25958 . "도타울 돈") (35930 . "돼지 돈") (38931 . "조아릴 돈")
                 (24775 . "두터울 돈, 도타울 돈") (27788 . "기름 덩어리 돈, 혼탁할 돈")
                 (29129 . "불 성할 돈") 35114 36975 (26109 . "늘 돋을 돈, 먼동틀 돈")
                 (26302 . "아침해 돈") (22697 . "돈대 돈") (28958 . "귀갑 지지는 불 돈")
                 28822 (28513 . "큰 물 돈") 27220 22136 29540 15167
                 15968 16123 30981 20229 22244 22698 24563 25189
                 25737 36280 39039 39272 40064 (24372 . "그림 그린 활 돈")
                 29938 (36489 . "거룻배 돈") 37907 23864 24201 25213
                 25780 26396 29340 30744 35928 36478 36568 36871
                 38869 39145 39504 39784 40663 14511 14699 14928
                 15007 16614 17723 19075 19786 19789))
       (46028 . ((31361 . "부딪칠 돌") (20077 . "이름 돌, 사람이름 돌")
                 (21636 . "꾸짖을 돌") (22551 . "굴똑 돌") 33135 28261
                 25580 26606 13329 14318 16096 39283 23434 22153
                 24610 33878 36812 37423 37710 39167 40351 40757
                 15837 16592 18440))
       (46039 . (13839))
       (46041 . ((21516 . "같을 동, 한가지 동") (26481 . "동녘 동")
                 (20908 . "겨울 동") (21205 . "움직일 동") (27934 . "마을 동, 고을 동")
                 (31461 . "아이 동") (37509 . "구리 동") (20941 . "얼 동")
                 (33891 . "바로잡을 동, 바를 동") (26704 . "오동나무 동")
                 (26847 . "용마루 동, 동자기둥 동") (24999 . "동경할 동, 흐리멍덩할 동")
                 (30140 . "아플 동") (30643 . "눈동자 동") (33012 . "큰 창자 동")
                 (20189 . "同의 古字, 같을 동") (28540 . "물 이름 동, 물결높을 동")
                 (37676 . "보습날 동") (20718 . "아이 동") (34155 . "황모 동")
                 35532 20685 (24420 . "붉은 칠 동") 22508 (20375 . "무지할 동")
                 (23762 . "산이름 동") 26312 14850 31324 20319 (34624 . "무지개 동")
                 (28087 . "마룻대 동") (22412 . "항아리 동") (26403 . "달 뜰 동")
                 35447 33191 14345 (27238 . "나무이름 동") 25359 28884
                 32103 39175 (33375 . "배 동") (33596 . "쏙갓 동") 25026
                 28265 29534 32639 39846 40725 23741 19996 20923
                 21160 26635 33000 38108 40086 40491 (21699 . "클말할 동")
                 (33523 . "겨울살이 동") (39831 . "전어 동") 20530 20549
                 21229 21658 22693 23003 23099 23326 23746 23773
                 23840 23852 24221 25113 26168 26189 27651 27681
                 27693 28021 28853 29137 29341 29418 30510 30780
                 30800 31217 31511 31677 31905 33110 33732 34324
                 34450 34901 36200 36853 37230 37462 37493 38680
                 39399 39903 40327 13384 13514 13734 13794 14005
                 14397 14501 14560 14607 15073 15349 15495 15858
                 15895 16167 16175 16191 16520 16546 16794 17270
                 18322 19381 19492 19568 19657 19659 19712 19796
                 19889))
       (46160 . ((38957 . "머리 두") (35712 . "구절 두, 읽을 두")
                 (26007 . "말 두") (35910 . "콩 두") (26460 . "막을 두")
                 (20828 . "투구 두, 둘러쌀 두") (30168 . "마마 두") (31431 . "구멍 두")
                 (36887 . "머무를 두") (33651 . "콩 두") (26515 . "두공 두, 주두 두")
                 32015 (25985 . "섞을 두") (34873 . "좀 두") (32922 . "밥통 두")
                 34855 (38497 . "절벽 두") (34474 . "올챙이 두") 39190
                 14259 (25238 . "퍼주거릴 두") 15031 20832 16054 (33072 . "목 두")
                 38439 64035 16890 18575 19623 22836 31398 38061
                 39294 21515 (20128 . "의미없는 토") 20408 21061 21562
                 21783 21790 26794 27255 27564 27629 28002 31740
                 33648 34104 34713 37078 37208 37380 37568 38295
                 (39717 . "싸움") 13553 14034 14293 14468 14894 15407
                 16353 16648 16838 17734 17777 18101 19238 19535
                 19785))
       (46161 . (15026))
       (46164 . ((37389 . "무딜 둔, 둔할 둔") (23663 . "진칠 둔")
                 (36929 . "달아날 둔") (33216 . "볼기 둔") (33434 . "채소 이름 둔, 나무싹 둔")
                 (36975 . "달아날 둔, 숨을 둔") (31360 . "광중 둔") 21544
                 (36813 . "머뭇거릴 둔") 38045 22345 33227 36402 38677
                 14932 16138 18227 18716))
       (46168 . ((20071 . "음역자 둘") 13320 13337))
       (46179 . (13709))
       (46181 . (15027))
       (46301 . ((24471 . "얻을 득") 28098 22042 38173 26831
                 37696 14198 18039 18040))
       (46321 . ((31561 . "가지런할 등, 무리 등") (30331 . "오를 등")
                 (29128 . "등잔 등") (39472 . "오를 등") (34276 . "등나무 등")
                 (37159 . "나라 이름 등, 등나라 등") (23965 . "고개 등, 비탈 등")
                 (35588 . "베낄 등") (27225 . "등자나무 등, 궤 등") 17601
                 (28373 . "물 솟을 등") 36460 (37913 . "등자 등") (28783 . "燈의 略字")
                 (30964 . "돌 비탈길 등, 돌다리 등") 17581 (32290 . "봉할 등")
                 (34723 . "등사 등") (22705 . "자드락길 등") 31782 35915
                 (20979 . "걸상 등") 39523 16197 31832 33199 38565
                 13569 33150 35466 37011 38251 (31824 . "대 기구 등")
                 20700 20783 22100 23297 24144 25125 27336 28443
                 29842 31475 33376 34373 35252 36998 38703 39398
                 39975 40735 15530 17541 17714 17768 17907 18014
                 18476 18478 19380 19597 19618 19710))
       (46629 . (13743 13744))
       (46896 . (13346))
       (46972 . ((32645 . "새그물 라") (35064 . "벌거벗을 라, 벌거숭이 라")
                 (25078 . "게으를 라") (34746 . "소라 라") (25295 . "붙잡을 라")
                 (37007 . "순행할 라, 돌 라") (30313 . "약물 중독 라")
                 (34367 . "무 라, 소나무겨우살이 라") (21895 . "나팔")
                 35260 (39486 . "노새 라") (22217 . "소리 읽힐 라") (38012 . "징 라")
                 (30256 . "연주창 라") (33245 . "벌거벗을 라") 34819
                 30754 28757 35254 29568 36534 14213 32409 16136
                 16852 32599 25042 26916 29473 30302 31657 33078
                 33821 36923 38179 38233 39585 13417 (20526 . "알몸 라")
                 (26346 . "날 흐릴 라") (39512 . "옹 솥 라") 20157 20792
                 21062 21872 22201 22230 25694 25886 25901 26095
                 27407 31854 32598 33121 33999 35225 37807 37837
                 39264 40449 13481 14764 14945 15361 15483 16203
                 16346 16979 17064 17201 17204 19393))
       (46973 . ((33853 . "떨어질 락") (27138 . "즐거울 락, 즐길 락")
                 (35582 . "대답할 락, 허락할 락") (32097 . "헌솜 락, 연락할 락")
                 (27931 . "강이름 락") (39409 . "낙타 락") (28889 . "지질 락")
                 (37226 . "진한 유즙 락, 타락 락") (29662 . "구슬 목걸이 락")
                 (29334 . "얼룩소 락") 30796 28668 37499 38610 27898
                 32476 33638 38124 39289 39558 21984 23752 31551
                 39182 39845 40252 13538 14753 16425 16951 17290))
       (46976 . ((20025 . "붉을 란") (21365 . "알 란") (20098 . "어지러울 란")
                 (34349 . "난초 란") (27396 . "난간 란") (29211 . "문드러질 란")
                 (40478 . "난새 란") (25078 . "게으를 란") (28734 . "물결 란, 큰물결 란")
                 (27410 . "나무 이름 란, 난나무 란") 20081 (38348 . "가막을 란")
                 (38014 . "방울 란") (25876 . "막을 란") 18856 28390
                 28772 (35188 . "난삼 란") (23358 . "게으를 란") (24177 . "내리다지 란")
                 26005 22301 22302 (28755 . "적실 란") 28769 31843
                 20848 25318 26003 26639 26686 28572 28866 35888
                 37550 38247 38417 40510 20138 22226 23364 23375
                 26347 27415 29143 29220 29907 30325 31991 32649
                 33189 34378 35157 35197 35659 35733 36509 37344
                 37997 38020 38858 40265 13843 14441 14760 15437
                 15467 15573 16633 17484 17794 19085))
       (46980 . ((21068 . "어그러질 랄") (36771 . "매울 랄")
                 (21895 . "말굽히할 랄") (22482 . "바자울 랄") 25419
                 (36770 . "辣과 同字") 29549 22403 22483 25574 25575
                 25867 26955 29902 30220 34674 38961 39694 39931
                 15882 16075 16093 16435 17344 17637 17654 19563
                 19867))
       (46988 . ((35261 . "볼 람") (28651 . "퍼질 람") (34253 . "쪽 람")
                 (31811 . "바구니 람") (25900 . "잡을 람") (23888 . "남기 람")
                 (32412 . "닻줄 람") (27414 . "감람나무 람") (25829 . "攬과 同字, 잡아다릴 람")
                 (35172 . "누더기 람") 35239 (23146 . "탐할 람") 28768
                 22728 15535 19472 14940 (24783 . "떨릴 람") 22229
                 13411 23706 25597 27012 28389 28452 31726 32518
                 34013 35124 35272 17216 20758 21833 22146 25058
                 28008 28742 29155 29175 29185 29222 29884 31031
                 32383 33915 34347 39026 13845 14054 14126 14249
                 14811 14891 15379 15382 16830 17195 18988))
       (46989 . ((25289 . "꺽을 랍, 꺾을 랍") (33240 . "납향 랍")
                 (34847 . "밀 랍") 25722 (37982 . "땜납 랍") 33224 29193
                 17910 38260 21862 25626 26566 30934 32715 33736
                 34635 37003 38817 13639 14452 16560 16627 16881))
       (46993 . ((37086 . "사나이 랑, 사내 랑") (28010 . "물결 랑")
                 (24266 . "복도 랑") (29436 . "이리 랑") (26391 . "밝을 랑")
                 (29701 . "옥 이름 랑, 옥같은돌 랑") (37070 . "사나이")
                 (34690 . "蜋과 同字, 버마재비 랑") 28711 (29807 . "고을 이름 랑, 옥같은돌 랑")
                 36361 (38316 . "솟을 대문 랑") (34571 . "사마귀 랑")
                 (31234 . "강아지 풀 랑") (27028 . "나무 이름 랑") 26745
                 28922 (30816 . "돌 부딪는 소라 랑") 33992 37571 (33704 . "수크령 랑")
                 27444 33350 15088 38162 38406 21190 21879 22476
                 22641 23247 23808 26390 26404 27139 31588 33943
                 34018 35471 36532 37074 39418 13546 13919 14285
                 14467 15294 15458 16319 16801 17242 17758 18521
                 19414 19574))
       (47000 . ((20358 . "올 래") (33802 . "명아주 래") 36058
                 (23821 . "산 이름 래") (24480 . "올 래") 26469 21201
                 24469 33713 16728 29453 29724 (28150 . "강 이름 래")
                 (39435 . "큰 말 래") 20488 31618 38978 26870 30558
                 37688 40291 17137 18053 23811 20459 26814 28062
                 30544 38140 20797 21819 23137 24242 36904 37106
                 39904 40326 40627 13971 15674 16290 16574 18898))
       (47021 . ((20919 . "찰 랭") 21797))
       (47029 . ((30053 . "다스릴 략, 간략할 략") (25504 . "노략질할 략")
                 (30055 . "略과 同字") 25866 22297 25730 27932 13436
                 13769 14908 16558 17166 18138))
       (47049 . ((20841 . "두 량, 둘 량") 22580 (33391 . "좋을 량, 어질 량")
                 (37327 . "헤아릴 량") (20937 . "涼의 俗字, 서늘할 량")
                 (26753 . "푸조나무 량, 들보 량") (31975 . "양식 량, 곡식 량")
                 (35538 . "믿을 량") (20142 . "밝을 량") (27153 . "서늘할 량, 들보 량")
                 (36635 . "수레 량, 수레 한채 량") (31921 . "기장 량")
                 (20486 . "재주 량, 공교할 량") (31918 . "糧과 同字, 양식 량")
                 20004 20001 (28092 . "서늘할 량") (39758 . "도깨비 량")
                 (36361 . "뛸 량") 24778 16279 (26891 . "들보 량") (24738 . "슬퍼할 량")
                 28280 32201 35058 36620 36652 20429 20457 21793
                 35845 36742 36748 39753 (21928 . "소리 맑을 량")
                 21748 21858 22682 25498 25999 26238 29047 31767
                 32161 32682 33084 34621 13491 13568 13597 14167
                 15937 16438 16449 17635 18294 18435 18684 19051
                 19306))
       (47140 . ((26053 . "군사 려, 나그네 려") (24942 . "생각할 려")
                 (40599 . "고울 려") (21237 . "힘쓸 려") (21570 . "음률 려, 풍류 려")
                 (24300 . "오두막집 려, 원두막 려") (31018 . "거친 숫돌 려")
                 (39530 . "가라말 려, 검은 말 려") (38317 . "이문 려")
                 (25150 . "여그러질 려, 어그러질 려") (40654 . "검을 려, 무리 려")
                 (20406 . "짝 려") (28670 . "겨를 려, 씻을 려") 29313
                 (39522 . "나귀 려") (34268 . "나라 이름 려, 아름다운 옥 려")
                 (20791 . "짝 려") (34851 . "굴 려") (27354 . "종려나무 려")
                 (21426 . "갈 려") 21525 (33242 . "살갗 려") (30296 . "창질 려")
                 27828 (33154 . "등골 뼈 려") (34849 . "좀먹을 려") 33620
                 (31986 . "현미 려") 20029 20387 (21811 . "울 려") 30445
                 29896 (26784 . "평고대 려") 27408 34808 40679 (37986 . "줄 려")
                 26841 31341 24306 34264 25894 27350 29192 33560
                 34599 13414 17787 21169 20458 21385 24208 27016
                 28388 30112 30778 31238 31901 33002 34510 38109
                 38398 39540 39562 40097 (20770 . "맘에 하기 싫을 려")
                 (37004 . "천천히 걸을 려") 21236 22215 23151 24001
                 24759 25364 25435 26334 26341 26651 27829 28946
                 29653 30667 31075 31154 31326 32125 32159 33168
                 34528 34823 37080 37569 37867 37975 39359 39481
                 40049 40058 40313 13479 13551 14030 14367 14368
                 15194 15360 15482 15645 16272 16276 16347 17003
                 17066 17630 17699 17703 17761 17869 17928 18643
                 18792 18815 19337 19477 19745))
       (47141 . ((21147 . "힘 력") (27511 . "지낼 력, 지날 력")
                 (26310 . "책력 력") (28701 . "거를 력, 스밀 력") (31019 . "조약돌 력")
                 (36706 . "삐걱거릴 력, 수레바퀴에 치는 력") (38722 . "벼락 력")
                 (27359 . "상수리나무 력") (27370 . "말구유 력") 27508
                 39730 26278 17673 21423 30378 (37192 . "땅 이름 력")
                 23668 26424 27306 30975 31024 36498 21382 21590
                 22364 26533 26638 27813 30124 30782 33480 36318
                 36729 37094 38217 38643 25866 (30311 . "연주창 력")
                 (36707 . "갈 력") 20782 21412 21499 22182 22754 25853
                 29199 29893 29905 30771 31197 33946 34294 34488
                 34679 35323 35720 36210 37784 38731 38903 40051
                 40168 13583 14371 14832 15119 15332 15481 15860
                 16033 16122 16193 16360 16623 17253 17277 18383
                 18384 18713 18806 19365 19483 19484))
       (47144 . ((36899 . "잇닿을 련, 연할 련") (32244 . "익힐 련")
                 (37706 . "불릴 련, 쇠불릴 련") (24976 . "불쌍히 여길 련, 가련할 련")
                 (25088 . "사모할 련") (32879 . "잇달 련, 연할 련") (34030 . "연밥 련, 연꽃 련")
                 (28451 . "물놀이 련, 물놀이칠 련") (36646 . "손수레 련, 당길 련")
                 (29001 . "불릴 련, 쇠불릴 련") (24604 . "연리할 련")
                 (29833 . "호련 련, 종묘제기 련") (25891 . "걸릴 련")
                 24651 37676 22556 (33248 . "저민고기 련") (23372 . "아름다울 련")
                 (39985 . "연어 련") (26973 . "멀구슬나무 련") 25689
                 17352 (37832 . "쇠사슬 련") 32314 32872 14139 38142
                 23048 25371 25781 26815 28063 28860 29711 32451
                 32852 33044 33714 35042 36743 36830 40098 23423
                 (28229 . "누일 련") (39946 . "고기 이름 련") 20678 22009
                 22098 22897 23201 23294 24474 24937 25862 27108
                 27363 29779 30324 32414 32756 32875 32878 33190
                 33840 35123 35632 36453 37179 14083 14101 14152
                 14272 14677 14721 15005 15368 15756 16038 16145
                 16603 18188))
       (47148 . ((21015 . "줄 렬, 벌일 렬") (28872 . "세찰 렬, 매울 렬")
                 (21155 . "못할 렬, 용렬할 렬") (35010 . "찢을 렬") (20925 . "찰 렬, 찬바람 렬")
                 (27916 . "맑을 렬") (25449 . "비틀 렬") (39090 . "사나운 바람 렬")
                 33570 36862 37602 40247 15209 38154 19733 25362
                 21181 21671 21751 23028 25368 26677 27990 28910
                 29037 29188 30553 33055 34522 34550 36244 36835
                 37597 39844 14624 15151 15198 15929 16223 16704
                 18425 19339))
       (47156 . ((24265 . "청렴할 렴") (28610 . "내 이름 렴, 엷을 렴")
                 (25986 . "거둘 렴") (31806 . "발 렴") (27566 . "염할 렴")
                 27483 24088 31839 (28722 . "넘칠 렴") (22889 . "화장 상자 렴")
                 14488 29547 (30927 . "거친 숫돌 렴") 23322 37934
                 25627 28307 34197 34333 34826 22849 25947 28491
                 33718 34105 38256 20151 21126 21298 21299 28592
                 28627 29073 29163 31284 31842 31848 32695 33217
                 34207 34334 34698 35229 39697 14202 14424 15000
                 15340 15464 15761 16147 16480 16770 17182 18042
                 18789 19281))
       (47157 . ((29557 . "사냥 렵, 사냥할 렵") (39715 . "갈기 렵")
                 (36496 . "밟을 렵") 20768 25848 29471 24036 29347
                 32855 40050 15359 15537 16131 16178 16509 17005
                 18226 18259 19081))
       (47161 . ((20196 . "명령 령, 시킬 령, 하여금 령")
                 (38936 . "옷깃 령, 거느릴 령") (23527 . "편안할 령")
                 (38728 . "신령 령") (23994 . "재 령, 산고개 령") (38646 . "조용히 오는 비 령, 떨어질 령")
                 (37428 . "방울 령") (29618 . "옥이름 령, 금옥소리 령")
                 (22265 . "옥 령, 감옥 령") (40801 . "나이 령") (36894 . "굳셀 령, 통할 령")
                 (32838 . "들을 령") (24604 . "영리할 령") (20278 . "영리할 령, 악공 령, 광대 령")
                 (32718 . "깃 령, 날개 령") (32666 . "영양 령") (23738 . "재 령, 깊은산 령")
                 (31533 . "도꼬마리 령, 작은 통 령") (21478 . "나눌 령, 다를 령, 헤어질 령")
                 (23725 . "산 이름 령") (27872 . "깨우칠 령") 28789 (33491 . "도꼬마리 령")
                 22397 (40210 . "할미새 령") 15443 (34505 . "장자리 령")
                 27386 (27422 . "격자창 령") 37309 38683 29940 (21604 . "속삭일 령")
                 29393 (26148 . "날 빛 영롱할 령") 33330 24446 31451
                 36297 38685 29167 35397 37187 26818 38083 38666
                 39046 40496 40802 40836 (22984 . "계집 슬기로울 령")
                 (31202 . "벼 처음 익을 령") (36584 . "사냥 수레 령")
                 (40567 . "나이 령") 21026 23361 25294 26382 26563
                 28586 28841 29223 30346 30769 32055 34166 34342
                 34897 34954 38301 38462 38679 39382 39807 40610
                 40855 13969 14453 14765 14933 15022 15137 15338
                 15494 15923 16079 16265 16677 16961 16982 17017
                 17186 17221 17752 17797 18021 18482 18556 18559
                 18985 19405 19505 19719 19730 19755))
       (47168 . ((31150 . "예도 례") (20363 . "법식 례") (35914 . "禮의 古字")
                 (38583 . "隸와 同字, 붙을 례") (37300 . "단술 례") (38584 . "붙을 례")
                 (28583 . "강 이름 례") 31036 13841 40162 (40039 . "가물치 례")
                 27348 27418 28671 29345 29925 30725 14402 14625
                 16221 16708 19817))
       (47196 . ((32769 . "늙은이 로") (36335 . "길 로") (21214 . "일할 로")
                 (24594 . "성냉 로, 성낼 로") (38706 . "이슬 로") (30439 . "밥그릇 로")
                 (39791 . "노둔할 로") (29200 . "화로 로, 화초 로") (34310 . "갈대 로")
                 (40442 . "해오라기 로") (34396 . "포로 로") (25796 . "사로잡을 로")
                 (25736 . "잡을 로, 건져낼 로") (36613 . "수레 로") (40565 . "소금 로, 개펄 로")
                 (28510 . "강 이름 로, 물이름 로") 33446 (27347 . "방패 로")
                 (28696 . "강 이름 로") 28518 21171 (37994 . "화로 로")
                 (40056 . "농어 로") (22746 . "흑토 로") 39486 (33386 . "艫와 同字")
                 15587 (39025 . "머리뼈 로") (33387 . "뱃머리 로") (40469 . "가마우지 로")
                 28551 37876 (36708 . "고패 로") 29840 (29576 . "검을 로")
                 29561 32591 25870 31797 19667 (31362 . "우리 로")
                 21694 23274 27368 29679 30673 31788 33379 39645
                 21348 27257 28809 21172 21346 21792 22108 22406
                 23810 25438 25523 25784 26636 27655 27896 28061
                 30184 32802 33339 34383 36723 36738 38097 38137
                 39045 40065 40072 40492 40557 (28407 . "소금밭 로")
                 (30278 . "종독 로") 30944 (37930 . "부레 로") (39639 . "해골 로")
                 20695 21744 22062 22165 22183 22647 23959 24645
                 24997 24998 25852 26405 26675 27152 27247 27660
                 28022 28571 28674 29419 29904 30803 30961 31785
                 31834 32401 (32770 . "늙을로엄") 32814 34046 34135
                 34370 34791 34854 36546 37536 37824 37906 39794
                 39857 40696 13607 13610 13798 14281 14490 14515
                 14831 14867 15021 15188 15325 15341 16342 16597
                 16877 18190 18222 18241 18510 19599 19600 19791))
       (47197 . ((32160 . "초록빛 록") (37636 . "기록할 록")
                 (31103 . "복 록") (40575 . "사슴 록") (40595 . "신기슭 록, 산기슭 록")
                 (30860 . "돌 모양 록") (33737 . "조개풀 록") 24405 30429
                 (28133 . "밭을 록") 31833 31108 (28425 . "거를 록")
                 37249 32209 (24404 . "나무 깍을 록") (39428 . "말이름 록")
                 (31759 . "대 상자 록") (36678 . "도르래 록") 24280 37682
                 13736 30569 34061 36911 32511 28172 31635 36760
                 (22309 . "버섯 록") 22646 23101 25693 26810 26882
                 27162 27695 29085 29741 29994 31798 31926 33172
                 34736 35310 36258 36439 37684 39484 40294 40867
                 14105 14308 14727 14998 15101 15327 15542 16190
                 16601 17170 17249 17297 17340 17410 17973 18052
                 18431 19054 19546 19754))
       (47200 . ((35542 . "말할 론") 25476 24768 30870 33749
                 35770 23144 28323 14651))
       (47217 . ((24324 . "희롱할 롱") (31840 . "대그릇 롱")
                 40848 (32894 . "귀머거리 롱") (22751 . "언덕 롱") (29903 . "환할")
                 (26407 . "흐릿할 롱, 침침할 롱") (28711 . "적실") (38580 . "고개 이름 롱")
                 27379 22752 (31025 . "갈 롱") (26344 . "어스레할 롱")
                 24003 (34338 . "개여뀌 롱") (20785 . "건목칠 롱") 21730
                 31026 (25871 . "누를 롱") 17354 22184 31433 16047
                 32999 21392 21657 22404 22405 25314 26173 26634
                 27895 28381 29649 30779 31548 32843 33551 38471
                 40846 24004 25354 25397 26759 27102 30822 31725
                 34858 35011 35185 35909 37863 37992 38727 40471
                 40851 13405 13913 14046 14294 14469 15373 15589
                 16781 18497 19082))
       (47280 . ((38647 . "우뢰 뢰, 우리 뢰") (36084 . "힘 입을 뢰")
                 (36034 . "뇌물 줄 뢰") (20769 . "영락할 뢰, 무너질 뢰")
                 (29282 . "우리 뢰") (30922 . "돌무더기 뢰") (36058 . "줄 뢰")
                 (28712 . "여울 뢰, 물결 뢰") (32589 . "술독 뢰") (32786 . "쟁기 뢰")
                 38972 31839 (37241 . "부을 뢰") 28716 (35460 . "뇌사 뢰")
                 (31015 . "바위너설 뢰") (32391 . "실 마디 뢰") 15714
                 18963 (34174 . "꽃봉오리 뢰") 34302 37939 38978 20651
                 27345 29891 31016 17593 28625 31809 35796 36162
                 36169 36182 38253 25858 (30988 . "돌 굴러내릴 뢰")
                 22760 25794 27281 27380 27937 29681 30078 30295
                 30949 31577 34845 35184 35716 36704 37511 37649
                 37976 38008 38721 38939 38954 39075 40041 13389
                 13478 13603 14373 15509 15741 16141 16340 16676
                 17002 17251 17455 18166 18477 18606 18626 19345
                 19610))
       (47308 . ((26009 . "되질할 료, 헤아릴 료") (20102 . "마칠 료")
                 (20698 . "동료 료, 동관 료") (36988 . "멀 료") (30274 . "병고칠 료")
                 (32842 . "귀 울 료, 애오라지 료") (23534 . "벼슬아쳐 료, 동관 료")
                 (29134 . "화톳불 료, 햇불 료") (30637 . "밝을 료, 멀리보는 료")
                 (24278 . "공허할 료, 성 료") (23525 . "쓸쓸할 료") (39719 . "시끄러울")
                 (34044 . "여뀌 료") (28518 . "큰 비 료") (37290 . "막걸리 료")
                 (32346 . "감길 료") (33163 . "발기름 료") (25769 . "다스릴 료")
                 24960 25005 40431 18276 (26296 . "밝을 료") (29536 . "밤사냥 료")
                 29849 (37904 . "은 료") 28819 15731 (22073 . "울 료")
                 23582 27217 29198 36689 14155 30103 32557 36797
                 38243 40553 (23293 . "외조모 료") (39106 . "높이 부는 바람 료")
                 (39113 . "바람 료") 20332 21502 23589 23590 23658
                 23962 23963 24299 25977 26338 29102 29202 29235
                 31410 31415 31426 31773 32719 33195 34260 34761
                 34783 35906 36095 36440 36477 37149 38269 39007
                 39630 13929 14240 14329 14459 15787 16678 17020
                 17286 17504 17599 17769 18189 18599 18949 19021
                 19292))
       (47329 . ((40845 . "용 룡") 30675 30508 40857 (40850 . "龍의 古字")
                 24511 29206 31452 34860 36122 36504 39521 14435))
       (47336 . ((27155 . "다락 루") (28431 . "샐 루") (28122 . "눈물 루")
                 (32047 . "묶을 루") (23650 . "창 루") (38475 . "줍을 루, 좁을 루")
                 (22744 . "진 루") (32311 . "실 루") (23105 . "별 이름 루, 어리석은 루")
                 (37860 . "새길 루, 강철 루") (35128 . "남루할 루") (34078 . "쑥 루, 물쑥 루")
                 (30267 . "부스럼 루, 헌곳 루") 27004 23649 (34747 . "땅강아지 루")
                 27882 34684 22593 (20674 . "구부릴 루") 25695 (39631 . "해굴 루")
                 22655 (23937 . "봉우리 루") (24954 . "정성스러울 루")
                 19371 23256 23642 24276 19179 30232 20603 21949
                 22418 23044 23901 25602 31699 32533 32807 33932
                 35099 38210 39621 16470 (22029 . "시끄러울 루") (32812 . "씨 뿌리는 기구 루")
                 21437 27344 27648 28057 28295 28426 29089 29962
                 30266 30323 30620 31757 33186 33371 35633 35991
                 36545 36977 38843 40412 40762 13453 13623 13948
                 14114 14330 14430 14698 15033 15502 15619 15775
                 15950 16015 16114 16667 16761 16962 18255 18650
                 19582 19726))
       (47448 . ((27969 . "흐를 류") (30041 . "머무를 류") (26611 . "버들 류")
                 (39006 . "무리 류") (21129 . "죽일 류, 성 류") (35628 . "그릇될 류")
                 (28316 . "방울져 떨어질 류, 증류수 류") (29705 . "유리 류, 유리돌 류")
                 (30244 . "혹 류") 32326 (29792 . "琉와 同字, 구슬이름 류")
                 (26066 . "깃발 류, 깃술 류") (27060 . "석류나무 류")
                 (30827 . "유황 류") (28687 . "맑을 류") 29804 (32397 . "갇힐 류")
                 (40377 . "올빼미 류") (32306 . "포승 류") 38692 (36955 . "머무를 류")
                 39470 37840 26056 16642 (27210 . "榴의 本字") 37775
                 32630 39104 34271 37798 39109 32107 32640 34365
                 34374 36435 39392 39413 17851 21016 27983 31867
                 32506 32551 34050 35884 38086 38157 38223 38240
                 39127 39311 39581 40552 39106 22176 22639 23225
                 23356 23911 24263 25072 26625 26746 27151 27246
                 27417 28475 28561 28741 29080 29643 29858 30020
                 30065 30081 30277 30914 31159 32185 32413 33352
                 33957 33989 34124 34288 34353 34354 35031 37466
                 37622 37890 38625 39161 39230 39505 39736 39969
                 40410 40467 40589 13356 13927 14805 15557 15791
                 16204 16222 16267 16999 17884 18087 18240 18662
                 19231 19480 19542 19849))
       (47449 . ((20845 . "여섯 륙") (38520 . "뭍 륙") (25134 . "죽일 륙")
                 20679 31249 (21216 . "협력할 륙") 31307 38470 21113
                 21198 22388 28117 30943 36379 39909 40305 13520
                 14060 18524))
       (47452 . ((20523 . "인륜 륜") (36650 . "바퀴 륜, 수레바퀴 륜")
                 (23833 . "산 이름 륜") (32184 . "낚싯줄 륜, 푸른인끈 륜")
                 (28138 . "물놀이 륜, 빠질 륜") (20374 . "둥글 륜, 뭉치 륜")
                 (25476 . "가릴 륜") 37632 36378 23832 22279 26822
                 20177 20262 22261 25249 27814 32438 36718 21991
                 22504 30958 31248 32803 33088 34598 38511 39913
                 13510 13742 15821 16897 17523))
       (47456 . ((24459 . "법 률") (29575 . "헤아릴 률") (26647 . "밤나무 률")
                 (24900 . "두려워할 률, 두려울 률") (23874 . "가파를 률")
                 31717 (28327 . "강 이름 률") 14323 30793 29806 33183
                 15258 23757 20371 20616 20947 22415 22619 23549
                 24461 25646 31675 32322 33870 40389 40604 13696
                 13786 14992 17665 17694 18499 19206))
       (47469 . ((38534 . "클 륭, 높을 륭") 28700 (30275 . "느른할 륭")
                 (31423 . "활꼴 륭") 28927 23952 34159 28272 38707
                 13957 14187 14741 16615 18786))
       (47477 . ((21202 . "굴레 륵") (32907 . "갈비 륵") 29583
                 20162 25168 (27856 . "돌 갈라질 륵") 40147 22046
                 24519 26906 27707 31483 31765 33403 38430 39987))
       (47484 . (13625))
       (47492 . ((20956 . "찰 름") (24297 . "곳집 름") 27305
                 (20955 . "찰 름") 24298 25037 25044 (28575 . "서늘할 름")
                 27265 30299 30301 33787 13357 14854 17778))
       (47497 . ((38517 . "큰 언덕 릉") (26974 . "모 릉, 네모질 릉")
                 (20940 . "능가할 릉") (32190 . "비단 릉") (33777 . "蔆과 同字, 마름 릉")
                 (31260 . "모 릉") 26865 23834 28137 22796 (20528 . "속일 릉")
                 24241 25493 30550 39914 32491 40110 (34054 . "마름 릉")
                 22542 23112 24867 29708 30864 31102 34020 34192
                 35052 36380 36632 37634 13739 13828 14660 15461
                 16906 16964 17686 18905 19211 19354))
       (47532 . ((37324 . "마을 리") (21033 . "날카로울 리, 이로울 리")
                 (29702 . "다스릴 리, 이치 리") (30064 . "다를 리") (26446 . "오얏 리")
                 (21519 . "벼슬아치 리, 관리 리") (38626 . "떼 놓을 리, 떠날 리")
                 (23653 . "신 리, 밟을 리") (35023 . "속 리") (26792 . "태나무 리, 배 리")
                 39530 (32633 . "근심 리, 만날 리") (35041 . "裏와 同字, 옷속 리")
                 (31852 . "울타리 리") (21400 . "釐의 俗字, 티끌 리")
                 (20442 . "속될 리") (30178 . "설사 리, 이질 리") (24743 . "영리할 리, 똑똑할 리")
                 (32696 . "여윌 리, 파리할 리") (37328 . "다스릴 리, 털끝 리")
                 (29432 . "貍와 同字, 이리 리") (29313 . "얼룩소 리")
                 (29827 . "유리 리") (39881 . "잉어 리") (21774 . "가는 소리 리")
                 (33673 . "말리 리, 사과꽃 리") (28012 . "해리 리, 바다잇수 리")
                 (28435 . "스며들 리") (33669 . "다다를 리") (31163 . "산신 리")
                 35400 (34733 . "교륭 리") (20432 . "똑똑할 리") (23264 . "과부 리")
                 (39761 . "도깨비 리") 32410 (35981 . "삵 리") (37008 . "이어질 리")
                 (28054 . "다다를 리") 40477 25691 37288 33950 (21737 . "어조사 리")
                 34362 35125 32301 30432 21145 23771 24434 24733
                 27142 29314 34110 (34570 . "참조개 리") (40656 . "끈끈이 리")
                 20460 23052 26819 27650 17254 18032 31729 32545
                 34016 36902 38146 40100 40514 (21114 . "벗길 리")
                 20835 21013 21075 21937 22175 22212 23091 23371
                 23415 23778 23794 24424 24737 25889 26793 26808
                 28016 29339 29441 29709 29948 29957 30557 30621
                 30678 31346 31472 31587 31898 31924 31950 32332
                 33079 33347 33650 33758 34765 34859 35623 37611
                 37616 37637 39681 39887 39916 40630 13503 14052
                 14630 14670 14738 15386 15520 15856 15898 15944
                 16115 16150 16278 16668 16763 16848 17037 17125
                 17248 17670 17853 17855 18069 18667 18714 18820
                 18889 19228 19544 19771 19795))
       (47536 . ((38563 . "鄰의 俗字, 이웃 린") (40607 . "기린 린")
                 (40023 . "비늘 린") (21533 . "아낄 린, 인색할 린") (36522 . "짓밟을 린, 짖밟을 린")
                 (29136 . "도깨비불 린, 반딧불 린") (29848 . "옥빛 린")
                 (34298 . "골풀 린, 뇌양이 린") (28542 . "맑을 린, 물맑을 린")
                 (29532 . "튼튼할 린") (37168 . "이웃 린") (30967 . "돌 을 틈물이 흐르는 모양 린")
                 (31932 . "물 맑을 린") 36980 (23961 . "가파를 린") (39502 . "워라말 린")
                 (36692 . "수레 소리 린") (32343 . "이을 린") 24673 30645
                 17906 30084 (24715 . "아낄 린") (36505 . "짓밟을 린")
                 25755 (37883 . "굳셀 린") 20719 21432 36709 34106
                 36495 36762 37051 40158 (31910 . "도깨비 불 린")
                 20099 22755 26036 26301 27209 28955 29968 32759
                 36472 38325 40592 13570 13697 15840 17006 18063
                 18092 18607 19184 19388 19567))
       (47548 . ((26519 . "수풀 림") (33256 . "임할 림") (28107 . "물뿌릴 림, 물댈 림")
                 (29747 . "아름다운 옥 림") (38678 . "장마 림") (30195 . "임질 림")
                 24783 20903 23818 26237 26877 28718 28726 30852
                 31638 37250 14173 14569 15689 19152))
       (47549 . ((31435 . "설 립") (31520 . "우리 립, 삿갓 립")
                 (31890 . "알 립, 쌀알 립") (30764 . "돌 소리 립") (23718 . "산 우뚝할 립")
                 38644 33497 40215 28225 37469 13688 16841 18562
                 19614 19864))
       (47560 . ((39340 . "말 마") (40635 . "삼 마") (30952 . "갈 마")
                 (25705 . "갈 마") (39764 . "마귀 마") (30194 . "저릴 마, 홍역 마")
                 (29802 . "마노 마, 옥돌이름 마") (30908 . "마노 마, 옥돌 마")
                 (40637 . "잘 마") 40636 (40639 . "갈 마") (21144 . "깍을 마")
                 34758 20103 21966 31137 17677 (23229 . "어미 마")
                 22043 (34718 . "말거머리 마") (34759 . "개구리 마")
                 37815 14724 21527 21787 22920 23351 26473 29368
                 29595 30721 31043 34434 39532 20620 21504 21750
                 22044 22162 22180 22192 23282 23332 23579 25057
                 25090 25845 27050 28324 29336 29505 31027 32817
                 34100 34278 34321 39261 39629 39701 39970 40396
                 13315 13367 16314 16634 17899 17950 18182 18645
                 18646 18910 19019 19305 19426 19592 19704))
       (47561 . ((33707 . "없을 막, 말 막") (24149 . "막 막, 장막 막")
                 (28448 . "사막 막, 아득할 막") (33180 . "막 막, 살속의 가죽 막")
                 (23518 . "쓸쓸할 막") (37000 . "멀 막, 아득할 막") 30268
                 34256 24153 (37836 . "칼 이름 막") 26287 (30617 . "눈 흐릴 막")
                 22012 37146 39235 39309 38214 21168 22651 30608
                 32312 14387 14749 15475 15852 17580 19372))
       (47564 . ((33836 . "일만 만") (28415 . "찰 만") (26217 . "저물 만, 늦을 만")
                 (24930 . "게으를 만, 거만할 만") (28459 . "질펀할 만, 물질펀한 만")
                 (34875 . "오랑캐 만") (28771 . "물굽이 만") (23081 . "해산할 만")
                 (25405 . "당길 만") (34067 . "덩굴 만") (30622 . "속일 만, 거슴츠레한 만")
                 (36627 . "끌 만, 수레끌 만") (24398 . "굽을 만, 활당길 만")
                 (39237 . "만두 만") (39995 . "뱀장어 만") (21325 . "만자 만, 범자의만 만")
                 (19975 . "일만 만") (26364 . "끌 만, 길 만") (24018 . "뫼 만, 산봉우리 만")
                 (35646 . "속일 만") 28286 26202 28288 (24148 . "막 만")
                 29107 25059 15109 (23258 . "업신여길 만") (36451 . "비틀거릴 만")
                 39707 (22657 . "흙손 만") 24727 (32309 . "무늬 없는 비단 만")
                 28587 33045 (37853 . "흙손 만") 20680 (39704 . "머리장식 만")
                 25713 30540 30677 34766 37588 14434 17469 34542
                 23782 24367 28385 30610 32550 34728 35881 36434
                 38232 39071 39314 40151 (37835 . "금 만") 21328 22598
                 22762 23640 24946 26778 27134 27168 28523 28599
                 28663 29516 34052 34352 34755 35156 36110 37156
                 38804 39010 39703 13500 13764 13768 13922 15704
                 15896 16116 16376 16715 16764 17057 17573 17749
                 17934 18225 18273 18274 18540 18701 19446 19467))
       (47568 . ((26411 . "끝 말") (38778 . "버선 말, 붉은 끈 말")
                 (25273 . "바랄 말, 바를 말") (35178 . "버선 말") (27819 . "거품 말, 물방울 말")
                 (21788 . "끝") (33545 . "말리 말") 26153 24085 (31203 . "꼴 말")
                 34972 22970 38856 30492 26554 24083 24637 28686
                 30348 30576 30750 31011 31894 35001 38273 38884
                 39785 13435 14904 15184 15788 17258 17374 19252
                 19430 19525 19762))
       (47576 . (37572 37693 37803 14055 18769))
       (47581 . ((20129 . "망할 망") (26395 . "바랄 망") (24536 . "잊을 망")
                 (24537 . "바쁠 망") (22916 . "허망할 망, 망령될 망")
                 (32596 . "그물 망, 없을 망") (33579 . "아득할 망") (32178 . "그물 망")
                 (33426 . "까끄라기 망, 싹 망") (24792 . "멍할 망") (33725 . "우거질 망, 풀이 우거진 망")
                 (37017 . "산 이름 망, 북망산 망") (36638 . "바퀴테 망")
                 (39757 . "도깨비 망") (33726 . "마디 짧은 대 망") (34770 . "이무기 망")
                 (32593 . "그물") 20158 37609 34830 (28461 . "넓을 망")
                 13498 14592 20838 13685 17092 (27730 . "황급할 망")
                 24702 26402 26455 34313 35511 34628 37359 14335
                 15122 19510 32434 36747 38099 21506 22782 22848
                 23055 26471 26850 28985 29284 29726 30173 30451
                 30821 30829 31488 (32594 . "그물") 32595 33595 33699
                 33781 34535 14284 14288 16499 16722 17118 17231
                 17550 17597 17660 18760 19491 19561 19816))
       (47588 . ((36067 . "팔 매") (36023 . "살 매") (27599 . "매양 매")
                 (22969 . "누이 매, 손아랫누이 매") (26757 . "매화나무 매")
                 (23186 . "중매 매") (22475 . "묻을 매") (26522 . "줄기 매")
                 (39749 . "도깨비 매") (26151 . "새벽 매, 먼동틀 매")
                 (23504 . "잠잘 매, 잔다 매") (32629 . "욕할 매") (36993 . "갈 매, 멀리갈 매")
                 (29028 . "그을음 매") (21574 . "어리석을 매") (26995 . "梅의 古字")
                 28028 38718 (30491 . "어두울 매") (33683 . "나무딸기 매")
                 18986 31126 (29611 . "매괴 매") 27598 25274 27820
                 30167 (33530 . "딸기 매") (37238 . "술밑 매") 20293
                 20930 21233 28232 33028 38862 39393 39741 39554
                 (38665 . "매우 매") 20080 21154 21334 22770 33644
                 36808 21573 22058 22342 22650 25396 27089 29691
                 33058 33116 34130 34230 36298 37570 40438 40675
                 13889 14117 14751 15185 15691 16051 16411 16412
                 16498 16768 17032 17241 17702 18065 18197 18232
                 18690 19496 19498 19810))
       (47589 . ((40613 . "보리 맥") (33032 . "맥 맥") (35978 . "북방 종족 맥, 큰곰 맥")
                 (38476 . "두렁 맥, 밭둑길 맥") (39488 . "말탈 맥") (33033 . "脈의 古字")
                 38689 35977 38690 30525 (35971 . "오랑캐 맥") 35227
                 40614 34022 (35992 . "짐승 이름 맥") 24094 30527
                 32072 34536 34887 15737 15982 17937 18987 19374
                 19694))
       (47609 . ((23391 . "맏 맹") (30431 . "맹세할 맹") (29467 . "사나울 맹")
                 (30450 . "소경 맹, 눈어두울 맹") (33804 . "싹 맹") (27667 . "백성 맹")
                 40701 29965 30015 34673 33355 34427 34116 37171
                 38192 20762 21200 25529 25821 27223 29950 33684
                 34594 37683 39917 40710 16453 16839 16909 16986
                 17629 17823 17864 18800))
       (47616 . (20060))
       (47728 . (26048 21436 14513))
       (47729 . ((35219 . "찾을 멱, 구할 멱") (27752 . "물이름 멱")
                 33986 (20906 . "덮을 멱") 32643 (31992 . "실") 35220
                 (24142 . "덮을 멱") 24130 32415 35269 (20886 . "덮다")
                 22611 24166 28159 28446 28631 31770 31993 40719
                 15314 16959 17168 17809 18686 19373))
       (47732 . ((38754 . "낯 면") (20813 . "면할 면") (21193 . "힘쓸 면")
                 (30496 . "잠잘 면") (32191 . "이어질 면, 솜 면") (20885 . "면류관 면")
                 (27796 . "물 흐를 면") 23081 (20443 . "힘쓸 면") (32236 . "가는 실 면, 아득할 면")
                 (30468 . "애꾸눈 면, 곁눈질할 면") (40629 . "麪과 同字, 밀가루 면")
                 (26825 . "목화 면") (32220 . "햇솜 면") 38755 (40618 . "밀가루 면")
                 (31942 . "麪과 同字") (28238 . "빠질 면") 19983 20589
                 15632 21204 24848 27339 33708 33858 40619 32501
                 32517 33148 19974 21909 23106 23188 23349 (23424 . "움집")
                 27312 27717 30664 30666 30671 33265 33415 34642
                 39864 40634 13465 14063 14147 14192 14880 14906
                 15244 15363 16398 16583 18121 18162 18692 19189
                 19475))
       (47736 . ((28357 . "멸망할 멸, 멸할 멸") (34065 . "업신여길 멸, 없을 멸")
                 (34890 . "모독할 멸") (31742 . "대 껍질 멸") 13525 34843
                 25635 27351 28781 23405 24173 25073 25595 34190
                 37974 40052 40211 13469 14946 16510 16956 17193
                 17930 18552 19023))
       (47749 . ((26126 . "밝을 명") (21629 . "목숨 명") (21517 . "이름 명")
                 (40180 . "울 명, 새소리 명") (37528 . "새길 명") (20901 . "어두울 명")
                 (28319 . "어두울 명, 바다 명") (26269 . "어두울 명")
                 (30399 . "그릇 명") (34719 . "마디충 명, 며루 명") (37225 . "술 취할 명, 취한 모양 명")
                 (30609 . "눈 감을 명") (33986 . "명협 명, 담벽풀 명")
                 (33559 . "차 싹 명, 차의 싹 명") (26919 . "홈 통 명")
                 40701 14176 (27962 . "강 이름 명") 30464 20949 15077
                 35450 17347 38125 40483 20338 23027 23239 24911
                 25525 26393 27040 28989 29072 29501 30515 31549
                 35245 37133 40264 13505 13806 14320 16665 16809
                 17029 17548 19172 19679))
       (47756 . ((34946 . "소매 몌, 옷소매 몌")))
       (47784 . ((27597 . "어미 모") (27611 . "털 모, 터럭 모")
                 (26286 . "저물 모") (24917 . "그리워할 모, 사모할 모")
                 (21215 . "모을 모, 뽑을 모") (27169 . "법 모") (26576 . "아무 모")
                 (35584 . "꾀할 모") (20882 . "가릴 모, 무릅쓸 모") (20398 . "업신여길 모")
                 (35980 . "얼굴 모, 모양 모") (35624 . "꾀 모") (29279 . "소 우는 소리 모, 클 모")
                 (24125 . "모자 모") (33541 . "띠 모") (30683 . "창 모, 세모진창 모")
                 (32791 . "줄 모, 없을 모") (25720 . "찾을 모, 본뜰 모")
                 (29281 . "수컷 모") (31970 . "模의 俗字, 모호할 모")
                 (30520 . "눈동자 모") (25721 . "베낄 모") (29761 . "서옥 모, 대모 모")
                 (33468 . "풀 우거질 모, 나물 모") (22982 . "여스승 모")
                 (32772 . "늙은이 모") (20372 . "가지런 할 모") (39654 . "다팔머리 모")
                 (26052 . "깃대 장식 모") (23202 . "강새암 할 모") (34762 . "蝥와 同字")
                 20880 40624 (23275 . "예쁠 모") (30474 . "눈 흐릴 모")
                 14412 (30339 . "貌와 同字") 35625 (23013 . "할미 모")
                 (34661 . "해충 모") 27649 31183 19145 (31885 . "밀리메트로 모")
                 27205 (37502 . "칼 끝 모") 17829 29286 35851 35871
                 (24648 . "탐할 모") 20830 20867 21726 23058 23764
                 24916 26502 26626 27615 27626 27630 27639 27936
                 29519 29605 29937 30762 31443 32606 33362 33850
                 34025 34513 34640 34801 35218 36574 37205 39801
                 40254 40348 13380 13493 13499 13756 13866 14028
                 14175 14274 14791 14796 15006 15640 15992 16058
                 16350 17091 17867 18522 19319))
       (47785 . ((30446 . "눈 목") (26408 . "나무 목") (29287 . "칠 목, 기를 목")
                 (30566 . "화목할 목") (31302 . "화목할 목, 화할 목")
                 (27792 . "머리감을 목") (40361 . "집오리 목") 26968
                 33711 (33500 . "거여목 목") 38658 19697 14088 27619
                 28817 38076 40540 (20969 . "찬바람 목") 22390 26466
                 29383 34462 37484 38638 14542 16263 16434 17086
                 17525 18236 19639))
       (47792 . ((27794 . "가라 앉을 몰, 빠질 몰") (27519 . "죽을 몰")
                 27809 27521 22333 26416 27517 27518 33672))
       (47800 . (39885))
       (47805 . ((22818 . "꿈 몽") (33945 . "입을 몽, 어릴 몽")
                 (26406 . "풍부할 몽, 풍부한 몽") (30663 . "소경 몽")
                 (28635 . "가랑비 올 몽") 26790 (26330 . "어두울 몽")
                 (30626 . "어두울 몽") 25077 20897 (24170 . "덮을 몽")
                 25052 17475 34835 14355 (38650 . "안개 몽") 39259
                 (25054 . "어두울 몽") (33384 . "싸움배 몽") 30674 38719
                 18707 18249 40562 20762 28309 (40463 . "비둘기 몽")
                 22819 27308 27659 29556 29998 31006 37176 38693
                 38720 13489 13945 14132 14193 14938 15667 16388
                 17397 17477 17552 18022 18025 18405 18466 18754
                 19474 19611 19724 19775 19782))
       (47896 . ((21359 . "토끼 묘") (22937 . "묘할 묘") (24287 . "사당 묘, 종묘 묘")
                 (22675 . "무덤 묘") (31186 . "까끄라기 묘") (33495 . "모 묘, 싹 묘")
                 (29483 . "고양이 묘") (26164 . "별자리 이름 묘, 별이름 묘")
                 (25551 . "그릴 묘") 30045 (28218 . "아득할 묘") (26483 . "어두울 묘, 아득할 묘")
                 (31447 . "땅 이름 묘") (37672 . "닻 묘") 24217 (30471 . "애꾸눈 묘")
                 (34256 . "작을 묘") 32242 30034 29573 (28156 . "물 아득할 묘")
                 (35987 . "猫와 同字") 33542 31694 32520 38170 40523
                 20871 21941 23180 23289 23745 24255 25148 27862
                 28213 30022 30596 31543 32226 35205 40025 40339
                 13412 14009 16487 16742 17370 17826))
       (47904 . (20078))
       (47924 . ((28961 . "없을 무") (27494 . "굳셀 무, 자만할 무, 군인 무, 호반 무")
                 (21209 . "일 무, 힘쓸 무, 권장할 무") (25098 . "다섯째 천간 무, 무성할 무")
                 (33538 . "우거질 무, 무성할 무") (33310 . "춤출 무")
                 (36031 . "바꿀 무, 물건 살 무, 무역할 무") (38695 . "안개 무, 어두울 무")
                 (35491 . "무고할 무, 깔볼 무, 속일 무") (24043 . "무당 무, 산 이름 무")
                 (27595 . "말 무") (25771 . "어루만질 무, 누를 무") (30045 . "밭이랑")
                 (34154 . "거칠어질 무, 거친풀 무, 황무지 무") (25006 . "어루만질 무, 애무할 무")
                 (25287 . "엄지손가락 무") (26080 . "없을 무") (25035 . "힘쓸 무, 노력할 무")
                 (32326 . "묶을 무, 얽을 무, 삼 무") (40289 . "앵무새 무")
                 (29687 . "옥돌 무, 옥돌이름 무") (26969 . "무성할 무, 아름다울 무, 모과나무 무")
                 (24289 . "집 무, 처마 무, 지붕 무") 25242 30592 (39446 . "달릴 무, 질주할 무")
                 (33204 . "포 무, 크게 저민 고기 무") 34980 37738
                 23162 23285 30868 24160 29970 20533 22565 30644
                 33665 37479 13447 20137 21153 21586 22953 24209
                 24579 32554 33436 35820 36152 38654 39579 40512
                 40521 (20763 . "舞와 同字") (22072 . "분명하지 않을 무, 모호한 모양 무")
                 20203 21178 21603 22706 22886 23084 23885 24855
                 25924 26259 26581 27206 28501 29059 29299 29841
                 30046 30062 32991 34992 35464 35669 36359 36492
                 37166 37691 38682 38826 39667 40272 40417 14420
                 14579 15180 15242 15256 15440 15559 15730 15747
                 15867 16215 16420 16977 17143 17175 17250 17388
                 17545 17646 18428 18768 18811 18945 19541 19655))
       (47925 . ((22696 . "먹 묵, 형벌 이름 묵") 40665 (40664 . "묵묵할 묵, 고요할 묵, 잠잠할 묵")
                 (22079 . "고요할 묵, 말 아니할 묵") 32390 22172 29189
                 23292 28665 30310 34772 14927 15087 15861 16508
                 16511 17923))
       (47928 . ((38272 . "문 문") (21839 . "물을 문, 알릴 문")
                 (25991 . "무늬 문, 얼룩 문, 글월 문") (32862 . "들을 문")
                 (32010 . "어지러울 문, 얽힐 문") (27766 . "내 이름 문, 수치 문, 물이름 문")
                 (32011 . "무늬 문, 주름 문") (34442 . "모기 문") (21563 . "입술 문, 입가 문")
                 (21006 . "목 벨 문, 자를 문, 끊을 문") (20497 . "들 문, 무리 문")
                 (38639 . "구름 무늬 문") 38395 38382 38376 (25451 . "어루만질 문, 쓰다듬을 문")
                 25041 (25059 . "번거로울 문") 14742 33047 (25222 . "닦을 문, 문지를 문")
                 24542 (24727 . "잊어버릴 문") (28806 . "따뜻할 문")
                 26282 32123 34441 38335 39356 20204 25194 32441
                 38036 38412 (21249 . "몸메 문") 24543 (29834 . "붉은 옥 문")
                 20230 21245 21601 22927 24419 28162 29683 29882
                 32947 33440 33755 34379 34721 34753 37419 37702
                 38277 38330 38341 39367 39792 40188 40205 40740
                 13763 14408 15203 15693 16074 17055 17337 17341
                 17927 18370 18490 18857 19154 19482))
       (47932 . ((29289 . "만물 물, 무리 물") (21247 . "말 물, 아닐 물, 깃발 물")
                 (27797 . "아득할 물, 숨길 물, 잠길 물") 23689 26130
                 31877 15066 18812))
       (48120 . ((32654 . "아름다울 미, 맛이 좋을 미") (31859 . "쌀 미")
                 (26410 . "아닐 미") (21619 . "맛 미, 뜻 미") (23614 . "꼬리 미, 등 미")
                 (24494 . "작을 미, 자질구레할 미, 가늘 미") (36855 . "미혹할 미, 전념할 미")
                 (30473 . "눈썹 미, 노인 미") (24396 . "두루 미, 널리 미, 그칠 미")
                 (38753 . "쓰러질 미, 쏠릴 미, 흐트러질 미") (23194 . "아첨할 미, 풍치가 아름다울 미, 예쁠 미")
                 (34183 . "고삐 미, 고비 미") (26979 . "문미 미, 처마 미, 인중방 미")
                 (28228 . "물 가 미, 더운 물 미") (23883 . "산이름 미")
                 (28220 . "물놀이 미, 내 이름 미, 물결무늬 미") (35598 . "수수께끼 미")
                 (40692 . "곰팡이 미, 검을 미, 썪을 미, 기미낄 미")
                 (26806 . "나무 끝 미") (24357 . "彌와 同字") (24365 . "활고자 미, 그칠 미")
                 (31964 . "죽 미, 싸라기 미") (32315 . "고삐 미, 얽어맬 미")
                 15046 (20153 . "힘쓸 미, 문채가 있는 모양 미") 30511
                 (28720 . "물 넓을 미") (40587 . "큰사슴 미, 부서질 미")
                 (25929 . "어루만질 미") 20894 27939 32601 14578 (23226 . "착하고 아름다울 미")
                 (29564 . "원숭이 미") 40603 (23059 . "장황할 미, 힘쓸 미")
                 14135 18324 37119 14123 15613 (34364 . "장미 미, 천궁 미")
                 27992 30507 14225 29461 33416 35868 38209 38213
                 40539 (23172 . "빛 고울 미") 33535 20366 21673 21674
                 22579 23309 23370 23876 23990 24510 25444 25823
                 25879 25888 26006 28211 28326 28758 29021 29218
                 29496 29762 30291 30530 30584 30599 30656 31065
                 31304 31683 32651 33042 33353 33649 33739 33886
                 33919 33982 34077 34346 34654 35257 35448 36542
                 37274 37310 37311 37316 37540 37762 37767 38270
                 38714 39815 40357 40586 14086 14181 14300 14375
                 14534 14685 15673 15711 15887 16719 16778 16891
                 16971 16992 17010 17034 17075 17115 17173 17240
                 17658 17779 17783 18047 18151 18793 18808 18869
                 19263 19303 19530 19746))
       (48124 . ((27665 . "백성 민") (25935 . "민첩할 민") (38292 . "위문할 민, 성 민")
                 (25003 . "불쌍히 여길 민") (24758 . "번민할 민") (26107 . "하늘 민, 가을하늘 민")
                 (26108 . "온화할 민") (29641 . "아름다운돌 민, 옥돌 민")
                 (29599 . "옥돌 민") (24845 . "근심할 민, 불쌍한 민")
                 (27887 . "망할 민, 빠질 민") (23735 . "산이름 민") (32225 . "낚시줄 민, 입힐 민")
                 (40701 . "힘쓸 민, 맹꽁이 민") (38313 . "종족이름 민")
                 40702 20726 28177 (24542 . "강인할 민") 28576 (26251 . "굳셀 민")
                 (32205 . "낚시줄 민") 24924 (28259 . "낚시줄 민") 28515
                 25923 29144 30203 30727 24751 28950 32535 38389
                 38391 38397 40152 (24543 . "강인할 민") (32608 . "그물 민")
                 (33504 . "속대 민") (39989 . "대구 민") 20922 21025
                 21188 22410 22980 23823 24587 25279 25450 25967
                 29148 29720 29725 29769 30463 30856 31522 31778
                 37425 37641 37746 40214 14262 14313 14317 14659
                 14679 14712 14857 15075 15110 16469 16549 16585
                 17099 18279 18408 18513 18555 19128 19588))
       (48128 . ((23494 . "빽빽할 밀") (34588 . "꿀 밀") (35600 . "고요할 밀")
                 23443 23483 35879 (27154 . "침향 밀") (28405 . "물 흐르는 모양 밀")
                 22055 23770 27329 28135 34084 34252 34848 15064
                 15604 15669 15717 16455 17167 18129 18697))
       (48149 . ((26420 . "나무껍질 박, 질박할 박") (36843 . "닥칠 박, 핍박할 박")
                 (21338 . "넓을 박") (27850 . "머무를 박, 배댈 박") (25293 . "칠 박")
                 (34180 . "숲 박, 엷을 박") (39361 . "얼룩말 박") (33334 . "배 박, 큰배 박")
                 (21085 . "벗길 박, 찢을 박") (32283 . "묶을 박") (25778 . "업어질 박, 부딪칠 박")
                 (25615 . "칠 박, 두드릴 박") (27192 . "통나무 박, 순박할 박")
                 (29632 . "호박 박") (31636 . "발 박") (31893 . "지게미 박")
                 (33162 . "포 박, 어깨 박") (38649 . "누리 박, 우박 박")
                 (29854 . "옥덩어리 박") 21093 25920 (20147 . "땅이름 박")
                 31028 35182 24313 24893 19112 (39406 . "짐승이름 박")
                 (37787 . "종 박") 31009 27279 25800 29055 37998 22691
                 24436 31367 14951 30632 18796 32538 38027 38082
                 38216 38244 39121 39310 39539 (27394 . "주두 박")
                 (29268 . "박공 박") (39622 . "어깨쭉지 박") 21866 22103
                 22167 22308 25066 28807 28894 29350 29403 29500
                 29919 30774 31769 32969 33354 33825 33962 36277
                 37337 37457 37879 39086 39226 39602 39625 13687
                 13784 13944 15117 15638 15713 15954 16042 16142
                 16378 16525 16599 16696 17272 17285 17480 17882
                 17999 18788 18956 18964 18979 18992 19079 19097
                 19242 19302 19328 19466 19848))
       (48152 . ((21322 . "반 반") (21453 . "돌이킬 반") (39151 . "밥 반")
                 (29677 . "나눌 반, 벌려설 반") (33324 . "옮길 반, 되돌아올 반")
                 (36820 . "돌이킬 반, 돌아올 반") (20276 . "짝 반") (30436 . "소반 반")
                 (21467 . "배반할 반") (38930 . "나눌 반, 반포할 반")
                 (27075 . "쟁반 반") (26001 . "얼룩 반, 아롱질 반") (28504 . "뜨물 반, 물이름 반")
                 (25644 . "옮길 반") (30971 . "강이름 반, 반계 반") (25856 . "더위잡을 반, 끌어당길 반")
                 (30036 . "지경 반, 밭두둑 반") (34784 . "서릴 반, 쥐며느리 반")
                 (32070 . "줄 반, 얽을 반") (31020 . "백반 반") (25292 . "버릴 반")
                 (27886 . "학교 반, 얼음풀릴 반") (30460 . "곁눈질 할 반, 눈자위 반")
                 (30928 . "넓을 반, 반석 반") (30242 . "흉터 반, 부스럼자국 반")
                 30424 (32982 . "희생 반") 36451 35195 38838 (25203 . "끌 반")
                 30285 (25643 . "덜 반") (38934 . "풀릴 반") 24995 35402
                 (25917 . "나눌 반") (26380 . "구실 반") 26002 29257
                 14134 15293 18112 (20206 . "反과 同字") 30718 32458
                 39041 39277 (23227 . "비틀거릴 반") (32934 . "나눌 반")
                 (34700 . "가뢰 반") 20920 22370 22884 22981 23119
                 23310 23316 24139 24593 26568 27804 28351 28682
                 28813 31194 32271 33968 34408 35113 36312 36780
                 37473 37612 37788 38011 38342 38781 39152 39788
                 40187 13364 13976 14486 15029 15072 15207 15767
                 16195 16249 16320 16593 16626 16826 16946 16970
                 17021 17065 18026 18506 18674 19251 19465 19476))
       (48156 . ((30332 . "쏠 발, 일어날 발") (39662 . "터럭 발")
                 (25300 . "뺄 발") (36299 . "밟을 발") (37474 . "바리때 발")
                 (28497 . "뿌릴 발, 활발할 발") (28196 . "바다이름 발")
                 (25765 . "다스릴 발") (21187 . "갑작스러울 발, 발끈할 발")
                 (39747 . "가물 귀신 발") (37297 . "술 괼 발") 38069
                 23387 (28001 . "일어날 발") 30330 24386 32141 39658
                 27831 14989 (37432 . "방울 발") 22690 33543 14814
                 (40275 . "집비둘기 발") 18406 (21745 . "어지러울 발")
                 34970 35151 25244 29358 36599 13814 26605 31201
                 32968 37882 39326 18670 21457 25320 27900 34991
                 37222 38073 39293 40069 40513 (33046 . "배꼽 발")
                 20412 21456 22394 22957 24834 25388 26738 28300
                 28838 (30326 . "걷다") 30329 30411 30773 32573 33656
                 33757 33895 35417 36467 37091 37271 37581 39185
                 39323 39499 39809 40013 40741 13575 13596 14040
                 14066 14216 14289 14794 15678 15807 16052 17256
                 17952 18088 18395 18572 18666 18813 19116 19311
                 19330 19403 19649 19658))
       (48164 . (28276))
       (48169 . ((26041 . "모 방, 방위 방") (25918 . "놓을 방")
                 (38450 . "둑 방, 막을 방") (25151 . "방 방") (35370 . "찾을 방, 찾아볼 방")
                 (37030 . "나라 방") (22952 . "방해할 방, 거리낄 방")
                 (33459 . "꽃다울 방, 향기 방") (20621 . "곁 방") (20515 . "본뜰 방, 본받을 방")
                 (26049 . "두루 방, 넓을 방") (24439 . "거닐 방, 방황할 방")
                 (32033 . "자을 방, 길쌈 방") (40848 . "어지러울 방, 흑백의 잡색 방")
                 (22346 . "동네 방, 고을이름 방") (27036 . "매 방, 게시판 방")
                 (26121 . "마침 방, 밝을 방") (35607 . "헐뜯을 방, 나무랄 방")
                 26834 (26507 . "다목 방, 박달 방") (24135 . "幫과 同字, 도울 방")
                 (32938 . "기름 방, 비계 방") (23592 . "삽살개 방") (33152 . "쌍배 방, 방광 방")
                 (28354 . "비 퍼부을 방, 기름질 방") (34444 . "방합 방, 금조개 방")
                 (30917 . "돌 떨어지는 소리 방") (33323 . "배 방, 사공 방")
                 (33953 . "인동 덩굴 방, 우엉 방") 38710 (20223 . "헤멜 방")
                 (29267 . "패 방") (39651 . "비슷할 방") (21398 . "클 방")
                 32129 36868 37377 38641 29429 15236 (39796 . "방어 방")
                 (25618 . "배저을 방") 24236 (24171 . "도울 방") 21732
                 22469 33006 39417 (34691 . "방게 방") 26136 27768
                 27997 28115 30470 33174 34436 14797 16749 18982
                 24110 24222 32442 32465 35775 35876 38059 38225
                 40066 40145 (24492 . "시중들 방") (26058 . "옹기장 방")
                 (26758 . "목어 방") (33341 . "배 방") (37770 . "깎을 방")
                 40846 (21274 . "상자") 21977 22457 22543 22596 22621
                 23246 24154 25399 27654 28816 29285 29307 29604
                 29932 31254 31715 32269 32810 32936 33365 34518
                 34607 35243 36285 37025 37035 37690 38820 39624
                 39967 40203 40365 13378 13673 13675 14161 14595
                 14646 14700 14861 15044 15179 15222 16374 16540
                 16689 17402 17406 17549 17835 18457 18573 18614
                 18817 18907 19063 19511 19633))
       (48176 . ((21271 . "달아날") (25308 . "절 배") (26479 . "잔 배")
                 (36649 . "무리 배") (32972 . "등 배") (37197 . "아내 배, 짝 배")
                 (20493 . "곱 배, 갑절 배") (22521 . "북돋을 배, 북돋울 배")
                 (25490 . "밀칠 배, 물리칠 배") (35061 . "성") (20467 . "광대 배")
                 (36064 . "물어줄 배") (38506 . "쌓아올릴 배, 모실 배")
                 (24472 . "노닐 배, 배회할 배") (32986 . "아이 밸 배")
                 (28227 . "물결 이는 모양 배, 물소리 배") (35060 . "옷 치렁치렁할 배, 옷이 치렁치렁한 모양 배")
                 (30403 . "杯의 俗字, 잔 배") (35097 . "속적삼") (28953 . "불에 쬘 배, 말릴 배")
                 36592 (22351 . "언덕 배") 29615 26734 36744 25309
                 37253 31129 26616 27632 (22383 . "坏와 同字") (29746 . "구슬 꿰미 배")
                 32935 14000 36180 (25170 . "뺄 배") (34003 . "꽃봉오리 배")
                 20243 20438 20573 21857 22938 23951 25520 25593
                 28811 29445 29731 30874 33622 34012 34883 36651
                 38443 38875 13608 13777 14301 14356 14643 15246
                 15297 15366 15764 15784 15786 16087 16294 17108
                 17139 17556 17682 18649 19129 19146))
       (48177 . ((30334 . "일백 백") (30333 . "흰 백") (20271 . "맏 백")
                 (26658 . "柏의 俗字, 잣나무 백") (24091 . "비단 백")
                 (39748 . "넋 백") (26575 . "나무 이름 백, 잣나무 백")
                 (20336 . "백사람의어른") 33513 37510 27942 40076
                 (31912 . "힉터메트르 백") 20833 23734 28240 29944
                 31457 32084 39818 13542 14413 16159 16163 16351
                 18335 19654))
       (48192 . (13745))
       (48264 . ((30058 . "갈마들 번, 번수 번") (29033 . "괴로워할 번, 번거로울 번")
                 (32321 . "많을 번, 성할 번") (39132 . "뒤칠 번, 날 번")
                 (30971 . "강이름 번") (34115 . "우거질 번, 불 번") (34281 . "덮을 번, 울타리 번")
                 (29140 . "구울 번") (24161 . "기 번, 수건 번") (27146 . "울 번, 어수선할 번")
                 (32763 . "날 번") 30056 (34345 . "산흰쑥 번") 29856
                 (32345 . "되풀이 풀이할 번") 26075 (33200 . "제사 고기 번")
                 36691 26073 28735 31538 36463 22694 27214 16021
                 23311 31827 34844 28902 (34978 . "속옷 번") 20704
                 21227 22083 22911 26853 28714 32208 32691 34208
                 35150 37895 37953 40021 40429 15883 16978 17123
                 17126 17171 17776 19099 19108 19190 19379))
       (48268 . ((20240 . "칠 벌") (32624 . "죄 벌, 벌줄 벌")
                 (38309 . "공훈 벌, 가문 벌") (31567 . "떼 벌") (27203 . "떼 벌")
                 32602 38400 (32632 . "罰과 同字") 20640 22433 26672
                 27980 30594 34245 13689 13882 16837 17492 18681))
       (48276 . ((20961 . "무릇 범") (29359 . "범할 범") (31684 . "법 범")
                 (33539 . "풀 이름 범, 벌 범") (27726 . "뜰 범") (27867 . "뜰 범")
                 (24070 . "돛 범") (26805 . "범어 범, 웅얼거릴 범") (27710 . "넘칠 범, 물넘칠 범")
                 20962 36563 (39103 . "돛 범") 26443 14390 33316 13672
                 19317 38034 (31541 . "법 범") (35337 . "말 많을 범")
                 20963 24539 26569 28412 30421 33319 35546 37353
                 15616 16384 17530 17574 17582 18410 19226))
       (48277 . ((27861 . "법 법") (29754 . "법랑 법, 유약 법")
                 15570 30138 29648 20337 28747 37701))
       (48317 . ((22721 . "벽 벽") (30887 . "푸를 벽") (20731 . "후미질 벽")
                 (38370 . "열 벽") (29863 . "동근 옥 벽, 둥근옥 벽")
                 (30294 . "적취 벽, 버릇 벽") (21128 . "쪼갤 벽") (25816 . "엄지손가락 벽, 나눌 벽")
                 (38713 . "벼락 벽") (34327 . "황경나무 벽") (27287 . "황벽나무 벽, 황백나무 벽")
                 (36767 . "임금 벽") 34204 (25815 . "가슴칠 벽") 30357
                 (29971 . "벽돌 벽") 36484 26897 (35166 . "주름 벽")
                 (30088 . "가를 벽") 36483 28258 28604 32188 37940
                 17183 40126 19737 (40447 . "농병아리 벽") (40714 . "거북 벽")
                 22140 22555 23417 24294 25013 25344 25546 27288
                 30996 30997 31275 31978 32372 37373 37538 39951
                 40458 13856 14909 15480 15720 15862 17472 18728
                 18903 19360))
       (48320 . ((20415 . "똥오줌 변, 문득 변") (35722 . "변할 변")
                 (37002 . "가 변") (36783 . "말잘할 변") (36776 . "분별할 변")
                 (21342 . "조급할 변, 성 변") (24321 . "고깔 변") (39432 . "나란히 할 변")
                 29613 (31849 . "제기 이름 변") 25306 37001 (25219 . "손뼉칠 변")
                 24487 (24557 . "기뻐할 변") 22793 27764 (37318 . "분별하다")
                 21464 36794 36782 (39394 . "騈의 俗字") 26154 (33089 . "살갗 틀 변")
                 16057 21310 36336 36038 33020 19131 19329 31550
                 33476 36777 36779 36793 39560 33849 (39615 . "통갈비 변")
                 (40216 . "매 변") 21285 23749 27763 28830 29375 29603
                 29954 31272 31940 34250 35213 36769 38302 14073
                 14200 14544 14746 15187 15522 15566 15735 16088
                 17016 17428 17578 18130 18381))
       (48324 . ((21029 . "나눌 별, 분별 별") (30629 . "언뜻 볼 별, 눈을 깜짝하는 별")
                 (40009 . "鼈과 同字, 자라 별") (40712 . "자라 별") 21035
                 25735 36457 33666 (40425 . "금계 별") 28494 18469
                 30250 40150 (24390 . "활 뒤틀릴 별") (20031 . "삐치다")
                 23283 24502 24971 26300 30303 34380 34498 34782
                 35154 35221 37893 40862 13601 14524 16348 16887
                 17122 17176 17375 17975 18404 18722 18773 18950
                 19313 19684))
       (48337 . ((20853 . "군사 병") (30149 . "병 병, 병들 병")
                 (19993 . "남녘 병") (31453 . "아우를 병") (23643 . "병풍 병")
                 (31177 . "잡을 병") (28851 . "밝을 병") (26564 . "자루 병")
                 (20482 . "아우를 병, 나란할 병") (26170 . "昞과 同字, 밝을 병")
                 (26142 . "밝을 병") (29953 . "병 병") (39200 . "떡 병, 밀가루떡 병")
                 (24183 . "아우를") (39432 . "나란히 할 병, 두말이 한멍에 멘 병")
                 27996 (36647 . "거마 소리 병") (26821 . "柄과 同字, 자루 병")
                 (20006 . "竝과 同字") 24182 23631 20341 39173 36908
                 (36856 . "달아날 병") 29942 (39394 . "騈의 俗字") (32574 . "두레박 병")
                 24114 37044 25618 20641 38467 24097 24626 26655
                 31470 32864 14848 (22592 . "담 병") (32099 . "무늬없는 비단 병, 이을 병, 먹줄 튀길 병")
                 20050 24136 24690 27956 30009 33514 34499 34530
                 36607 38953 16898 39292 37429 (37500 . "판금 병")
                 (37618 . "넓은 못 병, 징 병") 20555 22442 23502 24240
                 25254 25440 25682 26922 27678 29717 30506 31369
                 31667 31752 33397 33619 35457 36395 37105 38790
                 39849 40295 13593 13891 16066 17105 17224 17674
                 17874 19003 19459 19765 19844))
       (48372 . ((27493 . "걸을 보, 걸음 보") (20445 . "지킬 보, 도울 보")
                 (22577 . "갚을 보") (23542 . "보배 보") (35036 . "기울 보, 도울 보")
                 (26222 . "널리 보, 넓을 보") (35676 . "계보 보, 족보 보")
                 (36628 . "덧방나무 보, 도울 보") (29995 . "클 보") 36875
                 (22561 . "작은 성 보") (28541 . "물이름") (33769 . "보살 보, 보리나무 보")
                 (27921 . "보 보, 보마기 보") (35091 . "포대기 보") (29668 . "寶의 古字, 보배 보")
                 (28282 . "보") 23453 38732 (31776 . "제기 이름 보")
                 (20428 . "도울 보") (40700 . "수 보") (33862 . "풀 더부룩할 보")
                 23539 33670 27497 36356 26938 36082 25431 38836
                 (40199 . "능에 보") 22562 34293 13915 25253 34917
                 35889 36741 38248 40488 (30425 . "簠의 本字") 22487
                 23212 23514 26268 27496 28915 29042 32229 34132
                 34469 35561 37920 39362 39852 40181 13893 15835
                 16068 16073 16287 16431 17120 17520 19017 19417
                 19605 19656 19696 19728))
       (48373 . ((31119 . "복 복") (26381 . "옷 복, 입을 복")
                 (24489 . "돌아올 복") (20239 . "엎드릴 복") (21340 . "점 복, 점칠 복")
                 (33145 . "배 복") (35206 . "뒤집힐 복, 도리어 복") (35079 . "겹옷 복")
                 (39333 . "향기 복") (20693 . "종 복, 시중꾼 복") (25778 . "닦을 복")
                 (36667 . "바퀴살 복") (21264 . "길 복, 엉금엉금길 복")
                 (39954 . "전복 복") 27921 (23443 . "성 복, 편안할 복")
                 (33583 . "복령 복") (34068 . "무우 복, 치자꽃 복") (36665 . "복토 복, 바퀴통 복")
                 20166 34993 (25169 . "칠 복") (24158 . "건 복") (28654 . "강 이름 복")
                 13458 (40297 . "새 이름 복") 33744 (25908 . "치다")
                 33869 (25909 . "치다") (34670 . "살무사 복") 26687 35204
                 26949 35142 (22691 . "흙덩이 복") 37293 (34656 . "박쥐 복")
                 28563 22797 36752 40134 (31641 . "전동 복") (33748 . "무 복")
                 22424 26868 26929 29531 30032 30039 30273 31274
                 31321 32101 32384 34151 34393 34418 35173 35560
                 36108 36414 36476 36688 37713 37730 39858 40170
                 40349 40468 13504 13903 13958 14341 15164 15351
                 15531 16636 16858 16911 17489 17742 17744 17905
                 18355 18414 18916 18993 19073 19165 19718 19799))
       (48376 . ((26412 . "밑 본, 근본 본") 21600 26957 32074
                 14455 15290 15355 15793 19249))
       (48380 . ((20086 . "땅이름")))
       (48393 . ((22857 . "받들 봉") (36898 . "만날 봉") (23553 . "봉할 봉")
                 (40179 . "봉새 봉") (23791 . "봉우리 봉, 산봉우리 봉")
                 (34562 . "벌 봉") (34028 . "쑥 봉") (20472 . "녹 봉") (32299 . "꿰맬 봉")
                 (25447 . "받들 봉") (37586 . "칼끝 봉, 칼날 봉") (28925 . "봉화 봉")
                 (26834 . "몽둥이 봉") (23792 . "峯과 同字, 봉우리 봉")
                 (29090 . "연기 자욱할 봉, 불기운 봉") (29739 . "칼집 장식 봉, 칼집장식옥 봉")
                 (20016 . "예쁠 봉") 31688 (32152 . "縫과 同字") 40175
                 (31735 . "뜸 봉") 22620 33873 36085 33411 25408 33346
                 34861 39676 (22790 . "끌 봉") (33782 . "풀 무성할 봉")
                 21802 23862 28110 33681 37856 20964 32541 36183
                 38155 (40204 . "鳳의 古字") 20217 22468 22514 22643
                 22950 25683 26747 27120 27173 28018 28247 28292
                 28456 28968 29326 29714 29997 30461 31484 32388
                 34325 35202 37821 37981 38811 38904 38908 39700
                 14036 14082 14382 14429 14720 15803 15848 15853
                 15855 15932 16113 16433 16775 17149 17568 17894
                 17900 18012 18539 18895 19052 19068 19312 19390
                 19678 19766))
       (48512 . ((29238 . "아비 부") (19981 . "아닐 부") (22827 . "지아비 부")
                 (37096 . "거느릴 부, 나눌 부") (23500 . "가멸 부, 부자 부")
                 (24489 . "다시 부, 돌아올 부") (21542 . "아닐 부") (23142 . "며느리 부")
                 (28014 . "뜰 부") (25206 . "도울 부, 붙들 부") (24220 . "곳집 부, 마을 부")
                 (21103 . "버금 부") (20184 . "줄 부, 부탁할 부") (36000 . "질 부, 짐질 부")
                 (36070 . "구실 부") (22521 . "무덤 부") (36212 . "나아갈 부, 달릴 부")
                 (38468 . "붙을 부, 의탁할 부") (31807 . "장부 부, 문서 부")
                 (31526 . "부신 부, 병부 부") (33104 . "썩을 부") (37340 . "가마 부")
                 (20613 . "스승 부") (25975 . "펼 부, 베풀 부") (35331 . "부고 부, 통부 부")
                 (38428 . "언덕 부") (33178 . "살갗 부, 피부 부") 20443
                 (20463 . "구푸릴 부, 구부릴 부") (36091 . "부의 부")
                 (26023 . "도끼 부") (21078 . "쪼갤 부") (33433 . "부용 부, 연꽃 부")
                 (39385 . "곁마 부, 가까울 부") (33105 . "장부 부") (22496 . "선창 부, 부두 부")
                 (21648 . "분부할 부") (23413 . "알 깔 부") (23386 . "미쁠 부, 믿을 부")
                 (28325 . "펼 부, 클 부") (40167 . "오리 부, 물오리 부")
                 (32566 . "장군 부") (33705 . "풀 이름 부, 갈대청 부")
                 (36282 . "책상다리할 부, 도사리어 앉을 부") (33344 . "작은 배 부")
                 (20166 . "엎드릴 부") (31060 . "합사할 부") (20440 . "사로잡을 부")
                 23554 (37383 . "도끼 부") (25290 . "어루만질 부") (35026 . "모을 부")
                 (26740 . "마룻대 부") (34048 . "빈지문 부") (27025 . "부상 부")
                 (25482 . "그러모을 부") 20165 (33531 . "귀목풀 부")
                 (39826 . "붕어 부") (34569 . "하루살이 부") 30177 29951
                 20904 26835 34973 37083 (38955 . "머리 숙일 부")
                 (25236 . "움킬 부") (28074 . "물거품 부") 37345 (40617 . "밀기울 부")
                 (29598 . "옥돌 부") (36311 . "발등 부") 24045 28367
                 40172 (32600 . "그물 부") (34472 . "파랑강충이 부")
                 23555 24612 25418 26574 (31583 . "대청 부") (32981 . "장부 부")
                 21637 26250 30726 31231 15210 (32614 . "그물 부")
                 (33443 . "질경이 부") 22185 24355 25452 26510 29315
                 29879 33839 35394 37639 13537 14392 15820 17712
                 18302 19149 20971 22919 24576 32932 35747 36127
                 36171 36185 38187 39544 40075 40632 (23181 . "婦와 同字")
                 20245 20585 21115 21199 21541 21579 21753 22399
                 22458 22923 22983 23056 23069 23108 23754 24044
                 24269 25071 25241 26057 26920 27541 27885 28179
                 28912 28964 29624 29704 30025 30935 31139 31235
                 31438 31617 31728 31920 31952 32017 32040 32146
                 32238 32569 32571 33117 33652 34489 34519 34652
                 34925 35092 35943 36366 37022 37081 37110 37148
                 37212 37468 (38429 . "우부방") 38490 38636 39083
                 39202 39732 40186 40192 40620 40625 40633 64019
                 13642 13678 14282 14379 14380 14612 14641 14665
                 14837 14864 14948 15065 15205 15412 15511 15706
                 15814 16236 16264 16548 16639 16686 17087 17128
                 17228 17229 17234 17293 17300 17319 17405 17511
                 17536 17540 17543 17615 17640 17653 17860 17920
                 18332 18351 18360 18388 18485 18509 18851 18967
                 19092 19094 19215 19320 19355 19453 19536 19669
                 19677 19768 19770 19838))
       (48513 . ((21271 . "북녘 북, 달아날 북") 36387 20720
                 37491))
       (48516 . ((20998 . "나눌 분") (24996 . "결낼 분, 분할 분")
                 (22894 . "떨칠 분, 성낼 분") (31881 . "가루 분") (32027 . "어지러워질 분, 어지러울 분")
                 (22868 . "달릴 분, 달아날 분") (22707 . "무덤 분") (38930 . "큰 머리 분")
                 (33452 . "향기로울 분") (28954 . "불사를 분") (30406 . "동이 분")
                 (24575 . "성낼 분, 분할 분") (31966 . "똥 분") (22132 . "뿜을 분, 뿜어낼 분")
                 (38640 . "안개 분") (25198 . "꾸밀 분, 잡을 분") (21545 . "뿜을 분")
                 (36033 . "클 분") (27774 . "클 분, 물이름 분") (26128 . "햇빛")
                 (20307 . "용렬할 분") (27675 . "기운 분") 20712 (26876 . "마룻대 분")
                 (22348 . "먼지 분") (29319 . "달아날 분") (28614 . "뿜을 분")
                 (34145 . "들깨 분") 34465 40642 (26508 . "나무 이름 분")
                 32706 34927 (30042 . "삼태기 분") 29602 31182 32693
                 40726 (26380 . "頒과 同字") (28243 . "용솟음할 분")
                 (36690 . "병거 분") 29132 29931 35356 35958 36526
                 37948 39257 20606 21943 22367 24868 31914 32439
                 35950 36146 38171 40124 (24073 . "걸레 분, 행주 분")
                 26875 (30735 . "큰 소리 분") (31528 . "거칠 분") (32934 . "머리 클 분")
                 (33209 . "고기 삶은 국 분") (40698 . "옷에 오색 수 놓을 분")
                 (40738 . "두더지 분") 20532 20829 21935 22347 22873
                 22946 23694 24169 24325 25465 25770 26374 26739
                 26788 27240 27477 27853 28160 28725 28803 29139
                 29526 29936 31445 32658 32713 32760 32833 33519
                 33872 33974 34464 36588 36669 36905 37210 37398
                 37659 38571 39220 39322 39337 39797 40029 13753
                 14243 14603 14611 14713 14887 15269 15331 15477
                 15818 15918 16334 19071 19323 19717))
       (48520 . ((19981 . "아닐 불") (20315 . "부처 불") (25282 . "떨 불, 떨칠 불")
                 (24343 . "아닐 불, 말 불") (24447 . "비슷할 불") 21640
                 (40699 . "수 불") (32049 . "인끈 불") 24063 (39668 . "비슷할 불")
                 32060 (31059 . "푸닥거리할 불") 24619 (33536 . "풀 우거질 불")
                 33470 26146 21020 (23722 . "산길 불") (33396 . "발끈할 불")
                 22386 24087 29016 31536 20175 25173 32450 32459
                 38888 (38861 . "폐슬 불") 20032 20921 26603 26811
                 27679 28837 30006 32711 37464 39812 13868 13973
                 14264 14980 16059 18334 19310))
       (48531 . (13722))
       (48533 . ((26379 . "벗 붕") (23849 . "무너질 붕, 산무너질 붕")
                 (40300 . "대붕새 붕") (26842 . "시렁 붕, 사다리 붕")
                 (30844 . "붕산 붕, 약이름 붕") (32323 . "묶을 붕") 33824
                 24376 39685 22644 22539 28464 25508 26902 32179
                 32503 38234 40527 20503 20656 22051 28969 30189
                 36454 37872 14666 15135 15727 16090 18006 18972))
       (48708 . ((38750 . "아닐 비") (40763 . "코 비") (24754 . "슬플 비")
                 (20633 . "깆출 비, 갖출 비") (27604 . "견줄 비") (39131 . "날 비")
                 (31192 . "숨길 비, 비밀할 비") (36027 . "쓸 비, 허비할 비")
                 (22915 . "왕비 비") (21329 . "낮을 비") (30865 . "돌기둥 비, 비석 비")
                 (32933 . "살찔 비") (25209 . "칠 비, 비평할 비") (23138 . "여자종 비, 여종 비")
                 (21290 . "대상자 비") (19989 . "클 비, 으뜸 비") (27606 . "삼갈 비")
                 (27852 . "샘물 흐르는 모양 비, 졸졸흐르는 모양 비")
                 (27608 . "毗와 同字, 밝을 비") (37145 . "다라울 비, 더러울 비")
                 (25161 . "문짝 비, 사립문 비") (35692 . "비유할 비")
                 (24199 . "덮을 비") (33218 . "팔 비") (35048 . "도울 비")
                 (27832 . "끓을 비") (24970 . "고달플 비") (32203 . "붉은 빛 비, 붉은 비단 비")
                 (33086 . "지라 비, 비위 비") (35513 . "헐뜯을 비, 그르다할 비")
                 (30202 . "암메추라기 비, 새이름 비") (34586 . "바퀴 비, 풍뎅이 비")
                 (29749 . "비파 비") (21269 . "비수 비") (30738 . "비상 비")
                 (32737 . "물총새 비, 비취새 비") (31189 . "쭉정이 비")
                 (22947 . "죽은 어미 비") 36033 (27607 . "도울 비, 밝을 비")
                 (33778 . "엷을 비, 나물이름 비") (26000 . "오락가락할 비, 아롱질 비")
                 (31875 . "쭉정이 비") (27047 . "비자나무 비") (26503 . "비파나무 비, 참빗 비")
                 (20478 . "더할 비") 31706 (31061 . "귀신 비, 비밀 비")
                 (38303 . "문 닫을 비") (38671 . "눈 펄펄 내릴 비") (36705 . "고삐 비")
                 30016 (22318 . "무너질 비") (27800 . "강 이름 비") 30122
                 31110 (39441 . "곁마 비") (32646 . "큰 곰 비") 24192
                 (39616 . "넓적다리 비") 30193 (26832 . "도지개 비")
                 (30565 . "흘겨볼 비") (30174 . "뱃속 결릴 비") (40729 . "마상북 비")
                 35062 (37043 . "클 비") (32021 . "가선 비") (35988 . "비휴 비")
                 (36116 . "힘쓸 비") (24753 . "표현 못할 비") (26897 . "술통 비")
                 (33107 . "장딴지 비") 24243 (38516 . "성가퀴 비") (20211 . "떠날 비")
                 21850 23218 38814 (33432 . "풀 이름 비") 20286 (28125 . "강 이름 비")
                 (31718 . "빗치개 비") 22896 31200 (33470 . "작은 모양 비")
                 (28638 . "물 소리 비") (34445 . "왕개미 비") 26610 31954
                 35964 (29394 . "비비 비") 16413 (21077 . "발 벨 비")
                 (22500 . "더할 비") (33798 . "비해 비") 22189 28164
                 34863 18105 (38836 . "풀무 비") 40303 20475 20601
                 21406 23123 23251 23645 26162 31620 32859 33165
                 38507 39329 14924 18741 (30201 . "저릴 비") 22791
                 24811 32432 32495 32628 35837 36153 36177 36756
                 38042 38212 39134 40113 (23617 . "방귀 비") (28128 . "강 이름 비")
                 (29385 . "삵의 새끼 비") (34006 . "아주까리 비") (37099 . "고을 이름 비")
                 (39451 . "빠른 말 비") 20471 20675 21624 21860 22141
                 22154 22354 22720 22838 22876 23124 23350 23620
                 23727 23845 24588 24706 25271 26243 26314 26375
                 26428 26504 27020 27614 27636 28178 28983 29333
                 29446 30017 30101 30143 30564 30919 31507 31622
                 31794 31882 32950 33076 33111 33325 33508 33801
                 33822 34140 34221 34493 34608 34609 34709 34743
                 34790 35171 35520 35917 35966 37162 37384 37402
                 37471 37645 37790 37928 38448 38566 38725 38751
                 39133 39205 39379 39660 39790 39806 39905 13421
                 13576 13591 13627 13792 13818 13848 13865 14386
                 14425 14529 14705 14809 14970 14991 15093 15280
                 15594 15698 15738 15906 15939 16061 16176 16225
                 16345 16363 16415 16419 16505 16638 16783 16849
                 16904 17022 17036 17063 17333 17476 17544 17729
                 17751 17953 17997 18076 18096 18125 18442 18443
                 18527 18550 18661 18703 19005 19006 19008 19009
                 19088 19103 19137 19148 19168 19232 19334 19441
                 19457 19494 19641 19773))
       (48712 . ((36007 . "가난할 빈") (36051 . "손 빈") (38971 . "자주 빈")
                 (24428 . "빛날 빈") (23338 . "아내 빈, 귀녀 빈") (27567 . "염할 빈")
                 (28657 . "물가 빈") (22188 . "찡그릴 빈") (28693 . "물가 빈")
                 36103 (29277 . "암컷 빈") (25996 . "빛날 빈, 문채날 빈")
                 (27996 . "물가 빈, 갯고랑 빈") (29613 . "구슬 이름 빈")
                 (27315 . "빈랑나무 빈") (39714 . "살쩍 빈") (25839 . "물리칠 빈")
                 (20752 . "인도할 빈") (34315 . "네가래 빈") (37024 . "나라 이름 빈")
                 33529 (39024 . "찡그릴 빈") (32381 . "어지러울 빈")
                 (35955 . "나라 이름 빈") 36052 (38694 . "옥 광채 빈")
                 (37964 . "강철 빈") 17305 20221 39338 29880 39657
                 (39682 . "鬢의 俗字") 34226 34841 28661 (30665 . "찡그릴 빈")
                 (33231 . "종지뼈 빈") 26809 20647 23252 23486 25672
                 27103 27553 28392 28626 32548 33169 36139 38228
                 39057 39078 39628 39699 16013 23078 26409 26901
                 29553 29816 30999 31334 35945 39022 39518 39637
                 14295 14438 15357 15371 16094 18068 18924 18952))
       (48729 . ((27703 . "얼음 빙") (32856 . "찾아가다 빙, 청할 빙")
                 39342 (24977 . "기대다 빙, 의지할 빙") (39425 . "달리다 빙, 달릴 빙")
                 (20912 . "氷의 本字") (20973 . "기대다 빙") (23049 . "장가들다 빙")
                 20444 30767 16443 39563 (20980 . "凭과 同字") 20172
                 (20907 . "얼음") 24959 27893 28036 28124 38736 13512
                 15703 18147))
       (49104 . (20858 21723))
       (49324 . ((22235 . "넉 사, 넷 사") (20107 . "일 사") (39135 . "밥 먹일 사")
                 (22763 . "선비 사, 벼슬 사") (27515 . "죽을 사") (24605 . "생각할 사")
                 (20351 . "하여금 사") (21490 . "역사 사, 사기 사") (31169 . "사사 사")
                 (23546 . "절 사") (24107 . "스승 사") (20181 . "벼슬할 사")
                 (32114 . "실 사") (33293 . "집 사") (35613 . "사례할 사")
                 (24051 . "여섯째 지지 사, 뱀 사") (23556 . "궁술 사, 쏠 사")
                 (21496 . "맡을 사") (36781 . "말 사, 말씀 사") (20284 . "같을 사")
                 (36060 . "줄 사") (31038 . "토지의 신 사, 모일 사")
                 (26031 . "이 사") (31040 . "제사 사") (26619 . "사실할 사, 캐물을 사")
                 (27801 . "모래 사") (35422 . "말씀 사") (23531 . "베낄 사")
                 (25448 . "버릴 사") (37034 . "간사할 사") (35408 . "속일 사")
                 (26012 . "비낄 사, 비뚤어질 사") (34503 . "뱀 사") (36198 . "용서할 사")
                 (31072 . "사당 사, 봄제사 사") (27863 . "물 이름 사")
                 22472 (29509 . "사자 사") (39164 . "먹일 사") (21766 . "부추길 사, 가르칠 사")
                 (21987 . "이을 사, 대이을 사") (24473 . "옮길 사") (32023 . "깁 사")
                 (20123 . "적을 사") (22882 . "사치할 사") (28681 . "토할 사, 쏟을 사")
                 (23057 . "춤출 사, 춤추는모양 사") (40605 . "사향노루 사")
                 31761 (32902 . "방자할 사") (20447 . "기다릴 사") (20045 . "잠깐 사")
                 (20282 . "엿볼 사") (34001 . "도롱이 사") (30722 . "모래 사")
                 (33678 . "향부자 사, 귀뚜라미 사") (39391 . "사마 사")
                 (35039 . "가사 사") (26797 . "북 사") (31721 . "체 사, 왕대 사")
                 (26614 . "수저 사, 숟가락 사") (28195 . "찌끼 사") (20735 . "잘게 부슬 사, 정성없는 사")
                 36766 27086 (31458 . "기다릴 사") (34593 . "납향 사")
                 (31992 . "가는 실 사") (31525 . "상자 사") (27053 . "정자 사")
                 39427 26597 36054 (22994 . "동서 사") 28056 32796
                 (21368 . "풀 사") 34032 23651 20905 38349 31745 25749
                 36050 (26946 . "떼 사") 25394 (39387 . "달릴 사") 32567
                 36772 31136 33294 (21082 . "찌를 사") (21643 . "잠간 사")
                 34421 (27740 . "지류 사") 36765 39679 (20638 . "취하여 춤추는 모양 사")
                 31145 32898 34938 26596 21401 21956 28423 36445
                 (35223 . "엿볼 사") (39882 . "문절망둑 사") 25146 26572
                 33938 34156 37734 18627 20889 27899 19997 21389
                 21661 24072 29422 31579 32433 34547 35784 35789
                 35874 36170 36176 38090 39282 39303 39542 39543
                 40104 40502 19731 23869 (30183 . "쥐통 사") 30390
                 (39782 . "鯊와 同字") (39972 . "새우 사") 20106 20118
                 20288 20312 20531 20541 20667 20749 20777 (21430 . "사사")
                 21459 21798 21861 21939 22053 22109 22638 23088
                 23392 23787 23894 24662 25393 25592 25699 26731
                 26732 26962 27065 27165 27840 27876 27917 27977
                 28091 28334 28675 28794 29133 29293 29440 29470
                 29497 29793 30395 30751 30832 30900 30915 31127
                 31764 31853 31878 32781 32896 33679 33912 34172
                 34267 34386 34596 34692 34774 34804 35375 35484
                 35972 36421 36519 36900 37362 37366 37430 37435
                 37442 37447 37448 37488 37551 37590 37889 38319
                 39140 39223 39431 39462 39795 39867 39883 40190
                 40371 40421 40758 40772 13366 13524 13648 13660
                 13693 13759 13926 14006 14038 14081 14157 14275
                 14394 14396 14456 14916 15186 15444 15453 15514
                 15635 15660 15666 15683 15700 15740 15930 15931
                 15953 15980 16040 16048 16196 16282 16308 16313
                 16389 16661 16833 16835 16995 17315 17380 17662
                 17675 17681 17710 18128 18319 18486 18633 18675
                 18732 18801 18814 18825 19030 19170 19255 19268
                 19439 19593 19814))
       (49325 . ((25976 . "자주 삭") (21066 . "깎을 삭, 깍을 삭")
                 (32034 . "동아줄 삭") (26388 . "초하루 삭") (37984 . "녹일 삭")
                 (27082 . "창 삭") (29197 . "빛날 삭") 30687 25632 31678
                 28865 38084 (33972 . "말오줌대 삭") 21965 25585 28345
                 29537 37785 14914 15286 16421 16507 16756 17159
                 17819))
       (49328 . ((23665 . "뫼 산") (29987 . "낳을 산") (25955 . "흩을 산, 흩어질 산")
                 (31639 . "셀 산, 셈놓을 산") (37240 . "초 산, 초.실 산")
                 (20632 . "우산 산") (21034 . "깎을 산, 깍을 산") (29642 . "산호 산")
                 (30109 . "산증 산, 산중 산") (38704 . "싸라기눈 산, 싸락눈 산")
                 (33948 . "달래 산") (27733 . "오구 산, 통발 산") 29986
                 31597 (32342 . "일산 산") (35349 . "헐뜯을 산") 21024
                 39189 15002 36314 (28536 . "눈물흐를 산") 27647 (29435 . "사자 산")
                 (37855 . "대패 산") 28411 (28568 . "潸의 略字") (21111 . "깎을 산")
                 14723 (27253 . "산자 산") 20176 30176 38561 20135
                 38130 20254 22999 23402 27637 27984 35754 38377
                 39315 (22989 . "헐뜯을 산") (23423 . "쌍둥이 산") (38282 . "비색할 산")
                 21300 21377 22328 22765 23932 25676 26467 28533
                 29414 31064 31495 31749 31972 33314 34262 36216
                 36565 37014 37886 38274 39242 14077 14507 14996
                 15292 15302 15948 15978 16607 16896 16968 17074
                 17454 18098 18268 18546 18999 19758))
       (49332 . ((27578 . "죽일 살") (34081 . "내칠 살") (34217 . "보살 살")
                 (25746 . "뿌릴 살") (29022 . "죽일 살") (20087 . "음역자 살")
                 26432 25675 13322 34097 33832 27037 27175 33038
                 34372 38327 17051))
       (49340 . ((19977 . "석 삼") (21443 . "석 삼") (34072 . "인삼 삼")
                 (26862 . "나무빽빽할 삼") (28402 . "샐") (28183 . "거를")
                 (34923 . "적삼 삼, 옷 삼") (33439 . "벨 삼, 풀벨 삼")
                 (26441 . "삼나무 삼") 21445 32404 25723 (27647 . "털긴 삼")
                 (31965 . "나물죽 삼") (24417 . "터럭") (39702 . "헝클어질 삼")
                 32319 27118 35138 31303 31356 31937 38032 26905
                 27388 (37348 . "낫 삼") 20200 20437 21116 22023 24147
                 24334 26321 26926 27270 27613 28033 29337 29991
                 31938 31971 32615 33888 34017 34195 36098 37842
                 38288 14414 14540 14834 15173 15377 15701 16017
                 16400 16735 17033 17039 18759 18818 19141 19177
                 19182 19783))
       (49341 . ((25591 . "꽂을 삽") (28545 . "떫을 삽, 깔깔할 삽")
                 (39087 . "바람소리 삽") (37394 . "창 삽, 새길 삽") (27459 . "마실 삽")
                 (32739 . "운삽 삽") (37732 . "가래 삽") (21317 . "서른 삽")
                 (38670 . "가랑비 삽") 25407 28544 37789 25554 (38661 . "비올 삽")
                 29264 31633 (21820 . "쪼아먹을 삽") 38776 36287 28073
                 28171 38033 38200 39122 20571 21922 24121 27504
                 30004 31679 32732 33279 33808 35653 36970 38637
                 39354 13454 13995 14076 14929 15018 15409 15641
                 15716 16194 16905 17724 18019 18173 18250 19203
                 19210 19356 19362))
       (49345 . ((19978 . "위 상") (30456 . "서로 상") (24819 . "생각할 상")
                 (24120 . "항상 상") (36062 . "상줄 상") (21830 . "헤아릴 상, 장사 상")
                 (23577 . "오히려 상") (21930 . "죽을 상, 상사 상") (20663 . "상처 상, 상할 상")
                 (38684 . "서리 상") (29376 . "형상 상") (22039 . "맛볼 상")
                 (35443 . "자세할 상") (31077 . "상서로울 상") (35937 . "코끼리 상")
                 (24202 . "牀의 俗字, 평상 상") (26705 . "뽕나무 상")
                 (20687 . "형상 상") (20767 . "갚을 상") (35059 . "치마 상")
                 (24224 . "학교 상, 우나라태학 상") (31665 . "상자 상")
                 (29245 . "시원할 상") (35316 . "잔 상") (32724 . "빙빙돌아날 상")
                 (23360 . "과부 상") (24258 . "행랑 상") (28248 . "강이름 상, 물이름 상")
                 (29248 . "평상 상") (27233 . "상수리나무 상, 도토리 상")
                 23270 28711 (23776 . "고개 상") 29366 23578 20504
                 (29982 . "맛볼 상") (27556 . "일찍죽을 상") 24851 24678
                 22439 21410 29818 31029 (39001 . "이마 상") 20007
                 (24476 . "노닐 상") (32215 . "담황색 상") 37631 37851
                 25156 (26188 . "정오 상") 35626 40472 17966 40040
                 (22653 . "높고밝은땅 상") 27145 36120 39526 39837
                 15102 32497 40094 20260 20607 22674 23581 27527
                 32515 35294 35814 36175 39073 39590 40127 40564
                 (37791 . "방울소리 상") (39738 . "삶을 상") 19972 20201
                 20664 21971 22160 23953 24929 24943 25633 26706
                 26715 27105 28403 28449 28474 28498 28736 30921
                 32116 32212 32276 33881 34063 34730 34768 34771
                 34864 35116 35152 37871 37900 37980 39483 39895
                 40012 40054 40414 14080 14276 14722 14780 16984
                 17198 17561 17687 17706 17902 18534 19161 19178
                 19363 19714 19824 19836))
       (49352 . ((22622 . "변방 새, 변망 새") (29885 . "도장 새, 옥새 새")
                 (36093 . "굿할 새, 국할 새") 22736 (39955 . "뿔심 새")
                 29626 36187 40131 23869 22139 24866 29909 31802
                 13815 13844 16930 18081 19460 19800))
       (49353 . ((33394 . "빛 색") (22622 . "막힐 색, 변망 색")
                 (32034 . "찾을 색, 동아줄 색") (21959 . "아낄 색, 인색할 색")
                 (34196 . "아낄 색") (31329 . "거둘 색") (24876 . "놀랍고두려울 색")
                 36094 25717 (27117 . "나뭇잎 떨어져 앙상할 색")
                 (28615 . "껄끄러울 색") 30688 32364 37773 21868 31313
                 36188 38127 (28690 . "濇의 俗字") 21990 25038 25804
                 26652 27502 27509 27854 27923 30263 31343 36694
                 37547 40794 40816 14577 14710 14937 15270 15431
                 16266 16569 16994 17026 18365))
       (49373 . ((29983 . "날 생") (30465 . "덜 생, 살필 생")
                 (29298 . "희생 생") (29989 . "생질 생, 사위 생") (31513 . "생황 생")
                 (30490 . "눈에백태낄 생") 27881 26637 40746 (37454 . "녹 생")
                 27525 28262 29636 33532 36025 64033 13762 16210
                 16211 18055 19644 19748))
       (49436 . ((35199 . "서녘 서") (26360 . "쓸 서, 글 서")
                 (24207 . "차례 서") (26257 . "더울 서") (24246 . "여러 서, 뭇 서")
                 (24464 . "천천할 서") (32626 . "관청 서, 마을 서") (24661 . "용서할 서")
                 (32214 . "실마리 서") (35475 . "맹세할 서") (25933 . "차례 서, 베풀 서")
                 (36893 . "갈 서") (29790 . "상서 서") (33298 . "펼 서")
                 (26329 . "새벽 서") (26866 . "살 서, 쉴 서") (32997 . "서로 서")
                 (40653 . "기장 서") (40736 . "쥐 서") (29312 . "무소 서, 뿔난소 서")
                 (34223 . "참마 서, 감자 서") (23996 . "섬 서") (25234 . "풀 서, 끄어낼 서")
                 (23167 . "사위 서") (22779 . "사위 서") (26646 . "깃들일 서, 깃드릴 서")
                 (31598 . "점대 서, 산가지 서") (37604 . "호미 서") (32110 . "솜 서, 헌솜 서")
                 (22661 . "농막 서") (25471 . "깃들일 서, 깃드릴 서")
                 21465 (32019 . "느슨할 서") (22124 . "씹을 서") 25944
                 36990 38314 28294 25749 (37455 . "호미 서") 37265
                 35549 34573 (28241 . "거를 서") (28584 . "물가 서")
                 (33447 . "상수리 서") 14688 28469 28539 24251 31944
                 32210 20582 20422 22293 22463 23638 24711 17106
                 17493 20070 23679 32446 32490 35870 38148 (25567 . "나무이름 서")
                 (32801 . "구실이름 서") 21349 21777 23276 24454 24659
                 26952 27176 27685 27775 27776 28490 28630 29817
                 30297 30802 31280 31789 31902 31980 32259 32851
                 32863 33342 33955 34247 34641 34868 35298 36809
                 37859 39779 40042 40046 40048 40296 40737 13352
                 13566 13863 14025 14143 14360 14516 14589 15168
                 15317 15682 15728 15810 16240 16311 16884 16925
                 16960 17121 17265 17626 18737 18877 18888 18942
                 19564))
       (49437 . ((30707 . "돌 석") (22805 . "저녁 석") (24109 . "자리 석")
                 (26132 . "옛 석") (24796 . "아낄 석") (37323 . "풀 석")
                 (26512 . "가를 석, 쪼갤 석") (37675 . "주석 석") (22893 . "클 석")
                 (30889 . "클 석") (26227 . "밝을 석, 분석할 석") (28511 . "개펄 석")
                 (27728 . "조수 석, 저녁조수 석") (28101 . "쌀일 석")
                 (33990 . "자리 석, 클 석") (33098 . "포 석") (33283 . "신 석")
                 37322 31352 34731 31055 (34597 . "도마뱀 석") (37456 . "놋쇠 석")
                 (26224 . "밝을 석") 35068 28972 30361 35179 37020
                 26718 26852 27297 28859 30551 33284 30805 37320
                 38177 (30717 . "석비레 석") (40747 . "석서 석") 20326
                 20895 21255 21821 23317 24769 25208 27823 28569
                 28959 30966 32198 33765 34158 35119 37368 39830
                 13482 13949 14059 14061 14172 15098 15178 15460
                 16506 16534 16537 16542 16695 17832 18971 19645))
       (49440 . ((20808 . "먼저 선") (20185 . "신선 선") (21892 . "착할 선")
                 (21934 . "고을이름 선") (32218 . "실 선, 기울 선") (33337 . "배 선")
                 (36984 . "가릴 선") (39854 . "고울 선") (27927 . "조촐할 선, 율이름 선")
                 (23459 . "베풀 선") (26059 . "돌 선") (31146 . "봉선 선, 전위할 선")
                 (33203 . "반찬 선") (32341 . "기울 선") (25159 . "사립문 선, 부채 선")
                 (29053 . "부칠 선, 부추킬 선") (29887 . "아름다운옥 선")
                 (29831 . "아름다운옥 선, 구슬 선") (29764 . "도리옥 선, 둥근옥 선")
                 (30316 . "옴 선, 버짐 선") (33146 . "샘 선, 멍울 선")
                 (20120 . "구할 선") (32680 . "부러워할 선, 부끄러워할 선")
                 (37521 . "끌 선, 윤택할금 선") (34796 . "매미 선") (35445 . "많을 선, 말전할 선")
                 (39245 . "膳과 同字, 반찬 선") (23307 . "고울 선") (38704 . "싸락눈 선, 별 선")
                 (20682 . "춤출 선") (34330 . "이끼 선") (36323 . "맨발 선, 발벗을 선")
                 (37925 . "복자 선") (28210 . "바림 선, 물적실 선") (29697 . "옥 선, 구슬 선")
                 (25982 . "글잘쓸 선") 28282 (23584 . "적을 선") 32263
                 29177 40092 32558 22688 (35668 . "가르칠 선") (32171 . "線과 同字")
                 29550 (37831 . "갈이틀 선") 36506 (39480 . "불깔 선")
                 (24835 . "쾌할 선") 20686 34810 (22689 . "백토 선")
                 23319 (31557 . "솔 선") 33063 39402 29103 25550 34798
                 35144 36560 15757 15869 16772 (40059 . "생선 선")
                 20688 28457 33321 37708 40020 15707 31109 32447
                 38115 23157 29405 30307 31868 34259 34633 35804
                 36345 36873 38239 39583 40157 22989 23034 (23257 . "예쁠 선")
                 (23583 . "尠과 同字") (23667 . "날 선") (27482 . "고을 선")
                 (35697 . "善의 古字") (40019 . "두렁허리 선") 20627
                 20722 22599 22910 23288 24303 25639 26294 27272
                 27624 27673 28759 28791 28877 29655 30905 30960
                 31122 31176 31666 32100 32316 32335 32528 33518
                 34073 34561 35132 35590 36219 36340 36462 37167
                 37232 37753 39092 39877 39962 13480 13615 13621
                 13859 14821 14922 14983 15016 15200 15296 15508
                 15596 15722 16060 16268 16325 16482 16483 16608
                 16672 16691 16701 17011 17275 17888 17968 17995
                 18034 18036 18074 18467 18538 18605 18622 18748
                 18791 18821 18968 19527 19559 19586 19686))
       (49444 . ((38634 . "눈 설") (35498 . "말씀 설") (35373 . "베풀 설")
                 (33292 . "혀 설") (27844 . "샐 설, 발설할 설") (34203 . "맑은대쑥 설, 풀이름 설")
                 (21352 . "사람이름 설") (23633 . "가루 설, 구차할 설")
                 (27945 . "샐 설, 물이샐 설") (28203 . "칠 설, 더러울 설")
                 (35131 . "더러울 설, 평복 설") (26964 . "문설주 설, 기둥 설")
                 (40807 . "물 설, 씹을 설") 25352 (20592 . "맑을 설")
                 (25586 . "셀 설") 32079 (29191 . "불사를 설") (32050 . "고삐 설")
                 31271 18013 26555 (23199 . "깔볼 설") 22227 (30879 . "가죽다룰 설")
                 28971 36512 32228 (26284 . "설만할 설") 35081 27184
                 16097 35828 20149 32449 35774 40149 20084 20275
                 20673 21758 25268 25587 27021 30134 31164 31951
                 32108 33188 33373 34062 34533 35500 36684 36773
                 36778 40008 13349 13549 13582 14229 14428 14575
                 14877 15615 15662 15677 16369 18271 18809 18990
                 19537))
       (49452 . ((26297 . "해돋을 섬, 햇살오를 섬") (32406 . "가늘 섬")
                 (34814 . "두꺼비 섬") (38493 . "고을이름 섬") (27570 . "다죽일 섬, 멸할 섬")
                 (38275 . "번쩍할 섬, 피할 섬") (36109 . "넉넉할 섬, 도울 섬")
                 (21089 . "날카로울") (25016 . "간사할 섬") (25723 . "섬섬할 섬")
                 (37531 . "가래 섬") 27569 29203 (35691 . "헛소리 섬")
                 (30546 . "번개빛 섬") 32398 26830 (38897 . "산부추 섬")
                 25502 15672 16632 32420 25530 27516 32330 35893
                 36193 38118 38378 (23365 . "가늘 섬") 31512 23312
                 25761 25877 26225 28854 29068 29125 35187 35234
                 35741 13970 14563 14875 14949 14990 15367 15370
                 16036 16441 16481 16625 16782 17073 17081 17773
                 18494 19437 19449))
       (49453 . ((33865 . "고을이름 섭, 성 섭") (28041 . "물 건널 섭, 지나칠 섭, 돌아다닐 섭")
                 (25885 . "몰아 잡을 섭, 끌 섭, 단정하게 할 섭, 겸할 섭, 거둘 섭, 기록할 섭, 항복할 섭")
                 (29166 . "불에 익힐 섭, 불꽃 섭, 조화로울 섭") (36513 . "밟을 섭, 신 신을 섭, 오를 섭")
                 (25086 . "무서울 섭, 두려울 섭") 27481 29205 25666
                 (22209 . "말하려다 멈출 섭, 중얼거릴 섭, 입 다물 섭")
                 35723 (32886 . "소곤거릴 섭, 낄 섭, 성씨 섭") (38007 . "쪽집게 섭")
                 23655 31819 28169 22801 37745 27399 38818 36510
                 21995 24913 25668 28384 32834 36433 38218 39070
                 (28740 . "흠치르르할 섭, 물이름 섭, 떼 섭") (39027 . "귀밑 뼈 섭")
                 23647 24381 24482 30092 34818 35423 35736 36354
                 36377 37560 38872 13476 14762 15380 15420 15623
                 15881 15886 16202 18195 18593 18734 19392 19397))
       (49457 . ((25104 . "이룰 성, 거듭 성, 마칠 성, 화목할 성")
                 (26143 . "별 성, 희뜩희뜩할 성, 세월 성, 성시 성")
                 (30465 . "살필 성") (24615 . "성품 성, 마음 성, 바탕 성, 색욕 성")
                 (22995 . "일가 성, 아이 낳을 성, 백성 성") (22478 . "보루 성, 재 성, 서울 성")
                 (35488 . "미쁠 성, 공경할 성, 살필 성, 진실 성, 정성 성")
                 (32854 . "성인 성, 착할 성, 통할 성, 지극할 성")
                 (32882 . "소리 성, 풍류 성, 명예 성") (30427 . "담을 성, 이룰 성, 정제할 성, 많을 성, 성할 성")
                 (26207 . "밝을 성, 햇살 퍼질 성") (37266 . "술 깰 성, 꿈 깰 성, 깨달을 성")
                 (23468 . "사고 성, 도서실 성, 창고 성, 사기창고 성")
                 (24826 . "깨달을 성, 똑똑할 성, 조용할 성") (33125 . "비린내 성, 더러울 성, 날고기 성, 냄새 성")
                 (29689 . "옥이름") (29481 . "성성이 성") (31596 . "바디 성, 대나무이름 성")
                 28219 22454 (39426 . "붉은 소 성, 누른말 성, 활 물쩡물쩡할 성")
                 23800 29388 26208 29003 37077 39219 37614 20567
                 31669 32126 33663 35587 22768 32988 35802 38102
                 39565 29766 22701 23053 26320 27576 30376 30578
                 31386 31682 35306 35314 37727 39823 39886 39929
                 13914 14189 14872 15248 15843 16169 16179 16238
                 16298 16478 16591 16840 17868 18070 19142))
       (49464 . ((19990 . "세상 세, 평생 세, 백년 세, 인간 세")
                 (35498 . "달랠 세, 쉴 세, 말씀 세") (27506 . "해 세, 곡식 익을 세, 돐 세, 풍년 세")
                 (32048 . "가늘 세, 작을 세, 좀놈 세") (27927 . "씻을 세, 세수 그릇 세")
                 (21218 . "권세 세, 위엄 세, 불알 세, 기회 세") (31237 . "구실 세, 거둘 세, 놓을 세, 쉴 세, 세금 세")
                 (36016 . "빌릴 세, 세 바칠 세, 세낼 세") (31545 . "가는 대 세")
                 (27922 . "씻을 세, 설치할 세") 31246 23895 27507
                 (32336 . "가늘고 성긴 베 세") (34555 . "매미 허물 세")
                 (24104 . "차는 수건 세, 손수건 세") 25385 35038 27269
                 28055 19835 23681 32454 36147 19991 20119 21323
                 26385 28058 31089 34087 36329 36558 36682 37556
                 40795 40805 13397 13599 13626 14519 15441 15927
                 19261))
       (49548 . ((23567 . "작을 소, 좁을 소, 적을 소, 어릴 소, 적게 여길 소")
                 (23569 . "적을 소, 조금 소, 멸시할 소, 잠깐 소, 젊을 소")
                 (25152 . "바 소, 곳 소, 쯤 소, 가질 소, 얼마 소")
                 (31505 . "웃음 소") (28040 . "다할 소, 꺼질 소, 풀릴 소, 해어질 소, 사라질 소")
                 (32032 . "흴 소, 생초 소, 빌 소, 바탕 소, 원래 소")
                 (21484 . "부를 소, 청할 소, 과부 소") (26157 . "소명할 소, 소목 소, 태평세월 소, 밝을 소")
                 (35380 . "하소연할 소, 송사할 소") (25475 . "쓸 소, 상투 소")
                 (34311 . "차조기 소, 나무할 소, 까무러칠 소, 깨어날 소, 성씨 소")
                 (29138 . "불 붙을 소, 들불 소, 불 놓을 소, 불사를 소")
                 (39479 . "근심스러울 소, 흔들릴 소, 급할 소, 처량할 소, 소동할 소")
                 (30094 . "성길 소, 드물 소") (34092 . "나물 소")
                 (24034 . "새집 소, 집 지을 소, 새 보금자리 소, 적진 망보는 높을 수레 소, 도둑굴 소, 큰 피리 소, 채소이름 소")
                 (32057 . "이을 소, 얽을 소, 소개할 소") (37045 . "땅이름 소, 성씨 소")
                 (27836 . "못 소, 늪 소") (34157 . "쑥 소, 쓸쓸할 소, 말 우는 소리 소, 바람소리 소")
                 (23477 . "밤 소, 작을 소, 벌레이름 소") (36877 . "노닐 소, 거닐 소")
                 (31787 . "퉁소 소, 활고지 소") 26786 (25620 . "긁을 소, 휘저을 소, 분분할 소")
                 (36961 . "거스를 소, 맞이할 소, 하소연할 소") (26803 . "얼레빗 소, 머리빗을 소")
                 (22609 . "허수아비 소") (29990 . "깨어날 소, 숨 쉴 소")
                 (30233 . "옴 소, 종기 소") (30095 . "뚫릴 소, 나눌 소, 멀 소, 추할 소, 드물 소, 성길 소")
                 (22063 . "휘파람 소, 세게 뿜을 소, 읊을 소") (37559 . "녹을 소, 사라질 소, 녹일 소")
                 (38902 . "아름다울 소, 이을 소, 순의 풍류 소, 봄 소")
                 (28703 . "물 맑을 소, 비바람 몰아칠 소, 강이름 소, 비바람치는 모양 소")
                 (28836 . "밝을 소") (28335 . "거슬러 올라갈 소") 37335
                 (31712 . "가는 대 소, 조릿대 소") 18429 15037 (36360 . "疎의 譌字")
                 (33368 . "배 소") (38660 . "하늘 소, 진눈깨비 소") 21682
                 (27869 . "거슬러 올라갈 소, 흘러갈 소") 31308 28935
                 (22525 . "쓸 소, 그릴 소, 언덕 소") (32731 . "날개치는 소리 소, 모지라질 소")
                 (24876 . "고할 소, 참소할 소, 알릴 소") 38808 28988
                 38144 29117 31602 (37221 . "타락죽 소") (32325 . "고치 실 뽑을 소")
                 (21362 . "높을 소, 성씨 소") 35599 34856 21165 (34552 . "거미알 소, 연가시 알집 소")
                 20611 15633 (20431 . "거문고 뒤치는 소리 소") 29631
                 (25422 . "덜 소, 흔들거릴 소, 추릴 소, 버릴 소")
                 (39752 . "산도깨비 소") 30175 30658 (20299 . "소개할 소, 사신 소, 댓수의 차례 소")
                 31678 21951 29829 39442 13787 (21961 . "멀떠구니 소")
                 24875 24901 26067 27385 27479 28344 29713 33487
                 (31601 . "가는 대 소") 20851 21880 24035 25195 28487
                 28903 31267 31659 32461 32555 33831 34767 35785
                 39578 40121 40139 (22608 . "塑와 同字") (27156 . "풀막 소, 너스레 소")
                 (33158 . "살찔 소, 멀떠구니 소") (39865 . "문어 소, 낙지 소")
                 (39994 . "비린내 소의 譌字") 22056 22220 23051 23242
                 23573 24054 24368 24391 25531 26582 27041 27375
                 27612 27657 28318 28421 28530 29258 31420 31524
                 31615 31768 32128 32164 32292 33348 33702 33847
                 34726 34800 34961 36355 36622 37147 39093 39102
                 39678 39874 39925 39968 40034 40813 13439 13573
                 13654 13900 14493 14704 14823 14878 15010 15097
                 15284 15496 15510 15527 15541 15621 15790 15899
                 15924 16331 16899 16944 17061 17396 17529 17557
                 17562 17696 17701 17707 17733 17944 18044 18174
                 18233 18282 18989 19248 19590 19739))
       (49549 . ((36895 . "빠를 속, 부를 속, 더러울 속, 사슴 발자취 속")
                 (20439 . "익을 속, 버릇 속, 풍속 속, 세상 속") (32396 . "이을 속")
                 (26463 . "묶을 속, 단나무 속, 약속할 속, 비단 다섯끝 속")
                 (23660 . "무리 속, 좇을 속, 붙이 속, 거느릴 속, 붙일 속")
                 (31903 . "좁쌀 속, 겉곡식 속, 조 속") (36118 . "속바칠 속, 무역할 속")
                 (28049 . "빨") (35606 . "일어날") 23646 34060 39191
                 35307 27948 37577 (36972 . "공손할 속, 줄어들 속")
                 26752 35470 31756 32154 32493 35873 36174 20723
                 24991 27157 27536 28517 34263 34266 36250 13403
                 13572 15128 15304 16711 17178 17678 17705 18326
                 18772 19591 19760))
       (49552 . ((23403 . "손자 손, 움 돋을 손, 순할 손, 피할 손")
                 (25613 . "덜 손, 잃을 손, 피곤할 손, 삼갈 손, 상할 손, 괘 이름 손")
                 (36956 . "순할 손, 겸손할 손, 도망할 손, 사양할 손")
                 (24061 . "사양할 손, 낮은체 할 손, 부드러울 손, 괘 이름 손, 동과 남의 사이방향 손")
                 (39137 . "飱과 同字, 삼킬 손") (33984 . "난초") (39143 . "저녁밥 손, 물에 만 밥 손")
                 (39153 . "飧의 俗字") 22080 21936 23385 25439 29426
                 33642 36874 24058 24891 25614 27074 28512 29499
                 34165 34206 39016 14514 16474 17443 17750))
       (49556 . ((29575 . "거느릴 솔, 좇을 솔, 쓸 솔, 대강 솔, 공경할 솔, 뱀 이름 솔, 헤아릴 솔")
                 21339 (34752 . "귀뚜라미 솔") (31395 . "느릿느릿 걸을 솔, 우지직거릴 솔, 구멍으로 나올 솔")
                 20090 25684 29993 30911 34614 15518 18598))
       (49567 . (13694))
       (49569 . ((36865 . "보낼 송, 가질 송, 활쏘기 능통할 송, 줄 송")
                 (26494 . "소나무 송, 향풀 송, 땅이름 송, 솔 송")
                 (35359 . "송사할 송, 시비할 송, 꾸짖을 송, 괘 이름 송")
                 (35494 . "풍유할 송, 읽을 송, 말할 송, 원망할 송, 욀 송")
                 (38924 . "기릴 송, 칭송할 송, 찬양하는 노래 송")
                 (23435 . "송나라 송, 성씨 송") (24730 . "두려울 송, 송구할 송, 두려워할 송")
                 (28126 . "강이름 송") 33635 (31462 . "공경할 송, 두려울 송, 높일 송, 솟을 송")
                 (26583 . "松과 同字") (39686 . "터럭 더부룩할 송")
                 14963 14238 35772 35829 38198 39042 33752 20527
                 20935 24250 24440 24879 25021 25426 26496 26537
                 26789 26980 27303 34585 37750 37817 39224 39415
                 14800 14886 15133 15268 15288 15952 18150 18592
                 19443))
       (49576 . ((24802 . "의심낼 솨") 17437))
       (49584 . (21808))
       (49604 . ((27578 . "내릴 쇄, 감할 쇄, 빠를 쇄, 죽일 쇄")
                 (21047 . "긁을 쇄, 고칠 쇄, 빗을 쇄, 문지를 쇄, 인쇄할 쇄")
                 (37782 . "자물쇠 쇄, 쇠사슬 쇄, 가둘 쇄, 항쇄 쇄, 자물쇄 쇄")
                 (28753 . "뿌릴 쇄, 깜짝 놀랄 쇄") (30862 . "부술 쇄, 잘 쇄")
                 27922 (29795 . "옥가루 쇄, 가늘 쇄, 좀스러울 쇄, 대궐문 아로새길 쇄")
                 26194 37819 26348 37825 38145 32304 37801 21993
                 22499 15136 21794 29712 38121 24802 21998 28078
                 28305 30741 32320 13662 16919 18310 18644 19776))
       (49632 . ((34928 . "쇠할 쇠, 모손할 쇠, 약할 쇠")
                 (37335 . "쇠") 38026 21752 (22794 . "천천히걷다") 38806
                 15524 19086))
       (49688 . ((27700 . "물 수, 고를 수") (25163 . "손 수, 잡을 수, 칠 수")
                 (25976 . "헤아릴 수, 수죄할 수, 몇 수, 이치 수, 자주 수")
                 (23432 . "지킬 수, 보살필 수, 원님 수, 기다릴 수")
                 (39318 . "머리 수, 먼저 수, 비롯할 수, 임금 수, 향할 수, 자백할 수")
                 (27193 . "나무 수, 막을 수, 세울 수, 심을 수") (21463 . "이을 수, 얻을 수, 담을 수, 받을 수")
                 (25480 . "줄 수, 부칠 수") (25910 . "모을 수, 거둘 수, 잡을 수, 수레바퀴 수")
                 (38614 . "비록 수, 벌레이름 수") (20462 . "닦을 수, 옳게할 수, 정리할 수, 꾸밀 수")
                 (22781 . "목숨 수, 장수할 수, 나이 수, 축복할 수")
                 (38920 . "수염 수, 기다릴 수, 모름지기 수") (35504 . "누구 수, 무엇 수")
                 (31168 . "빼어날 수, 이삭 수, 벼 패일 수, 아름다울 수")
                 (24833 . "근심 수, 염려할 수, 탄식할 수") (36930 . "사무칠 수, 나아갈 수, 이룰 수, 마침내 수")
                 (38568 . "따를 수, 괘 이름 수, 나라이름 수, 맡길 수")
                 (22234 . "가둘 수, 갇힐 수, 묶일 수") (27530 . "죽을 수, 베일 수, 끊어질 수, 지나갈 수, 다를 수")
                 (36664 . "떨어뜨릴 수, 쏟을 수, 보낼 수, 짐바리 수")
                 (38656 . "음식 수, 머뭇거릴 수, 기다릴 수, 찾을 수, 괘 이름 수")
                 (29560 . "짐승 수") (22402 . "드리울 수, 변방 수, 거의 수, 남길 수")
                 (24101 . "주장할 수, 장수 수") (27929 . "물가 수, 물읾 수")
                 (30561 . "졸음 수, 잠 수") (25628 . "찾을 수, 더듬을 수, 화살 빨리 가는 소리 수, 어지러울 수")
                 (38539 . "수나라 수") (39635 . "뼛속 기름 수, 마음속 수, 뼈골 수")
                 (37526 . "저울눈 수") (33936 . "꼭두서니 수, 모을 수, 숨을 수, 찾을 수")
                 (37228 . "술 권할 수, 갚을 수") (32670 . "부끄러울 수, 음식 수, 부끄러워할 수")
                 (32353 . "수놓을 수, 성씨 수") (31929 . "정할 수, 순수할 수")
                 (34966 . "소매 수") (30246 . "파리할 수, 가늘 수") (35728 . "讎와 同字, 원수 수")
                 (31466 . "세울 수, 서있을 수, 더벅머리 수") 22014
                 (29417 . "겨울사냥 수, 순행할 수") (23234 . "형수 수")
                 (31319 . "이삭 수, 벼이삭 수") (33065 . "포 수, 길 수, 마를 수, 닦을 수")
                 (25101 . "수자리 수, 막을 수, 집 수") (32143 . "편안할 수, 수레고삐 수, 물러갈 수, 여우모양 수")
                 (34282 . "큰 늪 수, 덤불 수, 똬리 수") (32172 . "인끈 수")
                 (36995 . "깊숙할 수") (39706 . "턱수염 수, 아랫수염 수")
                 (23723 . "바위구멍 수, 바위굴 수") (29159 . "봉화 수, 불 수, 나무 문질러 불낼 수")
                 (38567 . "무덤길 수, 구멍 수, 길 수") 28049 (22013 . "기침할 수, 입가실 수")
                 (28465 . "양치질할 수, 빨래할 수") (29703 . "귀막이 수, 옥돌 수, 귀고리 수")
                 (23744 . "岫와 同字, 바위굴 수") (33585 . "수유 수")
                 (29874 . "서옥 수, 패옥 수") (34010 . "기쁠") (37561 . "동록 수, 녹슬 수")
                 38543 21454 25968 (21471 . "어른 수, 쌀 씻는 소리 수")
                 (35726 . "짝 수, 원수 수, 대거리할 수, 비교할 수, 갚을 수")
                 (21806 . "팔 수, 갚을 수") (35918 . "세울 수, 내시 수, 더벅머리 수")
                 26220 (30559 . "똑바로 볼 수, 윤택할 수") (38514 . "변방 수, 위태할 수")
                 25436 30562 (31071 . "빌미 수") 34222 34429 33696
                 (30605 . "소경 수, 장님 수, 판수 수") 39620 (39100 . "바람소리 수")
                 32137 (27571 . "치다") 35162 37221 28705 20501 28395
                 18584 (24267 . "숨길 수") (28338 . "오줌 수, 반족할 수, 적실 수")
                 35446 32379 37802 18239 35510 28484 31268 31730
                 37223 17346 (33156 . "파리할 수") 24805 25851 27262
                 27362 31298 33753 35385 37307 (33082 . "얼굴 윤택할 수, 머리덜미 수")
                 25164 27286 33092 13496 16819 27845 22784 26662
                 28603 30185 31149 38612 39767 16853 18242 28617
                 (31828 . "조리 수, 곡식 세는 단위 휘 수, 또아리 수")
                 (33208 . "髓와 同字") 23299 23334 23986 26488 (27701 . "삼수변")
                 28374 28717 33085 37885 13423 14686 17465 18196
                 18745 23551 31446 32483 38152 39035 20861 24069
                 25822 26641 29539 31883 32485 32502 32525 35841
                 35847 36755 38114 38204 38624 39125 39306 39312
                 17217 (39240 . "드릴 수, 반찬 수") 20450 20609 20929
                 21404 21468 21974 22202 22323 22440 22464 23095
                 23213 23320 23999 24129 24256 24830 25521 25673
                 25687 26078 27143 27446 (27706 . "아래물") 27731
                 28029 28077 29035 29504 29843 29901 30262 30440
                 31327 31956 32082 32135 32333 32376 32407 33127
                 33273 33359 33516 33662 33987 34024 34150 34699
                 35019 35682 36069 36567 36928 37131 37273 37760
                 37829 37833 37861 37894 37929 38294 38478 39231
                 39466 39810 39997 13396 13464 13727 13740 13852
                 14032 14050 14062 14072 14096 14154 14205 14316
                 14481 14538 14661 14696 15425 15505 15507 15513
                 15523 15642 15710 15718 15874 15905 15947 16095
                 16125 16161 16247 16662 16737 16747 16851 16941
                 16953 16972 16996 17142 17227 17456 17560 17721
                 17722 17745 17871 17883 18165 18192 18220 18256
                 18452 18492 18541 18549 18603 18695 18777 18790
                 18892 19059 19218 19273 19304 19309 19359 19421
                 19461 19473 19576 19680))
       (49689 . ((23487 . "잘 숙, 지킬 숙, 머물 숙") (21460 . "아재비 숙, 주울 숙, 어릴 숙, 끝 숙")
                 (28113 . "맑을 숙, 착할 숙, 사모할 숙") (23408 . "누구 숙, 어느 숙, 살필 숙, 익을 숙")
                 (29087 . "익힐 숙, 삶을 숙, 풍년들 숙, 익을 숙")
                 (32901 . "공손할 숙, 공경할 숙, 경계할 숙, 엄숙할 숙")
                 (22809 . "이를 숙, 이미 숙, 빠를 숙, 일찍 일어날 숙")
                 (33789 . "콩") (22654 . "사랑방 숙, 기숙사 숙, 서당 숙")
                 (29881 . "옥그릇") (28506 . "빠를") (29729 . "옥이름")
                 17304 29851 (20495 . "잠간 숙, 홀연 숙, 개의 빨리 달리는 모양 숙")
                 (20470 . "비롯할 숙, 처음 숙, 지을 숙") 22513 (27226 . "길고 꼿꼿할 숙")
                 34047 20496 28930 40016 (20789 . "잿빛 숙, 남해 임금 숙, 독화당할 숙, 갑자기 숙")
                 (40427 . "숙상새 숙") 29578 (39500 . "좋은말 숙, 숙상말 숙")
                 25491 15766 16612 19001 31899 32899 39573 40532
                 (23116 . "궁녀 벼슬이름 숙") 20788 23575 25677 29663
                 33700 34410 35540 36350 39835 13385 13392 13528
                 14106 14171 14923 15017 15588 15627 15676 16606
                 17355 17535 19545))
       (49692 . ((38918 . "좇을 순, 부드러울 순, 차례 순, 성씨 순, 순할 순")
                 (32020 . "실 순, 순수할 순, 온전할 순, 부드러울 순, 순전할 순")
                 (33059 . "입술 순") (24033 . "돌 순, 순행할 순, 두루 순")
                 (24490 . "좇을 순, 의지할 순, 돌 순, 차례 순, 돌아다닐 순")
                 (26092 . "열흘 순, 두루할 순, 고를 순, 가득할 순, 요사스런 별 순")
                 (27529 . "죽은 사람을 따라 죽을 순, 구할 순, 좇을 순")
                 (30636 . "눈 깜작일 순, 잠깐 순, 눈을 깜짝이는 순")
                 (33308 . "순임금 순, 무궁화 순") (28147 . "맑을 순, 순박할 순")
                 (29667 . "옥그릇 순, 옥 이름 순") (30462 . "방패 순, 벼슬이름 순")
                 (27957 . "웅덩이물 순, 소리없이 눈물흘릴 순, 믿을 순")
                 (31565 . "죽순 순, 대싹 순") (37255 . "전국술 순, 두터울 순, 삼갈 순, 순수할 순")
                 (39348 . "길들일 순, 착할 순, 길드릴 순") (33600 . "풀 이름 순, 성씨 순, 순풀 순")
                 21767 (35426 . "꾀할 순, 물을 순, 믿을 순") (35524 . "거듭 일러줄 순, 지극할 순, 정성스러울 순")
                 (24455 . "조리 돌릴 순, 따라 죽을 순, 빠를 순, 부릴 순")
                 (24642 . "진실할 순, 믿을 순, 온공스러울 순") (34036 . "순나물 순, 순마룰 순")
                 (26991 . "난간 순, 방패 순, 뺄 순, 상여 순") (37662 . "사발종 순")
                 28958 (34147 . "무궁화 순") (26642 . "경쇠걸이 순, 순나무 순")
                 (27219 . "무궁화나무 순") 31499 (32939 . "광대뼈 순, 정성스러울 순")
                 (40329 . "메추라기 순, 옷 해질 순") 23755 24309 36660
                 (32003 . "신에 실선 두를 순, 실 띠 순, 법 순, 좇을 순")
                 28440 14695 (29413 . "徇의 俗字") (39690 . "머리 풀어헤칠 순")
                 (30516 . "눈짓할 순") (20378 . "좇을 순, 빠를 순, 조리 돌릴 순")
                 14735 17144 18766 33724 32431 35810 35846 39034
                 39535 40529 40544 (30457 . "눈 감을 순, 졸 순") 39400
                 20264 20593 21680 22878 23024 24972 25559 26442
                 27051 27078 27621 28017 28283 28387 28483 29321
                 30019 30618 30628 30903 30957 31664 31784 33435
                 33939 34912 36048 37063 37269 38489 39897 13456
                 13628 13981 14148 14151 14404 14874 15084 15104
                 15404 15692 15726 15799 15914 16230 16418 16437
                 17371 17415 17502 17616 17690 17842 17961 17993
                 18214 18664 19716))
       (49696 . ((25100 . "개 술, 열한째 지지 술, 때려 부술 술")
                 (36848 . "이을 술, 좇을 술, 조회할 술, 밝힐 술, 지을 술")
                 (34899 . "재주 술, 심술부릴 술, 술법 술") (37477 . "돗바늘 술, 인도할 술, 큰바늘 술")
                 21369 37514 16055 28495 33623 29676 27821 32073
                 27440 33921 34965 37680 40336 13701 14218 16257
                 18590 18596 19333 19699))
       (49709 . ((23815 . "높을 숭, 공경할 숭, 마칠 숭, 산이름 숭")
                 (23913 . "높을 숭, 산이름 숭") (23847 . "산 불끈 솟을 숭, 산 웅장할 숭, 산이름 숭")
                 28373 (33752 . "배추 숭") 30841 13565 14307 14557
                 19447))
       (49772 . ((20485 . "버금 쉬, 아직 벼슬자리에 오르지 않은 아들 쉬, 원님 쉬")
                 (26220 . "돐 쉬, 첫 생일 쉬") (28140 . "칼 담글 쉬, 물들일 쉬, 범할 쉬")
                 (28960 . "물들일 쉬, 칼날 물에 담글 쉬, 지질 쉬, 태울 쉬")
                 20252 29719 15555 17977))
       (49836 . ((29791 . "비파 슬, 거문고 슬, 깨끗한체할 슬, 바람소리 슬")
                 (33181 . "무릎 슬, 종지뼈 슬, 치슬말 슬") (34664 . "이 슬")
                 (34417 . "蝨과 同字") 16109 21376 40122 39115 28623
                 29873 39924 16041 16078))
       (49845 . ((32722 . "익힐 습, 거듭 습") (25342 . "주울 습, 거둘 습, 팔찌 습, 벼슬이름 습")
                 (35186 . "옷 덧입을 습, 벼슬 대물릴 습, 반복할 습, 합할 습, 염습할 습, 엄습할 습")
                 (28629 . "젖을 습, 소 귀 벌룩거릴 습, 근심 습") (35126 . "슬갑 습")
                 28287 (38576 . "진펄 습") 24948 (29088 . "환할 습, 선명할 습, 반딧불 반짝거릴 습")
                 38699 35637 20064 34989 40155 21827 23949 27106
                 28348 28445 39105 39485 39996 13450 13801 14340
                 14779 16327 17353 17537 17873))
       (49849 . ((21213 . "이길 승, 경치가 좋을 승, 가질 승")
                 (25215 . "받들 승, 이을 승, 도울 승, 차례 승") (20056 . "오를 승, 곱할 승, 셈하는 단위 승, 같은 물건 네 개 승, 탈 승")
                 (20711 . "중 승") (26119 . "해 돋을 승, 풍년들 승, 오를 승")
                 (21319 . "되 승, 오를 승, 피륙의 짜인 날을 세는 단위 새 승, 괘 이름 승")
                 (32361 . "노끈 승, 법 승, 다스릴 승, 이을 승") (19998 . "이을 승, 도울 승, 벼슬의 이름 승, 향상할 승")
                 (38494 . "오를 승, 올릴 승") (34821 . "파리 승") 20055
                 34631 22605 17138 (39705 . "터럭 더부룩할 승") 40038
                 26889 27702 15457 23882 25012 32260 32499 20924
                 21583 22552 25229 26008 26529 26717 27066 27251
                 27910 28311 28986 30075 31444 33024 34114 34687
                 35677 38441 38521 39468 40319 13372 14268 15281
                 15629 17551 19370 19526))
       (49884 . (21313 (26178 . "때 시, 끼니 시, 이 시, 엿볼 시, 가끔 시")
		       (22987 . "비로소 시, 처음 시, 시작할 시, 별이름 시")
		       (31034 . "보일 시, 바칠 시, 가르칠 시") (24066 . "저자 시, 흥정할 시, 집 많은 동리 시")
		       (23546 . "내관 시") (35222 . "볼 시, 견줄 시, 본받을 시, 대접 시")
		       (35433 . "귀글 시, 풍류가락 시, 받들 시, 글귀 시")
		       (35430 . "시험할 시, 더듬을 시, 비교할 시") (26159 . "이 시, 바를 시, 곧을 시")
		       (26045 . "베풀 시, 쓸 시, 더할 시, 안팎 곱사등이 시, 버릴 시, 줄 시, 은혜 시, 공로 시")
		       (20365 . "모실 시, 가까울 시, 좇을 시") (30690 . "살 시, 소리살 시, 곧을 시, 베풀 시, 똥 시, 화살 시")
		       (26612 . "섶 시, 불 땔 나무 시, 시제지낼 시") (23629 . "주검 시, 송장 시, 시체 시")
		       (35553 . "행장 시, 시호 시") (24337 . "윗사람 죽일 시")
		       (21273 . "숟가락 시, 열쇠 시") (29468 . "의심낼 시, 시기할 시, 두려워할 시, 사나울 시")
		       (35962 . "승냥이 시, 늑대 시") (23204 . "시집 시") (26623 . "감나무")
		       (23608 . "시동 시, 주검 시, 시체 시") (24643 . "믿을 시, 의지할 시, 어머니 시")
		       (33997 . "뺑대쑥 시, 시초 시, 뺑때쑥 시") 35610 (35925 . "돼지 시, 돝 시")
		       (32709 . "날개 시, 깃 시, 뿐 시") (23630 . "똥 시, 변 시")
		       (22070 . "말 울 시, 목 쉴 시, 울부짖을 시, 말이우는 시")
		       (33940 . "소회향 시, 모종낼 시") (26160 . "是와 同字")
		       (26558 . "柹의 本字, 감 시") (21883 . "뿐 시, 쯤 시")
		       (21422 . "부릴 시, 마부 시, 나눌 시") (32230 . "보름새베 시, 시마복 시, 석달복 시")
		       (28556 . "물 잦을 시, 다할 시") (24285 . "부릴 시, 천할 시, 마부 시")
		       (26617 . "감 시") 32065 (20594 . "간절히 책망할 시, 살피고 힘쓸 시, 굳셀 시")
		       (31124 . "복 시, 편안할 시") 25749 33134 39096 (35551 . "이 시, 살필 시, 다스릴 시")
		       (20821 . "외뿔난 소 시") 24338 (35913 . "메주 시, 콩자반 시")
		       17188 (22610 . "홰 시") (38987 . "뺨 시, 볼 시") 30478
		       40178 (26546 . "수삼 시, 도꼬마리 시") (35568 . "눈 휘둥글할 시, 곧은말 시")
		       20952 32744 19629 20072 27640 28454 35193 (37315 . "술 거를 시")
		       25548 29362 35127 16544 22488 26102 32524 33715
		       35270 35797 35799 37246 38088 39124 40101 40484
		       32740 (37705 . "숟가락 시, 열쇠 시") 20824 21598
		       21686 23465 23868 23925 24485 25120 25555 26105
		       27055 27618 27830 28260 28321 30466 31035 31073
		       31542 31771 32627 32708 33194 33864 33913 37055
		       37424 37443 39971 40461 40749 13451 13486 13502
		       13613 13751 14008 14228 14536 15085 15259 15567
		       15965 16007 16098 16917 17043 17203 17377 17576
		       17649 17872 18046 18120 18229 18235 18391 18547
		       18564 18848 19011 19625))
       (49885 . ((39135 . "밥 식, 먹을 식, 씹을 식, 제 식, 헛말할 식")
                 (24335 . "법 식, 쓸 식, 제도 식, 수레 앞 가로막대 식")
                 (26893 . "심을 식, 세울 식, 초목 식") (35672 . "알 식")
                 (24687 . "숨쉴 식, 그칠 식, 자식 식, 기를 식") (39166 . "꾸밀 식, 문채날 식, 분 바를 식")
                 (28252 . "물 맑을 식, 엄숙할 식") (27542 . "날 식, 심을 식, 성할 식, 불릴 식")
                 (36606 . "수레앞 가로막이 나무 식, 수레 안에서 절할 식")
                 (34645 . "벌레먹을 식, 일식 식, 월식 식, 좀먹을 식")
                 (25325 . "닦을 식, 다듬을 식, 씻을 식") (29060 . "불 꺼질 식, 없어질 식, 마칠 식")
                 (23508 . "이 식, 참 식, 뿐 식") (22516 . "찰진흙 식, 흙 이길 식, 더듬어 걸을 식")
                 (31698 . "대밥통") (26683 . "점판 식") 37501 28882 21936
                 (23219 . "며느리 식") 34432 36732 39267 39280 29046
                 30236 31107 31810 33952 36936 37134 37812 39136
                 39197 14656 15273 15655 15699 18733 19282))
       (49888 . ((36523 . "몸 신, 아이 밸 신, 몸소 신") (20449 . "믿을 신, 밝힐 신, 맏길 신, 이틀 밤을 잘 신, 사신 신")
                 (33251 . "신하 신, 두려울 신") (26032 . "새 신, 처음 신, 고울 신")
                 (31070 . "천신 신, 영검할 신, 신명 신, 정신 신, 귀신 신")
                 (30003 . "펼 신, 기지개켤 신, 거듭 신, 아홉째 지지 신, 원숭이 신")
                 (36784 . "날 신") (36763 . "매울 신, 고생 신, 여덟째 천간 신")
                 (24892 . "삼갈 신, 정성스러울 신, 고요할 신") (20280 . "펼 신, 다스릴 신, 기지개 신")
                 (26216 . "샛별 신, 아침을 아뢸 신, 새벽 신") (32051 . "큰 띠 신, 벼슬아치 신")
                 (35338 . "물을 신, 다스릴 신, 죄 조사할 신, 알릴 신, 꾸짖을 신")
                 (36805 . "빠를 신, 억셀 신") (33102 . "콩팥 신, 불알 신, 자지 신")
                 (29180 . "깜부기불 신, 불똥 신, 재난의 뒤 신") (23480 . "집 신, 처마 신, 대궐 신")
                 (34218 . "섶 신, 땔나무 신, 월급 신, 성씨 신") (21627 . "읊조릴 신, 끙끙거릴 신")
                 (34563 . "큰 조개 신, 이무기 신, 술잔 신") (23072 . "아이밸 신, 마부 신, 임신할 신")
                 (34254 . "갈골 신, 나아갈 신, 남을 신") (33688 . "약 이름 신, 많은 모양 신, 나라이름 신")
                 (20353 . "떼지어 갈 신, 여럿이 행함 신") 27733 (30695 . "하물며 신, 잇속 드러날 신")
                 (36112 . "노자 신, 노수 신") (21698 . "빙그레 웃을 신, 비웃을 신")
                 24910 (27739 . "물뿌릴 신") (38947 . "눈들고 볼 신")
                 (33060 . "사직제 지낸 고기 신") 25275 36078 20831
                 20265 26587 (39402 . "말 우물거릴 신, 빨리 걸을 신")
                 29878 16277 23799 37600 14122 28908 32453 32958
                 33641 35759 35782 36166 (22138 . "이야기 신") (22239 . "숨구멍 신, 정수리 신")
                 (23034 . "나라이름 신") 21314 21554 22973 23390 23678
                 24350 25183 25938 26138 26335 27018 27680 28901
                 29130 29637 29985 30482 30488 30692 30775 31091
                 31497 31862 31864 32962 33566 33664 35025 35353
                 35360 35371 35383 37486 38432 38998 39891 39984
                 40290 40400 40590 13360 13399 13992 14041 14043
                 14521 15091 15201 15406 15599 15782 15795 16760
                 16803 17025 17558 18097 18140 18152 18290 18565
                 18619 18629 19135 19264 19394 19488))
       (49892 . ((23460 . "집 실, 아내 실, 방 실") (22833 . "잃을 실, 그르칠 실")
                 (23526 . "열매 실") (24713 . "모두 실, 알 실") (34763 . "귀뚜라미 실")
                 31416 23454 23455 39115 24632 27009 34693 37869
                 39960 13805 14576 16025 17053 19873))
       (49900 . ((24515 . "마음 심") (29978 . "심할 심, 무엇 심")
                 (28145 . "깊을 심") (27784 . "성 심, 가라앉을 심") (23563 . "찾을 심")
                 (23529 . "살필 심") (28683 . "즙낼") (27777 . "스며들 심, 물이름 심")
                 (35574 . "참 심, 믿을 심") (33455 . "등심초 심") 28144
                 (27785 . "沈과 同字") (28527 . "물가 심") 24854 35543
                 26490 26787 20232 34120 15665 26937 (29142 . "삶을 심")
                 35153 14199 (33882 . "뽕나무 열매 심") 23479 26516
                 39733 28993 30635 35262 16099 21539 23158 23457
                 23547 27988 28182 35842 35852 38241 40095 13706
                 (37908 . "칼날 아래 코등이 심") (40015 . "칼철갑상어 심")
                 20410 21538 22106 22548 23352 (24516 . "심방변")
                 25011 25907 26730 27187 27244 29845 30222 34803
                 35717 35849 37029 37161 37386 39787 40024 13604
                 13982 13999 14084 14118 14586 14600 15362 15762
                 16248 16967 18067 18689))
       (49901 . ((21313 . "열 십, 전부 십") (25342 . "十의 갖은자, 주울 십")
                 (20160 . "열 사람 십") (36795 . "네거리 십") 20825 21324
                 26910 29927 31437 31861 36231 18873))
       (49920 . (13336))
       (49933 . ((38617 . "짝 쌍") 21452 18951 33389 23367
                 27398 64014 13664 17014 18244))
       (50472 . ((27663 . "성 씨")))
       (50491 . (13842))
       (50500 . ((20818 . "아이 아") (25105 . "자기 아, 나 아")
                 (38597 . "아담할 아") (20126 . "버금 아") (39187 . "주릴 아, 굶을 아")
                 (33469 . "싹 아") (29273 . "어금니 아, 대장 기 아")
                 (38463 . "언덕 아, 아름다울 아, 알랑거릴 아") (34905 . "마을 아")
                 (20420 . "갑자기 아") (35357 . "맞아들일 아, 의심할 아")
                 (21854 . "벙어리 아, 놀랄 아") (40201 . "갈가마귀 아, 검을 아")
                 (23784 . "산 높을 아") (23077 . "예쁠 아") (40285 . "거위 아")
                 (33706 . "지칭개 아, 새발쑥 아") (34558 . "누에나방 아, 누에나비 아")
                 30206 (36819 . "마중할 아") (30132 . "앓을 아") (21734 . "놀람의 어조사 아")
                 20816 23149 20124 (23104 . "娿와 同字") (20011 . "가장귀 아")
                 (23785 . "峨와 同字") 26895 31247 (23103 . "아리따울 아")
                 (30737 . "갈 아") 22968 40310 20473 32867 17101 20122
                 21713 22445 23045 25372 26720 27689 30166 35766
                 38070 38100 38165 39295 40486 40517 16713 (31500 . "대싹 아")
                 (37647 . "투구의 목가림 아") (40286 . "鵝와 同字")
                 20258 21386 21526 21782 21834 22975 23410 23641
                 24204 25495 27692 28048 29373 29601 29684 30120
                 30210 30354 30539 30826 34149 34460 35198 (35200 . "덮다")
                 35472 37427 37650 37914 39424 40790 14003 14174
                 14476 14480 14769 14798 14836 15411 16383 16577
                 16649 16688 17129 17130 17235 17848 18589 19125
                 19671 19672))
       (50501 . ((27138 . "풍류 악, 즐길 악") (24801 . "나쁠 악, 악할 악")
                 (23731 . "큰 산 악") (25569 . "잡을 악, 손아귀 악, 움켜질 악")
                 (24853 . "놀랄 악, 억지부릴 악") (22538 . "흰 흙 악")
                 (38990 . "턱 악, 턱뼈 악") (23997 . "높은 산 악, 큰산 악")
                 (24132 . "장막 악") (28197 . "두터울 악, 비젖을 악")
                 (39952 . "악어 악") (37122 . "나라이름 악, 고을이름 악")
                 (37716 . "칼날끝 악") (40823 . "악착할 악, 악착같을 악")
                 27005 (33852 . "꽃받침 악") (35556 . "직언할 악") (40346 . "물수리 악")
                 (21666 . "놀랄 악") (21908 . "닭 우는 소리 악, 아첨해 웃는 소리 악")
                 40465 (22121 . "놀랄 악") 36940 23871 34138 15257
                 (21374 . "윗턱 악") 22574 24694 40132 (33133 . "잇몸 악")
                 20048 22441 24746 35860 38199 39066 40535 40844
                 (20563 . "거리낄 악") (35240 . "오래 볼 악") (40822 . "잇몸 악")
                 20564 20715 25459 26947 27374 28226 29735 33115
                 34625 35725 37993 40055 13557 13758 13761 14311
                 14347 14733 15946 18446 18654 18755 19191))
       (50504 . ((23433 . "편안할 안, 어찌 안") (26696 . "책상 안, 생각할 안")
                 (30524 . "눈 안") (38996 . "顏과 同字, 얼굴 안") (23736 . "물가 언덕 안")
                 (38593 . "鴈과 同字, 기러기 안") (38797 . "안장 안")
                 (25353 . "살필 안, 어루만질 안") (26191 . "늦을 안, 편안할 안")
                 (39839 . "아귀 안, 천징어 안") 38991 (40200 . "기러기 안")
                 40387 36119 (29364 . "들개 안") (36107 . "가짜 안")
                 (26697 . "案과 同字") 19254 26956 15859 16705 36189
                 39068 38133 39032 20370 20560 22453 22547 22943
                 23026 23145 23766 26277 27688 27933 30712 33612
                 36549 37544 37644 38796 38919 39476 40171 40243
                 40683 13812 13846 14273 15080 15913 16386 18623
                 18947 19351 19395 19675))
       (50508 . ((35585 . "아뢸 알, 뵈올 알") (38332 . "막을")
                 (36555 . "수레 삐걱거릴 알, 빼앗을 알") (26017 . "돌릴 알, 주선할 알")
                 (36943 . "막을 알") (35344 . "들추어 낼 알") 25115 (25118 . "창 알")
                 (38942 . "콧마루 알") (27513 . "부서진뼈") 23981 (25568 . "뽑을 알")
                 29488 22304 25470 (40246 . "뻐꾸기 알") 22568 38016
                 19805 35750 35858 36711 38415 (22030 . "새소리 알")
                 (31349 . "구멍 알") 21148 21689 22048 23669 25366
                 25820 27514 28643 28965 31403 32848 33018 37687
                 40771 40830 64031 13717 13718 14793 14965 15822
                 16401 16458 17726 18783 19384 19401 19445 19506))
       (50516 . ((26263 . "어두울 암, 몰래 암, 어리석을 암")
                 (24022 . "바위 암, 험할 암") (30284 . "종양 암") (24245 . "초막 암, 암자 암")
                 (38343 . "어두울 암, 어리석을 암, 망루 암") (23721 . "巖의 俗字, 바위 암")
                 (33780 . "암자 암, 쑥 암") 20474 (21813 . "머금을")
                 (35571 . "욀 암") 23890 (40687 . "어두울 암") (23891 . "바위 암, 가파를 암")
                 38967 (39331 . "향기로울 암") (26235 . "어두울 암")
                 30878 (23189 . "머뭇거릴 암") 38909 25566 40685 (21885 . "잠꼬대 암")
                 30199 33851 40298 40676 19706 22511 35865 39060
                 40524 (33124 . "고기 삶을 암") (33866 . "풀이름 암")
                 (34029 . "암자 암") 21308 21873 22583 22759 24012
                 24023 30438 30443 31033 32623 35066 35485 38540
                 38648 40341 40601 40611 40684 13849 14074 14109
                 14212 16226 16726 17815 18948 19169 19219 19429))
       (50517 . ((22739 . "누를 압") (25276 . "수결 둘 압, 누를 압")
                 (40232 . "오리 압, 집오리 압") 23724 (29390 . "친근할 압, 업신여길 압, 익힐 압")
                 40493 24210 30991 32616 23691 23030 13609 17514
                 21387 22311 20753 21262 22292 24216 26353 28832
                 30736 30790 38821 39978 13646 14037 15564 15816
                 16270 16792 17509 18201))
       (50521 . ((20208 . "우러를 앙, 우러러볼 앙") (27523 . "재앙 앙")
                 (22830 . "가운데 앙") (24591 . "원망할 앙") (26171 . "오를 앙, 밝을 앙")
                 (40230 . "암원앙새 앙") (31207 . "모 앙") (26114 . "밝을,높을")
                 (30414 . "동이 앙") (38789 . "가슴걸이 앙, 소고삐 앙")
                 (27889 . "끝없을 앙") (21356 . "자기 앙") (22385 . "티끌 앙")
                 26506 19321 25264 32998 14277 16410 40495 20306
                 20639 21641 22990 23687 23711 24899 28852 30479
                 32059 32699 35399 36590 37280 37472 38645 39386
                 13461 15231 15975 16156 16366 16643 16870 17547
                 17570 17959 19029 19244 19322 19520))
       (50528 . ((24859 . "사랑 애") (21696 . "슬플 애, 가여울 애")
                 (28079 . "물가 애") (30861 . "礙의 俗字, 그칠 애") (22467 . "티끌 애, 속세 애, 먼지 애")
                 (33406 . "쑥 애") (21917 . "목 쉴 애") (23830 . "낭떠러지 애")
                 (38552 . "좁을 애") (38724 . "아지랭이 애, 아지랑이 애")
                 (26326 . "희미할 애, 흐릴 애") (21395 . "언덕 애, 물가 애")
                 (31001 . "막을 애") (34297 . "화기로울 애") (27448 . "한숨쉴 애")
                 (39427 . "어리석을 애, 말달릴 애") 29815 25969 29233
                 (30554 . "눈흘길 애") (30362 . "흴 애") (30649 . "흐릿할 애")
                 (25384 . "밀칠 애") 38305 14197 (20734 . "어렴풋할 애")
                 (34182 . "숨겨질 애") (38729 . "구름낄 애") (25457 . "막을 애")
                 (29507 . "어리석을 애") 22738 (30929 . "맷돌 애") 17725
                 (23085 . "계집종 애") 35690 (21769 . "대답하는 소리 애")
                 21571 27600 29862 14587 22003 23250 26279 30353
                 30809 34108 38207 38402 38701 (21824 . "물어뜯을 애")
                 (22127 . "트림할 애") (23829 . "崖와 同字") (28420 . "물가 애")
                 20236 21446 21710 21810 22026 22544 22631 23102
                 23329 23910 25043 25053 25971 26169 28653 29168
                 29466 30375 30777 30795 36089 37552 37764 37952
                 39218 39332 40043 40241 13644 13778 13879 14597
                 14728 15471 15804 16271 16324 16748 17743 18301
                 18976 19253 19875))
       (50529 . ((38989 . "이마 액, 수량 액") (21380 . "재앙 액")
                 (28082 . "즙 액, 진액 액") (33099 . "겨드랑이 액") (25212 . "움켜쥘 액, 움겨질 액")
                 (32266 . "목 맬 액") (25494 . "겨드랑이에 낄 액") 35451
                 (38440 . "막힐 액") (25636 . "움켜쥘 액") (25145 . "좁을 액")
                 36571 38943 (21597 . "울 액") 32546 36717 39069 23753
                 28081 30760 30912 33482 34437 35935 35990 36598
                 37418 38456 14319 14806 17976 18248 19065 19566))
       (50547 . (21393))
       (50549 . ((40367 . "꾀꼬리 앵") (27387 . "앵두나무 앵")
                 (40474 . "앵무새 앵") (32588 . "술병 앵") 40462 (22198 . "새 소리 앵")
                 29974 (40426 . "꾀꼬리 앵") 19725 (32579 . "술병 앵")
                 22052 26716 27185 32578 33722 40550 (23240 . "예쁠 앵")
                 22883 29959 35707 15620 15961 15962 16477 18953))
       (50556 . ((22812 . "밤 야") (37326 . "들판 야") (20063 . "단정의 어조사 야")
                 (33509 . "땅이름 야, 지혜(범어의 음역) 야, 반야 야")
                 23556 (32822 . "의문의 어조야 야, 아비 야, 어조사 야")
                 (24825 . "끌릴 야, 어지러울 야, 이끌 야") (20539 . "땅이름 야")
                 (26928 . "야자나무 야") (29242 . "아비 야, 노인 야, 노인존칭 야")
                 (20918 . "단련할 야, 대장장이 야, 쇠불릴 야") (25590 . "희롱할 야")
                 (22492 . "들판 야") 25427 26514 21903 29706 37603
                 22050 28978 29784 18739 29239 38104 20059 20145
                 20556 21524 22724 23164 25832 28444 31042 37374
                 37761 40314 13729 13827 13906 15208 17609 18810))
       (50557 . ((24369 . "약할 약, 어릴 약") (34277 . "약 약, 약할 약")
                 (32004 . "대략 약, 맹세할 약, 간소할 약, 맺을 약, 기약할 약")
                 (33509 . "같을 약, 자네 약, 만약 약, 반야 약") (30053 . "간략할 약, 노략질할 약, 잔꾀 약")
                 (36493 . "뛸 약") (25504 . "노략질할 약") (33903 . "꽃밥 약, 부들풀싹 약")
                 (33979 . "구약나물 약, 부들속 약") (38000 . "자물쇠 약")
                 (31845 . "피리 약, 잠글 약") 28729 (31156 . "종묘의 여름제사 약")
                 31039 (29210 . "스러질 약") (31707 . "죽순 약") 15736
                 (39983 . "멸치 약, 정어리 약") 31660 31673 34341
                 16337 33647 (40864 . "피리 약") 21727 32422 34220
                 36291 (40376 . "댓닭 약, 투계 약") 21458 21938 23926
                 26953 27906 28163 32389 34270 37120 39961 40473
                 13474 14128 15387 17124 17438 17795 18085 18385
                 18479 18688 19891))
       (50577 . ((32650 . "양 양") (20841 . "두 량, 둘 양") (38525 . "볕 양, 드러낼 양, 시월 양, 거짓 양")
                 (27915 . "큰 바다 양, 넘칠 양") (33391 . "어질 량, 좋을 량, 남편 량")
                 (39178 . "기를 양, 다스릴 양, 봉양할 양") (37327 . "헤아릴 량, 용량 량")
                 (35731 . "사양할 양") (25562 . "드높일 양, 칭찬할 양, 들날릴 양")
                 (20937 . "서늘할 량") (26753 . "들보 량, 다리 량") (27171 . "모양 양, 모범 양, 본보기 양")
                 (31975 . "양식 량, 곡식 양") (26954 . "갯버들 양") (22756 . "흙 양, 풍년들 양, 기름진흙 양")
                 (35538 . "헤아릴 량, 믿을 양") (20142 . "밝을 량, 임금의 상중 량")
                 (35140 . "오를 양, 장사지낼 양, 도울 양") (23363 . "계집아이 양, 아씨 양")
                 (30221 . "머리헐 양, 종기 양") (25880 . "물리칠 양, 빼앗을 양, 밀칠 양")
                 (37312 . "술 빚을 양") (24665 . "근심 양") (30306 . "가려울 양")
                 (31155 . "푸닥거리할 양, 기도할 양") (30162 . "가려울 양, 옴 양")
                 (20335 . "거짓 양, 어정거릴 양, 속일 양") (26264 . "해 돋을 양, 해돋이 양")
                 (31344 . "풍년들 양, 볏짚 양") (28673 . "물결 일렁일 양, 내이름 양")
                 (25965 . "오를 양, 칭찬할 양") (29036 . "구울 양, 녹일 양")
                 (39524 . "날뛰는 말 양") 35698 27096 (28478 . "물결 출렁거릴 양")
                 14910 26679 26140 (39098 . "날릴 양, 높일 양") (24457 . "배회할 양")
                 37722 25081 21239 17483 23861 13358 (38002 . "거푸집 속 양")
                 (28732 . "이슬 많은 모양 양, 강이름 양") 32405 22732
                 39263 17012 22431 27392 29910 35577 22199 31123
                 38438 13606 25196 38262 38451 20859 23330 26104
                 26472 28800 30113 35753 37247 37304 38038 39119
                 39317 39591 (28874 . "구울 양, 녹일 양") (30523 . "눈 아름다울 양")
                 (34328 . "양하 양") (36656 . "상여수레 양") 21175 21224
                 22861 24512 25065 25857 27304 27676 27687 27697
                 29209 29565 29660 29924 31010 31331 32655 32661
                 34520 34630 36511 37898 38711 39716 39953 40249
                 40457 14372 14777 15125 15601 16010 16465 17257
                 17478 17798 17849 19223 19258 19280 19301 19822))
       (50612 . ((39770 . "물고기 어") (35486 . "말씀 어") (28417 . "물고기 잡을 어")
                 (26044 . "방향의 어조사 어, 대신할 어") (24481 . "거느릴 어, 모실 어")
                 (31142 . "막을 어, 그칠 어") (22276 . "가둘 어, 옥 어")
                 (30208 . "어혈질 어") (39341 . "말 부릴 어") (40812 . "이 어긋날 어")
                 25205 (22281 . "마부 어, 변방 어") (39147 . "먹기 싫을 어, 배부를 어")
                 34323 (25940 . "금할 어, 풍류그칠 어") (28132 . "앙금 어, 진흙 어")
                 23807 31838 33784 37593 28180 35821 38139 39275
                 39533 40060 40841 33571 20178 21817 26844 31191
                 31626 31741 34041 34316 37287 39239 39922 40416
                 16489 17400 17459 17697 17896 18601 18767 19515
                 19575 19619))
       (50613 . ((20740 . "억 억, 편안할 억, 이바지할 억, 헤아릴 억, 인민 억")
                 (25014 . "생각 억, 기억할 억") (25233 . "누를 억, 억울할 억")
                 (33222 . "가슴 억, 가득할 억") (27277 . "참죽나무 억, 박달나무 억, 싸리나무 억")
                 34191 23991 28602 (32374 . "땋은 노 억") 20159 24518
                 24741 30292 32906 35378 39209 15047 17793 17911))
       (50616 . ((35328 . "말씀 언, 자기 언, 우뚝할 언")
                 (28937 . "어찌 언, 의심쩍을 언, 어조사 언") (24422 . "彥과 同字, 선비 언")
                 (35578 . "상말 언") (22576 . "방죽 언") (20547 . "자빠질 언, 누울 언")
                 24421 (35742 . "죄 의논할 언") 29698 21761 (23267 . "생긋생긋 웃을 언")
                 28473 34091 (37154 . "땅이름 언") (40756 . "두더쥐 언, 소 같은 큰 쥐 언")
                 (21309 . "숨길 언, 길 곁 언, 한데 뒷간 언") 21933
                 34648 (40761 . "鼴과 同字") 35744 35866 35891 (20671 . "鄢과 同字")
                 22037 22677 23086 23875 24861 26939 29290 29975
                 31734 33810 35095 35329 36541 36931 37118 38529
                 39947 40352 40798 40820 14471 15043 15421 16473
                 16894 17602 17889 18305 18743))
       (50620 . ((34326 . "그루터기 얼, 싹 얼") (23420 . "첩자식 얼, 요물 얼, 첩의자식 얼")
                 23421 31985 (33260 . "법 얼, 문지방 얼, 과녁 얼")
                 20091 33266 38353 38537 23922 31989 26559 14136
                 27377 34853 38221 24021 27127 37811 13918 15238
                 15658 18558))
       (50628 . ((22196 . "굳셀 엄, 공경할 엄, 엄할 엄")
                 (25513 . "거둘 엄, 가릴 엄") (22852 . "문득 엄, 가릴 엄, 그칠 엄, 오랠 엄, 문뜩 엄")
                 (20796 . "공경할 엄, 머리쳐들 엄") (20474 . "자기 엄, 클 엄")
                 (28153 . "담글 엄, 오래머물 엄") 28208 (24191 . "바윗집 엄")
                 25564 (38329 . "고자 엄, 내시 엄") 26235 (23846 . "해 지는 산 이름 엄")
                 24327 (32616 . "그물 엄") 39017 14655 14058 38546
                 20005 20456 (21378 . "언덕") 33100 35267 38409 (26350 . "해 다니는 길 엄")
                 (37251 . "절일 엄, 김치 엄") 37317 21094 21384 21427
                 22110 23373 23958 27413 30845 35214 40849 13991
                 14466 14491 15226 16651 17015 17790 18163 18637
                 19603 19883 19886))
       (50629 . ((26989 . "일 업, 위태할 업, 씩씩할 업, 벌써 업, 공경할 업")
                 (23978 . "산이 높고 웅장할 업, 산높은모양 업") 19994
                 14436 (37172 . "땅이름 업, 성씨 업") (23979 . "嶪과 同字")
                 28594 37050 27543 39203 39516 40456 13820 14411
                 14993 15433 15907 16454 17180 17320 18742 18920
                 19538 19612))
       (50631 . (26069))
       (50633 . (15048))
       (50640 . ((24666 . "성낼 에, 원망할 에") (26304 . "음산할 에, 가리울 에")
                 27562 39248 14636))
       (50644 . ((20870 . "일본화폐단위")))
       (50668 . ((22899 . "계집 녀, 시집보낼 녀") (22914 . "같을 여, 만약 여, 어떠할 여, 이를 여")
                 (39192 . "남을 여, 나라이름 여") (33287 . "너울너울할 여, 더불어 여, 같을 여, 미칠 여")
                 (20313 . "자기 여, 남을 여, 나 여") (27741 . "물 이름 여, 물에 젖을 여, 너 여")
                 (26053 . "나그네 려, 베풀 려") (20104 . "자기 여, 나 여")
                 (40599 . "고울 려, 빛날 려, 베풀 려") (36671 . "수레바탕 여, 짐질 여")
                 (21237 . "힘쓸 려, 가다듬을 려") (21570 . "풍류 려, 등골뼈 려, 성씨 려")
                 (24300 . "초가 려, 움막 려, 원두막 여") (31018 . "돌 단단할 려, 숫돌 여")
                 (39530 . "가라말 려, 나라이름 려, 산이름 려, 검은 말 여")
                 (38317 . "마을 려, 이문 여") (40654 . "무리 려, 동틀 려")
                 (28670 . "씻을 려, 맑을 려") (27487 . "그런가할 어조사 여")
                 (36701 . "輿와 同字, 가마 여") (33593 . "띠 뿌리 여, 받을 여, 헤아릴 여")
                 (29877 . "보배옥 여") (33349 . "나룻배 여, 나뭇배 여")
                 (30998 . "여돌 여, 섞인돌 여") 19982 (33281 . "마주들 여")
                 27428 30060 27955 26079 16522 34167 40253 20227
                 22948 28582 31865 39296 28394 30066 33286 34019
                 38135 20382 23337 24556 24710 25049 26695 27378
                 29427 29593 30061 30818 31004 31333 31566 31813
                 32919 34144 37018 37539 38611 39869 40466 13468
                 14747 15771 16030 16130 16274 16539 17474))
       (50669 . ((21147 . "힘 력, 부지런할 력") (20134 . "또 역, 모두 역")
                 (26131 . "바꿀 역") (36870 . "거스를 역") (27511 . "지낼 력, 겪을 력, 지날 역")
                 (24441 . "부릴 역, 국경 지킬 역") (39515 . "잇닿을 역, 쌀 뾰족뾰족할 역, 역말집 역")
                 (35695 . "통변할 역, 번역할 역") (26310 . "셀 력, 세월 력, 책력 역")
                 (22495 . "경계 역, 범위 역, 지경 역") (30123 . "염병 역")
                 (32377 . "삭일 역, 베풀 역, 실 끝 찾을 역") (36706 . "수레바퀴에 치일 력, 서로 부닥칠 력, 수레바퀴에 치는 역")
                 25985 22520 34606 26233 (23975 . "연달아 있는 산 역")
                 15634 (25036 . "기꺼울 역") 28249 24095 39721 (38334 . "문지방 역")
                 26859 29153 30570 40322 (28130 . "빨리 흐를 역")
                 22340 27485 29721 39754 23748 24639 32462 35379
                 35793 38408 39365 39551 40541 40543 20231 22299
                 22460 22719 23664 25131 26318 26861 28808 30188
                 30811 32206 32268 32621 34377 35931 37299 37408
                 39907 40323 40394 13386 13825 13980 14530 15617
                 15776 16091 16227 17246 17608 17642 17804 18523
                 18868 18901 19294 19353 19493 19692))
       (50672 . ((24180 . "해 년, 나이 년, 나아갈 년") (28982 . "불사를 연, 그러할 연, 허락할 연, 그러나 연, 그럴 연")
                 (36899 . "이을 련, 끌릴 련, 연할 연") (29017 . "연기 연")
                 (32244 . "이길 련, 익힐 련, 겪을 련") (30799 . "연마할 연, 갈 연")
                 (37706 . "불린 쇠 련, 단련할 련, 쇠불릴 연") (23476 . "잔치할 연, 편안할 연")
                 (24310 . "미칠 연, 닿을 연") (32227 . "인연 연, 서두를 연")
                 (28436 . "펼칠 연, 긴 물줄기 연, 통할 연, 윤택할 연, 넓힐 연")
                 (24976 . "불쌍할 련, 사랑할 련, 가련할 연") (25088 . "생각할 련, 사모할 련")
                 (32879 . "잇닿을 련, 관계할 련, 연할 연") (34030 . "연밥 련, 연꽃 련")
                 (29123 . "불사를 연, 연등절 연, 불탈 연") (36575 . "부드러울 연, 연할 연")
                 (28149 . "못 연, 깊을 연, 북소리 둥둥할 연, 모래톱 연")
                 (29141 . "제비 연, 편안할 연, 잔치 연, 나라이름 연")
                 (27839 . "물 따라 내려갈 연, 좇을 연") (37467 . "납 연, 분 연")
                 21693 (34893 . "성할 연, 넓을 연, 물이 넘칠 연, 상자 연, 아름다울 연, 흐를 연")
                 (25424 . "버릴 연, 병들어 죽을 연") (30831 . "벼루 연, 돌 연")
                 (28451 . "물 놀이칠 련, 눈물 줄줄 흘릴 련") (23032 . "고을 연, 사랑스러울 연, 고울 연")
                 (31605 . "대자리 연") (36646 . "당길 련, 궁중의 길 련")
                 25062 (26941 . "서까래 연") (40182 . "솔개 연") (25754 . "잡을 년, 닦을 년, 밟을 년, 손 끝으로 비빌 년")
                 (29001 . "쇠 불릴 련, 반죽할 련") (28895 . "煙과 同字, 연기 연")
                 (28051 . "물방울 떨어질 연, 조촐할 연") (29833 . "종묘 제기 련")
                 (27783 . "물 졸졸 흐를 연, 물흐르는모양 연") (23071 . "어여쁠 연, 춤추는 모양 연")
                 (28046 . "침 연, 물 졸졸 흐를 연") (22567 . "빈 땅 연, 변두리땅 연")
                 (22181 . "침 삼킬 연, 목구멍 연") (25403 . "당길 연, 달아날 연, 손바닥으로 칠 연")
                 (31178 . "年의 本字, 해 연") (32303 . "길 연, 늘일 연")
                 33877 30740 20822 (35724 . "잔치 연, 모여 말할 연")
                 (33241 . "목구멍 연, 연지 연") (33690 . "만연할 연, 풀 이름 연")
                 32536 22742 34578 21550 (25534 . "아전 연, 기인할 연")
                 (24705 . "분할 연, 근심할 연") (20823 . "고을이름 연, 믿을 연, 단정할 연")
                 (22479 . "땅 가장자리 연") 36653 37382 30877 15250
                 24862 32782 (28215 . "물 이름 연") 32150 37579 22925
                 29658 32257 15554 32944 34837 (40432 . "제비 연, 씨름 연")
                 34574 34653 15577 25739 (23359 . "아름다울 연") 25217
                 28737 29772 29888 31005 37308 16553 27260 28170
                 30746 36719 38085 40482 (22246 . "물 깊을 연") (26339 . "더울 연, 청명할 연")
                 (27358 . "연나무 연") (34613 . "우렁찰 연, 벌레 이름 연")
                 21064 22059 22126 23010 23083 23306 23357 25541
                 26134 26857 26940 27182 27242 28161 28166 28181
                 28302 28923 29439 32219 32334 32921 33005 34053
                 35051 35326 36456 36683 37076 37184 39520 40728
                 40733 13444 13556 13667 14115 14170 14739 14862
                 15071 15147 15175 15556 15597 15626 15772 15825
                 15868 15888 15926 16070 16237 16275 16373 16380
                 16445 16564 16663 16787 17049 17278 17652 17715
                 17788 17796 17870 17914 17991 18119 18217 18306
                 18597 18954 19207 19260 19570 19683 19711 19727))
       (50676 . ((35498 . "기꺼울 열, 말씀 열") (29105 . "뜨거울 열, 쏠릴 열, 더울 열")
                 (21015 . "벌릴 렬, 무리에 들어갈 렬, 항렬 렬, 베풀 렬, 벌일 열")
                 (24709 . "즐거울 열, 복종할 열, 성씨 열, 기뻐할 열")
                 (28872 . "불 활활 붙을 렬, 빛날 렬, 업적 렬, 아름다울 렬, 사나울 렬, 매울 열")
                 (21155 . "용렬할 렬, 못날 렬, 어릴 렬, 서툴 렬")
                 (35010 . "찢어질 렬, 비단 자투리 렬, 찢을 열") (38321 . "겪을 열, 군대 점호할 열, 용납할 열, 볼 열")
                 (21693 . "목 멜 열, 막힐 열") (28037 . "극락갈") 25553
                 29191 (22094 . "목 멜 열") 25341 28529 38322 24689
                 28284 38405 24742 28909 34862 13750 16789 16802
                 17062 19271))
       (50684 . ((24565 . "생각할 념, 읽을 념, 스물 념, 대단히 짧은시각 념")
                 (28814 . "불꽃 염, 임금의 호 염") (40573 . "소금 염, 절일 염, 후렴 염")
                 (26579 . "꼭두서니 염, 물들일 염") (24265 . "맑을 렴, 조촐할 렴, 청렴할 렴, 살필 렴")
                 (21421 . "넉넉할 염, 아름다울 염, 게으를 염, 만족할 염, 미워할 염, 편안할 염")
                 (38331 . "마을 염, 항간 염, 저승 염, 마을안의 문 염")
                 (28976 . "불당길") (31806 . "발 렴") (33398 . "고울 염, 얼굴 탐스러울 염")
                 (27566 . "염할 염") (33490 . "풀 우거질 염, 덧없을 염, 풀 우거진 모양 염")
                 (39653 . "구레나룻 염") (29744 . "아름다운 옥 염, 비취옥 염")
                 21089 (25467 . "손가락으로 찍을 념") 26511 37663
                 (22633 . "鹽의 俗字") (20873 . "가는 털 늘어질 염, 남생이 등 언저리 염, 침노할 염")
                 29124 (28750 . "灩의 俗字") (39260 . "싫을 염, 물릴 염")
                 (39768 . "잠꼬대할 염") 39663 26325 29458 15828 (28777 . "물 그득할 염, 물결 출렁거릴 염")
                 38760 (40694 . "검은 사마귀 염") 34977 35924 (25162 . "문 빗장 염")
                 28775 (27327 . "산뽕나무 염") 28977 20579 20868 21428
                 21605 23342 34470 16395 33395 21388 21411 24697
                 27539 28383 30416 37245 38379 38414 39181 39751
                 40118 40673 (25045 . "편안할 염") 27318 37317 20085
                 22747 22837 22988 23203 25064 25257 25957 26858
                 27369 28756 28948 29635 33399 33925 34490 34939
                 35923 37203 37302 38278 39920 13993 14740 15448
                 15748 15819 15866 16341 16743 17194 17283 17318
                 17571 18113 18680 18721 18866 19143 19890))
       (50685 . ((33865 . "잎 엽, 세대 엽, 성씨 엽") (29557 . "사냥할 렵, 진동할 렵, 어긋날 렵, 바람소리 렵")
                 (29121 . "불 이글글할 엽, 번쩍번쩍할 엽, 번쩍거릴 엽")
                 (26308 . "빛날 엽, 번개 번쩍거릴 엽") 21494 25586
                 26986 39768 39233 (29207 . "불 이글글할 엽, 번쩍번쩍할 엽")
                 30641 (29056 . "불빛 이글이글할 녑") (38760 . "웃는 볼 옴쏙 들어갈 엽, 얼굴의 검은 사마귀 엽")
                 26327 25819 (26309 . "曄과 同字") 26196 28904 38757
                 39308 20574 20727 22152 25834 25835 26556 26557
                 27548 30371 30648 37943 13487 14926 15864 16910
                 19278 19295))
       (50687 . (13746 13747))
       (50689 . ((27704 . "길 영, 오랠 영") (33521 . "꽃부리 영, 영웅 영, 구름 뭉게뭉게 일어날 영, 아름다울 영")
                 (20196 . "하여금 령, 시킬 령, 가령 령, 개 목소리 령, 벼슬이름 령")
                 (38936 . "옷깃 령, 고개 령, 거느릴 령, 받을 령")
                 (27054 . "영화로울 영, 오동나무 영, 추녀 영, 꽃다울 영")
                 (36814 . "맞을 영") (29151 . "지을 영, 헤아릴 영, 다스릴 영, 진 영")
                 (23527 . "편안할 녕, 차라리 녕, 어찌 녕") (38728 . "신령 령, 혼백 령, 좋을 령")
                 (23994 . "고개 령, 봉우리 령, 산고개 영") (24433 . "그림자 영, 형상 영")
                 (35424 . "읊을 영") (26144 . "비칠 영, 빛날 영") (27891 . "헤엄칠 영")
                 (38646 . "부서질 령, 비뚝비뚝 떨어질 령, 셈 나머지 령")
                 (30408 . "가득할 영, 넘칠 영, 물 흐를 영, 찰 영")
                 (22603 . "무덤 영, 산소 영") (37428 . "방울 령") (29787 . "옥빛 영, 수정 영, 옥광채 영")
                 (29618 . "옥소리 쟁그렁거릴 령, 아롱아롱할 령, 정교할 령, 금옥소리 영")
                 (22265 . "감옥 령") (26254 . "비칠 영") (23344 . "어릴 영, 더할 영, 머리치장할 영, 물요괴 영, 갓난아이 영")
                 (31310 . "이삭 영, 송곳 끝 영, 빼어날 영") (32403 . "갓끈 영, 말 가슴걸이 영, 얽힐 영")
                 (29801 . "귀막이 옥 영, 밝을 영") (32838 . "들을 령, 깨달을 령")
                 (24604 . "영리할 령") (27001 . "기둥 영, 하관들 영")
                 (28699 . "큰 바다 영, 신선 사는 산 영") (28481 . "물 이름 영, 성씨 영, 강이름 영")
                 (23992 . "산 높을 영, 산이험한 영") (38681 . "눈꽃 영, 구름 피어오를 영")
                 (29552 . "영악할") (28634 . "물 돌아나갈 영, 물도는 모양 영")
                 (28214 . "물 맑을 영") (28719 . "물소리 영, 물 돌아나갈 영, 물웅덩이 영")
                 (32666 . "영양 령, 큰 뿔 양 령") (29008 . "빛날 영")
                 (29908 . "옥돌 영, 구슬 목걸이 영") (37704 . "방울소리 영")
                 (21647 . "노래할 영, 읊을 영") (36111 . "이문 남길 영, 멜 영, 자랄 영")
                 (32264 . "얽힐 영, 맬 영") (23348 . "성씨 영, 풀어낼 영, 가득할 영, 아름다울 영")
                 33829 (37090 . "땅이름 영") 21942 23896 26640 31132
                 25878 38964 (30317 . "목 혹 영") (38906 . "풍류이름 영")
                 28764 35246 31837 31855 20544 23190 28299 28633
                 28724 34337 23156 25732 26628 29406 29838 30271
                 32492 32552 33830 34686 36194 38195 39053 39061
                 39062 (28007 . "잠길 영, 앙금 영") (28660 . "물 졸졸 흐를 영")
                 (30884 . "물 속 돌 영") (34833 . "도마뱀 영") 23339
                 23366 23422 24006 24010 24302 24634 24869 25708
                 25869 25882 26400 26573 26593 26796 26983 27391
                 28289 29187 29828 30401 30696 31023 32211 33854
                 34663 34867 35597 36047 37279 37915 37965 40359
                 13831 14130 14421 14477 14869 15264 15519 15724
                 15752 16344 16436 16464 16631 17044 17117 17481
                 17485 17766 18028 18054 18640 18859 18957 19119
                 19274 19287 19288))
       (50696 . ((31150 . "예도 례, 인사 례") (20363 . "본보기 례, 견줄 례, 대부분 례, 법식 예")
                 (34269 . "재주 예, 글 예, 가늠할 예, 극진할 예, 심을 예")
                 (35947 . "기쁠 예, 편안할 예, 미리 예") (35709 . "칭찬할 예, 이름날 예, 즐길 예, 기릴 예")
                 (37555 . "날카로울 예, 까끄라기 예") 38583 (38928 . "미리 예, 참여할 예")
                 (30591 . "슬기로울 예, 밝을 예, 통할 예") (27844 . "내칠 예, 날개 훨훨칠 예, 흩을 예")
                 (37300 . "단술 례, 고을이름 례") (28618 . "흐릴 예, 물 넘칠 예, 물넘치는 모양 예")
                 (33454 . "풀 뾰족뾰족할 예, 물가 예, 방패끈 예, 나라이름 예, 성씨 예")
                 (35427 . "이를 예, 나아갈 예, 학문 통달할 예") (26355 . "끌 예, 당길 예, 천천히 할 예")
                 (35028 . "옷 뒷자락 예, 변방 예, 후손 예") (31330 . "더러울 예, 거칠 예")
                 (27945 . "퍼질 예, 날개 훨훨칠 예") (38584 . "종 례, 노비 례, 검열할 례")
                 (21000 . "풀 벨 예, 죽일 예") (20034 . "풀 벨 예, 다스릴 예")
                 (20522 . "어릴 예, 도울 예, 끝 예") (38675 . "암무지개 예")
                 (29450 . "사자 예") (27757 . "물 이름 예, 물 굽이 예")
                 (34306 . "蕊와 同字, 꽃술 예") (21473 . "밝을 예, 임금 예, 어질 예")
                 (32755 . "어조사 예, 깃 일산 예, 가려 숨길 예, 새이름 예")
                 30238 (32324 . "검푸른 비단 예, 탄식하는 소리 예")
                 30568 (34122 . "꽃술 예, 꽃 예") (39906 . "암고래 예")
                 (29545 . "민족이름 예") 26304 34555 (26520 . "자루 예")
                 32703 21225 34617 (25341 . "당길 예") (34618 . "말매미 예, 암무지개 예")
                 26357 17650 (34185 . "더러울 예") 18134 (40406 . "갈매기 예, 봉황 예")
                 36631 (34123 . "蕊의 俗字") (34443 . "모기 예") 34042
                 35982 16604 (22216 . "잠꼬대할 예, 헛소리할 예")
                 22518 22532 26226 14460 (30614 . "눈에 백태 낄 예")
                 (40593 . "아기사슴 예, 사자 예") 26509 29823 34329
                 35308 37613 13935 35465 38160 21210 21587 30231
                 31229 33402 35811 39044 40117 40549 (23259 . "유순할 예, 갓난아이 예")
                 24802 (25500 . "부칠 예, 비길 예, 잡을 예, 당길 예")
                 (33477 . "벨 예, 자를 예") 20863 21613 22753 23127
                 23253 23537 24096 24584 24728 25433 26663 26879
                 27128 27236 27577 28131 29620 30257 31175 32352
                 34307 34979 35129 35196 35405 35739 36295 36357
                 37107 37839 38782 39254 40691 40815 13698 13920
                 13930 14067 14098 14179 14443 14628 14853 14994
                 15019 15498 15548 15709 15721 15981 16561 16641
                 16703 16780 16831 16868 16873 17141 17266 17288
                 17741 17802 17887 17981 18204 18221 18314 18554
                 18563 19352 19385))
       (50724 . ((20116 . "다섯 오") (21320 . "낮 오, 남쪽 오, 말 오")
                 (24801 . "미워할 오, 부끄러울 오, 악할 오") (21566 . "자기 오, 아들 오, 나 오")
                 (35492 . "그릇할 오") (28879 . "까마귀 오, 검을 오, 어찌 오, 탄식할 오")
                 (24735 . "깨우칠 오, 깨달을 오") (21555 . "오나라 오, 큰소리 지를 오, 성씨 오")
                 (21978 . "탄식할 오") (27738 . "더러울 오, 웅덩이 오")
                 (23067 . "즐거울 오") (20658 . "업신여길 오, 즐길 오, 거만할 오")
                 (22887 . "속 오, 비밀 오, 깊을 오") (26791 . "머귀나무 오, 버틸 오, 오동나무 오")
                 (22714 . "방 구들 오, 물가 언덕 오, 뭍 오") (20237 . "五의 갖은자, 다섯 사람 오, 무리 오")
                 (23524 . "잠 깨어날 오, 깨달을 오, 잠깰 오") (25034 . "원망할 오, 번뇌할 오, 사랑할 오")
                 31121 (28595 . "깊을 오, 물 이름 오") (39986 . "큰 자라 오")
                 (40711 . "매우 큰 자라 오") (26212 . "밝을 오, 만날 오")
                 (25942 . "희롱할 오, 거만할 오, 오만할 오") (22626 . "산 언덕 오, 마을 오")
                 (29100 . "볶을 오, 애 끓는 소리 오") (29522 . "사나운 개 오, 호박개 오")
                 (31613 . "버들 고리 오") (34568 . "지네 오") (20425 . "맞이할 오")
                 (26111 . "밝을 오, 대낮 오") (24548 . "거스릴 오, 미워할 오")
                 (22007 . "여럿이 걱정할 오") 26439 (23210 . "할미 오, 땅귀신 오")
                 (27737 . "汚와 同字") (36968 . "노닐 오") (37846 . "무찌를 오, 구리동이 오")
                 24734 19176 (35158 . "도포 오") 21556 27745 29310
                 36803 (39489 . "준마 오, 깔볼 오") 23228 (31417 . "아궁이 오")
                 (36821 . "만날 오, 어길 오") (40751 . "날다람쥐 오")
                 21577 27967 (32881 . "못 들은채 할 오, 말 더듬거릴 오")
                 21419 (21780 . "글 읽는 소리 오") (24928 . "거만할 오")
                 (35639 . "중얼거릴 오, 거만할 오") (38569 . "감출 오, 물가 언덕 오")
                 (25410 . "부닥칠 오, 거스릴 오") (37834 . "번철 오")
                 22114 37140 22008 25207 28015 34735 24274 29832
                 15987 (20213 . "짝 오, 같을 오") 22315 23912 23941
                 25817 29688 31078 37082 37845 17336 30941 37592
                 38550 40404 14719 40140 (20451 . "갈래질 오") (22316 . "흙손 오")
                 20044 21596 22366 22885 23089 23705 24750 34948
                 35823 38056 39580 (22081 . "성낼 오") (23279 . "업신여길 오")
                 (33571 . "차조기 오") 20036 21838 22497 22881 23082
                 23087 23988 24345 24546 25664 25710 27469 27771
                 27926 28329 28406 28570 28944 29075 29194 29523
                 29597 29798 30182 31375 32378 32927 33077 34076
                 34305 34704 35640 36278 36892 36987 37794 37933
                 38266 38558 39966 40238 13317 13416 14108 14121
                 14332 14338 14359 14506 14599 14944 15155 15271
                 15468 16077 16306 16368 16598 17471 17610 17818
                 18153 18193 18194 18331 18535 18781 18827 18844
                 19155 19343 19375 19424 19720 19744 19781))
       (50725 . ((29577 . "구슬 옥, 사랑할 옥, 이룰 옥")
                 (23627 . "집 옥, 지붕 옥, 거북 덮개 옥") (29508 . "감옥 옥")
                 (27779 . "기름질 옥, 부드러울 옥, 손 씻을 옥, 아리따울 옥")
                 (37434 . "보배 옥") 37576 29425 33721 38064 21101
                 23177 30753 31676 36553 40191 13511 17473 18333))
       (50728 . ((28331 . "따뜻할 온, 부드러울 온") (31337 . "편안할 온, 곡식 걷어 모을 온")
                 (34314 . "마름 온, 쌓일 온, 쌓을 온") (32277 . "모시 온, 묵은 솜 온")
                 (30239 . "염병 온, 온역 온") (29797 . "사람이름 온")
                 (37278 . "술 빚을 온") 28201 (24909 . "성낼 온, 병 온")
                 24864 34292 34160 (27699 . "기운 어릴 온, 기운 성할 온")
                 27698 (38878 . "감출 온, 붉은 빛 온") 23228 37270
                 39335 (26167 . "어질 온") 38832 (34176 . "마름 온, 쌓일 온")
                 (36672 . "온량차 온") 27058 33941 29061 22637 30226
                 37822 34164 37213 25598 27013 29044 32522 36754
                 38891 40129 (39234 . "밀 국수 온") (39982 . "멸치 온")
                 25653 26749 27551 32252 33920 35102 35953 36668
                 38374 39963 14183 15112 16148 16753 18847 18871
                 19283))
       (50732 . ((20800 . "우뚝할 올, 움직이지 않을 올, 오똑할 올")
                 33149 26444 21372 23676 (33155 . "해구 올, 살찔 올")
                 25188 (21986 . "목 멜 올") 30713 13327 20079 19122
                 28148 30595 38434 38768 40767 40768 13338 13363
                 16254 17506 18829))
       (50745 . ((32705 . "늙은이 옹, 새 목털 아래 옹") (25793 . "안을 옹, 품을 옹, 가질 옹")
                 (38605 . "조화로울 옹, 학교 옹, 화할 옹") (22725 . "막을 옹, 북돋울 옹")
                 (29973 . "항아리 옹, 물장군 옹, 독 옹") (37013 . "막힐 옹, 사람이름 옹")
                 (29934 . "항아리 옹, 독 옹") (39252 . "아침밥 옹") (30320 . "헌곳")
                 (38994 . "우러를 옹") 30152 (21889 . "물고기 입 오물거릴 옹")
                 (33994 . "옹풀 옹, 우거질 옹") (38621 . "조화로울 옹, 학교 옹")
                 (28355 . "구름 피어 오를 옹, 안개 자욱할 옹") (32587 . "甕과 同字")
                 (31162 . "땅 이름 옹") (24305 . "벽옹 옹") 33219 19066
                 13938 14168 25317 39065 (30293 . "등창 옹") 21212
                 21960 21985 22128 22613 23905 26273 28589 28745
                 30600 32876 34169 34697 37114 37779 39941 40370
                 40774 13858 14089 14485 14883 15276 15657 15823
                 16011 16126 16235 16328 16949 17445 17704 17912
                 19573))
       (50752 . ((33253 . "누울 와, 쉴 와") (29926 . "질그릇 와, 기와 와")
                 (35355 . "거짓말 와, 어긋날 와") (34680 . "달팽이 와")
                 (28198 . "소용돌이 와, 시끄러울 와") (34521 . "개구리 와")
                 (31401 . "굴 와, 움집 와, 감출 와") 37707 (31402 . "도랑 와, 깊을 와, 맑은물 와")
                 23043 21351 (27964 . "물 이름 와") (21703 . "음란한 소리 와, 게울 와")
                 23207 28084 (35660 . "거짓 와, 와전될 와, 요사스런 말 와")
                 (33845 . "상추 와, 부루 와") 40707 (22254 . "새 후릴 와")
                 21546 18975 (26521 . "옹이 와") (31370 . "웅덩이 와")
                 25456 28453 37387 28065 31389 33716 34583 35769
                 (23120 . "정숙할 와, 단정할 와") (29479 . "와개 와")
                 20324 21651 21801 25650 25896 28089 28315 30038
                 30745 36823 37047 38923 13861 14018 14810 15995
                 16152 16570 16807 18458 19456 19831))
       (50753 . (18610))
       (50756 . ((23436 . "완전할 완, 끝날 완, 지킬 완")
                 (32233 . "느즈러질 완, 더딜 완") (33694 . "빙그레 웃는 모양 완, 빙그레웃을 완")
                 (38929 . "완고할 완, 완악할 완") (23451 . "어슴푸레할 완, 언덕 위 언덕 완, 굽을 완")
                 (23113 . "순할 완, 예쁠 완, 젊을 완") (38446 . "성씨 완, 관이름 완")
                 (29715 . "구슬") (29609 . "희롱할 완, 보배 완") (28003 . "옷 빨 완, 씻을 완")
                 (33109 . "팔뚝 완") (32747 . "익숙할 완, 싫을 완, 구경할 완, 싫은 완")
                 (26880 . "주발 완, 사발 완") (29740 . "보석 완, 아름다운 옥 완, 서옥 완")
                 (30871 . "그릇 완, 주발 완") (26785 . "네 발 도마 완, 토막나무 완")
                 (33048 . "중완 완") (35916 . "동부 완, 완두 완") (24779 . "한탄할 완, 놀라 탄식할 완")
                 34189 (23695 . "산 뾰족할 완") 34623 (30412 . "은 바리 완")
                 (21011 . "깎을 완") 21084 (28084 . "물 굽이쳐 흐를 완, 진흙 묻을 완")
                 25231 36624 (24552 . "탐할 완, 사랑할 완") 22253
                 32531 39037 (22456 . "회 섞어 바를 완") (22951 . "좋을 완, 고울 완")
                 21773 21848 22502 23136 23880 24117 25430 25445
                 28919 30549 32689 33814 36006 36099 13860 14049
                 14196 14570 15463 16104 16157 17854 18115 18297
                 19400 19419))
       (50760 . ((26352 . "가로되 왈") 21014 21975 32841
                 18100))
       (50773 . ((29579 . "임금 왕, 할아버지 왕") (24448 . "갈 왕, 옛 왕, 이따금 왕")
                 (26106 . "고울 왕, 해무리 왕, 성할 왕, 왕성할 왕")
                 (27754 . "물 출렁출렁할 왕, 연못 왕, 깊고넓을 왕")
                 (26505 . "굽을 왕, 원통할 왕") 24451 (23586 . "절름발이")
                 (23595 . "절름발이 왕, 파리할 왕, 천상바라기 왕")
                 20220 (36844 . "往과 同字") 15551 (28679 . "아득할 왕, 넓을 왕, 고인 물 썪어 냄새날 왕")
                 23587 23593 23594 24442 24569 26240 34463 13388
                 13484 14263 15581 15609 15631 18705 19885))
       (50780 . ((20525 . "나라이름 왜, 일본") (30702 . "난장이 왜")
                 (23043 . "어여쁜 계집 왜, 얼굴 아름다울 왜, 예쁜계집 왜")
                 (27498 . "비뚤") (23207 . "여자이름 왜") 39463 23090
                 21176 32250 36535 13775 15402 19856))
       (50808 . ((22806 . "바깥 외, 다를 외, 잃을 외, 아버지 외")
                 (30031 . "두려울 외, 겁낼 외") (29477 . "더러울 외, 굽을 외, 섞일 외, 개 새끼 셋 낳을 외, 음란할 외")
                 (24013 . "높을 외, 높고 튼 모양 외") (23916 . "산 뾰족할 외, 산 울뚝질뚝할 외, 험할 외")
                 38536 24262 (32885 . "배냇귀머거리 외") (29032 . "불에 묻어 구울 외, 그슬릴 외, 사람이름 외")
                 26693 (38551 . "높을 외, 성씨 외") (30920 . "돌 사닥다리 외")
                 (20558 . "사랑할 외, 가물거릴 외") (23860 . "평평치 아니할 외")
                 26931 (30888 . "돌 자갈 땅 외, 바위 옹두라지 외")
                 35444 25024 25547 33138 32873 40130 (23892 . "산이름 외, 산 높을 외")
                 (28200 . "더러울 외") 23774 24507 24836 28350 30243
                 33896 37729 39840 39939 13911 14065 14215 14324
                 14357 16620 17151 18495 19173))
       (50827 . (22814))
       (50836 . ((35201 . "구할 요, 살필 요, 언약할 요, 허리 요, 기다릴 요, 부를 요, 반드시 요")
                 (27138 . "좋아할 요, 즐길 요") (26009 . "말질할 료, 헤아릴 료, 대금 료")
                 (20102 . "마칠 료, 깨달을 료") (35616 . "노래 요, 소문 요")
                 (20698 . "벗 료, 어여쁠 료, 동관 요") (36953 . "멀 요, 노닐 요")
                 (38518 . "화락할 요, 순임금의 신하 요") (25622 . "흔들 요, 회오리 바람 요, 별이름 요")
                 (33136 . "허리 요") (36988 . "멀 료, 강이름 료, 나라이름 료")
                 (22575 . "높을 요, 요임금 요") (22934 . "요괴로울 요, 생긋생긋 웃는 모양 요, 요사할 요")
                 (32768 . "빛날 요") (30274 . "병 나을 료, 병 고칠 료")
                 (26332 . "해 비칠 요, 요일 요") (23002 . "어여쁠 요, 날랠 요, 멀 요, 예쁠 요")
                 (23615 . "오줌 뇨") (25854 . "길들일 요, 번거로울 요, 길드릴 요")
                 (25747 . "긁을") (39250 . "용서할 요, 배부를 요") (36992 . "맞을 요, 구할 요, 부를 요, 맞이할 요")
                 (22829 . "어여쁠 요, 무성할 요, 재앙 요, 일찍 죽을 요, 뱃속에 든 아이 요, 끊어 죽일 요")
                 (23534 . "동관 료, 관리 료, 중의 집 료") (20709 . "거짓 요, 난장이 요, 요행 요")
                 (29134 . "뜰에 세운 횃불 료, 풀 태울 료, 햇불 요")
                 (31368 . "고요할 요, 깊을 요") 21676 (20985 . "오목할 요")
                 (25303 . "고집스러울 요, 꺾을 요") (31407 . "기와가마 요, 오지그릇 요")
                 (23525 . "잠잠할") (32350 . "얽힐 요, 둘릴 요") (29796 . "아름다운 옥 요, 못 이름 요, 옥돌 요")
                 (34044 . "병 나을 료, 여뀌 요") (23970 . "높을 요, 산이높은 요")
                 (32327 . "따를 요, 성할 요") (27208 . "노 요, 돛대 요")
                 (29183 . "비칠 요, 빛날 요") (34799 . "촌백충 요") 20040
                 (24186 . "작을 요, 어릴 요, 작다 요") (24493 . "부릴 요")
                 (28550 . "걸찰 요, 물 끌어올 요") (24508 . "돌아다닐 요, 순라 요, 변방 요, 구할 요")
                 (27520 . "단명할 요") 36602 (34136 . "나무할 요, 거칠 요")
                 25594 35617 25671 31377 (36982 . "둘릴 요") 29814
                 31408 (31365 . "움펑눈 요, 까마득할 요") 22169 (22387 . "凹와 同字")
                 (40386 . "장끼 요, 새매 요") 39099 15661 23597 (31045 . "재앙 요")
                 (31358 . "깊숙할 요, 골방 요") 37530 20636 23438
                 34958 28308 39445 17989 31380 16463 16791 (22685 . "메마른 밭 요, 자갈 땅 요")
                 23972 29671 33466 33917 35358 14897 16472 18595
                 19418 35875 20389 23046 23591 23779 25376 26721
                 27975 32469 33627 34546 36965 39126 39286 40144
                 40542 (20576 . "허리 가늘 요") (21907 . "벌레소리 요")
                 (23304 . "예쁜체 할 요") 20216 21510 21954 22407
                 22426 22448 23161 23217 23249 23686 23870 24506
                 24878 25023 25245 26266 26518 26602 26620 26766
                 26950 27034 27043 28881 29070 29397 29498 30661
                 30936 31556 33280 33374 33517 34004 34344 35155
                 35230 35407 35729 36586 36619 37776 38340 38562
                 38783 39174 39977 40193 40226 40405 13379 13425
                 13432 13438 13677 13869 13912 14164 14209 14321
                 14802 15055 15056 15955 15963 16354 16655 16793
                 16798 17153 17162 17179 17219 17668 17844 18106
                 18107 18459 19164 19225 19689 19879))
       (50837 . ((27442 . "욕심낼 욕, 하고자할 욕") (28020 . "깨끗이할 욕, 못 이름 욕, 목욕할 욕")
                 (36785 . "욕될 욕, 굽힐 욕") (24958 . "욕심낼 욕, 하고자할 욕")
                 (35109 . "요") (32287 . "꾸밀") (28349 . "유월 더위 욕, 무더울 욕")
                 40274 (34000 . "새싹 욕, 가을 맡은 귀신 욕") 37135
                 26137 32539 40518 21973 23223 23786 36621 37578
                 14778))
       (50857 . ((29992 . "쓸 용, 재물 용, 통할 용") (23481 . "얼굴 용, 감쌀 용, 용서할 용")
                 (21191 . "날랠 용, 용맹할 용") (40845 . "용 룡, 임금 룡")
                 (24248 . "쓸 용, 떳떳할 용, 수고로울 용, 어리석을 용, 물도랑 용")
                 (36362 . "뛸") (37854 . "큰 쇠북 용") (33592 . "풀 뾰죽뾰죽 날 용, 못생길 용, 녹용 용")
                 (37780 . "불릴 용, 녹일 용, 거푸집 용") (20653 . "머슴 용, 지을 용, 삯 용")
                 (28342 . "물 질펀히 흐를 용, 녹일 용, 녹을 용") (29794 . "패옥소리 용")
                 (29076 . "鎔의 俗字, 거푸집 용") (32883 . "솟을 용, 귀먹을 용, 공경할 용")
                 (28263 . "涌의 俗字, 물솟을 용") (33993 . "연꽃 용, 약 이름 용")
                 (28044 . "물 넘칠 용, 날칠 용, 물솟을 용") (20887 . "宂과 同字, 한산할 용")
                 (22665 . "작은 성 용, 담벽 용, 골목길 용") (20433 . "시체와 함께 매장하는 허수아비 인형 용")
                 (29996 . "물 솟아 오를 용, 골목길 용, 쇠북 꼭지 용, 곡식 셈하는 단위 휘 용")
                 (24898 . "권할 용, 거들어 말할 용") (27029 . "용나무 용, 나무이름 용")
                 (22471 . "골목길 용, 길 돋울 용") (33282 . "방아 찧을 용, 쇠북소리 용, 해가 질 용")
                 (24949 . "게으를 용") (23426 . "번잡할 용, 바쁠 용, 떠돌아 다닐 용")
                 (36404 . "뛸 용") 20323 (24963 . "천치 용") 37144 (23921 . "봉우리 쭝굿쭝굿할 용")
                 16184 31593 29271 25679 29253 39670 30823 (34553 . "번데기 용")
                 36438 18567 32824 38229 38235 40108 40153 (20635 . "혁혁할 용, 몸 굽힐 용")
                 (27110 . "병기 얹는 시렁 용") 20615 21192 22606 23222
                 23238 23262 24430 24703 24704 24849 24889 25608
                 25617 27061 27652 28413 30781 31297 31299 32281
                 33498 35107 36597 39890 39979 40005 40234 40411
                 13359 14042 14160 14334 14545 14774 14775 15082
                 15348 15517 15528 15794 16014 16702 16832 16879
                 16950 17892 18363 18460 18502 18533 18566 18698
                 18913 19064))
       (50864 . ((21491 . "오른쪽 우, 높일 우, 강할 우, 도울 우")
                 (29275 . "소 우, 별 이름 우") (38632 . "비 우") (21451 . "벗 우, 합할 우")
                 (21448 . "또 우, 용서할 우") (23431 . "집 우, 처마 기슭 우, 하늘 우")
                 (20110 . "갈 우, 여기 우, 만족할 우, 탄식의 어조사 우")
                 (23588 . "가장 우, 더욱 우, 탓할 우") (24962 . "근심 우, 상제될 우, 병 우")
                 (36935 . "만날 우, 대접할 우, 뜻밖에 우") (24858 . "어리석을 우, 어두울 우, 업신여길 우")
                 (20778 . "넉넉할 우, 부드러울 우, 나을 우, 결단성 없을 우")
                 (32701 . "깃 우, 오음의 하나 우") (37109 . "역말 우, 지날 우, 우편 우")
                 (20598 . "짝지울 우, 무리 우, 뜻밖에 우") (20305 . "도울 우")
                 (31056 . "도울 우, 다행할 우") (31161 . "하우씨 우, 성씨 우, 느릴 우")
                 (36802 . "굽을 우, 피할 우, 멀 우") (34398 . "염려할 우, 즐거울 우, 갖출 우, 우제 지낼 우")
                 (23507 . "살 우, 부탁할 우, 빙자할 우, 붙어살 우")
                 (38533 . "모퉁이 우, 기슭 우") (23886 . "산모퉁이")
                 (31121 . "복") (38633 . "기우제 우, 무지개 우") (32006 . "얽힐 우, 굽을 우")
                 (29760 . "옥돌 우, 옥다음가는 돌 우") (30402 . "바리 우, 사냥하는 진 우, 밥그릇 우")
                 (26100 . "해 돋을 우, 새벽 우") 20112 (33419 . "토란 우")
                 (34261 . "연뿌리 우") (29591 . "옥돌 우, 우기나무 우")
                 (37354 . "요령 우, 바리때 우, 풍류그릇 우") 21505
                 20111 22563 (32806 . "쟁기 우, 겨리 우, 짝 맞을 우")
                 (36413 . "타달거릴 우") 30449 (21889 . "서로 부를 우")
                 (31485 . "큰 생황 우") 35343 20417 36803 22313 31355
                 30115 (32816 . "고무래 우, 곰방메 우") 14178 31162
                 (40576 . "암사슴 우") 22648 27340 33475 22168 (40588 . "숫사슴 우")
                 25070 35367 13886 13989 16497 24253 26437 (37725 . "귀고리 우")
                 22131 25180 26944 28672 32395 32940 33122 14609
                 15826 17093 18974 21497 20451 20248 24551 24580
                 25200 29369 32417 37038 40063 40843 20148 20554
                 (21565 . "개 싸울 우, 개 짖는 소리 우") (35619 . "망령되게 말할 우")
                 (40818 . "충치 우") 21528 23168 23775 24611 24938
                 25181 25383 26713 27353 27510 27772 27787 28257
                 28574 29276 29558 30419 31076 32738 33420 33634
                 33837 34117 34422 34919 35366 37016 37125 37182
                 37201 37826 38483 39360 39596 39619 39799 39875
                 40454 40821 13441 13462 13659 13681 13905 13910
                 13997 14204 14448 14673 14693 14745 14976 15106
                 15434 15550 15571 16046 16102 16180 16680 16816
                 17009 17080 17154 17172 17218 17564 18084 18419
                 18579 18625 18687 18803 18816 18872 18878 18962
                 18965 19026 19069 19713))
       (50865 . ((26093 . "빛날 욱, 햇살 치밀 욱, 교만할 욱")
                 (37057 . "문채날 욱, 자욱할 욱, 향냐날 욱") (29020 . "비칠 욱, 불꽃 욱, 성할 욱, 빛날 욱")
                 (38922 . "머리 굽실거릴 욱, 별이름 욱, 사람이름 욱")
                 (26161 . "햇빛 밝을 욱") (21206 . "힘쓸 욱") 28595 (24423 . "빛날 욱, 문채날 욱, 무성할 욱")
                 (26671 . "산앵두 욱") (31286 . "서직 무성할 욱, 서속이 우거진 모양 욱")
                 (29152 . "더울 욱") 21207 38569 18699 34177 31266
                 15243 39036 21904 26094 26370 30983 14626 14781
                 15410 16987 18209))
       (50868 . ((38642 . "구름 운, 팔대손 운, 하늘 운")
                 (36939 . "움직일 운, 운수 운, 운전할 운") (20113 . "이를 운, 움직일 운, 돌아갈 운")
                 (38907 . "울림 운, 소리 맞을 운, 운 운, 운치 운")
                 (33464 . "향풀 운, 궁궁이 운, 촘촘할 운, 성씨 운, 재주 운")
                 (26248 . "무리 운, 어지러울 운") (38549 . "떨어질 운, 곤란할 운")
                 (27550 . "죽을 운, 떨어질 운") (32792 . "길 맬 운, 김맬 운")
                 (34131 . "운풀 운, 평지 운, 향내나는 풀 운") (28560 . "큰 물결 운")
                 (27218 . "나무 무늬 운") (29065 . "누른 빛 운") (32028 . "얼크러질 운")
                 (38901 . "韻과 同字") (24818 . "의논할 운, 혼후할 운")
                 (27780 . "물 콸콸 흐를 운, 끓을 운") 39211 37126
                 (38691 . "떨어질 운, 천둥에 비 몰아올 운") (31700 . "왕대나무 운")
                 37142 28339 22845 24874 25230 32284 38871 19061
                 27538 24701 31612 32429 36816 37075 37095 38504
                 21911 22249 22936 26527 28066 30467 30930 31184
                 32359 32826 33130 33975 34681 36081 40819 13955
                 14051 14095 14310 14920 16082 16812 16854 16985
                 18059 18347 18613 18998 19167 19632))
       (50872 . ((39729 . "향기로운 풀 울, 무성할 울") (34074 . "고을이름 울")
                 (20112 . "땅이름") 27421 15191 29096 33728 39728
                 29225 40678 16460 27406 28778 13339 14376 17799
                 19813))
       (50885 . ((38596 . "수컷 웅, 영웅 웅, 웅장할 웅")
                 (29066 . "곰 웅, 빛날 웅") 18938 20684 14258 15857))
       (50896 . ((20803 . "으뜸 원, 착하고 어질 원, 일 년의 첫 날 원, 임금 원, 백성 원, 하늘 원")
                 (21407 . "근본 원, 거듭 원, 언덕 원, 기인할 원, 들판 원")
                 (36960 . "멀 원, 심오할 원") (22290 . "동산 원, 능 원, 울타리 원")
                 (39000 . "하고자할 원, 바랄 원, 생각할 원, 원할 원")
                 (22291 . "둥글 원, 원만할 원, 화폐의 단위 원") (24616 . "원망할 원, 원수 원, 분낼 원")
                 (38498 . "원집 원, 공해 원, 절 원, 학교 원") (21729 . "관원 원, 둥글 원")
                 (28304 . "근원 원, 샘 원, 계속할 원") (25588 . "당길 원, 사다리 원, 구원할 원")
                 (33489 . "나랏동산 원, 큰 바람 원, 나라동산 원")
                 (29783 . "도리옥 원, 구멍 큰 둥근 옥 원") (34945 . "옷 치렁거릴 원, 성씨 원, 옷긴모양 원")
                 (23195 . "아리따운 여자 원, 여자 벼슬이름 원, 예쁜계집 원")
                 (23491 . "원통할 원, 굽힐 원, 죄 없을 원") (29503 . "원숭이 원, 잔나비 원")
                 (40219 . "숫원앙새 원, 수원앙새 원") (38446 . "원나라 원, 성씨 원, 관이름 원")
                 (36677 . "진문 원, 멍에 원, 별 이름 원, 멍에채 원")
                 (22435 . "낮은 담 원, 보호하는 사람 원, 별 이름 원")
                 (29232 . "당길 원, 이끌 원, 가로되 원") (24895 . "정성 원, 착할 원, 삼갈 원")
                 29740 (28274 . "물소리 원, 물 졸졸 흐를 원") (27781 . "물 이름 원")
                 (23236 . "여자이름 원") (27961 . "흐를 원, 물 이름 원")
                 20870 (20900 . "원통할 원, 원한 원") 35999 22300
                 36922 (40311 . "원추새 원") (34623 . "굼틀거릴 원")
                 30073 (40703 . "큰 자라 원") 26236 (35612 . "천천히 말할 원, 기꺼울 원")
                 22286 27038 (29480 . "猿과 同字") 14634 14707 19638
                 (26476 . "몸 주무를 원, 나무이름 원") 22808 30498
                 36384 (26981 . "떡갈나무 원, 실감개 원, 신골 원")
                 22807 28306 33554 34227 37005 40354 19018 19630
                 21592 22278 35816 36128 36757 36828 40499 40531
                 40715 17621 32169 (33451 . "고기 잡는 풀 원, 여뀌 원")
                 (34199 . "園과 同字") (37626 . "저울 바탕 원") (39477 . "배 희고 갈기 검은 말 원")
                 20487 20614 20618 21409 21429 22636 22964 23220
                 24780 26378 27052 29506 30454 31120 31502 31650
                 33918 33949 33964 34454 34671 34696 34895 35089
                 35108 35554 35954 37031 37211 37416 37809 38375
                 38992 39372 39789 40368 13374 14184 14322 14326
                 14398 14672 14892 15847 15945 16220 16712 16928
                 17357 17521 17730 17824 17828 17877 17980 18116
                 18167 18525 18761 19049 19117 19239 19651 19666
                 19752))
       (50900 . ((26376 . "달 월") (36234 . "넘을 월, 뛸 월, 월나라 월")
                 (37470 . "도끼 월") (31908 . "말 내킬 월, 곰곰히 생각할 월, 나라이름 월")
                 (21014 . "발꿈치 자를 월") 31925 27198 36559 25097
                 25224 29400 38074 38053 20212 23684 24628 26498
                 27879 34446 34447 36288 37381 14119 14269 15561
                 16692 17104 17331 18354 18400 18503 18561 18726
                 19202))
       (50948 . ((20301 . "벼슬 위, 임금의 신분 위, 자리 위, 방위 위")
                 (21361 . "위태할 위, 무너질 위, 상할 위, 높을 위, 별 이름 위")
                 (29234 . "할 위, 다스릴 위, 하여금 위, 어미원숭이 위, 이룰 위")
                 (23041 . "위엄 위, 두려울 위") (20553 . "클 위, 넉넉할 위, 기특할 위")
                 (35586 . "고할 위, 이를 위") (34907 . "衞와 同字, 호위할 위")
                 (22996 . "맡길 위, 버릴 위, 맘에 든든할 위, 벼이삭 고개 숙일 위")
                 (24944 . "위로할 위") (22285 . "둘레 위, 에워 쌀 위, 한아람 위, 지킬 위")
                 (20702 . "거짓 위, 속일 위") (32963 . "밥통 위, 별 이름 위, 성씨 위")
                 (36949 . "어길 위, 미적거릴 위, 되돌아올 위") (20525 . "순한 모양 위, 빙글 돌아올 위")
                 (32239 . "씨줄 위, 별 오른편으로 돌 위, 경위 위")
                 (23561 . "벼슬이름 위, 눌릴 위") (39759 . "위나라 위, 대궐 위, 우뚝할 위")
                 (34074 . "제비쑥 위, 초목 우거진 모양 위, 잔무늬 위")
                 (28205 . "물 이름 위, 속 끓일 위") (38859 . "다룬 가죽 위, 훌부들할 위, 군복 위, 성씨 위, 연할 위")
                 (33806 . "마를 위, 풀 이름 위, 쇠약해질 위") (29771 . "위옥 위, 노리개 위, 옥이름 위")
                 (33894 . "갈대 위, 작은 배 위") (34655 . "고슴도치 위")
                 (26256 . "햇빛 위, 빛 환할 위, 빛날 위") (35096 . "휘장 위, 아름다울 위")
                 (34111 . "풀 위, 성씨 위, 고을이름 위") 28858 (35529 . "핑계할 위, 버릴 위")
                 21890 (38344 . "대궐안의 문 위") (34910 . "막을 위, 지킬 위, 핏기운 위, 나라이름 위")
                 (36918 . "비틀거릴 위") (21919 . "한숨쉴 위") 29484
                 (38873 . "옳을 위") (30207 . "습병 위, 잘름거릴 위, 음위증 위")
                 (33907 . "초목 성할 위, 위유풀 위") 20605 25757 (29010 . "벌그레할 위, 환할 위")
                 (24131 . "향낭 위, 홑휘장 위") (29096 . "다리미 위, 고약 붙일 위")
                 (38881 . "꽃 환히 필 위") 13908 (39207 . "먹일 위")
                 34340 39595 28505 32635 34651 36503 13816 24491
                 28235 30794 39594 32261 34610 18865 23950 24839
                 34287 36507 38944 39593 17945 20026 20266 20255
                 21355 22258 22260 24079 28064 28828 29614 32428
                 33479 34950 35839 35859 36829 38385 38886 38890
                 20784 21470 21921 21940 22085 22101 22683 23169
                 23193 23206 23404 23530 23767 23843 26994 27912
                 28543 29023 29101 29338 29353 32237 33933 34358
                 34737 34785 35133 35235 35718 35727 36096 36370
                 37164 37711 38535 38696 38865 38874 39097 39221
                 39940 40154 13712 13902 14254 14314 14523 14566
                 14684 14755 15049 15129 15235 15817 16112 16935
                 17268 17271 18015 18918 19096 19216 19217 19315
                 19585 19751))
       (50976 . ((26377 . "있을 유, 얻을 유, 과연 유, 또 유")
                 (27969 . "흐를 류, 번져나갈 류, 구할 류, 내칠 류, 달아날 류, 무리 류, 귀양보낼 류")
                 (30001 . "말미암을 유, 쓸 유, 지날 유") (27833 . "기름 유, 구름 피어오를 유, 공손할 유")
                 (30041 . "머무를 류, 그칠 류, 오랠 류, 횡사할 류")
                 (29494 . "어미 원숭이 유, 같을 유, 오히려 유") (26611 . "버들 류, 별 이름 류")
                 (36986 . "잃어버릴 유, 남을 유, 끼칠 유") (36938 . "놀 유, 벗 사귈 유")
                 (37193 . "별 유, 열째 지지 유, 나아갈 유, 닭 유")
                 (21807 . "오직 유, 허락할 유") (24188 . "어릴 유, 사랑할 유")
                 (26580 . "부드러울 유, 순할 유, 복종할 유") (24799 . "꾀할 유, 오직 유, 생각할 유")
                 (39006 . "같을 류, 나눌 류, 착할 류, 무리 류") (35029 . "넉넉할 유, 너그러울 유")
                 (24736 . "멀 유, 생각할 유, 한가할 유, 근심할 유")
                 (35480 . "꾀일 유, 가르칠 유, 당길 유, 인도할 유")
                 (20083 . "젖 유, 종유석 유, 낳을 유, 젖먹일 유, 기를 유")
                 (20754 . "선비 유, 나장이 유, 유도 유, 광대 유")
                 (32173 . "벼리 유, 이을 유, 맺을 유, 오직 유") (24840 . "나을 유, 어질 유, 더욱 유, 병 나을 유")
                 (24189 . "숨을 유, 그윽할 유, 적을 유, 어두울 유, 귀신 유, 저승 유, 깊을 유")
                 (21129 . "이길 류, 죽일 류, 베풀 류, 성씨 류") (20842 . "그럴 유, 대답할 유, 공손할 유, 그러할 유")
                 (36400 . "넘을 유") (24254 . "노적 유, 곡식 셈하는 단위 휘 유")
                 (26977 . "느름나무 유, 서쪽 유") (32016 . "단추 뉴, 맺을 뉴")
                 (30290 . "병 나을 유, 병들 유") (35565 . "고할 유, 비유할 유")
                 (28216 . "헤엄칠 유, 노닐 유, 깃발 유") (23461 . "너그러울 유, 죄 사할 유, 도울 유")
                 (21929 . "깨우쳐 줄 유, 알릴 유, 비유할 유, 깨우칠 유")
                 (28316 . "처마물 류, 낙수물 류, 증류수 류") (35547 . "아첨할 유")
                 (37742 . "놋쇠 유") (29705 . "유리돌 류, 나라이름 류")
                 (24841 . "기뻐할 유, 즐거울 유") (26586 . "유자 유")
                 (36418 . "밟을 유, 쌀 씻을 유") (25540 . "당길 유, 칭찬할 유, 희롱할 유")
                 (25912 . "곳 유, 휙 달아날 유, 아득할 유, 대롱거릴 유")
                 (29495 . "꾀 유, 그릴 유, 옳을 유, 탄식할 유") (32143 . "깃발 늘어질 유")
                 (36926 . "갈 유, 지날 유") (28641 . "적실 유, 막힐 유, 곱고 윤택할 유")
                 (23418 . "젖먹이 유, 사모할 유, 딸릴 유") (33278 . "잠간 유, 착할 유, 나라이름 유, 활이 뒤로 젖혀질 유, 잠깐 유")
                 (29788 . "아름다운 옥 유") (20369 . "도울 유, 권할 유, 귀인을 모시고 밥먹을 유")
                 (30827 . "석류황 류, 유황 류") (26491 . "싸리 뉴") 32327
                 (33848 . "수유 유, 약이름 유") (27943 . "물이름 유")
                 (26978 . "부드러운 나무 유, 홰 유") (37321 . "물건 빛날 유")
                 (31858 . "부르짖을 유") (24119 . "휘장 유") 35174 (29270 . "엇살창 유, 들창 유, 밝을 유, 인도할 유")
                 (22749 . "토담 유, 단 둘러쌓은 담 유") (22271 . "엔담 유, 고루할 유")
                 20446 (33140 . "아랫배 살찔 유") (36876 . "빙그레할 유")
                 35238 21947 (33696 . "가라지 유, 추할 유") 22149
                 23899 (31941 . "섞일 유, 얽힐 유, 잡곡밥 유") 13532
                 (25545 . "풀 유, 부드럽게 할 유, 희일 유") (34133 . "냄새나는 풀 유, 유풀 유")
                 (34148 . "더북할 유") 35086 36662 (31404 . "판장문 유")
                 (40669 . "검푸를 유, 검은칠할 유") 34659 35087 23166
                 26328 34620 15610 (30224 . "죄인 주려 죽을 유") 32204
                 27999 (31411 . "이지러질 유, 약할 유, 짐승이름 유")
                 16212 18133 25833 27121 32657 (35721 . "성낼 유")
                 27641 29398 34231 36912 (21606 . "사슴 우는 소리 유, 새짐승 우는 소리 유")
                 28656 38531 (39850 . "상어 유") 23214 28410 16229
                 34837 (35093 . "꿩의 깃으로 꾸민 옷 유, 쾌자 유")
                 21347 30159 (27857 . "물빛이 검을 유, 물이름 유")
                 23121 27002 28202 29988 16004 17969 (20888 . "머뭇거릴 유, 갈 유")
                 (34484 . "굼틀거릴 유") (34660 . "하루살이 유") (40748 . "청서 유, 족제비 유")
                 23859 24622 27014 27464 29027 30574 31198 32685
                 33132 33863 36257 36371 36623 36854 37305 37438
                 37546 37714 39461 14682 15976 16286 28493 32500
                 33720 35278 35825 35840 35861 36951 38080 38101
                 39077 40073 40084 (26047 . "깃발 유, 깃술 유, 면류관술 유")
                 (30217 . "癒와 同字") (34480 . "노래기 유, 땅지네 유")
                 (38819 . "다룬 가죽 유, 마른 가죽 유") (40869 . "부르짖을 유")
                 20003 20580 21292 21760 22572 23031 23171 23208
                 23340 23399 23513 23728 23795 24238 25261 26004
                 26014 26741 26756 26992 27390 27642 28184 28277
                 28667 28706 29176 29296 29727 29768 29895 30481
                 31113 (31160 . "발자국") 32656 32840 33052 33429
                 33584 33661 33838 33935 34038 34125 34575 34643
                 34650 34724 34744 35969 35984 36654 37006 37123
                 37229 37248 37968 38870 39020 39455 39819 39943
                 40044 40340 13345 13365 13473 13617 13629 13655
                 13760 14044 14415 14584 14827 15053 15103 15152
                 15562 15579 15606 15639 15755 15769 15789 15960
                 16032 16064 16101 16140 16182 16213 16385 16523
                 16587 16718 16721 16727 16796 17427 17482 17531
                 17588 17664 17917 18051 18171 18181 18237 18325
                 18341 18570 18591 18935 19057 19083 19185 19200
                 19220 19462 19501 19504 19522 19665 19787))
       (50977 . ((20845 . "여섯 륙, 나라이름 륙") (32905 . "고기 육, 몸 육, 살 육")
                 (32946 . "기를 육, 날 육, 자랄 육") (38520 . "뭍 륙, 길 륙, 어긋날 륙, 뛸 륙, 두터울 륙")
                 (25134 . "죽일 륙, 육시할 륙, 욕할 륙") (22537 . "기름진 땅 육")
                 (27603 . "기를 육, 어릴 육, 많을 육") 28143 (20773 . "팔 육, 길 육")
                 37669 21706 21815 21893 23437 26843 28980 33966
                 34331 34591 34988 36915 14531 14676 16089 17133
                 17979))
       (50980 . ((20523 . "인륜 륜, 무리 륜, 조리 륜, 의리 륜, 가릴 륜")
                 (36650 . "바퀴 륜, 땅 길이 륜, 우렁찰 륜, 서릴 륜, 둘레 륜, 수레바퀴 윤")
                 (23609 . "다스릴 윤, 바를 윤, 벼슬이름 윤, 믿을 윤")
                 (20801 . "미쁠 윤, 마땅할 윤, 옳게 여길 윤, 좇을 윤, 허락할 윤, 진실로 윤")
                 (28516 . "불을") (38287 . "윤달") (23833 . "곤륜산 륜, 나라이름 륜, 산이름 윤")
                 (32996 . "맏아들 윤, 익힐 윤, 이을 윤") (37399 . "창 윤")
                 (28138 . "빠질 륜, 물놀이칠 륜, 거느릴 륜") (36103 . "예블")
                 (21243 . "가지런할 윤, 적을 윤") 30023 (22891 . "물 깊을 윤, 물이 충충할 윤, 물이 층층한 윤")
                 (29607 . "귀막이 구슬 윤") 37235 38304 29377 34657
                 14567 26112 38445 28070 36191 38384 37574 27213
                 33206 33658 34611 39355 14344 16217 18718 18846
                 19123))
       (50984 . ((24459 . "법 률, 풍률 률, 지을 률, 저울질할 률")
                 (29575 . "헤아릴 률, 과녁 률, 홀 한껏 다릴 률") (26647 . "밤 률, 신주 재목 률, 단단할 률, 무서울 률, 공손할 률")
                 (24900 . "두려울 률, 쭈그릴 률, 슬플 률, 떨 률")
                 (32895 . "마침내 율, 지을 율, 오직 율, 스스로 율, 붓 율, 드디어 율")
                 33567 (28495 . "물 흐를 율") 18582 (30686 . "송곳질할 율, 상서구름 율, 궤휼할 율, 귀신이름 율")
                 27429 40229 22090 29135 32344 37513 39496 40010
                 40231 13333 13344 15336 17110 19195 19335))
       (50997 . ((38534 . "성할 륭, 높을 륭, 가운데 우뚝할 륭, 괘 이름 륭")
                 (34701 . "부드러울 융, 밝을 융, 융통할 융, 불길 위로 오를 융, 귀신이름 융")
                 (25102 . "병장기 융, 싸움수레 융, 클 융, 너 융, 도울 융, 서쪽 오랑캐 융")
                 (32104 . "삶은 실 융, 가는 베 융") (28700 . "물 깊을 륭, 물깊은 모양 융")
                 32674 28427 23040 15196 (29416 . "원숭이 융") 32924
                 39397 32466 26343 27623 33561 34702 17964))
       (51008 . ((37504 . "은 은, 돈 은") (24681 . "은혜 은, 신세 은, 사랑할 은")
                 (38577 . "숨을 은, 아낄 은, 속걱정할 은, 불쌍히 여길 은, 점칠 은, 의지할 은, 쌓을 은")
                 (27575 . "많을 은, 무리 은, 클 은, 가운데 은, 은나라 은, 융성할 은, 천둥소리 은")
                 (35518 . "화평할 은") (22432 . "언덕 은, 하늘 가장자리 은, 경계 은")
                 (22331 . "언덕 은") (24903 . "은근할 은, 공손할 은, 간절할 은")
                 (35348 . "시비할 은") 38560 38544 (21548 . "벙긋거릴 은, 들을 은, 입이 큰 모양 은")
                 31311 31283 (24982 . "물을 은, 공근할 은, 힘쓸 은, 서러워할 은")
                 22438 14617 27267 (22170 . "어리석을 은, 말다툼할 은, 불칙하게 말할 은")
                 (29434 . "뭇 개 짖는 소리 은") 14703 27389 (28341 . "물소리 은, 물이름 은")
                 (27324 . "집 마룻대 은, 대공 은") (30318 . "두드러기 은")
                 27903 24941 (22461 . "해감 은, 찌끼 은") (37150 . "땅이름 은")
                 28646 30948 15759 24983 25050 29054 29374 33937
                 30270 35354 38134 40834 (22273 . "물이름 은") (29666 . "옥돌 은")
                 (40791 . "잇몸 은, 말다툼할 은, 미워할 은") 40806
                 (20058 . "숨을") 21999 23758 23806 23998 25665 26022
                 27309 29452 33981 34335 35732 13318 13466 13686
                 13932 14437 14716 14761 15624 15964 15966 16752
                 16885 17189 17604 17808 17820 19000 19297 19750))
       (51012 . ((20057 . "새 을, 천간 이름 을, 굽힐 을, 생선의 창자 을, 모모 을")
                 22314 38023 (40166 . "제비 을") 37332 17330))
       (51020 . ((38899 . "소리 음, 편지 음, 음악 음") (39150 . "마실 음")
                 (38512 . "음기 음, 부인에게 예 가르칠 음, 몰래 음, 세월 음, 그늘 음")
                 (21535 . "읊을 음, 끙끙거릴 음, 탄식할 음, 턱 끄덕거릴 음, 노래할 음, 말 더듬을 음, 울 음")
                 (28139 . "방탕할 음, 간음할 음, 음란할 음, 넘칠 음, 적실 음, 오랠 음")
                 (34093 . "덮을 음, 가릴 음, 그늘 음") 38530 (21905 . "소리지를 음, 안간힘 쓸 음, 목 쉴 음")
                 (24852 . "조용할 음, 화평할 음") (38698 . "장마 음")
                 17731 (23839 . "멧부리 음, 산 높을 음") 39154 30230
                 23148 38688 38437 38674 33643 38452 39278 (24277 . "덮을 음")
                 20049 20364 22102 22142 23855 25727 28158 28878
                 30282 30866 31400 33474 33654 35361 36251 37405
                 38910 40419 13634 13719 14244 15427 15890 18111
                 18238 18691 18969 18983 19113))
       (51021 . ((37009 . "고을 읍, 답답할 읍") (27875 . "눈물 줄줄 흘릴 읍, 부글부글 끓는 소리 읍, 소리없이울 읍")
                 (25558 . "읍할 읍, 나아갈 읍, 깍지낄 읍, 공손할 읍")
                 (25401 . "잡아당길 읍, 잔질할 읍") (24722 . "답답할 읍, 근심할 읍")
                 (28005 . "젖을 읍, 흠치르르할 읍") 35035 28230 19266
                 20427 21768 27486 28231 13834 15454 15636 15723
                 16892 16983 17603 17713))
       (51025 . ((25033 . "응당 응, 꼭 응, 사랑할 응, 대답할 응, 응할 응")
                 (20957 . "엉길 응, 이룰 응, 열중할 응") (40441 . "매 응")
                 (33210 . "가슴 응, 친할 응, 당할 응, 응할 응, 받을 응")
                 35661 24212 24540 40560 22111 36552 13475 14585
                 15760 15773 18937))
       (51032 . ((34915 . "옷 의, 입을 의") (24847 . "뜻 의, 생각 의, 의리 의, 형세 의")
                 (32681 . "옳을 의, 의리 의, 뜻 의") (20381 . "의지할 의, 기댈 의, 비슷할 의, 따를 의, 비유할 의, 병풍 의")
                 (37291 . "의원 의, 병 고칠 의") (30691 . "말 그칠 의, 어조사 의")
                 (35696 . "말할 의, 의논할 의, 꾀할 의") (23452 . "옳을 의, 마땅할 의, 유순할 의")
                 (20736 . "꼴 의, 모양 의, 법도 의, 좋을 의") (30097 . "의심할 의, 두려워할 의, 그럴듯할 의")
                 (35516 . "옳을 의, 다스릴 의") (25836 . "의논할 의, 헤아릴 의, 흡사할 의, 비길 의")
                 (27589 . "굳셀 의, 발끈 성낼 의") (26885 . "가래나무 의, 교의 의")
                 (32266 . "목 맬 의") (20506 . "기댈 의, 의지할 의, 믿을 의")
                 (25087 . "아름다울 의, 클 의") (34811 . "왕개미 의, 검을 의, 술구더기 의")
                 (34191 . "연밥 의, 율무 의") (33380 . "배 닿을 의, 배언덕에 댈 의")
                 (37665 . "세발 가마 의, 쇠뇌틀 의, 휘뚝거릴 의")
                 21307 (29463 . "불 깐 개 의, 탄식할 의, 길 의") (23991 . "산이름 의")
                 (27449 . "아름답다할 의, 거룩하다할 의") 20041 (28458 . "물놀이칠 의")
                 20893 25158 38999 (20759 . "서로 못믿을 의, 참람할 의")
                 23440 27301 27593 31465 16129 (21139 . "코 베일 의, 베일 의")
                 31125 (34712 . "목 매어 죽는 벌레 의") 26070 29316
                 34239 14364 14756 (39248 . "밥 쉴 의") 35689 14936
                 18264 19722 20202 25311 31054 34433 35758 35850
                 38129 38257 39291 (20946 . "눈서리 쌓일 의") (30994 . "바위 의")
                 20100 20591 21530 23313 23327 23412 23538 23980
                 24225 25742 26065 26924 27321 29169 29387 30057
                 31270 31822 32667 33811 34265 34720 (34916 . "옷의")
                 35322 35706 35929 36642 36697 36888 37116 37303
                 37400 37541 37951 39009 39183 40446 40451 13559
                 13640 13650 13832 14020 14166 14358 14474 14667
                 14729 15050 15166 15429 15474 15574 16200 16296
                 16370 16989 17000 17598 17657 18269 18657 18887
                 18919 19120 19153 19183 19237 19314 19481 19502
                 19503 19601))
       (51060 . ((20108 . "두 이, 풍신 이, 같을 이, 의심할 이")
                 (32819 . "귀 이, 조자리 이, 말 그칠 이, 여덟대 손자 이")
                 (37324 . "마을 리, 근심할 리, 잇수 리") (21033 . "날카로울 리, 이로울 리, 탐할 리, 편리할 리, 이자 리")
                 (29702 . "다스릴 리, 바를 리, 누늬낼 리, 성품 리, 이치 리, 고칠 리, 나무결 리")
                 (30064 . "다를 이, 괴이할 이, 나눌 이") (20197 . "써 이, 쓸 이, 까닭 이, 함께 이, 거느릴 이")
                 (32780 . "순접의 어조사 이, 너 이, 말이을 이, 이에 이")
                 (26446 . "오얏 리, 선비 천거리할 리, 역말 리, 보따리 리")
                 (24050 . "이미 이, 그칠 이, 버릴 이, 너무 이") (31227 . "옮길 이, 모낼 이, 변할 이")
                 (26131 . "쉬울 이, 다스릴 이, 게으를 이, 편할 이, 바꿀 이")
                 (21519 . "아전 리, 관리 리") (38626 . "떠날 리, 베풀 리, 지날 리, 아름다울 리, 둘 리, 외손 리")
                 (23653 . "가죽신 리, 신을 리, 밟을 리") (35023 . "옷 안 리, 속 리")
                 (26792 . "배 리, 벌레이름 리") (20234 . "저 이, 이 이, 오직 이, 답답할 이, 성씨 이")
                 (22839 . "동쪽 오랑캐 이, 평평할 이, 기꺼울 이, 상할 이, 베풀 이, 괘 이름 이")
                 (27877 . "진흙 니, 물이 더러워질 니, 흙손 니, 술취할 니, 이슬 맺힐 니")
                 (29669 . "귀막이 옥 이, 귀고리 이, 해무리 이, 햇무리 이")
                 (24347 . "늦출 이, 활 부릴 이, 방탕할 이, 해이할 이")
                 (36019 . "二의 갖은자, 버금 이, 의심낼 이, 마음 변할 이, 이별할 이, 두 이")
                 (24609 . "기쁠 이, 즐거울 이, 화할 이") (32633 . "만날 리, 근심할 리, 걸릴 리")
                 (39180 . "미끼 이, 이깝 이, 먹일 이, 흰떡 이") (35041 . "裏의 俗字, 옷속 이")
                 (30157 . "상할 이, 다칠 이, 흠집 이") (29246 . "너 이, 가까울 이, 오직 이")
                 (30178 . "이질 리, 곱똥 리, 설사 리") (23016 . "이모 이, 처이모 이")
                 (36029 . "끼칠 이, 줄 이") (36999 . "가까울 이") (24411 . "彝의 俗字, 떳떳할 이")
                 (32900 . "익힐 이, 수고할 이, 휘추리 이, 곁가지 이")
                 (33505 . "율무 이, 질경이 이") (39156 . "엿 이, 맛 이")
                 (33617 . "흰 비름 이, 베일 이, 벨 이") (24413 . "떳떳할 이, 법 이, 종묘 제기 이, 술통 이")
                 (36836 . "든든할 이") 38921 38948 32673 23570 24322
                 (35345 . "자랑할 이") 22319 (35409 . "자랑할 이") 34238
                 36841 (27935 . "콧물 이, 눈물 이, 못이름 이") 20766
                 21276 36014 35410 36806 27953 30500 36608 (21695 . "선웃음칠 이, 책 읽는 소리 이")
                 26461 25953 30489 36004 36700 23571 26246 26562
                 39404 24412 27919 24414 26669 37199 38949 29638
                 32672 33506 34534 40671 (26670 . "버섯 이, 느타리 이")
                 20289 21045 23866 24333 25157 25914 27186 28521
                 30513 30514 32155 33017 34524 34888 35643 36272
                 36859 39669 13382 13669 14383 17708 (23572 . "爾와 同字")
                 (38582 . "밑 이, 미치다 이") 24336 35794 36144 36155
                 38098 39056 39284 39285 40085 40504 (39847 . "창자젓 이")
                 22376 20033 20287 20340 20359 20373 21665 21670
                 23184 23463 23759 23763 24056 24281 24364 24670
                 26545 26699 26936 27467 27540 27622 27938 29098
                 29391 29941 31510 31671 31747 31915 32847 32916
                 32995 33008 33611 34708 34922 34968 34994 35003
                 35456 36320 36849 37487 37498 38481 38573 38986
                 39838 40239 40333 13324 13413 13443 13703 14021
                 14027 14087 14510 14525 14535 14708 14782 14816
                 15253 15337 15384 15389 15419 15651 15979 16063
                 16162 16684 17113 17210 17311 17312 17334 17386
                 17695 18041 18114 18287 18309 18677 18885 19039
                 19201 19246 19338 19532))
       (51061 . ((30410 . "더할 익") (32764 . "날개 익") (32714 . "도울 익")
                 (28346 . "빠질 닉, 약할 닉, 헤어 나오지 못할 닉")
                 (32716 . "다음날 익") (21311 . "숨을 닉, 숨길 닉, 덮어둘 닉, 몰래 붙을 닉, 쌀 닉, 술 담는 그릇 닉")
                 (35610 . "웃을 익, 웃는모양 익") (28727 . "강이름 익, 스며흐를 익")
                 27023 (24331 . "주살 익, 취할 익, 홰 익, 검을 익, 벼슬이름 익, 오랑캐 익")
                 (40385 . "익새 익, 돛대 끝 바람개비 익") 37364 (29092 . "빛날 익, 사람이름 익")
                 26457 21964 33367 40659 16681 35877 40546 28019
                 29116 29512 31315 33161 33413 34529 38591 38700
                 39598 40824 13620 13988 14091 14494 17114 17211
                 17949 18663 19398 19756))
       (51064 . ((20154 . "사람 인") (20161 . "어질 인") (22240 . "인할 인")
                 (24341 . "끌 인, 이끌 인") (21360 . "도장 인") (23493 . "세째지지 인, 범 인")
                 (24525 . "참을 인") (35469 . "알 인") (38563 . "鄰의 俗字, 이웃 인")
                 (23035 . "혼인 인, 혼인할 인") (21693 . "목구멍 인, 목멜 인")
                 (40607 . "기린 인") (20995 . "칼날 인") (28270 . "잠길 인, 없어질 인")
                 (40023 . "비늘 인") (21533 . "아낄 인, 인색할 인") (29136 . "도깨비불 인, 반딧불 인")
                 (34451 . "지렁이 인") (38765 . "韌과 同字, 질긴 고기 인")
                 (22553 . "막을 인") (29848 . "옥빛 인") (33589 . "자리 인, 사철쑥 인")
                 (32106 . "기운 인, 수삼 인") (38775 . "가슴걸이 인")
                 (34298 . "골풀 인, 뇌양이 인") (20190 . "길 인") (31115 . "제사지낼 인")
                 (22820 . "조심할 인") 35346 36564 (20799 . "어진사람")
                 (27684 . "기운성할 인") 22233 (35008 . "요 인") 32009
                 38345 25133 29283 (23139 . "姻과 同字") 35570 38860
                 37844 39408 16644 22420 28250 37372 38523 14588
                 (27911 . "湮과 同字") 20205 27437 27461 29836 31555
                 34750 32427 35748 35761 36715 38111 38887 39555
                 (25192 . "그러하나 인") (31870 . "벼 인") (33442 . "씨 인")
                 (20155 . "사람인변") 20944 20996 22236 23675 23683
                 (24308 . "길게걷다") 24520 24526 26372 26418 26450
                 26773 27557 27925 28645 31170 31221 31884 32248
                 32917 33562 33653 34089 36533 37391 37542 38769
                 38791 39772 39843 13883 14169 14388 14792 14818
                 15014 15549 15934 16252 16404 16405 16588 16857
                 17264 17366 17392 17490 17569 17648 18915 19615))
       (51068 . ((19968 . "한 일") (26085 . "날 일") (36920 . "달아날 일, 편안할 일")
                 (22777 . "한 일, 하나 일") (37808 . "중량 일, 스물넉냥쭝 일")
                 (20350 . "춤 일, 줄춤 일") (28322 . "넘칠 일") (20314 . "편안할 일, 편할 일")
                 (39353 . "역말 일, 역마 일") (36604 . "앞지를 일") 37360
                 (27846 . "끊을 일") 30396 34933 22769 36726 38226
                 39538 21166 21625 22264 22785 23300 24332 27970
                 29146 36791 37412 40233 40423 15569 17572 19327
                 19648))
       (51076 . ((26519 . "수풀 림, 더북더북 날 림") (22764 . "아홉째천간 임")
                 (20219 . "맡길 임, 맡을 임") (33256 . "임할 림, 군림할 림, 여럿이 울 림, 굽힐 림")
                 (36035 . "품팔이 임, 빌릴 임") (23001 . "妊과 同字, 아이밸 임")
                 (28107 . "지적지적할 림, 축일 림, 번지르르할 림, 물댈 임")
                 (31252 . "곡식익을 임") (33615 . "들깨 임, 콩 임") (24641 . "생각할 임, 믿을 임")
                 (22922 . "아이밸 임") (34941 . "옷깃 임") 34997 (39146 . "익힐 임")
                 (32077 . "짤 임, 길쌈할 임") 39169 32029 (37515 . "젖을 임")
                 26863 33101 37395 15797 16658 20020 32436 36161
                 39274 26656 26659 31053 31225 33741 36576 40256
                 14619 16878 17109 17365 18110 18136 19267))
       (51077 . ((20837 . "들 입") (31435 . "설 입") (31520 . "우리 입, 삿갓 입")
                 (31890 . "알 입, 쌀알 입") (21316 . "스물 입") (24319 . "스물 입")
                 36796 22310 25174 26433 39774 40176))
       (51079 . (13793))
       (51081 . ((21097 . "남을 잉") (23381 . "아이밸 잉") (20173 . "인할 잉, 이전의 잉")
                 (33471 . "새풀싹 잉") (23221 . "보낼 잉") 21104 36088
                 31037 38526 18313 25172 33185 33407 36792 13406
                 14010 14220 15169 16049 16679 18094 19828))
       (51088 . ((23376 . "아들 자") (33258 . "스스로 자") (32773 . "놈 자")
                 (23383 . "글자 자") (22985 . "姊의 俗字, 손윗누이 자")
                 (24904 . "사랑할 자") (36039 . "재물 자") (23039 . "맵시 자")
                 (29574 . "이 자") (32043 . "자주빛 자, 검붉을 자") (24675 . "방자할 자")
                 (21050 . "찌를 자") (28363 . "불을 자") (30133 . "흠 자, 죽은깨 자")
                 (38604 . "암컷 자") (30913 . "자석 자") (28825 . "고기구을 자, 고기구이 자")
                 (35566 . "물을 자, 꾀 자") (34249 . "깔개 자") (29038 . "삶을 자, 지질 자")
                 (20180 . "자세할 자") (34071 . "사탕수수 자") (29943 . "오지그릇 자")
                 28856 (21672 . "물을 자, 원망할 자") (23388 . "힘쓸 자, 부지런할 자")
                 (33576 . "가시나무 자, 납가새 자") 34234 20857 (33586 . "무성할 자")
                 22986 (36018 . "재물 자") (30502 . "眥와 同字") 25211
                 (23414 . "부지런할 자") (35390 . "헐뜯을 자") (36205 . "붉은흙 자")
                 (31906 . "피 자") (39661 . "코밑수염 자") (35292 . "털뿔 자")
                 (27866 . "맑을 자") (26584 . "산뽕나무 자") 29009 36262
                 26431 32980 (32788 . "복돋을 자") (40403 . "자고 자")
                 (30501 . "흘길 자") 36241 (40383 . "가마우지 자") (29304 . "암컷 자")
                 39176 23411 23915 24219 35391 28588 (33022 . "고깃점 자")
                 14807 (21616 . "주차할 자") 27048 (37793 . "호미 자")
                 26445 21867 26661 31213 31285 15936 19821 (33727 . "풀가시 자")
                 (39827 . "젓 자") 21618 22997 27252 31531 39604 16653
                 17361 19652 40538 34461 35864 36160 36164 38156
                 38211 40074 40093 40551 (23336 . "계집성품너그러고순할 자")
                 (23382 . "쌍둥이 자") (33544 . "지치 자") 34424 34481
                 (38975 . "코밑수염 자") 21054 21511 21849 21982 22416
                 22614 23473 27164 27601 28290 28320 28648 29099
                 29628 29633 29958 30148 30711 31008 31116 31172
                 31222 31869 31949 32213 32387 32975 33427 33546
                 33569 34187 34515 34694 34757 34816 36220 36224
                 37137 37281 37421 37573 38974 39159 39162 39626
                 39866 39974 40220 40384 40722 40781 13393 13544
                 13730 13826 13854 13881 13975 14141 14399 14464
                 14645 14808 15013 15086 15395 15415 15863 16006
                 16045 16261 16307 16547 16773 16920 17150 17666
                 17693 17762 17834 17898 17913 17955 18043 18123
                 18454 18875 18931 18959 19070 19299 19486 19509
                 19647))
       (51089 . ((20316 . "지을 작") (26152 . "어제 작") (29237 . "벼슬 작")
                 (37196 . "따를 작, 참작할 작") (26451 . "구기 작") (28796 . "사를 작, 구을 작")
                 (38592 . "참새 작") (40306 . "까치 작") (32189 . "너그러울 작")
                 (21242 . "구기 작, 잔질할 작") (22204 . "씹을 작") (33421 . "작약 작")
                 (37259 . "술잔을 돌릴 작") (28856 . "터질 작, 기름에 지질 작")
                 (26027 . "벨 작, 쪼갤 작") (37218 . "초 작") (24589 . "부끄러울 작")
                 (28975 . "밝을 작") 22913 (30863 . "삼갈 작") 40415
                 (26590 . "나무이름 작") 23141 29213 30381 (23709 . "산 높을 작")
                 (27723 . "삶을 작") (29363 . "아롱 짐승 작") 22324 25178
                 31130 31600 19548 29454 32496 40522 (26033 . "쪼갤 작")
                 20194 22960 23710 28917 28979 29094 29201 30389
                 31208 31251 31345 32347 33522 33675 35638 36254
                 37436 37935 38733 39157 13381 13824 14640 14961
                 14997 15149 15922 15985 16757 16846 17103 17310
                 17328 18283 18338 18819 18943 19637))
       (51092 . ((27544 . "해칠 잔, 나머지 잔") (30430 . "작은 술잔 잔")
                 (26855 . "잔교 잔, 사다리 잔") (23409 . "잔약할 잔")
                 (28538 . "물이 줄줄 흐르는 모양 잔, 물졸졸흐르는 모양 잔")
                 27531 (25108 . "쌓일 잔") (21079 . "깍을 잔") 37254
                 20701 21036 25099 26632 30415 39587 (39503 . "안장 얹지 않은 말 잔")
                 23960 25283 26719 28537 29718 33754 34405 34406
                 36634 36687 14302 14663 15245 15450 16117 17298
                 17404 17859 17985 18291 19014 19552 19772))
       (51096 . (22224 20093))
       (51104 . ((28507 . "잠길 잠") (26283 . "잠시 잠") (34870 . "누에 잠")
                 (31668 . "바늘 잠, 경계할 잠") (31786 . "비녀 잠") 28251
                 (23697 . "봉우리 잠, 높을 잠") (28508 . "潛의 俗字")
                 (34453 . "누에 잠") 36436 34360 22465 (28052 . "괸물 잠")
                 26787 31506 (28659 . "潛의 俗字") 31790 29112 31736
                 36090 25743 34874 25382 26242 33244 36186 20802
                 21906 23489 25565 25741 26141 30838 33250 34629
                 35687 36232 37909 13574 14245 14299 14533 16024
                 16120 16494 16609 16766 17276 17462 17885 17886
                 17929 18373 18655 18656 18704 19144 19285 19516
                 19595 19685))
       (51105 . ((38620 . "섞일 잡, 잡될 잡") 38296 21277
                 36810 (35149 . "雜의 本字") 38609 (30972 . "산 높을 잡")
                 24064 38661 21841 29024 21634 24065 26434 (21345 . "관 잡")
                 (22211 . "장단 잡을 잡") (30504 . "눈 깜작일 잡") 21652
                 22017 25272 25316 27806 30080 30776 35671 37460
                 38629 38900 14812 15105 17785 17988 18329 19007
                 19102 19837))
       (51107 . (13807))
       (51109 . ((38263 . "길 장") (22580 . "마당 장") (23559 . "장차 장, 장수 장")
                 (31456 . "문채 장, 글 장") (22767 . "씩씩할 장, 장할 장")
                 (29376 . "모양 형상 장") (24373 . "베풀 장") (25484 . "손바닥 장")
                 (19976 . "어른 장, 길 장") (33144 . "창자 장") (24115 . "휘장 장막 장")
                 (38556 . "막을 장, 막힐 장") (29518 . "권면할") (33900 . "장사 지낼 장")
                 (34255 . "감출 장") (33674 . "엄할 장, 씩씩할 장") (22715 . "牆과 同字, 담 장")
                 (35037 . "차려 입을 장, 행장 장") (31911 . "단장할 장")
                 (33247 . "오장 장") (21280 . "장인 장, 바치 장") (24196 . "장중할 장, 농막 장")
                 (29835 . "끝의 반을 깎아 뾰족하게 한 홀 장, 서옥 장")
                 (34083 . "줄 장, 과장풀 장") (27167 . "녹나무 장, 노나무 장")
                 (26454 . "지팡이 장") (20183 . "병장기 장, 의지할 장")
                 (29520 . "노루 장") (37292 . "장 장, 간장 장") (28479 . "미음 장, 초 장")
                 (27299 . "돛대 장") (34196 . "물여뀌 장, 장미꽃 장")
                 (22892 . "도울 장, 클 장") (33255 . "착할 장, 거둘 장")
                 (36115 . "장물 장") (29254 . "담 장") (27404 . "장농 장, 장롱 장")
                 (26290 . "햇발이 돋아올 장, 밝을 장") 23558 22766
                 22681 34101 (35013 . "裝의 俗字") 33624 (25109 . "죽일 장")
                 33171 (30260 . "장기 장") (23938 . "높고 가파른 산 장")
                 22888 (37848 . "금옥 소리 장") 33235 36124 (31962 . "妝과 同字")
                 22642 (28467 . "강이름 장") (33799 . "나무 이름 장")
                 (22872 . "클 장") 39634 39380 (36076 . "치부책 장")
                 (40606 . "노루 장") 27123 (29250 . "암양 장") 24926
                 23944 34751 (22941 . "꾸밀 장") 25177 26024 31931
                 27137 (23321 . "궁녀 장") (24295 . "담 장") 27238 (37155 . "나라이름 장")
                 34336 37284 25706 36045 36449 22870 33039 36163
                 22330 22918 23281 24080 24352 26728 26729 27183
                 27974 32928 33484 33931 34103 34688 36134 37233
                 38197 38271 39541 40137 (20669 . "놀랄 장") (32664 . "수양 장")
                 (39206 . "산자 장") 20012 20169 20199 20815 21278
                 21288 22623 22663 22671 22773 23076 23260 24155
                 24165 24329 26761 28939 (29247 . "나무조각") 29252
                 29946 30190 30252 30613 30979 31872 32585 33378
                 34049 34769 36056 36540 36967 37562 37873 37923
                 38006 38264 38813 39487 40002 40006 13871 13898
                 13923 14023 14483 14497 15260 15309 15732 15763
                 16187 16244 16245 16963 17035 17058 17255 17261
                 17538 17861 18155 18346 18438 19444 19777))
       (51116 . ((22312 . "있을 재, 살 재, 살필 재, 곳 재")
                 (25165 . "재주 재, 능할 재, 현인 재, 바탕 재") (26448 . "재목 재, 늘미 재, 성품 재, 재주 재")
                 (20877 . "두 번 재, 거듭 재") (36001 . "재물 재, 뇌물 재, 보배 재")
                 (21705 . "비로소 재, 답다 재, 그런가 재, 어조사 재")
                 (26685 . "심을 재") (36617 . "실을 재, 이길 재, 비롯할 재, 가득할 재")
                 (28797 . "재앙 재, 천벌 재, 횡액 재") (35009 . "마름질할 재, 헤아릴 재, 결단할 재, 옷마를 재")
                 (23472 . "주관할 재, 다스릴 재, 재상 재, 으뜸 재, 삶을 재")
                 (28371 . "앙금 재, 찌끼 재") (40779 . "재계할 재, 집 재")
                 (26771 . "가래나무 재, 목수 재, 우러러 볼 재, 고향 재, 노나무 재")
                 (32289 . "일 재, 실을 재") (28221 . "맑을 재") (40782 . "탄식할 재, 쌀 재, 가질 재")
                 (28798 . "災와 同字") (32404 . "겨우 재, 비롯할 재")
                 36075 14500 33745 27039 18292 36087 14770 25434
                 28886 36083 14771 36173 25995 36130 36733 40839
                 22792 25175 23869 20386 20644 20750 21901 25998
                 27909 28328 30581 37224 37358 37465 40796 13490
                 13773 13885 14835 14943 15024 15472 17225 17345
                 19241 19368 19757 19815))
       (51137 . ((29229 . "다툴 쟁, 다스릴 쟁, 분별할 쟁")
                 (37658 . "쇳소리 쟁그렁거릴 쟁, 징 쟁") (35533 . "간할 쟁")
                 (31631 . "쟁 쟁, 풍경 쟁") 20105 (23842 . "가파를 쟁, 산 높을 쟁")
                 (37783 . "금석소리 쟁, 창 쟁, 총 쟁") 23781 31581
                 (30624 . "똑바로 볼 쟁") 37946 25473 23837 (29732 . "옥소리 쟁")
                 25497 17131 25379 29424 35812 38126 (29465 . "밉살스럴 쟁, 짐승이름 쟁")
                 22092 23841 26854 30848 36255 37875 37887 39251
                 39687 13516 15161 17269 18411 18843 18870 19554))
       (51200 . ((20302 . "낮을 저, 값 쌀 저, 소리 약할 저, 숙일 저, 머뭇거릴 저")
                 (36015 . "쌓을 저, 감출 저, 둘 저") 35576 (33879 . "나타날 저, 널리 알려질 저, 글 지을 저, 옹저 저")
                 (24213 . "밑 저, 그칠 저") (25269 . "밀칠 저, 당할 저, 막을 저, 다달을 저, 씨름할 저")
                 (27822 . "그칠 저, 막을 저, 무너질 저, 공갈할 저, 물이 번질 저, 나라이름 저")
                 (36487 . "머뭇거릴 저") (29482 . "豬의 俗字, 돼지 저")
                 (37048 . "사처 저, 주막 저, 홀바탕 저, 병풍 저, 집 저")
                 (21632 . "씹을 저, 깨달을 저, 먹을 저") (31672 . "젓가락 저")
                 (35419 . "저주할") (29401 . "원숭이 저, 엿볼 저, 살필 저")
                 (35293 . "찌를 저, 받을 저, 씨름 저") (33511 . "모시 저")
                 (20786 . "쌓을 저, 버금 저, 용납할 저") (26990 . "닥나무 저, 종이돈 저, 감매나무 저")
                 (20295 . "오래 설 저, 기다릴 저, 잠깐 머물 저, 우두커니설 저")
                 (36889 . "맞을 저, 이것 저, 여기 저") (28186 . "물가 저, 물 갈라질 저")
                 (32053 . "모시 저") (26485 . "공이 저, 방망이 저, 혜성 저, 망치 저")
                 (30141 . "등창 저") (38606 . "저구 저, 증경이 저, 원앙 저")
                 (27159 . "가죽나무 저") (33785 . "김치 저, 푸서리 저")
                 (22992 . "맏누이 저, 아가씨 저, 교만할 저") (34295 . "마 저, 감자 저")
                 (40799 . "이 어긋날 저") (35398 . "꾸짖을 저, 알소할 저, 비방할 저")
                 (27664 . "낮을 저, 오랑캐 이름 저, 이를 저, 근본 저, 별이름 저")
                 20281 (23425 . "조회 받는 곳 저, 멈출 저") (26594 . "뿌리 저")
                 31599 (28710 . "물 괼 저, 못 저, 웅덩이 저") (33524 . "암삼 저, 꾸러미 저, 대지팡이 저, 신바닥 저")
                 27870 33861 36228 (26492 . "북 저, 도토리 저") (34971 . "속적삼 저")
                 32741 (28532 . "瀦와 同字") (29300 . "찌를 저, 대강 저, 씨름 저, 당할 저")
                 31450 28353 (34502 . "구더기 저, 지네 저") (32669 . "숫양 저")
                 (35098 . "주머니 저, 쌓을 저, 옷에 솜 둘 저") (35948 . "돼지 저, 물이 괼 저, 못이름 저")
                 (32605 . "짐승 그물 저") 16628 (23720 . "돌산에 흙 덮일 저")
                 33447 30497 27367 31720 35421 34963 24602 25263
                 25716 30135 30752 31111 14451 15069 21607 24356
                 30119 33547 14614 20267 20648 26592 27104 27237
                 32443 33486 35781 35787 36142 36825 40835 (38524 . "모래섬 저, 둔덕 저, 언덕 저")
                 20222 21167 21800 22373 22398 22712 22851 22905
                 23182 24197 25310 25483 27361 27371 28624 30493
                 32668 32860 33767 34127 34296 34667 34857 35936
                 36230 36313 36583 38458 39606 39923 40225 13391
                 13555 14153 14457 14982 15042 15212 15229 15305
                 15725 15750 15894 16315 16382 16586 16865 16908
                 17072 17222 17237 17309 17379 17431 17924 17954
                 18471 18532 18616 18636 18679 19126 19233 19286
                 19529))
       (51201 . ((36196 . "붉을 적, 금치 적") (30340 . "과녁 적, 꼭 그러할 적, 밝을 적")
                 (25973 . "대적할 적, 원수 적, 무리 적, 막을 적")
                 (36969 . "맞갖을 적, 편안할 적, 깨달을 적, 시집갈 적, 좇을 적, 주장할 적, 마침 적")
                 (36042 . "도적 적, 해칠 적, 도둑 적") (31309 . "포갤 적, 모을 적, 넓이 적, 부피 적, 쌓을 적")
                 (31821 . "문서 적, 왁자할 적, 호적 적, 압수할 적, 임금이 가는 밭 적")
                 (36321 . "자취 적") (32318 . "길쌈 적, 공 적, 이룰 적, 이을 적")
                 (25688 . "딸 적, 돋구어낼 적, 움직일 적") (23490 . "고요할 적, 쓸쓸할 적")
                 (28404 . "물방울 적, 물 댈 적") (36447 . "행적 적, 자취 적, 사적 적")
                 (31515 . "피리 적, 날라리 적") (28825 . "고기구이 적, 김 쏘일 적, 냄새 퍼질 적")
                 (36857 . "발자국 적, 자취 적, 뒤따를 적") (23265 . "정실 적, 맏아들 적")
                 (34249 . "어수선할 적, 확자할 적, 임금이 가는 밭 적")
                 (35627 . "귀양갈 적, 꾸짖을 적") (29380 . "북방 오랑캐 적, 꿩 그린 옷 적, 멀 적")
                 (21514 . "슬퍼할") (36842 . "나아갈 적, 순할 적, 열어줄 적")
                 (21219 . "공적 적, 사업 적") (32735 . "꿩의 깃 적, 황후의 옷 적")
                 (37841 . "화살촉 적") (33659 . "갈대 적, 쑥 적") 36374
                 21831 (31988 . "쌀 사들일 적, 빠를 적") (36886 . "멀 적")
                 (35263 . "볼 적") 24152 31732 38766 24312 (30951 . "돌무더기 적, 모래벌 적")
                 38932 33494 36919 32804 31818 25855 35713 (27136 . "추녀 적, 도토마리 적, 방망이 적")
                 22673 18417 22145 36271 15365 16019 24244 27149
                 27474 29963 30835 35136 40464 31215 30875 31860
                 32489 35276 35882 36156 38237 (33730 . "연밥 적")
                 (39344 . "별배기 적, 소태성 적") 21785 21874 22016
                 24301 25117 26099 26769 28030 28419 28795 28926
                 29109 29587 29899 30314 32891 32913 33859 34059
                 34064 34273 34824 35956 36516 39999 13440 14339
                 14505 14553 15193 15318 15949 16484 17054 17070
                 18061 18064 18608 18658 18738 18944 18980 19376
                 19452 19513 19693 19742 19778 19808))
       (51204 . ((30000 . "밭 전, 사냥할 전, 논 전") (21069 . "앞 전, 옛 전, 인도할 전")
                 (20840 . "온전할 전, 갖출 전") (38651 . "번개 전, 전기 전")
                 (25136 . "싸움 전, 경쟁할 전") (23637 . "펼칠 전, 열릴 전, 살필 전")
                 (20659 . "전할 전, 펼 전, 이을 전, 주막 전, 책 전")
                 (20856 . "책 전, 법 전, 맡을 전, 도덕 전") (37666 . "돈 전, 가래 전")
                 (23560 . "오로지 전, 전일할 전") (36681 . "구를 전, 변할 전, 굴릴 전")
                 (27583 . "대궐 전") (39002 . "이마 전, 엎드러질 전")
                 (30008 . "경기 전") (24731 . "고칠 전, 차례 전") (22880 . "정할 전")
                 (31661 . "화살 전, 약이름 전") (37523 . "사람 가릴 전, 저울질할 전")
                 (31627 . "기록할 전, 글 전") (22625 . "메일 전, 북소리 전, 순종할 전, 병들 전, 막힐 전")
                 (39198 . "전별잔치 전, 보낼 전") (31686 . "문체이름 전, 전자 전")
                 (32399 . "둘릴 전, 묶을 전, 얽을 전") (21098 . "가위 전")
                 (29006 . "달일 전, 불에 말릴 전, 다릴 전") (24283 . "전방 전, 터전 전")
                 (27656 . "단자리 전, 전방석 전") (36670 . "돌아 누울 전")
                 (39019 . "사지 떨릴 전, 머리 비뚤이 전") (28593 . "찌끼 전, 물 고일 전")
                 (30322 . "미칠 전, 지랄병 전") (26643 . "말뚝 전, 나무못 전")
                 (37931 . "새길 전") (35438 . "평론할 전, 갖출 전") 38603
                 (20346 . "신선이름") (20291 . "밭 맬 전, 머슴꾼 전, 사냥할 전, 밭갈 전")
                 (37439 . "비녀 전, 보배로 꾸민 그릇 전") (29728 . "옥이름")
                 28096 (31564 . "통발 전") (22652 . "벽돌 전") (30033 . "화전 전, 밭 전")
                 (38758 . "무안할 전, 물끄러미 볼 전") (29259 . "글 전, 표 전, 문체이름 전")
                 37549 28871 (30027 . "사냥할 전") (33094 . "두터울 전, 지극할 전, 많이 차려놓을 전")
                 31473 (38995 . "어리석을 전, 오로지 전") (32742 . "베어 없앨 전, 멸할 전")
                 25108 (29809 . "귀막이옥 전, 옥이름 전") (30154 . "병 나을 전")
                 (28244 . "빨 전, 씻을 전, 물 번질 전") 29777 32307
                 (26051 . "기 전, 말 그칠 전") (36500 . "햇길 전, 별자리 전")
                 (24019 . "산 꼭대기 전") 25126 36327 25132 31546
                 (32694 . "노린내날 전") (33603 . "향풀 전, 전풀 전")
                 38352 21112 (30938 . "벽돌 전, 기와 전") 22635 25129
                 (40455 . "구진매 전") 27658 (21550 . "빨 전") (29966 . "벽돌 전, 기와 전")
                 39256 (36997 . "머뭇거릴 전") 13897 (22208 . "새 지저귈 전, 후렴 전")
                 (39142 . "범벅 전, 된죽 전") 27215 20494 32402 33211
                 35710 39003 28127 (28598 . "물이름 전") 27834 (25539 . "가를 전, 자를 전, 가지런히 할 전")
                 25612 30248 32275 35629 (26676 . "향나무 전") 27097
                 28359 (23623 . "구멍 전") (39216 . "된죽 전") (40035 . "전어 전")
                 20622 28685 29303 27079 (37674 . "솥 전, 쇠 전") 25332
                 29820 30786 31412 31743 16883 (30048 . "밭 전") (31835 . "성씨 전")
                 (33182 . "썰 전, 질그릇 만드는 틀 전") (36615 . "달 전, 잴 전")
                 25655 26218 26392 29780 13651 15051 18514 35883
                 38220 19987 20253 20256 21869 24005 25112 27617
                 30005 30315 30742 32544 35277 35808 36578 36716
                 36737 36759 38065 38079 38120 38423 38589 39067
                 39072 39076 39279 40163 40559 19012 (23269 . "아름다울 전")
                 (30300 . "어루러기 전") (37181 . "가게 전") 37585 (38747 . "푸른 대 전")
                 (39691 . "수염 깎을 전, 귀밑 털 드릴 전") 20546 21415
                 21440 21818 22559 22722 22757 23038 23133 23152
                 23178 23400 23761 23918 24020 24116 24389 24686
                 25461 25846 25895 25951 26076 26295 26915 26923
                 26942 27024 27199 27364 27500 28281 29472 29776
                 29875 30198 30397 30455 30547 30858 30872 30924
                 31727 31947 32095 32254 32692 33056 33118 33223
                 33871 33893 33923 34119 34580 34788 35237 35539
                 35567 36063 36296 36429 36430 37151 37766 37816
                 37897 39330 39401 39450 39457 40004 40235 40390
                 40399 40827 13488 13497 13578 13947 14090 14239
                 14241 14381 14416 14470 14671 14766 14786 14901
                 15221 15285 15364 15388 15833 16071 16183 16428
                 16500 17273 17373 17420 17450 17638 17680 18402
                 18407 18436 18496 18520 18545 18641 18774 18883
                 18902 18904 18912 19013 19259 19571 19580 19690))
       (51208 . ((31680 . "마디 절, 절개 절, 대마디 절")
                 (32118 . "絕과 同字, 끊을 절") (20999 . "끊을 절, 저밀 절, 정성스럴 절, 급할 절, 진맥할 절")
                 (25240 . "꺾을 절, 자를 절, 휘일 절, 알맞을 절")
                 (31434 . "좀도둑 절, 얕을 절") (25130 . "끊을 절, 말 잘할 절")
                 (27993 . "쌀 씻을 절, 물이름 절") (30308 . "멍울 절, 부스럼 절")
                 (31363 . "竊의 俗字") 36604 (21353 . "병부") (26210 . "밝을 절, 비칠 절")
                 26211 (23690 . "산 모퉁이 우뚝 내밀 절") 34141 23995
                 32085 13630 32477 26817 30102 33410 21354 23568
                 24175 25767 25838 26867 26998 27791 32890 33478
                 34840 34846 34877 36227 38267 13321 13404 13571
                 13602 13803 14754 15039 15262 15877 15902 16448
                 17395 17606 17753 17915 17921 18393 19042 19132
                 19157 19609))
       (51216 . ((24215 . "가게 점, 주막 점") (40670 . "더러울 점, 뭉갤 점, 검은 점 점")
                 (21344 . "점칠 점, 날씨 볼 점, 점령할 점") (28472 . "번질 점, 물 흘러 들어갈 점, 점점 점, 차차 점")
                 (31896 . "붙일 점, 끈끈할 점") (38673 . "젖을 점, 비 지정거릴 점")
                 (28857 . "點의 略字, 검은 점 점") 25288 (23742 . "고개 점, 절이름 점")
                 (39822 . "메기 점") (29623 . "옥의 티 점, 흠각 점")
                 22379 35224 (20308 . "엿볼 점, 숙일 점") (31775 . "삿자리 점, 대 이름 점")
                 38461 (33515 . "이엉 점, 거적자리 점, 글씨 흘려 쓸 점")
                 (22666 . "빠질 점, 축축할 점") 30145 (40655 . "붙일 점, 풀 점")
                 (39085 . "펄렁거릴 점") (34090 . "풀 벨 점, 보리 팰 점")
                 25154 33959 15875 30969 40071 22443 28176 35271
                 39120 31512 34501 21816 22200 22860 24806 25474
                 29012 31205 32971 34745 34985 35479 36398 37865
                 38933 13536 14186 15768 15779 16173 16417 17220
                 17922 17931 18168 18379 18708 19015))
       (51217 . ((25509 . "사귈 접, 잇닿을 접, 대접할 접")
                 (34678 . "들나비 접") (28203 . "물결 출렁출렁할 접")
                 (25722 . "접을 접, 패할 접") 25586 (26986 . "널평상 접")
                 (39944 . "가자미 접, 넙치 접") (24948 . "두려울 접, 겁낼 접")
                 27293 (36416 . "저벅저벅 걸을 접") (36309 . "미끄러질 접")
                 33363 (26884 . "나무 접붙일 접, 형벌틀 접, 문설주 접")
                 35083 17425 40125 (34600 . "나비 접") 25365 25809
                 32849 33768 35189 13401 14478 17458 18243 18261
                 19091))
       (51221 . ((27491 . "바를 정, 과녁 정, 남쪽으로 난 창 정, 어른 정, 분별할 정")
                 (23450 . "정할 정, 편안할 정, 고요할 정") (19969 . "장정 정, 네째 천간 정, 고무래 정")
                 (25919 . "정사 정, 바르게 할 정, 조세 정") (24773 . "뜻 정, 마음속 정")
                 (24237 . "뜰 정, 곧을 정") (20572 . "머무를 정, 늦어질 정")
                 (31934 . "정교할 정, 세밀할 정, 정할 정, 성실할 정, 신령 정, 정기 정")
                 (38748 . "고요할 정, 조용할 정, 쉴 정, 편안할 정")
                 (35998 . "곧을 정, 굳을 정") (20117 . "우물 정, 단정할 정, 괘 이름 정, 천정 정, 저자 정")
                 (28136 . "맑을 정, 조촐할 정") (38914 . "이마 정, 꼭대기 정, 관 꼭대기 치장하는 장식 정")
                 (24311 . "조정 정, 바를 정, 곧을 정") (24449 . "칠 정, 갈 정")
                 (31243 . "법 정, 한정 정, 과정 정, 품수 정, 헤아릴 정, 길 정")
                 (20141 . "정자 정, 여관 정, 고를 정") (25972 . "가지런할 정, 정돈할 정")
                 (37165 . "나라이름 정, 성씨 정, 정나라 정") (35330 . "바로잡을 정, 의논할 정")
                 (26230 . "수정 정, 맑을 정, 빛날 정") (21576 . "보일 정, 드러낼 정")
                 (40718 . "솥 정, 바야흐로 정, 새로울 정") (26060 . "장목기 정, 표시할 정, 표할 정")
                 (25402 . "빼어날 정, 너그러울 정, 향초이름 정, 뺄 정")
                 (31118 . "상서로울 정") (20597 . "엿볼 정, 정탐할 정, 물을 정")
                 (27712 . "물가 정, 수렁 정") (26984 . "담틀 마구리대 정, 단단한 나무 정, 쥐똥나무 정")
                 (33351 . "거룻배 정, 작은 배 정") (29693 . "옥 이름 정, 뺄 정")
                 (38742 . "다스릴 정, 편안할 정, 생각할 정") (37336 . "창 정, 못 정")
                 (24128 . "그림족자 정") (37664 . "촛대 정, 덩이 정")
                 (30855 . "닻돌 정, 배 멈출 정") (31357 . "구덩이 정, 함정 정")
                 (30010 . "밭 두덕 정, 정보 정, 시가 정, 밭지경 정")
                 (37194 . "비틀거릴 정, 술취한 모양 정") (30555 . "눈 검은자위 정, 눈동자 정")
                 (28833 . "데칠 정, 불 번쩍거릴 정") (38662 . "벼락 정, 번개 정")
                 33729 (28191 . "물 고일 정, 도랑 정, 물가 정") (28096 . "배댈")
                 (26232 . "해돋는모양") (32142 . "띠술 정, 인끈 정")
                 (37478 . "징 정") (28702 . "맑을 정, 정결할 정") (29582 . "옥소리 정")
                 (35562 . "고를") 39894 (27273 . "능수버들 정, 수양버들 정")
                 (22979 . "계집단정할") (26622 . "나무결 바를 정, 나무바를 정")
                 (28254 . "물이름 정") (37580 . "화살촉 정, 달음박질할 정, 쇳덩이 정")
                 (35388 . "간할 정") 38745 20928 14158 38803 28783
                 27972 27871 (31603 . "대쪽 정") 15052 (38746 . "단장할 정, 밝을 정")
                 22501 (37234 . "술병 정, 술 깰 정") 38962 (26755 . "외줄기 정, 막대 정, 꼿꼿할 정")
                 30708 20409 28047 39139 20936 38449 (24596 . "황겁할 정, 가슴 두근두근할 정")
                 29716 36204 (21486 . "단단히 부탁할 정, 정성스러울 정")
                 (26838 . "문설주 정, 막대 정") (33691 . "풀 줄거리 정, 들보 정")
                 (23159 . "아리따울 정, 쌀쌀할 정") 20163 29732 38963
                 (29685 . "노리개 정, 패옥 정") 25537 35022 (30100 . "헌데 정")
                 37605 28918 33057 34579 24888 30556 35476 15742
                 (26735 . "서안 정, 걸상 정") 26061 26903 28622 31463
                 32821 40750 13867 15490 16647 16717 19834 35777
                 20390 24103 26536 26621 26722 27976 30529 31087
                 34511 35746 36126 36202 37073 38025 38066 38116
                 38147 38189 38739 39030 39051 39268 (20290 . "두려워할 정")
                 23143 (25503 . "둘러칠 정") (36937 . "순라군 정, 염탐할 정")
                 20028 20401 21864 21878 22306 22485 22505 22627
                 22901 22924 23063 23129 23196 23441 23498 23666
                 23881 23935 24068 24462 24496 24522 24732 25392
                 25576 26430 26645 26807 26975 27059 27256 27289
                 27755 27756 28650 30012 30447 30480 30536 30880
                 30901 30968 31467 31632 31997 32253 32787 32839
                 32857 32866 32868 32956 33069 33096 33114 33404
                 33827 33910 34209 34416 34634 34639 34806 36397
                 37010 37924 38318 38455 38743 38744 38762 38977
                 40202 40721 40753 13509 14432 14439 14543 14548
                 15040 15439 15679 15834 15990 16151 16192 16462
                 16571 16785 16821 16824 16888 16931 17042 17098
                 17148 17908 18323 18387 18832 18874 19040 19062
                 19413 19507 19539))
       (51228 . ((24351 . "아우 제") (31532 . "집 제, 차례 제")
                 (38988 . "이마 제, 제목 제") (35576 . "모든 제, 모두 제")
                 (31085 . "제사 제") (38500 . "섬돌 제, 덜 제") (24093 . "임금 제")
                 (35069 . "만들 제, 지을 제") (21046 . "억제할 제, 제도 제")
                 (28639 . "나루 제, 건널 제") (25552 . "끌 제, 들 제")
                 (38555 . "사이 제, 모을 제") (40778 . "가지런할 제")
                 (22564 . "둑 제, 방축 제") (21137 . "약 지을 제") (24716 . "공경할 제")
                 (36420 . "굽 제") (26799 . "사다리 제") (21884 . "울 제, 새지저귀는 소리 제")
                 (38717 . "갤 제, 비갤 제") (33229 . "배꼽 제") (34234 . "냉이 제")
                 (37261 . "맑은 술 제") (33617 . "띠싹 제") 40782 (20757 . "동배 제")
                 (36491 . "오를 제") 33488 21092 35078 (25824 . "물리칠 제")
                 28379 (38532 . "堤와 同字") (23075 . "여동생 제") (31124 . "복 제")
                 25993 38574 30974 30304 31335 34368 (30535 . "흘끗 볼 제")
                 (31242 . "강아지풀 제") (29765 . "제당 제") 31300 32136
                 40343 (39927 . "메기 제") (32249 . "붉은 비단 제") (36406 . "찰 제")
                 (29464 . "미친개 제") 22156 28123 20666 40284 28199
                 38830 (36431 . "蹄와 同字") 21953 25056 (38898 . "양념할 제")
                 20672 27890 28424 29886 20557 20569 24498 25685
                 33954 37959 39456 15100 16440 20394 21058 21724
                 25380 27982 28168 32488 32519 33040 33632 34548
                 35832 36347 38161 38469 38657 39064 40090 40128
                 40520 40784 40785 (37719 . "큰 가마 제") (39847 . "복 제")
                 20125 20452 21399 22494 23036 23170 23198 24607
                 24831 24952 24975 25994 26668 26762 27333 28477
                 28941 29438 29686 30588 30894 30898 31157 32612
                 33123 33385 33813 34139 34669 34832 34947 35034
                 35605 36263 37563 39814 39863 39919 39990 40045
                 40250 40345 40609 40780 40783 13351 13418 13714
                 13795 14146 14562 14856 14905 15625 15876 16181
                 16529 16581 16670 16674 16698 16736 16893 17252
                 17338 17394 17422 17441 17517 17519 17718 17817
                 18083 18161 18206 18433 18577 18897 18961 18966
                 19032 19034 19104 19105 19106 19107 19243 19262
                 19265 19386 19471 19557 19560 19569 19853 19857
                 19858 19859 19881))
       (51312 . ((26397 . "아침 조") (31062 . "조상 조, 할아버지 조")
                 (21161 . "도울 조") (40165 . "새 조") (35519 . "고를 조")
                 (36896 . "지을 조") (26089 . "새벽 조, 이를 조") (20806 . "조짐 조")
                 (26781 . "가지 조, 조목 조") (29031 . "비출 조, 비칠 조")
                 (25805 . "잡을 조") (31199 . "구실 조, 부세 조") (24340 . "위문할 조, 조상할 조")
                 (28526 . "조수 조") (32068 . "끈 조, 짤 조") (36339 . "뛸 조")
                 (36249 . "조나라 조") (26362 . "성 조") (31066 . "복 조")
                 (29157 . "마를 조, 불에말릴 조") (25514 . "둘 조") (36973 . "만날 조")
                 (37347 . "낚시 조") (24427 . "새길 조") (26361 . "무리 조")
                 (35412 . "고할 조, 조서 조") (38459 . "험할 조, 막힐 조")
                 (31895 . "거칠 조, 간략할 조") (28437 . "배로 실어 나를 조, 배로실어올 조")
                 (20939 . "시들 조") (31264 . "빽빽할 조, 조밀할 조")
                 (34299 . "조류 조, 글 조") (32903 . "비롯할 조, 비로소 조")
                 (22066 . "조롱할 조, 희롱할 조") (29226 . "손톱 조")
                 (30522 . "바라볼 조, 멀리볼 조") (36481 . "떠들 조, 움직일 조")
                 (26839 . "대추나무 조") (31967 . "지게미 조") 35419
                 (32368 . "야청 통견 조, 아청통견 조") (38613 . "독수리 조")
                 21514 (20430 . "도마 조") (28836 . "照와 同字") (31381 . "정숙할 조, 안존할 조")
                 (34468 . "벼룩 조") (27133 . "구유 조, 마구 조") (26177 . "朝의 古字, 아침 조")
                 (29866 . "면루관 드림 옥 조, 면류관드림옥 조") 26465
                 (22122 . "떠들썩할 조") (30337 . "하인 조") (31079 . "조묘 조")
                 (24450 . "갈 조") 38460 (32985 . "복 조") 30338 (31990 . "쌀 내어 팔 조")
                 (31432 . "부엌 조") (21405 . "措와 同字") 24933 33025
                 31900 (20993 . "바라 조") 32091 (35679 . "시끄러울 조")
                 (40304 . "수리 조") (28577 . "씻을 조") 32617 26064
                 (34601 . "매미 조") (27522 . "죽을 조") (40706 . "朝와 同字")
                 (33226 . "누릴 조") (22024 . "지껄일 조") 26387 21825
                 29244 31413 (20347 . "방정맞을 조") 29745 (34086 . "담쟁이 조")
                 39880 (37530 . "쟁개비 조") 31409 34732 (31961 . "매조미쌀 조")
                 26316 27808 29639 30654 32295 34023 22423 24652
                 33671 34251 37314 39371 16820 (25214 . "채울 조")
                 27950 (35458 . "꾈 조") 26176 30857 31427 31464 32634
                 33345 36270 37289 18262 18882 32486 26531 28790
                 31374 32452 32562 33553 33646 35791 35843 36213
                 38028 38035 38110 38123 40102 40119 40479 (20654 . "마칠 조")
                 (23333 . "날씬할 조") (25030 . "근심할 조") (31498 . "조리 조")
                 (32105 . "견직물 날실 수 조") (33370 . "거룻배 조")
                 (37407 . "낚시 조") (37629 . "불리지 않은 쇠 조") (39899 . "도미 조")
                 20228 21023 21500 21781 21795 22877 23464 23942
                 24167 24227 25235 26523 26630 26765 27170 27172
                 27558 27720 29171 (29227 . "손톱조") 29411 29813
                 30265 30404 30615 31314 31628 31763 32576 32644
                 32846 32897 32904 32911 33692 33892 33961 34040
                 34235 34497 34729 35135 35161 35228 35339 35631
                 35647 36242 36455 37173 37333 37553 37594 37677
                 37797 37818 37866 37936 37955 38655 38734 38779
                 38807 39395 39646 39777 39817 39841 39991 40582
                 40716 13467 13550 13658 13963 14102 14296 14417
                 14431 14461 14591 14852 15165 15204 15291 15333
                 15358 15830 15854 15904 16016 16527 16666 16774
                 16903 16966 17244 17262 17325 17350 17452 17539
                 17554 17667 17688 17689 17827 17850 17890 17951
                 18186 18356 18413 18487 18576 18756 19046 19454
                 19540 19650 19826))
       (51313 . ((36275 . "발 족") (26063 . "겨레 족") (31751 . "모일 족, 가는대 족")
                 (37827 . "화살촉 족") 34079 21739 (30255 . "피부병 이름 족")
                 38238 36415 14266 15680))
       (51316 . ((23384 . "있을 존") (23562 . "존귀할 존, 높을 존")
                 29127 (25333 . "꽂을 존") 37903 27202 35000 37516))
       (51320 . ((21330 . "군사 졸") (25305 . "서툴 졸, 졸할 졸")
                 (29469 . "갑자기 졸") 20485 25469 13330 33082 31265
                 26890 36388 21318 28842 31652 13325 15413 16528
                 17946 18077 19455 19555))
       (51333 . ((32066 . "끝낼 종, 마칠 종") (23447 . "마루 종")
                 (31278 . "씨 종, 종류 종") (24478 . "좇을 종") (37912 . "쇠북 종")
                 (32305 . "늘어질 종, 세로 종") (33131 . "부스럼 종")
                 (29742 . "옥홀 종, 서옥이름 종") (32156 . "모을 종")
                 (36405 . "발꿈치 종") (36394 . "자취 종") (24939 . "권할 종, 놀랄 종")
                 (37758 . "술병 종, 술잔 종") (28121 . "물 소리 종")
                 (24752 . "즐길 종, 즐거울 종") (26837 . "종려나무 종")
                 (20519 . "상고 신인 종, 옛적신인 종") 24467 (36452 . "자취 종")
                 32294 (30215 . "수종 종") (34749 . "누리 종") 23928
                 (26934 . "棕과 同字") 39459 34031 34007 39683 36072
                 36392 (31933 . "주악 종") 24470 24554 (20224 . "두려워할 종")
                 20490 23600 26401 31846 31945 32195 32245 34931
                 37712 38047 20174 24578 26526 30125 32437 32456
                 32508 32959 33473 38202 39572 39577 (24914 . "생각할 종")
                 (26570 . "방망이 종") (27141 . "전나무 종") (29821 . "패옥소리 종")
                 20657 21027 21920 22571 22585 22928 23107 23406
                 23887 23893 23927 24146 24468 24504 26158 26288
                 27505 27767 27848 28480 28802 28996 29095 29460
                 29475 30258 30619 30850 30955 31279 31981 32223
                 32746 33113 33360 33916 34080 34668 35508 35557
                 35957 36073 36465 37567 37661 39436 39460 39689
                 39735 39918 39932 40228 40334 40744 13808 13921
                 13959 14113 14422 14484 14571 14573 14678 14865
                 15350 15971 16092 16467 16714 16855 16934 16954
                 17247 17451 17528 17893 17972 17986 18005 18251
                 18542 18600 19703))
       (51340 . ((24038 . "왼 좌, 왼쪽 좌") (22352 . "앉을 좌")
                 (20304 . "도울 좌") (24231 . "자리 좌") (25387 . "꺽을 좌, 꺾을 좌")
                 13876 33054 (39677 . "북상투 좌") (21065 . "꺽을 좌")
                 (33693 . "여물 좌") 34959 38153 (30180 . "뾰루지 좌")
                 20403 21655 22798 (23662 . "왼손") 30537 30700 33996
                 34009 36246 36979 37564 13611 13880 14039 14206
                 14279 15211 16563 18422))
       (51396 . ((32618 . "허물 죄") 36768 23989 27276 14353
                 14995))
       (51452 . ((20027 . "주인 주") (20303 . "살 주, 머무를 주")
                 (36208 . "달릴 주") (27880 . "물댈 주") (26205 . "낮 주")
                 (26417 . "붉을 주") (23449 . "집 주") (37202 . "술 주")
                 (24030 . "고을 주") (21608 . "두루 주") (33311 . "배 주")
                 (26609 . "기둥 주") (27954 . "섬 주, 물가 주") (29664 . "구슬 주, 진주 주")
                 (26666 . "그루 주") (22863 . "아뢸 주") (37956 . "쇠 부어 만들 주, 쇠불릴 주")
                 (39376 . "머무를 주, 말머무를 주") (30087 . "밭두둑 주")
                 (36913 . "돌 주, 주일 주") (36490 . "머뭇거릴 주, 주저할 주")
                 (35461 . "벨 주") (35387 . "주낼 주") (32044 . "명주 주")
                 (20570 . "지을 주, 만들 주") (31264 . "빽빽할 주") (24282 . "부엌 주")
                 (32002 . "껑거리끈 주, 말고삐 주") (22014 . "시킬")
                 (21610 . "빌 주, 저주 주") (36659 . "모일 주, 몰려들 주")
                 20881 (32964 . "후사 주, 투구 주") (31820 . "투호 살 주, 셈대 주")
                 (28234 . "모일 주, 물모일 주") (32162 . "얽힐 주, 빽빽할 주")
                 (28557 . "단비 주") (28855 . "심지 주") (34523 . "거미 주")
                 (23005 . "예쁠 주") (37198 . "세 번 빚은 술 주") (20367 . "난쟁이 주, 난장이 주")
                 26515 40799 21416 (38676 . "장마 주") (32920 . "팔꿈치 주")
                 36464 (20756 . "무리 주") (25284 . "떠받칠 주") (35055 . "홑이불 주")
                 (36946 . "다가설 주") (36057 . "진휼할 주") 36501 35704
                 31808 (36616 . "끌채 주") 40648 15675 37054 32056
                 18118 40584 (20380 . "속일 주") (24172 . "휘장 주")
                 (33120 . "살결 주") 36844 (38668 . "운우 모양 주") 20279
                 22115 28109 (30787 . "주사 주") 21678 26221 36326
                 13649 (34079 . "태주 주") 26886 29656 (23564 . "세울 주")
                 21778 (22965 . "예쁠, 사람 이름") 25549 13632 (31826 . "주문 주")
                 (34496 . "나무좀 주") (35403 . "呪와 同字") 24606 26993
                 30128 31228 32081 34229 36596 39349 13647 14433
                 15652 17191 26172 20002 20454 20945 24113 27249
                 30068 31609 32419 32504 35803 35818 36178 36364
                 36736 36751 37619 38117 38136 39547 40508 17207
                 (19999 . "갈 주") (36238 . "사람이름 주") (37458 . "쇳돌 주")
                 (40708 . "거미 주") (20022 . "불똥") 21438 21650 21900
                 22155 22772 23140 23444 23872 24174 24479 25060
                 26018 27174 27337 27365 27574 27904 28863 28880
                 29181 29352 29355 30107 30359 30441 30698 30763
                 31081 31897 32604 33751 33860 34773 34805 35006
                 36209 36307 36630 36878 37102 37506 37545 39155
                 39407 39410 39430 39842 39888 40248 40259 13631
                 13700 14251 14653 14750 14773 14819 15094 15973
                 16027 16338 16359 16864 16921 17048 17287 17335
                 17339 17389 17619 17631 17822 18284 18398 18475
                 18669 18899 19036 19090 19606 19620 19847))
       (51453 . ((31481 . "대 죽") (31909 . "죽 죽, 미음 죽")
                 39739 20476))
       (51456 . ((28310 . "수준기 준, 법 준") (20426 . "준걸 준")
                 (36981 . "좇을 준") (20934 . "승인할 준, 법 준") (27994 . "깊을 준")
                 (23803 . "높을 준") (39423 . "준마 준") (22472 . "가파를 준, 높을 준")
                 (28652 . "칠 준, 깊을 준") (26201 . "밝을 준") (27197 . "술그릇 준, 술단지 준")
                 (31459 . "마칠 준, 일마칠 준") (34850 . "꿈틀거릴 준, 꿈실거리는 준")
                 (20737 . "雋·俊과 同字, 영특할 준") (38603 . "새살찔")
                 (30063 . "俊과 同字, 농부 준") (36897 . "뒷걸음칠 준, 물러갈 준")
                 (23535 . "모일 준") (28940 . "굽을 준, 불땔 준") (32583 . "樽·尊과 同字")
                 20950 (36466 . "걸터앉을 준") (38588 . "새매 준") (39189 . "대궁 준")
                 (25753 . "누를 준") 30531 (24823 . "어수선할 준") 13657
                 (32167 . "피륙의 폭의 넓이 준") (22082 . "수군거릴 준")
                 18804 (36358 . "그칠 준") 18349 (30388 . "주름 준")
                 36403 13434 16080 25432 27582 38486 40277 14418
                 39298 39567 40159 (22523 . "과녘 준") (22699 . "樽·罇과 同字")
                 (37903 . "창고달 준") (40018 . "송어 준") (40276 . "금계 준")
                 20550 20692 22783 22795 23442 23967 25039 31253
                 31476 31674 32348 33904 35376 35664 36863 37800
                 40381 40439 13395 13470 14182 15850 16106 16177
                 17423 17727 18665 19358))
       (51460 . ((33537 . "풀 처음 나는 모양 줄, 싹 줄")
                 (20092 . "줄 줄") 23826 23850 31371 31516 14613))
       (51471 . (13791))
       (51473 . ((20013 . "가운데 중") (37325 . "무거울 중")
                 (34886 . "무리 중") (20210 . "버금 중") 35080 20247
                 (30526 . "衆의 本字") 22933 23185 29382 31575 33327
                 33597 34459 34665 34934 36826 37409 38528 13370
                 13398 14561 15540 15650 17196 18278 18879 19695))
       (51593 . (21063 (21373 . "곧 즉") (21363 . "卽의 俗字")
		       39933 21918 21799 33844 23857 33709 34637 40087
		       40107 25572 30349 33629 39938 40033 13409 17434))
       (51600 . ((27355 . "빗 즐") (39469 . "수말 즐") 28676
                 (21918 . "두런거릴 즐") 22578 26633 39576 25843 26966
                 13833))
       (51608 . (24590))
       (51609 . (36655 (27713 . "즙 즙, 진액 즙") (33914 . "기울 즙, 지붕일 즙")
		       (26987 . "노 즙, 돛대 즙") 28616 (27293 . "楫과 同字")
		       (34170 . "삼백초 즙") 28242 38709))
       (51613 . ((26366 . "일찍 증") (22686 . "불을 증, 더할 증")
                 (35657 . "증거 증, 보증할 증") (36104 . "보낼 증, 줄 증")
                 (30151 . "증세 증, 병증세 증") (24974 . "미워할 증")
                 (33976 . "찔 증") (25327 . "건질 증") (28893 . "김 오를 증")
                 (29969 . "시루 증") (32338 . "비단 증") 35388 26365
                 (23954 . "산 높고 험할 증") (30704 . "주살 증") 22679
                 29844 27239 37163 29111 39507 (32638 . "어망 증")
                 17302 32559 36192 22222 25756 28519 30963 31474
                 31708 35652 40027 14546 15845 16234 16880 16981
                 17585 17732 17856 18018 18517 18764 18797 19485))
       (51648 . ((22320 . "땅 지") (30693 . "알 지") (33267 . "이를 지")
                 (24535 . "뜻 지") (20043 . "갈 지") (27490 . "발 지, 그칠 지")
                 (32025 . "종이 지") (25345 . "가질 지") (25351 . "손가락 지")
                 (35672 . "표할 지, 알 지") (25903 . "가를 지, 지탱할 지")
                 (21482 . "다만 지") (26525 . "가지 지") (26234 . "슬기 지, 지혜 지")
                 (27744 . "못 지") (35468 . "기록할 지") (36978 . "늦을 지, 더딜 지")
                 (26088 . "맛있을 지, 뜻 지") (22336 . "터 지") (33437 . "지초 지")
                 (33026 . "기름 지, 비계 지") (32930 . "사지 지, 팔다리 지")
                 (31049 . "복 지") (21675 . "길이 지, 여덟치 지") (25711 . "잡을 지, 쥘 지")
                 (26547 . "탱자나무 지") (31063 . "공경할 지") (36286 . "발 지")
                 (31047 . "마침 지") (30757 . "숫돌 지") (36100 . "폐백 지, 패백 지")
                 (28460 . "담글 지, 물거품 지") (27802 . "물가 지, 모래톱 지")
                 (33463 . "구리때 지, 백지 지") (34584 . "거미 지") (22656 . "계단 위의 공지 지")
                 36933 15062 (31210 . "벼 처음 익을 지") 31190 (36499 . "넘어질 지")
                 31630 (33296 . "핥을 지") 31084 17965 36618 (36383 . "머뭇거릴 지")
                 (31722 . "저 이름 지") (40409 . "맹금 지") (27056 . "주춧돌 지")
                 (22395 . "모래섬 지") 32989 40183 (25210 . "손바닥 지")
                 (36601 . "굴대 머리 지") 24075 17499 (27868 . "강 이름 지")
                 25624 34483 (38447 . "터 지") 21390 22700 37589 39495
                 (30179 . "사마귀 지") 24445 24649 25255 36959 18009
                 18311 19893 25370 28173 32440 36157 36725 36734
                 36831 40503 (39848 . "젓갈 지") 20455 20481 20970
                 21159 21553 21707 22337 22356 22360 22474 23073
                 23739 23987 24559 26475 26682 26813 26917 27749
                 27750 27924 28157 28522 30139 30731 31187 31486
                 31554 31595 31624 32149 32977 33299 34940 35231
                 35287 35368 35608 36030 36223 37231 39350 40242
                 40709 13636 14242 14248 14260 14454 14495 14849
                 14972 15057 15300 15476 15501 15515 15663 15903
                 16231 16251 16545 16650 16745 16859 17391 17611
                 17612 17628 17633 17641 18180 18295 18390 18401
                 18470 18553 18685 18720 18765 18932 19111 19528
                 19584 19628 19653 19833))
       (51649 . ((30452 . "곧을 직") (32887 . "벼슬 직, 직분 직")
                 (32340 . "짤 직") (31287 . "기장 직, 메기장 직") (31257 . "올벼 직")
                 28333 15272 (31133 . "사람 이름 직") 32832 29318
                 27188 23298 32455 32844 33201 34357 34777 36548
                 14219 15940 17416))
       (51652 . ((30494 . "참 진") (36914 . "나아갈 진") (30433 . "다할 진")
                 (36784 . "다섯째지지 진, 날 진") (38515 . "늘어놓을 진, 묵을 진")
                 (25391 . "떨칠 진") (38663 . "벼락 진, 진동할 진") (38499 . "줄 진, 진칠 진")
                 (29645 . "보배 진") (37805 . "누를 진, 진압할 진") (26185 . "進과 同字, 나아갈 진")
                 (22645 . "띠끌 진, 티끌 진") (27941 . "나루 진") (31206 . "벼 이름 진, 진나라 진")
                 (35386 . "볼 진, 진맥할 진") (21972 . "성낼 진, 꾸짖을 진")
                 (30137 . "홍역 진, 마마 진") (36049 . "구휼할 진, 넉넉할 진")
                 (26187 . "晉의 俗字, 나아갈 진") (36587 . "수레 뒤턱 나무 진, 구를 진")
                 (21767 . "놀랄 진") (25634 . "꽂을 진") (33275 . "이를 진")
                 (27524 . "끊어질 진, 다할 진") (32265 . "搢과 同字, 분홍빛 진")
                 (27035 . "개암나무 진") (34967 . "홑옷 진, 고운 옷 진")
                 (29800 . "아름다운 돌 진, 옥돌 진") (30043 . "두렁길 진, 밭갈피 진")
                 (29857 . "옥돌 진") (32285 . "삼실 진, 맺을 진") (30603 . "부릅뜰 진, 눈부릅뜰 진")
                 (28337 . "많을 진, 성할 진") (26733 . "평고대 진, 대청 진")
                 (34095 . "더워지기 진, 사철쑥 진") 30495 (36225 . "좇을 진")
                 31131 37806 (20760 . "盡과 同字") 29809 36226 (31289 . "떨기로 날 진")
                 30114 30521 32983 19139 (33985 . "우거질 진") 28301
                 29646 (29650 . "옥 이름 진") 23019 (39698 . "숱 많을 진")
                 (20402 . "동자 진") (37441 . "보배 진") 25974 27031
                 30485 32062 16396 25656 28636 34236 34707 36675
                 36788 13361 16444 23613 23576 32444 32537 32540
                 35786 36168 36728 36827 38215 38453 38472 (34740 . "설렐 진")
                 20397 21994 22157 22587 22630 22743 23232 23235
                 23335 23634 24106 24363 25262 25355 25936 26147
                 26499 27140 27196 27549 27989 29513 29710 29799
                 31816 32022 32124 32293 32836 33272 33942 34237
                 34564 35030 35499 35603 36569 36839 37681 38741
                 39383 40688 13720 14528 14706 15120 15132 15172
                 15352 15369 15488 16390 16550 16551 16943 17078
                 17436 17488 17903 18281 18420 18537 18991 19134
                 19698))
       (51656 . ((36074 . "바탕 질") (30142 . "병 질") (31209 . "차례 질")
                 (23018 . "조카 질") (36845 . "갈마들 질, 침노할 질")
                 (36300 . "넘어질 질, 달아날 질") (31378 . "막을 질")
                 (21489 . "꾸짖을 질") (23241 . "미워할 질, 질투할 질")
                 (24089 . "책갑 질") (20314 . "흐릴 질") (26702 . "차꼬 질, 조카 질")
                 (33187 . "새살 날 질, 음도 질") (20356 . "어리석을 질, 굳을 질")
                 (34541 . "거머리 질") (29894 . "사람 이름 질") (32112 . "질 질")
                 32779 36604 36499 26186 39469 (22436 . "개밋둑 질")
                 25270 37517 34976 (37061 . "고을 이름 질") (37973 . "모루 질")
                 (33978 . "납가새 질") 29918 34975 31017 31057 21669
                 26163 32720 25347 22967 23756 24654 26595 32778
                 13772 32470 36136 38106 38183 20776 21141 21396
                 24226 24881 25116 27081 27341 27959 28492 30499
                 30512 31223 32041 32965 33119 33271 33525 34703
                 34738 35396 35921 35922 36013 13429 13799 13837
                 14129 14140 14955 15539 15546 17348 17479 18208
                 18376 19257 19523))
       (51664 . ((26389 . "나 짐") (26015 . "술 따를 짐, 짐작할 짐")
                 (40198 . "짐새 짐") 37209 37206 15330 40489 26650
                 13872 15032 15283 15656 16137 18933 19636))
       (51665 . ((38598 . "모일 집, 모을 집") (22519 . "잡을 집")
                 (36655 . "모을 집") (20160 . "세간 집, 열사람 집") (32221 . "낳을 집, 길쌈 집")
                 (28503 . "샘솟을 집, 물끓을 집") (37878 . "판금 집, 쇳조각 집")
                 (25122 . "그칠 집") 35583 32310 14349 24953 (21664 . "참소할 집")
                 13931 14350 17174 34043 17429 25191 32119 32521
                 36753 20156 20590 21337 23983 29921 33381 35642
                 37715 38630 38631 13770 13809 14572 15479 15637
                 16466 16740 16965 19056 19382))
       (51667 . (21997))
       (51669 . ((24501 . "부를 징") (25074 . "혼날 징, 징계할 징")
                 (28548 . "맑을 징") (28546 . "澄과 同字") 24500 (28691 . "맑을 징")
                 24981 (30634 . "바로 볼 징") 24809 (30309 . "적취 징")
                 14366))
       (52264 . ((36554 . "수레 차") (27425 . "버금 차") (27492 . "이 차")
                 (19988 . "또 차") (20511 . "빌 차, 빌릴 차") (24046 . "어긋날 차, 어길 차")
                 (33590 . "차 차") (21449 . "깍지낄 차, 두갈래 차") (36974 . "막을 차, 가릴 차")
                 (21983 . "탄식할 차, 슬플 차") (36425 . "넘어질 차, 지날 차")
                 (31642 . "차자 차, 글 차") (23919 . "우뚝 솟을 차, 험할 차")
                 (37365 . "비녀 차") (30923 . "윤낼 차, 갈 차") (20376 . "실의할 차, 자랑할 차")
                 (27086 . "뗏목 차") 33524 23033 16552 21124 27722
                 26440 33687 (29811 . "깨끗할 차") 21814 25199 (20349 . "도울 차")
                 25766 (22867 . "오만할 차") (30824 . "조개 이름 차")
                 40570 (23700 . "갈림길 차") 25363 35341 36304 36407
                 37277 14859 30743 33315 36710 40574 (20566 . "찢을 차")
                 (24483 . "빌릴 차") 20300 20583 20584 21391 21779
                 21967 22011 22898 23237 23923 25184 25619 25661
                 26267 26664 31521 32001 32088 32926 33366 33414
                 33963 34070 34392 34556 34921 35112 35655 37535
                 37768 40775 40825 13804 14230 14482 14590 14731
                 15842 15897 16174 16232 16285 16479 16536 16590
                 17068 17426 17444 17496 17505 17586 17813 17843
                 18339 18389 18465 18536 18652 18824 19448 19464
                 19497 19577 19664 19774 19801 19877))
       (52265 . ((30528 . "붙을 착, 부딪칠 착") (33879 . "분명할 착")
                 (25417 . "잡을 착") (37679 . "섞일 착") (36487 . "머뭇거릴 착")
                 (25662 . "짤 착") (38015 . "뚫을 착") (31364 . "좁을 착, 끼일 착")
                 (40810 . "악착할 착, 악착같을 착") (26034 . "깎을 착")
                 26037 (25139 . "창으로 찌를 착") 26030 35537 23062
                 27998 (25801 . "찌를 착") 31534 23061 29695 20991
                 35836 38169 40842 20021 21074 25775 25798 28738
                 31323 31719 31758 31857 31987 32274 33697 (36789 . "쉬엄쉬엄가다")
                 (36790 . "책받침") 36906 37596 37985 13442 13560
                 13983 15499 16869 17614 17780 18776 18787))
       (52268 . ((36106 . "도울 찬") (35738 . "기릴 찬, 칭찬할 찬")
                 (39184 . "먹을 찬, 삼킬 찬") (29914 . "제기 찬, 옥그릇 찬")
                 (29158 . "빛날 찬, 찬란할 찬") (38013 . "끌 찬, 뚫을 찬")
                 (29864 . "빛날 찬") (25776 . "지을 찬, 글지을 찬") (39244 . "반찬 찬")
                 (32386 . "모을 찬") (31762 . "빼앗을 찬") (31428 . "숨을 찬, 숨길 찬")
                 (32408 . "이을 찬") (31922 . "정미 찬, 정한 쌀밥 찬")
                 39137 (28591 . "맑을 찬") 36059 28236 (25890 . "모일 찬")
                 32393 (29224 . "불땔 찬") (27409 . "모일 찬") 15889
                 (24017 . "높이 솟을 찬") 25861 35715 37191 (36274 . "놀라 흩어질 찬")
                 31713 28669 31347 37978 (20775 . "모일 찬") 29897
                 31158 (20793 . "모을 찬") 25883 28754 31851 36517
                 20019 27349 35192 37180 36190 25786 25874 28799
                 29906 31388 32565 36273 36479 36508 37186 38249
                 39316 (21143 . "끊을 찬") 20751 22219 27561 29110
                 29208 31737 31825 32356 32700 34194 36518 38009
                 39265 14033 14138 14365 14547 14732 15108 15879
                 15997 16023 16110 16526 16980 17013 17187 18364
                 18382 18557 19236 19256 19478 19543))
       (52272 . ((23519 . "살필 찰") (21049 . "절 찰") (26413 . "패 찰, 편지 찰")
                 (25830 . "뿌릴 찰, 문지를 찰") (32046 . "감을 찰, 묶을 찰")
                 21681 (25166 . "뺄 찰") (25334 . "핍박할 찰") 21747
                 24000 21070 27307 31032 35431 37972 38113 38258
                 20082 20602 22163 25859 26714 27346 31012 31948
                 32037 32811 34491 34879 37720 14217 14763 15452
                 15568 15592 16624 17645 17747 18807 19882))
       (52280 . ((21443 . "간여할 참, 석 참") (24920 . "참혹할 참, 슬플 참")
                 (24921 . "부끄러울 참") (26028 . "벨 참") (31449 . "우두커니 설 참, 역마을 참")
                 (35730 . "참소할 참, 간악할 참") (20717 . "참람할 참, 거짓 참")
                 (35734 . "참서 참, 비결 참") (22649 . "구덩이 참") (25082 . "뉘우칠 참")
                 (21445 . "의논할 참") 21442 24808 (24922 . "부끄러울 참")
                 (39490 . "곁마 참") (35670 . "참소할 참") 20707 (40690 . "검푸르죽죽할 참")
                 13656 (24009 . "가파를 참") (23940 . "높을 참") (27111 . "판 참")
                 21441 (38001 . "보습 참") (25007 . "슬퍼할 참") 25723
                 21444 (25881 . "찌를 참") (39262 . "탐할 참") 26101
                 (27395 . "살별 참") 18203 (20787 . "어긋날 참") 22667
                 35634 25076 27638 35392 37286 18207 21142 22086
                 23934 23946 26369 28730 35249 36283 22545 23853
                 24813 25600 26025 26912 30214 30876 35863 35886
                 35894 37694 38261 39307 39574 40682 (27610 . "토끼 참")
                 (37864 . "끌 참") 20650 22197 22733 23328 23345 23939
                 25714 29521 30254 30939 30947 33388 35675 37185
                 39769 14103 14110 14309 14331 14337 14759 14907
                 15403 15909 16037 16233 16330 16490 16513 17486
                 18371 18712 18731 18927 19084 19583 19707 19880))
       (52285 . ((21809 . "노래 창, 노래부를 창") (31379 . "창 창")
                 (26124 . "창성할 창") (20489 . "곳집 창") (21109 . "비롯할 창, 비로소 창")
                 (33980 . "푸를 창") (26274 . "펼 창, 화창할 창") (24432 . "밝을 창")
                 (28356 . "찰 창, 서늘할 창") (25950 . "높을 창, 열 창")
                 (26166 . "밝을 창") (20513 . "여광대 창, 계집광대 창")
                 (28466 . "불을 창, 넘칠 창") (27085 . "창 창, 나무창 창")
                 (30241 . "부스럼 창") (24884 . "슬퍼할 창") (33081 . "배부를 창, 배불룩할 창")
                 (23100 . "몸 파는 여자 창, 창녀 창") (29462 . "미쳐 날뛸 창, 놀랄 창")
                 (33750 . "창포 창") (24288 . "헛간 창") (33369 . "선창 창")
                 (24757 . "슬퍼할 창") (25654 . "닿을 창") 22257 (39727 . "술")
                 (21041 . "비롯할 창") 29269 (20480 . "미칠 창") (36428 . "추창할 창")
                 37783 38326 (27653 . "새털 창") (24797 . "멍할 창")
                 (31419 . "창 창") 28112 24963 25680 21071 21081 29262
                 (20630 . "천할 창") 37862 (31383 . "창 창") 30076 (29810 . "옥 소리 창")
                 37625 40364 29778 31724 21424 30946 35602 37673
                 26538 20179 20261 20263 21019 21595 24581 24582
                 25111 25250 27815 28072 28829 29594 29617 30021
                 30126 32960 33329 33485 36292 38166 38176 38410
                 40115 40487 (20948 . "찰 창") (25127 . "다칠 창") 37638
                 26905 20218 20696 20997 21958 22250 23906 24784
                 25700 26239 28108 28664 28987 29079 29514 29737
                 34725 35054 35503 36086 39911 40730 13449 13855
                 14847 15076 16189 16669 16731 16750 16811 17323
                 17865 18062 18318 18432 18594 18700 18762 19048
                 19350 19581 19613))
       (52292 . ((25505 . "캘 채") (33756 . "나물 채") (37319 . "캘 채, 취할 채")
                 (24425 . "무늬 채, 채색 채") (20661 . "빚 채, 빚질 채")
                 (34081 . "거북 채, 채나라 채") (22512 . "영지 채, 사패땅 채")
                 (23528 . "울짱 채, 나무우리 채") (32181 . "비단 채, 오색비단 채")
                 (37365 . "비녀 채") (23488 . "녹봉 채, 동관 채") (30758 . "울타리 채, 울타리칠 채")
                 30261 34822 (33565 . "구리때 채") 30245 (26828 . "참나무 채")
                 20536 22214 20538 34431 38039 17213 23111 27064
                 30572 32297 36393 38763 14674 14725 15575 16576
                 17192 17414 17457 17933 18327 18453 18635 19458))
       (52293 . ((36012 . "꾸짖을 책") (20874 . "책 책") (31574 . "채찍 책, 꾀 책")
                 (26613 . "울짱 책, 우리 책") (32735 . "꿩 책") 26629
                 (31744 . "살평상 책") 20876 (22038 . "외칠 책") (30932 . "찢을 책")
                 (24152 . "건 책") 31582 16895 14717 33332 33574 35630
                 21863 24123 31654 36131 (31527 . "책 상자 책") (34481 . "벼메뚜기 책")
                 21383 23271 24993 25338 25927 30367 30612 31651
                 31863 31907 33815 34011 34102 34420 35534 36846
                 37519 37884 38937 13757 14297 14898 14925 14942
                 15611 16365 18234 18352 18372 19412 19878))
       (52376 . ((34389 . "살 처, 곳 처") (22971 . "아내 처")
                 (24765 . "슬퍼할 처") (20932 . "쓸쓸할 처") 17807 (33803 . "풀 성하게 우거진 모양 처")
                 22788 20966 (28114 . "쓸쓸할 처") 35248 (35255 . "엿볼 처")
                 35281 (35076 . "깃의 가 처") (37098 . "고을 이름 처")
                 21022 31436 32042 32192 32797 35259 38667 40328
                 14104 16486))
       (52377 . ((23610 . "자 척") (25299 . "주울 척") (26021 . "물리칠 척, 내칠 척")
                 (25114 . "겨레 척") (21050 . "찌를 척") (38587 . "새 한 마리 척, 짝 척")
                 (38495 . "오를 척, 나아갈 척") (28364 . "씻을 척") (25842 . "던질 척, 버릴 척")
                 (30240 . "파리할 척") (33034 . "등성마루 척") (20508 . "대범할 척, 빼어날 척")
                 (24957 . "근심할 척") (21076 . "바를 척, 없앨 척") (36448 . "밟을 척, 뛸 척")
                 (24789 . "두려워할 척") 35073 (36310 . "발바닥 척")
                 24956 (25709 . "주울 척") (36497 . "머뭇거릴 척") 36432
                 (20470 . "시작할 척") 25510 40378 (34612 . "도마뱀 척")
                 38578 (22601 . "메마른 땅 척") 36450 36386 22660
                 25431 22668 40732 25527 28068 36399 40545 (21582 . "길이 단위 척")
                 (22375 . "기지 척") 20662 21498 23924 (24435 . "자축거리다")
                 24463 24720 24790 30825 30953 31886 33164 34439
                 37850 40803 13870 14237 14463 14715 16218 17095
                 17534 17897 18336 18358 19420))
       (52380 . ((22825 . "하늘 천") (21315 . "일천 천") (24029 . "내 천")
                 (27849 . "샘 천") (28154 . "얕을 천") (36983 . "옮길 천")
                 (36068 . "천할 천") (34214 . "천거할 천") (36368 . "밟을 천")
                 (20018 . "꿸 천") (37351 . "팔찌 천") (25797 . "멋대로 천, 오로지 천")
                 (31359 . "뚫을 천, 통할 천") (21912 . "헐떡거릴 천, 헐떡일 천")
                 (38369 . "열 천, 밝힐 천") (38433 . "두렁 천, 밭둑길 천")
                 (33307 . "어그러질 천, 어수선할 천") (38854 . "그네 천")
                 (20191 . "일천 천, 천사람 천") (29588 . "옥고리 천")
                 (34453 . "지렁이 천") (33616 . "거듭할 천") (20521 . "예쁠 천")
                 36801 36932 (28666 . "흩뿌릴 천") (27914 . "이를 천")
                 (34118 . "경계할 천") 21003 (33960 . "대가 우거질 천")
                 29120 33564 36046 (33418 . "풀 무성할 천") 26667
                 38857 (20468 . "엷을 천") 25322 21110 22903 33608
                 (20739 . "머뭇거릴 천") (36799 . "천천히 걸을 천")
                 26804 26824 27940 28278 32170 38350 38848 27973
                 36341 38030 (31046 . "하늘 천") 20865 24527 25190
                 28293 33927 36145 38031 38416 38831 40136 (20706 . "어그러질 천")
                 (29929 . "킬로그램 천") (31873 . "킬로메트르 천") (33270 . "거듭 천")
                 (38749 . "하늘 천") 20383 20827 20850 22213 22284
                 22321 22322 22844 (24027 . "내") 24157 25596 26363
                 26436 27006 27343 27458 27674 27724 27736 28723
                 28763 28773 29294 29485 29652 29775 30360 30815
                 31439 31685 31711 32351 33128 33906 34099 35650
                 35896 36644 36658 37370 38323 38740 39937 13354
                 13960 14315 14564 15308 15451 16402 16406 16697
                 16804 16805 16998 17230 17526 17590 18102 18604
                 18612 18752 19792))
       (52384 . ((37941 . "쇠 철") (21746 . "밝을 철") (24505 . "통할 철, 관철할 철")
                 (25764 . "거둘 철, 걷을 철") (28552 . "물 맑을 철")
                 (21894 . "哲과 同字, 밝을 철") (36685 . "바퀴 자국 철, 수레자국 철")
                 (32180 . "꿰맬 철, 맺을 철") (20984 . "볼록할 철, 뾰족할 철")
                 (36639 . "그칠 철") 37444 (37525 . "鐵의 古字") (25479 . "주울 철")
                 (21852 . "마실 철") 33103 37921 (24793 . "근심할 철")
                 30071 (39214 . "탐할 철") 19349 (27488 . "마실 철")
                 (21087 . "깎을 철") 37258 15321 24714 15130 24443
                 32512 36749 36761 38081 38151 (22481 . "밝을 철")
                 (37667 . "물미 철") (39163 . "탐할 철") 21238 21461
                 21856 22174 22205 25952 25962 28080 28914 28942
                 29217 30638 30739 32620 32837 34504 34567 35056
                 35521 37227 37275 37608 39510 13901 16652 17542
                 18082 18357 18430 18795 19150))
       (52392 . ((28155 . "더할 첨") (23574 . "뾰족할 첨") (30651 . "볼 첨, 쳐다볼 첨")
                 (20681 . "다 첨") (35522 . "아첨할 첨") (31844 . "제비 첨")
                 (27838 . "더할 첨, 절일 첨") (35449 . "이를 첨") (31805 . "농 첨")
                 (29979 . "달 첨, 단 첨") (24541 . "더럽힐 첨") (31799 . "처마 첨")
                 29980 25016 (27280 . "처마 첨") (27111 . "편지 첨")
                 (35164 . "행주치마 첨") 38850 31830 26653 24767 28233
                 25517 (24777 . "팰 첨") 27799 35047 (28728 . "적실 첨")
                 20927 22113 23126 29122 37999 14593 19100 31614
                 20325 35043 35844 38168 (24168 . "휘장 첨") (27388 . "쐐기 첨")
                 21379 23974 25921 28688 28746 31841 33300 33306
                 33790 34371 35165 35719 37215 37587 37737 37937
                 38298 39170 40647 13313 13394 13584 13741 13940
                 14002 14124 14440 14608 14768 15154 15239 15687
                 15802 16262 16673 16700 16861 17132 17382 17498
                 18316 18472 18835))
       (52393 . ((22974 . "첩 첩") (35548 . "염탐할 첩, 이간할 첩")
                 (24086 . "표제 첩, 문서 첩") (29266 . "글씨판 첩, 편지 첩")
                 (30090 . "거듭 첩, 포갤 첩") (36028 . "붙을 첩, 붙일 첩")
                 (25463 . "이길 첩") (36626 . "문득 첩") (22558 . "성가퀴 첩, 담 첩")
                 (30571 . "속눈썹 첩, 속눈섭 첩") (35126 . "덧옷 첩, 겹옷 첩")
                 36633 (21899 . "재잘거릴 첩") 39944 (24599 . "고요할 첩")
                 14407 (35130 . "겹옷 첩") 30082 20543 27662 32820
                 19764 23125 29251 30089 31317 19038 38075 21472
                 30067 35853 36148 36740 (20514 . "빠를 첩") (21611 . "맛 볼 첩, 귀에 대고 소근거릴 첩, 앙징할 첩")
                 20604 23211 23848 24821 26337 28097 32193 33083
                 33820 34037 35505 36373 36389 36832 37446 39871
                 39900 14644 14664 14871 14889 14968 14969 15096
                 15146 15215 15436 15538 15591 17164 17494 17736
                 18463 18785 18922 19729 19839))
       (52397 . ((38737 . "푸를 청") (28152 . "맑을 청") (35531 . "청할 청")
                 (32893 . "들을 청") (26228 . "갤 청") (24307 . "관청 청")
                 (33729 . "부추꽃 청, 초목무성한 모양 청") (39894 . "청어 청")
                 38738 20521 28165 21548 13652 32892 (20938 . "서늘할 청")
                 (22282 . "뒷간 청") 32884 24304 21381 24193 35831
                 40109 19734 23143 (34619 . "귀뚜라미 청") (40324 . "교청새 청")
                 21403 22813 25477 26258 27696 29984 30851 37100
                 15705 18300 18317 18973))
       (52404 . ((39636 . "몸 체") (20999 . "자를 체, 끊을 체")
                 (36958 . "갈마들 체") (28399 . "막힐 체, 쌓일 체") (26367 . "바꿀 체, 대신할 체")
                 (36910 . "미칠 체, 잡아가둘 체") (32224 . "맺을 체")
                 (28053 . "눈물 체") (32180 . "꿰멜 체") (35558 . "살필 체")
                 (21059 . "머리 깎을 체, 머리깍을 체") (20307 . "몸 체")
                 (36550 . "몸 체") 36934 28382 36883 31128 (30732 . "섬돌 체")
                 25507 (26851 . "산앵두나무 체") (24408 . "돼지 체")
                 27866 34142 (34069 . "작은 가시 체") 36976 34201
                 26453 36528 39650 30096 34627 18007 (33922 . "작은 가시 체")
                 23933 34734 (27554 . "나른할 체") (38726 . "구름 낄 체")
                 39199 23644 25048 26648 35077 25573 25712 13956
                 19078 23625 32532 35867 36798 36882 17878 19736
                 (22159 . "재채기 할 체") 39664 21447 22164 22662
                 23865 25390 25611 25686 26996 29839 30200 30345
                 30709 30940 31094 31501 36815 36822 37361 37482
                 37520 37919 38708 39605 39680 39684 40392 40393
                 13561 13624 14423 15153 16323 16409 16671 16871
                 17185 17326 17349 17453 18423 18464 19855))
       (52488 . ((33609 . "풀 초") (21021 . "처음 초") (25307 . "부를 초")
                 (36229 . "뛰어넘을 초") (30990 . "주춧돌 초") (25220 . "베낄 초")
                 (32918 . "닮을 초") (31186 . "시간 단위 초") (21736 . "망 볼 초")
                 (26970 . "회초리 초, 초나라 초") (28966 . "그슬릴 초, 탈 초")
                 (30977 . "암초 초") (31245 . "벼 줄기 끝 초, 점점 초")
                 (27189 . "땔나무 초") (35970 . "담비 초, 돈피 초") (26786 . "나무 끝 초, 농기 초")
                 (28818 . "볶을 초") (24980 . "수척할 초, 파리할 초")
                 (30813 . "초석 초") (34121 . "파초 초") (37259 . "초산 초")
                 (37294 . "초례 초") (26898 . "산초나무 초, 후추 초")
                 (37218 . "초 초") (21119 . "죽일 초, 노략질할 초") (33400 . "풀 초")
                 28267 (33493 . "능소화 초, 능토풀 초") (36834 . "멀 초")
                 (35482 . "꾸짖을 초") (21222 . "수고롭게 할 초") (24708 . "근심할 초")
                 (32131 . "생명주실 초") (37396 . "노략질할 초") (36602 . "수레 초")
                 (23789 . "가파를 초") (24586 . "슬퍼할 초") (24832 . "정색할 초")
                 (39659 . "다박머리 초") (38808 . "칼집 초") (35673 . "꾸짖을 초")
                 (26474 . "끝 초") (22093 . "먹을 초") (23719 . "산이 높을 초")
                 36372 26904 (29131 . "홰 초") (40422 . "뱁새 초") (39014 . "야윌 초")
                 21165 20431 17296 (36256 . "멀 초") 25015 (20716 . "밝게 살필 초")
                 (23957 . "높을 초") (37740 . "가래 초") 23737 24360
                 25768 30503 (20578 . "얼이 빠져 있을 초") 38585 27290
                 33202 33408 34797 37450 17967 32481 (33401 . "초두")
                 35788 35822 35887 36730 38046 38201 40554 40838
                 (23286 . "수척할 초") (32790 . "밭 거듭 갈 초") (37739 . "가래 초")
                 (40800 . "이를 갈 초") 20198 20207 20730 20746 21121
                 21131 21557 22961 23628 23971 24016 24105 24890
                 25719 27381 27433 28496 28619 28963 29005 29150
                 29876 30212 30276 30532 30631 31058 33582 33657
                 33772 33812 34161 34286 35288 35372 35589 36269
                 36680 37349 37731 37902 38487 38825 38866 39474
                 40616 40742 40828 13817 14616 14702 14960 15170
                 15335 15532 15653 15780 15813 15880 15974 16515
                 16682 16844 16958 17155 17197 17284 17553 18184
                 18369 18434 18778 19020 19192 19199 19499 19587
                 19888))
       (52489 . ((35320 . "받을 촉, 느낄 촉, 지날 촉, 범할 촉, 찌를 촉")
                 (23660 . "이을 촉, 닿을 촉, 부탁할 촉, 모을 촉, 조심할 촉")
                 (29165 . "爥의 俗字, 촛불 촉") (20419 . "핍박할 촉, 촉박할 촉, 좁을 촉, 빠를 촉, 짧을 촉, 재촉할 촉")
                 (34560 . "촉규화벌레 촉, 큰 닭 촉, 나라이름 촉")
                 (22225 . "부탁할 촉") (30679 . "곧을 촉, 우뚝 솟을 촉, 우뚝할 촉")
                 37827 22065 (36485 . "자취 촉, 강동강동 뛸 촉, 철쭉꽃 촉")
                 30633 (30682 . "볼 촉") (39633 . "해골 촉") 26040 30211
                 34827 (29221 . "비칠 촉") 21146 (26351 . "비칠 촉")
                 27484 28767 35302 28891 13577 (34213 . "촉규화 촉, 자귀밥 촉")
                 20109 23374 26038 27416 31440 33064 33221 34878
                 35169 35177 36247 36352 38019 40448 13895 15342
                 16127 16602 16845 17205 17292 17789 18033 18148
                 18377 18399 18481 19077))
       (52492 . ((23544 . "치 촌, 마디 촌, 헤아릴 촌, 조금 촌")
                 (26449 . "마을 촌, 밭집 촌, 시골 촌") (24534 . "헤아릴 촌, 짐작할 촌")
                 (37032 . "村과 同字, 마을 촌") 21004 (21515 . "인치 (inch) 촌") 23639 28554 31871))
       (52509 . ((32317 . "꿰맬 총, 거느릴 총, 합할 총, 끄덩이 잡을 총")
                 (32880 . "귀 밝을 총, 민첩할 총, 귀맑을 총") (37507 . "도끼 구멍 총, 총 총")
                 (23541 . "사랑할 총, 임금께 총애받을 총, 은혜 총, 첩 총")
                 (21474 . "떨기 총, 모을 총, 번잡할 총") (22618 . "冢의 俗字, 무덤 총")
                 (25696 . "거느릴 총, 모을 총, 무리 총, 장수 총")
                 (24740 . "바쁠 총, 덤빌 총") (34085 . "파 총, 파 뿌리 총, 푸를 총")
                 (24961 . "바쁠") (33905 . "풀 더북할 총") 20898 25460
                 32207 25556 21254 20588 28430 (39492 . "총이말 총")
                 24625 32235 32870 20655 (34031 . "풀 더북할 총")
                 26823 28520 32865 (37862 . "창 총, 찌를 총") (39448 . "驄과 同字")
                 24553 29825 34242 24803 28439 31733 15827 17623
                 18744 19995 23456 24635 32874 38131 39586 22507
                 24865 25603 27180 27191 27401 28488 28621 28743
                 28967 29084 29162 29212 32258 32369 33552 34764
                 35621 37743 37843 14257 15941 16825 16929 16997
                 17419 17786 17875 18543 18925 19468))
       (52524 . ((25774 . "머리 끄덩이 잡을 촬, 당길 촬, 비칠 촬, 사진찍을 촬")
                 23098 25893 31393 32339 35146 13581 15214 15734
                 16815 19829))
       (52544 . (21840 21851 31101 37642 18852))
       (52572 . ((26368 . "가장 최, 우뚝할 최, 넉넉할 최")
                 (20652 . "재촉할 최, 핍박할 최, 일어날 최") (23828 . "산 우뚝할 최, 높고 가파를 최, 성씨 최")
                 (25703 . "꺾을 최, 망할 최, 저지할 최") 34142 28140
                 (27057 . "서까래 최") (29824 . "옥빛 찬란할 최, 구슬 주렁주렁 달릴 최")
                 (28476 . "깊을 최, 고을 최, 눈물 줄줄 흘릴 최, 눈서리 쌓일 최")
                 (32279 . "상복 최, 상옷 최") 14177 22010 33063 (22060 . "물 최, 산적 한 입에 넣을 최")
                 (30954 . "험하고 높을 최") 22676 24923 27119 32183
                 32542 20951 27190 27452 29091 29525 30368 31325
                 32035 32679 33175 37623 37849 14112 14190 15695
                 17067 17716 17938 18001 18941 19741))
       (52628 . ((31179 . "가을 추, 세월 추, 때 추, 말이 뛰놀 추, 성씨 추")
                 (25512 . "옮길 추, 가릴 추, 궁구할 추, 차례로옮길 추")
                 (19985 . "수갑 추, 사람이름 추") (36861 . "쫓을 추, 따를 추")
                 (25277 . "뺄 추, 뽑을 추, 당길 추, 거둘 추") (37276 . "추할 추, 같을 추, 무리 추, 자라똥구멍 추, 부끄러울 추")
                 (36264 . "달아날 추, 추창할 추") (37138 . "추나라 추, 나라이름 추")
                 (27000 . "가래나무 추, 노나무 추, 산유자 추, 바둑판 추")
                 (26894 . "쇠몽둥이 추, 칠 추, 짓찧을 추, 참나무 추, 쇠뭉치 추")
                 (27166 . "밑둥 추, 지두리 추, 고동 추, 북두 첫째 별 추, 긴요할 추, 달 추, 가운데 추, 지도리 추")
                 (22684 . "떨어질 추, 잃을 추, 추락할 추") (37195 . "괴수 추, 두목 추, 숙 익을 추, 끝날 추")
                 (33467 . "꼴 추, 짐승먹이 추, 풀벨 추") (37648 . "송곳 추")
                 (27084 . "칠 추") (37656 . "저울눈 추, 마치 추, 저울 추")
                 (37786 . "쇠마치 추, 저울 추") (39949 . "미꾸라지 추")
                 34282 (38619 . "새 새끼 추, 병아리 추, 어린아이 추, 날짐승 새끼 추")
                 (39478 . "거덜 추, 마부 추, 짐승이름 추") (30394 . "쭈그러질 추, 추할 추")
                 (35535 . "꾀할 추, 물을 추") (28267 . "늪 추, 폭포 추, 찌푸릴 추, 서늘할 추")
                 (26491 . "수갑 추, 조막손이 추") (33833 . "맑은 대쑥 추, 가래나무 추")
                 (40612 . "성길 추, 클 추, 거칠 추") (33965 . "芻와 同字")
                 (30259 . "병 나을 추, 덜릴 추") 36237 (24774 . "실심할 추, 섭섭할 추")
                 (31753 . "버금 추, 가지런히 날 추") (38508 . "벼루 추, 정월 추, 부끄러울 추, 마을 추")
                 (31648 . "채찍 추, 볼기채 추") (21886 . "두런거릴 추, 찍찍거릴 추")
                 24832 (31634 . "비 추") 31704 (25637 . "북 칠 추") (24090 . "비 추, 털 추")
                 (38822 . "그네 추, 말고들개 추") (20710 . "임금 추, 고용할 추")
                 (29955 . "우물 벽돌 추") (25462 . "종아리 칠 추, 짓찧을 추")
                 (32267 . "줄에 달릴 추") (39755 . "북상투 추") 39915
                 40577 (23093 . "젊을 추, 물고기 추, 예쁜 여자 추")
                 33809 (39429 . "청부루말 추") (39948 . "미꾸라지 추, 빙자할 추, 해추 추")
                 23863 27259 37177 14888 (32272 . "주름 추, 구길 추")
                 37740 (40342 . "두루미 추") 40373 23497 26864 33734
                 22959 34660 (38585 . "새 추") 25538 26871 29052 32197
                 35528 35993 38976 39692 18984 26530 38180 21005
                 22368 23604 30385 32457 32530 35833 36235 37049
                 38181 38607 39546 39571 40112 40133 40537 18265
                 (25579 . "묶을 추, 거둘 추, 모을 추") (40315 . "아롱비둘기 추")
                 40580 (40861 . "秋의 古字") 20404 20733 21532 22526
                 23197 23216 23607 25009 25515 25578 25610 26872
                 27019 27552 27805 28190 28268 28269 28340 29034
                 29331 29952 30573 30597 30657 30846 30984 31180
                 31312 31619 31693 31829 32231 33159 33783 34290
                 34677 34775 34852 35285 36261 36954 37104 37268
                 37655 37958 38823 39374 39695 39902 40003 40640
                 40817 40826 13383 13427 13580 13737 13779 13785
                 14016 14343 14690 14917 14918 14974 15216 15237
                 15282 15374 15733 15811 15829 16083 16153 16301
                 16317 16375 16442 16738 16755 16779 16822 16876
                 17107 17112 17146 17343 17421 17424 17442 17532
                 17679 18228 18258 18451 18671 18828 18946 19118
                 19348 19558 19617 19681 19832 19846))
       (52629 . ((31069 . "빌 축, 끊을 축, 비로소 축") (19985 . "소 축, 축시 축, 북동쪽 축")
                 (31689 . "다질 축, 쌓을 축") (36880 . "쫓을 축, 물리칠 축")
                 (32302 . "줄어들 축, 물러갈 축, 거둘 축, 모자랄 축, 오그라질 축")
                 (30044 . "가축 축, 쌓을 축, 그칠 축") (33988 . "쌓을 축, 모을 축, 감출 축")
                 (36600 . "바디집 축, 질책 축, 속바퀴 축, 얇을 축, 굴대 축")
                 (36468 . "찰 축, 밟을 축") (26586 . "북 축") (36441 . "찡그릴 축, 쭈그러질 축")
                 (31569 . "비파 축") (31482 . "나라이름 축") 36391 26615
                 (33331 . "고물 축, 배 꼬리 축") (36444 . "발 끌어 디딜 축")
                 28352 (35926 . "발 얽은 돼지걸음 축") 25280 (27117 . "단풍나무 축")
                 38310 (22959 . "동서 축") 31489 34027 18462 32553
                 36724 (40704 . "두꺼비 축") 19986 20631 22076 22088
                 25930 26019 27559 31288 31723 32300 33599 36469
                 37136 39011 40001 13796 14057 14085 15383 16293
                 18002 18008 18031 18798 19377))
       (52632 . ((26149 . "봄 춘, 남녀의 정 춘, 술 춘, 온화할 춘")
                 (26943 . "참죽나무 춘, 대춘나무 춘, 어르신네 춘")
                 (29763 . "옥이름") 27332 26486 27201 18320 22590
                 23179 26110 26265 26493 28279 30582 33797 33846
                 34685 36080 39942 40350 13754 15081 16356 18362
                 18501 18862 19616))
       (52636 . ((20986 . "날 출, 게울 출, 도망할 출, 보일 출, 낳을 출, 물러갈 출, 생질 출")
                 (40668 . "내칠 출, 물리칠 출") (26414 . "삽주뿌리 출")
                 26415 24629 32064 (31211 . "차조 출") 23680 30784
                 32460 24631 27150 27434 27855 28834 33677 40674
                 13377 13592 17956 18403 18618))
       (52649 . ((24544 . "충성 충, 곧을 충") (20805 . "가득찰 충, 막을 충, 당할 충")
                 (34802 . "벌레 충, 김 오를 충") (34909 . "충돌할 충, 찌를 충, 거리 충, 돌파할 충")
                 (34935 . "가운데 충, 정성 충, 착할 충, 속곳 충, 마음 충")
                 (27798 . "깊을 충, 온화할 충, 어릴 충") (20914 . "깊을 충, 온화할 충, 어릴 충")
                 (34411 . "벌레 충") 31181 (24545 . "근심할 충") 17821
                 32704 30405 29675 33375 25584 28026 28468 33594
                 14621 33575 22019 29214 30155 31052 13517 14681
                 14820 15252 15576 15582 16524 16788 17962 18257
                 18418 18548))
       (52684 . ((24756 . "근심할 췌, 파리할 췌, 피폐할 췌, 초췌할 췌")
                 (33795 . "모을 췌, 괘 이름 췌") (36101 . "붙일 췌, 데릴사위 췌, 혹 췌, 군더더기 췌")
                 (33205 . "첩육경 췌, 지라 췌, 췌장 췌") 30209 25571
                 (24820 . "두려워할 췌") 38983 36184 30121 24560))
       (52712 . ((21462 . "거둘 취, 찾을 취, 받을 취, 빼앗을 취, 장가들 취, 취할 취")
                 (23601 . "좇을 취, 이룰 취, 곧 취, 가령 취, 능할 취, 저자 취, 나아갈 취")
                 (21561 . "불 취, 숨쉴 취, 악기 불 취, 부를 취, 충동할 취")
                 (36259 . "추창할 취, 뜻 취") (33261 . "냄새 취, 향기 취, 썪을 취, 고약한 이름 취")
                 (37257 . "술취할 취, 궤란할 취") (32858 . "모을 취, 고을 취, 걷을 취, 쌓을 취")
                 (28810 . "불 지필 취, 밥 지을 취, 불땔 취") (23094 . "장가들 취, 중개인 취")
                 (32736 . "비취 취, 푸를 취, 깃푸른새 취") (33030 . "연할 취, 약할 취")
                 (39519 . "몰아갈 취, 달릴 취, 별안간 취, 말몰 취")
                 (40434 . "독수리 취") (22068 . "부리 취") (25571 . "잴 취, 시험할 취, 요량할 취")
                 37204 (35292 . "부리 취") (27635 . "솜털 취, 연하고 부드러울 취")
                 31425 32710 39588 40555 (20899 . "쌓을 취, 모을 취")
                 27207 22143 23945 27100 27271 27400 28642 29883
                 33027 33196 33230 33264 37243 40865 15004 15316
                 15324 15446 17756 19892))
       (52769 . ((28204 . "측량할 측, 깊을 측, 맑을 측, 날카로울 측, 헤아릴 측")
                 (20596 . "곁 측, 기울어질 측, 저물 측, 배반할 측, 가까울 측, 미천할 측, 아플 측")
                 (24827 . "슬플 측, 감창할 측, 아플 측, 불쌍할 측, 불쌍히여길 측")
                 (20164 . "기울 측, 성조 측, 물이 괄괄 흐를 측, 희미할 측, 옆 측, 돈 이름 측")
                 (21408 . "뒷간 측, 평상 가장자리 측, 섞일 측, 기울 측")
                 24257 26115 31236 21397 20391 24699 27979 24194
                 25425 26135 27716 30047 15553 16927))
       (52772 . (35183 27372 34301 40788 20781))
       (52784 . (38358))
       (52789 . ((23652 . "겹 층, 거듭 층, 층층대 층") 36461
                 23618 16492))
       (52824 . ((27835 . "다스릴 치, 다듬을 치, 익을 치, 치료할 치")
                 (33268 . "이를 치, 극진할 치, 버릴 치, 불러올 치, 풍치 치, 드릴 치, 보낼 치, 연구할 치")
                 (40786 . "이 치, 나이 치, 같을 치") (32622 . "베풀 치, 버릴 치, 역말 치, 안치할 치, 둘 치")
                 (24677 . "부끄럼 치, 욕될 치, 부끄러울 치") (20516 . "값 치, 가치있을 치, 만날 치, 당할 치")
                 (24501 . "치성 치") (38601 . "꿩 치, 폐백 치, 성 윗담 치, 목맬 치")
                 (30196 . "癡의 俗字, 어리석을 치") (31258 . "어린 벼 치, 어릴 치, 늦을 치")
                 (23769 . "산이 우뚝 솟을 치, 갖출 치, 쌓을 치, 산우뚝할 치")
                 (39347 . "달릴 치, 거동길 치, 전할 치") (20360 . "사치할 치, 넓을 치, 많을 치, 풍부할 치")
                 (29118 . "불 활활 붙을 치, 불성할 치") (24159 . "깃대 치, 표기 치")
                 (21988 . "비웃을 치, 냉소 치") (32251 . "톡톡할 치, 빽빽할 치, 헌 옷 기울 치")
                 (30164 . "치질 치, 풍병 치") (30305 . "어리석을 치, 미련할 치, 미치광이 치, 주근깨 치")
                 (32199 . "검을 치, 검은비단 치") (34473 . "치벌레 치, 어리석을 치, 업신여길 치, 벌레이름 치")
                 (31305 . "稚와 同字, 어린벼 치") (23919 . "산 울뚝질뚝할 치")
                 (22516 . "찰진흙 치, 흙 이길 치") (26772 . "치자 치")
                 (28100 . "물이름 치, 검을 치") (36636 . "짐수레 치, 휘장 친 수레 치")
                 32827 (23512 . "둘 치, 폐할 치, 쉴 치") (40223 . "솔개 치, 수알치 치, 부엉이 치, 말똥구리 치, 토란 치, 마음대로 날릴 치, 올빼미 치")
                 34733 (24257 . "뒷간 치, 평상 가장자리 치, 섞일 치, 버금 치")
                 (24053 . "잔 치, 연지 치") 27503 25061 (33745 . "따비밭 치")
                 (37657 . "저울눈 치, 조금 치") 17417 (35115 . "옷 빼앗을 치, 벗을 치, 넋 잃을 치")
                 (35960 . "발 없는 벌레 치, 풀 치, 해치양 치, 발없는벌레의총칭 치")
                 (32122 . "가는 칡베 치, 가는 갈포 치") 23224 (34201 . "풀 깎을 치")
                 (22786 . "뒤처져오다") 20587 (21702 . "입 딱 벌일 치")
                 (30052 . "제터 치") 30517 (21358 . "술잔 치, 연지 치")
                 (40260 . "솔개 치") 31290 35318 20095 24268 (36337 . "머뭇거릴 치, 주저할 치, 갖출 치")
                 20540 26900 27501 34995 38474 39482 14870 15424
                 (38436 . "떨어질 치, 헐릴 치, 언덕 치, 사태날 치")
                 40656 24228 28365 29947 31974 33013 35053 37079
                 14475 14565 16447 26624 24092 28861 32513 35311
                 36750 38193 39536 40123 40497 40831 (30163 . "풍병 치, 악할 치")
                 (39892 . "숭어 치") (40217 . "雉와 同字") 20479 20610
                 21366 22401 22417 22939 23856 23867 24477 24640
                 24964 25067 25336 25649 26072 26456 28285 30014
                 31218 32014 32795 33430 33548 33580 33614 33880
                 34274 34990 35175 35381 35459 35514 36334 36666
                 37497 37759 39396 40325 (40697 . "바느질하다") 40797
                 14035 14465 14689 15289 15344 15390 15643 15796
                 16008 16322 16336 16371 16710 16724 16776 16874
                 16988 17079 17084 17206 17321 17383 17740 18086
                 18099 18288 18307 18366 18368 18638 18863 18909
                 19336 19489 19860))
       (52825 . ((21063 . "법칙 칙, 본받을 칙, 모범 칙, 조목 칙")
                 (21189 . "신칙할 칙, 칙령 칙") (39149 . "갖출 칙, 닦을 칙, 부지런할 칙")
                 24668 20377 (25941 . "경계할 칙, 다스릴 칙, 삼갈 칙, 바를 칙, 칙서 칙, 기릴 칙, 신칙할 칙")
                 24919 36971 40338 40408 21017 39276 28116 36265
                 37459 19701))
       (52828 . ((35242 . "사랑할 친, 친할 친, 몸소 친, 일가 친, 사돈 친")
                 (35183 . "속옷 친, 접근할 친, 베풀 친") (27372 . "관 친, 무궁화나무 친, 땔나무 친")
                 22187 28697 18315 20146 27015 34924 40832 (40788 . "이 갈 친, 어릴 친")
                 34301 20780 23175 23540 28597 40787 18483 18484))
       (52832 . ((19971 . "일곱 칠, 글체이름 칠") (28422 . "옻나무 칠, 물이름 칠")
                 (26578 . "漆의 俗字, 옻칠할 칠") 13522 15181 26748
                 13314 13564 13567 13585 15299 18185 18651))
       (52840 . ((37341 . "바늘 침, 침 침, 바느질할 침, 찌를 침")
                 (27784 . "장마물 침, 진펄 침, 잠길 침, 채색할 침, 고요할 침, 가라앉을 침")
                 (20405 . "범할 침, 침노할 침, 습격할 침, 흉작 침")
                 (23522 . "잠잘 침, 쉴 침, 방 침, 못생길 침") (28024 . "적실 침, 불릴 침, 번질 침, 빠질 침")
                 (26517 . "베개 침, 수레뒤 가로댄 나무 침, 소말뚝 침, 베게 침")
                 (37756 . "바늘 침, 침 침") (30759 . "방칫돌 침, 다듬잇돌 침")
                 (28251 . "빠질 침, 잠길 침") (29723 . "보배 침") 35574
                 (24561 . "믿을 침, 정성 침") 31090 (27785 . "沈의 俗字")
                 (39416 . "말 몰아 달릴 침") (37599 . "새길 침") (23510 . "젖을 침, 점점 침")
                 26787 (26937 . "모탕 침, 도끼바탕 침, 토막 침") 30890
                 20244 23505 25228 36408 25557 26542 26795 36061
                 16814 17507 23517 38024 38163 39566 (37108 . "땅이름 침")
                 20819 21786 22480 22822 23964 25227 28613 32133
                 32157 33680 33817 33908 35483 35523 37378 37718
                 38659 39868 40053 13428 13508 13518 13684 14194
                 15003 15622 15668 15785 16283 16924 17566 17970
                 18048 18049 18231 18719 18784 19124 19158 19180
                 19289))
       (52841 . ((34756 . "벌레 움츠릴 칩, 우물거릴 칩")
                 39357 34544 28432 37342 13943 14223 14234 15249))
       (52845 . ((31281 . "저울질할 칭, 일컬을 칭, 헤아릴 칭, 같을 칭")
                 (31204 . "저울 칭, 백 근 칭") 31216 31338 20545 29231
                 14069 17757))
       (52852 . (16084 20327 33001))
       (53132 . ((24555 . "기분이 좋을 쾌, 시원할 쾌, 빠를 쾌, 쾌할 쾌")
                 (29546 . "간교할 쾌, 교활할 쾌") (22828 . "결단할 쾌, 나누어 정할 쾌, 괘 이름 쾌")
                 20744 (22130 . "목구멍 쾌, 훨할 쾌") 20393 21721
                 21455 31607))
       (53440 . ((20182 . "다를 타, 저 타, 간사할 타") (25171 . "때릴 타, 칠 타")
                 (22949 . "편안할 타, 일 없을 타, 타협할 타, 떨어질 타")
                 (22702 . "떨어질 타, 상투 타, 잃을 타") 38539 (24816 . "게으를 타, 태만할 타")
                 (21822 . "침 타, 버릴 타") (38464 . "비탈 타, 땅이름 타")
                 (39389 . "약대 타, 곱사등이 타") (33333 . "키 타") (26965 . "橢의 略字, 나무그릇 타")
                 (39345 . "탈 타, 짐 실을 타") (26422 . "나무가지 휘늘어질 타, 떨기 타, 움킬 타, 휘늘어진 타")
                 (21668 . "꾸짖을 타, 슬플 타, 뿜을 타, 쩍쩍 씹는소리 타")
                 (25302 . "拕와 同字, 당길 타") (36836 . "어정거릴 타")
                 22549 (39364 . "馱의 俗字") 38465 (36529 . "피할 타")
                 (27825 . "물이 갈래질 타, 큰 비 타, 눈물이 흐를 타")
                 23427 35409 (36302 . "미끄러질 타") (35435 . "자랑할 타, 풍칠 타, 신통할 타")
                 (20311 . "다를 타, 저 타, 마음에 든든할 타, 더할 타, 입을 타, 짊어질 타")
                 21522 (40713 . "자라 타") (39390 . "駞와 同字") 37217
                 (26561 . "키 타") 22194 13904 (25301 . "끌 타, 당길 타")
                 22428 21660 22908 (27234 . "둥글길죽할 타, 수레통 가운데그릇 타, 길죽한 그릇 타")
                 (39808 . "모래무지 타") 27826 22430 38436 23200 23966
                 26421 38479 13507 20152 26925 35815 38167 39534
                 39548 40501 40717 22376 39506 (40213 . "타조 타")
                 21044 21057 21635 21649 21722 22427 22517 22582
                 22703 23287 23598 23726 23836 24437 24988 25185
                 25349 25350 25777 26711 26927 27620 27643 28086
                 28793 28839 28840 29280 30161 30755 30756 30882
                 31173 32061 32158 34953 36243 36325 36346 36530
                 36547 37754 39958 40270 40743 14022 14026 14286
                 14473 14697 14858 15312 15376 15920 15921 16103
                 16160 16259 16541 16732 16795 17259 17512 17518
                 17587 17987 18020 18191 18468 18512 18729 18730
                 18747 19028 19089 19519 19594 19761))
       (53441 . ((24230 . "꾀할 탁, 헤아릴 탁, 벼슬이름 탁, 법도 탁")
                 (28609 . "흐릴 탁, 더러울 탁, 물이름 탁") (25299 . "물리칠 탁, 헤칠 탁, 밀칠 탁, 주울 탁")
                 (25176 . "밀칠 탁, 떡국 탁, 차반 탁") (21331 . "높을 탁, 뛰어날 탁, 책상 탁, 성씨 탁")
                 (28655 . "씻을 탁, 빨래할 탁, 적실 탁, 클 탁, 빛날 탁, 산 민둥민둥할 탁, 살찔 탁, 놀 탁")
                 (35351 . "맡길 탁, 부탁할 탁, 핑계할 탁") (29730 . "옥 쪼을 탁, 옥 다듬을 탁, 가릴 탁, 쫄 탁")
                 (25826 . "뽑을 탁, 뺄 탁, 솟을 탁, 이길 탁") (37944 . "요령 탁, 목탁 탁, 큰방울 탁")
                 (22396 . "찢을 탁, 터질 탁, 난산할 탁, 싹 트일 탁")
                 (21828 . "쪼을 탁, 문 두드릴 탁") (20524 . "환할 탁, 클 탁")
                 (26589 . "쪼갤 탁, 조두 탁, 목탁 탁") (26219 . "환할 탁, 밝을 탁")
                 (29752 . "사람이름 탁") (25286 . "터질 탁, 찢어질 탁, 열릴 탁")
                 (27216 . "전대 탁, 자루 탁, 도가니 탁, 공잇소리 탁, 짐승이름 탁")
                 20039 27094 (28095 . "물방울 떨어질 탁, 칠 탁, 물이름 탁, 땅이름 탁")
                 (36916 . "멀 탁, 절름거릴 탁, 뛰어날 탁") (31836 . "대 껍질 탁")
                 (36372 . "우뚝 설 탁, 앙감질 탁") 26899 20354 36256
                 36293 39776 (27824 . "떨어뜨릴 탁, 돌 던질 탁, 떨어질 탁")
                 (30714 . "나무이름 탁") (34304 . "떨어질 탁, 마를 탁")
                 20187 25984 26452 14622 14843 27978 31656 35756
                 38094 38255 39270 20274 20953 21090 21099 21125
                 21829 24249 26700 27093 27729 30842 31399 33818
                 34839 34981 35159 36401 37420 37938 39141 39346
                 39517 40299 13612 15020 15999 16031 17470 17644
                 19347))
       (53444 . ((27470 . "탄식할 탄, 아름답다 할 탄, 화답할 탄")
                 (24392 . "쏠 탄, 악기부릴 탄, 탄알 탄, 탄핵할 탄, 퉁길 탄")
                 (28845 . "숯 탄, 불똥 탄, 볶일 탄") (35477 . "기를 탄, 클 탄, 속일 탄, 넓을 탄, 탄생할 탄, 방탕할 탄")
                 (28760 . "여울 탄, 해 이름 탄") (32187 . "옷 터질 탄, 나타날 탄, 파할 탄")
                 (24986 . "수고로울 탄, 꺼릴 탄") (21585 . "삼킬 탄, 휩쌀 탄, 감출 탄, 멸할 탄")
                 (22374 . "너그러울 탄, 평평할 탄, 넓을 탄") (22022 . "한숨쉴 탄, 탄식할 탄")
                 (27563 . "다할 탄") 24814 27546 24377 21534 (30321 . "중풍증 탄, 사지 틀릴 탄")
                 (25892 . "열 탄, 펼칠 탄") 30227 25763 20708 22077
                 25528 21876 24382 25674 28393 30186 30251 32509
                 35806 (25019 . "坦과 同字") (39506 . "돈짝무늬총이말 탄")
                 24633 25849 26298 27709 27718 28256 30083 30899
                 32688 35680 39464 13716 14070 14839 14863 17102
                 17978))
       (53448 . ((33067 . "벗어날 탈, 간략할 탈, 뼈 바를 탈, 그르칠 탈, 벗을 탈")
                 (22890 . "빼앗을 탈, 잃어버릴 탈, 좁은 길 탈, 깎을 탈")
                 26802 25939 (20411 . "간사할 탈, 가벼울 탈, 간략할 탈")
                 22842 25437 33073 25946 27634 30181 33676 39861
                 40317 14558 19140))
       (53456 . ((25506 . "더듬을 탐, 찾을 탐, 취할 탐, 염탐할 탐")
                 (36010 . "탐할 탐, 욕심낼 탐") (32829 . "즐길 탐, 귀 축 늘어질 탐, 웅크리고 볼 탐, 즐거울 탐, 깊고 멀 탐")
                 (30472 . "노려볼 탐, 즐길 탐") 37267 25762 (37206 . "술 즐길 탐")
                 36525 36138 36181 (22015 . "여럿이 먹는 소리 탐")
                 24528 20683 27786 36071 14654 15671 15685 17369
                 17381))
       (53457 . ((22612 . "탑 탑") (25645 . "모뜰 탑, 박을 탑, 탈 탑")
                 (27067 . "평상 탑, 모직 탑, 자리 탑") 38354 39944
                 (25640 . "비문 박을 탑, 베낄 탑") (22604 . "낮은 땅 탑, 초벌갈 탑, 떨어질 탑")
                 21970 25752 28463 15534 37961 38812 38835 13811
                 38424 40142 (20637 . "답답할 탑, 불안할 탑, 어리석을 탑, 용렬할 탑")
                 22147 22678 27646 28347 29415 30249 31138 32823
                 36962 37652 37777 39833 39976 14053 15315 15529
                 15994 16947 17160 17500 17533))
       (53461 . ((31958 . "엿") (28271 . "물 끓일 탕, 물이름 탕, 씻을 탕, 끓을 탕")
                 (34153 . "넓고 클 탕, 없어질 탕, 방탕할 탕, 소탕할 탕")
                 (23445 . "골집 탕, 방탕할 탕, 석수 탕") (24081 . "나라 곳집 탕, 처자식 탕")
                 (30442 . "씻을 탕, 움직일 탕, 진동할 탕, 그릇 부실 탕, 뭍 배질할 탕")
                 36927 (30893 . "무늬진 돌 탕, 흰 기운 탕, 넘칠 탕, 산이름 탕")
                 31772 34666 36412 (34351 . "盪과 同字") 29847 37899
                 33633 27748 28907 30720 38132 (29145 . "씻을 탕, 불에 델 탕")
                 20562 22325 23160 25701 33770 34202 36260 38652
                 16146 17495 18834))
       (53468 . ((22826 . "클 태, 굵을 태, 심할 태, 처음 태, 콩 태")
                 (27888 . "클 태, 통할 태, 너그러울 태, 편안할 태, 괘 이름 태, 타이")
                 (24907 . "모양 태, 뜻 태") (27526 . "위태할 태, 가까이할 태, 비롯할 태")
                 (24608 . "게으를 태, 느릴 태, 거만할 태") (21488 . "별 태, 늙을 태")
                 (32974 . "아이 밸 태, 시작 태, 애밸 태") (20812 . "기쁠 태, 지름길 태, 모일 태, 통할 태, 곧을 태, 구멍 태, 바꿀 태, 괘 이름 태")
                 (36294 . "밟을 태") (39089 . "몹시 부는 바람 태, 태풍 태")
                 (27760 . "미끄러울 태, 넘칠 태, 씻길 태, 사치할 태, 미끄러질 태")
                 (33492 . "이끼 태") (31518 . "볼기칠 태") 21574 (37040 . "태나라 태")
                 39364 (36840 . "미칠 태") 20817 32063 34581 34555
                 39384 19326 39824 24533 21588 (22509 . "보뚝 태")
                 20810 24562 14615 28313 14701 24577 32464 38043
                 39552 40080 17212 (23079 . "기뻐할 태, 더딜 태")
                 23393 (25260 . "칠 태") 20643 20909 22268 22835 25147
                 25464 25988 26307 28849 28850 29156 31887 32144
                 32255 32957 33318 33773 36585 37214 37414 39088
                 39422 13376 13431 13890 14283 14541 14550 15602
                 16922 19208))
       (53469 . ((23429 . "집 택, 자리 택, 살 택, 정할 택, 묘 구덩이 택")
                 (25799 . "가릴 택, 추릴 택, 차별할 택") (28580 . "못 택, 진펄 택, 진액 택, 윤택할 택, 비와 이슬 택, 늪 택")
                 25246 25321 27810 27901 20268 28898 30987 34828
                 40453 14447 15206 17737 17770))
       (53489 . ((25745 . "撐의 俗字, 버틸 탱") 31424 (25744 . "버틸 탱, 취할 탱, 헤칠 탱")
                 27160 27222 29274 27221 15313))
       (53552 . ((25860 . "펼칠 터, 흩을 터, 날칠 터, 비길 터, 헤칠 터")
                 25669))
       (53664 . ((22303 . "흙 토, 뿌리 토, 나라 토, 뭍 토")
                 (35342 . "다스릴 토, 칠 토, 찾을 토, 벨 토, 꾸짖을 토")
                 (21520 . "토할 토, 나올 토, 펼 토") (20814 . "兔의 俗字, 토끼 토")
                 20820 33759 22305 33556 35752 38029 22541 27746
                 33423 33717 37367 40309 15078))
       (53668 . (22136 (22091 . "啍과 同字") 29938 21837
		       30077 19823))
       (53672 . (13323))
       (53685 . ((36890 . "뚫릴 통, 사무칠 통, 형통할 통, 사귈 통, 다닐 통, 간음할 통")
                 (27934 . "구렁 통, 깊을 통, 빌 통, 밝을 통, 고을 통")
                 (32113 . "거느릴 통, 벼리 통, 실마디 통, 근본 통, 이을 통")
                 (30171 . "아플 통, 상할 통, 심할 통, 몹시 통") (24927 . "서러울 통, 애통할 통")
                 (31570 . "사통대 통, 통 통") (26742 . "엿되들이 통 통")
                 (24683 . "슬플 통, 아플 통, 앓을 통") 20375 15809
                 24965 (31593 . "대나무통 통, 낚시 통") 29093 32130
                 24696 32479 (27147 . "대 홈 통, 어름나무 통") 22005
                 25413 30156 34026 14554 14988 15558))
       (53748 . ((36864 . "물러갈 퇴, 겸양할 퇴, 물리칠 퇴")
                 26894 (38969 . "기울어질 퇴, 사나운 바람 퇴, 쇠할 퇴, 무너질 퇴, 대머리 퇴")
                 (22534 . "흙무더기 퇴 놓을 퇴, 언덕 퇴") (27084 . "몽둥이 퇴, 너스레 퇴, 누에시렁 퇴, 던질 퇴")
                 (33151 . "넓적다리 퇴") (37786 . "옥 다듬을 퇴") (35114 . "옷 벗을 퇴, 꽃빛 옅어질 퇴")
                 28958 25637 (38564 . "무너질 퇴, 미끄러질 퇴, 부드러울 퇴, 땅이름 퇴")
                 38973 39755 20691 34039 30205 38970 20416 23605
                 19147 39059 22422 22624 23903 24346 26712 29050
                 30931 31336 33070 34284 34312 34689 36422 36458
                 38510 38951 39613 40237 13645 14210 14236 14472
                 14662 14851 15459 15839 16316 16329 16343 16387
                 16847 17050 17384 18179 19002 19284))
       (53804 . ((25237 . "던질 투, 버릴 투, 줄 투, 의탁할 투")
                 (39722 . "싸움 투") (36879 . "통할 투, 사무칠 투, 지나칠 투")
                 (22871 . "전례 투, 덮게 투") 37742 (22956 . "투기할 투, 자식 없는 계집 투, 질투할 투")
                 (20600 . "훔칠 투, 엷을 투, 구차할 투, 인정머리없을 투")
                 (28189 . "빛 변할 투, 더러워질 투, 고을이름 투")
                 20599 38360 (22930 . "투기할 투, 미워할 투, 강샘할 투, 부러워할 투")
                 39725 (39600 . "주사위 투") 21731 25960 29263 31226
                 32240 34339 39718 39724 13691 13731 13968 14479
                 14999 18144 18348 18397 18693))
       (53825 . (20319))
       (53945 . ((29305 . "우뚝할 특, 숫소 특, 뛰어날 특, 다만 특, 세 살 먹은 짐승 특, 특별할 특")
                 (24925 . "간악할 특, 간사할 특, 더러울 특") (24530 . "변할 특, 의심할 특, 어기어질 특")
                 34723 34776 36003 38141 24529 33062 34478 37617
                 40207 14658 14841))
       (53960 . ((38358 . "엿볼") 38383))
       (54028 . ((30772 . "깨뜨릴 파, 다할 파, 갈라질 파")
                 (27874 . "물결 파, 물 젖을 파, 눈 광채 파, 움직일 파, 달빛 파")
                 (32631 . "파할 파, 내칠 파") (25773 . "심을 파, 펼칠 파, 버릴 파, 까불 파, 달아날 파")
                 (27966 . "물 갈래 파, 보낼 파, 나눌 파, 파벌 파, 물가닥 파")
                 (38935 . "비뚤어질 파, 자못 파") (25226 . "잡을 파, 헤칠 파")
                 (22369 . "언덕 파, 제방 파") (24052 . "꼬리 파, 뱀 파, 땅이름 파")
                 (23110 . "할머니 파, 너풀너풀 춤출 파, 할미 파")
                 (36315 . "절뚝발이 파, 절름발이 파") (29750 . "비파 파")
                 (33453 . "파초 파") (29228 . "긁을 파, 기어다닐 파")
                 (25850 . "열 파, 헤칠 파, 제거할 파, 손뼉칠 파")
                 38466 (26487 . "비파나무 파, 써레 파, 칼자루 파, 악기이름 파")
                 38712 (21493 . "못할 파, 드디어 파, 술잔 파") (24597 . "두려울 파")
                 (33897 . "꽃봉오리 파") (30372 . "흴 파, 배 불룩할 파, 귀 밑에 희끗희끗할 파")
                 22761 (24085 . "머리 동이수건 파") (31800 . "까부를 파, 키 파")
                 32610 (28766 . "물이름 파") 30116 (29627 . "파려옥 파, 유리옥 파")
                 24074 (31494 . "가시 대 파, 울타리 파") 38774 16777
                 (37169 . "땅이름 파") 23955 (22929 . "새앙머리 파, 갈래머리 파")
                 22459 21543 27419 37376 25670 22365 38063 38071
                 39047 (23717 . "산 비알질 파") (29240 . "아비 파") (32793 . "쇠시랑 파")
                 (33760 . "시금치 파") 21877 22041 22847 23552 23708
                 24349 27335 27734 28502 30341 30854 31544 31586
                 31653 31889 32038 32659 33317 33934 34082 34438
                 34494 34969 35666 35933 36276 36289 37461 37635
                 37763 39370 40067 13790 14097 14207 14222 14855
                 14959 15195 15917 16364 16635 17324 17329 17700
                 18267 18799 19442 19518 19861))
       (54032 . ((21028 . "판단할 판, 쪼갤 판, 맡을 판")
                 (26495 . "널조각 판, 널기와 판, 풍류이름 판") (29256 . "조각 판, 호적 판, 국경 판, 한 길 판, 인쇄할 판")
                 (36009 . "장사 판, 팔 판") (38442 . "산비탈 판, 언덕 판")
                 (36774 . "힘들일 판, 갖출 판") (22338 . "언덕 판, 산비탈 판, 순나라 서울이름 판")
                 (29923 . "외씨 판, 실과씨 판, 꽃잎 판") (37393 . "불린 금 판, 불린 금덩이 판")
                 27764 36775 26116 33320 34626 21150 36137 38051
                 23685 29930 30024 30469 31876 40293 14598))
       (54036 . ((20843 . "여덟 팔") (21485 . "나팔 팔, 입 벌릴 팔")
                 (25420 . "깨뜨릴 팔, 나눌 팔, 칠 팔") 25170 26419
                 27715 20168 21749 29584 37343 15213))
       (54047 . (24060))
       (54056 . ((35997 . "조개 패, 재물 패, 비단 패") (25943 . "헐어질 패, 무너질 패, 깨어질 패, 썪을 패, 패할 패")
                 (29260 . "방 붙일 패, 호패 패, 방패 패") (24726 . "거스를 패, 어지러울 패, 거슬릴 패")
                 (35207 . "霸의 俗字, 두목 패") (20329 . "패옥 패, 찰 패, 마음에 먹을 패, 물이 흘러서 합할 패")
                 (27803 . "둥둥 뜰 패, 비 쏟아질 패, 넉넉할 패, 비가쏟아지는모양 패")
                 (31255 . "돌피 패, 조그마할 패") (21764 . "염불소리 패, 노래부를 패")
                 (29437 . "이리 패") (28031 . "물이름 패, 물가 패") (38664 . "비 쏟아질 패")
                 (38712 . "으뜸 패, 패왕 패") (23387 . "혜성 패") (26054 . "기 패, 깃발 날릴 패")
                 29371 (29678 . "노리개 패, 찰 패") 35478 37046 26833
                 25453 15770 26046 30706 37093 21591 29384 36125
                 36133 38049 20226 23029 26113 26774 29292 29448
                 29692 31529 31738 31792 31930 33501 33591 34557
                 36097 37121 37575 39351 13605 13783 14045 14596
                 14825 15090 15560 15916 15933 17027 17840 18426
                 18585 19031 19067 19517 19549))
       (54077 . (24196 (24429 . "성씨 팽, 땅이름 팽, 방패 팽, 총소리 팽, 북소리 팽")
		       30844 (33192 . "배 불룩할 팽, 큰배 팽") (28558 . "물소리 팽, 물결 부딛는 형세 팽")
		       (28921 . "삶을 팽, 요리 팽, 삶는 팽") 20283 (31050 . "사당문 제사 팽")
		       38285 30896 (30768 . "여울 물소리 팽") (34778 . "蟛과 同字")
		       (34779 . "방게 팽") 22061 23917 24969 25533 26760
		       29967 30351 30801 30942 31261 36591 36643 37643
		       38351 39471 13407 14627 14840 15478 16158 16664
		       17515 17984 18763))
       (54085 . ((24846 . "사나울 퍅, 고집할 퍅")))
       (54200 . ((20415 . "편할 편, 소식 편, 익힐 편, 오줌 편, 문득 편")
                 (31687 . "책 편, 편차할 편") (29255 . "조각 편, 쪼갤 편, 성씨 편")
                 (36941 . "두루 편") (20559 . "치우칠 편, 무리 편, 오십명 편, 간사한 모양 편")
                 (32232 . "책편 편, 기록할 편, 첩지 편, 벌릴 편, 엮을 편")
                 (25153 . "특별할 편, 거룻배 편, 작을 편") (38829 . "채찍 편, 볼기채 편, 태장 편")
                 (39449 . "속일 편, 말에 뛰어 오를 편") (32745 . "훌쩍 날 편, 오락가락할 편, 훌쩍나는 모양 편")
                 24487 (35082 . "좁을 편, 급할 편, 옷이 몸에 낄 편")
                 29048 36417 26985 (33361 . "거룻배 편") (34649 . "박쥐 편")
                 21310 (35550 . "공교한 말 편") 26948 39934 39935
                 32527 32534 35869 39575 40138 (24828 . "조급할 편, 편협할 편")
                 (32246 . "꿰맬 편") 33849 22248 23205 25561 29265
                 29327 29493 30885 31663 35217 36021 37757 38952
                 39447 39800 40355 13554 15644 15908 16144 16299
                 17362 18530 19279))
       (54212 . ((36022 . "떨어질 폄, 덜릴 폄, 꺾을 폄, 귀양보낼 폄, 감할 폄")
                 (31366 . "하관할 폄, 광중 폄") (30765 . "돌침 폄") 36140))
       (54217 . ((24179 . "평탄할 평, 바를 평, 고를 평, 다스릴 평, 쉬울 평, 풍년 평, 화친할 평, 재관관 평, 평평할 평")
                 (35413 . "평론할 평, 헤아릴 평, 고칠 평, 기롱할 평")
                 (22378 . "벌판 평, 평평할 평, 평수 평") (33805 . "마름 평, 머구리밥 평, 개구리밥 평")
                 (26544 . "장기판 평, 바둑판 평, 회양목 평") (33529 . "맑은 대쑥 평, 사과 평")
                 (25256 . "탄핵할 평") (24614 . "맘 급할 평, 칼칼할 평")
                 29622 (27865 . "물소리 평, 물결 셀 평") 21615 18837
                 35780 40070 (34033 . "마름 평, 머구리밥 평, 개구리밥 평")
                 (39811 . "가자미 평") 21257 23000 23740 25340 32979
                 34482 38299 39373 14017 16056 17260 17617))
       (54224 . ((38281 . "닫을 폐, 마칠 폐, 가릴 폐") (24290 . "폐할 폐, 내칠 폐, 집 쏠릴 폐, 떨어질 폐")
                 (32954 . "허파 폐, 부아 폐, 친할 폐, 마음속 폐")
                 (24330 . "해질 폐, 폐단 폐, 곤할 폐, 곰곰 궁리할 폐, 결단할 폐, 엎드릴 폐")
                 (34109 . "가리울 폐, 다할 폐, 정할 폐, 가릴 폐")
                 (24163 . "폐백 폐, 예물 폐, 재물 폐, 돈 폐") (38491 . "대궐 섬돌 폐")
                 (25987 . "죽을 폐, 엎드러질 폐, 자빠질 폐") (21536 . "짖을 폐, 땅이름 폐")
                 (23318 . "사랑할 폐, 윗사람으로부터 사랑받을 폐, 벼슬이름 폐")
                 (29528 . "곤할 폐, 폐단 폐, 죽을 폐, 엎드러질 폐")
                 (25949 . "옷 해질 폐, 버릴 폐, 무너질 폐, 실패할 폐, 활줌통 폐")
                 24259 (30280 . "고질병 폐") 26478 (29428 . "짐승이름 폐, 옥 폐, 들개 폐")
                 26768 14498 24164 24223 27609 31621 38381 27360
                 29529 30761 34143 34572 37160 37950 38279 40739
                 13892 14400 15303 16305 17004 17760 18456 19415))
       (54252 . ((24067 . "베 포, 피륙 포, 벌릴 포, 돈 포, 베풀 포")
                 (25265 . "안을 포, 품을 포, 낄 포") (26292 . "사나울 포, 급할 포, 맨손으로 칠 포, 불끈 일어날 포, 마를 포, 상할 포")
                 (25429 . "사로잡을 포") (28006 . "물가 포, 갯가 포, 개 포")
                 (32990 . "태보 포, 한 배 포, 동포 포") (21253 . "꾸릴 포, 용납할 포, 더부룩히 날 포, 보따리 포, 쌀 포")
                 (39165 . "배부를 포, 물릴 포, 먹기 싫을 포, 흡족할 포")
                 (30770 . "큰 대포 포, 돌쇠뇌 포, 돌 튀기는 화살 포")
                 (25243 . "拋와 同字, 버릴 포") (37610 . "전방 포, 베풀 포, 문고리 포, 펼 포, 아플 포, 벌릴 포")
                 (36875 . "달아날 포, 포흠할 포, 도망갈 포") (24598 . "두려울 포, 놀래킬 포")
                 (39825 . "절인 생선 포, 성씨 포") (27873 . "물거품 포, 물이 솟아날 포, 물 흐를 포, 성할 포")
                 (33889 . "포도 포, 들머리 포, 나라이름 포") (35090 . "포장할 포, 옷 뒤길 포, 도포 포, 고을이름 포")
                 (34957 . "도포 포, 관디 포, 앞섶 포, 두루마기 포")
                 (33970 . "부들풀 포, 창포 포, 개버들 포") (22275 . "채마밭 포, 밭 갈 포, 곳 포, 산이름 포, 나물밭 포")
                 (33071 . "포 포, 말린 과실 포") (21754 . "씹어 먹을 포, 먹일 포")
                 (26333 . "볕 쬘 포, 볕 쏘일 포") (21261 . "엉금엉금 길 포, 엎드러질 포")
                 (21638 . "고함지를 포, 짐승소리 포, 불끈 성낼 포, 으르릉거릴 포")
                 (30129 . "부풀 포, 부르틀 포") (24214 . "푸줏간 포, 부엌 포")
                 (33502 . "그령풀 포, 밑둥 포, 초목이 다복히 날 포, 꾸러피 포, 꾸러미 포")
                 (20296 . "펼칠 포, 널리알릴 포") (21263 . "박 포, 바가지 포, 퉁소 포")
                 (28846 . "그슬릴 포, 싸서 구울 포, 형벌 포") (26209 . "신시 포, 해질 포")
                 (39188 . "저녁 곁두리 포, 먹을 포") 25291 37242 (33302 . "鋪의 俗字")
                 23474 31022 (28848 . "그슬 포, 까불거릴 포") 26553
                 (35495 . "꾀할 포, 클 포, 도울 포, 간할 포") 20772
                 35139 34403 22289 33969 40581 (38788 . "가방 포, 가죽 다로는 장인 포")
                 31523 (39878 . "매가리 포, 물아지 포") 24585 33068
                 33303 36305 40579 15565 21032 38072 38087 38138
                 39281 40077 40837 (26323 . "급할 포, 창졸 포") (37451 . "대패 포")
                 20328 (21241 . "싸다") 21656 22409 22484 22853 23394
                 23788 24239 24513 24721 25258 26600 29230 29389
                 29917 30384 31007 33762 33826 34475 34565 34956
                 35046 35100 35592 36595 37437 37988 38337 38496
                 38756 39665 40271 40621 40793 13877 14004 14015
                 15079 15141 15143 15329 15506 16940 16955 17031
                 17238 17282 17685 18124 18293 18710 19037 19197
                 19275 19346 19852))
       (54253 . ((26292 . "햇빛 쪼일 폭, 나타날 폭, 드러날 폭, 사나울 폭")
                 (29190 . "불터질 폭, 폭발할 폭") (24133 . "폭 폭, 겉치레할 폭")
                 (28689 . "폭포수 폭, 소나기 폭") (26333 . "햇볕에 말릴 폭")
                 (36667 . "바퀴살통 폭") 26323 17145 17215))
       (54271 . (21944))
       (54273 . (20051))
       (54364 . ((34920 . "겉 표, 윗옷 표, 밝을 표") (27161 . "표시할 표, 높은 가지 표, 적을 표, 나무 끝 표, 표할 표")
                 (31080 . "쪽지 표, 문서 표, 불날릴 표, 훌쩍 날 표, 날랠 표")
                 (28418 . "뜰 표, 움직일 표, 으스스할 표, 높이 훨훨 날 표")
                 (26451 . "북두자루 표, 북두자리 표") (35961 . "표범 표, 아롱범 표")
                 (39108 . "회오리바람 표, 나부낄 표, 떨어질 표, 회리바람 표")
                 (21117 . "표독할 표, 긁을 표, 겁박할 표, 끝 표, 찌를 표, 빠를 표")
                 (24915 . "급할 표") (29922 . "표주박 표") (20469 . "나누어 줄 표, 흩어질 표, 가마니 표")
                 (24426 . "칡범 표, 문채 표, 작은 범 표") (39491 . "날쌜 표, 누런 말 표")
                 (39111 . "회오리바람 표") (32313 . "옥빛 표, 훌쩍 날 표")
                 39110 (25725 . "칠 표, 두를 표, 찔러 밀칠 표, 가슴 만질 표")
                 (27533 . "굶어 죽을 표") (37987 . "말 재갈 표, 성할 표")
                 39095 37285 (35057 . "목수건 표") 28146 34280 (20676 . "몸이 가벼울 표, 진중치 못할 표")
                 (23254 . "가벼울 표, 날랠 표, 색에 빠질 표") 39112
                 39114 24431 29186 29451 31342 33176 34056 34232
                 35134 35620 39531 39762 16188 16762 26631 32549
                 38230 38259 39128 39129 39130 39561 39584 40148
                 (21217 . "겁탈할 표, 겁박할 표") (22028 . "휙 불 표, 몰아갈 표")
                 (29083 . "불똥 튈 표, 성낼 표, 붉을 표, 빛날 표")
                 (37858 . "칼날 표, 비수 표") (39647 . "머리털 희뜩희뜩할 표, 깃발 날릴 표, 머리 늘일 표, 머리털이길게드리워진모양 표")
                 (39998 . "부레 표") 20774 22658 23114 24150 24497
                 26074 27302 28398 28684 29154 29349 30253 30379
                 30623 30950 31739 32754 33087 33237 34741 35544
                 36102 37686 38365 39008 39081 39107 13840 14333
                 14354 14844 15123 15345 15353 15503 15729 15806
                 15998 16451 16493 17351 17720 17775 18035 18308
                 19389 19562 19753 19842))
       (54392 . (19212))
       (54408 . ((21697 . "뭇 품, 품수 품, 가지 품, 벼슬 차례 품")
                 (31263 . "여쭐 품, 줄 품, 받을 품, 성품 품, 품할 품")
                 (31104 . "여쭐 품, 사뢸 품, 받을 품, 성품 품") 27008))
       (54413 . ((39080 . "바람 풍, 흘레할 풍, 울릴 풍, 풍속 풍, 경치 풍, 위엄 풍, 병풍 풍, 모양 풍")
                 (35914 . "豐의 略字, 풍년 풍") (26963 . "단풍나무 풍, 신나무 풍")
                 (35575 . "비유로 간할 풍, 외울 풍, 욀 풍") (39342 . "벼슬이름 풍, 성씨 풍, 물귀신이름 풍, 고을이름 풍")
                 35920 20016 37190 37175 33873 28194 20561 20974
                 28739 34356 20911 26539 27811 30127 30748 35773
                 39118 (30219 . "두풍 풍, 미치광이 풍") 20732 20968
                 20972 22584 23543 26660 27282 27816 29000 29478
                 30904 38715 38730 39082 39116 40631 13373 13477
                 14374 17358 19780))
       (54588 . ((30382 . "가죽 피, 껍질 피, 성씨 피") (24444 . "저 피")
                 (34987 . "이불 피, 덥힐 피, 미칠 피, 나타날 피, 창피할 피, 덮을 피")
                 (36991 . "숨을 피, 어길 피, 싫어할 피, 피할 피")
                 (30130 . "피곤할 피, 나른할 피, 느른할 피") (25259 . "헤칠 피, 흩어질 피, 찢어질 피, 펼 피")
                 (36315 . "기울어 지게 설 피") (38466 . "기울어질 피, 치우칠 피")
                 36767 (35414 . "말 잘할 피, 간사할 피, 알소할 피")
                 26560 24084 39603 29395 37433 20298 (38785 . "말 가슴걸이 피, 안갑 피")
                 31195 15455 35792 38093 40079 18003 23617 (39666 . "머리쓰개 피, 첩지 피")
                 24630 26055 27669 29348 32052 32717 32794 34275
                 35180 35983 36017 37524 39821 13535 13802 14512
                 14882 15301 17378 17398))
       (54589 . ((33143 . "뜩 막힐 픽, 답답할 픽, 닭 홰치는 소리 픽")
                 29007))
       (54596 . ((24517 . "반드시 필, 오로지 필, 살필 필, 기약 필")
                 (31558 . "붓 필, 오랑캐 이름 필") (21305 . "짝 필, 둘 필, 한마리 필, 벗 필, 무리 필")
                 (30050 . "마칠 필, 토끼그물 필, 다할 필, 편지 필, 책 필, 고기 꿰 다는 나무 필")
                 (24380 . "도울 필, 도지개 필, 거듭 필") (27852 . "졸졸흐르는 모양")
                 (30091 . "짝 필, 끝 필") (33534 . "향기날") (29644 . "칼 장식 옥 필")
                 (39325 . "향내날 필") (36437 . "길 치울 필") (34045 . "휘추리 필, 약이름 필")
                 (38880 . "슬갑 필") (31731 . "대 사립문 필, 나무 실은 수레 필")
                 (35313 . "바람이 쌀쌀할 필, 대평수 필") (20310 . "점잔피울 필, 가득할 필")
                 39494 24379 (28397 . "샘 끓어 솟을 필") 27027 16118
                 (39388 . "살찐 말 필") (40303 . "갈가마귀 필") 29082
                 32967 34960 39158 18374 21716 27605 28375 31508
                 31578 33628 34385 36344 38091 40526 (21639 . "향기날 필")
                 (32636 . "토끼그물 필") (38840 . "슬갑 필") 21537 22006
                 22972 24387 24621 28535 32298 33481 35141 37042
                 37453 37838 39238 39763 39813 40196 40413 13526
                 14518 14520 15012 15295 15612 16107 16694 17368
                 17833 19035 19198 19227 19361))
       (54605 . ((20047 . "없을 핍, 옹색할 핍, 다할 핍, 모자랄 핍")
                 (36924 . "가까울 핍, 궁핍할 핍, 핍박학 핍, 핍박할 핍")
                 24842 (20586 . "핍박할 핍, 가까울 핍") 22978 40212))
       (54616 . ((19979 . "아래 하") (22799 . "여름 하") (27827 . "물 하")
                 (20309 . "어찌 하") (36032 . "축하할 하, 하례할 하")
                 (33655 . "연 하, 연꽃 하") (29781 . "티 하, 옥의 티 하")
                 (36944 . "멀 하") (38686 . "놀 하, 노을 하") (34662 . "새우 하, 개구리 하")
                 (21414 . "큰 집 하") (24264 . "큰 집 하") (39957 . "새우 하, 암고래 하")
                 (26160 . "是와 同字, 여름 하") (22151 . "놀라게 할 하")
                 22031 (32581 . "틈 하") 32256 (21568 . "입 벌릴 하")
                 (37724 . "목투구 하") 30577 35898 (28998 . "불사를 하")
                 39458 37868 (23688 . "휑뎅그렁할 하") 21735 25266
                 33743 13896 34430 36154 (25047 . "속일 하") (30229 . "이질 하")
                 19973 20023 21740 21996 22327 22803 30108 30859
                 30892 33424 34168 34964 35541 36206 37820 38293
                 38300 38800 39084 13767 13823 13924 15396 15418
                 15697 18445 19095 19159))
       (54617 . ((23416 . "배울 학") (40372 . "두루미 학") (34384 . "사나울 학, 혹독할 학")
                 (22737 . "골짜기 학") (35604 . "희롱할 학") (30247 . "학질 학")
                 23398 (30830 . "확실할 학") (28088 . "마를 학") 32751
                 (37085 . "고을 이름 학") 35319 (29410 . "오소리 학")
                 21955 29063 (40445 . "메까치 학") 28585 30672 13680
                 19709 30111 35857 40500 40548 (30380 . "흴 학") 21952
                 22171 22191 22726 23747 23976 25992 30283 30840
                 34842 35614 35976 38628 38735 40236 40366 40470
                 13665 15378 15749 16357 16706 17800 17803 18709
                 18955 19364))
       (54620 . ((38867 . "나라 이름 한, 한나라 한") (28450 . "한수 한")
                 (23506 . "찰 한") (38480 . "한정할 한") (38289 . "한가할 한")
                 (24680 . "한탄할 한, 한될 한") (26097 . "가물 한") (27735 . "땀 한")
                 (32752 . "날개 한, 벼슬이름 한") (37039 . "고을 이름 한")
                 (32597 . "드물 한") (24717 . "사나울 한, 모질 한") (28579 . "빨 한, 빨래할 한")
                 (38290 . "한가할 한, 겨를 한") (28698 . "넓고 큰 한, 넓고클 한")
                 (26751 . "막대기 한") 25182 (25421 . "막을 한") (38280 . "마을 문 한")
                 29408 20724 (26293 . "말릴 한") (40766 . "코골 한")
                 35253 29364 30647 29442 (20713 . "노할 한") 37015
                 39597 (23290 . "우아할 한") 29103 25738 40436 22462
                 25002 27212 15873 18142 18840 38386 40519 23092
                 27721 28938 38380 38889 (23291 . "우아할 한") (39419 . "사나운 말 한")
                 (40435 . "소리개 한") 20192 20343 20668 20847 21755
                 23080 23272 23677 26200 27258 28011 28566 30356
                 34002 34058 34525 34706 35963 35979 37356 37554
                 38615 40382 40444 40626 13455 14048 14462 14504
                 14987 15319 15526 15684 16450 16517 16936 17007
                 17232 17233 17327 17399 17613 17651 17791 17881
                 18780 18853 19367))
       (54624 . ((21106 . "벨 할") (36676 . "다스릴 할, 맡아볼 할")
                 34638 40672 20612 21180 (30606 . "애꾸눈 할") 40353
                 32278 33309 19160 36758 34829 20068 21968 25651
                 30792 30925 34715 37771 38868 40375 13595 13600
                 14188 15028 15275 16355 17763 17784 18838 18858
                 19876))
       (54632 . ((21688 . "모두 함, 다 함") (38519 . "빠질 함")
                 (21547 . "머금을 함") (20989 . "상자 함") (33382 . "싸움배")
                 (21898 . "소리 함, 고함칠 함") (28085 . "젖을 함") (37532 . "재갈 함")
                 (32216 . "봉할 함") (27323 . "우리 함, 난간 함") (40569 . "짤 함")
                 (21859 . "재갈 함, 머금을 함") (35572 . "화할 함") (33761 . "연봉오리 함")
                 38967 18911 21979 28106 20990 37262 (38366 . "범 소리 함")
                 38993 22277 15648 17663 25000 34904 35901 39328
                 17942 21765 27995 31592 (33695 . "꽃봉오리 함") 29696
                 29961 39805 14868 16029 18740 19874 34900 27099
                 32516 33328 38426 39301 (36702 . "함거 함") 22735
                 23074 26199 26770 27430 28559 28947 31982 33277
                 34559 34604 34605 35648 35907 35919 36657 37601
                 37646 38501 38980 39201 39723 39956 13543 13732
                 13829 13838 13850 14078 14287 14292 14647 15277
                 15354 15417 15792 16022 16168 16201 16900 17295
                 17412 17748 17812 17974 18416 18683 18753 18977
                 19010 19186 19556 19607 19843 19871 19872))
       (54633 . ((21512 . "합할 합, 같을 합, 짝 합, 대답할 합")
                 (38492 . "고을이름") (34532 . "조개 합, 두꺼비 합")
                 (30418 . "소반 뚜껑 합, 합 합") (38308 . "도장 합, 침방 합, 개구리소리 합, 규방 합")
                 (21704 . "한 모금 합, 마실 합") (38356 . "문짝 합, 닫을 합")
                 (30413 . "덮을 합, 합할 합, 어찌 아니할 합") (28312 . "문득 합, 이를 합, 의지할 합")
                 (27068 . "술그릇 합, 물통 합, 뚜껑 합") (37059 . "땅이름 합")
                 21623 35429 40255 (26585 . "우리 합, 향나무 합")
                 31701 (21260 . "기운 답답할 합") 14163 (21969 . "잔말할 합, 웃는 소리 합, 입 다물 합, 괘 이름 합")
                 23750 27441 28890 30407 39855 18580 38485 38422
                 39052 40509 20334 23751 24261 25966 29062 29672
                 31220 31917 32712 32800 38940 39803 13857 14714
                 15192 15408 16799 17806 17814 18117 18939 19854))
       (54637 . ((34892 . "시장 항, 항렬 항, 굳셀 항") (24658 . "늘 항, 옛 항, 항상 항")
                 (38477 . "항복할 항, 떨어질 항") (38917 . "목덜미 항, 클 항, 항목 항, 목뒤 항")
                 (25239 . "막을 항, 들 항, 겨룰 항, 항거할 항") (24055 . "거리 항, 마을 항, 복도 항, 내시 항, 골목 항")
                 (28207 . "물 갈라질 항, 항구 항, 뱃길 항") (33322 . "쌍배 항, 배질할 항")
                 (27782 . "큰 물 항, 진펀할 항, 이슬 항") (20130 . "목 항, 높을 항, 별이름 항, 기둥 항, 가릴 항")
                 (32568 . "항아리 항") (32923 . "똥구멍 항, 배 뚱뚱할 항, 항문 항")
                 (26477 . "건널 항, 거루 항, 고을이름 항, 모진 배 항")
                 (20233 . "강직할 항, 우뚝할 항, 배필 항, 겨룰 항, 짝 항")
                 (23022 . "항아 항, 달에 사는 미녀 항, 달 부르는 다른 이름 항, 계집이름 항")
                 (26689 . "수갑 항, 배다리 항, 횃대 항, 시렁 항")
                 (23270 . "항아") (24646 . "恒의 本字") 39599 (38927 . "목 항, 오르락 내리락할 항")
                 21549 38284 (28821 . "마를 항, 구울 항, 구들 항")
                 34902 (22831 . "멜 항, 다질 항") 20333 26043 38336
                 32942 38058 38390 39033 39043 (32575 . "항통 항, 문서함 항, 벙어리 항")
                 22594 23730 26649 31504 32577 33472 34466 36005
                 36333 36818 37023 37200 37415 38338 39783 13616
                 14303 14618 15392 16426 16496 17941 18230 18392
                 18621 18632 19635))
       (54644 . ((28023 . "바다 해, 많을 해, 세계 해") (23475 . "해할 해, 시기할 해, 죽일 해")
                 (35299 . "쪼갤 해, 꺠우칠 해, 풀 해, 괘 이름 해, 벗을 해")
                 (20133 . "열 둘째 지지 해, 서북쪽과 북쪽의 사이 방향 해, 음력 시월 해, 돼지 해")
                 (35442 . "갖출 해, 모두 해, 겸할 해, 마땅할 해, 그 해")
                 (22874 . "어찌 해, 종 해, 큰 배 해") (20565 . "함께할 해, 굳셀 해")
                 (39405 . "놀랄 해, 북 울릴 해") (39608 . "뼈 해") (35559 . "조화할 해, 글 해, 기롱지거리할 해, 고를 해")
                 (26999 . "해나무 해, 본뜰 해, 해자 해, 법 해") (25032 . "게으를 해")
                 (21683 . "기침 해, 침 뱉을 해, 방글방글 웃을 해")
                 (36994 . "우연히 만날 해") (23401 . "어린아이 해, 방글방글 웃을 해")
                 (22419 . "땅 가장자리 해, 햇수 해, 계단 해") (28707 . "찬이슬 해, 이슬 기운 해")
                 (34809 . "게 해") (37282 . "젓 해, 젓 담글 해, 젓 담그는 형벌 해")
                 (24296 . "관청 해, 공해 해") 35303 28581 (34212 . "염교 해, 부추 해")
                 (29548 . "신통한 양 해, 해갓 해") 29774 (27436 . "기침할 해, 크게 부를 해, 일깨울 해")
                 26192 38484 (21645 . "비웃을 해") 33604 34831 20357
                 39412 14629 (23984 . "골짝 사이 해, 골 이름 해")
                 30049 36037 (30158 . "학질 해, 하루거리 해") (39853 . "어채 해, 복 해")
                 23760 27294 33166 38896 14817 19109 35813 35856
                 36165 39055 39559 40081 (38950 . "턱 해") 21992 22640
                 22926 23007 24896 27686 32111 32370 33010 34210
                 35295 35941 36044 36614 37067 37244 39168 39258
                 40868 13472 13800 13936 13962 14142 14226 15393
                 15399 15401 15484 16258 16300 16429 16843 16975
                 18104 18489 18493 18529 18831 19408 19578 19602))
       (54645 . ((26680 . "실과 핵, 자세할 핵, 씨 핵") (21182 . "캐물을 핵, 핵실할 핵, 죄를캐물 핵")
                 (35208 . "핵실할 핵, 겨무거리 핵, 씨 핵, 액색할 핵")
                 (32750 . "깃촉 핵, 쭉지 핵, 칼 갓 핵") 26678 21183
                 16594))
       (54665 . ((34892 . "다닐 행, 길귀신 행, 오행 행, 그릇 얄긋거릴 행, 행서 행, 행실 행, 항렬 행")
                 (24184 . "다행할 행, 바랄 행") (26447 . "살구 행, 은행 행")
                 (20502 . "요행 행, 괴일 행, 아당할 행, 친할 행, 사랑할 행")
                 (33607 . "마름풀 행, 조아기 행") (24763 . "발끈 성낼 행")
                 23134 28076 31573 16172 32471 40507 21720 21832
                 22451 22588 27920 28870 32078 32200 33019 33685
                 35012 40244 40262 64040 13521 16452 16532 17655
                 18157 19410 19490))
       (54693 . ((21521 . "향할 향, 나아갈 향, 북쪽 창 향, 기울어질 향")
                 (37141 . "시골 향, 고향 향") (39321 . "향내 향, 약이름 향, 향기 향")
                 (20139 . "드릴 향, 제사지낼 향, 잔치 향, 흠향할 향")
                 (38911 . "소리 울릴 향, 소리 마주칠 향, 악기 향")
                 (39255 . "잔치할 향, 흠향할 향") (22190 . "향할 향, 누릴 향, 울릴 향, 받을 향")
                 (39177 . "먹일 향, 군량 향, 점심 향") (29670 . "옥이름")
                 37130 (34188 . "곡기 향, 향기 향, 기장 향") 26193
                 26319 37111 34817 33207 37129 37508 40600 18867
                 20065 21709 33431 39144 39287 20143 23008 27007
                 31269 33835 34435 40028 40641 13695 13821 15649
                 16349 16744 17041 17838 18861))
       (54728 . ((35377 . "허락할 허, 나아갈 허, 기약할 허, 여럿이 힘쓰는 소리 허")
                 (34395 . "빌 허, 헛될 허, 다할 허, 약할 허, 거짓 허, 하늘 허")
                 (22687 . "옛 터 허, 장 허, 큰 두덕 허, 옛성터 허")
                 (22099 . "불 허, 뿜을 허, 탄식할 허") 34394 35425
                 (27476 . "한숨 쉴 허, 흑흑 느껴 울 허, 코로 숨 내쉴 허")
                 26665 34391 14346 39766 22040 27986 35768 24968
                 37158 39497 39804 14256 17683))
       (54732 . ((25010 . "법 헌, 표준될 헌, 고시할 헌, 기뻐할 헌")
                 (29563 . "바칠 헌, 음식 헌, 어진이 헌, 드릴 헌")
                 (36562 . "초헌 헌, 껄껄 웃을 헌, 주적댈 헌, 추녀끝 헌, 고기 굵게 저밀 헌")
                 (27382 . "나무이름") 29486 25472 24024 28695 (24026 . "봉우리 헌, 시루봉 헌")
                 (24176 . "수레 안장 헌, 휘장 헌") 23466 36713 (25863 . "비길 헌, 물건 맬 헌")
                 20186 20321 25631 14757 16813 17926 18210 18926))
       (54736 . ((27463 . "쉴 헐, 나른할 헐, 다할 헐, 으슥할 헐, 스러질 헐, 흩어질 헐")
                 34638 28362 34829))
       (54744 . ((38570 . "험할 험, 험난할 험") (39511 . "시험할 험, 중험할 험, 보람 험")
                 25016 (23982 . "산 가파를 험") (29569 . "獫과 同字")
                 (29547 . "험윤 오랑캐 험") 30839 23812 29443 38505
                 38522 39443 39564 23021 24570 26484 27011 30982
                 35683 39440 15871 15901 16332))
       (54785 . ((38761 . "가죽 혁, 고칠 혁, 갑주 혁") (36203 . "불 이글이글할 혁, 빛날 혁, 성할 혁, 밝을 혁")
                 (29184 . "불빛 혁, 빛날 혁") (22869 . "아름다울 혁, 클 혁, 차례 혁, 바둑 혁, 빛날 혁, 근심할 혁")
                 (22151 . "노할 혁, 꾸짖을 혁") (27947 . "넘칠 혁, 빌 혁, 봇도랑 혁")
                 34891 (39721 . "송사할 혁, 싸울 혁") (24328 . "바둑둘 혁")
                 34409 20368 28931 28977 27528 36201 15836 21523
                 38411 20277 25415 28994 30434 30593 35236 36197
                 36248 14648 14758 15144 15261 15846 17607 18050
                 18091 18839 19547 19825))
       (54788 . ((35211 . "나타날 현, 드러날 현, 있을 현, 보일 현, 볼 현")
                 (29694 . "나타날 현, 지금 현, 옥빛 현") (36066 . "어질 현, 좋을 현, 구멍 현")
                 (39023 . "나타날 현, 밝을 현, 통달할 현, 높을 현")
                 (29572 . "검을 현, 검붉을 현, 하늘 현, 아득할 현, 고요할 현")
                 (32291 . "매달릴 현, 끊어질 현, 고을 현") (25080 . "매달 현, 멀 현")
                 (32067 . "줄풍류 현") (24358 . "시위 현, 반달 현, 맥박 잦을 현, 땅이름 현, 활시위 현")
                 (37449 . "솥귀 현") (23796 . "고개 현, 산이름 현") (28843 . "밝을 현, 불빛 현")
                 (27957 . "멀 현") (30505 . "아찔할 현, 현혹할 현") (34898 . "자랑할 현, 팔릴 현")
                 (32098 . "채색무늬 현, 수놓을 현") (26203 . "햇발 현, 볕기운 현")
                 (20436 . "연탐할 현, 풍세보는 깃 현, 정탐할 현")
                 (29625 . "옥돌") (27883 . "물 깊을 현, 이슬 맺힐 현, 눈물 줄줄 흘릴 현")
                 (30541 . "물끄러미 볼 현, 고울 현") (33335 . "뱃전 현")
                 38997 34984 (20743 . "영리할 현, 빠를 현") (23323 . "산뜻할 현")
                 28774 25979 (26145 . "햇빛 현") 35490 31013 36114
                 29700 30516 (32367 . "얽힐 현, 둘릴 현, 맬 현") 22455
                 33703 15118 (30147 . "힘줄 당기는 병 현, 적병 현")
                 31591 21743 22477 32984 34495 34825 37607 37990
                 18894 (30476 . "고을 현, 지방 현") 31046 20259 21439
                 23704 24748 26174 29616 32474 32563 33483 34476
                 36132 38089 (32766 . "파뜩파뜩 날 현, 경박할 현, 영리할 현, 급할 현")
                 (34566 . "가막조개 현, 호수이름 현") (37591 . "노구솥 현, 퉁노구 현")
                 (39421 . "돗총이 현, 철총이 현") 20269 21599 21662
                 22966 23050 23097 23153 24370 24624 24804 26763
                 30670 33236 33252 34404 35374 35434 35453 35512
                 35678 35714 36121 38809 38853 13835 14409 14522
                 14795 14822 15083 15225 15969 16121 16399 17239
                 17401 17918 18095 18169 18280 18286 18794 18891
                 19033 19332 19643))
       (54792 . ((34880 . "피 혈, 피붙이 혈, 씩씩할 혈")
                 (31348 . "굴 혈, 움 혈, 틈 혈, 굿 혈") (23377 . "외로울 혈, 나머지 혈, 창갈구리 혈")
                 (38913 . "머리 혈, 마리 혈, 페이지(page) 혈") (32092 . "헤아릴 혈, 묶을 혈")
                 27884 (36240 . "새가 떼로 날 혈, 나아갈 혈") 34658
                 32243 16823 18855 39029 21559 22393 22858 23054
                 23716 26710 28885 30118 33555 35015 13334 16797
                 17097 17468 17592 17781 18045 18126 18833 19044
                 19047 19196))
       (54800 . ((23244 . "싫어할 혐, 의심할 혐, 유쾌하지 않을 혐")
                 39334 16318 40760 16012 16208 19788))
       (54801 . ((21332 . "조화할 협, 맞을 협, 도울 협")
                 (33029 . "갈빗대 협, 위협할 협") (29433 . "좁을 협")
                 (23805 . "물 낀 두메 협, 산이름 협") (38492 . "좁을 협")
                 (25406 . "낄 협, 도울 협, 품을 협") (20448 . "의기 협, 사이에 낄 협, 협기 협")
                 (38960 . "뺨 협, 천천히 말할 협") (22846 . "곁 협, 낄 협, 잡을 협, 성씨 협, 가까울 협")
                 (28025 . "사무칠 협, 둘릴 협") (33031 . "脅과 同字, 갈비 협")
                 (37583 . "칼 손잡이 협, 집개 협, 집게 협") (33698 . "콩 꼬투리 협, 명협 협")
                 (21494 . "화합할 협") (24650 . "겁낼 협, 위협할 협")
                 (24860 . "쾌할 협, 뜻에 맞을 협") 31083 (31691 . "상자 협, 행담 협")
                 23777 24812 24719 18289 31604 34554 21158 20926
                 21232 33061 30820 38527 25337 29421 20384 21327
                 22841 25375 27971 30806 31655 32961 33626 34545
                 38103 38122 39050 39288 16882 (21287 . "옷상자 협, 꿰맬 협")
                 22473 21963 22437 24886 26090 26780 29057 29170
                 32138 33035 37503 38792 38956 39172 13917 14029
                 14517 14642 14694 14711 15241 15800 16170 16224
                 16284 16339 16459 17385 17390 17755 19205 19869))
       (54805 . ((20804 . "맏이 형") (24418 . "모양 형, 몸 형, 형상 형")
                 (21009 . "형벌 형, 죽일 형") (20136 . "형통할 형, 제사 형")
                 (34722 . "개똥벌레 형, 반딧불 형") (34913 . "저울대 형")
                 (28847 . "빛날 형, 밝을 형") (39336 . "향기 형, 향기로울 형")
                 (37026 . "나라 이름 형, 땅 이름 형") (28677 . "맑을 형, 개천 형, 물맑을 형")
                 (22411 . "거푸집 형, 본보기 형") (33610 . "모형나무 형, 곤장 형, 가시 형")
                 (29074 . "등불 형, 밝을 형") (29801 . "밝을 형") (27842 . "멀 형, 깊고 넓은 모양 형, 차가운 형, 찰 형")
                 (29673 . "노리개 형, 갓끈 형") (36872 . "멀 형, 빛날 형")
                 28913 (26689 . "시렁 형, 마개 형") (28366 . "실개천 형, 물결 일 형, 맑을 형")
                 (37795 . "줄 형, 꾸밀 형") (28752 . "물 이름 형") (36837 . "막을 형, 차단할 형")
                 33606 (35415 . "염탐할 형, 구할 형") 37494 (22800 . "멀 형, 바라보는 모양 형")
                 30798 28027 34309 37065 (38488 . "지레목 형, 비탈 형")
                 33053 14496 20352 28069 16062 16134 27894 28386
                 28486 33637 33639 33828 34021 35783 38095 38473
                 (23065 . "여관 이름 형, 예쁠 형") 21756 22355 23908
                 24729 25828 28704 34240 34509 34549 35118 36079
                 37598 40337 13353 13533 14014 14556 15202 15311
                 16155 16765 18072 18587 18630 18735 18736 19673))
       (54812 . ((24800 . "은혜 혜, 사랑할 혜") (24935 . "슬기로울 혜, 총명할 혜, 지혜 혜")
                 (20846 . "어조사 혜") (24407 . "비 혜, 꼬리별 혜") (37295 . "초 혜, 위대로울 혜, 식혜 혜")
                 (38795 . "신 혜, 가죽신 혜") (36426 . "지름길 혜, 건널 혜")
                 (34137 . "혜초 혜, 난초 혜") (26291 . "잔별") 24693
                 (30459 . "흘겨볼 혜, 돌아볼 혜") (23879 . "산이름")
                 (24495 . "기다릴 혜, 사잇길 혜") (27109 . "널 혜, 작은 관 혜")
                 40759 (20626 . "묶을 혜") (22034 . "가냘플 혜") 38837
                 28499 23878 27230 34794 35667 35904 22166 24979
                 14094 18103 18917 (23533 . "밝힐 혜, 깨달을 혜")
                 (35601 . "창피줄 혜") (35711 . "슬기로울 혜") 20705
                 (21304 . "감추다") 22177 24538 25075 27069 27200
                 27352 29860 33973 34695 35951 37880 38882 39473
                 13979 14691 14873 14952 15145 15397 15423 16381
                 17306 17546 17998 18122 18177 18696 19171 19626))
       (54840 . ((22909 . "좋을 호, 옳을 호") (25142 . "지게 호")
                 (34399 . "울부짖을 호, 부르짖을 호") (28246 . "호수 호")
                 (20046 . "인가 호, 어조사 호") (21628 . "부를 호, 숨 내쉴 호")
                 (34382 . "범 호") (32993 . "턱밑살 호, 오랑캐 호") (35703 . "보호할 호, 감쌀 호, 호위할 호")
                 (27627 . "가는 털 호") (28009 . "클 호") (35946 . "호걸 호, 귀인 호")
                 (20114 . "서로 호, 함께 호") (25160 . "뒤따를 호, 호위할 호")
                 (37804 . "호경 호") (30355 . "흴 호, 깨끗할 호") (31946 . "풀 호, 끈끈할 호, 풀칠할 호")
                 (26122 . "하늘 호, 큰 모양 호, 여름하늘 호") (28640 . "해자 호, 강 이름 호, 고을이름 호")
                 (31068 . "복 호") (26215 . "밝을 호, 빛날 호, 해돋을 호")
                 (28564 . "浩와 同字, 채색빛날 호") (22741 . "해자 호, 성밑해자 호")
                 (29392 . "여우 호") (29786 . "산호 호") (24359 . "활 호, 나무활 호")
                 (29733 . "호박 호") (22778 . "병 호, 단지 호, 항아리 호")
                 27073 (33983 . "쑥 호, 향기날 호, 지칠 호, 다북쑥 호")
                 (28649 . "퍼질 호") (33446 . "芐와 同字, 蔖의 俗字, 갈대 호")
                 (28765 . "넓을 호, 콩 삶은 물 호, 물줄기멀 호") (29920 . "표주박 호, 단지 호")
                 (32286 . "명주 호, 흴 호, 흰비단 호") (39013 . "클 호, 빛나는 모양 호")
                 (23733 . "산 호, 산초목 호") (34676 . "나비") (28408 . "물가 호")
                 (38912 . "구할 호, 지킬 호, 풍류이름 호") (28111 . "맑을 호")
                 (33899 . "마늘 호, 조롱박 호, 굵은 마늘 호") 21495
                 (29330 . "호궤할 호") 25143 14783 (24601 . "믿을 호")
                 (27789 . "冱의 譌字") 26120 30373 (31630 . "긴 대 호")
                 20053 17281 34836 (39212 . "기식할 호") 37143 (30366 . "밝을 호, 흴 호")
                 21804 30369 (37264 . "제호 호") (22071 . "짖을 호, 외칠 호")
                 (28360 . "장마 호") 22775 26285 34390 37152 (20913 . "찰 호, 얼 호, 막을 호")
                 (23278 . "아름다울 호, 시기할 호") (28396 . "강 이름 호")
                 29490 22033 26972 33528 17782 18109 (30364 . "흴 호, 흰 모양 호")
                 24077 26276 26317 26513 28409 31815 33792 35644
                 35705 37695 17024 22774 25252 27818 28751 32543
                 36727 38224 39074 40533 (20779 . "영웅 호") 23277
                 (29474 . "원숭이 호") (32853 . "들릴 호, 긴 귀 호")
                 (39693 . "수염 호") 20042 20624 20916 21728 21910
                 21989 22097 22158 22400 22832 23209 23804 24342
                 24655 24718 25144 25149 25698 26150 26534 26883
                 27124 27473 27864 28709 28864 29043 29097 29510
                 29515 29524 29939 31670 31748 31888 32148 33326
                 34096 34179 34181 34243 (34381 . "범") 34397 34906
                 36580 37417 37752 38608 38653 39793 39921 39959
                 39965 40184 40344 40358 13638 13765 13937 13967
                 13994 14012 14144 14267 14361 14551 14726 14893
                 14941 15124 15158 15323 15686 15910 16304 16856
                 17023 17082 17088 17094 17447 17589 17858 18224
                 18270 18277 18450 18723 18906 18923 19004 19060
                 19133 19276 19277 19435 19495 19747))
       (54841 . ((25110 . "혹 혹") (24785 . "미혹할 혹") (37239 . "독할 혹, 혹독할 혹")
                 (40288 . "고니 혹") 35952 25501 37721 38276 38966
                 21952 14767 15844 16009 16199 18353))
       (54844 . ((28151 . "섞을 혼, 섞일 혼") (23130 . "혼인할 혼")
                 (26127 . "어두울 혼") (39746 . "넋 혼") (28222 . "흐릴 혼")
                 28999 (29759 . "아름다운옥") (28343 . "어지러울 혼")
                 (38333 . "문지기 혼") 30568 (24795 . "흐릴 혼") (28956 . "빛날 혼")
                 24829 30575 25485 28259 39926 40754 19487 25381
                 27985 29682 38413 39300 (22274 . "뒷간 혼") 20529
                 22554 23147 24566 26156 26836 27545 28093 28957
                 30567 32247 32329 35304 39195 13422 13735 14064
                 15279 15466 16729 17040 17308 17418 18080 18160
                 18531 18928 19357 19767))
       (54848 . ((24573 . "소흘히 할 홀, 홀연 홀") (31503 . "홀 홀")
                 (24794 . "황홀할 홀") 32087 26358 33460 14846 19213
                 38186 (22251 . "온전할 홀") 20069 21282 21291 21823
                 21842 23523 37699 13748 14830 14881 15607 16000
                 16491 17636 18246 18970 19024 19759))
       (54861 . ((32005 . "붉을 홍") (27946 . "큰물 홍, 넓을 홍")
                 (24344 . "넓을 홍, 클 홍") (40251 . "큰 기러기 홍")
                 (27859 . "깊을 홍, 물깊을 홍") (35340 . "무너질 홍, 무너뜨릴 홍")
                 (34425 . "무지개 홍") (21700 . "떠들썩할 홍") (28888 . "횃불 홍")
                 (27742 . "수은 홍") 27930 38311 (39720 . "싸울 홍")
                 (28562 . "수은 홍") 33874 22917 21789 32418 33645
                 35751 40511 (26190 . "날 밝으려 할 홍") (31690 . "홈통 홍")
                 (37495 . "돌쇠뇌 홍") 20188 21503 21962 22444 23042
                 23672 28209 28482 28788 28962 30804 31904 33520
                 33875 34171 35900 35902 37566 38672 38687 39775
                 13715 13894 15156 15801 16423 16555 16596 16810
                 17018 18219 18321 18886 18958 19110 19193 19624))
       (54868 . ((28779 . "불 화") (33457 . "꽃 화") (21270 . "될 화")
                 (35441 . "말할 화, 이야기 화") (21644 . "화할 화") (30069 . "그림 화")
                 (33775 . "꽃 화, 빛날 화") (36008 . "재화 화, 재물 화")
                 (31117 . "재화 화") (31166 . "벼 화") (38772 . "신 화")
                 (27194 . "자작나무 화, 벗나무 화") (23301 . "여자 이름 화, 탐스러울 화")
                 (35649 . "시끄러울 화, 떠들썩할 화") 27159 (30059 . "그림 화")
                 30011 22821 25837 (22057 . "譁와 同字") 28565 (39498 . "준마 화")
                 27116 22711 38846 21642 37877 (20464 . "화할 화")
                 20249 26951 29837 35384 14351 21010 21326 21719
                 26726 31096 35805 36135 38060 38119 39557 40160
                 (23154 . "예쁠 화") (40866 . "풍류 조화될 화") 21529
                 21853 22486 22843 22976 23816 23819 24786 25702
                 26084 26489 26528 26567 26907 27790 (28780 . "불화발")
                 30409 30836 31477 31936 33305 33458 33794 34128
                 35502 35694 37033 37355 37413 37452 37685 39780
                 40047 40424 13670 13822 14278 14730 15177 15608
                 15777 16640 16767 17577 17698 18145))
       (54869 . ((30906 . "굳을 확, 확실할 확") (25844 . "넓힐 확")
                 (31339 . "벼 벨 확, 곡식거둘 확") (24275 . "둘레 확, 클 확")
                 28649 (25899 . "붙잡을 확, 움킬 확") (30907 . "굳을 확, 확실할 확")
                 (37962 . "가마 확") 30830 28088 18981 38252 38616
                 25837 30705 (30669 . "두리번거릴 확") 34838 31021
                 24397 25092 24393 25865 29571 33243 33383 38723
                 24416 25193 25313 40561 30689 21136 23347 24399
                 24415 25348 26340 27316 29236 29889 30662 31703
                 31814 31856 32815 33234 33958 35996 38697 40460
                 14363 14748 14895 15121 15142 15893 16034 16488
                 16516 16901 16991))
       (54872 . ((24739 . "근심 환") (27489 . "기뻐할 환") (36996 . "돌아올 환")
                 (25563 . "바꿀 환") (20024 . "알 환, 둥글 환") (29872 . "고리 환, 옥고리 환")
                 (21914 . "부를 환") (29029 . "불꽃 환, 빛날 환") (26707 . "푯말 환, 굳셀 환")
                 (24187 . "변할 환, 허깨비 환") (23462 . "벼슬 환, 환관 환")
                 (39973 . "환어 환, 홀아비 환") (39529 . "기뻐할 환, 환말 환")
                 (28185 . "흩어질 환, 물성할 환") (32008 . "흰 비단 환")
                 (22864 . "빛날 환, 클 환") (26213 . "환할 환") (25085 . "기뻐할 환")
                 (22300 . "두를 환") 36909 (23536 . "기내 환") 30287
                 (35938 . "기를 환") (39711 . "쪽찐 머리 환") 23503 29773
                 16800 (25808 . "입을 환") 35737 40030 (36696 . "환형 환")
                 28470 15114 (29915 . "옥홀 환") (30534 . "가득 차 있는 모양 환")
                 (37744 . "무게 단위 환") 27725 38372 (32089 . "끈목 환")
                 (37942 . "고리 환") 28596 32367 24366 33412 15111
                 32132 18090 18260 21796 22850 25442 27426 27475
                 28067 28949 32424 36824 38206 38254 38427 40105
                 40143 22456 (30358 . "샛별 환") 21915 23768 24844
                 25025 25868 26245 26777 27125 28873 29361 29566
                 30533 30627 31490 31979 32678 32914 33601 33800
                 33823 34279 35995 37188 37371 37582 38600 39879
                 39936 40197 40269 40558 13653 14047 14221 14450
                 15025 15758 15957 15958 16362 16403 16501 17274
                 17563 17583 17901 18216 18272 18441 18849 19316
                 19416 19721 19723 19743))
       (54876 . ((27963 . "살 활") (38346 . "트일 활, 넓을 활")
                 (28369 . "미끄러울 활, 어지러울 활") (29502 . "교활할 활")
                 (35905 . "뚫린 골 활, 골짜기 활") 28662 16424 25932
                 38420 20344 (34526 . "괄태충 활") 22895 23009 30483
                 30918 31214 31219 33855 34710 36239 13513 13682
                 15688 16646 16657 18850 19579 19763))
       (54889 . ((40643 . "누를 황") (30343 . "임금 황") (27841 . "하물며 황")
                 (33618 . "거칠 황") (28361 . "물 깊고 넓을 황") (24908 . "어렴풋할 황, 어리둥절할 황")
                 (24488 . "노닐 황, 방황할 황") (26179 . "밝을 황, 햇빛 황")
                 (24822 . "두려워할 황, 두려울 황") (36945 . "허둥거릴 황, 급할 황")
                 (29004 . "빛날 황, 환히밝을 황") (20976 . "봉황새 황, 암봉황새 황")
                 (28514 . "웅덩이 황") (29852 . "서옥 황, 반둥근패옥 황")
                 (34647 . "누리 황, 메뚜기 황") (27045 . "책상 황") (38541 . "해자 황, 산골짜기 황")
                 (31681 . "대숲 황, 대밭 황") (24880 . "밝을 황, 마음밝을 황")
                 (31783 . "혀 황, 생황 황") (24653 . "황홀할 황") (24140 . "휘장 황")
                 (28255 . "해자 황, 물결빨리내려가는 모양 황") (26180 . "晃과 同字, 밝을 황")
                 20917 (24627 . "멍할 황") (36026 . "줄 황") 40644 (32915 . "명치 끝 황")
                 17177 30365 29119 (21924 . "어린아이 울음 황") (37796 . "종소리 황")
                 20575 33358 35594 29056 14427 30272 31316 32296
                 (29789 . "옥 소리 황") 22708 25560 34881 35428 16476
                 35854 36150 40135 (23187 . "어머니 황") 20836 22573
                 22595 22875 23482 23858 24031 26306 26394 27003
                 27342 28862 29530 30377 30486 33887 34789 35579
                 36266 36582 38905 39213 39452 39945 40017 40428
                 14377 14389 14403 14406 14580 14650 14890 15163
                 15681 15746 16302 16430 16659 16739 17047 17059
                 17279 17440 17461 17503 18361 19076 19378 19688))
       (54896 . (32619 22118 32765 21717 22119 28549 32729))
       (54924 . ((22238 . "돌 회, 돌아올 회") (26371 . "모일 회, 모을 회")
                 (24724 . "뉘우칠 회") (25079 . "품을 회") (28142 . "강 이름 회, 물이름 회")
                 (24674 . "넓을 회, 클 회") (28784 . "재 회") (24315 . "돌 회, 돌아올 회")
                 (27292 . "노송나무 회") (33214 . "회 회, 회칠 회") 28618
                 (26214 . "그믐 회") (24458 . "노닐 회, 배회할 회") (35496 . "가르칠 회")
                 (36036 . "뇌물 회, 재물 회") (32362 . "그림 회") (34516 . "거위 회")
                 (28590 . "봇도랑 회, 우물도랑 회") (21295 . "물돌 회, 물이돌아가는 회")
                 (29546 . "교활할 회, 간교할 회") (33588 . "회향풀 회")
                 28239 20250 25040 22252 (36852 . "回와 同字") 31148
                 29177 13433 (27908 . "거슬러 올라 갈 회") 34184 (35452 . "조롱할 회")
                 (40032 . "회 회") (20330 . "어정거릴 회") 35927 32354
                 (28377 . "匯와 同字") 38368 (38958 . "세수할 회") 27820
                 (30420 . "바리 회") 22232 36901 28708 24667 29871
                 34517 34717 35106 37174 21053 26727 27981 28905
                 29423 32117 32472 33037 33631 35801 35826 36159
                 37072 39058 40089 14815 (26627 . "상수리나무 회")
                 21130 21684 23314 23671 24317 25339 25811 25980
                 26468 27376 27907 28899 28920 29172 30160 30650
                 30808 32818 34289 34361 34456 34582 35121 35439
                 35545 37932 38759 39856 13457 14131 14208 14247
                 14370 14555 14884 15473 15812 15815 16069 16303
                 17735 17846 18139 18187 18245 18247 18568 18727
                 19016 19181 19293 19740 19827))
       (54925 . ((30069 . "畫의 俗字") (29554 . "얻을 획") (21123 . "쪼갤 획, 그을 획")
                 30789 (30059 . "그을 획") 25837 35618 25782 32355
                 13810 33719 23155 (22148 . "외칠 획") 21663 22207
                 23295 28273 30729 31394 35595 35735 38852 39454
                 13552 13738 14734 14919 17464 18845 19101 19209))
       (54945 . ((27243 . "가로 횡") (34216 . "많을 횡") (23446 . "집 울릴 횡, 평안할 횡")
                 (37892 . "종 횡, 큰 종 횡") 27178 (40652 . "글방 횡")
                 37584 32045 31441 (37404 . "쇳소리 횡") 40649 (28555 . "물이 빙 돌 횡")
                 21552 22173 23797 24395 25544 25748 35897 36663
                 37957 14508 15751 17236 17301 19194 19222 19229))
       (54952 . ((23389 . "효도 효, 효도할 효") (25928 . "본받을 효")
                 (26313 . "새벽 효") (37237 . "삭일 효, 술괼 효") (29243 . "효 효, 괘이름 효")
                 (22150 . "울릴 효, 부르짖을 효") (21742 . "으르렁거릴 효, 성낼 효")
                 (26783 . "올배미 효, 올빼미 효") (32948 . "안주 효")
                 (28102 . "뒤섞일 효, 흙탕칠 효") (39501 . "날랠 효")
                 (25989 . "가르칠 효, 깨우칠 효") (28045 . "성 효, 물가 효")
                 21177 (22210 . "시끄럽다 효") 26241 22069 26549 22032
                 (20634 . "본받을 효") (27581 . "섞일 효") 40222 27466
                 (23844 . "산 이름 효") 25990 24994 27594 (29063 . "불김 효")
                 (30363 . "나타날 효") (34387 . "울부짖을 효") 22027
                 (39194 . "반찬 효") 29538 31562 39623 21715 22179
                 26195 26541 33014 39553 40494 20414 20466 20484
                 21626 22203 23115 23471 24232 25772 25961 27944
                 28391 28785 28786 29447 30170 30370 30819 31320
                 31385 31493 33198 34272 34376 35364 35432 35487
                 35509 37097 39012 39632 64039 13690 13984 14602
                 14957 15092 15219 15233 15743 15986 16039 16538
                 17008 17090 17565 17620 17767 17792 18805 19166
                 19224))
       (54980 . ((24460 . "뒤 후") (21402 . "두터울 후") (20399 . "과녁 후, 벼슬이름 후")
                 (21518 . "임금 후") (20505 . "물을 후, 기후 후") (21897 . "목구멍 후")
                 (21564 . "울 후, 우는소리 후") (26429 . "섞을 후, 썩을 후")
                 (36869 . "만날 후") (21957 . "맡을 후, 냄새맡을 후")
                 (29030 . "따뜻하게 할 후, 뜨거울 후") (24127 . "과녁 후")
                 (29661 . "옥 이름 후") 35436 28088 (22421 . "厚의 古字")
                 31943 (35425 . "자랑할 후") (37207 . "주정할 후") (29492 . "원숭이 후")
                 (22560 . "봉화대 후") (21923 . "불 후") 40773 (39217 . "건량 후")
                 (31692 . "공후 후") 22977 30586 30694 37064 20317
                 37741 29659 40769 40031 13771 18673 35817 40078
                 40088 (35651 . "망녕된 말 후") 21565 20884 21581
                 27913 29372 29476 30110 30218 32749 32757 33876
                 34705 35934 37127 37527 39610 39836 39928 13699
                 13788 14527 14631 15063 15115 15266 15405 15449
                 15647 15838 15928 16005 16072 16243 16521 17359
                 17801 17876 17992 18304 18343 18676 18881 19163
                 19687))
       (54984 . ((35347 . "가르칠 훈") (34224 . "향불 훈, 향풀 훈")
                 (29071 . "연기 낄 훈, 불기운 훈") (21235 . "공 훈")
                 (22734 . "진나팔 훈, 질나팔 훈") (26248 . "무리 훈")
                 (22628 . "壎과 同字, 질나팔 훈") (21211 . "勳의 古字, 공적 훈")
                 (28932 . "연기에 그을릴 훈, 향내 훈") (29179 . "연기 낄 훈, 불기운성할 훈")
                 21234 (26331 . "삭양 빛 훈") 37306 (32385 . "분홍빛 훈")
                 (33911 . "매운 채소 훈") 22339 34219 (29551 . "오랑캐 이름 훈")
                 30660 (37954 . "금빛 투색할 훈") 22489 21195 26197
                 33636 35757 (29195 . "불에 말릴 훈") 22161 22758
                 30611 33232 34066 34317 17202 17916 18023 18461
                 19819))
       (54988 . ((27451 . "문득 훌") 27480 28924 39750 13813))
       (55001 . ((34216 . "죽을 훙") 39021))
       (55012 . ((21927 . "의젓할 훤, 지껄일 훤") (33841 . "원추리 훤")
                 (26244 . "따뜻할 훤, 날따뜻할 훤") (29002 . "따뜻할 훤")
                 32743 (35580 . "속일 훤") (35552 . "잊을 훤") 26181
                 (28892 . "마를 훤") 26982 34300 31662 16468 21690
                 24843 34646 35974 35862 21509 22206 23191 26125
                 29407 30587 31140 33842 34175 34320 13545 18073
                 18093))
       (55016 . (17280))
       (55036 . ((27585 . "毀의 俗字, 헐 훼") (21913 . "부리 훼, 주둥이 훼")
                 (21321 . "풀 훼") (34411 . "벌레 훼") 27584 (34426 . "살무사 훼")
                 (29164 . "불 훼") 27591 (33428 . "풀 훼") 27283 35693
                 39018 39215 13424 14931 15824 16533 16611 18172))
       (55064 . ((25582 . "휘두를 휘") (36637 . "빛날 휘") (24509 . "아름다울 휘")
                 (35569 . "꺼릴 휘, 피할 휘") (40638 . "대장기 휘") (24409 . "무리 휘, 모을 휘")
                 (26249 . "빛 휘, 햇빛 휘") (28999 . "빛날 휘") 35096
                 (25757 . "찣을 휘") (32748 . "훨훨날 휘") 24410 26930
                 31112 17943 24145 26958 27719 26198 35763 36745
                 14801 23176 27851 28680 28787 32730 39988 14947
                 15054 15183 15942 16020))
       (55085 . (36964))
       (55092 . ((20241 . "쉴 휴") (25658 . "끌 휴, 가질 휴, 떠날 휴, 나눌 휴, 연속할 휴")
                 22702 (28875 . "경사로울 휴, 아름다울 휴") (34407 . "이지러질 휴")
                 (30054 . "밭두둑 휴") (38579 . "무너뜨릴 휴") 20111
                 30562 (40258 . "수리부엉이 휴") (21691 . "떠들 휴")
                 24229 38004 35973 (25813 . "携와 同字") (39673 . "옻칠할 휴")
                 25884 24002 29911 30509 33568 35327 38623 39117
                 40506 20512 20790 23118 23368 28876 30579 34355
                 34869 35321 35325 35989 37189 37533 38547 39528
                 39652 39860 13586 14687 14935 14950 15447 15580
                 15739 15983 16239 17387 18263 19340 19469))
       (55093 . ((30044 . "기를 휵, 용납할 휵") 24905 25616
                 27090 18137))
       (55100 . ((24676 . "구휼할 휼, 근심할 휼") (35662 . "속일 휼, 간사할 휼")
                 (40440 . "황새") 38705 21369 36985 37901 17935 28495
                 29533 17289 35890 40556 25008 28869 30642 32951
                 36041 14620 15578 18581 19204))
       (55113 . ((20982 . "흉할 흉") (33016 . "가슴 흉") (21256 . "가슴")
                 (20807 . "흉악할 흉, 나쁜 흉") (27958 . "물살세찰 흉, 물소리 흉")
                 27769 13355 (33015 . "胸과 同字") (24671 . "두려워할 흉")
                 24567 35369 35771 21701 35454 13529 13683 13692
                 18455))
       (55121 . ((40657 . "검을 흑") 40658 28534 37339 15428))
       (55124 . ((30165 . "흉터 흔") (27427 . "기뻐할 흔, 기쁠 흔")
                 (26133 . "아침 흔, 해돋을 흔") (28824 . "화끈거릴 흔")
                 (37313 . "피바를 흔") (24571 . "기뻐할 흔") (35362 . "기뻐할 흔")
                 (25472 . "치켜들 흔") 33291 30458 24897 (28974 . "불사를 흔")
                 20434 (24456 . "패려궂을 흔") 25323 38798 38184 34885
                 18211 20477 22945 24205 24798 29025 33066 37028
                 37697 14709 15306 16281))
       (55128 . ((27773 . "汔과 同字") (35350 . "이를 흘") (21507 . "말더듬을 흘, 말더듬는 흘")
                 (23673 . "산 우뚝솟을 흘, 산모양 흘") (32007 . "질 낮은 명주실 흘, 실밑 흘")
                 (36804 . "이를 흘") 24532 (27732 . "거의 흘") (20193 . "날랠 흘")
                 25186 34428 31866 40615 32421 35755 40833 (30105 . "쥐부스럼 흘")
                 (40789 . "깨물 흘") 32912 36279 37363 13369 16836
                 17356 18560 19235))
       (55136 . ((27453 . "공경할 흠") (27424 . "하품 흠, 하품할 흠")
                 (27462 . "받을 흠, 흠향할 흠") (24286 . "진열할 흠")
                 25599 38054 22135 23324 25779 37995 13368 18199))
       (55137 . ((21560 . "숨 들이쉴 흡") (27965 . "윤태하게 할 흡, 화할 흡")
                 (24688 . "마치 흡, 흡족할 흡") 25201 (32725 . "합할 흡")
                 (28509 . "빨리 흐르는 소리 흡") (27481 . "줄일 흡")
                 (22095 . "숨 들이쉴 흡") 29115 (32726 . "翕과 同字")
                 38367 14921 13446 14410 15015 15131 15544 16207
                 16495 18345 18960 19851))
       (55141 . ((33288 . "일 흥, 일어날 흥") 20852 23353
                 33238 39339 13495 17759))
       (55148 . ((24076 . "바랄 희") (21916 . "기뻐할 희, 기쁠 희")
                 (31232 . "드물 희") (25137 . "戲의 俗字, 희롱할 희")
                 (23305 . "즐길 희, 아름다울 희") (29081 . "빛날 희")
                 (22123 . "탄식할 희, 슬플 희") (32690 . "숨 희, 복희 희")
                 (31143 . "복 희") (29351 . "희생 희") (23020 . "성 희, 계집 희")
                 (29113 . "성할 희, 밝을 희") (24985 . "성할 희, 기뻐할 희")
                 (24894 . "한숨 희") (20694 . "기쁠 희, 즐거울 희") (26342 . "햇빛 희")
                 (20958 . "빛날 희, 화할 희") (26206 . "마를 희") (24984 . "憙의 古字, 기쁠 희")
                 (29114 . "熹와 同字, 밝을 희") (22221 . "쌍희 희") (25138 . "탄식할 희, 희롱할 희")
                 25135 29013 39193 29344 (22075 . "웃을 희") (39228 . "꾸밀 희")
                 29177 (27447 . "흐느낄 희") 20441 (21775 . "슬퍼할 희")
                 24007 23661 29430 (26303 . "웃을 희") (21669 . "허허 웃을 희")
                 15754 39246 (24725 . "슬퍼할 희") 28000 22193 27250
                 29064 30542 34786 35474 15138 (35944 . "멧돼지 희")
                 21612 26744 29707 32357 38716 15430 23619 25103
                 29306 35830 39273 (29204 . "불 희") 40662 23635 24549
                 24620 27478 28911 28929 28936 29058 29160 29332
                 30630 30978 33667 35654 35959 38581 39889 40026
                 40279 40770 13430 13951 13952 13990 14127 14150
                 14395 14956 15381 15400 15831 15849 15885 15991
                 16035 16512 17056 17413 17810 17847 18149 18995
                 19342))
       (55176 . (23630 24574 13671 17430 18841))
       (55184 . ((35440 . "물을 힐, 힐문할 힐") (40672 . "약을 힐")
                 32953 (38945 . "곧은목 힐") (32392 . "홀치기염색 힐")
                 25847 32952 25783 32556 35800 39049 (29365 . "오랑캐이름 힐")
                 (35181 . "옷자락 꽂을 힐") 24644 27439 32723 14954))))
    (message "")))

;; List of current conversion status.
;; The first element is the strating position of shown list.
;; It is a group number each split by `hanja-list-width'.
;; The second element is the position of selected element.
;; The third element is a list of suitable Hanja candidate.
(defvar hanja-conversions
  nil)

(defvar hanja-keymap
  (let ((map (make-sparse-keymap)))
    (define-key map "\r" 'hanja-list-terminate)
    (define-key map [return] 'hanja-list-terminate)
    (define-key map [up] 'hanja-list-prev-group)
    (define-key map [down] 'hanja-list-next-group)
    (define-key map [left] 'hanja-list-prev)
    (define-key map [right] 'hanja-list-next)
    (define-key map "\C-p" 'hanja-list-prev-group)
    (define-key map "\C-n" 'hanja-list-next-group)
    (define-key map "\C-b" 'hanja-list-prev)
    (define-key map "\C-f" 'hanja-list-next)
    (define-key map "0" (lambda ()
			  (aset hanja-conversions 1 0) (hanja-list-terminate)))
    (define-key map "1" (lambda ()
			  (aset hanja-conversions 1 1) (hanja-list-terminate)))
    (define-key map "2" (lambda ()
			  (aset hanja-conversions 1 2) (hanja-list-terminate)))
    (define-key map "3" (lambda ()
			  (aset hanja-conversions 1 3) (hanja-list-terminate)))
    (define-key map "4" (lambda ()
			  (aset hanja-conversions 1 4) (hanja-list-terminate)))
    (define-key map "5" (lambda ()
			  (aset hanja-conversions 1 5) (hanja-list-terminate)))
    (define-key map "6" (lambda ()
			  (aset hanja-conversions 1 6) (hanja-list-terminate)))
    (define-key map "7" (lambda ()
			  (aset hanja-conversions 1 7) (hanja-list-terminate)))
    (define-key map "8" (lambda ()
			  (aset hanja-conversions 1 8) (hanja-list-terminate)))
    (define-key map "9" (lambda ()
			  (aset hanja-conversions 1 9) (hanja-list-terminate)))
    map)
  "Keymap for Hanja (Korean Hanja Converter).")

(defun hanja-filter (condp lst)
  "Construct a list from the elements of LST for which CONDP returns true."
  (delq
   nil
   (mapcar (lambda (x) (and (funcall condp x) x)) lst)))

(defun hanja-list-prev-group ()
  "Select the previous group of hangul->hanja conversions."
  (let ((base (aref hanja-conversions 0)))
    (if (> base 0)
        (aset hanja-conversions 0 (1- base)))))

(defun hanja-list-next-group ()
  "Select the next group of hangul->hanja conversions."
  (let* ((base (aref hanja-conversions 0))
         (offset (aref hanja-conversions 1))
         (next-list (nthcdr (* (1+ base) hanja-list-width)
                            (aref hanja-conversions 2))))
    (when next-list
      (aset hanja-conversions 0 (1+ base))
      (if (< (1- (length next-list)) offset)
          (aset hanja-conversions 1 (1- (length next-list)))))))

(defun hanja-list-prev ()
  "Select the previous candidate of hangul->hanja conversion."
  (let ((base (aref hanja-conversions 0))
        (offset (aref hanja-conversions 1)))
    (cond ((> offset 0)
           (aset hanja-conversions 1 (1- offset)))
          ((> base 0)
           (hanja-list-prev-group)
           (aset hanja-conversions 1 (1- hanja-list-width))))))

(defun hanja-list-next ()
  "Select the next candidate of hangul->hanja conversion."
  (let* ((base (aref hanja-conversions 0))
         (offset (aref hanja-conversions 1))
         (hanja-list (aref hanja-conversions 2)))
    (cond ((and (< offset (1- hanja-list-width))
                (< offset (1- (length (nthcdr (* base hanja-list-width)
                                              hanja-list)))))
           (aset hanja-conversions 1 (1+ offset)))
          ((nthcdr (* (1+ base) hanja-list-width) hanja-list)
           (hanja-list-next-group)
           (aset hanja-conversions 1 0)))))

(defun hanja-list-show ()
  "Show hanja list in the echo area."
  (let* ((base (aref hanja-conversions 0))
         (offset (aref hanja-conversions 1))
         (hanja-list (aref hanja-conversions 2))
         (partial-hanja-list (nthcdr (* base hanja-list-width)
                                     hanja-list))
         (idx 0)
         (msg "")
         (message-log-max nil)
         char-meaning)
    (while (and (< idx hanja-list-width)
                partial-hanja-list)
      (setq msg (format "%s%d.%c " msg idx (car (car partial-hanja-list))))
      (when (= offset idx)
        (put-text-property
         (+ 2 (string-match (format "%d\\.[^ ]*" idx) msg))
         (match-end 0)
         'face 'highlight msg)
        (setq char-meaning (cdr (car partial-hanja-list))))
      (setq idx (1+ idx)
            partial-hanja-list (cdr partial-hanja-list)))
    (if char-meaning
        (setq msg (format "%s   (뜻 : %s)" msg char-meaning)))
    (message "%s" msg)))

(defun hanja-list-terminate ()
  "Exit from the hanja listing mode with the currently selected character."
  (let ((base (aref hanja-conversions 0))
        (offset (aref hanja-conversions 1))
        (hanja-list (aref hanja-conversions 2)))
    (throw 'exit-input-loop
	   (car (nth (+ (* base hanja-list-width) offset) hanja-list)))))

(defun hangul-to-hanja-char (char)
  "Show the list of hanja characters converted from CHAR.
The value is a hanja character that is selected interactively."
  (hanja-init-load)
  (let ((echo-keystrokes 0)
        (input-method-function nil))
    (setq hanja-conversions
          (vector
           0 0
           ;; Filter characters that can not be decoded.
           ;; Maybe it can not represent characters in current terminal coding.
           (hanja-filter (lambda (x) (car x))
                         (mapcar (lambda (c)
                                   (if (listp c)
                                       (cons (decode-char 'ucs (car c)) (cdr c))
				     (list (decode-char 'ucs c))))
                                 (aref hanja-table char)))))
    (unwind-protect
	(when (aref hanja-conversions 2)
	  (catch 'exit-input-loop
	    (while t
	      (hanja-list-show)
	      (let* ((seq (read-key-sequence nil))
		     (cmd (lookup-key hanja-keymap seq)))
		(if (functionp cmd)
		    (funcall cmd)
		  (setq unread-command-events
                        (nconc (listify-key-sequence seq)
                               unread-command-events))
		  (throw 'exit-input-loop nil))))))
      (setq hanja-conversions nil))))

(provide 'hanja-util)

;;; hanja-util.el ends here
