
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 *	waitc.c
 *
 *	Wait context descriptors.
 *
$Log$
 */

#include "global.h"
#include "protoglarp.h"
#include "myalloc.h"
#include "waitc.h"
#include "mesg.h"
#include "listmac.h"

/***************
 **  Globals  **
 **           **
 ***************/

extern void bailout();

struct waitc *waitlist = 0;


/***************
 **  Private  **
 **           **
 ***************/

static int nextwait = 1;


/*****************
 **  Interface  **
 **             **
 *****************/

/*	wait_init()
*
*	Call this boy before any other wait stuff.
*/

void
wait_init()
{
	if (!waitlist) {
		waitlist = TALLOC(1, struct waitc, "wait");
		bzero((char*)waitlist, sizeof(struct waitc));
		waitlist->wa_link = waitlist;
		waitlist->wa_rlink = waitlist;
	}
}


/*	wait_new()
*
*	Create new wait context of given kind with no peers.  Add to
*	active list.
*/

struct waitc *
wait_new(kind)
	int kind;
{
	struct waitc *wp;

	if (!(wp = TALLOC(1, struct waitc, "wait"))) {
		log_error("wait_new() can't get memory\n");
		bailout(0);
	}
	wp->wa_wid = nextwait++;
/* XXX this will wrap and cause problems.  need a way to get a
   XXX free id cheaply, without scanning the waitlist, at least more
   XXX than once */
	wp->wa_kind = kind;
	wp->wa_peer = wp->wa_rpeer = wp;

	wp->wa_on = wp->wa_tid = wp->wa_dep = 0;
	wp->wa_mesg = 0;
	wp->wa_spec = 0;

	LISTPUTBEFORE(waitlist, wp, wa_link, wa_rlink);
	return wp;
}


/*	wait_find()
*
*	Find a wait context in active list by id.  Returns pointer
*	or 0 if not found.
*/

struct waitc *
wait_find(wid)
	int wid;
{
	struct waitc *wp;

	for (wp = waitlist->wa_link; wp != waitlist; wp = wp->wa_link)
		if (wp->wa_wid == wid)
			return wp;
	return (struct waitc*)0;
}


/*	wait_delete()
*
*	Remove a wait context from the active list, disassociate from
*	any peers and destroy it.
*/

void
wait_delete(wp)
	struct waitc *wp;
{
	if (wp->wa_mesg)
		mesg_unref(wp->wa_mesg);

	if (wp->wa_link) {
		LISTDELETE(wp, wa_link, wa_rlink);
	}
	if (wp->wa_peer) {
		LISTDELETE(wp, wa_peer, wa_rpeer);
	}
	PVM_FREE(wp);
}


