
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 *	umbuf.c
 *
 *	Libpvm message descriptors
 *
$Log$
 */

#include <stdio.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include "pvm3.h"
#include "myalloc.h"
#include "frag.h"
#include "umbuf.h"
#include "listmac.h"

#ifdef IMA_I860
#include <memory.h>
#define bzero(s,n)         memset(s,0,n)
#endif

/***************
 **  Globals  **
 **           **
 ***************/

struct mhp *midh = 0;				/* heap of all msgs */
int midhsiz = 0;					/* length of midh */


/***************
 **  Private  **
 **           **
 ***************/

static char rcsid[] = "$Id$";
static char etext[512];				/* scratch for error log */
static int midhfree = 0;			/* head of midh free list */


/*	mid_new()
*
*	Allocate a message id for a new message.
*	Returns mid (>0) or PvmNoMem if can't get memory.
*/

int
mid_new()
{
	static int nxtmidhsiz;
	int mid;
	int ns;

	/* check free list empty */

	if (!midhfree) {

	/* extend midh array */

		if (midhsiz) {
			ns = nxtmidhsiz;
			nxtmidhsiz += midhsiz;
			midh = TREALLOC(midh, ns, struct mhp);
		} else {
			nxtmidhsiz = 13;
			ns = 8;
			midh = TALLOC(ns, struct mhp, "mid");
		}
		if (!midh)
			return PvmNoMem;

	/* put new entries on free list */

		while (midhsiz < ns) {
			midh[midhsiz].m_umb = 0;
			midh[midhsiz].m_free = midhfree;
			midhfree = midhsiz++;
		}
	}

	/* take next if from free list */

	mid = midhfree;
	midhfree = midh[midhfree].m_free;
	return mid;
}


/*	mid_free()
*
*	Return a message id to the free list.
*/

void
mid_free(mid)
	int mid;
{
	if (mid <= 0 || mid >= midhsiz || !midh[mid].m_umb)
		return;
	midh[mid].m_umb = 0;
	midh[mid].m_free = midhfree;
	midhfree = mid;
}


/*	umbuf_new()
*
*	Create a new empty message.
*	Return the mid or PvmNoMem if malloc fails.
*/

int
umbuf_new()
{
	struct umbuf *up;

	if (!(up = TALLOC(1, struct umbuf, "umb")))
		goto nomem;

	bzero((char*)up, sizeof(struct umbuf));
	if ((up->ub_mid = mid_new()) < 0)
		goto nomem;
	midh[up->ub_mid].m_umb = up;
	up->ub_frag = fr_new(0);
	return up->ub_mid;

nomem:
	if (up)
		PVM_FREE(up);
	return PvmNoMem;
}


/*	umbuf_free()
*
*	Free a message and all its resources.  Unlink it from any list
*	and recycle its mid.
*	Returns 0 if okay, else PvmNoSuchBuf if id bogus.
*/

int
umbuf_free(mid)
	int mid;
{
	struct umbuf *up;

	if (mid <= 0 || mid >= midhsiz)
		return PvmBadParam;
	if (!midh[mid].m_umb)
		return PvmNoSuchBuf;

	up = midh[mid].m_umb;
	if (up->ub_link && up->ub_rlink) {
		LISTDELETE(up, ub_link, ub_rlink);
	}
	fr_unref(up->ub_frag);
	mid_free(up->ub_mid);
	PVM_FREE(up);
	return 0;
}


/*	midtobuf()
*
*	Return pointer to message buffer given int message id.
*/

struct umbuf*
midtobuf(mid)
	int mid;
{
	return ((mid > 0 && mid < midhsiz) ? midh[mid].m_umb : 0);
}


umbuf_dump(mid)
	int mid;
{
	struct umbuf *up;
	struct frag *fp;

	if (mid <= 0 || mid >= midhsiz) {
		sprintf(etext, "umbuf_dump() BadParam\n");
		log_error(etext);
		return PvmBadParam;
	}
	if (!midh[mid].m_umb) {
		sprintf(etext, "umbuf_dump() NoSuchBuf\n");
		log_error(etext);
		return PvmNoSuchBuf;
	}

	up = midh[mid].m_umb;
	sprintf(etext, "umbuf_dump() mid=%d 0x%x len=%d cod=%d enc=%d src=t%x\n",
			mid, up, up->ub_len, up->ub_cod, up->ub_enc, up->ub_src);
	log_error(etext);
	for (fp = up->ub_frag->fr_link; fp != up->ub_frag; fp = fp->fr_link) {
		sprintf(etext, " frag=0x%x max=%d ofs=%d len=%d\n",
			fp, fp->fr_max, fp->fr_dat - fp->fr_buf, fp->fr_len);
		log_error(etext);
		hdump(fp->fr_dat, fp->fr_len, "  ");
/*
		hdump(fp->fr_dat, 32, "  ");
*/
	}
	return 0;
}


umbuf_list()
{
	int i;
	struct umbuf *up;
	struct frag *fp;
	int rlen;

	for (i = 1; i < midhsiz; i++)
		if (up = midh[i].m_umb) {
			rlen = 0;
			for (fp = up->ub_frag->fr_link; fp != up->ub_frag;
					fp = fp->fr_link)
				rlen += fp->fr_max;
			sprintf(etext,
			"umbuf_list() mid=%d cod=%d enc=%d src=t%x len=%d rlen=%d\n",
					i, up->ub_cod, up->ub_enc, up->ub_src, up->ub_len, rlen);
			log_error(etext);
		}
}


