
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 * PVM3.0-I860
 *
 *  lpvm.c:  Libpvm core for unix environment.
 * 
 * 		Ported to I860.
 * 
 *			send bug reports to:  wjiang@cs.utk.edu
 *
 */

#include <rpc/types.h>
#include <rpc/xdr.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#ifdef  SYSVSTR
#include <string.h>
#else
#include <strings.h>
#endif
#include <errno.h>
#include "pvm3.h"
#include "global.h"
#include "tdpro.h"
#include "ddpro.h"
#include "myalloc.h"
#include "frag.h"
#include "umbuf.h"
#include "listmac.h"
#include "mycube.h"

#ifdef IMA_I860
#include <memory.h>
#define bcopy(fr,to,n)    memcpy(to,fr,n)
#define bzero(s,n)         memset(s,0,n)
#endif

/* task debug mask */

#define TDMPACKET   1		/* packet tracing */
#define TDMMESSAGE  2		/* message tracing */


extern struct encvec *enctovec();



/***************
 **  Globals  **
 **           **
 ***************/

extern int      errno;		/* from libc */
extern char    *sys_errlist[];
extern int      sys_nerr;

extern int      rbufmid;	/* from pack.c */
extern int      sbufmid;	/* from pack.c */

char           *pvm_errlist[] = {	/* error messages for -pvm_errno */
	"Error 0",
	"Error 1",
	"Bad parameter",
	"Error 3",		/* PvmMismatch */
	"Error 4",		/* not used */
	"End of buffer",
	"No such host",
	"No such file",
	"Error 8",		/* not used */
	"Error 9",		/* not used */
	"Malloc failed",
	"Error 11",		/* not used */
	"Can't decode message",
	"Error 13",		/* not used */
	"Can't contact local daemon",
	"No current buffer",
	"No such buffer",
	"Error 17",		/* PvmNullGroup */
	"Error 18",		/* PvmDupGroup */
	"Error 19",		/* PvmNoGroup */
	"Error 20",		/* PvmNotInGroup */
	"Error 21",		/* PvmNoInst */
	"Host failed",
	"No parent task",
	"Not implemented",
	"Pvmd system error",
	"Version mismatch",
	"Out of resources",
	"Duplicate host",
	"Can't start pvmd",
	"Already in progress",
};

int             myndf = 0;	/* host native data enc, init XDR */
int             mypvmptid = -1;	/* parent task id */
int             mypvmtid = -1;	/* this task id */
int             myunixpid = -1;	/* process unix pid */
int             ourudpmtu = UDPMAXLEN;	/* local UDP MTU */
int             pvm_errno = 0;	/* last libpvm error code */
int             pvm_nerr = sizeof(pvm_errlist)
/ sizeof(pvm_errlist[0]);	/* exported num of errors */
struct umbuf   *rxlist = 0;	/* not-recvd msg list */

int             tidhmask = TIDHOST;	/* mask for host field of tids */
int             tidcmask = TIDCUBE;	/* mask for cube field of tids */
int             tidnode = TIDNODE;	/* mask for node field of tids */
int             tidnmask = TIDCLOCAL;	/* mask for node field of tids */
long            my_host = -1;	/* node id of my host */


/***************
 **  Private  **
 **           **
 ***************/

static char     rcsid[] = "$Id$";
static int      debugmask = 0;	/* which debugging info */
static char     etext[512];	/* scratch for error log */
#ifdef NOUNIXSOC
static struct sockaddr_in loclsadr;	/* address of local socket */
static struct sockaddr_in pvmdsadr;	/* address of pvmd socket */
#else
static struct sockaddr_un loclsadr;
static struct sockaddr_un pvmdsadr;
#endif
static int      loclslen = 0;	/* loclsadr length */
static int      loclsock = -1;	/* pvmd-task udp socket */
static int      pvmdslen = 0;	/* pvmdsadr length */
static struct umbuf *rxfrag = 0;/* not-assembled incm msgs */
static int      useruid = -1;	/* user's unix uid */
static int      xfertok = 1;	/* we have send token */


/**************************
 **  Internal Functions  **
 **                      **
 **************************/

/*
 * bailout()
 * 
 * Called by low-level stuff in f.e. frag.c.  Don't really want to bail in
 * libpvm.
 */

void
bailout(n)
	int             n;
{
}


/*
 * log_error()
 * 
 * Log a libpvm error message.  Prepends a string identifying the task.
 */

log_error(s)
	char           *s;
{
	FILE           *fp;

	if (mypvmtid == -1)
		fprintf(stderr, "libpvm [pid%d]: %s", myunixpid, s);
	else
		fprintf(stderr, "libpvm [t%x]: %s", mypvmtid, s);
}


/*
 * log_perror()
 * 
 * Log a libpvm error message.  Prepends a string identifying the task and
 * appends the system error string for _errno.
 */

log_perror(s)
	char           *s;
{
	char           *em;
	FILE           *fp;

	em = ((errno >= 0 && errno < sys_nerr)
	      ? sys_errlist[errno] : "Unknown Error");
	if (mypvmtid == -1)
		fprintf(stderr, "libpvm [pid%d]: %s: %s\n", myunixpid, s, em);
	else
		fprintf(stderr, "libpvm [t%x]: %s: %s\n", mypvmtid, s, em);
}


hdump(p, n, pad)
	char           *p;	/* bytes */
	int             n;	/* length */
	char           *pad;
{
	int             i;
	pad = pad ? pad : "";
	for (i = 0; n-- > 0; i = ++i & 15) {
		fprintf(stderr, "%s%02x%s",
			i ? "" : pad,
			0xff & *p++,
			n && i != 15 ? " " : "\n");
	}
}


/*
 * mxfer()
 * 
 * Move message frags between task and pvmd. Returns when outgoing message (if
 * any) fully sent (if block is true) one message fully received Returns >=0
 * the number of complete messages downloaded, or negative on error.
 */

int
mxfer(mid, dtid, code, block)
	int             mid;	/* message */
	int             dtid;	/* dest */
	int             code;	/* type code */
	int             block;	/* get at least one message */
{
	struct umbuf   *txup;	/* tx message or null */
	struct frag    *txfp = 0;	/* cur tx frag or null */
	char            dummy[TDFRAGHDR];	/* space to make null-pkt
						 * header */
	char           *cp;	/* data to send */
	int             len;	/* length of data */
	int             ff;	/* frag flags */
	struct frag    *rxfp;	/* rx frag */
	struct umbuf   *rxup;	/* rx message */
	int             gotem = 0;	/* num complete msgs downloaded */
	int             retry;
	int             src;
	static int      first = 1;
	int				tonode;		/* to another node or pvmd? */
	int				probe, recvmask;


	tonode = TIDISNODE(dtid);
	recvmask = 0x80000000 + (1 << PMTPVMD) + (1 << PMTNODE);

	if (txup = midtobuf(mid)) {
		txfp = txup->ub_frag->fr_link;
		txfp = txfp->fr_buf ? txfp : 0;
	}

out:
	while (txfp) {

		cp = txfp->fr_dat;
		len = txfp->fr_len;
		ff = 0;
		if (txfp->fr_rlink == txup->ub_frag) {	/* first frag */
			/* tack on t-t header */
			cp -= TTMSGHDR;
			len += TTMSGHDR;
			pk32(cp, code);
			pk32(cp + 4, txup->ub_enc);
			ff = FFSOM;
		}
		if (txfp->fr_link == txup->ub_frag)
			ff |= FFEOM;

		/* tack on t-d header */
		cp -= TDFRAGHDR;
		pk32(cp, dtid);
		pk32(cp + 4, mypvmtid);
		pk32(cp + 8, len);
		pk8(cp + 12, ff);
		len += TDFRAGHDR;
		if (debugmask & TDMPACKET) {
			sprintf(etext, "mxfer() to t%x len %d\n", dtid, txfp->fr_len);
			log_error(etext);
		}
		txfp = txfp->fr_link;
		if (!txfp->fr_buf)
			txfp = 0;

		if (tonode) {
			if (_csend(PMTNODE, cp, (long)len, (long)(dtid & tidnmask), 0) < 0){
				sprintf(etext, "mxfer() csend to t%x failed", dtid);
				log_perror(etext);
				return PvmSysErr;
			}
		} else {
/*
			if (!xfertok) {
				char	token[1];

				crecv(PMTTOKEN, token, 0);
			}
*/
			/* send pkt to pvmd (via cubed) */
			if (_csend(PMTNODE, cp, (long) len, my_host, CUBEDPID) < 0) {
				log_perror("mxfer() csend to cubed failed");
				return PvmSysErr;
			}
			xfertok = 0;
		}
	}

in:
	while ((block && !gotem) || (probe = _iprobe(recvmask))) {
		if (probe == -1) {
			log_perror("mxfer() err iprobe");
			break;
		}
		rxfp = fr_new(ourudpmtu);
		if (_crecv(recvmask, rxfp->fr_dat, (long) rxfp->fr_max) < 0) {
			log_perror("mxfer() err crecv to pvmd");
			fr_unref(rxfp);
			return PvmSysErr;
		}
		if ((len = _infocount()) < 0) {
			log_perror("mxfer() err info on recvd msgs\n");
			return PvmSysErr;
		}
	/*
	 * hdump(rxfp->fr_dat, len, "mxfer() pkt: ");
	 */

		/* decode frag header */
		cp = rxfp->fr_dat;
	/*
	 * dst = up32(cp); XXX might unpack and check
	 */
		src = up32(cp + 4);
		ff = up8(cp + 12);
		rxfp->fr_len = len - TDFRAGHDR;
		rxfp->fr_dat += TDFRAGHDR;

		if (debugmask & TDMPACKET) {
			sprintf(etext, "mxfer() from t%x len %d\n", src, rxfp->fr_len);
			log_error(etext);
		}

		if (ff & FFSOM) {	/* start of message */
		/* make umbuf, decode header, link to frag pile */
			cp += TDFRAGHDR;
			rxfp->fr_len -= TTMSGHDR;
			rxfp->fr_dat += TTMSGHDR;
			rxup = midtobuf(umbuf_new());
			rxup->ub_cod = up32(cp);
			rxup->ub_enc = up32(cp + 4);
			rxup->ub_src = src;
			LISTPUTBEFORE(rxfrag, rxup, ub_link, ub_rlink);
		}
	/* locate frag's message */
		for (rxup = rxfrag->ub_link; rxup != rxfrag; rxup = rxup->ub_link)
			if (rxup->ub_src == src)
				break;

		if (rxup == rxfrag) {	/* uh oh, no message for it */
			log_error("mxfer() frag with no message\n");
			fr_unref(rxfp);

		} else {
			LISTPUTBEFORE(rxup->ub_frag, rxfp, fr_link, fr_rlink);
			rxup->ub_len += rxfp->fr_len;

			if (ff & FFEOM) {	/* end of message */
				LISTDELETE(rxup, ub_link, ub_rlink);
				rxup->ub_codef = enctovec(rxup->ub_enc);
				LISTPUTBEFORE(rxlist, rxup, ub_link, ub_rlink);
				gotem++;
			}
		}
	}

in2:
	return gotem;
}
/*
 * msendrecv()
 * 
 * Single op to send a system message (usually to our pvmd) and get the reply.
 * Returns message handle or negative if error.
 */

int
msendrecv(other, code)
	int             other;	/* dst, src tid */
	int             code;	/* message code */
{
	int             cc;
	struct umbuf   *up;

	if (sbufmid <= 0)
		return PvmNoBuf;

	/* send code to other */
	if (debugmask & TDMMESSAGE) {
		sprintf(etext, "msendrecv() to t%x code %d\n", other, code);
		log_error(etext);
	}
	if ((cc = mxfer(sbufmid, other, code, 1)) < 0)
		return cc;

	/* recv code from other */
	for (up = rxlist->ub_link; 1; up = up->ub_link) {
		if (up == rxlist) {
			up = up->ub_rlink;
			if ((cc = mxfer(0, 0, 0, 1)) < 0)
				return cc;
			up = up->ub_link;
		}
		if (debugmask & TDMMESSAGE) {
			sprintf(etext, "msendrecv() cmp from t%x code %d\n",
				up->ub_src, up->ub_cod);
			log_error(etext);
		}
		if (up->ub_src == other && up->ub_cod == code)
			break;
	}
	LISTDELETE(up, ub_link, ub_rlink);
	if (rbufmid > 0)
		umbuf_free(rbufmid);
	rbufmid = 0;
	if (cc = pvm_setrbuf(up->ub_mid))
		return cc;
	return up->ub_mid;
}

int
unmksocs()
{
}

/*
 * beatask()
 * 
 * Config process as a task.  This is called as the first step of each libpvm
 * function.
 */

#ifdef  L_tmpnam
#define LEN_OF_TMP_NAM  L_tmpnam
#else
#define LEN_OF_TMP_NAM  64
#endif

int
beatask()
{
    int sbf = 0, rbf = 0;           /* saved rx and tx message handles */
    int prver;                      /* protocol version */
    int altpid = 0;         		/* pid assigned by pvmd if diff from real */
    int cc;
    char authfn[LEN_OF_TMP_NAM];    /* auth file name */
    int authfd = -1;                /* auth fd to validate pvmd ident */
	int my_phys;					/* direct node number */

	if (mypvmtid != -1)
		return 0;
    authfn[0] = 0;

    if ((useruid = getuid()) == -1) {
        log_error("can't getuid()\n");
        return PvmSysErr;
    }

	if ((my_host = _myhost()) < 0 || (my_phys = _mydirect()) < 0) {
		log_perror("beatask() can't get host node #");
		return PvmSysErr;
	}
	if (_crecv(PMTTID, &mypvmtid, sizeof(int)) < 0) {
		log_perror("beatask() can't crecv from cubed");
        return PvmSysErr;
    }
	mypvmtid += my_phys;
	if (_isend(PMTTID, &mypvmtid, sizeof(int), my_host, CUBEDPID) < 0){
		log_perror("beatask() can't isend to cubed");
		return PvmSysErr;
	}
	rxfrag = TALLOC(1, struct umbuf, "umb");
	bzero((char *) rxfrag, sizeof(struct umbuf));
	rxfrag->ub_link = rxfrag->ub_rlink = rxfrag;

	rxlist = TALLOC(1, struct umbuf, "umb");
	bzero((char *) rxlist, sizeof(struct umbuf));
	rxlist->ub_link = rxlist->ub_rlink = rxlist;

    /*
    *   create empty t-auth file that pvmd must write and we'll check later
    */

    (void)tmpnam(authfn);
    if ((authfd = open(authfn, O_RDONLY|O_CREAT|O_TRUNC, 0600)) == -1) {
        log_perror("beatask() creat t-auth file");
        return PvmSysErr;
    }

	sbf = pvm_setsbuf(pvm_mkbuf(PvmDataFoo));
	rbf = pvm_setrbuf(0);
	prver = TDPROTOCOL;
	pvm_pkint(&prver, 1, 0);
    pvm_pkstr(authfn);
    if ((cc = msendrecv(TIDPVMD, TM_CONNECT)) <= 0)
        goto bail;
    pvm_upkint(&prver, 1, 0);
    if (prver != TDPROTOCOL) {
        sprintf(etext, "beatask() t-d protocol mismatch (%d/%d)\n",
            TDPROTOCOL, prver);
        log_error(etext);
        cc = PvmSysErr;
        goto bail;
    }
    pvm_upkint(&cc, 1, 0);
    if (!cc) {
        log_error("beatask() pvmd refuses connection\n");
        goto bail;
    }

    /*
    *   check our t-auth file; write in pvmd d-auth file
    */

    if ((cc = read(authfd, (char*)&cc, 1)) == -1) {
        log_perror("beatask() read authfile\n");
        cc = PvmSysErr;
        goto bail;
    }
    if (cc != 1) {
        log_error("beatask() pvmd didn't validate itself\n");
        cc = PvmSysErr;
        goto bail;
    }
    (void)close(authfd);
    (void)unlink(authfn);

    pvm_upkstr(authfn);
    if ((authfd = open(authfn, O_WRONLY, 0)) == -1) {
        log_perror("beatask() open d-auth file\n");
        authfn[0] = 0;
        cc = PvmSysErr;
        goto bail;
    }
    (void)write(authfd, authfn, 1);
    (void)close(authfd);
    authfd = -1;
    authfn[0] = 0;

    /*
    *   send second connect message to pvmd
    */

    pvm_initsend(PvmDataFoo);
	pvm_pkint(&myunixpid, 1, 0);
    pvm_pkint(&altpid, 1, 0);
    if ((cc = msendrecv(TIDPVMD, TM_CONN2)) <= 0)
        goto bail;
    pvm_upkint(&cc, 1, 0);
    if (!cc) {
        log_error("beatask() pvmd refuses connection\n");
        goto bail;
    }
	pvm_upkint(&mypvmptid, 1, 0);
	pvm_upkint(&ourudpmtu, 1, 0);
	pvm_upkint(&myndf, 1, 0);

	if (debugmask & TDMPACKET) {
        sprintf(etext, "beatask() mypvmptid=%x\n", mypvmptid);
		log_error(etext);
    }

    pvm_freebuf(pvm_setrbuf(rbf));
    pvm_freebuf(pvm_setsbuf(sbf));

    return 0;

bail:
    if (pvm_getrbuf() > 0)
        pvm_freebuf(pvm_getrbuf());
    if (pvm_getsbuf() > 0)
        pvm_freebuf(pvm_getsbuf());
    pvm_setrbuf(rbf);
    pvm_setsbuf(sbf);

    if (authfd != -1)
        (void)close(authfd);
    if (authfn[0])
        (void)unlink(authfn);

	return cc;
}


int
endtask()
{
    if (mypvmtid != -1) {
        mypvmtid = -1;
    }

    /* XXX free rxfrag and rxlist */

    return 0;
}


/************************
 **  Libpvm Functions  **
 **                    **
 ************************/

int
pvm_perror(s)
	char           *s;
{
	if (mypvmtid == -1)
		fprintf(stderr, "libpvm [pid%d]: ", myunixpid);
	else
		fprintf(stderr, "libpvm [t%x]: ", mypvmtid);
	fprintf(stderr, "%s: %s\n",
		(s ? s : "(null)"),
		(pvm_errno <= 0 && pvm_errno > -pvm_nerr
		 ? pvm_errlist[-pvm_errno] : "Unknown Error"));
	return 0;
}


int
pvm_setdebug(mask)
	int             mask;
{
	int             old = debugmask;

	debugmask = mask;
	return old;
}
