
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 * PVM3.0-I860
 *
 *  ipack.c
 *
 *  Typed packing/unpacking, message buffer manip.
 *
 *		Ported to I860. Send bug reports to:  wjiang@cs.utk.edu
 *
 */

#ifdef IMA_BSD386
#include <machine/endian.h>
#endif
#include <rpc/types.h>
#include <rpc/xdr.h>
#include "pvm3.h"
#include "global.h"
#include "tdpro.h"
#include "ddpro.h"
#include "frag.h"
#include "umbuf.h"
#include "listmac.h"

#ifdef IMA_I860
#include <memory.h>
#define bcopy(fr,to,n)    memcpy(to,fr,n)
#endif

struct dat_blk {
    int type;       /* data type */
    void *base;     /* base address */
    int cnt;        /* number of elements */
    int std;        /* lead to next element */
};

/* struct dat_blk type codes */
#define INP_BYTE    1
#define INP_INT     2
#define INP_SHORT   3
#define INP_LONG    4
#define INP_FLOAT   5
#define INP_DFLOAT  6
#define INP_CPLX    7
#define INP_DCPLX   8


/***************
 **  Globals  **
 **           **
 ***************/

extern struct mhp *midh;        /* from umbuf.c */
extern int midhsiz;             /* from umbuf.c */
extern int myndf;               /* from lpvm.c */
extern int mypvmtid;            /* from lpvm.c */
extern int ourudpmtu;           /* from lpvm.c */

int rbufmid = 0;                /* current recv/unpack mid */
int sbufmid = 0;                /* current send/pack mid */


/***************
 **  Private  **
 **           **
 ***************/

static char rcsid[] = "$Id$";


/***************************
 **  Raw-format encoders  **
 **                       **
 ***************************/

/*  bytepk()
*
*   Insert a stream of bytes into sbuf.  Allocate more fragments as
*   necessary.
*   Returns 0 else PvmNoMem if malloc fails.
*/

static int
bytepk(cp, num, siz, lnc)
    char *cp;   /* base of data */
    int num;    /* num of chunks */
    int siz;    /* size of chunk */
    int lnc;    /* lead to next chunk */
{
    struct umbuf *up = midtobuf(sbufmid);   /* working message */
    struct frag *fp;            /* working frag */
    int togo;                   /* bytes left in chunk */
    int r;                      /* bytes (space) left in frag */

    if (siz == lnc) {       /* if contiguous, treat as single chunk */
        lnc = (siz *= num);
        num = 1;
    }
    lnc -= siz;     /* now bytes between chunks */

    while (num-- > 0) {     /* copy chunks until done */

        for (togo = siz; togo > 0; ) {
            fp = up->ub_frag->fr_rlink;
            r = fp->fr_max - (fp->fr_dat - fp->fr_buf) - fp->fr_len;

            if (togo <= r) {    /* space in frag for entire chunk */
                bcopy(cp, fp->fr_dat + fp->fr_len, togo);
                fp->fr_len += togo;
                cp += togo;
                togo = 0;

            } else {
                if (r > 0) {    /* space for part of chunk */
                    bcopy(cp, fp->fr_dat + fp->fr_len, r);
                    fp->fr_len += r;
                    togo -= r;
                    cp += r;

                } else {        /* no space, add new frag */
                    if (r = enc_step())
                        return r;
                }
            }
        }
        cp += lnc;
    }
    return 0;
}


/*  byteupk()
*
*   Extract bytes from rbuf.
*   Returns 0 else PvmNoData if end of message reached early.
*/

static int
byteupk(cp, num, siz, lnc)
    char *cp;   /* base of data */
    int num;    /* num of chunks */
    int siz;    /* size of chunk */
    int lnc;    /* lead to next chunk */
{
    struct umbuf *up = midtobuf(rbufmid);   /* working message */
    struct frag *fp;            /* working frag */
    int togo;                   /* bytes left in chunk */
    int r;                      /* bytes (data) left in frag */

    if (siz == lnc) {       /* if contiguous, treat as single chunk */
        lnc = (siz *= num);
        num = 1;
    }
    lnc -= siz;     /* now bytes between chunks */

    while (num-- > 0) {     /* copy chunks until done */

        for (togo = siz; togo > 0; ) {
            fp = up->ub_cfrag;
            r = fp->fr_len - up->ub_cpos;

            if (togo <= r) {    /* frag contains rest of chunk */
                bcopy(fp->fr_dat + up->ub_cpos, cp, togo);
                up->ub_cpos += togo;
                cp += togo;
                togo = 0;

            } else {
                if (r > 0) {    /* frag contains part of chunk */
                    bcopy(fp->fr_dat + up->ub_cpos, cp, r);
                    up->ub_cpos += r;
                    togo -= r;
                    cp += r;

                } else {        /* no space, add new frag */
                    if (r = dec_step())
                        return r;
                }
            }
        }
        cp += lnc;
    }
    return 0;
}


static int
enc_raw_init()
{
    return 0;
}


static int
dec_raw_init()
{
    struct umbuf *up = midtobuf(rbufmid);

    up->ub_cfrag = up->ub_frag->fr_link;
    up->ub_cpos = 0;
    return 0;
}


static int
enc_raw_any(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    return bytepk((char*)vp, cnt, siz, std * siz);
}


static int
dec_raw_any(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    return byteupk((char*)vp, cnt, siz, std * siz);
}


/***************************
 **  XDR-format encoders  **
 **                       **
 ***************************/

static int
enc_ext_short(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register short *np;
    register char *cp;
    char     buf[2];
    int cc = 0;

    for (np = (short*)vp; cnt-- > 0; np += std) {
        cp = (char *)np;
        buf[1] = *cp;
        buf[0] = *(cp+1);
        if (cc = bytepk(buf, 2, 1, 1))
            return cc;
    }
    return 0;
}

static int
enc_ext_int(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register int *np;
    register char *cp;
    char     buf[4];
    int cc = 0;

    for (np = (int *)vp; cnt-- > 0; np += std) {
        cp = (char *)np;
        buf[3] = *cp;
        buf[2] = *(cp+1);
        buf[1] = *(cp+2);
        buf[0] = *(cp+3);
        if (cc = bytepk(buf, 4, 1, 1))
            return cc;
    }
    return 0;
}

static int
enc_ext_long(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register long *np;
    register char *cp;
    char     buf[4];
    int cc = 0;

    for (np = (long *)vp; cnt-- > 0; np += std) {
        cp = (char *)np;
        buf[3] = *cp;
        buf[2] = *(cp+1);
        buf[1] = *(cp+2);
        buf[0] = *(cp+3);
        if (cc = bytepk(buf, 4, 1, 1))
            return cc;
    }
    return 0;
}

static int
enc_ext_float(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register float *np;
    register char *cp;
    char     buf[4];
    int cc = 0;

    for (np = (float *)vp; cnt-- > 0; np += std) {
        cp = (char *)np;
        buf[3] = *cp;
        buf[2] = *(cp+1);
        buf[1] = *(cp+2);
        buf[0] = *(cp+3);
        if (cc = bytepk(buf, 4, 1, 1))
            return cc;
    }
    return 0;
}

static int
enc_ext_double(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register double *np;
    register char *cp;
    char     buf[4];
    int cc = 0;

    for (np = (double *)vp; cnt-- > 0; np += std) {
        cp = (char *)np;
        buf[7] = *cp;
        buf[6] = *(cp+1);
        buf[5] = *(cp+2);
        buf[4] = *(cp+3);
        buf[3] = *(cp+4);
        buf[2] = *(cp+5);
        buf[1] = *(cp+6);
        buf[0] = *(cp+7);
        if (cc = bytepk(buf, 8, 1, 1))
            return cc;
    }
    return 0;
}

typedef struct {float r, i} complex;
typedef struct {double r, i} dcplx;

static int
enc_ext_cplx(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
	register complex *np;
    register char *cp;
	char buf[8];
    int cc = 0;

    for (np = (complex *)vp; cnt-- > 0; np += std) {
        cp = (char *)np;
		buf[3] = *cp;
        buf[2] = *(cp+1);
        buf[1] = *(cp+2);
        buf[0] = *(cp+3);
        buf[7] = *(cp+4);
        buf[6] = *(cp+5);
        buf[5] = *(cp+6);
        buf[4] = *(cp+7);
        if (cc = bytepk(buf, 8, 1, 1))
            return cc;
    }
    return 0;
}

static int
enc_ext_dcplx(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register dcplx *np;
    register char *cp;
    char buf[16];
    int cc = 0;

    for (np = (dcplx *)vp; cnt-- > 0; np += std) {
        cp = (char *)np;
        buf[7] = *cp;
        buf[6] = *(cp+1);
        buf[5] = *(cp+2);
        buf[4] = *(cp+3);
        buf[3] = *(cp+4);
        buf[2] = *(cp+5);
        buf[1] = *(cp+6);
        buf[0] = *(cp+7);
		buf[15] = *(cp+8);
		buf[14] = *(cp+9);
		buf[13] = *(cp+10);
		buf[12] = *(cp+11);
		buf[11] = *(cp+12);
		buf[10] = *(cp+13);
		buf[9] = *(cp+14);
		buf[8] = *(cp+15);
        if (cc = bytepk(buf, 16, 1, 1))
            return cc;
    }
    return 0;
}

static int
dec_ext_short(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register short *np;
    register char *cp;
    char     buf[2];
    int cc = 0;

    for (np = (short*)vp; cnt-- > 0; np += std) {
        if (cc = byteupk(buf, 2, 1, 1))
            return cc;
        cp = (char *)np;
        *cp = buf[1];
        *(cp+1) = buf[0];
    }
    return 0;
}

static int
dec_ext_int(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register int *np;
    register char *cp;
    char     buf[4];
    int cc = 0;

    for (np = (int *)vp; cnt-- > 0; np += std) {
        if (cc = byteupk(buf, 4, 1, 1))
            return cc;
        cp = (char *)np;
        *cp = buf[3];
        *(cp+1) = buf[2];
        *(cp+2) = buf[1];
        *(cp+3) = buf[0];
    }
    return 0;
}

static int
dec_ext_long(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register long *np;
    register char *cp;
    char     buf[4];
    int cc = 0;

    for (np = (long *)vp; cnt-- > 0; np += std) {
        if (cc = byteupk(buf, 4, 1, 1))
            return cc;
        cp = (char *)np;
        *cp = buf[3];
        *(cp+1) = buf[2];
        *(cp+2) = buf[1];
        *(cp+3) = buf[0];
    }
    return 0;
}

static int
dec_ext_float(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register float *np;
    register char *cp;
    char     buf[4];
    int cc = 0;

    for (np = (float *)vp; cnt-- > 0; np += std) {
        if (cc = byteupk(buf, 4, 1, 1))
            return cc;
        cp = (char *)np;
        *cp = buf[3];
        *(cp+1) = buf[2];
        *(cp+2) = buf[1];
        *(cp+3) = buf[0];
    }
    return 0;
}

static int
dec_ext_double(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register double *np;
    register char *cp;
    char     buf[4];
    int cc = 0;

    for (np = (double *)vp; cnt-- > 0; np += std) {
        if (cc = byteupk(buf, 8, 1, 1))
            return cc;
        cp = (char *)np;
        *cp = buf[7];
        *(cp+1) = buf[6];
        *(cp+2) = buf[5];
        *(cp+3) = buf[4];
        *(cp+4) = buf[3];
        *(cp+5) = buf[2];
        *(cp+6) = buf[1];
        *(cp+7) = buf[0];
    }
    return 0;
}

static int
dec_ext_cplx(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register complex *np;
    register char *cp;
    char     buf[8];
    int cc = 0;

    for (np = (complex *)vp; cnt-- > 0; np += std) {
        if (cc = byteupk(buf, 8, 1, 1))
            return cc;
        cp = (char *)np;
        *cp = buf[3];
        *(cp+1) = buf[2];
        *(cp+2) = buf[1];
        *(cp+3) = buf[0];
		*(cp+4) = buf[7];
		*(cp+5) = buf[6];
		*(cp+6) = buf[5];
		*(cp+7) = buf[4];
    }
    return 0;
}

static int
dec_ext_dcplx(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register dcplx *np;
    register char *cp;
    char     buf[16];
    int cc = 0;

    for (np = (dcplx *)vp; cnt-- > 0; np += std) {
        if (cc = byteupk(buf, 16, 1, 1))
            return cc;
        cp = (char *)np;
        *cp     = buf[7];
        *(cp+1) = buf[6];
        *(cp+2) = buf[5];
        *(cp+3) = buf[4];
        *(cp+4) = buf[3];
        *(cp+5) = buf[2];
        *(cp+6) = buf[1];
        *(cp+7) = buf[0];
		*(cp+8) = buf[15];
		*(cp+9) = buf[14];
		*(cp+10) = buf[13];
		*(cp+11) = buf[12];
		*(cp+12) = buf[11];
		*(cp+13) = buf[10];
		*(cp+14) = buf[9];
		*(cp+15) = buf[8];
    }
    return 0;
}


/*************************
 **  In-place encoders  **
 **                     **
 *************************/

static int
enc_inp_init()
{
    return 0;
}


static int
enc_inp_byte(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    struct dat_blk d;

    d.type = INP_BYTE;
    d.base = vp;
    d.cnt = cnt;
    d.std = std;
    return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_short(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    struct dat_blk d;

    d.type = INP_SHORT;
    d.base = vp;
    d.cnt = cnt;
    d.std = std;
    return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_int(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    struct dat_blk d;

    d.type = INP_INT;
    d.base = vp;
    d.cnt = cnt;
    d.std = std;
    return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_long(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    struct dat_blk d;

    d.type = INP_LONG;
    d.base = vp;
    d.cnt = cnt;
    d.std = std;
    return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_float(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    struct dat_blk d;

    d.type = INP_FLOAT;
    d.base = vp;
    d.cnt = cnt;
    d.std = std;
    return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_double(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    struct dat_blk d;

    d.type = INP_DFLOAT;
    d.base = vp;
    d.cnt = cnt;
    d.std = std;
    return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_cplx(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    struct dat_blk d;

    d.type = INP_CPLX;
    d.base = vp;
    d.cnt = cnt;
    d.std = std;
    return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_dcplx(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    struct dat_blk d;

    d.type = INP_DCPLX;
    d.base = vp;
    d.cnt = cnt;
    d.std = std;
    return bytepk((char*)&d, 1, sizeof(d), 0);
}


/*******************
 **  Foo encoders **
 **               **
 *******************/

static int
enc_foo_init()
{
    return 0;
}


static int
enc_foo_byte(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    return bytepk((char*)vp, cnt, siz, std * siz);
}


static int
enc_foo_int(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register int *np;
    char buf[4];
    int n;
    int cc;

    for (np = (int*)vp; cnt-- > 0; np += std) {
        n = *np;
        buf[0] = n >> 24;
        buf[1] = n >> 16;
        buf[2] = n >> 8;
        buf[3] = n;
        if (cc = bytepk(buf, 4, 1, 1))
            return cc;
    }
    return 0;
}


static int
dec_foo_init()
{
    struct umbuf *up = midtobuf(rbufmid);

    up->ub_cfrag = up->ub_frag->fr_link;
    up->ub_cpos = 0;
    return 0;
}


static int
dec_foo_byte(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    return byteupk((char*)vp, cnt, siz, std * siz);
}


static int
dec_foo_int(vp, cnt, std, siz)
    void *vp;
    int cnt, std, siz;
{
    register int *np;
    char buf[4];
    int cc;

    for (np = (int*)vp; cnt-- > 0; np += std) {
        if (cc = byteupk(buf, 4, 1, 1))
            return cc;
        *np = (0x80 & buf[0] ? ~0xffffffff : 0)
        + ((0xff & buf[0]) << 24)
        + ((0xff & buf[1]) << 16)
        + ((0xff & buf[2]) << 8)
        + (0xff & buf[3]);
    }
    return 0;
}


static int
enc_foo_err()
{
    return PvmNotImpl;  /* don't implement this :-) */
}


/*******************************************
 **  Alien (can't do it) en/decoder stub  **
 **                                       **
 *******************************************/

static int
enc_alien()
{
    return PvmBadMsg;
}


/****************************
 **  Encoder switch table  **
 **                        **
 ****************************/

static struct encvec encoders[] = {

    {   enc_raw_init, dec_raw_init,     /* Raw (native) 0 */
        enc_raw_any, dec_raw_any,
        enc_raw_any, dec_raw_any,
        enc_raw_any, dec_raw_any,
        enc_raw_any, dec_raw_any,
        enc_raw_any, dec_raw_any,
        enc_raw_any, dec_raw_any,
        enc_raw_any, dec_raw_any,
        enc_raw_any, dec_raw_any },

    {   enc_foo_init, dec_foo_init,     /* XDR 1 */
        enc_foo_byte, dec_foo_byte,
        enc_ext_short, dec_ext_short,
        enc_ext_int, dec_ext_int,
        enc_ext_int, dec_ext_int,
        enc_ext_float, dec_ext_float,
        enc_ext_double, dec_ext_double,
        enc_ext_cplx, dec_ext_cplx,
        enc_ext_dcplx, dec_ext_dcplx },

    {   enc_inp_init, enc_alien,        /* In place (encode only) 2 */
        enc_inp_byte, enc_alien,
        enc_inp_short, enc_alien,
        enc_inp_int, enc_alien,
        enc_inp_long, enc_alien,
        enc_inp_float, enc_alien,
        enc_inp_double, enc_alien,
        enc_inp_cplx, enc_alien,
        enc_inp_dcplx, enc_alien },

    {   enc_foo_init, dec_foo_init,     /* Foo (for pvmd) 3 */
        enc_foo_byte, dec_foo_byte,
        enc_foo_err, enc_foo_err,
        enc_foo_int, dec_foo_int,
        enc_foo_err, enc_foo_err,
        enc_foo_err, enc_foo_err,
        enc_foo_err, enc_foo_err,
        enc_foo_err, enc_foo_err,
        enc_foo_err, enc_foo_err },

    {   enc_alien, enc_alien,           /* Alien (can't deal) 4 */
        enc_alien, enc_alien,
        enc_alien, enc_alien,
        enc_alien, enc_alien,
        enc_alien, enc_alien,
        enc_alien, enc_alien,
        enc_alien, enc_alien,
        enc_alien, enc_alien,
        enc_alien, enc_alien },
};


/*  enctovec()
*
*   Map data encoding value to encode/decode function vector
*/

struct encvec *
enctovec(enc)
    int enc;
{
    if (enc == 0)           /* xdr */
        return &encoders[1];
    if (enc == 1)           /* foo */
        return &encoders[3];
    if (enc == myndf)       /* native */
        return &encoders[0];
    return &encoders[4];    /* alien */
}


int
enc_step()
{
    struct frag *fp = midtobuf(sbufmid)->ub_frag;
    struct frag *nfp;

    if (!(nfp = fr_new(ourudpmtu)))
        return PvmNoMem;
    nfp->fr_dat += MAXHDR;
    LISTPUTBEFORE(fp, nfp, fr_link, fr_rlink);
    return 0;
}


int
dec_step()
{
    struct umbuf *up = midtobuf(rbufmid);

    up->ub_cpos = 0;
    if (up->ub_cfrag == up->ub_frag)        /* no message left */
        return PvmNoData;
    up->ub_cfrag = up->ub_cfrag->fr_link;
    if (up->ub_cfrag == up->ub_frag)
        return PvmNoData;
    return 0;
}


/************************
 **  Libpvm Functions  **
 **                    **
 ************************/

int
pvm_mkbuf(enc)
    int enc;        /* data format */
{
    int mid;
    struct umbuf *up;
    struct frag *fp;

    switch (enc) {                      /* verify encoding */
    case PvmDataDefault:
        enc = 0;        /* XXX this just forces XDR */
        break;

    case PvmDataRaw:
        enc = myndf;
        break;

    case PvmDataFoo:
        enc = 1;
        break;

    case PvmDataInPlace:
        return uliberr("pvm_mkbuf", PvmNotImpl);
        break;

    default:
        return uliberr("pvm_mkbuf", PvmBadParam);
    }
    if ((mid = umbuf_new()) <= 0)
        return uliberr("pvm_mkbuf", mid);

    up = midtobuf(mid);                 /* set encoders */
    up->ub_enc = enc;
    up->ub_codef = enctovec(enc);
    up->ub_src = mypvmtid;

    fp = fr_new(ourudpmtu);             /* make a blank frag to write in */
    fp->fr_dat += MAXHDR;
    LISTPUTBEFORE(up->ub_frag, fp, fr_link, fr_rlink);

    return mid;
}


int
pvm_freebuf(mid)
    int mid;        /* message handle */
{
    int cc;

    if (mid <= 0 || mid >= midhsiz)
        return uliberr("pvm_freebuf", PvmBadParam);
    if (sbufmid == mid)
        sbufmid = 0;
    if (rbufmid == mid)
        rbufmid = 0;
    if ((cc = umbuf_free(mid)) < 0)
        return uliberr("pvm_freebuf", cc);
    return 0;
}


int
pvm_setsbuf(mid)
    int mid;
{
    int omid;
    struct umbuf *up = 0;

    if (mid < 0 || mid >= midhsiz)
        return uliberr("pvm_setsbuf", PvmBadParam);
    if (mid) {
        if (!(up = midh[mid].m_umb))
            return uliberr("pvm_setsbuf", PvmNoSuchBuf);
        if (mid == rbufmid)
            rbufmid = 0;
    }
    omid = sbufmid > 0 ? sbufmid : 0;
    sbufmid = mid;
    if (up && !(up->ub_flag & UB_PACK)) {
        (up->ub_codef->enc_init)();
        up->ub_flag &= ~UB_UPACK;
        up->ub_flag |= UB_PACK;
    }
    return omid;
}


int
pvm_getsbuf()
{
    return sbufmid > 0 ? sbufmid : 0;
}


int
pvm_setrbuf(mid)
    int mid;
{
    int omid;
    struct umbuf *up = 0;

    if (mid < 0 || mid >= midhsiz)
        return uliberr("pvm_setrbuf", PvmBadParam);
    if (mid) {
        if (!(up = midh[mid].m_umb))
            return uliberr("pvm_setrbuf", PvmNoSuchBuf);
        if (mid == sbufmid)
            sbufmid = 0;
    }
    omid = rbufmid > 0 ? rbufmid : 0;
    rbufmid = mid;
    if (up && !(up->ub_flag & UB_UPACK)) {
        (up->ub_codef->dec_init)();
        up->ub_flag &= ~UB_PACK;
        up->ub_flag |= UB_UPACK;
    }
    return omid;
}


int
pvm_getrbuf()
{
    return rbufmid > 0 ? rbufmid : 0;
}


int
pvm_initsend(enc)
    int enc;
{
    int mid;

    if ((mid = pvm_mkbuf(enc)) < 0)
        return uliberr("pvm_initsend", mid);
    if (sbufmid > 0)
        pvm_freebuf(sbufmid);
    pvm_setsbuf(mid);
    return mid;
}


int
pvm_bufinfo(mid, len, code, tid)
    int mid;
    int *len;
    int *code;
    int *tid;
{
    struct umbuf *up;

    if (mid <= 0 || mid >= midhsiz)
        return uliberr("pvm_bufinfo", PvmBadParam);
    if (!(up = midh[mid].m_umb))
        return uliberr("pvm_bufinfo", PvmNoSuchBuf);
    if (len)
        *len = up->ub_len;
    if (code)
        *code = up->ub_cod;
    if (tid)
        *tid = up->ub_src;
    return PvmOk;
}


int
pvm_pkbyte(cp, cnt, std)
    char *cp;
    int cnt, std;
{
    int cc;

    if (sbufmid <= 0)
        return uliberr("pvm_pkbyte", PvmNoBuf);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_byte)((void*)cp, cnt, std, 1))
        return uliberr("pvm_pkbyte", cc);
    return PvmOk;
}


int
pvm_upkbyte(cp, cnt, std)
    char *cp;
    int cnt, std;
{
    int cc;

    if (rbufmid <= 0)
        return uliberr("pvm_upkbyte", PvmNoBuf);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_byte)((void*)cp, cnt, std, 1))
        return uliberr("pvm_upkbyte", cc);
    return PvmOk;
}


int
pvm_pkcplx(xp, cnt, std)
    float *xp;
    int cnt, std;
{
    int cc;

    if (sbufmid <= 0)
        return uliberr("pvm_pkcplx", PvmNoBuf);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_cplx)
            ((void*)xp, cnt, std, sizeof(float)*2))
        return uliberr("pvm_pkcplx", cc);
    return PvmOk;
}


int
pvm_upkcplx(xp, cnt, std)
    float *xp;
    int cnt, std;
{
    int cc;

    if (rbufmid <= 0)
        return uliberr("pvm_upkcplx", PvmNoBuf);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_cplx)
            ((void*)xp, cnt, std, sizeof(float)*2))
        return uliberr("pvm_upkcplx", cc);
    return PvmOk;
}


int
pvm_pkdcplx(zp, cnt, std)
    double *zp;
    int cnt, std;
{
    int cc;

    if (sbufmid <= 0)
        return uliberr("pvm_pkdcplx", PvmNoBuf);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_dcplx)
            ((void*)zp, cnt, std, sizeof(double)*2))
        return uliberr("pvm_pkdcplx", cc);
    return PvmOk;
}


int
pvm_upkdcplx(zp, cnt, std)
    double *zp;
    int cnt, std;
{
    int cc;

    if (rbufmid <= 0)
        return uliberr("pvm_upkdcplx", PvmNoBuf);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_dcplx)
            ((void*)zp, cnt, std, sizeof(double)*2))
        return uliberr("pvm_upkdcplx", cc);
    return PvmOk;
}


int
pvm_pkdouble(dp, cnt, std)
    double *dp;
    int cnt, std;
{
    int cc;

    if (sbufmid <= 0)
        return uliberr("pvm_pkdouble", PvmNoBuf);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_double)
            ((void*)dp, cnt, std, sizeof(double)))
        return uliberr("pvm_pkdouble", cc);
    return PvmOk;
}


int
pvm_upkdouble(dp, cnt, std)
    double *dp;
    int cnt, std;
{
    int cc;

    if (rbufmid <= 0)
        return uliberr("pvm_upkdouble", PvmNoBuf);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_double)
            ((void*)dp, cnt, std, sizeof(double)))
        return uliberr("pvm_upkdouble", cc);
    return PvmOk;
}


int
pvm_pkfloat(fp, cnt, std)
    float *fp;
    int cnt, std;
{
    int cc;

    if (sbufmid <= 0)
        return uliberr("pvm_pkfloat", PvmNoBuf);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_float)
            ((void*)fp, cnt, std, sizeof(float)))
        return uliberr("pvm_pkfloat", cc);
    return PvmOk;
}


int
pvm_upkfloat(fp, cnt, std)
    float *fp;
    int cnt, std;
{
    int cc;

    if (rbufmid <= 0)
        return uliberr("pvm_upkfloat", PvmNoBuf);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_float)
            ((void*)fp, cnt, std, sizeof(float)))
        return uliberr("pvm_upkfloat", cc);
    return PvmOk;
}


int
pvm_pkint(np, cnt, std)
    int *np;
    int cnt, std;
{
    int cc;

    if (sbufmid <= 0)
        return uliberr("pvm_pkint", PvmNoBuf);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_int)
            ((void*)np, cnt, std, sizeof(int)))
        return uliberr("pvm_pkint", cc);
    return PvmOk;
}


int
pvm_upkint(np, cnt, std)
    int *np;
    int cnt, std;
{
    int cc;

    if (rbufmid <= 0)
        return uliberr("pvm_upkint", PvmNoBuf);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_int)
            ((void*)np, cnt, std, sizeof(int)))
        return uliberr("pvm_upkint", cc);
    return PvmOk;
}


int
pvm_pklong(np, cnt, std)
    long *np;
    int cnt, std;
{
    int cc;

    if (sbufmid <= 0)
        return uliberr("pvm_pklong", PvmNoBuf);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_long)
            ((void*)np, cnt, std, sizeof(long)))
        return uliberr("pvm_pklong", cc);
    return PvmOk;
}


int
pvm_upklong(np, cnt, std)
    long *np;
    int cnt, std;
{
    int cc;

    if (rbufmid <= 0)
        return uliberr("pvm_upklong", PvmNoBuf);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_long)
            ((void*)np, cnt, std, sizeof(long)))
        return uliberr("pvm_upklong", cc);
    return PvmOk;
}


int
pvm_pkshort(np, cnt, std)
    short *np;
    int cnt, std;
{
    int cc;

    if (sbufmid <= 0)
        return uliberr("pvm_pkshort", PvmNoBuf);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_short)
            ((void*)np, cnt, std, sizeof(short)))
        return uliberr("pvm_pkshort", cc);
    return PvmOk;
}


int
pvm_upkshort(np, cnt, std)
    short *np;
    int cnt, std;
{
    int cc;

    if (rbufmid <= 0)
        return uliberr("pvm_upkshort", PvmNoBuf);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_short)
            ((void*)np, cnt, std, sizeof(short)))
        return uliberr("pvm_upkshort", cc);
    return PvmOk;
}


int
pvm_pkstr(cp)
    char *cp;
{
    int cc;
    int l = strlen(cp) + 1;

    if (sbufmid <= 0)
        return uliberr("pvm_pkstr", PvmNoBuf);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_int)
            ((void*)&l, 1, 1, sizeof(int)))
        return uliberr("pvm_pkstr", cc);
    if (cc = (midtobuf(sbufmid)->ub_codef->enc_byte)
            ((void*)cp, l, 1, 1))
        return uliberr("pvm_pkstr", cc);
    return PvmOk;
}


int
pvm_upkstr(cp)
    char *cp;
{
    int l;
    int cc;

    if (rbufmid <= 0)
        return uliberr("pvm_upkstr", PvmNoBuf);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_int)
            ((void*)&l, 1, 1, sizeof(int)))
        return uliberr("pvm_upkstr", cc);
    if (cc = (midtobuf(rbufmid)->ub_codef->dec_byte)
            ((void*)cp, l, 1, 1))
        return uliberr("pvm_upkstr", cc);
    return PvmOk;
}


