/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 * PVM3.0-I860
 *
 *  cubed.c:  Cube Daemon, special task to be spawned by pvmd to 
 *		manage node processes
 *			usage: cubed file count path1 path2 ...
 *
 *      send bug reports to:	wjiang@cs.utk.edu
 *
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <stdio.h>
#include <cube.h>
#include "pvm3.h"
#include "global.h"
#include "ddpro.h"
#include "tdpro.h"
#include "myalloc.h"
#include "mycube.h"

#define MAXPATHLEN  CANBSIZ
#define NUMARGS	    3		/* minimum number of command-line arguments */

extern short    qm_cubeid;	/* from libhost.a */
extern int      sbufmid;	/* from pack.c */
extern int		mypvmtid;	/* from lpvm.c */
extern int		tidhmask;	/* from lpvm.c */
extern int		ourudpmtu;	/* from lpvm.c */
int tidcmask = TIDCUBE;     /* mask for cube field of tids */
int tidnmask = TIDCLOCAL;   /* mask for node field of tids */
int tidnode = TIDNODE;      /* this bit turned on for node tid */

/* private */
static int      cube_spawn();
static char		etext[512];
static int		cubepart;	/* cube portion of tids */

main(argc, argv)
	int     argc;
	char    **argv;
{
	int		pid;
	char    cname[CUBENAMELEN];
	int     err, cc;
	int     sbf, rbf;
	int     npaths = 0;		/* the number of search paths for executable */
	char    **epaths = 0;
	int     cid = -1;
	int 	log_fd = -1;	/* error log file */
	char 	fname[32];
	int		ntasks;			/* number of tasks to start */
	int		*tids = 0;
	char	*buf;
	int		msglen;
	long	node;

	if (argc < NUMARGS) {
		fputs("usage: cubed file count path1 path2 ...\n", stderr);
		exit(1);
	}
/*
	sprintf(fname, CUBELOGFILE, getuid());
	if ((log_fd = open(fname, O_WRONLY|O_CREAT|O_TRUNC|O_APPEND, 0666)) != -1) {
		dup2(log_fd, 1);
		dup2(1, 2);
	}
*/
	if (cc = beatask()) {
		fputs("cubed: can't connect to pvmd", stderr);
		exit(1);
	}
	pid = getpid();
	MAKECUBENAME(cname, pid);
	err = _getcube(cname, argv[2], NULL, 0);
	if (err < 0) {
		err = PvmNoCube;
		perror("cubed: getcube");
	} else {
		cid = qm_cubeid;
		if (_setpid(CUBEDPID) < 0)
			perror("cubed: can't setpid\n");
		if (argc > NUMARGS) {
			npaths = argc - NUMARGS;
			epaths = &(argv[NUMARGS]);
		}
		ntasks = atoi(argv[2]);
		err = cube_spawn(epaths, npaths, argv[1], ntasks);
	}

	sbf = pvm_setsbuf(pvm_mkbuf(PvmDataFoo));
	rbf = pvm_setrbuf(0);
	pvm_pkint(&err, 1, 0);
	pvm_pkstr(cname);
	if (!err) {
		int		i, *tid;

		pvm_pkint(&cid, 1, 0);
		tids = TALLOC(ntasks, int, "tids");
		for (i = 0, tid = tids; i < ntasks; i++, tid++)
			if (_crecv(PMTTID, tid, sizeof(int)) < 0) {
				perror("cubed: crecv tids");
				*tid = PvmDSysErr;
			}
		CTOHL(tids,ntasks);
		pvm_pkint(tids, ntasks, 1);
	}
	if ((cc = msendrecv(TIDPVMD, TM_SPAWNACK)) > 0)
		pvm_freebuf(pvm_setrbuf(rbf));
	else
		pvm_setrbuf(rbf);
	pvm_freebuf(pvm_setsbuf(sbf));
	if (cc < 0)
		uliberr("cubed: TM_SPAWNACK", cc);
	if (tids)
		PVM_FREE(tids);
	if (err == PvmNoCube) {
		pvm_exit();
		exit(1);
	}

	/* relay messages between pvmd and nodes */

	buf = TALLOC(ourudpmtu, char, "msg");
	while (_crecv(PMTNODE, buf, ourudpmtu) != -1 
			&& (msglen = _infocount()) != -1
			&& (node = _infonode()) != -1) {
		int		dst;	/* destination node */
		char	token[1];

		if ((cc = sendtopvmd(buf, msglen)) < 0)
			uliberr("cubed: sendto", cc);
	}
	perror("cubed exiting: err crecv\n", stderr);
	pvm_exit();
	exit(2);
}

/* load process(es) onto allocated cube */
static int
cube_spawn(epaths, npaths, file, count)
	char          **epaths;
	int             npaths;
	char           *file;
	int             count;
{
	char            path[MAXPATHLEN];
	struct stat     sb;
	char          **ep;
	int    			i;

	/* search for file */

	for (; npaths--; epaths++) {
		(void) strcpy(path, getenv("HOME"));	/* load() needs absolute path */
		(void) strcat(path, "/");
		(void) strcat(path, *epaths);
		(void) strcat(path, "/");
		(void) strncat(path, file, sizeof(path) - strlen(path) - 1);

		if (stat(path, &sb) == -1
		    || ((sb.st_mode & S_IFMT) != S_IFREG)
		    || !(sb.st_mode & S_IEXEC)) {
			continue;
		}

		/* file found */
		cubepart = CUBEPART(mypvmtid & tidhmask, qm_cubeid, tidcmask);
		cubepart |= tidnode;
		HTOCL(&cubepart,1);
		if (count == (1 << (ffs(count) - 1))) { 		/* power of 2 */
			if (_load(path, -1, 0) < 0) {
				perror("cubed: load");
				return PvmDSysErr;
			} else {
				if (_csend(PMTTID, &cubepart, sizeof(int), -1, 0) < 0)
					perror("cubed: csend");
			}
		} else {
			for (i = 0; i < count; i++)
				if (_load(path, (long)i, 0) < 0) {
					sprintf(etext, "loading node %d", i);
					perror(etext);
					return PvmDSysErr;
				} else {
					if (_csend(PMTTID, &cubepart, sizeof(int), (long *)i, 0) < 0)
						perror("csend");
				}
		}
		return 0;
	}
	return PvmNoFile;
}
