static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * This program is free software: you can redistribute it and/or modify       \n"
" * it under the terms of the GNU General Public License as published by       \n"
" * the Free Software Foundation; either version 3 of the License, or          \n"
" * (at your option) any later version.                                        \n"
" *                                                                            \n"
" * This program is distributed in the hope that it will be useful,            \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              \n"
" * GNU General Public License for more details.                               \n"
" *                                                                            \n"
" * You should have received a copy of the GNU General Public License          \n"
" * along with this program.  If not, see <http://www.gnu.org/licenses/>.      \n"
" *                                                                            \n"
" * This operation is a port of the GIMP Apply lens plug-in                    \n"
" * Copyright (C) 1997 Morten Eriksen mortene@pvv.ntnu.no                      \n"
" *                                                                            \n"
" * Porting to GEGL:                                                           \n"
" * Copyright 2013 Emanuel Schrade <emanuel.schrade@student.kit.edu>           \n"
" * Copyright 2013 Stephan Seifermann <stephan.seifermann@student.kit.edu>     \n"
" * Copyright 2013 Bastian Pirk <bastian.pirk@student.kit.edu>                 \n"
" * Copyright 2013 Pascal Giessler <pascal.giessler@student.kit.edu>           \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_double (refraction_index, _(\"Lens refraction index\"), 1.7)         \n"
"  value_range (1.0, 100.0)                                                    \n"
"                                                                              \n"
"property_boolean (keep_surroundings, _(\"Keep original surroundings\"), FALSE)\n"
"  description(_(\"Keep image unchanged, where not affected by the lens.\"))   \n"
"                                                                              \n"
"property_color (background_color, _(\"Background color\"), \"none\")          \n"
"  //ui_meta (\"role\", \"color-secondary\")                                   \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_C_SOURCE apply-lens.c                                         \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <math.h>                                                             \n"
"#include <stdio.h>                                                            \n"
"                                                                              \n"
"/**                                                                           \n"
" * Computes the projected position (projx, projy) of the                      \n"
" * original point (x, y) after passing through the lens                       \n"
" * which is given by its center and its refraction index.                     \n"
" * See: Ellipsoid formula: x^2/a^2 + y^2/b^2 + z^2/c^2 = 1.                   \n"
" *                                                                            \n"
" * @param a2 semiaxis a                                                       \n"
" * @param b2 semiaxis b                                                       \n"
" * @param c2 semiaxis c                                                       \n"
" * @param x coordinate x                                                      \n"
" * @param y coordinate y                                                      \n"
" * @param refraction refraction index                                         \n"
" * @param projy inout of the projection x                                     \n"
" * @param projy inout of the projection y                                     \n"
" */                                                                           \n"
"static void                                                                   \n"
"find_projected_pos (gfloat  a2,                                               \n"
"                    gfloat  b2,                                               \n"
"                    gfloat  c2,                                               \n"
"                    gfloat  x,                                                \n"
"                    gfloat  y,                                                \n"
"                    gfloat  refraction,                                       \n"
"                    gfloat *projx,                                            \n"
"                    gfloat *projy)                                            \n"
"{                                                                             \n"
"  gfloat z;                                                                   \n"
"  gfloat nxangle, nyangle, theta1, theta2;                                    \n"
"  gfloat ri1 = 1.0;                                                           \n"
"  gfloat ri2 = refraction;                                                    \n"
"                                                                              \n"
"  z = sqrt ((1 - x * x / a2 - y * y / b2) * c2);                              \n"
"                                                                              \n"
"  nxangle = acos (x / sqrt(x * x + z * z));                                   \n"
"  theta1 = G_PI / 2 - nxangle;                                                \n"
"  theta2 = asin (sin (theta1) * ri1 / ri2);                                   \n"
"  theta2 = G_PI / 2 - nxangle - theta2;                                       \n"
"  *projx = x - tan (theta2) * z;                                              \n"
"                                                                              \n"
"  nyangle = acos (y / sqrt (y * y + z * z));                                  \n"
"  theta1 = G_PI / 2 - nyangle;                                                \n"
"  theta2 = asin (sin (theta1) * ri1 / ri2);                                   \n"
"  theta2 = G_PI / 2 - nyangle - theta2;                                       \n"
"  *projy = y - tan (theta2) * z;                                              \n"
"}                                                                             \n"
"                                                                              \n"
"/**                                                                           \n"
" * Prepare function of gegl filter.                                           \n"
" * @param operation given Gegl operation                                      \n"
" */                                                                           \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  const Babl *format = gegl_operation_get_source_format (operation, \"input\");\n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"input\", format);                   \n"
"  gegl_operation_set_format (operation, \"output\", format);                  \n"
"}                                                                             \n"
"                                                                              \n"
"/**                                                                           \n"
" * Returns the cached region. This is an area filter, which acts on the whole image.\n"
" * @param operation given Gegl operation                                      \n"
" * @param roi the rectangle of interest                                       \n"
" * @return result the new rectangle                                           \n"
" */                                                                           \n"
"static GeglRectangle                                                          \n"
"get_cached_region (GeglOperation       *operation,                            \n"
"                   const GeglRectangle *roi)                                  \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"/**                                                                           \n"
" * Process the gegl filter                                                    \n"
" * @param operation the given Gegl operation                                  \n"
" * @param input the input buffer.                                             \n"
" * @param output the output buffer.                                           \n"
" * @param result the region of interest.                                      \n"
" * @param level the level of detail                                           \n"
" * @return True, if the filter was successfull applied.                       \n"
" */                                                                           \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *input,                                          \n"
"         GeglBuffer          *output,                                         \n"
"         const GeglRectangle *result,                                         \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglProperties         *o = GEGL_PROPERTIES (operation);                    \n"
"  const Babl  *input_format = gegl_buffer_get_format (input);                 \n"
"  const int bytes_per_pixel = babl_format_get_bytes_per_pixel (input_format); \n"
"                                                                              \n"
"  /**                                                                         \n"
"   * src_buf, dst_buf:                                                        \n"
"   * Input- and output-buffers, containing the whole selection,               \n"
"   * the filter should be applied on.                                         \n"
"   *                                                                          \n"
"   * src_pixel, dst_pixel:                                                    \n"
"   * pointers to the current source- and destination-pixel.                   \n"
"   * Using these pointers, the reading and writing can be delayed till the    \n"
"   * end of the loop. Hence src_pixel can also point to background_color if   \n"
"   * necessary.                                                               \n"
"   */                                                                         \n"
"  guint8    *src_buf, *dst_buf, *src_pixel, *dst_pixel, background_color[bytes_per_pixel];\n"
"                                                                              \n"
"  /**                                                                         \n"
"   * (x, y): Position of the pixel we compute at the moment.                  \n"
"   * (width, height): Dimensions of the lens                                  \n"
"   * pixel_offset: For calculating the pixels position in src_buf / dst_buf.  \n"
"   */                                                                         \n"
"  gint         x, y,                                                          \n"
"               width, height,                                                 \n"
"               pixel_offset, projected_pixel_offset;                          \n"
"                                                                              \n"
"  /**                                                                         \n"
"   * Further parameters that are needed to calculate the position of a projected\n"
"   * further pixel at (x, y).                                                 \n"
"   */                                                                         \n"
"  gfloat       a, b, c,                                                       \n"
"               asqr, bsqr, csqr,                                              \n"
"               dx, dy, dysqr,                                                 \n"
"               projected_x, projected_y,                                      \n"
"               refraction_index;                                              \n"
"                                                                              \n"
"  gboolean     keep_surroundings;                                             \n"
"                                                                              \n"
"  width = result->width;                                                      \n"
"  height = result->height;                                                    \n"
"                                                                              \n"
"  refraction_index = o->refraction_index;                                     \n"
"  keep_surroundings = o->keep_surroundings;                                   \n"
"  gegl_color_get_pixel (o->background_color, input_format, background_color); \n"
"                                                                              \n"
"  a = 0.5 * width;                                                            \n"
"  b = 0.5 * height;                                                           \n"
"  c = MIN (a, b);                                                             \n"
"  asqr = a * a;                                                               \n"
"  bsqr = b * b;                                                               \n"
"  csqr = c * c;                                                               \n"
"                                                                              \n"
"  /**                                                                         \n"
"   * Todo: We might want to change the buffers sizes, as the memory consumption\n"
"   * could be rather large.However, due to the lens, it might happen, that one pixel from the\n"
"   * images center gets stretched to the whole image, so we will still need   \n"
"   * at least a src_buf of size (width/2) * (height/2) * 4 to be able to      \n"
"   * process one quarter of the image.                                        \n"
"   */                                                                         \n"
"  src_buf = gegl_malloc (width * height * bytes_per_pixel);                   \n"
"  dst_buf = gegl_malloc (width * height * bytes_per_pixel);                   \n"
"                                                                              \n"
"  gegl_buffer_get (input, result, 1.0, input_format, src_buf,                 \n"
"                   GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);                     \n"
"                                                                              \n"
"  for (y = 0; y < height; y++)                                                \n"
"    {                                                                         \n"
"      /* dy is the current pixels distance (in y-direction) of the lenses center */\n"
"      dy = -((gfloat)y - b + 0.5);                                            \n"
"                                                                              \n"
"      /**                                                                     \n"
"       * To determine, whether the pixel is within the elliptical region affected\n"
"       * by the lens, we furthermore need the squared distance. So we calculate it\n"
"       * once for each row.                                                   \n"
"       */                                                                     \n"
"      dysqr = dy * dy;                                                        \n"
"      for (x = 0; x < width; x++)                                             \n"
"        {                                                                     \n"
"          /* Again we need to calculate the pixels distance from the lens dx. */\n"
"          dx = (gfloat)x - a + 0.5;                                           \n"
"                                                                              \n"
"          /* Given x and y we can now determine the pixels offset in our buffer. */\n"
"          pixel_offset = (x + y * width) * bytes_per_pixel;                   \n"
"                                                                              \n"
"          /* As described above, we only read and write image data once. */   \n"
"          dst_pixel = &dst_buf[pixel_offset];                                 \n"
"                                                                              \n"
"          if (dysqr < (bsqr - (bsqr * dx * dx) / asqr))                       \n"
"            {                                                                 \n"
"              /**                                                             \n"
"               * If (x, y) is inside the affected region, we can find its projected\n"
"               * position, calculate the projected_pixel_offset and set the src_pixel\n"
"               * to where we want to copy the pixel-data from.                \n"
"               */                                                             \n"
"              find_projected_pos (asqr, bsqr, csqr, dx, dy, refraction_index, \n"
"                                  &projected_x, &projected_y);                \n"
"                                                                              \n"
"              projected_pixel_offset = ( (gint)(-projected_y + b) * width +   \n"
"                                         (gint)( projected_x + a) ) * bytes_per_pixel;\n"
"                                                                              \n"
"              src_pixel = &src_buf[projected_pixel_offset];                   \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              /**                                                             \n"
"               * Otherwise (that is for pixels outside the lens), we could either leave\n"
"               * the image data unchanged, or set it to a specified 'background_color',\n"
"               * depending on the user input.                                 \n"
"               */                                                             \n"
"              if (keep_surroundings)                                          \n"
"                src_pixel = &src_buf[pixel_offset];                           \n"
"              else                                                            \n"
"                src_pixel = background_color;                                 \n"
"            }                                                                 \n"
"                                                                              \n"
"          /**                                                                 \n"
"           * At the end, we can copy the src_pixel (which was determined above), to\n"
"           * dst_pixel.                                                       \n"
"           */                                                                 \n"
"          memcpy (dst_pixel, src_pixel, bytes_per_pixel);                     \n"
"        }                                                                     \n"
"  }                                                                           \n"
"                                                                              \n"
"  gegl_buffer_set (output, result, 0, gegl_buffer_get_format (output), dst_buf,\n"
"                   GEGL_AUTO_ROWSTRIDE);                                      \n"
"                                                                              \n"
"  gegl_free (dst_buf);                                                        \n"
"  gegl_free (src_buf);                                                        \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"  gchar                    *composition =                                     \n"
"    \"<?xml version='1.0' encoding='UTF-8'?>\"                                \n"
"    \"<gegl>\"                                                                \n"
"    \"<node operation='gegl:apply-lens'>\"                                    \n"
"    \"  <params>\"                                                            \n"
"    \"    <param name='refraction_index'>1.7</param>\"                        \n"
"    \"    <param name='keep_surroundings'>false</param>\"                     \n"
"    \"    <param name='background_color'>rgba(0, 0.50196, 0.50196, 0.75)</param>\"\n"
"    \"  </params>\"                                                           \n"
"    \"</node>\"                                                               \n"
"    \"<node operation='gegl:load'>\"                                          \n"
"    \"  <params>\"                                                            \n"
"    \"    <param name='path'>standard-input.png</param>\"                     \n"
"    \"  </params>\"                                                           \n"
"    \"</node>\"                                                               \n"
"    \"</gegl>\";                                                              \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  operation_class->threaded                = FALSE;                           \n"
"  operation_class->prepare                 = prepare;                         \n"
"  operation_class->get_cached_region       = get_cached_region;               \n"
"  filter_class->process                    = process;                         \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",        \"gegl:apply-lens\",                                     \n"
"    \"title\",       _(\"Apply Lens\"),                                       \n"
"    \"categories\",  \"map\",                                                 \n"
"    \"license\",     \"GPL3+\",                                               \n"
"    \"description\", _(\"Simulates the optical distoration caused by having an elliptical lens over the image\"),\n"
"    \"reference-composition\", composition,                                   \n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
