/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bsesequencer.h"
#include	"bsesong.h"
#include	"bsemem.h"
#include	"bsemixer.h"

void
bse_sequencer_start (BseSong		  *song,
		     BseSequencerTickCB	   callback,
		     gpointer		   callback_data)
{
  BseSequencer *sequencer;
  
  g_return_if_fail (song != NULL);
  g_return_if_fail (song->sequencer == NULL);
  
  sequencer = bse_sequencer_chunk_new0 ();
  sequencer->loop_type = BSE_LOOP_NONE;
  sequencer->next_pattern_row = 0;
  sequencer->first_pattern_row = 0;
  sequencer->last_pattern_row = 0;
  sequencer->next_pattern = 0;
  sequencer->first_pattern = 0;
  sequencer->last_pattern = 0;
  
  sequencer->step_threshold = 0;
  sequencer->step_counter = 0;
  
  sequencer->tick_callback = callback;
  sequencer->tick_callback_data = callback_data;
  sequencer->tick_pattern = NULL;
  sequencer->tick_row = 0;
  
  song->sequencer = sequencer;
  
  bse_sequencer_recalc (song);
  sequencer->step_counter = sequencer->step_threshold;
}

void
bse_sequencer_recalc (BseSong *song)
{
  BseSequencer *sequencer;
  guint buf_p_m;
  
  g_return_if_fail (song != NULL);
  g_return_if_fail (song->sequencer != NULL);
  
  sequencer = song->sequencer;
  buf_p_m = bse_mixer_get_mix_freq () / bse_mixer_get_n_buffer_values () * 60;
  sequencer->step_threshold = buf_p_m / song->bpm;
  sequencer->step_counter %= sequencer->step_threshold;
}

void
bse_sequencer_set_loop (BseSong	       *song,
			guint		start_pattern_index,
			guint		end_pattern_index)
{
}

void
bse_sequencer_set_pattern_loop (BseSong	       *song,
				guint		start_pattern_row,
				guint		end_pattern_row)
{
}

void
bse_sequencer_step (BseSong	   *song,
		    guint	    n_voices,
		    BseVoice	   *voices)
{
  BseSequencer *sequencer;
  BsePattern *pattern;
  guint channel;
  BseNote *note;
  
  g_return_if_fail (song != NULL);
  g_return_if_fail (song->sequencer != NULL);
  g_return_if_fail (voices != NULL);
  
  sequencer = song->sequencer;
  
  sequencer->step_counter++;
  if (sequencer->step_counter < sequencer->step_threshold)
    return;
  
  sequencer->step_counter = 0;
  
  pattern = bse_song_get_pattern_from_list (song, sequencer->next_pattern);
  
  if (!pattern)
    {
      // bse_sequencer_stop (song); /* FIXME */
      // return;
      sequencer->next_pattern = 0;
      pattern = bse_song_get_pattern_from_list (song, sequencer->next_pattern);
    }
  
  /* FIXME: polyphony */
  g_return_if_fail (n_voices >= pattern->n_channels);
  
  for (channel = 0; channel < song->n_channels; channel++)
    {
      note = bse_pattern_get_note (pattern, channel, sequencer->next_pattern_row);
      
      bse_mixer_activate_voice (&voices[channel], note);
    }
  while (channel < n_voices)
    bse_voice_reset (&voices[channel++]);
  
  sequencer->tick_pattern = pattern;
  sequencer->tick_row = sequencer->next_pattern_row;
  
  sequencer->next_pattern_row++;
  if (sequencer->next_pattern_row >= song->pattern_length)
    {
      sequencer->next_pattern_row = 0;
      sequencer->next_pattern++;
    }
}

void
bse_sequencer_trigger_tick (BseSong	   *song)
{
  g_return_if_fail (song != NULL);
  g_return_if_fail (song->sequencer != NULL);
  
  if (song->sequencer->tick_pattern)
    {
      if (song->sequencer->tick_callback)
	song->sequencer->tick_callback (song->sequencer->tick_callback_data,
					song,
					song->sequencer->tick_pattern,
					song->sequencer->tick_row);
      song->sequencer->tick_pattern = NULL;
      song->sequencer->tick_row = 0;
    }
}

void
bse_sequencer_stop (BseSong	   *song)
{
  g_return_if_fail (song != NULL);
  g_return_if_fail (song->sequencer != NULL);
  
  if (song->sequencer->tick_callback)
    song->sequencer->tick_callback (song->sequencer->tick_callback_data,
				    song,
				    NULL,
				    0);
  
  bse_sequencer_chunk_free (song->sequencer);
  song->sequencer = NULL;
}
