/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bseioload.h"
#include	"bsesample.h"
#include	"bsesong.h"
#include	"bseparser.h"
#include        <unistd.h>
#include        <fcntl.h>
#include	<sys/stat.h>


BseIoData*
bse_io_load_auto (const gchar     *file_name,
		  BseIoDataFlags   loader_flags)
{
  g_return_val_if_fail (file_name != NULL, NULL);

  return bse_io_load_auto_perc (file_name, loader_flags, NULL, NULL);
}

BseIoData*
bse_io_load_auto_perc (const gchar     *file_name,
		       BseIoDataFlags   loader_flags,
		       BsePercentageCB  callback,
		       gpointer         callback_data)
{
  BseIoData *load_data;
  int fd;
  GScanner *scanner;
  BseIoDataFlags data_flags;
  
  g_return_val_if_fail (file_name != NULL, NULL);

  load_data = bse_io_data_new (loader_flags);
  
  if (!S_ISREG (g_scanner_stat_mode (file_name)))
    {
      load_data->error = BSE_ERROR_FILE_NOT_FOUND;
      return load_data;
    }
  
  fd = open (file_name, O_RDONLY);
  if (fd < 0)
    {
      load_data->error = BSE_ERROR_FILE_IO;
      return load_data;
    }
  
  /* identify BSE-Wave/BSE-Song
   */
  load_data->error = bse_parser_identify_fd (fd, &data_flags);

  if (load_data->error || (loader_flags & data_flags) == 0)
    {
      close (fd);
      return load_data;
    }

  /* setup parser
   */
  scanner = bse_parser_setup (file_name, fd, 1, 0, NULL, 0);

  /* parse text part
   */
  load_data->error = bse_parser_parse (scanner, loader_flags, NULL, NULL);

  if (!load_data->error &&
      loader_flags & BSE_IO_DATA_SAMPLES &&
      loader_flags & BSE_IO_DATA_BINARY_APPENDIX)
    load_data->error = bse_parser_read_binary_appendices (scanner);

  if (!load_data->error)
    {
      GSList *slist;
      
      if (loader_flags & BSE_IO_DATA_SAMPLES)
	{
	  bse_parser_complete_samples (scanner);
	  
	  for (slist = bse_parser_get_sample_list (scanner); slist; slist = slist->next)
	    {
	      BseSample *sample;
	      
	      sample = slist->data;
	      bse_sample_ref (sample);
	      load_data->samples = g_list_prepend (load_data->samples, sample);
	    }
	}

      if (loader_flags & BSE_IO_DATA_SONGS)
	{
	  bse_parser_complete_songs (scanner);

	  for (slist = bse_parser_get_song_list (scanner); slist; slist = slist->next)
	    {
	      BseSong *song;
	      
	      song = slist->data;
	      bse_song_ref (song);
	      load_data->songs = g_list_prepend (load_data->songs, song);
	    }
	}
    }
  
  bse_parser_shutdown (scanner);

  close (fd);

  return load_data;
}
