/*
   File: ReSize.c
   Authors: Mike Schmidt,
            K.R. Sloan  
   Last Modified: 27 September 1989
   Purpose: Pass a WFF format image file from stdin to stdout
            Make the new viewport LeftV, BottomV, RightV, TopV

            This is a specialization of ReSample - so the code may look
            a bit strange in parts.  Not to worry.

 */
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,
          "Usage:\n\t%s BottomV LeftV TopV RightV\n",
          RoutineName);
 }

static void
CheckBounds (Bottom,Left,Top,Right)
 int Bottom,Left,Top,Right;
 {
  if ( (Top < 0)      || (1023 < Top)   ||
       (Right < 0)    || (1023 < Right) ||
       (Top < Bottom) || (Right < Left)   )
   {
    fprintf (stderr, 
             "Bounds Bottom=%d Left=%d Top=%d Right=%d are illegal.\n",
             Bottom,Left,Top,Right);
    usage();
    exit (-1);
   }
 }

static int
IsInWindow (x,y,Bottom,Left,Top,Right)
 int x,y,Bottom,Left,Top,Right;
 {
  return ( (y >= Bottom) && (y <= Top) &&
           (x >= Left)   && (x <= Right)  );
 }
 
static void
Pass(fdIn,fdOut,BottomV,LeftV,TopV,RightV)
 FILE *fdIn, *fdOut;
 int BottomV,LeftV,TopV,RightV;
 {
  FrameBufferType *FBin, *FBout;
  int BottomI, LeftI, TopI, RightI;
  int BottomW, LeftW, TopW, RightW;
  char Name[NameLength], Value[ValueLength];
  int xv,yv,n;
  unsigned short Pixel[99];  /* that ought to be enough... */
  unsigned short FillPixel[99]; /* C's default value is zeros... */
  double sx,sy,xwd,ywd,xwdsave;
  int passed = 0;

  CheckBounds (BottomV,LeftV,TopV,RightV);
  
  FBin = (FrameBufferType *) 0;
  FBout= (FrameBufferType *) 0;

  if (FAILURE == OpenFB(&FBin))           {                        exit(-1); }
  if (FAILURE == ReadImage(fdIn, FBin))   { (void)CloseFB(&FBin);  exit(-1); }
  if (FAILURE == OpenFB(&FBout))          { (void)CloseFB(&FBin);  exit(-1); }
 

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    if ( (0 != strcmp(Name,"Name")) && (0 != strcmp(Name,"Title")) )
     SetDescriptor(FBout, Name, Value);
   }
  SetBounds (FBout,BottomV,LeftV,TopV,RightV);

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); exit(-1); }

  /* Finally, pass the pixels */
  if (FAILURE == GetBounds(FBin, &BottomI, &LeftI, &TopI, &RightI)) 
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  BottomW = BottomI; LeftW = LeftI; TopW = TopI; RightW = RightI;
  /* Calculate scaling factors. */
  sx = (double)(RightW-LeftW+1)/(double)(RightV-LeftV+1);
  sy = (double)(TopW-BottomW+1)/(double)(TopV-BottomV+1);
  ywd = (double)BottomW + 0.5 - sy;
  xwdsave = (double)LeftW + 0.5 -sx;

  for (yv=BottomV;yv<=TopV;yv++)
   {
    ywd += sy;
    xwd = xwdsave;
    for (xv=LeftV;xv<=RightV;xv++)
     {
      xwd += sx;
      if (IsInWindow ((int)xwd,(int)ywd,BottomI,LeftI,TopI,RightI))
       {
        GetPixel (FBin,(int)xwd,(int)ywd,Pixel);
        if (FAILURE == NextPixelOut(FBout,Pixel))
         {
          (void)CloseFB(&FBin);
          (void)CloseFB(&FBout);
          exit(-1);
         }
       }
      else
       if (FAILURE == NextPixelOut (FBout,FillPixel))
        {
         (void)CloseFB(&FBin);
         (void)CloseFB(&FBout);
         exit(-1);
        }
     }
    wffFlush(FBout);
   }

  (void)CloseFB(&FBin);
  (void)CloseFB(&FBout);

 }

main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  int BottomW, LeftW, TopW, RightW;
  int BottomV, LeftV, TopV, RightV;
 
  RoutineName = argv[ArgsParsed++];
  
  if (argv[ArgsParsed][0] == '-') { usage(); exit(-1);}
  if ((argc-ArgsParsed) >= 4)
   {
    BottomW = atoi(argv[ArgsParsed++]); BottomV = BottomW;
    LeftW   = atoi(argv[ArgsParsed++]); LeftV   = LeftW;
    TopW    = atoi(argv[ArgsParsed++]); TopV    = TopW;
    RightW  = atoi(argv[ArgsParsed++]); RightV  = RightW; 
   }
  else { usage(); exit(-1);}

  if ((argc-ArgsParsed) >= 4)
   {
    BottomV = atoi(argv[ArgsParsed++]);
    LeftV   = atoi(argv[ArgsParsed++]);
    TopV    = atoi(argv[ArgsParsed++]);
    RightV  = atoi(argv[ArgsParsed++]);
   }
  if (ArgsParsed < argc)  { usage(); exit(-1); }	

  Pass(stdin,stdout,
       BottomV, LeftV, TopV, RightV);
  exit(0);
}
