/* 
** graph_w.c: outer graph widget routines
**
** Written by:	Upinder S. Bhalla 	(bhalla@aurel.caltech.edu)
**		John Uhley 		(uhley@tybalt.caltech.edu)
**
** Copyright 1988 by the California Institute of Technology
**
** Permission to use, copy, modify, and distribute this
** software and its documentation for any purpose and without
** fee is hereby granted, provided that the above copyright
** notice appear in all copies and that both that copyright
** notice and this permission notice appear in supporting
** documentation, and that the name of the California Institute
** of Technology not be used in advertising or publicity pertaining 
** to distribution of the software without specific, written prior 
** permission.  The California Institute of Technology makes no 
** representations about the suitability of this software for any purpose.
** It is provided "as is" without express or implied warranty.
** 
** The Xgraph Widget library was written by Upinder S. Bhalla and
** John Uhley.  Portions of the labeling routines were taken from
** David Harrison's xgraph (X10V4) program with his permission.
**
** The X Window system is copyright 1985, 1986, 1987 By the
** Massachusetts Institute of Technology, without whom none of this
** code would have been possible.
**
*/
#include "widg_ext.h"
#ifdef X11R3
#include <X11/LabelP.h>
#else
#include <X11/Xaw/LabelP.h>
#endif X11R3
#include "LayoutP.h"
#include "innergraph_wP.h"
#include "graph_wP.h"

#define nlog10(x)	(x == 0.0 ? 0.0 : log10(x) + 1e-15)

static double RoundUp();

static XtResource resources[] = {
	{XtNxmin, XtCGraphXmin, XtRString, sizeof(char *),
		XtOffset(GraphWidget, graph.xmin_str), XtRString,
			"0.0"
	},
	{XtNymin, XtCGraphYmin, XtRString, sizeof(char *),
		XtOffset(GraphWidget, graph.ymin_str), XtRString,
			"0.0"
	},
	{XtNxmax, XtCGraphXmax, XtRString, sizeof(char *),
		XtOffset(GraphWidget, graph.xmax_str), XtRString,
			"100.0"
	},
	{XtNymax, XtCGraphYmax, XtRString, sizeof(char *),
		XtOffset(GraphWidget, graph.ymax_str), XtRString,
			"100.0"
	},
	{XtNtitle, XtCGraphTitle, XtRString, sizeof(char *),
		XtOffset(GraphWidget, graph.title), XtRString,
			"X11 Graph Widget"
	},
	{XtNXUnits, XtCXUnits, XtRString, sizeof(char *),
		XtOffset(GraphWidget, graph.XUnits), XtRString,
			"XUNITS"
	},
	{XtNYUnits, XtCYUnits, XtRString, sizeof(char *),
		XtOffset(GraphWidget, graph.YUnits), XtRString,
			"YUNITS"
	},
};


static Boolean graph_set_values();
static void Destroy();
static void graph_initialize();
static void graph_display();
void graph_rescale();
static void graph_axis();
static void graph_name_curves();
static int get_graph_maxname();
static void compute_inner_offset();
void GraphPrint();

static char translations[] = "Ctrl<Key>P:GraphPrint()";

static XtActionsRec actionsList[] = {
		{"GraphPrint",  GraphPrint}, 
};
GraphClassRec graphClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &compositeClassRec,
    /* class_name		*/	"Graph",
    /* widget_size		*/	sizeof(GraphRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	graph_initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	actionsList,
    /* num_actions		*/	XtNumber(actionsList),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	graph_rescale,
    /* expose			*/	graph_display,
    /* set_values		*/	graph_set_values,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  }, {
	NULL,
	NULL,
	XtInheritInsertChild,
	XtInheritDeleteChild,
	NULL,
	},
};

WidgetClass graphWidgetClass = (WidgetClass)&graphClassRec;

static void 
graph_initialize(request,new)
	GraphWidget request,new;
{
	InnerGraphWidget	i_w;
	int n;
	Position	x,y;
	Dimension	w,h;
	Arg	arg[20];

	compute_inner_offset(new, &x, &y, &w, &h);
	if (x < 0) x = 0; 
	if (y < 0) y = 0; 
	if (w < MIN_OUTER_GRAPH_WIDTH) w = MIN_OUTER_GRAPH_WIDTH;
	if (h < MIN_OUTER_GRAPH_HEIGHT) h = MIN_OUTER_GRAPH_HEIGHT;
	sscanf(new->graph.xmin_str, "%f", &(new->graph.xmin));
	sscanf(new->graph.xmax_str, "%f", &(new->graph.xmax));
	sscanf(new->graph.ymin_str, "%f", &(new->graph.ymin));
	sscanf(new->graph.ymax_str, "%f", &(new->graph.ymax));

	n = 0;
	XtSetArg(arg[n], XtNx, x); n++;
	XtSetArg(arg[n], XtNy, y); n++;
	XtSetArg(arg[n], XtNwidth, w); n++;
	XtSetArg(arg[n], XtNheight, h); n++;
	XtSetArg(arg[n], XtNxmin, new->graph.xmin_str); n++;
	XtSetArg(arg[n], XtNymin, new->graph.ymin_str); n++;
	XtSetArg(arg[n], XtNxmax, new->graph.xmax_str); n++;
	XtSetArg(arg[n], XtNymax, new->graph.ymax_str); n++;
	XtSetArg(arg[n], XtNforeground, graph_fg); n++;
	XtSetArg(arg[n], XtNbackground, graph_bg); n++;

	i_w = (InnerGraphWidget)
		XtCreateManagedWidget("internal_graph", innergraphWidgetClass,
		new, arg, n);
	new->graph.inner_widget = i_w;
	inner_graph_rescale(i_w);
}

static void Destroy(w)
		 Widget w;
{
	if (XtHasCallbacks(w, XtNcallback) == XtCallbackHasSome)
		XtRemoveAllCallbacks(w, XtNcallback);
}

/*
** graph_display:	Redraw the current graph.  This calls the
**			axis label routines and then calls the
**			inner_graph_display routine for the inner graph.
*/
static void graph_display(gw,event,region)
	GraphWidget gw;
	XEvent	*event;
	Region	region;
{
	Display	*display;
	Window	wind;
	GC	gc;
	XGCValues	values;
	InnerGraphWidget internal_graph;

	if (XtWindow(gw) == NULL)
		return;

	internal_graph = (InnerGraphWidget)XtNameToWidget(gw, "internal_graph");

	if (internal_graph == NULL) {
		fprintf(stderr,"graph_display: no internal graph\n");
		return;
	}
	/* if the widget is smaller than our defined mininum 
	** draw something saying this...
	*/
	display = XtDisplay(gw);
	wind = XtWindow(gw);
	if (wind == NULL)
		return;
	gc = XtGetGC(gw, GCForeground, &values);
	if (gw->graph.title != NULL) {
		if (XDisplayCells(display,XDefaultScreen(display)) > 254)
			XSetForeground(display,gc,graph_fg);
		else
			XSetForeground(display,gc,
				XBlackPixel(display,XDefaultScreen(display)));
		XPSDrawString(display, wind, gc, 
			(gw->core.width - (strlen(gw->graph.title) *
			title_width)) / 2 , 
			2 * PADDING, 
			gw->graph.title, strlen(gw->graph.title));
	}
	if (gw->core.width < MIN_OUTER_GRAPH_WIDTH || gw->core.height < 
		MIN_OUTER_GRAPH_HEIGHT) {
		XSetForeground(display,gc,graph_fg_error);
		XFillRectangle(display, wind, gc, 0, 0,
			gw->core.width, gw->core.height);
		XSetForeground(display,gc,graph_fg);
		return;
	}
	graph_axis(gw);
	graph_name_curves(gw);
	inner_graph_display(internal_graph, NULL, NULL);
}

/*
** graph_rescale:	Called when the widget is resized.  Reconverts
**			raw data to the new scaled data.  Redoes axis.
**
*/
void graph_rescale(gw) 
GraphWidget	gw;
{
	InnerGraphWidget internal_graph;
	Position	x,y;
	Dimension	w,h;
	Display	*display;
	Window	wind;
	GC	gc;
	int	i;
	XGCValues	values;
	XtCallbackRec	*callback;

	if (XtWindow(gw) == NULL)
		return;
	internal_graph = (InnerGraphWidget)XtNameToWidget(gw, "internal_graph");
	if (internal_graph == NULL) {
		fprintf(stderr,"graph_display: no internal graph\n");
		return;
	}
	compute_inner_offset(gw, &x, &y, &w, &h);
	XtConfigureWidget(internal_graph, x,y,w,h,1);
	
	if (((LayoutConstraints)
		(gw->core.constraints))->layout.layout_callback 
		!= NULL) {
			callback = (XtCallbackRec *)
				(((LayoutConstraints)
				(gw->core.constraints))->
				layout.layout_callback);
			i = 0;
			while (callback[i].callback != NULL) {
				(callback[i].callback)(gw,
				callback[i].closure, NULL);
				i++;
			}
	}

	if (gw->core.visible)  {
		XClearArea(XtDisplay(gw), XtWindow(gw), 0, 0, gw->core.width, 
			gw->core.height, False);
		XFlush(XtDisplay(gw));
	}
	display = XtDisplay(gw);
	wind = XtWindow(gw);
	if (wind == NULL)
		return;
	if (gw->core.width < MIN_OUTER_GRAPH_WIDTH || gw->core.height < 
		MIN_OUTER_GRAPH_HEIGHT) {
		gc = XtGetGC(gw, GCForeground, &values);
		XSetForeground(display,gc,graph_fg_error);
		XFillRectangle(display, wind, gc, 0, 0,
			gw->core.width, gw->core.height);
		XSetForeground(display,gc,graph_fg);
		fprintf(stderr,"You have made the graph too small.\n");
		return;
	}
	graph_name_curves(gw);
	graph_axis(gw);
	inner_graph_rescale(internal_graph);
}

/*
** compute_inner_offset:	Given an widget return x,y,w,h
**				rectagle for inner widget.
**
**	Based on code written by David Harrison, Xgraph for X10, UC Berkeley.
**
*/
static void compute_inner_offset(outer_gw, x, y, w, h)
	GraphWidget	outer_gw;
	Position	*x, *y;
	Dimension	*w, *h;
{
	InnerGraphWidget	gw;
	int		left_width;
	int		max_name;
	int		x_origin, y_origin;
	int		x_finish, y_finish;


	gw = (InnerGraphWidget)XtNameToWidget(outer_gw,"internal_graph");
	if (gw != NULL)
		max_name = get_graph_maxname(gw->graph.graphInfo);
	else
		max_name = 15;

	x_origin = (2 * PADDING) + (7 * axis_width); 
	y_origin = (3 * PADDING) + (axis_height / 2)  + axis_height + 
		title_height;

	/* 
	** Compute the space we'll need on the right hand and lower 
	** portions of the graph.
	**
	*/
	left_width = (strlen(outer_gw->graph.XUnits)+8) * axis_width;
	if ((max_name * axis_width) + PADDING > left_width)
		left_width = max_name * axis_width + PADDING;

	x_finish =  outer_gw->core.width - PADDING - left_width;
	y_finish =  outer_gw->core.height - (2 * PADDING) - axis_height;

	/*
	** The remaining space is where we'll put the graph.  It's
	** the size of our "bounding box" for the data.
	**
	*/

	*w =  x_finish - x_origin;
	*h = y_finish - y_origin;
	*x = outer_gw->graph.xoffset = x_origin;
	*y = outer_gw->graph.yoffset = y_origin;

}


/*
** graph_axis:	Compute and draw the axis
**
**	Based on code from xgraph for XV10.  David Harrison, UC Berkeley.
*/
static void graph_axis(outer_gw)
	GraphWidget	outer_gw;
{
	GC	gc;
	XGCValues	values;
	int	expX, expY;
	int	startX, y_spot, x_spot;
	double y_units_per_pixel;
	double x_units_per_pixel;
	double	x_inc, y_inc, x_start, y_start, y_index, x_index, larger;
	char	power[10], value[10];
	InnerGraphWidget	gw;
	Display	*display;
	Window	wind;

	display = XtDisplay(outer_gw);
	wind = XtWindow(outer_gw);
	if (wind == NULL)
		return;

	gw = (InnerGraphWidget)XtNameToWidget(outer_gw, "internal_graph");
	if (gw == NULL) {
		fprintf(stderr,"graph_axis: unable to find internal_graph\n");
		return;
	}

	if (gw->graph.xmin >= gw->graph.xmax) {
		printf("xmin > xmax, inverting\n");
		larger = gw->graph.xmin;
		gw->graph.xmin = gw->graph.xmax;
		gw->graph.xmax = larger;
	}

	if (gw->graph.ymin >= gw->graph.ymax) {
		printf("ymin > ymax, inverting\n");
		larger = gw->graph.ymin;
		gw->graph.ymin = gw->graph.ymax;
		gw->graph.ymax = larger;
	}

	if (fabs(gw->graph.xmin) > fabs(gw->graph.xmax))
		larger = fabs(gw->graph.xmin);
	else
		larger = fabs(gw->graph.xmax);
	
	expX = ((int)floor(nlog10(larger)/3.0)) *3;
	
	if (fabs(gw->graph.ymin) > fabs(gw->graph.ymax))
		larger = fabs(gw->graph.ymin);
	else
		larger = fabs(gw->graph.ymax);
	expY = ((int)floor(nlog10(larger)/3.0)) * 3;

	gc = XtGetGC(outer_gw, GCForeground, &values);
	XSetFont(display, gc, graph_axis_font->fid);

	if (XDisplayCells(display,XDefaultScreen(display)) > 254)
		XSetForeground(display,gc,graph_fg);
	else
		XSetForeground(display,gc,
			XBlackPixel(display,XDefaultScreen(display)));

	XSetSubwindowMode(display, gc, IncludeInferiors);
	
	XPSDrawString(display, wind, gc, PADDING, (2 * PADDING) +
		(axis_height / 2) + (title_height), outer_gw->graph.YUnits, 
			strlen(outer_gw->graph.YUnits));
	if (expY != 0) {
		XPSDrawString(display, wind, gc, PADDING + 
			(strlen(outer_gw->graph.YUnits) * axis_width), 
			(2 * PADDING) + (axis_height / 2) + (title_height), 
			" x 10", 5);
		sprintf(power, "%d", expY);
		XPSDrawString(display, wind, gc, PADDING + 
			( (strlen(outer_gw->graph.YUnits)+5) * axis_width), 
			(2 * PADDING) + (axis_height / 4) + (title_height), 
			power, strlen(power));
	}
		
	sprintf(power, "%d", expX);
	startX = outer_gw->core.width - (8 * PADDING) - 
			(strlen(outer_gw->graph.XUnits) * 
		axis_width) - (strlen(power) * axis_width);
	XPSDrawString(display, wind, gc, startX,
		outer_gw->core.height - (axis_height / 2), 
		outer_gw->graph.XUnits, strlen(outer_gw->graph.XUnits));
	if (expX != 0) {
		XPSDrawString(display, wind, gc, startX + 
			(strlen(outer_gw->graph.XUnits) * axis_width), 
			outer_gw->core.height - (axis_height/2), " x 10", 5);
		XPSDrawString(display, wind, gc, startX + 
			((strlen(outer_gw->graph.XUnits) +5 )* axis_width), 
			outer_gw->core.height - (axis_height), power, 
			strlen(power));
	}

	/* Now we need to draw the tick marks */
	y_units_per_pixel = 
		(gw->graph.ymax - gw->graph.ymin)/(gw->core.height);


	y_inc = (3 + axis_height) * y_units_per_pixel;
	y_inc = RoundUp(y_inc);

	y_start = y_inc + gw->graph.ymin;

	for (y_index = y_start; y_index <= gw->graph.ymax;
	 	y_index += y_inc) {
		y_spot = (gw->core.height +  outer_gw->graph.yoffset) -
			((int)(((y_index - gw->graph.ymin))/
			y_units_per_pixel - 0.5));
		WriteValue(value, y_index, expY);
		XPSDrawString(display, wind, gc, 
			PADDING, y_spot + (axis_height/2),
			value, strlen(value) );
		XPSDrawLine(display, wind, gc, 
			outer_gw->graph.xoffset - TICKSIZE/2, y_spot ,
			outer_gw->graph.xoffset + TICKSIZE/2, y_spot);
		XPSDrawLine(display, wind, gc, 
			outer_gw->graph.xoffset + gw->core.width + TICKSIZE/2,
			y_spot,
			outer_gw->graph.xoffset + gw->core.width - TICKSIZE/2,
		 	y_spot);
	}

	x_units_per_pixel = (gw->graph.xmax - gw->graph.xmin)/ (gw->core.width);
	x_inc = (3 + (axis_width * 7)) * x_units_per_pixel;
	x_inc = RoundUp(x_inc);
	x_start = x_inc + gw->graph.xmin;
	for (x_index = x_start; x_index < gw->graph.xmax; x_index += x_inc) {
		x_spot = (outer_gw->graph.xoffset) + 
			((int)((x_index  - gw->graph.xmin)/ x_units_per_pixel) + 0.5);
		WriteValue(value, x_index, expX);
		XPSDrawString(display, wind, gc, 
			x_spot - (axis_width * 7)/2, outer_gw->core.height,
			value, strlen(value) );
		XPSDrawLine(display, wind, gc, 
			x_spot, outer_gw->graph.yoffset - TICKSIZE/2,
			x_spot, outer_gw->graph.yoffset + TICKSIZE/2);
		XPSDrawLine(display, wind, gc, 
			x_spot, outer_gw->graph.yoffset - TICKSIZE/2 +
				 gw->core.height,
			x_spot, outer_gw->graph.yoffset + TICKSIZE/2 +
				gw->core.height);
	}
}

/*
** WriteValue:
**	
**	Get useful strings to print.  Lifted directly from Xgraph V10.
**	David Harrision, UC Berkeley.
**
*/
static int WriteValue(str, val, exp)
char *str;			/* String to write into */
double val;			/* Value to print       */
int exp;			/* Exponent             */
/*
 * Writes the value provided into the string in a fixed format
 * consisting of seven characters.  The format is:
 *   -ddd.dd
 */
{
    int index;

    if (exp < 0) {
	for (index = exp;  index < 0;  index++) {
	    val *= 10.0;
	}
    } else {
	for (index = 0;  index < exp;  index++) {
	    val /= 10.0;
	}
    }
    (void) sprintf(str, "%7.2f", val);
}

/*
** This routine rounds up the given positive number such that
** it is some power of ten times either 1, 2, or 5.  It is
** used to find increments for grid lines.
**/
static double RoundUp(val)
double val;			/* Value */
{
    int exponent, index;

    exponent = (int) floor(nlog10(val));
    if (exponent < 0) {
	for (index = exponent;  index < 0; index++) {
	    val *= 10.0;
	}
    } else {
	for (index = 0;  index < exponent; index++) {
	    val /= 10.0;
	}
    }
    if (val > 5.0) val = 10.0;
    else if (val > 2.0) val = 5.0;
    else if (val > 1.0) val = 2.0;
    else val = 1.0;
    if (exponent < 0) {
	for (index = exponent;  index < 0;  index++) {
	    val /= 10.0;
	}
    } else {
	for (index = 0;  index < exponent;  index++) {
	    val *= 10.0;
	}
    }
    return val;
}

/*
** graph_name_curves:
**
**	Print the names of the various curves being plotted.
**
*/
static void graph_name_curves(outer_gw)
	GraphWidget	outer_gw;
{
	InnerGraphWidget	gw;
	GraphInfo	*info;
	Position	x, y;
	GC	gc;
	XGCValues	values;
	Display		*display;
	int	max_name;

	gw= (InnerGraphWidget)XtNameToWidget(outer_gw, "internal_graph");
	if (gw == NULL) {
		fprintf(stderr,"outer_graph_name_curves: no inernal graph\n");
		return;
	}
	if (XtWindow(gw) == NULL)
		return;
	info = gw->graph.graphInfo;

	if (info == NULL)
		return;
	max_name = get_graph_maxname(info);
	x = outer_gw->core.width - ((5+max_name) * title_width);
	y = 10;
	gc = XtGetGC(outer_gw, GCForeground, &values);
	display = XtDisplay(outer_gw);
	while (info != NULL) {
		if (info->name != NULL) {
			XSetFont(display, gc, 
				graph_title_font->fid);

			if (XDisplayCells(display,XDefaultScreen(display)) > 254)
				XSetForeground(display,gc,name_to_color(info->color));
			else
				XSetForeground(display,gc,
					XBlackPixel(display,XDefaultScreen(display)));
			XPSDrawString(display, 
				XtWindow(outer_gw), gc, x, y, 
				info->name, strlen(info->name));
			y = y + title_height + 5;
			info = info->next;
		}
	}
	XSetForeground(display,gc,graph_fg);
}


/*
** get_graph_maxname:	Search through a linked list of graph curves
**			and return the char-length of the largest curve name.
**
*/
static int get_graph_maxname(info)
	GraphInfo	*info;
{
	int	ret = 0;
	while (info != NULL) {
		if (info->name != NULL && strlen(info->name) > ret)
			ret = strlen(info->name);
		info = info->next;
	}
	return(ret);
}

/*
** graph_set_values:	
**
**	Exists solely to convert "string" values for xmin,xmax,ymin,ymax
**	into the correct floating point values used internally by this
**	graph.  Also passes copies down into the "real" graph widget
**	which does the graphing.  Horrible Horrible Kludge.
**
*/
static Boolean graph_set_values(current, request, new)
	GraphWidget	current;
	GraphWidget	request;
	GraphWidget	new;
{
	InnerGraphWidget	inner_w;
	int	n;
	Arg	arg[5];

	if (strcmp(current->graph.xmin_str, request->graph.xmin_str) != 0 ||
	   strcmp(current->graph.xmax_str, request->graph.xmax_str) != 0 ||
	   strcmp(current->graph.ymin_str, request->graph.ymin_str) != 0 ||
	   strcmp(current->graph.ymax_str, request->graph.ymax_str) != 0 ) {
		sscanf(new->graph.xmin_str, "%f", &(new->graph.xmin));
		sscanf(new->graph.xmax_str, "%f", &(new->graph.xmax));
		sscanf(new->graph.ymin_str, "%f", &(new->graph.ymin));
		sscanf(new->graph.ymax_str, "%f", &(new->graph.ymax));
		inner_w = (InnerGraphWidget)XtNameToWidget(new, 
			"internal_graph");
		if (inner_w != NULL)  {
			n = 0;
			XtSetArg(arg[n], XtNxmin, new->graph.xmin_str); n++;
			XtSetArg(arg[n], XtNymin, new->graph.ymin_str); n++;
			XtSetArg(arg[n], XtNxmax, new->graph.xmax_str); n++;
			XtSetArg(arg[n], XtNymax, new->graph.ymax_str); n++;
			XtSetValues(inner_w, arg, n);
		}
		return(TRUE);
	}
	return(FALSE);
}
/*
** graph_add_pts:	
**
**	Add points to the graph - A PUBLIC routine.
**
*/
graph_add_pts(w,plotname,color,x,y)
	GraphWidget	w;
	char		*plotname;
	char		*color;
	float		x,y;
{
	if (w == NULL) {
		fprintf(stderr,"graph_add_pts:	called with null widget\n");
		return;
	}
	if (inner_graph_add_points(w->graph.inner_widget,plotname, color,x, y)) 
		graph_name_curves(w);
}
void GraphPrint(w, event)
	GraphWidget	w;
	XEvent	*event;
{
	InnerGraphWidget inner;
	inner = (InnerGraphWidget)w->graph.inner_widget;
	PreparePS(XtDisplay(w), XtWindow(w), 1.0);
	graph_rescale(w);
	XPSDrawLine(XtDisplay(w), XtWindow(w), 
		DefaultGC(XtDisplay(w), 0),
		inner->core.x, inner->core.y, inner->core.x + 
		inner->core.width, inner->core.y);
	XPSDrawLine(XtDisplay(w), XtWindow(w), 
		DefaultGC(XtDisplay(w), 0),
		inner->core.x + inner->core.width, inner->core.y,
		inner->core.x + inner->core.width,
		inner->core.y + inner->core.height);
	XPSDrawLine(XtDisplay(w), XtWindow(w), 
		DefaultGC(XtDisplay(w), 0),
		inner->core.x + inner->core.width,
		inner->core.y + inner->core.height,
		inner->core.x,
		inner->core.y + inner->core.height);
	XPSDrawLine(XtDisplay(w), XtWindow(w), 
		DefaultGC(XtDisplay(w), 0),
		inner->core.x,
		inner->core.y + inner->core.height,
		inner->core.x , inner->core.y );
	inner_graph_display(w->graph.inner_widget, NULL, NULL);
	FinishPS();
}
	
