/****************************************************************
**                                                             **
**                        SPIKE_PARMS                          **
**                                                             **
**          By Upinder S. Bhalla, May 1990. Caltech            **
**                                                             **
** This file contains routines for extracting a number of      **
** parameters of a spike that may be useful for comparing      **
** different spike traces. The objective is to get a set of    **
** such parameters which will enable one to evaluate the       **
** validity of a neuronal simulation and to constrain the      **
** parameters used in the simulation.                          **
**                                                             **
****************************************************************/
#include <stdio.h>
#include <math.h>

#define MAXPTS 5000
#define MAXSPIKES 50
#define RES MAXSPIKES-1
#define DEV MAXSPIKES-2
#define ENDT 10.0
#define RESTING_POTL -0.07
#define PEAKRANGE 0.005	/* 5 mV should be plenty */
#define DEFAULTMAX -0.02	/* -10 mV as a lower acceptable spike */
#define DEFAULTLO -0.05	/* -50 mV as an upper acceptable AHP */
#define TOL 0.0001
#ifndef EPSILON
#define EPSILON 1e-10
#endif

typedef struct parm_struct {
	float	t,v,dv,d2v;
	int		index;
	float	vp,dvp,d2vp;
} Parm;

float	findmax();
float	findmin();
float	findmaxslope();
float	findminslope();
float	finddiffmin();
float	x[MAXPTS],y[MAXPTS],d2y[MAXPTS];
float	v0 = 0.0,tau0 = 1.0;
float	v1 = 0.0,tau1 = 1.0;
float	globt,globv,dt;
float	startt;
int		npts = 0;

char *do_spike_parms(argc,argv)
	int argc;
	char	**argv;
{
	int		peak[MAXSPIKES],valley[MAXSPIKES];
	int		peakno = 0,valleyno=0;
	int		plotno = 0;
	float	t;
	float	temp;
	float	e0,e1;
	float	endt = ENDT;
	float	peakest = DEFAULTMAX,hyperpolest = DEFAULTLO;
	int		i,k;
	int		abs_exp_flag = 0;
	float	restV = RESTING_POTL;
	char	*outputfile,*filemode,*notes;
	char	*expfile,*diffile;

	/* Variables to be evaluated */
	Parm	Max[MAXSPIKES];
	Parm	Min[MAXSPIKES];
	Parm	Rise[MAXSPIKES];
	Parm	Fall[MAXSPIKES];
	Parm	Rec[MAXSPIKES];
	Parm	Thresh[MAXSPIKES];

	float	*allvars[50];
	float	*vardevs[50];
	char	*varnames[50];
	int		nvars;

	FILE	*fp,*fopen();

	if (argc < 2) {
		printf("usage : %s filename [-plot plotno] [-start time] [-end time]\n",argv[0]);
		printf("[-peak V] [-hyperpol V] [-restV V] [-file outputfile mode] [-notes notes]\n");
		printf("[-exp0 v0 tau0] [-exp1 v1 tau1] [-abs_exp] [-expfile file] [-diffile file]\n");
		printf("Valid filemodes : multifile onefile onefileappend\n");
		printf("The exp terms are added to the interspike curves before analysis.\n");
		printf("By default, v0 and v1 are scaled by ((v0+v1+RestV-MinV)/(v0+v1))\n");
		printf("In abs_exp mode the scaling is not done\n");
		printf("The expfile contains the exp decay curve.\n");
		printf("The diffile contains the difference plot between the spike and the exp decay.\n");
		return;
	}
	Max[RES].index = 0;
	Min[RES].index = 0;
	Rise[RES].index = 0;
	Fall[RES].index = 0;
	Rec[RES].index = 0;
	Thresh[RES].index = 0;
	v0 = 0.0,tau0 = 1.0;
	v1 = 0.0,tau1 = 1.0;
	startt = 0.0;

	outputfile = NULL;
	filemode = NULL;
	notes = NULL;
	expfile = NULL;
	diffile = NULL;
	for (i = 2 ; i < argc ; i++) {
		if (strcmp(argv[i],"-plot") == 0) {
			i++; plotno = atoi(argv[i]);
		}
		if (strcmp(argv[i],"-start") == 0) {
			i++; startt = atof(argv[i]);
		}
		if (strcmp(argv[i],"-end") == 0) {
			i++; endt = atof(argv[i]);
		}
		if (strcmp(argv[i],"-peak") == 0) {
			i++; peakest = atof(argv[i]);
		}
		if (strcmp(argv[i],"-hyperpol") == 0) {
			i++; hyperpolest = atof(argv[i]);
		}
		if (strcmp(argv[i],"-notes") == 0) {
			i++; notes = argv[i];
		}
		if (strcmp(argv[i],"-file") == 0) {
			if (argc < (i + 2)) {
				printf("syntax : -file outputfile mode \n");
				return;
			}
			i++; outputfile = argv[i];
			i++; filemode = argv[i];
		}
		if (strcmp(argv[i],"-exp0") == 0) {
			if (argc < (i + 2)) {
				printf("syntax : -exp0 v0 tau0\n");
				return;
			}
			i++; v0 = atof(argv[i]);
			i++; tau0 = atof(argv[i]);
		}
		if (strcmp(argv[i],"-exp1") == 0) {
			if (argc < (i + 2)) {
				printf("syntax : -exp1 v1 tau1\n");
				return;
			}
			i++; v1 = atof(argv[i]);
			i++; tau1 = atof(argv[i]);
		}
		if (strcmp(argv[i],"-abs_exp") == 0) {
			abs_exp_flag = 1;
		}
		if (strcmp(argv[i],"-restV") == 0) {
			i++; restV = atof(argv[i]);
		}
		if (strcmp(argv[i],"-expfile") == 0) {
			i++; expfile = argv[i];
		}
		if (strcmp(argv[i],"-diffile") == 0) {
			i++; diffile = argv[i];
		}
	}
	nvars = 0;
	allvars[nvars] = &Max[RES].t; vardevs[nvars] = &Max[DEV].t;
	varnames[nvars] = "MaxT"; nvars++;
	allvars[nvars] = &Min[RES].t; vardevs[nvars] = &Min[DEV].t;
	varnames[nvars] = "MinT"; nvars++;
	allvars[nvars] = &Rise[RES].t; vardevs[nvars] = &Rise[DEV].t; 
	varnames[nvars] = "RiseT"; nvars++;
	allvars[nvars] = &Fall[RES].t; vardevs[nvars] = &Fall[DEV].t; 
	varnames[nvars] = "FallT"; nvars++;

	allvars[nvars] = &Max[RES].v; vardevs[nvars] = &Max[DEV].v; 
	varnames[nvars] = "MaxV"; nvars++;
	allvars[nvars] = &Min[RES].v; vardevs[nvars] = &Min[DEV].v; 
	varnames[nvars] = "MinV"; nvars++;
	allvars[nvars] = &Rise[RES].v; vardevs[nvars] = &Rise[DEV].v; 
	varnames[nvars] = "RiseV"; nvars++;
	allvars[nvars] = &Fall[RES].v; vardevs[nvars] = &Fall[DEV].v; 
	varnames[nvars] = "FallV"; nvars++;

	allvars[nvars] = &Max[RES].dv; vardevs[nvars] = &Max[DEV].dv; 
	varnames[nvars] = "MaxdV"; nvars++;
	allvars[nvars] = &Min[RES].dv; vardevs[nvars] = &Min[DEV].dv; 
	varnames[nvars] = "MindV"; nvars++;
	allvars[nvars] = &Rise[RES].dv; vardevs[nvars] = &Rise[DEV].dv; 
	varnames[nvars] = "RisedV"; nvars++;
	allvars[nvars] = &Fall[RES].dv; vardevs[nvars] = &Fall[DEV].dv; 
	varnames[nvars] = "FalldV"; nvars++;

	allvars[nvars] = &Max[RES].d2v; vardevs[nvars] = &Max[DEV].d2v; 
	varnames[nvars] = "Maxd2V"; nvars++;
	allvars[nvars] = &Min[RES].d2v; vardevs[nvars] = &Min[DEV].d2v; 
	varnames[nvars] = "Mind2V"; nvars++;
	allvars[nvars] = &Rise[RES].d2v; vardevs[nvars] = &Rise[DEV].d2v;
	varnames[nvars]= "Rised2V";nvars++;
	allvars[nvars] = &Fall[RES].d2v; vardevs[nvars] = &Fall[DEV].d2v;
	varnames[nvars]= "Falld2V";nvars++;

	if (v0 > EPSILON || v0 < -EPSILON) {
		allvars[nvars] = &Rec[RES].t; vardevs[nvars] = &Rec[DEV].t; 
		varnames[nvars] = "RecT"; nvars++;
		allvars[nvars] = &Rec[RES].v; vardevs[nvars] = &Rec[DEV].v; 
		varnames[nvars] = "RecV"; nvars++;
		allvars[nvars] = &Rec[RES].dv; vardevs[nvars] = &Rec[DEV].dv; 
		varnames[nvars] = "RecdV"; nvars++;
		allvars[nvars] = &Rec[RES].d2v; vardevs[nvars] = &Rec[DEV].d2v; 
		varnames[nvars] = "Recd2V"; nvars++;
		allvars[nvars] = &Rec[RES].vp; vardevs[nvars] = &Rec[DEV].vp; 
		varnames[nvars] = "RecVp"; nvars++;
		allvars[nvars] = &Rec[RES].dvp; vardevs[nvars] = &Rec[DEV].dvp; 
		varnames[nvars] = "RecdVp"; nvars++;
		allvars[nvars]= &Rec[RES].d2vp;vardevs[nvars]= &Rec[DEV].d2vp; 
		varnames[nvars] = "Recd2Vp"; nvars++;

		allvars[nvars] = &Thresh[RES].t;
		vardevs[nvars] = &Thresh[DEV].t; 
		varnames[nvars] = "ThreshT"; nvars++;
		allvars[nvars] = &Thresh[RES].v; 
		vardevs[nvars] = &Thresh[DEV].v; 
		varnames[nvars] = "ThreshV"; nvars++;
		allvars[nvars] = &Thresh[RES].dv; 
		vardevs[nvars] = &Thresh[DEV].dv; 
		varnames[nvars] = "ThreshdV"; nvars++;
		allvars[nvars] = &Thresh[RES].d2v;
		vardevs[nvars] = &Thresh[DEV].d2v;
		varnames[nvars]="Threshd2V"; nvars++;
		allvars[nvars] = &Thresh[RES].vp; 
		vardevs[nvars] = &Thresh[DEV].vp; 
		varnames[nvars] = "ThreshVp"; nvars++;
		allvars[nvars] = &Thresh[RES].dvp; 
		vardevs[nvars] = &Thresh[DEV].dvp; 
		varnames[nvars]="ThreshdVp"; nvars++;
		allvars[nvars] = &Thresh[RES].d2vp;
		vardevs[nvars] = &Thresh[DEV].d2vp;
		varnames[nvars]="Threshd2Vp"; nvars++;
	}

	if (!read_plot2(argv[1],plotno,&dt,y,MAXPTS,&npts,startt,endt)) {
		printf("Read plot failed\n");
		return("failed");
	}
	for (i = 0, t = startt ; i < npts ; i++, t+=dt)
		x[i] = t;
	endt = t - dt;

/*
** Evaluate spline to get d2y/dx2 table
*/
	spline(x-1,y-1,npts,0.0,0.0,d2y);

/* Finding the peaks */
	for (i = 1 ; i < (npts - 1) ; i++) {
		if (y[i] > y[i-1] && y[i] > y[i+1] && 
			y[i] > (peakest-PEAKRANGE)){
			splinfit2(x,y,x[i]-0.002,x[i]+0.002,&(Max[peakno]),findmax);
			peakno++;
		}
/* Finding the valleys */
		if (y[i] < y[i-1] && y[i] < y[i+1] &&
			y[i] < (hyperpolest+PEAKRANGE)){
			splinfit2(x,y,x[i]-0.002,x[i]+0.002,&(Min[valleyno]),
				findmin);
			/*
			Min[valleyno].index = i;
			splinfit(x,y,i,&(Min[valleyno].t),&(Min[valleyno].v),&(Min[valleyno].dv),&(Min[valleyno].d2v),findmin);
			*/
			valleyno++;
		}
	}
/* Finding the first-order terms */
	for (k = 0 ; k < peakno ; k++) {
/* Finding Rise */
		splinfit2(x,y,Max[k].t-0.005,Max[k].t-dt,
			&(Rise[k]),findmaxslope);
/* Finding Fall */
		splinfit2(x,y,Max[k].t+dt,Max[k].t+0.002,
			&(Fall[k]),findminslope);
	}


	if (valleyno == peakno && v0 > EPSILON || v0 < (-EPSILON)) {
		if (abs_exp_flag == 0) {
			temp = (v0+v1+restV-Min[0].v)/(v0+v1);
			v0 *= temp;
			v1 *= temp;
		}

		if (expfile) {
			fp = fopen(expfile,"w");
			for (k = 0 ; k < (peakno - 1) ; k++) {
				for (i = Min[k].index ; i < Max[k+1].index; i++) {
					temp = v0*(1.0-exp((Min[k].t-x[i])/tau0)) +
						v1*(1.0-exp((Min[k].t-x[i])/tau1));
					fprintf(fp,"%g %g\n",x[i],temp);
				}
			}
			fclose(fp);
		}
		if (diffile) {
			fp = fopen(diffile,"w");
			for (k = 0 ; k < (peakno - 1) ; k++) {
				for (i = Min[k].index ; i < Max[k+1].index; i++) {
					temp = y[i] - (v0*(1.0-exp((Min[k].t-x[i])/tau0)) +
						v1*(1.0-exp((Min[k].t-x[i])/tau1)));
					fprintf(fp,"%g %g\n",x[i],temp);
				}
			}
			fclose(fp);
		}

/* Finding Second-order terms */
		for (k = 0 ; k < (peakno - 1) ; k++) {
/* Finding Rec */
			globt = Min[k].t;
			globv = Min[k].v;
			splinfit2(x,y,Min[k].t,Max[k+1].t,&(Rec[k]),
				finddiffmin);

			e0 = v0*exp((Min[k].t-Rec[k].t)/tau0);
			e1 = v1*exp((Min[k].t-Rec[k].t)/tau1);
			Rec[k].vp = v0-e0 + v1-e1 - Rec[k].v - globv;
			Rec[k].dvp = e0/tau0 + e1/tau1 - Rec[k].dv;
			Rec[k].d2vp = -e0/(tau0*tau0) - e1/(tau1*tau1) - Rec[k].dv;
/* Finding Thresh */
			splinroot(x,y,Rec[k].t,Max[k+1].t,&(Thresh[k]),
				finddiffmin);
			e0 = v0*exp((Min[k].t-Thresh[k].t)/tau0);
			e1 = v1*exp((Min[k].t-Thresh[k].t)/tau1);
			Thresh[k].vp = v0-e0 + v1-e1 - Thresh[k].v - globv;
			Thresh[k].dvp = e0/tau0 + e1/tau1 - Thresh[k].dv;
			Thresh[k].d2vp= -e0/(tau0*tau0)-e1/(tau1*tau1)-Thresh[k].dv;
		}
	}

	/* Putting all times relative to the spike peak */
	for(i = 0 ; i < peakno ; i++) {
		t = Max[i].t;
		Min[i].t -= t;
		Rise[i].t -= t;
		Fall[i].t -= t;
		Rec[i].t -= t;
		Thresh[i].t -= t;
	}

	if (valleyno == peakno) {
		parm_stats(Max,0,peakno,RES,DEV);
		parm_stats(Min,0,peakno,RES,DEV);
		parm_stats(Rise,0,peakno,RES,DEV);
		parm_stats(Fall,0,peakno,RES,DEV);
		if (peakno > 1) {
			parm_stats(Rec,0,peakno-1,RES,DEV);
			parm_stats(Thresh,0,peakno-1,RES,DEV);
			find_ISI(Max,peakno,RES,DEV);
		}
	}

	if (outputfile && strlen(outputfile) > 0) {
		spikeparmoutput(allvars,vardevs,varnames,nvars,outputfile,filemode,notes);
	}

	printf("%d peaks, %d valleys, over %d pts and from %f to %f sec\n",
		peakno,valleyno,npts,startt,endt);
	if (notes)
	printf("\n%s\n",notes);
	for (i = 0 ; i < nvars ; i++) {
		if (vardevs) {
			fprintf(stdout,"%-12s	%-12g	+/-	%-12g	(%1.4f %%)\n",
				varnames[i],*(allvars[i]), *(vardevs[i]),
				*(vardevs[i]) * 100.0/ *(allvars[i]));
		} else {
			fprintf(stdout,"%-12s	%g	",varnames[i],*(allvars[i]));
			if ((i%3) == 2)
				fprintf(stdout,"\n");
		}
	}	
	printf("\n");

	return("didit");
}

spikeparmoutput(allvars,vardevs,varnames,nvars,filename,filemode,notes)
	float **allvars;
	float **vardevs;
	char **varnames;
	int	nvars;
	char	*filename;
	char	*filemode;
	char	*notes;
{
	int i;
	static char	parmfile[200];
	FILE	*fp,*fopen();

	if (strcmp(filemode,"multifile") == 0) {
		for (i = 0 ; i < nvars ; i++) {
			sprintf(parmfile,"%s.%s",filename,varnames[i]);
			fp = fopen(parmfile,"a");
			if (vardevs) {
				if (notes)
					fprintf(fp,"%s	%g	%g\n",notes,*(allvars[i]),
						*(vardevs[i]));
				else
					fprintf(fp,"%g	%g\n",*(allvars[i]),*(vardevs[i]));
			} else {
				if (notes)
					fprintf(fp,"%s	%g\n",notes,*(allvars[i]));
				else
					fprintf(fp,"%g\n",*(allvars[i]));
			}
			fclose(fp);
		}
	}
	if (strcmp(filemode,"onefile") == 0) {
		fp = fopen(filename,"w");
		if (notes)
			fprintf(fp,"\n%s\n",notes);
		for (i = 0 ; i < nvars ; i++) {
			if (vardevs) {
				fprintf(fp,"%-12s	%-12g	+/-	%-12g	(%1.4f %%)\n",
					varnames[i],*(allvars[i]), *(vardevs[i]),
					*(vardevs[i]) * 100.0/ *(allvars[i]));
			} else {
				fprintf(fp,"%s	%g	",varnames[i],*(allvars[i]));
				if ((i%3) == 2)
					fprintf(fp,"\n");
			}
		}	
		fprintf(fp,"\n");
		fclose(fp);
	}
	if (strcmp(filemode,"onefileappend") == 0) {
		fp = fopen(filename,"a");
		if (notes)
			fprintf(fp,"\n%s\n",notes);
		for (i = 0 ; i < nvars ; i++) {
			fprintf(fp,"%s	%g	",varnames[i],*(allvars[i]));
			if ((i%3) == 2)
				fprintf(fp,"\n");
		}	
		fprintf(fp,"\n");
		fclose(fp);
	}
}

/*
** Upisplint is a version of splint which returns the first and
** second derivatives of the splined function as well. It is
** effectively doing linear interpolation in d2y, and the rest follows.
*/
void upisplint(xa,ya,y2a,n,x,y,dy,d2y)
float xa[],ya[],y2a[],x,*y,*dy,*d2y;
int n;
{
	int klo,khi,k;
	float h,b,a;
	float y2lo,y2hi;
	void nrerror();

	klo=1;
	khi=n;
	while (khi-klo > 1) {
		k=(khi+klo) >> 1;
		if (xa[k] > x) khi=k;
		else klo=k;
	}
	h=xa[khi]-xa[klo];
	if (h == 0.0) nrerror("Bad XA input to routine SPLINT");
	a=(xa[khi]-x)/h;
	b=(x-xa[klo])/h;
	y2lo=y2a[klo];
	y2hi=y2a[khi];
	*y=a*ya[klo]+b*ya[khi]+((a*a*a-a)*y2lo+(b*b*b-b)*y2hi)*(h*h)/6.0;
	*dy=(ya[khi]-ya[klo])/h-((3.0*a*a-1.0)*y2lo-(3.0*b*b-1.0)*y2hi)*h/6.0;
	*d2y=a*y2lo+b*y2hi;
}

float findmax(xv)
	float xv;
{
	float yv,dyv,d2yv;

	upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv);
	return(-yv);
}

float findmin(xv)
	float xv;
{
	float yv,dyv,d2yv;

	upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv);
	return(yv);
}

float findmaxslope(xv)
	float xv;
{
	float yv,dyv,d2yv;

	upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv);
	return(-dyv);
}

float findminslope(xv)
	float xv;
{
	float yv,dyv,d2yv;

	upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv);
	return(dyv);
}

float finddiffmin(xv)
	float xv;
{
	float yv,dyv,d2yv;

	upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv);
	yv -= globv + v0*(1.0-exp((globt-xv)/tau0)) +
		v1*(1.0-exp((globt-xv)/tau1));
			
	return(yv);
}


int splinfit(x,y,i,t,v,dv,d2v,func)
	float	x[],y[],*t,*v,*dv,*d2v,(*func)();
	int		i;
{
	float yv;
	float brent();

	yv = brent(x[i-5],x[i],x[i+5],func,TOL,t);
	upisplint(x-1,y-1,d2y-1,npts,*t,v,dv,d2v);
}

int splinfit2(x,y,lo,hi,ret,func)
	float	x[],y[],(*func)();
	Parm	*ret;
	float	lo,hi;
{
	float	t,v,dv,d2v;
	float	yv;
	float	brent();

	yv = brent(lo,(hi + lo)/2.0,hi,func,TOL,&t);
	upisplint(x-1,y-1,d2y-1,npts,t,&v,&dv,&d2v);
	ret->t = t;
	ret->v = v;
	ret->dv = dv;
	ret->d2v = d2v;
	ret->index = (int)(0.5+(t-startt)/dt);
}

int splinroot(x,y,lo,hi,ret,func)
	float	x[],y[],(*func)();
	Parm	*ret;
	float	lo,hi;
{
	float	t,v,dv,d2v;
	float	zbrent();

	t = zbrent(func,lo,hi,TOL);
	upisplint(x-1,y-1,d2y-1,npts,t,&v,&dv,&d2v);
	ret->t = t;
	ret->v = v;
	ret->dv = dv;
	ret->d2v = d2v;
	ret->index = (int)(0.5+(t-startt)/dt);
}

parm_stats(parm,lo,hi,result,dev)
	Parm	parm[];
	int		lo,hi,result,dev;
{
	int i,j;
	float	data[MAXSPIKES];
	float	svar,skew,curt,adev;

	for(i=lo,j=1;i<hi;i++,j++) data[j] = parm[i].t;
	moment(data,j-1,&(parm[result].t),&adev,&(parm[dev].t),&svar,&skew,&curt);

	for(i=lo,j=1;i<hi;i++,j++) data[j] = parm[i].v;
	moment(data,j-1,&(parm[result].v),&adev,&(parm[dev].v),&svar,&skew,&curt);

	for(i=lo,j=1;i<hi;i++,j++) data[j] = parm[i].dv;
	moment(data,j-1,&(parm[result].dv),&adev,&(parm[dev].dv),&svar,&skew,&curt);

	for(i=lo,j=1;i<hi;i++,j++) data[j] = parm[i].d2v;
	moment(data,j-1,&(parm[result].d2v),&adev,&(parm[dev].d2v),&svar,&skew,&curt);

	for(i=lo,j=1;i<hi;i++,j++) data[j] = parm[i].vp;
	moment(data,j-1,&(parm[result].vp),&adev,&(parm[dev].vp),&svar,&skew,&curt);

	for(i=lo,j=1;i<hi;i++,j++) data[j] = parm[i].dvp;
	moment(data,j-1,&(parm[result].dvp),&adev,&(parm[dev].dvp),&svar,&skew,&curt);

	for(i=lo,j=1;i<hi;i++,j++) data[j] = parm[i].d2vp;
	moment(data,j-1,&(parm[result].d2vp),&adev,&(parm[dev].d2vp),&svar,&skew,&curt);
}

find_ISI(Max,peakno,result,dev)
	Parm	Max[];
	int		peakno,result,dev;
{
	int i;
	float	data[MAXSPIKES];
	float	svar,skew,curt,adev;

	for(i=1;i<peakno;i++) data[i] = Max[i].t-Max[i-1].t;
	moment(data,i-1,&(Max[result].t),&adev,&(Max[dev].t),&svar,&skew,&curt);
}
