#include <stdio.h>
#include <math.h>
#include "sim_ext.h"
#include "tools.h"

#define EGG_AVERAGE 1
#define EGG_ROOT 2
#define EGG_SHELL 3
#define BOX_AVERAGE 4
#define BOX_ROOT 5
#define BOX_SHELL 6
#ifndef EPSILON
#define EPSILON 1.0e-60
#endif
#define	RAXIAL	1
#define	AXIAL	2

#define MAX_EL_SIZE	5000

#define dot_prd(x1,y1,z1,x2,y2,z2) (x1 * x2 + y1 * y2 + z1 * z2)

#define MAX_NELEMENTS 500

ElementList	*find_somas();

do_rancoord(argc,argv)
	int	argc;
	char	**argv;
{
	Element	*elm;

	if (argc < 3) {
		fprintf(stderr,"Usage : %s element fraction\n",argv[0]);
		return;
	}

	if (!(elm= GetElement(argv[1]))) {
		fprintf(stderr,"Element %s not found\n",argv[1]);
		return;
	}
	rancoord(elm,atof(argv[2]),atof(argv[2]),atof(argv[2]));
}

do_rotate(argc,argv)
	int argc;
	char	**argv;
{
	Element	*elm;
	ElementList	*el;
	float	ox = 0.0,oy = 0.0,oz = 0.0;
	float	ax = 0.0,ay = 0.0,az = 1.0;
	int		translatekids = 0;
	int		fixkids = 0;
	int		i;
	float	theta;

	if (argc < 3) {
		printf(
			"Usage : %s element angle [-x] [-y] [-z] [-a(xis) x y z]\n",
			argv[0]);
		printf("[-c(enter) x y z] [-t(ranslatekids)] [-f(ixkids)]\n");
		printf("The x,y,z and a(xis) options select the axis of rotation. x,y and z select \n");
		printf("a coordinate axis. The a(xis) option sets a vector for the rotation. \n");
		printf("Default is z axis\n");
		printf("c(enter) sets the center of the rotation. Default is the origin\n");
		printf("By default, the transform applies to all children of theselected element\n");
		printf("t(ranslatekids) causes the children to be translated parallel to the parent,\n");
		printf("rather than rotated. f(ixkids) prevents the children from being moved\n");
		return;
	}

	for( i = 3 ; i < argc ; i++) {
		str = argv[i];
		if (*str == '-') {
			str++;
			switch(*str) {
				case 'x' :
					ax = 1.0; ay = 0.0 ; az = 0.0;
					break;
				case 'y' :
					ax = 0.0; ay = 1.0 ; az = 0.0;
					break;
				case 'z' :
					ax = 0.0; ay = 0.0 ; az = 1.0;
					break;
				case 'a' :
					if (argc < (i + 3)) {
						printf("Usage : %s element angle [-x] [-y] [-z] [-a(xis) x y z]\n", argv[0]);
						return;
					}
					i++; ax = atof(argv[i]);
					i++; ay = atof(argv[i]);
					i++; az = atof(argv[i]);
					break;
				case 'c' :
					if (argc < (i + 3)) {
						printf("usage : [-c(enter) x y z]\n");
						return;
					}
					i++; cx = atof(argv[i]);
					i++; cy = atof(argv[i]);
					i++; cz = atof(argv[i]);
					break;
				case 't' :
					translatekids = 1;
					break;
				case 'f' :
					fixkids = 1;
					break;
				default :
					break;
			}
		}
	}
	if (!(elm = GetElement(argv[1]))) {
		printf("Element '%s' not found\n",argv[1]);
		return;
	}
	theta = atof(argv[2]);

	if (translatekids || fixkids) {
		el = (ElementList *) calloc(1,sizeof(ElementList));
		el->nelements = 1;
		el->element = (Element **) calloc(1,sizeof(Element *));
		el->element[0] = elm;
		dx = elm->x; dy = elm->y; dz = elm->z;
		RotateAboutAxis(elmlist,cx,cy,cz,ax,ay,az,theta);
		dx = elm->x - dx; dy = elm->y - dy; dz = elm->z - dz;
		if (translatekids) {
			rel_position(elm,dx,dy,dz);
			elm->x -= dx; elm->y -= dy; elm->z -= dz;
		}
	} else {
		el = find_all_dends(elm,1);
		RotateAboutAxis(el,cx,cy,cz,ax,ay,az,theta);
	}
	FreeEL(el);
}

/*
** Positions the element at x,y,z, then moves all kids by same amount
*/
abs_position(parent,x,y,z)
	Element	*parent;
	float x,y,z;
{
	Element	*kid;
	float dx,dy,dz;
	
	dx = x - parent->x;
	dy = y - parent->y;
	dz = z - parent->z;

	rel_position(parent,dx,dy,dz);
}

rel_position(parent,dx,dy,dz)
	Element	*parent;
	float dx,dy,dz;
{
	Element	*kid;
	
	parent->x += dx;
	parent->y += dy;
	parent->z += dz;
	for(kid = parent->child ; kid; kid = kid->next) {
		rel_position(kid,dx,dy,dz);
	}
}

rel_position_dends(parent,dx,dy,dz)
    Element *parent;
    float dx,dy,dz;
{
    Element *kid;
	MsgIn	*msg;

    parent->x += dx;
    parent->y += dy;
    parent->z += dz;
    
    rel_position(parent,dx,dy,dz);

    for (msg=parent->msg_in;msg;msg = msg->next) {
        if (msg->type == RAXIAL) {
            kid = msg->src;
            rel_position_dends(kid,dx,dy,dz);
        }
    }
}

ElementList *find_all_dends(parent,find_ch)
	Element	*parent;
	int find_ch;
{
	ElementList	*sel,*find_somas();
	ElementList	*el;
	int i;


	el = (ElementList *) calloc(1,sizeof(ElementList));
	el->nelements = 0;
	el->element = (Element **)calloc(MAX_EL_SIZE, sizeof(Element *));

	sel = find_somas(parent);

	for (i = 0 ; i < sel->nelements ; i++) {
		if (find_ch)
			traverse_dends_and_ch(sel->element[i],el);
		else
			traverse_dends(sel->element[i],el);
	}

	free(sel);
	return(el);
}

traverse_dends(parent,el)
	Element	*parent;
	ElementList	*el;
{
	Element *temp,*child;
	MsgIn	*msg;

	el->element[el->nelements] = parent;
	(el->nelements)++;

	for (msg=parent->msg_in;msg;msg = msg->next) {
		if (msg->type == RAXIAL) {
			child = msg->src;
			traverse_dends(child,el);
		}
	}
}

traverse_dends_and_ch(parent,el)
	Element	*parent;
	ElementList	*el;
{
	Element *temp,*child;
	MsgIn	*msg;

	traverse_ch(parent,el);

	for (msg=parent->msg_in;msg;msg = msg->next) {
		if (msg->type == RAXIAL) {
			child = msg->src;
			traverse_dends_and_ch(child,el);
		}
	}
}

traverse_ch(parent,el)
	Element	*parent;
	ElementList	*el;
{
	Element *temp,*child;

	el->element[el->nelements] = parent;
	(el->nelements)++;

	for (child = parent->child ; child ; child = child->next) {
		traverse_ch(child,el);
	}
}

ElementList	*find_somas(parent)
	Element	*parent;
{
	ElementList	*el;

	el = (ElementList *)calloc(1,sizeof(ElementList));
	el->element = (Element **)calloc(MAX_NELEMENTS,sizeof(Element *));

	if (strcmp(parent->object->name,"compartment") == 0) {
	/* This element itself is a compartment, so proceed */
		el->element[0] = parent;
		el->nelements = 1;
	} else {
	/* Look among children for somas : no AXIAL input messages */
		scan_somas(parent,el);
	}
	return(el);
}

FreeEL(el)
	ElementList	*el;
{
	if (el->nelements > 0)
		free(el->element);
	free(el);
}

scan_somas(parent,el)
	Element	*parent;
	ElementList	*el;
{
	Element *gramp,*child;
	MsgIn	*msg;

	if (strcmp(parent->object->name,"compartment") == 0) {
		gramp = NULL;
		for (msg=parent->msg_in;msg;msg = msg->next) {
			if (msg->type == AXIAL) {
				gramp = msg->src;
				break;
			}
		}
		if (!gramp) {
			el->element[el->nelements] = parent;
			el->nelements += 1;
		}
	} else {
		for (child = parent->child; child ; child = child->next) {
			scan_somas(child,el);
		}
	}
}

/*
** rancoord : If the given element is not a compartment, it 
** looks for children which are compartments without incoming AXIAL
** messages : i.e. somas.
** If the given element is a compartment, it starts the traversal there
** itself.
**
** Having found the root elements, it then traverses them by their 
** message trees, relatively positioning the children.
*/
rancoord(parent,fx,fy,fz)
	Element	*parent;
	float	fx,fy,fz;
{
	ElementList	*el,*find_somas();
	int	i;
	
	Element	*child;

	el = find_somas(parent);

	for(i = 0 ; i < el->nelements ; i++)
		ran_traverse(el->element[i],NULL,fx,fy,fz);
	FreeEL(el);
}

ran_traverse(parent,gramp,fx,fy,fz)
	Element	*parent,*gramp;
	float	fx,fy,fz;
{
	Element *temp,*child;
	float x,y,z,dx,dy,dz;
	MsgIn	*msg;

	if (gramp) {
		dx = parent->x - gramp->x;
		dy = parent->y - gramp->y;
		dz = parent->z - gramp->z;
		dx = frandom(dx,-dx) * fx;
		dy = frandom(dy,-dy) * fy;
		dz = frandom(dz,-dz) * fz;

		rel_position_dends(parent,dx,dy,dz);

	}
	for (msg=parent->msg_in;msg;msg = msg->next) {
		if (msg->type == RAXIAL) {
			child = msg->src;
			ran_traverse(child,parent,fx,fy,fz);
		}
	}
}

do_egg_constrain(argc,argv)
	int	argc;
	char	**argv;
{

	if (argc < 9) {
		fprintf(stderr,"usage : %s path start x y z len wid ht\n",
			argv[0]);
		return;
	}

	constrain(argv[1],argv[2],EGG_AVERAGE,
		atof(argv[3]), atof(argv[4]), atof(argv[5]),
		atof(argv[6]), atof(argv[7]), atof(argv[8]));
}


/*
** Path specifies the elements to be constrained
** Mode specifies how to constrain them : using an ellipsoid or
** a box
** Vectors 1 and 2 specify the bounds of the ellipsoid/box
*/
constrain(path,start,mode,x1,y1,z1,x2,y2,z2)
	char	*path;
	char	*start;
	int		mode;
	float	x1,y1,z1,x2,y2,z2;
{
	ElementList	*el; /* nelements, **element */
	Element	*elm;
	float	x = 0, y = 0, z = 0;

	el = WildcardGetElement(path,0);
	elm = GetElement(start);

	switch(mode)  {
		case EGG_AVERAGE :
			/* Take the average direction (by summing all vectors)
			** and turn so that this faces to the middle of the volume
			*/
			sum_vectors(elm,&x,&y,&z);
			rotate3d(el,elm->x,elm->y,elm->z,x,y,z,x1,y1,z1);
			break;
		case EGG_ROOT :
			/* Take the vector specified by the start and its parent,
			** and turn so that this faces to the middle of the volume
			*/
			break;
		case EGG_SHELL :
			/* Check all vectors and move into the vol.
			*/
			break;
		case BOX_AVERAGE :
			break;
		case BOX_ROOT :
			break;
		case BOX_SHELL :
			break;
		default :
			break;
	}
}

sum_vectors(parent,x,y,z)
	Element	*parent;
	float	*x,*y,*z;
{
	MsgIn	*msg;
	Element	*child;

	for (msg=parent->msg_in;msg;msg = msg->next) {
		if (msg->type == RAXIAL) {
			child = msg->src;
			*x + = child->x - parent->x;
			*y + = child->y - parent->y;
			*z + = child->z - parent->z;
			sum_vectors(child,x,y,z);
		}
	}
}
	

IsInEgg(elm,x1,y1,z1,x2,y2,z2)
	Element	*elm;
	float	x1,y1,z1,x2,y2,z2;
{
	float x,y,z,ex,ey,ez;
	float r;

	ex = fabs((x1-x2)/2.0);
	ey = fabs((y1-y2)/2.0);
	ez = fabs((z1-z2)/2.0);

	if (ex < EPSILON || ey < EPSILON || ez < EPSILON) {
		fprintf(stderr,"Ellipsoid is too narrow\n");
		return(0);
	}

	x = elm->x - (x1 + x2)/2.0;
	y = elm->y - (y1 + y2)/2.0;
	z = elm->z - (z1 + z2)/2.0;

	if ((x/ex * x/ex + y/ey * y/ey + z/ez * z/ez) <= 1.0)
	/* Point is within ellipsoid */
		return(1);
	return(0);
}

IsInBox(elm,x1,y1,z1,x2,y2,z2)
	Element	*elm;
	float	x1,y1,z1,x2,y2,z2;
{
	float temp;

	if (x1 > x2) {
		temp = x1 ; 
		x1 = x2 ;
		x2 = temp;
	}
	if (y1 > y2) {
		temp = y1 ; 
		y1 = y2 ;
		y2 = temp;
	}
	if (z1 > z2) {
		temp = z1 ; 
		z1 = z2 ;
		z2 = temp;
	}

	if (elm->x > x1 && elm->x < x2 &&
		elm->y > y1 && elm->y < y2 && 
		elm->z > z1 && elm->z < z2) {
	/* Point is within box */
		return(1);
	 }
	 return(0);
}

/*
** 
**	rotate3d(elmlist,ox,oy,oz,rx,ry,rz,px,py,pz)
** elmlist is the list of elements to be rotated
**	o is the center of rotaion
**	r is the original position
**	p is the destination direction
** The axis of rotation is normal to the plane defined by 
**	o,r,p. It is centered at o
** The rotation brings r into line with p and o
** All the elements in the elmlist are rotated according to the
** transform thus calculated.
*/
rotate3d(elmlist,ox,oy,oz,rx,ry,rz,px,py,pz)
	ElementList	*elmlist;
	float	ox,oy,oz;
	float	rx,ry,rz;
	float	px,py,pz;
{
	Element	*elm;
	int		i;

	/* coords of elements being transformed */
	float x,y,z;

	/* Vectors for the axes of the new frame of reference */
	float	ix,iy,iz;
	float	jx,jy,jz;
	float	kx,ky,kz;

	/* matrix for intermediate transform */
	float m11,m12,m13;
	float m21,m22,m23;
	float m31,m32,m33;

	/* matrix for final transform */
	float t11,t12,t13;
	float t21,t22,t23;
	float t31,t32,t33;

	/* Length of original vector rel to origin */
	float dr;

	/* length of destination vector rel to origin */
	float dp;

	/* Yet another length */
	float len;

	/* original vector rel to origin */
	float drx,dry,drz;

	/* dest vector rel to origin */
	float dpx,dpy,dpz;

	/* sin and cos variables */
	float s,c;


	/* finding the vectors rel to the origin */
	drx = rx - ox;
	dry = ry - oy;
	drz = rz - oz;
	dr = sqrt(drx * drx + dry * dry + drz * drz);

	dpx = px - ox;
	dpy = py - oy;
	dpz = pz - oz;
	dp = sqrt(dpx * dpx + dpy * dpy + dpz * dpz);

	/* ensuring that the source and dest vectors are not parallel */
	c = dot_prd(drx,dry,drz,dpx,dpy,dpz) / (dr * dp);
	if (c > 0.9999) /* ridiculously near parallel */
	/* return original coords */
		return;

	/* Setting up the axes in the new frame */
	ix = drx/dr; iy = dry/dr ; iz = drz/dr;

	cross_prd(&kx,&ky,&kz,drx,dry,drz,dpx,dpy,dpz);
	len = sqrt(kx * kx + ky * ky + kz * kz);
	kx = kx/len ; ky = ky/len ; kz = kz/len;

	cross_prd(&jx,&jy,&jz,kx,ky,kz,ix,iy,iz);

	/* Doing rotation about z axis in new frame */
	/* First, finding sin and cos : */
	c = dot_prd(drx,dry,drz,dpx,dpy,dpz) / (dr * dp);
	s = len / (dr * dp);


	/* then, pre-multiplying the axes transform by the rot transf */
	m11=c * ix - s * jx; m12=c * iy - s * jy; m13=c * iz - s * jz;
	m21=s * ix + c * jx; m22=s * iy + c * jy; m23=s * iz + c * jz;
	m31=kx; m32 = ky; m33 = kz;

	/* Finally, pre-multiplying by the inverse transform to the
	** original frame. This is just the transpose of the first
	** transform matrix, since the columns were orthogonal */
	t11 = dot_prd(ix,jx,kx,m11,m21,m31);
	t12 = dot_prd(ix,jx,kx,m12,m22,m32);
	t13 = dot_prd(ix,jx,kx,m13,m23,m33);
	t21 = dot_prd(iy,jy,ky,m11,m21,m31);
	t22 = dot_prd(iy,jy,ky,m12,m22,m32);
	t23 = dot_prd(iy,jy,ky,m13,m23,m33);
	t31 = dot_prd(iz,jz,kz,m11,m21,m31);
	t32 = dot_prd(iz,jz,kz,m12,m22,m32);
	t33 = dot_prd(iz,jz,kz,m13,m23,m33);

	/* The transformation itself : easy, innit ? */
	for (i = 0 ; i < elmlist->nelements ; i++) {
		elm = elmlist->element[i];
		x = elm->x - ox ; y = elm->y - oy ; z = elm->z - oz;
		elm->x = dot_prd(t11,t12,t13,x,y,z) + ox;
		elm->y = dot_prd(t21,t22,t23,x,y,z) + oy;
		elm->z = dot_prd(t31,t32,t33,x,y,z) + oz;
	}
}

cross_prd(x1,y1,z1,x2,y2,z2,x3,y3,z3)
	float *x1,*y1,*z1,x2,y2,z2,x3,y3,z3;
{
	*x1 = y2 * z3 - y3 * z2;
	*y1 = z2 * x3 - z3 * x2;
	*z1 = x2 * y3 - x3 * y2;
}

/*
** 
**	RotateInPlane(elmlist,ox,oy,oz,rx,ry,rz,px,py,pz)
** elmlist is the list of elements to be rotated
**	o is the center of rotaion
**	r is the original vector : angle is calculated from this.
**	p is the third vector defining the plane of rotation.
**	theta is the angle of rotation relative to r
** The axis of rotation is normal to the plane defined by 
**	o,r,p. It is centered at o
** All the elements in the elmlist are rotated according to the
** transform thus calculated.
*/
RotateInPlane(elmlist,ox,oy,oz,rx,ry,rz,px,py,pz,theta)
	ElementList	*elmlist;
	float	ox,oy,oz;
	float	rx,ry,rz;
	float	px,py,pz;
	float	theta;
{
	Element	*elm;
	int		i;

	/* coords of elements being transformed */
	float x,y,z;

	/* Vectors for the axes of the new frame of reference */
	float	ix,iy,iz;
	float	jx,jy,jz;
	float	kx,ky,kz;

	/* matrix for intermediate transform */
	float m11,m12,m13;
	float m21,m22,m23;
	float m31,m32,m33;

	/* matrix for final transform */
	float t11,t12,t13;
	float t21,t22,t23;
	float t31,t32,t33;

	/* Length of original vector rel to origin */
	float dr;

	/* length of destination vector rel to origin */
	float dp;

	/* Yet another length */
	float len;

	/* original vector rel to origin */
	float drx,dry,drz;

	/* dest vector rel to origin */
	float dpx,dpy,dpz;

	/* sin and cos variables */
	float s,c;


	/* finding the vectors rel to the origin */
	drx = rx - ox;
	dry = ry - oy;
	drz = rz - oz;
	dr = sqrt(drx * drx + dry * dry + drz * drz);

	dpx = px - ox;
	dpy = py - oy;
	dpz = pz - oz;
	dp = sqrt(dpx * dpx + dpy * dpy + dpz * dpz);

	/* ensuring that the original and plane vectors are not parallel */
	c = dot_prd(drx,dry,drz,dpx,dpy,dpz) / (dr * dp);
	if (c > 0.9999) /* ridiculously near parallel */
	/* return original coords */
		return;

	/* Setting up the axes in the new frame */
	ix = drx/dr; iy = dry/dr ; iz = drz/dr;

	cross_prd(&kx,&ky,&kz,drx,dry,drz,dpx,dpy,dpz);
	len = sqrt(kx * kx + ky * ky + kz * kz);
	kx = kx/len ; ky = ky/len ; kz = kz/len;

	cross_prd(&jx,&jy,&jz,kx,ky,kz,ix,iy,iz);

	/* Doing rotation about z axis in new frame */
	/* First, finding sin and cos : */
	c = cos(theta);
	s = sin(theta);

	/* then, pre-multiplying the axes transform by the rot transf */
	m11=c * ix - s * jx; m12=c * iy - s * jy; m13=c * iz - s * jz;
	m21=s * ix + c * jx; m22=s * iy + c * jy; m23=s * iz + c * jz;
	m31=kx; m32 = ky; m33 = kz;

	/* Finally, pre-multiplying by the inverse transform to the
	** original frame. This is just the transpose of the first
	** transform matrix, since the columns were orthogonal */
	t11 = dot_prd(ix,jx,kx,m11,m21,m31);
	t12 = dot_prd(ix,jx,kx,m12,m22,m32);
	t13 = dot_prd(ix,jx,kx,m13,m23,m33);
	t21 = dot_prd(iy,jy,ky,m11,m21,m31);
	t22 = dot_prd(iy,jy,ky,m12,m22,m32);
	t23 = dot_prd(iy,jy,ky,m13,m23,m33);
	t31 = dot_prd(iz,jz,kz,m11,m21,m31);
	t32 = dot_prd(iz,jz,kz,m12,m22,m32);
	t33 = dot_prd(iz,jz,kz,m13,m23,m33);

	/* The transformation itself : easy, innit ? */
	for (i = 0 ; i < elmlist->nelements ; i++) {
		elm = elmlist->element[i];
		x = elm->x - ox ; y = elm->y - oy ; z = elm->z - oz;
		elm->x = dot_prd(t11,t12,t13,x,y,z) + ox;
		elm->y = dot_prd(t21,t22,t23,x,y,z) + oy;
		elm->z = dot_prd(t31,t32,t33,x,y,z) + oz;
	}
}

/*
** 
**	RotateAboutAxis(elmlist,ox,oy,oz,ax,ay,az)
** elmlist is the list of elements to be rotated
**	o is the center of rotaion
**	k is the axis vector
**	theta is the angle of rotation.
** All the elements in the elmlist are rotated according to the
** transform thus calculated.
*/
RotateAboutAxis(elmlist,ox,oy,oz,kx,ky,kz,theta)
	ElementList	*elmlist;
	float	ox,oy,oz;
	float	kx,ky,kz;
	float	theta;
{
	Element	*elm;
	int		i;

	/* coords of elements being transformed */
	float x,y,z;

	/* Vectors for the axes of the new frame of reference */
	float	ix,iy,iz;
	float	jx,jy,jz;

	/* matrix for intermediate transform */
	float m11,m12,m13;
	float m21,m22,m23;
	float m31,m32,m33;

	/* matrix for final transform */
	float t11,t12,t13;
	float t21,t22,t23;
	float t31,t32,t33;

	/* Length of original vector rel to origin */
	float dr;

	/* length of destination vector rel to origin */
	float dp;

	/* Yet another length */
	float len;

	/* original vector rel to origin */
	float drx,dry,drz;

	/* dest vector rel to origin */
	float dpx,dpy,dpz;

	/* sin and cos variables */
	float s,c;

	/* Normalising the axis vector */
	len = sqrt(kx * kx + ky * ky + kz * kz);
	kx = kx/len ; ky = ky/len ; kz = kz/len;

	/* finding vectors for the other axes.
	** Trick : to get a vector nonparallel to k, we just rotate 
	** its components. No particular directions are needed.
	*/
	cross_prd(&jx,&jy,&jz,kx,ky,kz,ky,kz,kx);
	cross_prd(&ix,&iy,&iz,jx,jy,jz,kx,ky,kz);

	/* Doing rotation about z axis in new frame */
	/* First, finding sin and cos : */
	c = cos(theta);
	s = sin(theta);

	/* then, pre-multiplying the axes transform by the rot transf */
	m11=c * ix - s * jx; m12=c * iy - s * jy; m13=c * iz - s * jz;
	m21=s * ix + c * jx; m22=s * iy + c * jy; m23=s * iz + c * jz;
	m31=kx; m32 = ky; m33 = kz;

	/* Finally, pre-multiplying by the inverse transform to the
	** original frame. This is just the transpose of the first
	** transform matrix, since the columns were orthogonal */
	t11 = dot_prd(ix,jx,kx,m11,m21,m31);
	t12 = dot_prd(ix,jx,kx,m12,m22,m32);
	t13 = dot_prd(ix,jx,kx,m13,m23,m33);
	t21 = dot_prd(iy,jy,ky,m11,m21,m31);
	t22 = dot_prd(iy,jy,ky,m12,m22,m32);
	t23 = dot_prd(iy,jy,ky,m13,m23,m33);
	t31 = dot_prd(iz,jz,kz,m11,m21,m31);
	t32 = dot_prd(iz,jz,kz,m12,m22,m32);
	t33 = dot_prd(iz,jz,kz,m13,m23,m33);

	/* The transformation itself : easy, innit ? */
	for (i = 0 ; i < elmlist->nelements ; i++) {
		elm = elmlist->element[i];
		x = elm->x - ox ; y = elm->y - oy ; z = elm->z - oz;
		elm->x = dot_prd(t11,t12,t13,x,y,z) + ox;
		elm->y = dot_prd(t21,t22,t23,x,y,z) + oy;
		elm->z = dot_prd(t31,t32,t33,x,y,z) + oz;
	}
}
