#include <stdio.h>
#include <math.h>
#include "sim_ext.h"
#include "seg_struct.h"
#ifndef	RAXIAL
#define	RAXIAL 1
#endif	RAXIAL
/*
**		rall_calc : By Upinder S. Bhalla May 1990
** 
** This file contains various routines for calculating cell parameters.
** The input resistance can be calculated directly from the parameters
** of the compartments. The value of Rm can be calculated form the 
** input resistance (experimental) of the cell assuming
** that all of the parameters in the compartments have been calculated
** using the same Rmest. 
** The values for input resistance and Rm are a few (generally < 5%)
** percent off the values you would get by dividing the Voltage change
** (found by running the simulation)
** by the injected current for the passive membrane. This is because
** this calculation uses Rall's analytic solution for the dendritic
** tree, whereas the simulation uses discrete compartments. The error
** increases if you use large compartments, and is around 3 % for 
** 10 compartments per length constant. The error may be reduced by
** terminating the dendritic branches with a 2x higher Ra, when using
** symmetric compartments.
**
** These calculations are based on the procedure outlined
** in 'Electric current flow in excitable cells' by Jack, Noble and
** Tsien, OUP, 1983, and Rall, Biophys J, Vol 9 1969 pp1483
**
** Formulae used :

---------------------------------------------------------------------

0 - Calculating Bj

Bj = SIGMAk[Bjk*( {(rm*ri)|(j-1)}/{(rm*ri)|(jk)} )^0.5]
where 

Bjk = ( B(j+1)k + tanh({(ri/rm)|jk}^0.5) ) /
		( 1 +  B(j+1)k * tanh({(ri/rm)|jk}^0.5) )

{(ri/rm)|jk} is ri/rm evaluated for compartment jk
B(j+1)k is B indexed at (j+1)k

---------------------------------------------------------------------

1 - input cond of cell

Gn = 1/rmsoma + SIGMAj(Boj * (1/(rmoj * rioj))^0.5)

Where
	Gn = input conductivity of cell : to be calculated.
	rmsoma = membrane res of cell
	SIGMA is the summation symbol, summation is over j th dend branch
	Boj is the B for the jth dend branch off the soma.
	rmoj is the memb res for the jth branch compt off the soma
	rioj is axial res for the jth branch compt off the soma

---------------------------------------------------------------------

2 - Value of Rm from input cond of cell.

Rm = Rmest/(Beta^2)

where 

Beta = {-A + ( A^2 + 4 * Gn/rmsoma)^0.5 } / {2 / rmsoma}
A = SIGMAj(Boj * (1/(rmoj * rioj))^0.5)
Gn = input cond of cell, measured
Rmest = Estimated value for Rm
rmsoma = membrane res of soma based on Rmest
all values of   rm = membrane res of compt, based on Rmest.

---------------------------------------------------------------------
*/

float calcBj();
float calcA();
float calcGn();
float gcompt();

float calcBj(elm,lastrmri)
	struct compartment_type *elm;
	float lastrmri;
{
	struct compartment_type *child;
	float Bj = 0.0;
	float Bjk,nextBj;
	float ri,rm,tr,rmri;
	MsgIn	*msg;

	ri = elm->Ra;
	rm = elm->Rm;

	rmri = ri * rm;

	for (msg = elm->msg_in ; msg ; msg = msg->next) {
		if (msg->type == RAXIAL) {
			child = (struct compartment_type *)(msg->src);
			ri = child->Ra;
			rm = child->Rm;
			tr = tanh(sqrt(ri/rm));
			nextBj = calcBj(child,rm * ri);
			Bjk = (nextBj + tr) / (1 + nextBj * tr);
			Bj = Bj + Bjk * sqrt(((lastrmri) / (rm * ri)));
		}
	}

	return(Bj);
}

float calcA(soma)
	struct compartment_type *soma;
{
	float A = 0.0;
	float Boj;
	float somarmri,somarm;
	float rm,ri,tr;
	struct compartment_type *child;
	MsgIn	*msg;

	somarm = soma->Rm;
	somarmri = somarm * soma->Ra;

	for (msg = soma->msg_in ; msg ; msg = msg->next) {
		if (msg->type == RAXIAL) {
			child = (struct compartment_type *)(msg->src);
			ri = child->Ra;
			rm = child->Rm;
			tr = tanh(sqrt(ri/rm));
			Boj = calcBj(child,rm * ri);
			Boj = (Boj + tr) / (1.0 + Boj * tr);
			A += Boj * (sqrt(1.0/(rm * ri)));
		}
	}
	return(A);
}

float calcGn(soma)
	struct compartment_type *soma;
{
	float Gn;

	Gn = calcA(soma) + 1.0 / soma->Rm;

	return(Gn);
}

/*
** calcRm assumes that the cell was set up with RM as memb resistivity
** I do not really understand what I was doing here.
*/
float calcRm(soma,Gn,RM)
	struct compartment_type *soma;
	float Gn;
	float RM;
{
	float Rm;
	float Beta,A;
	float rmsoma;

	rmsoma = soma->Rm;

	A = calcA(soma);
	Beta = (-A + sqrt( A * A + 4.0 * Gn/rmsoma)) / (2.0 / rmsoma);
	
	Rm = RM/(Beta * Beta);
	return(Rm);
}

char *do_sumCm(argc,argv)
	int		argc;
	char	**argv;
{
	Element *elm;
	float cap = 0.0;
	float res = 0.0;
	ElementList	*elmlist, *WildcardGetElement();
	char	path[80];
	char	*parent;
	int		i;

	if (argv[1])
		parent = argv[1];
	else
		parent = "";

	if (strlen(parent) == 0)
		sprintf(path,"./##[TYPE=compartment]");
	else
		sprintf(path,"%s/##[TYPE=compartment]",parent);
	if (!(elmlist = WildcardGetElement(path,0))) {
		printf("no elements found on '%s'\n",path);
		return("0.0");
	}
	for (i = 0 ; i < elmlist->nelements ; i++)
		cap += ((struct compartment_type *)(elmlist->element[i]))->Cm;
	return(ftoa(cap));
}

char *do_rall_calc_Rm(argc,argv)
	int		argc;
	char	**argv;
{
	float Gn;
	Element	*soma;

	if (argc < 2) {
		printf("usage : %s soma\n",argv[0]);
		return("0.0");
	}
	if (!(soma = GetElement(argv[1]))) {
		printf("element '%s' not found\n",argv[1]);
		return("0.0");
	}
	Gn = calcGn(soma);
	return(ftoa(1.0/Gn));
}


float gcompt(elm)
	struct compartment_type *elm;
{
	Element *child;
	float gc;
	MsgIn	*msg;

	gc = 1/(elm->Rm);

	for (msg = elm->msg_in ; msg ; msg = msg->next) {
		if (msg->type == RAXIAL) {
			child = msg->src;
			gc += gcompt(child);
		}
	}
	return(1.0/(elm->Ra+1.0/gc));
}

float gsoma(soma)
	struct compartment_type *soma;
{
	Element *child;
	float gc;
	MsgIn	*msg;

	gc = 1/(soma->Rm);

	for (msg = soma->msg_in ; msg ; msg = msg->next) {
		if (msg->type == RAXIAL) {
			child = msg->src;
			gc += gcompt(child);
		}
	}
	return(gc);
}


char *do_calc_Rm(argc,argv)
	int		argc;
	char	**argv;
{
	float Gn;
	Element	*soma;

	if (argc < 2) {
		printf("usage : %s soma\n",argv[0]);
		return("0.0");
	}
	if (!(soma = GetElement(argv[1]))) {
		printf("element '%s' not found\n",argv[1]);
		return("0.0");
	}
	Gn = gsoma(soma);
	return(ftoa(1.0/Gn));
}



#undef	RAXIAL
