#include "sim_ext.h"

#define Trapezoid(y,A,B,dt) \
	(((y)*(1-(dt)*(B)/2.0)+(dt)*(A))/(1+(dt)*(B)/2.0))

#define INTEG_AB2() \
	value = y + ( \
	1.5 *dy \
	- 0.5 *PreviousEvent(buffer,0)->magnitude \
	) * dt

#define INTEG_AB3() \
	value = y + ( \
	23.0 *dy  \
	- 16.0 *PreviousEvent(buffer,0)->magnitude \
	+ 5.0 *PreviousEvent(buffer,1)->magnitude \
	) * dt/12.0

#define INTEG_AB4() \
	value = y + ( \
	55.0 *dy  \
	- 59.0 *PreviousEvent(buffer,0)->magnitude \
	+ 37.0 *PreviousEvent(buffer,1)->magnitude \
	- 9.0 *PreviousEvent(buffer,2)->magnitude \
	) * dt/24.0

#define INTEG_AB5() \
	value = y + ( \
	1901.0 *dy  \
	- 2774.0 *PreviousEvent(buffer,0)->magnitude \
	+ 2616.0 *PreviousEvent(buffer,1)->magnitude \
	- 1274.0 *PreviousEvent(buffer,2)->magnitude \
	+ 251.0 *PreviousEvent(buffer,3)->magnitude \
	) * dt/720.0

double DirectIntegrate(method,segment,state,dy,dt,name)
int	method;
Segment	*segment;
double	state;
double	dy;
double	dt;
char *name;
{
double D;
double result;

    switch(method){
	case 0:
	case 1:
	case 4:
	    Error();
	    printf("integration method %d unavailable\n",method);
	    result = 0;
	    break;
	case 2:
	    /*
	    ** Adams-Bashforth 2 step
	    */
	    result = AB2(segment->parent,&(segment->child),
			    state,dy,dt,
			    name);
	    break;
	case 3:
	    /*
	    ** Adams-Bashforth 3 step
	    */
	    result = AB3(segment->parent,&(segment->child),
			    state,dy,dt,
			    name);
	    break;
	case -1:
	    /*
	    ** Euler: a definite must
	    */
	    result = Euler(state,dy,dt);
	    break;
    }
    return(result);
}

double IntegrateMethod(method,segment,state,A,B,dt,name)
int	method;
Segment	*segment;
double	state;
double	A;
double	B;
double	dt;
char *name;
{
double D;
double result;

    switch(method){
	case 0:
	    /*
	    ** the default is the exponential integration form
	    */
	    D = exp(-B*dt);
	    result = state*D + (A/B)*(1-D);
	    break;
	case 1:
	    /*
	    ** Gear 2nd order
	    */
	    result = Gear(segment->parent,&(segment->child),
			    state,A,B,dt,name);
	    break;
	case 2:
	    /*
	    ** Adams-Bashforth 2 step
	    */
	    result = AB2(segment->parent,&(segment->child),
			    state,A - state*B,dt,
			    name);
	    break;
	case 3:
	    /*
	    ** Adams-Bashforth 3 step
	    */
	    result = AB3(segment->parent,&(segment->child),
			    state,A - state*B,dt,
			    name);
	    break;
	case 4:
	    /*
	    ** Trapezoidal
	    */
	    result = Trapezoid(state,A,B,dt);
	    break;
	case -1:
	    /*
	    ** Euler: a definite must
	    */
	    result = Euler(state,A - state*B,dt);
	    break;
    }
    return(result);
}


#define NSTEPS 2

double Gear(element,y,alpha,beta,dt,y_buffername)
Element *element;
double y;
double alpha;
double beta;
double dt;
char *y_buffername;
{
Buffer	*buffer;
Event	*event;
double		value;

    /*
    ** try to get the buffer by name
    */
    if((buffer = (Buffer *)FindElement(element,y_buffername)) == NULL){
	/*
	** if it fails then create the buffer needed
	*/
	buffer=(Buffer *)Create("passive_buffer",y_buffername,element,NULL,0);
	Block(buffer);
    }
    /*
    ** check the size to make sure it is the correct size
    */
    if(buffer->size < NSTEPS){
	/*
	** if it is too small then expand it
	*/
	ExpandBuffer(buffer, NSTEPS - buffer->size); 
    }
    /*
    ** check the buffer to make sure it is full before trying
    ** to get events from it
    */
    if(PreviousEvent(buffer,0)){
	/*
	** do the 2nd order Gear
	*/
	value = (4.0*y - PreviousEvent(buffer,0)->magnitude + 2.0*dt*alpha) /
		(3.0*(1 + 2.0*dt*beta/3.0));

    } else {
	/*
	** use Trapezoidal to start up
	*/
	value = Trapezoid(y,alpha,beta,dt);
    }
    /*
    ** add the current state to the buffer
    */
    PutEvent(buffer, value, NULL, WRAP);
    return(value);
}

#undef NSTEPS

#define NSTEPS 2

double AB2(element,buffer_head,y,dy,dt,dy_buffername)
Element *element;
Buffer **buffer_head;
double y;
double dy;
double dt;
char *dy_buffername;
{
Buffer	*buffer;
Event	*event;
extern int	ClearBuffer();
double		value;

    /*
    ** try to get the buffer by name
    */
    if((buffer = (Buffer *)FindElement(element,dy_buffername)) == NULL){
	/*
	** if it fails then create the buffer needed
	*/
	buffer=(Buffer *)Create("passive_buffer",dy_buffername,element,NULL,0);
	Block(buffer);
    }
    /*
    ** check the size to make sure it is the correct size
    */
    if(buffer->size < NSTEPS){
	/*
	** if it is too small then expand it
	*/
	ExpandBuffer(buffer, NSTEPS - buffer->size); 
    }
    /*
    ** check the buffer to make sure it is full before trying
    ** to get events from it
    */
    if(PreviousEvent(buffer,0)){
	/*
	** do the 2 step Adams Bashforth integration
	*/
	INTEG_AB2();
    } else {
	/*
	** use Eulers to start up
	*/
	value = Euler(y,dy,dt);
    }
    /*
    ** add the current derivative to the buffer
    */
    PutEvent(buffer, dy, NULL, WRAP);
    return(value);
}

#undef NSTEPS

#define NSTEPS 3

double AB3(element,buffer_head,y,dy,dt,dy_buffername)
Element *element;
Buffer **buffer_head;
double y;
double dy;
double dt;
char *dy_buffername;
{
Buffer	*buffer;
Event	*event;
extern int	ClearBuffer();
double		value;

    /*
    ** try to get the buffer by name
    */
    if((buffer = (Buffer *)FindElement(element,dy_buffername)) == NULL){
	/*
	** if it fails then create the buffer needed
	*/
	buffer=(Buffer *)Create("passive_buffer",dy_buffername,element,NULL,0);
	Block(buffer);
    }
    /*
    ** check the size to make sure it is the correct size
    */
    if(buffer->size < NSTEPS){
	/*
	** if it is too small then expand it
	*/
	ExpandBuffer(buffer, NSTEPS - buffer->size); 
    }
    /*
    ** check the buffer to make sure it is full before trying
    ** to get events from it
    */
    if(PreviousEvent(buffer,1)){
	/*
	** do the 3 step Adams Bashforth integration
	*/
	INTEG_AB3();
    } else 
    /*
    ** try the next step down
    */
    if(PreviousEvent(buffer,0)){
	/*
	** do the 2 step Adams Bashforth integration
	*/
	INTEG_AB2();
    } else {
	/*
	** use Eulers to start up
	*/
	value = Euler(y,dy,dt);
    }
    /*
    ** add the current derivative to the buffer
    */
    PutEvent(buffer, dy, NULL, WRAP);
    return(value);
}

#undef NSTEPS

#define NSTEPS 4

double AB4(element,buffer_head,y,dy,dt,dy_buffername)
Element *element;
Buffer **buffer_head;
double y;
double dy;
double dt;
char *dy_buffername;
{
Buffer	*buffer;
Event	*event;
extern int	ClearBuffer();
double		value;

    /*
    ** try to get the buffer by name
    */
    if((buffer = (Buffer *)FindElement(element,dy_buffername)) == NULL){
	/*
	** if it fails then create the buffer needed
	*/
	buffer=(Buffer *)Create("passive_buffer",dy_buffername,element,NULL,0);
	Block(buffer);
    }
    /*
    ** check the size to make sure it is the correct size
    */
    if(buffer->size < NSTEPS){
	/*
	** if it is too small then expand it
	*/
	ExpandBuffer(buffer, NSTEPS - buffer->size); 
    }
    /*
    ** check the buffer to make sure it is full before trying
    ** to get events from it
    */
    if(PreviousEvent(buffer,2)){
	/*
	** do the 4 step Adams Bashforth integration
	*/
	INTEG_AB4();
    } else
    /*
    ** try the next step down
    */
    if(PreviousEvent(buffer,1)){
	/*
	** do the 3 step Adams Bashforth integration
	*/
	INTEG_AB3();
    } else 
    if(PreviousEvent(buffer,0)){
	/*
	** do the 2 step Adams Bashforth integration
	*/
	INTEG_AB2();
    } else {
	/*
	** use Eulers to start up
	*/
	value = Euler(y,dy,dt);
    }
    /*
    ** add the current derivative to the buffer
    */
    PutEvent(buffer, dy, NULL, WRAP);
    return(value);
}

#undef NSTEPS

#define NSTEPS 5

double AB5(element,buffer_head,y,dy,dt,dy_buffername)
Element *element;
Buffer **buffer_head;
double y;
double dy;
double dt;
char *dy_buffername;
{
Buffer	*buffer;
Event	*event;
extern int	ClearBuffer();
double		value;

    /*
    ** try to get the buffer by name
    */
    if((buffer = (Buffer *)FindElement(element,dy_buffername)) == NULL){
	/*
	** if it fails then create the buffer needed
	*/
	buffer=(Buffer *)Create("passive_buffer",dy_buffername,element,NULL,0);
	Block(buffer);
    }
    /*
    ** check the size to make sure it is the correct size
    */
    if(buffer->size < NSTEPS){
	/*
	** if it is too small then expand it
	*/
	ExpandBuffer(buffer, NSTEPS - buffer->size); 
    }
    /*
    ** check the buffer to make sure it is full before trying
    ** to get events from it
    */
    if(PreviousEvent(buffer,3)){
	/*
	** do the 5 step Adams Bashforth integration
	*/
	INTEG_AB5();
    } else 
    /*
    ** try the next step down
    */
    if(PreviousEvent(buffer,2)){
	/*
	** do the 4 step Adams Bashforth integration
	*/
	INTEG_AB4();
    } else 
    if(PreviousEvent(buffer,1)){
	/*
	** do the 3 step Adams Bashforth integration
	*/
	INTEG_AB3();
    } else 
    if(PreviousEvent(buffer,0)){
	/*
	** do the 2 step Adams Bashforth integration
	*/
	INTEG_AB2();
    } else {
	/*
	** use Eulers to start up
	*/
	value = Euler(y,dy,dt);
    }
    /*
    ** add the current derivative to the buffer
    */
    PutEvent(buffer, dy, NULL, WRAP);
    return(value);
}

#undef NSTEPS
