#include "sim_ext.h"

ElementList delete_list;

/*
** recursively delete the element tree in a depth first fashion
*/
void FreeTree(element)
Element *element;
{
    if(element->child){
	/*
	** free the children
	*/
	FreeTree(element->child);
    } 
    if(element->next){
	/*
	** and the siblings
	*/
	FreeTree(element->next);
    }
    /*
    ** then free the element
    */
    FreeElement(element);
}

DeleteElement(element)
Element *element;
{
Element *child;
Element *parent;

    if(element == NULL || (parent = element->parent) == NULL){
	    return(0);
    }
    /*
    ** try to remove the reference to the element in its parent
    */
    /*
    ** find the location of the element in the child list
    */
    for(child = parent->child;
	(child && child->next && !(child->next == element) && 
	!(child == element));
	child=child->next);
    /*
    ** the element was not found
    */
    if(child == NULL){
	    return(0);
    } else
    /*
    ** it reached the end of the list but still no match
    */
    if(child->next == NULL && child != element){
	    return(0);
    } else
    /*
    ** a match was found as the first child
    ** this can only happen if it is the first child
    */
    if(child == element){
	    parent->child = element->next;
    } else 
    /*
    ** a match was found
    */
    if(child->next == element){
	child->next = element->next;
    } else {
	return(0);
    }
    /*
    ** traverse the subtree and free the children
    */
    element->next = NULL;
    FreeTree(element);
    return(1);
}

do_clean(argc,argv)
int argc;
char **argv;
{

    if(argc < 2){
	printf("\n*** WARNING ***\n");
	printf("This function removes all elements from the simulation.\n");
	printf("If you really want to do this use 'clean -f'.\n");
	printf("\n");
	printf("usage: %s [-f][-all]\n",argv[0]);
	return;
    } else
    if(strcmp(argv[1],"-f") == 0){
	FreeTree(RootElement());
	sfreeall();
	StartupElements();
    } else 
    if(strcmp(argv[1],"-all") == 0){
	FreeTree(RootElement());
	sfreeall();
	SimStartup();
    } else {
	printf("usage: %s [-f][-all]\n",argv[0]);
	return;
    }
    ClearWorkingSchedule();
    CloseAllAsciiFiles();
    ResetErrors();
    DefaultPrompt();
    printf("Simulator cleaned out.\n");
    OK();
}

FreeElement(element)
Element *element;
{
char 	*ptr;
int 	i;
PFI	action_func;
MsgOut	*msgout;
MsgIn	*msgin;
Element	*target;
static Action delete_action = { "DELETE", DELETE, NULL, 0, NULL, NULL };
MsgOut	*GetMsgOutByMsgIn();

    if(element == NULL){
	return(0);
    }
    /*
    ** if the current element is about to be deleted then
    ** change the current element to someplace safe
    */
    if(element == WorkingElement()){
	SetWorkingElement(RootElement());
    }
    /*
    ** should remove all possible dependencies
    ** on the element including map references, connections to
    ** segments on the element
    */
    /*
    ** delete all incoming messages
    */
    while(msgin = element->msg_in){
	if((msgout = GetMsgOutByMsgIn(msgin->src,msgin))== NULL){
	    Error();
	    printf("could not get OUT msg from '%s'\n",
	    Pathname(msgin->src));
	    return(0);
	}
	if(!DeleteMsgOut(msgin->src,msgout)){
	    printf("unable to delete msgout\n");
	    return(0);
	}
	if(!DeleteMsgIn(element,msgin)){
	    printf("unable to delete msgin\n");
	    return(0);
	}
    }
    /*
    ** delete all outgoing messages
    */
    while(msgout = element->msg_out){
	if(!DeleteMsgIn(msgout->dst,msgout->msg_in)){
	    printf("unable to delete msgin\n");
	    return(0);
	}
	if(!DeleteMsgOut(element,msgout)){
	    printf("unable to delete msgout\n");
	    return(0);
	}
    }
    /*
    ** try to call any special delete function 
    */
    if(action_func = GetActionFunc(element->object,DELETE)){
	action_func(element,&delete_action);
    }
    /*
    ** free the element
    */
    sfree(element);
    return(1);
}

Connection *FindConnectionToSegment(src_projection,target)
Projection *src_projection;
Segment *target;
{
Connection *conn;

    if(src_projection == NULL || target == NULL){
	return(NULL);
    }
    /*
    ** find the connection based on its destination
    */
    for(conn=src_projection->connection;conn;conn=conn->next){
	if(conn->target == target){
	    return(conn);
	}
    }
    return(NULL);
}

DeleteConnection(src_projection,target_connection)
Projection *src_projection;
Connection *target_connection;
{
Connection *connection;
Connection *delete_connection;

    if(src_projection == NULL || target_connection == NULL)
	return(0);
    /*
    ** find the connection to be deleted
    */
    for(connection = src_projection->connection;
    (connection && !(connection->next == target_connection)
    && !(connection == target_connection));
    connection=connection->next);
    /*
    ** couldnt find any connection which matched
    */
    if(connection == NULL){
	/*
	** failed
	*/
	return(0);
    }
    /*
    ** if there is only one connection then this condition
    ** could occur
    */
    if(connection == target_connection){
	delete_connection = connection;
	/*
	** delete the connection
	*/
	src_projection->connection = delete_connection->next;
	/*
	** free the memory used
	*/
	sfree(delete_connection);
	/*
	** success
	*/
	return(1);
    } else 
    if(connection->next == target_connection){
	/*
	** otherwise remove the connection
	*/
	delete_connection = connection->next;
	/*
	** point to the connection following the deleted connection
	*/
	connection->next = delete_connection->next;
	/*
	** free the memory taken by the connection
	*/
	sfree(delete_connection);
	/*
	** success
	*/
	return(1);
    } else {
	/*
	** failed
	*/
	return(0);
    }
}

do_delete_element(argc,argv)
int argc;
char **argv;
{
Element *element;

    if(argc < 2){
	printf("usage: %s element\n",argv[0]);
	return;
    }
    element = GetElement(argv[1]);
    if(element){
	if(!DeleteElement(element)){
	    printf("unable to delete element %s\n",argv[1]);
	return;
	}
    } else {
	printf("unable to find element %s\n",argv[1]);
	return;
    }
    OK();
}

void do_delete_connection(argc,argv)
int 	argc;
char 	**argv;
{
Projection 	*src_projection;
Segment 	*target;
Connection 	*connection;

    if(argc < 3){
	printf("usage: %s from_projection to_segment\n",argv[0]);
	return;
    }

    src_projection = (Projection *)GetElement(argv[1]);
    target = (Segment *)GetElement(argv[2]);
    if((connection=FindConnectionToSegment(src_projection,target))==NULL){
	printf("cant find connection from '%s' to '%s'\n",
	Pathname(src_projection),
	Pathname(target));
	return;
    } else {
	if(!DeleteConnection(src_projection,connection)){
	    printf("unable to delete connection\n");
	    return;
	}
    }
    OK();
}

