#include "olf_ext.h"

/*
** Calculates a gate state using tabulated values. The only assumption
** is that the gate obeys the following differential equation :
**
**	dm/dt = alpha * (1 - m) - beta * m
**
** where alpha and beta are interpolated tables giving the dependency
** on the activation field.
** The form of the object is lifted directly from Matt Wilson's
** Vdep gate object. The exponential calculation routines have been
** replaced with lookup tables instead.
*/

TabGate(gate,action)
register struct tab_gate_type *gate;
Action		*action;
{
MsgIn	*msg;
int	int_method = 0;
double	GateState();
double	GateSteadyState();
double	alpha,beta;
double	salpha,sbeta;
double	palpha,pbeta;
char	*field;
int		xdivs;
float	xmin,xmax;
Interpol *create_interpol();

    if(debug > 1){
	ActionHeader("VDepGate",gate,action);
    }

    SELECT_ACTION(action){
    case PROCESS:
        /*
        ** check all of the messages to the gate
        */
		salpha = 0;
		sbeta = 0;
		palpha = 1;
		pbeta = 1;
        MSGLOOP(gate,msg) {
            case 0:				/* compartment */
        	/*
        	** 0 = membrane potential
        	*/
        	gate->activation = MSGVALUE(msg,0);
        	break;
            case 1:				/* integration method */
        	int_method = MSGVALUE(msg,0);
        	break;
			case 2:				/* summed alphas */
				salpha += MSGVALUE(msg,0);
			break;
			case 3:				/* product alphas */
				palpha *= MSGVALUE(msg,0);
			break;
			case 4:				/* summed betas */
				sbeta += MSGVALUE(msg,0);
			break;
			case 5:				/* product betas */
				pbeta *= MSGVALUE(msg,0);
			break;
        }

        /* 
        ** calculate the voltage dependent state variable m
        */
		if (gate->alpha_alloced)
        	alpha = TabInterp(gate->alpha,gate->activation) * palpha + salpha;
		else
			alpha = palpha + salpha;

		if (gate->beta_alloced)
        	beta = TabInterp(gate->beta,gate->activation) * pbeta + sbeta;
		else
			beta = pbeta + sbeta;

        if(gate->instantaneous){
            /*
            ** calculate the steady state value of the state variable
            */

            gate->m = alpha/(alpha+beta);
        } else {
        	gate->m = IntegrateMethod(int_method,gate,gate->m,
        		alpha,alpha+beta,Clockrate(gate),"m");
        }
        break;

    case RESET:
        /*
        ** check all of the messages to the gate
        */
        MSGLOOP(gate,msg) {
            case 2:				/* Erest */
        	gate->activation = MSGVALUE(msg,0);
        	break;
        }
		if (!(gate->alpha_alloced && gate->beta_alloced))

        /*
        ** calculate the steady state value of the state variable
        */
        alpha = TabInterp(gate->alpha,gate->activation);
        beta = TabInterp(gate->beta,gate->activation);
        gate->m = alpha/(alpha+beta);
        break;

        /* Specially for the array fields */
    case TABCREATE:
		if (action->argc < 4) {
			printf("usage : %s field xdivs xmin xmax\n","tabcreate");
			return(0);
		}
		field = action->argv[0];
		if (strcmp(field,"alpha") != 0 &&
			strcmp(field,"beta") != 0) {
			printf("field '%s' not known\n",field);
			return(0);
		}
		xdivs = atoi(action->argv[1]);
		xmin = atof(action->argv[2]);
		xmax = atof(action->argv[3]);
		if (strcmp(field,"alpha") != 0 &&
			strcmp(field,"beta") != 0)
			return(0); /* do the normal set */
		if (strcmp(field,"alpha") == 0) {
			gate->alpha = create_interpol(xdivs,xmin,xmax);
			gate->alpha_alloced = 1;
		} else if (strcmp(field,"beta") == 0) {
			gate->beta = create_interpol(xdivs,xmin,xmax);
			gate->beta_alloced = 1;
		}
        break;
    }
}

TabGate_CALC_MINF(gate,action)
register struct tab_gate_type *gate;
Action      *action;
{
double  GateSteadyState();
double	alpha,beta;

    if(action->argc > 0){
    gate->activation = Atof(action->argv[0]);
    } else {
    Error();
    printf("CALC_MINF action requires voltage argument\n");
    }
    /*
    ** calculate the steady state value of the state variable
    */
	alpha = TabInterp(gate->alpha,gate->activation);
	beta = TabInterp(gate->beta,gate->activation);
    gate->m = alpha/(alpha+beta);
    action->passback = ftoa(gate->m);
}

