/*
 * slip.c - Slip specific code in diald.
 *
 * Copyright (c) 1994 Eric Schenk.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL ERIC SCHENK BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ERIC
 * SCHENK HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ERIC SCHENK SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND ERIC SCHENK HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include "diald.h"

/*
 * SLIP PACKET READING CODE FROM RFC 1055 by J. Romkey.
 *
 * RECV_PACKET: receives a packet into the buffer located at "p".
 *      If more than len bytes are received, the packet will
 *      be truncated.
 *      Returns the number of bytes stored in the buffer.
 */

/* SLIP special character codes */
#define END             0300    /* indicates end of packet */
#define ESC             0333    /* indicates byte stuffing */
#define ESC_END         0334    /* ESC ESC_END means END data byte */
#define ESC_ESC         0335    /* ESC ESC_ESC means ESC data byte */

int recv_packet(unsigned char *p, int len)
{
    int c;
    int received = 0;


    /* sit in a loop reading bytes until we put together
    * a whole packet.
    * Make sure not to copy them into the packet if we
    * run out of room.
    */
    while(1) {
       /* get a character to process
	*/
       c = getc(proxy_mfp);

       /* handle bytestuffing if necessary
	*/
       switch(c) {

       /* if it's an END character then we're done with
	* the packet
	*/
       case END:
	       /* a minor optimization: if there is no
		* data in the packet, ignore it. This is
		* meant to avoid bothering IP with all
		* the empty packets generated by the
		* duplicate END characters which are in
		* turn sent to try to detect line noise.
		*/
	       if(received)
		       return received;
	       else
		       break;

       /* if it's the same code as an ESC character, wait
	* and get another character and then figure out
	* what to store in the packet based on that.
	*/
       case ESC:
	       c = getc(proxy_mfp);

	       /* if "c" is not one of these two, then we
		* have a protocol violation.  The best bet
		* seems to be to leave the byte alone and
		* just stuff it into the packet
		*/
	       switch(c) {
	       case ESC_END:
		       c = END;
		       break;
	       case ESC_ESC:
		       c = ESC;
		       break;
		       }

       /* here we fall into the default handler and let
	* it store the character for us
	*/
       default:
	       if(received < len)
		       p[received++] = c;
	       }
     }
}

/*
 * Set the pty to an 8 bit clean mode and change it to the
 * desired SLIP line disciple, and run ifconfig to configure the
 * device as up.
 * The SLIP configuration is essentially what slattach
 * does, but we do it here so we know what interface (sl*)
 * gets opened as a result. (slattach doesn't return this)
 */

void proxy_up(void)
{
    int disc, sencap;

    /* change proxy_sfd to 8 bit clean line, 38400 speed */
    set_up_tty(proxy_sfd,1, 38400);

    if (ioctl(proxy_sfd, TIOCGETD, &orig_disc) < 0)
	syslog(LOG_ERR,"Can't get line discipline on proxy device: %m"), die(1);

    /* change line disciple to SLIP and set the SLIP encapsulation */
    disc = N_SLIP;
    if ((proxy_iface = ioctl(proxy_sfd, TIOCSETD, &disc)) < 0) {
	if (errno == ENFILE) {
	   syslog(LOG_ERR,"No free slip device available for proxy."), die(1);
	} else if (errno == EEXIST) {
	    syslog(LOG_ERR,"Proxy device already in slip mode!?");
	} else if (errno == EINVAL) {
	    syslog(LOG_ERR,"SLIP not supported by kernel, can't build proxy.");
	    die(1);
	} else
	   syslog(LOG_ERR,"Can't set line discipline: %m"), die(1);
    }

    if (ioctl(proxy_sfd, SIOCSIFENCAP, &slip_encap) < 0)
	syslog(LOG_ERR,"Can't set encapsulation: %m"), die(1);

    /* verify that it worked */
    if (ioctl(proxy_sfd, TIOCGETD, &disc) < 0)
	syslog(LOG_ERR,"Can't get line discipline: %m"), die(1);
    if (ioctl(proxy_sfd, SIOCGIFENCAP, &sencap) < 0)
	syslog(LOG_ERR,"Can't get encapsulation: %m"), die(1);

    if (disc != N_SLIP || sencap != slip_encap)
        syslog(LOG_ERR,"Couldn't set up the proxy link correctly!"), die(1);

    if (debug&DEBUG_VERBOSE)
        syslog(LOG_INFO,"Proxy device established on interface sl%d",
	    proxy_iface);
    proxy_config();
}

/*
 * Configure the proxy SLIP lines address.
 * This may change in a dynamic setting.
 */
void proxy_config(void)
{
    char buf[128];
    /* mark the interface as up */
    if (netmask) {
        sprintf(buf,"%s sl%d %s pointopoint %s netmask %s mtu %d up",
	    PATH_IFCONFIG,proxy_iface,local_ip,remote_ip,netmask,mtu);
    } else {
        sprintf(buf,"%s sl%d %s pointopoint %s mtu %d up",
	    PATH_IFCONFIG,proxy_iface,local_ip,remote_ip,mtu);
    }
    system(buf);
}

/*
 * Add in a direct and default route to the slip link.
 * The default route is only added if the "default" option was
 * requested by the user.
 */

void route_to_proxy()
{
    char buf[128];

    if (debug&DEBUG_VERBOSE)
	syslog(LOG_INFO, "Establishing routes to proxy device");
    /* Do I need or want the direct local route? */
    sprintf(buf,"%s add %s sl%d",PATH_ROUTE,remote_ip,proxy_iface); 
    system(buf);
    /* Add in a default route for the link */
    /* FIXME: should this refuse to add if a default route exists? */
    if (default_route) {
        sprintf(buf,"%s add default gw %s sl%d",
	    PATH_ROUTE,remote_ip,proxy_iface);
        system(buf);
    }
    /* FIXME: I should do proxyarp routing here as well */
    /* call addroute script */
    if (addroute) {
        sprintf(buf,"%s sl%d %s %s %s",addroute,proxy_iface,(netmask)?netmask:"default",local_ip,remote_ip);
        if (system(buf) < 0) {
	    syslog(LOG_ERR,"addroute script %s failed",delroute);
        }
    }
}

/*
 * Cal the delroute script and take the pty out of slip mode.
 */
void proxy_down()
{
    char buf[128];
    if (debug&DEBUG_VERBOSE)
	syslog(LOG_INFO,"taking proxy device down");
    if (delroute) {
	/* call delroute <iface> <netmask> <local> <remote> */
        sprintf(buf,"%s sl%d %s %s %s",delroute,proxy_iface,(netmask)?netmask:"default",local_ip,remote_ip);
        if (system(buf) < 0) {
	    syslog(LOG_ERR,"delroute script %s failed",delroute);
        }
    }
    /* clear the line discipline */
    if ((proxy_iface = ioctl(proxy_sfd, TIOCSETD, &orig_disc)) < 0)
	syslog(LOG_ERR,"Can't set line discipline: %m"), die(1);
}

static int grab_addr(char **var)
{
    int len, i = 0;
    int state = 0;
    unsigned char buffer[128];
    unsigned char c;

    while (1) {
	if ((len = read(modem_fd,&c,1)) < 0) {
	    if (errno != EINTR || terminate) {
	        syslog(LOG_ERR,"Error reading from modem: %m");
	        return 0;
            }
        }
	if (len == 1) {
	    switch (state) {
	    case 0:	/* wait for a number to come up */
		if (isdigit(c)) buffer[i++] = c, state = 1;
		break;
	    case 1:
		if (isdigit(c) || c == '.') buffer[i++] = c;
		else {
		    if (buffer[i-1] == '.') i--;   /* trim off trailing "." */
		    buffer[i++] = 0;
		    goto done;
		}
		break;
	    }
	    if (i >= 128)
		syslog(LOG_ERR,"Buffer overflow when reading IP address"),
		die(1);
	}
    }
done:
    *var = strdup(buffer);
    return 1;
}

/*
 * Read the local and remote addresses from the modem.
 * The server on the other side is assumed to have written
 * these out to the line before actually going into slip mode.
 */
void dynamic_slip(void)
{
    if (debug&DEBUG_VERBOSE)
	syslog(LOG_INFO,"Fetching IP addresses from SLIP server");
    if (dynamic_mode == DMODE_REMOTE || dynamic_mode == DMODE_REMOTE_LOCAL)
	if (!grab_addr(&remote_ip)) return;
    if (dynamic_mode != DMODE_REMOTE)
	if (!grab_addr(&local_ip)) return;
    if (dynamic_mode == DMODE_LOCAL_REMOTE)
	if (!grab_addr(&remote_ip)) return;
    syslog(LOG_INFO,"New addresses: local %s, remote %s.",
	local_ip,remote_ip);
    proxy_config();
}
