/*
 * modem.c - Modem control functions.
 *
 * Copyright (c) 1994 Eric Schenk.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL ERIC SCHENK BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ERIC
 * SCHENK HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ERIC SCHENK SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND ERIC SCHENK HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * Portions of this code were derived from the code for pppd copyright
 * (c) 1989 Carnegie Mellon University. The copyright notice on this code
 * is reproduced below.
 *
 * Copyright (c) 1989 Carnegie Mellon University.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Carnegie Mellon University.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include "diald.h"

/* local variables */

static struct termios inittermios;      /* Initial TTY termios */
static int restore_term = 0;
int pgrpid;

#if B9600 == 9600
/*
 * XXX assume speed_t values numerically equal bits per second
 * (so we can ask for any speed).
 */
#define translate_speed(bps)	(bps)

#else
/*
 * List of valid speeds.
 */
struct speed {
    int speed_int, speed_val;
} speeds[] = {
#ifdef B50
    { 50, B50 },
#endif
#ifdef B75
    { 75, B75 },
#endif
#ifdef B110
    { 110, B110 },
#endif
#ifdef B134
    { 134, B134 },
#endif
#ifdef B150
    { 150, B150 },
#endif
#ifdef B200
    { 200, B200 },
#endif
#ifdef B300
    { 300, B300 },
#endif
#ifdef B600
    { 600, B600 },
#endif
#ifdef B1200
    { 1200, B1200 },
#endif
#ifdef B1800
    { 1800, B1800 },
#endif
#ifdef B2000
    { 2000, B2000 },
#endif
#ifdef B2400
    { 2400, B2400 },
#endif
#ifdef B3600
    { 3600, B3600 },
#endif
#ifdef B4800
    { 4800, B4800 },
#endif
#ifdef B7200
    { 7200, B7200 },
#endif
#ifdef B9600
    { 9600, B9600 },
#endif
#ifdef B19200
    { 19200, B19200 },
#endif
#ifdef B38400
    { 38400, B38400 },
#endif
#ifdef EXTA
    { 19200, EXTA },
#endif
#ifdef EXTB
    { 38400, EXTB },
#endif
#ifdef B57600
    { 57600, B57600 },
#endif
#ifdef B115200
    { 115200, B115200 },
#endif
    { 0, 0 }
};

/*
 * Translate from bits/second to a speed_t.
 */
int translate_speed(int bps)
{
    struct speed *speedp;

    if (bps == 0)
	return 0;
    for (speedp = speeds; speedp->speed_int; speedp++)
	if (bps == speedp->speed_int)
	    return speedp->speed_val;
    syslog(LOG_WARNING, "speed %d not supported", bps);
    return 0;
}
#endif

/*
 * set_up_tty: Set up the serial port on `fd' for 8 bits, no parity,
 * at the requested speed, etc.  If `local' is true, set CLOCAL
 * regardless of whether the modem option was specified.
 */
void set_up_tty(int fd, int local, int spd)
{
    int speed, i;
    struct termios tios;

    if (tcgetattr(fd, &tios) < 0) {
	syslog(LOG_ERR, "could not get initial terminal attributes: %m");
    }

    tios.c_cflag = CS8 | CREAD | HUPCL;
    if (local || !modem) tios.c_cflag |= CLOCAL;
    if (crtscts == 1) tios.c_cflag |= CRTSCTS;
    tios.c_iflag = IGNBRK | IGNPAR;
    tios.c_oflag = 0;
    tios.c_lflag = 0;
    for (i = 0; i < NCCS; i++)
	tios.c_cc[i] = 0;
    tios.c_cc[VMIN] = 1;
    tios.c_cc[VTIME] = 0;

    speed = translate_speed(spd);
    if (speed) {
	cfsetospeed(&tios, speed);
	cfsetispeed(&tios, speed);
    } else {
	speed = cfgetospeed(&tios);
    }

    if (tcsetattr(fd, TCSAFLUSH, &tios) < 0) {
	syslog(LOG_ERR, "failed to set terminal attributes: %m");
    }
}

/*
 * setdtr - control the DTR line on the serial port.
 * This is called from die(), so it shouldn't call die().
 */
void setdtr(int fd, int on)
{
    int modembits = TIOCM_DTR;

    ioctl(fd, (on? TIOCMBIS: TIOCMBIC), &modembits);
}


/*
 * fork_dialer - run a program to connect the serial device.
 */
void fork_dialer(char *program, int in, int out)
{
    dial_pid = fork();

    if (dial_pid < 0) {
        syslog(LOG_ERR, "failed to fork dialer: %m");
        die(1);
    }

    if (dial_pid == 0) {
        setreuid(getuid(), getuid());
        setregid(getgid(), getgid());
        dup2(in, 0);
        dup2(out, 1);
        execl("/bin/sh", "sh", "-c", program, (char *)0);
        syslog(LOG_ERR, "could not exec /bin/sh: %m");
        _exit(99);
        /* NOTREACHED */
    }
    syslog(LOG_INFO,"Running connect (pid = %d).",dial_pid);
}

/*
 * Open up a modem and set up the desired parameters.
 */
void open_modem()
{
    int npgrpid;
    int i;
    /*
     * Open the serial device and set it up.
     */

    modem_hup = 0;
    modem_fd = -1;

    for (i = 0; i < device_count; i++) {
        /*
         * March down the device list till we manage to open one up or
  	 * we run out of devices to try.
         */

        if (lock_dev && lock(devices[i]) < 0)
	    continue;

	/* OK. Locked one, try to open it */
	if ((modem_fd = open(devices[i], O_RDWR | O_NDELAY)) >= 0)
	    break;
	else {
	   syslog(LOG_ERR,"Error opening device %s: %m",devices[i]);
	}

	/* That didn't work, get rid of the lock */
        if (lock_dev) unlock();
    }

    if (modem_fd < 0) {
	syslog(LOG_INFO,"Couldn't find a free device to call out on.");
	dial_status = -1;
	return;
    }

    /* Make sure we are the session leader */
    if ((npgrpid = setsid()) >= 0)
	pgrpid = npgrpid;

    /* set device to be controlling tty */
    /* This should only happen in SLIP mode */
    if (mode == MODE_SLIP) {
	if (ioctl(modem_fd, TIOCSCTTY, 1) < 0) {
	    syslog(LOG_ERR, "failed to set modem to controlling tty: %m");
	    die(1);
	}

	if (tcsetpgrp(modem_fd, pgrpid) < 0) {
	    syslog(LOG_ERR, "failed to set process group: %m");
	    die(1);
	}
    }

    /* Get rid of any initial line noise */
    tcflush(modem_fd, TCIOFLUSH);

    if (tcgetattr(modem_fd, &inittermios) < 0) {
	syslog(LOG_ERR, "failed to get initial modem terminal attributes: %m");
    }

    /* So we don't try to restore if we die before this */
    restore_term = 1;

    /* Clear the NDELAY flag now (line is in CLOCAL) */
    if (fcntl(modem_fd,F_SETFL,fcntl(modem_fd,F_GETFL)&~(O_NDELAY)) < 0)
	syslog(LOG_ERR, "failed to clear O_NDELAY flag: %m"), die(1);

    /* hang up and then start again */
    set_up_tty(modem_fd, 1, 0);
    sleep(1);
    set_up_tty(modem_fd, 1, inspeed);

    /* Get rid of any initial line noise after the hangup */
    tcflush(modem_fd, TCIOFLUSH);

    fork_dialer(connector, modem_fd, modem_fd);
}

void finish_dial()
{
    set_up_tty(modem_fd, 0, inspeed);
}

/*
 * Close the modem, making sure it hangs up properly!
 */
void close_modem()
{
    if (debug&DEBUG_VERBOSE)
        syslog(LOG_INFO,"Closing modem line.");

    if (modem_fd < 0) {
 	return;
    }

    /* Get rid of what ever might be waiting to go out still */
    tcflush(modem_fd, TCIOFLUSH);

    /*
     * Restore the initial termio settings.
     */

    if (restore_term) {
	if (tcsetattr(modem_fd, TCSANOW, &inittermios) < 0) {
	    syslog(LOG_ERR,
		"failed to restore initial modem terminal settings: %m");
	}
    }

    /*
     * Hang up the modem up by dropping the DTR.
     * We do this because the initial termio settings
     * may not have set HUPCL. This forces the issue.
     * We need the sleep to give the modem a chance to hang
     * up before we get another program asserting the DTR.
     */
    setdtr(modem_fd, 0);
    sleep(1);

    close(modem_fd);
    if (lock_dev) unlock();
    modem_fd = -1;
}
