#include "timer.h"
#include <linux/param.h>

#define FW_ID_LEN 16			/* number of bytes in a connection id */
#define FW_OFFSET(x) ((x)&0x7f)		/* offset into a header */
#define FW_IP_OFFSET(x) ((x)&0x7f)	/* construct a ip header offset */
#define FW_DATA_OFFSET(x) (0x80|((x)&0x7f))	/* offset into an ip data */
#define FW_IN_IP(x) (((x)&0x80)==0)	/* in ip hdr segment */
#define FW_IN_DATA(x) ((x)&0x80)	/* in data segment */
#define FW_PROTO_ALL(x) ((x)==255)	/* any protocol */

/* Direction indicators */
#define FW_DIR_IN 0
#define FW_DIR_OUT 1
#define FW_DIR_BOTH 2

/* Comparision operators */
#define FW_EQ 0
#define FW_NE 1
#define FW_GE 2
#define FW_LE 3

#define FW_MAX_TERMS 10			/* Max terms per filter struct */
#define FW_MAX_PRULES 32
#define FW_NRUNIT 16			/* max # of unit's FW can monitor */

/*
 * Externally visible structures.
 */

typedef struct firewall_prule {
    unsigned char protocol;		/* Protocol: 255 = match all. */
    unsigned char codes[FW_ID_LEN];	/* coding rule, byte offsets */
} FW_ProtocolRule;

typedef struct firewall_term {
    unsigned char shift:5;	/* 0-31 */
    unsigned char op:2;		/* operation: =, !=, >=, <= */
    unsigned char offset;	/* offset code. Same as in protocol rules */
    unsigned int mask;		/* mask value */
    unsigned int test;		/* test value */
} FW_Term;

/*
 * Firewall filter.
 */
typedef struct firewall_rule {
    unsigned char prule:5;	/* protocol rule, 0-31. */
    unsigned char dir:2;	/* direction match. 0 = in, 1 = out, 2 = both */
    unsigned char log:1;	/* log matches to this rule */
    unsigned char accept:1;	/* Is this an accept or reject rule? */
    unsigned char count:7;	/* number of terms. maximum FW_MAX_TERMS */
    unsigned short timeout;	/* timeout in seconds. Max approx 18 hours */
    FW_Term terms[FW_MAX_TERMS];	/* terms in the rule */
} FW_Filter;

/*
 * Firewall request structure for ioctl's.
 */

struct firewall_req {
    unsigned char unit;			/* firewall unit */
    union {
        char ifname[16];		/* FIXME! */
	FW_Filter filter;
	FW_ProtocolRule rule;
	int value;
    } fw_arg;
};

/*
 * Firewall IOCTL's
 */

#define IP_FW_QFLUSH	1	/* flush the timeout queue */
#define IP_FW_AIFACE	2	/* associate with an interface */
#define IP_FW_DIFACE	3	/* disassociate with an interface */
#define IP_FW_QCHECK	4	/* is the queue empty or not */
#define IP_FW_FFLUSH	5	/* flush the filters */
#define IP_FW_PFLUSH	6	/* flush the protocol rules */
#define IP_FW_AFILT     7	/* add a filter rule */
#define IP_FW_APRULE	8	/* add a protocol rule */
#define IP_FW_PCONN	9	/* print the connections */
#define IP_FW_PPRULE	10	/* print the rules */
#define IP_FW_PFILT	11	/* print the filters */
#define IP_FW_OPEN	12	/* print the filters */
#define IP_FW_CLOSE	13	/* print the filters */

/*
 * Internal data structures.
 */

/*
 * List of filters.
 */
typedef struct fw_filters {
    struct fw_filters *next;	/* next filter in the firewall chain */
    FW_Filter filt;
} FW_Filters;

/*
 * Identifier structure.
 */
typedef struct {
    unsigned char id[FW_ID_LEN];	/* identifier for this connection */
} FW_ID;

/*
 * Connection entry;
 */
typedef struct fw_connection {
    struct timer_list timer;		/* timer for this connection */
    FW_ID id;				/* identifier for this connection */
    struct fw_connection *next,*prev;	/* queue chain pointers */
} FW_Connection;

typedef struct {
    FW_ProtocolRule prules[FW_MAX_PRULES];
    FW_Filters *filters;
    FW_Filters *last;
    FW_Connection *connections;
    char used;				/* is this unit free */
    char nrules;			/* how many rules are assigned */
    short nfilters;			/* how many filters are assigned */
} FW_unit;

int ctl_firewall(int, struct firewall_req *);
int check_firewall(int, int, unsigned char *, int);
