/*
 * filter.c - Packet filtering for diald.
 *
 * Copyright (c) 1994 Eric Schenk.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL ERIC SCHENK BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ERIC
 * SCHENK HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ERIC SCHENK SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND ERIC SCHENK HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <syslog.h>
#include <fcntl.h>
#include <ctype.h>
#include <time.h>
#include <net/if.h>
#include <netdb.h>
#include <netinet/ip.h>
#include <netinet/ip_tcp.h>
#include <netinet/ip_udp.h>
#include <netinet/ip_icmp.h>
#include <netinet/in.h>
#include <arpa/inet.h>

/* #include "firewall.h" */
#include "diald.h"

extern void die(int);
void block_timer(void);
void unblock_timer(void);

static void del_connection(FW_Connection *);

static FW_unit units[FW_NRUNIT];

static int initialized = 0;

extern void init_timer(struct timer_list *);

int timestamp(void)
{
    int c;
    time((time_t *)&c);
    return c;
}

/*
 * Initialize the units.
 */

static void init_units(void)
{
    int i;

    for (i = 0; i < FW_NRUNIT; i++) {
	units[i].used = 0;
	units[i].filters = NULL;
	units[i].last = NULL;
	units[i].connections = malloc(sizeof(FW_Connection));
	units[i].nrules = 0;
	units[i].nfilters = 0;
	if (!units[i].connections) {
	    syslog(LOG_ERR,"Out of memory! AIIEEE!");
	    die(1);
	}
	units[i].connections->next = units[i].connections->prev
	    = units[i].connections;
    }
    initialized = 1;
}

/*
 * Add a connection to the queue.
 */


static void add_connection(FW_unit *unit, FW_ID *id, unsigned int timeout)
{
    FW_Connection *c = unit->connections->next;

    /* FIXME: In user space I need to block while updating */
    block_timer();
    while (c != unit->connections) {
	if (memcmp((unsigned char *)&c->id,
		(unsigned char *)id,sizeof(FW_ID))==0)
	   break;
	c = c->next;
    }
    if (c == unit->connections) {
	c = malloc(sizeof(FW_Connection));
	if (c == 0) {
	   syslog(LOG_ERR,"Out of memory! AIIEEE!");
	   die(1);
	}
	c->id = *id;
	init_timer(&c->timer);
	c->timer.data = (int)c;
	c->timer.function = (void *)(int)del_connection;
	c->next = unit->connections->next;
	c->prev = unit->connections;
	unit->connections->next->prev = c;
	unit->connections->next = c;
    } else {
	del_timer(&c->timer);
    }
#ifdef 0
    syslog(LOG_INFO,"Adding connection %d @ %d - timeout %d",(int)c,
	timestamp(),timeout);
#endif
    c->timer.expires = timeout*HZ;
    add_timer(&c->timer);
    /* FIXME: Unblock after this */
    unblock_timer();
}

/*
 * Get a connection out of a queue.
 */

static void del_connection(FW_Connection *c)
{
#ifdef 0
    syslog(LOG_INFO,"Deleting connection %d @ %d",(int)c,timestamp());
#endif
    c->next->prev = c->prev;
    c->prev->next = c->next;
}

static void fw_log_packet(int accept, struct iphdr *pkt, int len)
{
    char saddr[20], daddr[20];
    struct in_addr addr;
    int sport = 0, dport = 0;
    struct tcphdr *tcp = (struct tcphdr *)((char *)pkt + 4*pkt->ihl);
    struct udphdr *udp = (struct udphdr *)tcp;

    addr.s_addr = pkt->saddr;
    strcpy(saddr,inet_ntoa(addr));
    addr.s_addr = pkt->daddr;
    strcpy(daddr,inet_ntoa(addr));

    if (pkt->protocol == IPPROTO_TCP || pkt->protocol == IPPROTO_UDP)
	sport = ntohs(udp->source), dport = ntohs(udp->dest);

    syslog(LOG_INFO,"firewall %s proto %d packet %s,%d => %s,%d",
	(accept == -1)?"checking":((accept)?"accepted":"rejected"),
	pkt->protocol, saddr, sport, daddr, dport);
}

void print_filter(FW_Filter *filter)
{
    int i;
    syslog(LOG_INFO,"filter: prl %d dir %d log %d acc %d cnt %d tm %d",
	filter->prule,filter->dir,filter->log,filter->accept,
	filter->count,filter->timeout);
    for (i = 0; i < filter->count; i++) {
	syslog(LOG_INFO,"    term: shift %d op %d off %d%c msk %x tst %x",
	    filter->terms[i].shift, filter->terms[i].op,
	    filter->terms[i].offset&0x7f,
	    (filter->terms[i].offset&0x80)?'d':'h',
	    filter->terms[i].mask, filter->terms[i].test);
    }
}

int check_firewall(int unitnum, int dir, unsigned char *pkt, int len)
{
    FW_unit *unit;
    FW_Filters *fw;
    unsigned char *data;
    FW_ProtocolRule *prule;
    FW_Term *term;
    int i,v;

    if (!initialized) init_units();

    if (unitnum < 0 || unitnum >= FW_NRUNIT) {
	/* FIXME: set an errorno? */
	return -1;
    }

    unit = &units[unitnum];
    fw = unit->filters;

    data = pkt + 4*((struct iphdr *)pkt)->ihl;

    while (fw) {
#ifdef 0
	print_filter(&fw->filt);
#endif
	if (dir != fw->filt.dir && fw->filt.dir != FW_DIR_BOTH)
	    goto next_rule;

	/* Check the protocol rule */
	prule = &unit->prules[fw->filt.prule];
	if (!(FW_PROTO_ALL(prule->protocol)
	|| prule->protocol == ((struct iphdr *)pkt)->protocol))
	    goto next_rule;

	/* Check the terms */
	for (i = 0;
	(fw->filt.count > FW_MAX_TERMS) || (i < fw->filt.count); i++) {
	    if (i > FW_MAX_TERMS && fw->filt.count == 0) {
		fw = fw->next, i = 0;
		if (fw == NULL) break;
	    }
	    term = &fw->filt.terms[i];
	    v = (ntohl(*(int *)(&(FW_IN_DATA(term->offset)?data:pkt)
				  [FW_OFFSET(term->offset)]))
		    >> term->shift) & term->mask;
#ifdef 0
	    syslog(LOG_INFO,"testing ip %x:%x data %x:%x mask %x shift %x test %x v %x",
		ntohl(*(int *)(&pkt[FW_OFFSET(term->offset)])),
		*(int *)(&pkt[FW_OFFSET(term->offset)]),
		ntohl(*(int *)(&data[FW_OFFSET(term->offset)])),
		*(int *)(&data[FW_OFFSET(term->offset)]),
		term->mask,
		term->shift,
		term->test,
		v);
#endif
	    switch (term->op) {
	    case FW_EQ: if (v != term->test) goto next_rule; break;
	    case FW_NE: if (v == term->test) goto next_rule; break;
	    case FW_GE: if (v >= term->test) goto next_rule; break;
	    case FW_LE: if (v <= term->test) goto next_rule; break;
	    }
	}
	/* Ok, we matched a rule. What are we suppose to do? */
#ifdef 0
	if (fw->filt.log)
#endif
        if (debug)
	    fw_log_packet(fw->filt.accept,(struct iphdr *)pkt,len);
	if (fw->filt.accept && fw->filt.timeout > 0) {
	    /* store the connection */
	    FW_ID id;
	    for (i = 0; i < FW_ID_LEN; i++)
		id.id[i] = (FW_IN_DATA(prule->codes[i])?data:pkt)
			    [FW_OFFSET(prule->codes[i])];
	    add_connection(unit,&id,fw->filt.timeout);
	}
	/* Return 1 if accepting rule with non zero timeout, 0 otherwise */
	return (fw->filt.accept&&fw->filt.timeout>0);

next_rule: /* try the next filter */
	fw = fw->next;
    }
    /* Failed to match any rule. For now this means an accept,
     * otherwise the kernel will reject all packets until we
     * install at least one firewall rule.
     */
    if (debug)
	fw_log_packet(1,(struct iphdr *)pkt,len);
    return 1;
}

int ctl_firewall(int op, struct firewall_req *req)
{
    FW_unit *unit;
    if (!initialized) init_units();

    /* Need to check that req is OK */

    if (req && req->unit >= FW_NRUNIT) return -1; /* ERRNO */

    if (req) unit = &units[req->unit];
    else unit = units;
    
    switch (op) {
    case IP_FW_QFLUSH:
	if (!req) return -1; /* ERRNO */
	{
	    FW_Connection *c;
	    while ((c = unit->connections->next) != unit->connections) {
		del_timer(&c->timer);
		del_connection(c);
		free((void *)c);
	    }
	    return 0;
	}
    case IP_FW_AIFACE:
	/* Do nothing for now. In the kernel I need to add to this. */
	return 0;
    case IP_FW_DIFACE:
	/* Do nothing for now. In the kernel I need to add to this. */
	return 0;
    case IP_FW_QCHECK:
	if (!req) return -1; /* ERRNO */
	return (unit->connections->next == unit->connections);
    case IP_FW_PFLUSH:
	if (!req) return -1; /* ERRNO */
	unit->nrules = 0;
	return 0;
    /* PFLUSH implies FFLUSH */
    case IP_FW_FFLUSH:
	if (!req) return -1; /* ERRNO */
	{
	    FW_Filters *next, *filt = unit->filters;
	    while (filt)
	    	{ next = filt->next; free(filt); filt = next; }
	    unit->filters = NULL;
	    unit->last = NULL;
	}
	return 0;
    case IP_FW_AFILT:
	if (!req) return -1; /* ERRNO */
	{
	    FW_Filters *filters = malloc(sizeof(FW_Filters));
	    if (filters == 0) {
		syslog(LOG_ERR,"Out of memory! AIIEEE!");
		return -1; /* ERRNO */
	    }
	    filters->next = 0;
	    filters->filt = req->fw_arg.filter;
	    if (unit->last) unit->last->next = filters;
	    if (!unit->filters) unit->filters = filters;
	    unit->last = filters;
	    unit->nfilters++;
	}
	return 0;
    case IP_FW_APRULE:
	if (!req) return -1; /* ERRNO */
	if (unit->nrules >= FW_MAX_PRULES) return -1; /* ERRNO */
	unit->prules[(int)unit->nrules] = req->fw_arg.rule;
	return unit->nrules++;
    /* Printing does nothing right now */
    case IP_FW_PCONN:
	if (!req) return -1; /* ERRNO */
	{
	    unsigned long atime = time(0);
	    FW_Connection *c;
	    char saddr[20], daddr[20];
    	    struct in_addr addr;
	    for (c=unit->connections->next; c!=unit->connections; c=c->next) {
                addr.s_addr = c->id.id[1] + (c->id.id[2]<<8)
                        + (c->id.id[3]<<16) + (c->id.id[4]<<24);
                strcpy(saddr,inet_ntoa(addr));
                addr.s_addr = c->id.id[5] + (c->id.id[6]<<8)
                        + (c->id.id[7]<<16) + (c->id.id[8]<<24);
                strcpy(daddr,inet_ntoa(addr));
                syslog(LOG_INFO,
                        "ttl %d, %d - %s/%d => %s/%d",
                        c->timer.expected-atime, c->id.id[0],
                        saddr, c->id.id[10]+(c->id.id[9]<<8),
                        daddr, c->id.id[12]+(c->id.id[11]<<8));
	    }
	    return 0;
	}
	return 0;
    case IP_FW_PPRULE:
	if (!req) return -1; /* ERRNO */
	return 0;
    case IP_FW_PFILT:
	if (!req) return -1; /* ERRNO */
	return 0;
    /* Opening and closing firewalls is cooperative right now.
     * Also, it does nothing to change the behavior of a device
     * associated with the firewall.
     */
    case IP_FW_OPEN:
	{
	    int i;
	    for (i = 0; i < FW_NRUNIT; i++)
		if (units[i].used == 0) {
		    struct firewall_req mreq;
		    mreq.unit = i;
		    ctl_firewall(IP_FW_PFLUSH,&mreq);
		    units[i].used = 1;
		    return i;
		}
	    return -1;	/* ERRNO */
	}
    case IP_FW_CLOSE:
	{
	    struct firewall_req mreq;
	    if (!req) return -1; /* ERRNO */
	    mreq.unit = req->unit;
	    ctl_firewall(IP_FW_PFLUSH,&mreq);
	    unit->used = 0;
	    return 0;
	}
    }
    return -1; /* ERRNO */
}
