/*
 * diald.c - Demand dialing daemon for ppp.
 *
 * Copyright (c) 1994 Eric Schenk.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL ERIC SCHENK BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ERIC
 * SCHENK HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ERIC SCHENK SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND ERIC SCHENK HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * Portions of this code were derived from the code for pppd copyright
 * (c) 1989 Carnegie Mellon University. The copyright notice on this code
 * is reproduced below.
 *
 * Copyright (c) 1989 Carnegie Mellon University.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Carnegie Mellon University.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include "diald.h"
#include "version.h"

/* intialized variables. */
int modem_fd = -1;		/* modem device fp (for proxy reads) */
int modem_hup = 0;		/* have we seen a modem HUP? */
int request_down = 0;		/* has the user requested link down? */
int request_up = 0;		/* has the user requested link down? */
int pppd_pid = 0;		/* current pppd command pid */
int dial_pid = 0;		/* current dial command pid */
int dial_status = 0;		/* status from last dial command */
int state_timeout = -1;		/* state machine timeout counter */
int proxy_iface = 0;
int ppp_iface = 0;
int terminate = 0;

void main(int argc, char *argv[])
{
    int sel;
    struct timeval timeout;
    fd_set readfds;

    /* initialize system log interface */
    openlog("diald", LOG_PID | LOG_NDELAY | LOG_PERROR,  LOG_LOCAL2);

    /* initialize a firewall unit so we can store our options */
    /* If I get things into a device this should be an "open" */
    fwunit = ctl_firewall(IP_FW_OPEN,0);

    /* Get the default defs and config files first */
    parse_options_file(DIALD_DEFS_FILE);
    parse_options_file(DIALD_CONFIG_FILE);
    /* Get the command line modifications */
    parse_args(argc-1,argv+1);
    /* Do validity checks on the setup */
    check_setup();

    become_daemon();

    if (debug&DEBUG_VERBOSE)
        syslog(LOG_INFO,"Starting diald version %s",VERSION);

    signal_setup();
    filter_setup();

    /* get a pty and open up a proxy link on it */
    get_pty(&proxy_mfd,&proxy_sfd);
    proxy_mfp = fdopen(proxy_mfd,"r+");
    proxy_up();
    route_to_proxy();
    idle_filter_proxy();

    if (debug&DEBUG_VERBOSE)
	syslog(LOG_INFO,"Daild initial setup completed.");

    /* main loop */
    timeout.tv_sec = PAUSETIME;
    timeout.tv_usec = 0;
    while (!terminate) {
	/* wait up to a second for an event */
        FD_ZERO(&readfds);
        FD_SET(proxy_mfd,&readfds);
        FD_SET(skfd,&readfds);
	if (mode == MODE_SLIP && state == STATE_UP)
	    FD_SET(modem_fd,&readfds);
	sel = select(100,&readfds,0,0,&timeout);
	if (sel > 0) {
	    /* update the connection filters */
	    if (FD_ISSET(skfd,&readfds)) filter_read();

	    /* deal with packets coming into the pty proxy link */
	    if (FD_ISSET(proxy_mfd,&readfds)) proxy_read();

	    /* deal with packets coming over the modem */
	    if (FD_ISSET(modem_fd,&readfds))
		modem_read();
	}
	if (timeout.tv_sec == 0 && timeout.tv_usec == 0) {
	    /* advance the clock 1 second */
	    timeout.tv_sec = PAUSETIME;
	    timeout.tv_usec = 0;
	    if (state_timeout > 0) state_timeout--;
	    if (debug&DEBUG_TICK)
	        syslog(LOG_DEBUG,"--- tick --- state %d block %d",state,blocked);
	}
	change_state();
    }
    die(0);
}

/*
 * Change into a daemon.
 * Get rid of the stdio streams, and disassociate from the original
 * controling terminal, and become a group leader.
 */

void become_daemon()
{
    int pid;
    if (daemon) {
        close(0);
        close(1);
        close(2);
	/* go into the background */
	if ((pid = fork()) < 0) {
	    syslog(LOG_ERR,"Could not fork into background: %m");
	    die(1);
	}
	/* parent process is finished */
	if (pid != 0) exit(0);
    }
}

/*
 * Set up the signal handlers.
 */

void signal_setup()
{
    int mask;
    struct sigaction sa;
    /* set up signal handlers */

    sigemptyset(&mask);
    sigaddset(&mask, SIGHUP);
    sigaddset(&mask, SIGINT);
    sigaddset(&mask, SIGTERM);
    sigaddset(&mask, SIGUSR1);
    sigaddset(&mask, SIGUSR2);
    sigaddset(&mask, SIGCHLD);
    sigaddset(&mask, SIGALRM);

#define SIGNAL(s, handler)      { \
        sa.sa_handler = handler; \
        if (sigaction(s, &sa, NULL) < 0) { \
            syslog(LOG_ERR, "sigaction(%d): %m", s); \
            die(1); \
        } \
    }

    sa.sa_mask = mask;
    sa.sa_flags = 0;
    SIGNAL(SIGHUP, sig_hup);            /* Hangup: modem went down. */
    SIGNAL(SIGINT, sig_intr);           /* Interrupt: take demand dialer down */
    SIGNAL(SIGTERM, sig_term);          /* Terminate: user take link down */
    SIGNAL(SIGUSR1, linkup);            /* User requests the link to go up */
    SIGNAL(SIGUSR2, print_filter_queue); /* dump the packet queue to the log */
    SIGNAL(SIGCHLD, sig_chld);		/* reap dead kids */
    SIGNAL(SIGALRM, alrm_timer);	/* Deal with a timer expired */
}

/*
 * Get a pty and open both the slave and master sides.
 */

void get_pty(int *mfd, int *sfd)
{
    char *ptys = "0123456789abcdef";
    int i,c;
    static char buf[128];

    for (c = 'p'; c <= 's'; c++)
        for (i = 0; i < 16; i++) {
	    sprintf(buf,"/dev/pty%c%c",c,ptys[i]);
	    if ((*mfd = open(buf,O_RDWR)) >= 0) {
	    	sprintf(buf,"/dev/tty%c%c",c,ptys[i]);
		if ((*sfd = open(buf,O_RDWR|O_NOCTTY|O_NDELAY)) < 0) {
		    syslog(LOG_ERR,"Can't open slave side of pty: %m");
		    die(1);
		}
		return;
	    }
        }
}

/*
 * Deal with master side packet on the SLIP link.
 */
void proxy_read()
{
    char buffer[4096];
    int len;

    if (mode == MODE_SLIP && state == STATE_UP) {
    	/* Active SLIP transparent mode, just copy to the modem */
        len = read(proxy_mfd,buffer,4096);
	txtotal += len;
        write(modem_fd,buffer,len);
    } else {
        /* read the SLIP packet */
        len = recv_packet(buffer,4096);
	txtotal += len;
	rxtotal -= len;	/* since it will double count on the snoop */
        if (state == STATE_UP) {
	    struct sockaddr to;
	    to.sa_family = AF_INET;
	    strcpy(to.sa_data,snoop_dev);
	    /* if the external iface is up then we can send it on */
	    if (sendto(skfd,buffer,len,0,&to,sizeof(struct sockaddr)) < 0) {
		syslog(LOG_ERR,
		    "Error forwarding data packet to physical device: %m");
	    }
        }
    }
}

/*
 * Deal with modem side packet on the SLIP link.
 */
void modem_read()
{
    char buffer[4096];
    int len = read(modem_fd,buffer,4096);
    rxtotal += len;
    write(proxy_mfd,buffer,len);
}

/*
 * Terminate diald gracefully.
 */

static int in_die = 0;

void die(int i)
{
    int count;

    if (!in_die) {
	in_die = 1;
	/* We're killing without a care here. Uhggg. */
	if (pppd_pid) kill(pppd_pid,SIGINT);
	if (dial_pid) kill(dial_pid,SIGINT);
	/* Wait up to 30 seconds for them to die */
        for (count = 0; (pppd_pid || dial_pid) && count < 30; count++)
	    sleep(1);
	/* If they aren't dead yet, kill them for sure */
	if (pppd_pid) kill(pppd_pid,SIGKILL);
	if (dial_pid) kill(dial_pid,SIGKILL);
	/* Give the system a second to send the signals */
	if (pppd_pid || dial_pid) sleep(1);
	close_modem();
    	proxy_down();
    	exit(i);
    }
}

/*
 * Signal handlers.
 */

/*
 * Modem link went down.
 */
void sig_hup(int sig)
{
    syslog(LOG_INFO, "SIGHUP: modem got hung up on.");
    modem_hup = 1;
}

/*
 * TERM. User wants the link to go down.
 * (Perhaps there should be a 10 second delay? Configurable????)
 */
void sig_term(int sig)
{
    syslog(LOG_INFO, "SIGTERM: Link down request received.");
    request_down = 1;
    request_up = 0;
}

/*
 *  The user has requested that the link be put up.
 */
void linkup(int sig)
{
    syslog(LOG_INFO, "SIGUSR1. External link up request received.");
    request_down = 0;
    request_up = 1;
}

/*
 * A child process died. Find out which one.
 */
void sig_chld(int sig)
{
    int pid, status;

    if ((pid = waitpid(-1,&status,WNOHANG)) == -1) {
	if (errno != ECHILD)
	    syslog(LOG_ERR, "waitpid: %m");
	return;
    }
    if (pid == pppd_pid) pppd_pid = 0;
    if (pid == dial_pid) { dial_status = status; dial_pid = 0; }
    if (pid > 0) {
        if (WIFSIGNALED(status)) {
            syslog(LOG_WARNING, "child process %d terminated with signal %d",
                   pid, WTERMSIG(status));
        }
    }
}

/*
 * Interrupt. User wants diald to be terminated.
 */
void sig_intr(int sig)
{
    syslog(LOG_INFO, "SIGINT. Termination request received.");
    terminate = 1;
}
