/* misc.c -- miscelaneous functions used in git/gitps/gitview. I'll add here
   anything that won't fit anywhere else. */

/* Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by Tudor Hulubei and Andrei Pitis.  */


#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#else /* !HAVE_STDLIB_H */
#include "ansi_stdlib.h"
#endif /* !HAVE_STDLIB_H */

#include <sys/types.h>
#include <pwd.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "xstring.h"
#include "xmalloc.h"
#include "config.h"
#include "file.h"
#include "tty.h"
#include "misc.h"


static char CONFIGFILE_PREFIX[] = "/.gitrc.";
static char lib[] = LIB_DIRECTORY;

char *tty_name;
int  tty_name_len;
char *login_name;
int  login_name_len;


void configuration_fatal_error(configfile)
    char *configfile;
{
    fprintf(stderr, "%s: installation problem: \n", program);
    fprintf(stderr, "%s: cannot find configuration file '%s'.\n\n", program, configfile);
}


void configuration_warning(configfile)
    char *configfile;
{
    fprintf(stderr, "\n%s: cannot open configuration file '%s'.\n",
	    program, configfile);
    fprintf(stderr, "%s: See the manual page for details.\n", program);
    fprintf(stderr, "%s: If the TERM environment variable is, let's say, vt102, your\n",
	    program);
    fprintf(stderr, "%s: configuration file name should be '.gitrc.vt102'.\n",
	    program);
    fprintf(stderr, "%s: You can copy a configuration file in your home directory to\n",
    	    program);
    fprintf(stderr, "%s: overwrite the default one. \n", program);
    fprintf(stderr, "%s: Try modifying '.gitrc.generic'...\n\n", program);
}


void configuration_check()
{
    int config_ok;

    char *configfile = xmalloc(strlen(home) + 1 + strlen(CONFIGFILE_PREFIX) +
    			       strlen(tty_type) + 1);
    strcpy(configfile, home);
    strcat(configfile, CONFIGFILE_PREFIX);
    strcat(configfile, tty_type);

    config_ok = configuration_init(configfile);

    if (!config_ok)
    {
	free(configfile);
	configfile = xmalloc(strlen(lib) + 1 + strlen(CONFIGFILE_PREFIX) +
    			     strlen(tty_type) + 1);
	strcpy(configfile, lib);
	strcat(configfile, CONFIGFILE_PREFIX);
	strcat(configfile, tty_type);

	config_ok = configuration_init(configfile);
        
	if (!config_ok)
        {
	    configuration_warning(configfile);

	    free(configfile);
	    configfile = xmalloc(strlen(lib) + 1 + strlen(CONFIGFILE_PREFIX) +
    			         10 + 1);
	    strcpy(configfile, lib);
	    strcat(configfile, CONFIGFILE_PREFIX);
	    strcat(configfile, "generic");

	    config_ok = configuration_init(configfile);

	    if (!config_ok)
	    {
		configuration_fatal_error(configfile);
		exit(1);
	    }
        }
    }
}


void use_section(section)
    char *section;
{
    if (configuration_section(section) == -1)
    {
	fprintf(stderr,"%s: can't find section %s in the configuration file.\n",
		program, section);
	exit(1);
    }
}


int get_int_var(var_name, options, options_no, default_value)
    char *var_name, *options[];
    int options_no, default_value;
{
    int i;
    char *data;

    configuration_getvarinfo(var_name, &data, 1, DO_SEEK);

    if (data)
    {
	for (i = 0; i < options_no; i++)
	    if (strcmp(data, options[i]) == 0)
		break;

	if (i == options_no)
	    fprintf(stderr, "%s: invalid %s (%s).\n", program, var_name, data);
	else
	    return i;
    }

    return default_value;
}


int get_flag_var(var_name, default_value)
    char *var_name;
    int default_value;
{
    char *data;

    configuration_getvarinfo(var_name, &data, 1, DO_SEEK);

    if (data)
    {
	if (strcmp(data, "ON")  == 0) return 1;
	if (strcmp(data, "OFF") == 0) return 0;
	fprintf(stderr, "%s: invalid %s (%s).\n", program, var_name, data);
	return default_value;
    }

    return default_value;
}


char *get_string_var(var_name, default_value)
    char *var_name, *default_value;
{
    char *data;

    configuration_getvarinfo(var_name, &data, 1, DO_SEEK);

    if (data)
	return xstrdup(data);

    return default_value;
}


void get_colorset_var(charset, colorset_name, fields_no)
    int *charset;
    char *colorset_name[];
    int fields_no;
{
    int i, index;
    char *data;

    for (i = 0; i < fields_no; i++)
    {
	configuration_getvarinfo(colorset_name[i], &data, 1, DO_SEEK);

	if (data)
	{
	    index = tty_get_color_index(data);
	    if (index == -1)
		fprintf(stderr, "%s: invalid %s (%s).\n",
			program, colorset_name[i], data);
	    else
		charset[i] = index;
	}
    }
}


char *clear_path(path)
    char *path;
{
    char *cpath = path;
    char *opath = path;

    if (*opath == 0) return path;
    if (*opath == '/') *cpath++ = *opath++;

    while (*opath)
    {
	while (*opath == '/' || 
	       (*opath == '.' && (*(opath + 1) == '/' || *(opath + 1) == 0)))
	    opath++;
	while (*opath && *opath != '/') *cpath++ = *opath++;
	if (*opath) *cpath++ = '/';
    }
    if (*(cpath - 1) == '/' && cpath - path > 1) cpath--;
    *cpath = 0;
    return path;
}


void get_tty_name()
{
    if ((tty_name = ttyname(1)) == NULL)
    {
	fprintf(stderr, "%s: can't get terminal name.\n", program);
	exit(1);
    }

    tty_name_len = strlen(tty_name);
}


void get_login_name()
{
    struct passwd *pwd;
    int euid = geteuid();

    if ((pwd = getpwuid(euid)) == NULL)
    {
	fprintf(stderr, "%s: OOOPS, I can't get your user name (euid = %d) !\n", program, euid);
	fprintf(stderr, "%s: Your account no longer exists or you are a SYSTEM CRACKER !! :-)\n", program);
	fprintf(stderr, "%s: Correct the problem and try again.\n", program);
	exit(1);
    }

    login_name	   = xstrdup(pwd->pw_name);
    login_name_len = strlen(login_name);
}


void truncate_long_name(name, dest, len)
    char *name, *dest;
    int len;
{
    int name_len;

    if ((name_len = strlen(name)) > len)
    {
        dest[0] = dest[1] = dest[2] = '.';
        memcpy(dest + 3, name + name_len - len + 3, len - 3);
    }
    else
        memcpy(dest, name, name_len);
}


int get_file_length(handle)
    int handle;
{
    int tmp, length;
    
    tmp    = lseek(handle, 0, SEEK_CUR);
    length = lseek(handle, 0, SEEK_END);
    lseek(handle, tmp, SEEK_SET);
    return length;
}
