// SPDX-License-Identifier: GPL-2.0
/*
 * Normal 64-bit CRC calculation.
 *
 * This is a basic crc64 implementation following ECMA-182 specification,
 * which can be found from,
 * https://www.ecma-international.org/publications/standards/Ecma-182.htm
 *
 * Dr. Ross N. Williams has a great document to introduce the idea of CRC
 * algorithm, here the CRC64 code is also inspired by the table-driven
 * algorithm and detail example from this paper. This paper can be found
 * from,
 * http://www.ross.net/crc/download/crc_v3.txt
 *
 * crc64table[256] is the lookup table of a table-driven 64-bit CRC
 * calculation, which is generated by gen_crc64table.c in kernel build
 * time. The polynomial of crc64 arithmetic is from ECMA-182 specification
 * as well, which is defined as,
 *
 * x^64 + x^62 + x^57 + x^55 + x^54 + x^53 + x^52 + x^47 + x^46 + x^45 +
 * x^40 + x^39 + x^38 + x^37 + x^35 + x^33 + x^32 + x^31 + x^29 + x^27 +
 * x^24 + x^23 + x^22 + x^21 + x^19 + x^17 + x^13 + x^12 + x^10 + x^9 +
 * x^7 + x^4 + x + 1
 *
 * Copyright 2018 SUSE Linux.
 *   Author: Coly Li <colyli@suse.de>
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/crc64.h>
#include "crc64table.h"

MODULE_DESCRIPTION("CRC64 calculations");
MODULE_LICENSE("GPL v2");

/**
 * crc64_be - Calculate bitwise big-endian ECMA-182 CRC64
 * @crc: seed value for computation. 0 or (u64)~0 for a new CRC calculation,
	or the previous crc64 value if computing incrementally.
 * @p: pointer to buffer over which CRC64 is run
 * @len: length of buffer @p
 */
u64 __pure crc64_be(u64 crc, const void *p, size_t len)
{
	size_t i, t;

	const unsigned char *_p = p;

	for (i = 0; i < len; i++) {
		t = ((crc >> 56) ^ (*_p++)) & 0xFF;
		crc = crc64table[t] ^ (crc << 8);
	}

	return crc;
}
EXPORT_SYMBOL_GPL(crc64_be);
